#!/usr/bin/python3
#  coding: utf-8
#

import sys, os, glob

try:
    from PySide import QtCore, QtGui
except:
    from PyQt4 import QtCore, QtGui


#-------------------
#  TreeWidget class
#-------------------
class TreeWidget(QtGui.QTreeWidget):
    """ treeWidgetをoverloadして、にsignalを追加する"""
    try:
        reloadSignal = QtCore.pyqtSignal()
    except:
        reloadSignal = QtCore.Signal()


#-----------------------
#  treeWidget_test class
#-----------------------
class treeWidget_test:
    """ tree作成用のクラス
    
    Attribute:
        treeWidget (QTreeWidget)    :QtのtreeWidgte
        maskEvent (bool)            :expandEventのmask用
        iconFolder (QPixmap)        :folderのアイコン（TreeFoam内のicon）
        rootDir (str)               :最上位のdir
        selDir (str)                :selDirまで展開したtreeを作成"""

    def __init__(self, selDir):
        self.treeWidget = TreeWidget()
        self.maskEvent = True           #eventをmaskする
        #アイコンを設定
        iconDir = os.path.normpath("/opt/TreeFoam/icons")
        self.iconFolder = QtGui.QIcon()
        self.iconFolder.addPixmap(QtGui.QPixmap(iconDir + os.sep + "folder16.png"))
        #dirを設定
        self.rootDir = os.getenv("HOME")
        if type(selDir) != str:
            selDir = selDir.encode("utf-8")
        self.selDir = selDir
        #---- data format ------------
        # self.dataDict = {
        #     "/home/caeuser":                [],
        #     "/home/caeuser/CAE":            [0,0],
        #     "/home/caeuser/CAE/test":       [1, 2],
        #     "/home/caeuser/CAE/test1":      [3, 4],
        #     "/home/caeuser/CAE/test1/test": [6, 7],
        #     "/home/caeuser/TEST":           [0,0],
        #     "/home/caeuser/TEST/test_1":    [2, 1]
        #     }
        # self.rootDir = "/home/caeuser"
        # self.selDir = "/home/caeuser/CAE/test1"

    #
    #  main
    #-----------
    def main(self):
        """ dirを取得し、treeを作成する。"""
        rootDir = self.rootDir
        selDir = self.selDir
        #headerを作成
        headers = ["directory", "score", "note"]
        self.createTreeWidget(headers)
        #treeのデータを辞書で取得
        self.dataDict = self.createDataDictionary(rootDir, selDir)
        #辞書データをtreeDataに変換
        self.treeData = self.createTreeData(rootDir)
        #treeDataをtreeWidgetにセット
        self.setItemsToTreeWidget()
        #treeのselDirを展開し、選択
        self.selectDir(selDir)
        #col幅を調整する
        self.adjustColumnsSize()
        #eventMaskを解除
        self.createEvent()
        self.maskEvent = False          #eventMaskを解除

    def close(self):
        QtGui.qApp.quit()

    def createEvent(self):
        """ eventを作成"""
        self.treeWidget.itemChanged.connect(self.onItemChangeed)
        self.treeWidget.itemClicked.connect(self.onItemClicked)
        self.treeWidget.itemExpanded.connect(self.onItemExpanded)
        self.treeWidget.reloadSignal.connect(self.onReloadFolders)

    #------- event handler --------------
    #itemの内容が変更された時
    def onItemChangeed(self, selItem, col):
        pass
    #itemをクリックした時
    def onItemClicked(self, selItem, col):
        self.adjustColumnsSize()
    #itemを展開した時
    def onItemExpanded(self, expItem):
        if self.maskEvent == False:
            self.itemExpanded(expItem)
    #reload
    def onReloadFolders(self):
        self.reloadFolders()
    #-----------------------------------------

    def reloadFolders(self):
        """ folderを再読込。選択folderを記憶し、再表示後選択する。"""
        #treeのデータを辞書で取得
        rootDir = self.rootDir
        items = self.treeWidget.selectedItems()
        if len(items) == 0:
            selDir = self.selDir
        else:
            selDir = self.getDirFromItem(items[0])
        self.dataDict = self.createDataDictionary(rootDir, selDir)
        #辞書データをtreeDataに変換
        self.treeData = self.createTreeData(rootDir)
        #root以下を削除
        rootItem = self.treeWidget.topLevelItem(0)
        for _i in range(rootItem.childCount()):
            delItem = rootItem.child(0)
            rootItem.removeChild(delItem)
        #rootitem以下を再作成
        items = self.treeData[1]
        rootDir = self.treeData[0]
        self.addTreeNodes(rootItem, [rootDir], items)
        #treeのselDirを展開し、選択
        self.selectDir(selDir)
        #col幅を調整する
        self.adjustColumnsSize()

    def itemExpanded(self, item):
        """ itemを展開した時、そのfolder内のdataを再読み込みし、treeを再作成する"""
        #itemの子itemを全て削除（辞書も該当部を削除）
        self.deleteChildItems(item)
        #item以下のfolderを取得
        itemDir = self.getDirFromItem(item)
        folders = self.getFolders(itemDir)
        folders += self.getChildFolders(folders)
        #辞書を追加
        self.addDataToDataDict(folders)
        #treeDataを再作成
        rootDir = self.treeData[0]
        self.treeData = self.createTreeData(rootDir)
        #item以下を再作成
        items = self.getItemsFromTreeData(itemDir)
        #parentDir = [rootDir] + itemDir[len(rootDir)+1:].split("/")
        parentDir = [rootDir] + itemDir[len(rootDir)+1:].split(os.sep)
        self.addTreeNodes(item, parentDir, items)
        #col幅を調整する
        self.adjustColumnsSize()

    def deleteChildItems(self, parentItem):
        """ childItemを全て削除する。
        self.dataDict（辞書）も更新する。"""
        #parentDirを取得
        parentDir = self.getDirFromItem(parentItem)
        #辞書からparentDir以下を削除
        folders = list(self.dataDict.keys())
        for folderDir in folders:
            #if folderDir[:len(parentDir)+1] == parentDir + "/":
            if folderDir[:len(parentDir)+1] == parentDir + os.sep:
                dummy = self.dataDict.pop(folderDir)
        #treeWidgetからchildrenを削除
        for _idx in range(parentItem.childCount()):
            delItem = parentItem.child(0)
            parentItem.removeChild(delItem)

    def getItemsFromTreeData(self, selDir):
        """ dirをたどって、treeDataのitemsを取得する"""

        def getTreeNodes(items, name):
            """ treeData内のitems内からnameが持っているitemsを取得する"""
            for item in items:
                newItems = ""
                if type(item) == str:
                    newItems = name
                else:
                    if item[0] == name:
                        newItems = item[1]
                        break
            return newItems

        rootDir = self.treeData[0]
        items = self.treeData[1]
        #words = selDir[len(rootDir)+1:].split("/")
        words = selDir[len(rootDir)+1:].split(os.sep)
        for name in words:
            items = getTreeNodes(items, name)
        return items

    def addDataToDataDict(self, folders):
        """ folders内のデータをdataDictに追加する"""
        for folder in folders:
            colConts = self.getColContsAtFolder(folder)
            self.dataDict[folder] = colConts

    def createDataDictionary(self, rootDir, selDir):
        """ dirとそれに対応するdataの辞書を作成する。"""
        folders = self.getDirUnderRootToSelDir(rootDir, selDir)
        dataDict = {}
        for folder in folders:
            colConts = self.getColContsAtFolder(folder)
            dataDict[folder] = colConts
        return dataDict

    def getColContsAtFolder(self, folder):
        """ folderDirに対応するdata（colConts）を取得する"""
        #if os.path.exists(folder + "/score.csv"):
        if os.path.exists(folder + os.sep + "score.csv"):
            #f = open(folder + "/score.csv")
            f = open(folder + os.sep + "score.csv", encoding="utf-8")
            cont = f.read()
            f.close()
            colConts = [cont]
        else:    
            colConts = []
        return colConts

    def unselectAllItems(self):
        """ 全itemを非選択に設定する"""
        items = self.treeWidget.selectedItems()
        for item in items:
            item.setSelected(False)

    def selectDir(self, selDir):
        """ selDirを選択表示する。"""
        #全項目を非選択に設定
        self.unselectAllItems()
        #selDirのitemを取得
        selItem = self.getItemFromDir(selDir)
        #selItemまで展開
        self.expandToItem(selItem)
        #selItemを選択
        selItem.setSelected(True)
        #selItemまでscroll
        self.treeWidget.scrollToItem(selItem, QtGui.QAbstractItemView.PositionAtCenter)

    def expandToItem(self, selItem):
        """ rootからselItemまで展開する"""
        #eventMask
        self.maskEvent = True
        #展開するitemを取得
        items = [selItem]
        rootItem = self.treeWidget.topLevelItem(0)
        while selItem is not rootItem:
            selItem = selItem.parent()
            items = [selItem] + items
        items = [rootItem] + items
        #itemを展開
        for item in items:
            self.treeWidget.expandItem(item)
        #mask解除
        self.maskEvent = False

    def getItemFromDir(self, selDir):
        """ dirからitemを取得"""
        rootDir = self.treeData[0]
        #folders = selDir[len(rootDir)+1:].split("/")
        folders = selDir[len(rootDir)+1:].split(os.sep)
        parentItem = self.treeWidget.topLevelItem(0)
        for folder in folders:
            for idx in range(parentItem.childCount()):
                childItem = parentItem.child(idx)
                text = childItem.text(0)
                if text == folder:
                    break
            parentItem = childItem
        return parentItem

    def getDirFromItem(self, selItem):
        """ itemからdirを取得する"""

        def convQstringToText(string):
            """ python2, 3で動作が異なるので、ここで修正。"""
            #python2 or 3 対応に変換
            try:
                #python2の場合、変換する。
                string = string.toUtf8().data()
            except:
                #python3の場合、エラー発生。変換せず。
                pass
            
            #PySide or PyQt4対応で変換
            try:
                #PySideの場合、unicodeで戻るので、strに変換
                if type(string) != str:
                    string = string.encode("utf-8")
            except:
                #PyQt4の場合は、そのまま戻る
                pass
            return string

        words = []
        rootItem = self.treeWidget.topLevelItem(0)
        while selItem is not rootItem:
            text = selItem.text(0)
            text = convQstringToText(text)
            words = [text] + words
            selItem = selItem.parent()
        rootText = rootItem.text(0)
        rootText = convQstringToText(rootText)
        words = [rootText] + words
        #selDir = "/".join(words)
        selDir = os.sep.join(words)
        return selDir

    def createTreeWidget(self, headers):
        """ treeのcolumnタイトルを設定する"""
        nCol = len(headers)
        self.treeWidget.setColumnCount(nCol)
        for col in range(len(headers)):
            header = headers[col]
            self.treeWidget.headerItem().setText(col, header)

    def setItemsToTreeWidget(self):
        """ treeDataをTreeWidgetにセットする"""
        treeData = self.treeData
        dataDict = self.dataDict
        rootName = treeData[0]
        items = treeData[1]
        parentItem = QtGui.QTreeWidgetItem(self.treeWidget)
        parentItem.setText(0, rootName)
        parentItem.setIcon(0, self.iconFolder)
        colConts = dataDict[rootName]
        self.setColConts(parentItem, colConts)
        self.addTreeNodes(parentItem, [rootName], items)

    def setColConts(self, item, colConts):
        """ item中にcol内容をセットする"""
        for i in range(len(colConts)):
            value = colConts[i]
            col = i + 1
            if type(value) == str:
                item.setText(col, value)
            elif type(value) == int or type(value) == float:
                item.setText(col, str(value))
            else:
                item.setIcon(col, value)

    def addTreeNodes(self, parentItem, parentDir, items):
        """ TreeItemを作成する"""
        for item in items:
            if type(item) == str:
                itemDir = parentDir + [item]
                #colConts = self.dataDict["/".join(itemDir)]
                colConts = self.dataDict[os.sep.join(itemDir)]
                child = QtGui.QTreeWidgetItem(parentItem)
                child.setIcon(0, self.iconFolder)
                child.setText(0, item)
                self.setColConts(child, colConts)
            else:
                itemDir = parentDir + [item[0]]
                #colConts = self.dataDict["/".join(itemDir)]
                colConts = self.dataDict[os.sep.join(itemDir)]
                child = QtGui.QTreeWidgetItem(parentItem)
                child.setText(0, item[0])
                child.setIcon(0, self.iconFolder)
                self.setColConts(child, colConts)
                self.addTreeNodes(child, itemDir, item[1])

    def adjustColumnsSize(self):
        """ column幅をadjustする"""
        nCols = self.treeWidget.columnCount()
        for col in range(nCols):
            self.treeWidget.resizeColumnToContents(col)

    def createTreeData(self, rootDir):
        """ treeDataDictからtreeDataを返す"""
        dataDict = self.dataDict        
        rootName = "root"
        treeData = [rootDir, []]
        folders = []
        folderDirs = list(dataDict.keys())
        folderDirs.sort()
        for folder in folderDirs:
            if folder != rootDir:
                folderDir = rootName + folder[len(rootDir):]
                #words = folderDir.split("/")
                words = folderDir.split(os.sep)
                folders.append(words)
        for folderWords in folders:
            treeData = self.setTreeDataFolder(0, treeData, folderWords)
        return treeData

    def setTreeDataFolder(self, nest, tree, folder):
        """ folderをtree状に作成する"""
        nest += 1
        if nest > 50:
            print("error: folder nesting is too deep!")
            return
        #親folderまでskipする
        parentDir = folder[nest]
        ii = 0
        while ii < len(tree[1]):
            if type(tree[1][ii]) == list:
                if tree[1][ii][0] == parentDir:
                    self.setTreeDataFolder(nest, tree[1][ii], folder)
                    break
            else:
                if tree[1][ii] == parentDir:
                    tree[1][ii] = [parentDir, []]
                    self.setTreeDataFolder(nest, tree[1][ii], folder)
                    break
            ii += 1
        #親folderの位置ならchildを追加する
        if nest == len(folder) - 1:
            child = folder[-1]
            tree[1].append(child)
        return tree

    #--------- directoryの取得関連 ---------
    def getDirUnderRootToSelDir(self, rootDir, selDir):
        """ rootDirからselDir系列のdirectoryを取得する"""
        folders = self.getFoldersBetweenDir(rootDir, selDir)
        childFolders = self.getChildFolders(folders)
        folders += childFolders
        return folders

    def getFoldersBetweenDir(self, startDir, endDir):
        """ startDirからendDirまでのfolderを取得する"""
        folders = [startDir]
        folders += self.getFolders(startDir)
        if startDir == endDir:
            return folders
        currDir = ""
        for folder in folders:
            n = len(folder)
            if folder == endDir[:n]:
                currDir = folder
                break
        folders += self.getFolders(currDir)
        #names = endDir[len(currDir)+1:].split("/")
        names = endDir[len(currDir)+1:].split(os.sep)
        for name in names:
            #currDir += "/" + name
            currDir += os.sep + name
            folders += self.getFolders(currDir)
        return folders 

    def getFolders(self, currDir):
        """ curDirのfolderDirを取得する。
        currDirが「div*」の場合は、div*のfolderのみ取得する。"""
        if os.path.basename(currDir)[:len("div")] == "div":
            #items = glob.glob(currDir + "/div*")
            items = glob.glob(currDir + os.sep + "div*")
        else:
            #items = glob.glob(currDir + "/*")
            items = glob.glob(currDir + os.sep + "*")
        folders = list(filter(lambda x: os.path.isdir(x), items))
        return folders

    def getChildFolders(self, folders):
        """ foldersの子foldersを取得する"""
        subFolders = []
        for folder in folders:
            subFolders += self.getFolders(folder)
        return subFolders

    
# def showGui():
#     app = QtGui.QApplication(sys.argv)
#     #ui = treeWidget_test()
#     #ui.main()
#     sys.exit(app.exec_())

# if __name__ == "__main__":
#     showGui()
