#!/usr/bin/python3
#  coding: utf-8
#
#           meshioConvert.py
#
#       meshioを使ったmesh変換
#
#   25/04/05    新規追加
#      06/27    write:「!SECTION」「!END」行を追加
#

import os
import numpy as np
import meshio
import convertMesh as cm
import geometryFunc as geo
import pyFistr


#  meshioNameをFistrNameに変換
meshioFistrDict = {
    #meshio         fistr
    "line":         ["611", "1D"],
    "triangle":     ["731", "2D"],
    "quad":         ["741", "2D"],
    "tetra":        ["341", "3D"],
    "tetra10":      ["342", "3D"],
    "hexahedron":   ["361", "3D"],
    "hexahedron20": ["362", "3D"],
    "wedge":        ["351", "3D"],
    "wedge15":      ["352", "3D"]
    }

fistrMeshioDict = {
    #fistr  meshio
    "111": ["line",         "1D"],
    "112": ["line3",        "1D"],
    "611": ["line",         "1D"],
    "231": ["triangle",     "2D"],
    "731": ["triangle",     "2D"],
    "232": ["triangle6",    "2D"],
    "741": ["quad",         "2D"],
    "241": ["quad",         "2D"],
    "242": ["quad8",        "2D"],
    "341": ["tetra",        "3D"],
    "342": ["tetra10",      "3D"],
    "361": ["hexahedron",   "3D"],
    "362": ["hexahedron20", "3D"],
    "351": ["wedge",        "3D"],
    "352": ["wedge15",      "3D"]
    }

#拡張子
writeExtDict = {
    "abaqus":  "inp",    #Abaqus      (*.inp)
    "ansys":   "msh",    #ANSYS       (*.msh)
    "avsucd":  "avs",    #AVS-UCD     (*.avs)
    "exodus":  "e",      #Exodus      (*.e,*.exo)
    "gmsh":    "msh",    #Gmsh        (*.msh)
    "gmsh22":  "msh",    #Gmsh 2.2    (*.msh)
    "fistr":   "msh",    #FrontISTR   (*.msh)
    "mdpa":    "mdpa",   #Kratos/MDPA (*.mdpa)
    "med":     "med",    #Salome/med  (*.med)
    "medit":   "mesh",   #Medit       (*.mesh,*.meshb)
    "nastran": "bdf",    #Nastran     (*.bdf,*.fem,*.nas)
    "netgen":  "vol",    #Netgen      (*.vol)
    "permas":  "post",   #PERMAS      (*.post,*.dato)
    "vtk":     "vtk",    #VTK         (*.vtk)
    "vtu":     "vtu",    #VTU         (*.vtu)
    "xdmf":    "xdmf"    #XDMF        (*.xdmf,*.xmf)
    }

#----------------
#  read class
#----------------
class read:
    """ meshioを使って他形式のmeshの読み込み。

    使用例:
    import meshioOperation as meshOp
    fileName = "plate.inp"
    readMesh = meshOp.read(fileName, "abaqus")
    meshHeaderData = readMesh.meshio2fistr()
    readMesh.write("FistrModel.msh")
    """

    def __init__(self, fileName, format):
        #  読み込み辞書定義
        self.readFuncDict = {
            #keyword    function名
            "abaqus":  self.readAbaqusMesh,     #Abaqus      (*.inp)
            "ansys":   self.readAnsysMesh,      #ANSYS       (*.msh)
            "avsucd":  self.readAvsucdMesh,     #AVS-UCD     (*.avs)
            "exodus":  self.readExodusMesh,     #Exodus      (*.e,*.exo)
            "gmsh":    self.readGmshMesh,       #Gmsh        (*.msh)
            "mdpa":    self.readMdpaMesh,       #Kratos/MDPA (*.mdpa)
            "med":     self.readMedMesh,        #Salome/med  (*.med)
            "medit":   self.readMeditMesh,      #Medit       (*.mesh,*.meshb)
            "nastran": self.readNastranMesh,    #Nastran     (*.bdf,*.fem,*.nas)
            "netgen":  self.readNetgenMesh,     #Netgen      (*.vol)
            "permas":  self.readPermasMesh,     #PERMAS      (*.post,*.dato)
            "vtk":     self.readVtkMesh,        #VTK         (*.vtk)
            "vtu":     self.readVtuMesh,        #VTU         (*.vtu)
            "xdmf":    self.readXdmfMesh        #XDMF        (*.xdmf,*.xmf)
            }
        #読み込みfileName
        self.fileName = fileName
        self.format = format
        func = self.readFuncDict[format]
        self.mesh = func()
        self.meshHeaderData = []

    #-----読み込み関数定義
    def readAbaqusMesh(self):                   #abaqus
        mesh = meshio.abaqus.read(self.fileName)
        return mesh
    def readAnsysMesh(self):                    #ansys
        mesh = meshio.ansys.read(self.fileName)
        return mesh
    def readAvsucdMesh(self):                   #avsucd
        mesh = meshio.avsucd.read(self.fileName)
        return mesh
    def readExodusMesh(self):                   #exodus
        mesh = meshio.exodus.read(self.fileName)
        return mesh
    def readGmshMesh(self):                     #gmsh
        mesh = meshio.gmsh.read(self.fileName)
        return mesh
    def readMdpaMesh(self):                     #mdpa
        mesh = meshio.mdpa.read(self.fileName)
        return mesh
    def readMedMesh(self):                      #med
        mesh = meshio.med.read(self.fileName)
        return mesh
    def readMeditMesh(self):                    #medit
        mesh = meshio.medit.read(self.fileName)
        return mesh
    def readNastranMesh(self):                  #nastran
        mesh = meshio.nastran.read(self.fileName)
        return mesh
    def readNetgenMesh(self):                   #netgen
        mesh = meshio.netgen.read(self.fileName)
        return mesh
    def readPermasMesh(self):                   #permas
        mesh = meshio.permas.read(self.fileName)
        return mesh
    def readVtkMesh(self):                      #vtk
        mesh = meshio.vtk.read(self.fileName)
        return mesh
    def readVtuMesh(self):                      #vtu
        mesh = meshio.vtu.read(self.fileName)
        return mesh
    def readXdmfMesh(self):                     #xdmf
        mesh = meshio.xdmf.read(self.fileName)
        return mesh

    #  deleteIncludeShellBeamInSolid
    def deleteIncludeShellBeamInSolid(self, elmDict):
        """ solidに含まれるshell,beam、shellに含まれるbeamを削除"""
        #3Dをチェック
        node3dDict = {}
        for ioType in elmDict.keys():
            elType, dim = meshioFistrDict[ioType]
            if dim == "3D" and len(elmDict[ioType]) > 0:
                #nodeNoを取得
                elmData = elmDict[ioType]
                n = len(elmData[0])
                if n == 10:
                    #四面体2次
                    n = 4
                elif n == 20:
                    #六面体2次
                    n = 8
                elif n == 15:
                    #五面体2次
                    n = 6
                for lineData in elmData:
                    for nodeNo in lineData[:n]:
                        if not nodeNo in node3dDict.keys():
                            node3dDict[nodeNo] = elType
        if len(node3dDict) > 0:
            #shell、lineを削除
            for ioType in elmDict.keys():
                elType, dim = meshioFistrDict[ioType]
                if dim == "2D" and len(elmDict[ioType] > 0):
                    #shellを削除
                    newElmData = []
                    elmData = elmDict[ioType]
                    n = len(elmData[0])
                    if len(elmData[0]) == 6 or len(elmData[0]) == 8:
                        n = len(elmData[0]) / 2
                    for i in range(len(elmData)):
                        nodeNos = elmData[i]
                        flag = 0
                        for nodeNo in nodeNos[:n]:
                            if not nodeNo in node3dDict.keys():
                                flag = 1
                                break
                        if flag == 1:
                            newElmData.append(nodeNos)
                    elmDict[ioType] = np.array(newElmData)
                if dim == "1D" and len(elmDict[ioType] > 0):
                    #lineを削除
                    newElmData = []
                    elmData = elmDict[ioType]
                    for i in range(len(elmData)):
                        nodeNos = elmData[i]
                        flag = 1
                        if nodeNos[0] in node3dDict.keys():
                            if nodeNos[-1] in node3dDict.keys():
                                flag = 0
                        if flag == 1:
                            newElmData.append(nodeNos)
                    elmDict[ioType] = np.array(newElmData)
        #2Dをチェック
        node2dDict = {}
        for ioType in elmDict.keys():
            elType, dim = meshioFistrDict[ioType]
            if dim == "2D" and len(elmDict[ioType]) > 0:
                #nodeNoを取得
                elmData = elmDict[ioType]
                n = len(elmData[0])
                if n == 6 or n == 8:
                    n = n // 2
                for lineData in elmData:
                    for nodeNo in lineData[:n]:
                        if not nodeNo in node3dDict.keys():
                            node2dDict[nodeNo] = elType
        if len(node2dDict) > 0:
            #lineを削除
            for ioType in elmDict.keys():
                elType, dim = meshioFistrDict[ioType]
                if dim == "1D" and len(elmDict[ioType]) > 0:
                    #lineを削除
                    newElmData = []
                    elmData = elmDict[ioType]
                    for i in range(len(elmData)):
                        nodeNos = elmData[i]
                        flag = 1
                        if nodeNos[0] in node2dDict.keys():
                            if nodeNos[-1] in node2dDict.keys():
                                flag = 0
                        if flag == 1:
                            newElmData.append(nodeNos)
                    elmDict[ioType] = np.array(newElmData)
        return elmDict

    #  getElementDict
    def getElementDict(self, elmDict, nodeDict):
        """ meshio形式の要素をfistr形式要素に変換する"""
        #要素を取得
        elementDict = {}
        elmNo = 1
        for ioType in elmDict.keys():
            if len(elmDict[ioType]) > 0:
                elType = meshioFistrDict[ioType][0]
                elementDict[elType] = []
                cells = elmDict[ioType] + 1
                print("  " + elType + " elements are getting...")
                for cell in cells:
                    elType, newCell = cm.meshio2fistr_el(ioType, cell)
                    elementDict[elType].append([elmNo] + newCell)
                    elmNo += 1
        return elementDict

    #  getFirstDummyNodeNo
    def getFirstDummyNodeNo(self, nodeDict, dummyDict):
        """ 追加するdummyNodeNoを取得する"""
        if len(dummyDict) > 0:
            nodeNos = []
            for dummyName in dummyDict.keys():
                nodes = list(map(lambda x: x[0], dummyDict[dummyName]))
                nodeNos += nodes
        else:
            nodeNos = list(nodeDict.keys())
        nodeNo = max(nodeNos) + 1
        return nodeNo

    #  getDummyNodesDict
    def getDummyNodesDict(self, elmData, dummyNo):
        """ dummyNodesDict = {<nodeNo>:<dummyNodeNo, ...}を作成"""
        nodes = []
        for lineData in elmData:
            nodes += lineData[1:]
        nodes = list(set(nodes))
        dummyNodesDict = {}
        for nodeNo in nodes:
            dummyNodesDict[nodeNo] = dummyNo
            dummyNo += 1
        return dummyNodesDict

    #  addDummyNodesTo611
    def addDummyNodesTo611(self, elementDict, nodeDict, dummyDict):
        """ 611_beamにdummyを追加する"""
        for elType in elementDict.keys():
            if elType == "611":
                #nodeNoを取得
                dummyNo = self.getFirstDummyNodeNo(nodeDict, dummyDict)
                #dummyNodesDictを取得する
                elmData = elementDict[elType]
                dummyNodesDict = self.getDummyNodesDict(elmData, dummyNo)
                #elmData再作成
                for i in range(len(elmData)):
                    lineData = list(elmData[i])
                    elmNo, nodes = lineData[0], lineData[1:]
                    addNodes = []
                    for ii in range(len(nodes)):
                        dummyNo = dummyNodesDict[nodes[ii]]
                        addNodes.append(dummyNo)
                    lineData = lineData + addNodes
                    elmData[i] = lineData
                elementDict[elType] = elmData
                #dummyNodesを取得
                dummyNodes = []
                for nodeNo in dummyNodesDict.keys():
                    loc = list(nodeDict[nodeNo])
                    dummyNo = dummyNodesDict[nodeNo]
                    dummyNodes.append([dummyNo] + loc)
        if "611" in elementDict.keys():
            #611を641に変更
            data = elementDict.pop("611")
            elementDict["641"] = data
            dummyDict["dummyE641"] = dummyNodes
        return elementDict, dummyDict

    #  addDummyNodesTo731
    def addDummyNodesTo731(self, elementDict, nodeDict, dummyDict):
        """ 731_shellにdummyを追加する"""
        for elType in elementDict.keys():
            if elType == "731":
                #nodeNoを取得
                dummyNo = self.getFirstDummyNodeNo(nodeDict, dummyDict)
                #dummyNodeDictを取得する
                elmData = elementDict[elType]
                dummyNodesDict = self.getDummyNodesDict(elmData, dummyNo)
                for i in range(len(elmData)):
                    lineData = list(elmData[i])
                    elmNo, nodes = lineData[0], lineData[1:]
                    addNodes = []
                    for ii in range(len(nodes)):
                        dummyNo = dummyNodesDict[nodes[ii]]
                        addNodes.append(dummyNo)
                    lineData = lineData + addNodes
                    elmData[i] = lineData
                elementDict[elType] = elmData
                #dummyNodesを取得
                dummyNodes = []
                for nodeNo in dummyNodesDict.keys():
                    loc = list(nodeDict[nodeNo])
                    dummyNo = dummyNodesDict[nodeNo]
                    dummyNodes.append([dummyNo] + loc)
        if "731" in elementDict.keys():
            #731を761に変更
            data = elementDict.pop("731")
            elementDict["761"] = data
            dummyDict["dummyE761"] = dummyNodes
        return elementDict, dummyDict

    #  addDummyNodesTo741
    def addDummyNodesTo741(self, elementDict, nodeDict, dummyDict):
        """ 741_shellにdummyを追加する"""
        for elType in elementDict.keys():
            if elType == "741":
                #nodeNoを取得
                dummyNo = self.getFirstDummyNodeNo(nodeDict, dummyDict)
                #dummyNodeDictを取得する
                #dummyNodes = []
                elmData = elementDict[elType]
                dummyNodesDict = self.getDummyNodesDict(elmData, dummyNo)
                for i in range(len(elmData)):
                    lineData = list(elmData[i])
                    elmNo, nodes = lineData[0], lineData[1:]
                    addNodes = []
                    for ii in range(len(nodes)):
                        dummyNo = dummyNodesDict[nodes[ii]]
                        addNodes.append(dummyNo)
                    lineData = lineData + addNodes
                    elmData[i] = lineData
                elementDict[elType] = elmData
                #dummyNodesを取得
                dummyNodes = []
                for nodeNo in dummyNodesDict.keys():
                    loc = list(nodeDict[nodeNo])
                    dummyNo = dummyNodesDict[nodeNo]
                    dummyNodes.append([dummyNo] + loc)
        if "741" in elementDict.keys():
            #731を761に変更
            data = elementDict.pop("741")
            elementDict["781"] = data
            dummyDict["dummyE781"] = dummyNodes
        return elementDict, dummyDict

    #  checkConvertToDummyElement
    def checkConvertToDummyElement(self, elementDict, nodeDict):
        """ shell, beam要素をdummy付き要素に変換する"""
        dummyDict = {}
        #solid要素があるか？
        flag = 0
        for elType in elementDict.keys():
            elmConts = cm.elmContsDict[elType]
            if elmConts[0] == "solid":
                flag = 1
                break
        if flag == 0:
            #solid要素ない場合は、そのまま
            return elementDict, dummyDict
        #dummyNodeを追加する
        elementDict, dummyDict = self.addDummyNodesTo611(elementDict, nodeDict, dummyDict)
        elementDict, dummyDict = self.addDummyNodesTo731(elementDict, nodeDict, dummyDict)
        elementDict, dummyDict = self.addDummyNodesTo741(elementDict, nodeDict, dummyDict)
        return elementDict, dummyDict

    #  getMeshHeaderData
    def getMeshHeaderData(self, nodeDict, dummyDict, elementDict):
        """ fistrのmeshHeaderDataを取得して返す"""
        meshHeaderData = []
        #header
        header = "!HEADER\n"
        data = ["create mesh from meshio Mesh.\n"]
        meshHeaderData.append([header, data])
        #node
        header = "!NODE\n"
        data = []
        nodeNos = nodeDict.keys()
        for nodeNo in nodeNos:
            loc = list(nodeDict[nodeNo])
            nodeLine = [nodeNo] + loc
            data.append(nodeLine)
        meshHeaderData.append([header, data])
        print("  nodes:", len(nodeNos))
        #dummyNode
        if len(dummyDict) > 0:
            for ngrp in dummyDict.keys():
                header = "!NODE, NGRP=" + ngrp + "\n"
                data = []
                for node in dummyDict[ngrp]:
                    data.append(list(node))
                meshHeaderData.append([header, data])
                print("  " + ngrp + ":", len(data))
        #element
        #  定義順に並べ替え
        elmNoTypes = []
        elmTyps = elementDict.keys()
        for elmType in elmTyps:
            elmNo = elementDict[elmType][0][0]
            elmNoTypes.append([elmNo, elmType])
        elmNoTypes.sort()
        #  定義順に出力
        for elNo, elmType in elmNoTypes:
            header = "!ELEMENT, TYPE=" + elmType + ", EGRP=E" + elmType + "\n"
            data = []
            elms = elementDict[elmType]
            for elm in elms:
                data.append(list(elm))
            meshHeaderData.append([header, data])
            print("  elements type " + elmType + ":", len(elms))
        return meshHeaderData

    #
    #  meshio2fistr
    #---------------
    def meshio2fistr(self):
        """ fistr形式に変換"""
        print("converting " + self.format + " mesh to FrontISTR at meshio library...")
        print("  -- meshio version-" + meshio.__version__ + " --")
        #nodeを取得
        mesh = self.mesh
        nodes = mesh.points
        nodeDict = {}
        print("  nodes are getting...")
        for i in range(len(nodes)):
            nodeNo = i + 1
            loc = nodes[i]
            if len(loc) == 1:
                #1次元の場合、YZ軸を追加
                loc = list(loc) + [0.0, 0.0]
            elif len(loc) == 2:
                #2次元の場合、Z軸を追加
                loc = list(loc) + [0.0]
            nodeDict[nodeNo] = loc
        #elementを取得
        elmDict = {}
        elmDict["line"] = mesh.get_cells_type("line")
        elmDict["triangle"] = mesh.get_cells_type("triangle")
        elmDict["quad"] = mesh.get_cells_type("quad")
        elmDict["tetra"] = mesh.get_cells_type("tetra")
        elmDict["tetra10"] = mesh.get_cells_type("tetra10")
        elmDict["hexahedron"] = mesh.get_cells_type("hexahedron")
        elmDict["hexahedron20"] = mesh.get_cells_type("hexahedron20")
        elmDict["wedge"] = mesh.get_cells_type("wedge")
        #余分な要素を削除
        elmDict = self.deleteIncludeShellBeamInSolid(elmDict)
        #要素のnode並び順をfistr用に修正
        elementDict = self.getElementDict(elmDict, nodeDict)
        #shell,beamをdummy付きの要素に変換
        elementDict, dummyNodeDict = self.checkConvertToDummyElement(elementDict, nodeDict)
        #fistrMesh作成
        self.meshHeaderData = self.getMeshHeaderData(nodeDict, dummyNodeDict, elementDict)
        return self.meshHeaderData

    #
    #  write
    #----------
    def write(self, fileName):
        """ meshHeaderDataからfistr用のmeshFileを作成する
        拡張子は省略可能。"""
        lines = []
        #!NODE,!ELEMENTを作成
        for header, data in self.meshHeaderData:
            lines += [header]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!NODE" or words[0] == "!ELEMENT":
                for lineData in data:
                    line = ", ".join(list(map(str, lineData))) + "\n"
                    lines.append(line)
            else:
                for line in data:
                    lines.append(line)
        #!SECTIONを作成
        for header, data in self.meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                egrp = pyFistr.getValName(header, "EGRP")
                elType = pyFistr.getValName(header, "TYPE")
                sectType = cm.elmContsDict[elType][0].upper()
                sectLine = "!SECTION, TYPE=" + sectType + ", EGRP=" + egrp + "\n"
                lines.append(sectLine)
        #!ENDを作成
        line = "!END\n"
        lines.append(line)
        #fileName確認
        name = os.path.basename(fileName)
        if len(name.split(".")) == 1:
            ext = writeExtDict["fistr"]
            fileName += "." + ext
        #書き込み
        print("write FrontISTR mesh file '" + fileName + "'")
        f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()


#----------------
#  write class
#----------------
class write:
    """ fistr形式のmeshをmeshioを使って他形式に変換する。
    
    使用例:
    import meshioOperation as meshOp
    writeMesh = meshOp.write(meshHeaderData, "abaqus")
    writeMesh.write("test.inp")
    """

    def __init__(self, meshHeaderData, format):
        #  書き込み辞書定義
        self.writeFuncDict = {
            #keyword    function名
            "abaqus":  self.writeAbaqusMesh,    #Abaqus      (*.inp)
            "ansys":   self.writeAnsysMesh,     #ANSYS       (*.msh)
            "avsucd":  self.writeAvsucdMesh,    #AVS-UCD     (*.avs)
            "exodus":  self.writeExodusMesh,    #Exodus      (*.e,*.exo)
            "gmsh":    self.writeGmshMesh,      #Gmsh        (*.msh)
            "gmsh22":  self.writeGmsh22Mesh,    #Gmsh 2.2    (*.msh)
            "mdpa":    self.writeMdpaMesh,      #Kratos/MDPA (*.mdpa)
            "med":     self.writeMedMesh,       #Salome/med  (*.med)
            "medit":   self.writeMeditMesh,     #Medit       (*.mesh,*.meshb)
            "nastran": self.writeNastranMesh,   #Nastran     (*.bdf,*.fem,*.nas)
            "netgen":  self.writeNetgenMesh,    #Netgen      (*.vol)
            "permas":  self.writePermasMesh,    #PERMAS      (*.post,*.dato)
            "vtk":     self.writeVtkMesh,       #VTK         (*.vtk)
            "vtu":     self.writeVtuMesh,       #VTU         (*.vtu)
            "xdmf":    self.writeXdmfMesh       #XDMF        (*.xdmf,*.xmf)
            }
        self.format = format
        self.mesh = self.convertToMeshio(meshHeaderData)

    #-----書き込み関数定義
    def writeAbaqusMesh(self, fileName, mesh):        #abaqus
        meshio.abaqus.write(fileName, mesh)
    def writeAnsysMesh(self, fileName, mesh):         #ansys
        meshio.ansys.write(fileName, mesh)
    def writeAvsucdMesh(self, fileName, mesh):        #avsucd
        meshio.avsucd.write(fileName, mesh)
    def writeExodusMesh(self, fileName, mesh):        #exodus
        meshio.exodus.write(fileName, mesh)
    def writeGmshMesh(self, fileName, mesh):          #gmsh
        meshio.gmsh.write(fileName, mesh, fmt_version="4.1", binary=False)
    def writeGmsh22Mesh(self, fileName, mesh):        #gmsh22
        meshio.gmsh.write(fileName, mesh, fmt_version="2.2", binary=False)
    def writeMdpaMesh(self, fileName, mesh):          #mdpa
        meshio.mdpa.write(fileName, mesh)
    def writeMedMesh(self, fileName, mesh):           #med
        meshio.med.write(fileName, mesh)
    def writeMeditMesh(self, fileName, mesh):         #medit
        meshio.medit.write(fileName, mesh)
    def writeNastranMesh(self, fileName, mesh):       #nastran
        meshio.nastran.write(fileName, mesh)
    def writeNetgenMesh(self, fileName, mesh):        #netgen
        meshio.netgen.write(fileName, mesh)
    def writePermasMesh(self, fileName, mesh):        #permas
        meshio.permas.write(fileName, mesh)
    def writeVtkMesh(self, fileName, mesh):           #vtk
        meshio.vtk.write(fileName, mesh)
    def writeVtuMesh(self, fileName, mesh):           #vtu
        meshio.vtu.write(fileName, mesh)
    def writeXdmfMesh(self, fileName, mesh):          #xdmf
        meshio.xdmf.write(fileName, mesh)

    #  getNodeDictFromHeaderData
    def getNodeDictFromHeaderData(self, meshHeaderData):
        """ nodeDictを作成する"""
        nodeDict = {}
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!NODE":
                for lineData in data:
                    nodeNo, loc = (lineData[0], lineData[1:])
                    nodeDict[nodeNo] = loc
        return nodeDict

    #  delDummyNodesFrom641
    def delDummyNodesFrom641(self, meshHeaderData, nodeDict):
        """ 641要素からdummyNodeを削除して611要素名に換える"""
        for i in range(len(meshHeaderData)):
            header, data = meshHeaderData[i]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elType = pyFistr.getValName(header, "TYPE")
                if elType == "641":
                    newData = []
                    delNodes = []
                    for j in range(len(data)):
                        n = (len(data[j]) - 1) // 2 + 1
                        newNodes = data[j][:n]
                        newData.append(newNodes)
                        delNodes += data[j][n:]
                    newHeader = pyFistr.setValName(header, "TYPE=611")
                    meshHeaderData[i] = [newHeader, newData]
                    delNodes = list(set(delNodes))
                    for nodeNo in delNodes:
                        if nodeNo in nodeDict.keys():
                            _loc = nodeDict.pop(nodeNo)
        return meshHeaderData, nodeDict

    #  delDummyNodesFrom761
    def delDummyNodesFrom761(self, meshHeaderData, nodeDict):
        """ 761要素からdummyNodeを削除して731要素名に換える"""
        for i in range(len(meshHeaderData)):
            header, data = meshHeaderData[i]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elType = pyFistr.getValName(header, "TYPE")
                if elType == "761":
                    newData = []
                    delNodes = []
                    for j in range(len(data)):
                        n = (len(data[j]) -1) // 2 + 1
                        newNodes = data[j][:n]
                        newData.append(newNodes)
                        delNodes += data[j][n:]
                    newHeader = pyFistr.setValName(header, "TYPE=731")
                    meshHeaderData[i] = [newHeader, newData]
                    delNodes = list(set(delNodes))
                    for nodeNo in delNodes:
                        if nodeNo in nodeDict.keys():
                            _loc = nodeDict.pop(nodeNo)
        return meshHeaderData, nodeDict

    #  delDummyNodesFrom781
    def delDummyNodesFrom781(self, meshHeaderData, nodeDict):
        """ 781要素からdummyNodeを削除して741要素名に換える"""
        for i in range(len(meshHeaderData)):
            header, data = meshHeaderData[i]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elType = pyFistr.getValName(header, "TYPE")
                if elType == "781":
                    newData = []
                    delNodes = []
                    for j in range(len(data)):
                        n = (len(data[j]) - 1) // 2 + 1
                        newNodes = data[j][:n]
                        newData.append(newNodes)
                        delNodes += data[j][n:]
                    newHeader = pyFistr.setValName(header, "TYPE=741")
                    meshHeaderData[i] = [newHeader, newData]
                    delNodes = list(set(delNodes))
                    for nodeNo in delNodes:
                        if nodeNo in nodeDict.keys():
                            _loc = nodeDict.pop(nodeNo)
        return meshHeaderData, nodeDict

    #  getElementDictFromHeaderData
    def getElementDictFromHeaderData(self, meshHeaderData):
        """ elementDictを取得して返す"""
        elementDict = {}
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                for lineData in data:
                    elmNo = lineData[0]
                    nodes = lineData[1:]
                    elementDict[elmNo] = nodes
        return elementDict

    #  getSurfaceDictFromSolidElms
    def getSurfaceDictFromSolidElms(self, elmType, solidElmData):
        """ 表面のnodeNoをkeyとしたfaceDictを取得して返す
        faceDict={<nodeNos>:[elmNo, faceNo], ...}"""
        faceDict = {}
        for n in range(len(solidElmData)):
            elm = solidElmData[n]
            elmNo = elm[0]
            nodes = elm[1:]
            elmFaceNodes = cm.faceNodeOfElementDict[elmType]
            for i in range(1, len(elmFaceNodes)):
                order = elmFaceNodes[i]
                nodeNos = list(map(lambda x: nodes[x-1], order))
                nodeNos.sort()
                key = tuple(nodeNos)
                if key in faceDict.keys():
                    _dummy = faceDict.pop(key)
                else:
                    faceDict[key] = [elmNo, i]
        return faceDict

    #  getSurfaceElementsFromSolidElms
    def getSurfaceElementsFromSolidElms(self, elType, solidElmData, faceDict, elementDict):
        """ 表面要素を取得して返す。"""
        #newElmNoを取得
        newElmNo = max(list(elementDict.keys())) + 1
        #表面要素を取得
        elm3Data = []   #三角形1次
        elm4Data = []   #四角形1次
        elm6Data = []   #三角形2次
        elm8Data = []   #四角形2次
        for nodeKey in faceDict.keys():
            elmNo, faceNo = faceDict[nodeKey]
            elNodes = elementDict[elmNo]
            nodeNos = cm.fistr_elFaceNode(elType, elNodes, faceNo)
            if len(nodeNos) == 6:
                #三角形2次要素 連続並びを正しい並びに修正
                elm = [newElmNo] + nodeNos
                newElm = [elm[0], elm[1], elm[3], elm[5], elm[4], elm[6], elm[2]]
                elm6Data.append(newElm)
            elif len(nodeNos) == 8:
                #四角形2次要素 連続並びを正しい並びに修正
                elm = [newElmNo] + nodeNos
                newElm = [elm[0], elm[1], elm[3], elm[5], elm[7], elm[2], elm[4], elm[6], elm[8]]
                elm8Data.append(newElm)
            else:
                #1次要素は、そのまま
                newElm = [newElmNo] + nodeNos
                if len(nodeNos) == 3:
                    elm3Data.append(newElm)
                else:
                    elm4Data.append(newElm)
            elementDict[newElmNo] = nodeNos
            newElmNo += 1
        #headerDataを取得
        newHeaderData = []
        if len(elm3Data) > 0:
            header = "!ELEMENT, TYPE=231\n"
            headerData = [header, elm3Data]
            newHeaderData.append(headerData)
        if len(elm4Data) > 0:
            header = "!ELEMENT, TYPE=241\n"
            headerData = [header, elm4Data]
            newHeaderData.append(headerData)
        if len(elm6Data) > 0:
            header = "!ELEMENT, TYPE=232\n"
            headerData = [header, elm6Data]
            newHeaderData.append(headerData)
        if len(elm8Data) > 0:
            header = "!ELEMENT, TYPE=242\n"
            headerData = [header, elm8Data]
            newHeaderData.append(headerData)
        #headerData作成
        return newHeaderData, elementDict

    #  addSurfaceElements
    def addSurfaceElements(self, meshHeaderData, elementDict):
        """ solid要素の表面要素を取得する"""
        #surfaceを取得
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elType = pyFistr.getValName(header, "TYPE")
                if fistrMeshioDict[elType][1] == "3D":
                    #表面要素のelmNoとfaceNoを取得
                    faceDict = self.getSurfaceDictFromSolidElms(elType, data)
                    newHeaderData, elementDict = self.getSurfaceElementsFromSolidElms(elType, data, faceDict, elementDict)
                    meshHeaderData += newHeaderData
        return meshHeaderData, elementDict

    #  getEdgeDictFromSurfaceElms
    def getEdgeDictFromSurfaceElms(self, elmType, faceElmData):
        """ 全edgeの辞書を取得して返す
        edgeDict={<edgeNode>:[<face1, face2>], ...}"""
        edgeDict = {}
        for n in range(len(faceElmData)):
            elm = faceElmData[n]
            elmNo = elm[0]
            nodes = elm[1:]
            elmEdgeNodes = cm.lineNodeOfFaceDict[elmType]
            for i in range(1, len(elmEdgeNodes)):
                order = elmEdgeNodes[i]
                nodeNos = list(map(lambda x: nodes[x-1], order))
                nodeNos.sort()
                key = tuple(nodeNos)
                if key in edgeDict.keys():
                    edgeDict[key].append(nodes)
                else:
                    edgeDict[key] = [nodes]
        return edgeDict

    #  getNeedEdgeForEdgeElms
    def getNeedEdgeForEdgeElms(self, edgeDict, nodeDict):
        """ edgeDictから
        edgeを共有するfaceが1個のみ、共有faceの角度が30度以上を取得"""
        needDict = {}
        for key in edgeDict.keys():
            faces = edgeDict[key]
            if len(faces) == 1:
                #faceが1個を取得
                needDict[key] = edgeDict[key]
            else:
                nodes0, nodes1 = faces
                nodes0 = nodes0[:3]
                nodes1 = nodes1[:3]
                face0 = list(map(lambda x: nodeDict[x], nodes0))
                face1 = list(map(lambda x: nodeDict[x], nodes1))
                faceAngle = geo.facesAngle(face0, face1)
                #face角度30°以上？
                if abs(faceAngle) < 0.866:
                    #30°以上を取得
                    needDict[key] = edgeDict[key]
        return needDict

    #  getEdgeElementsFromEdgeDict
    def getEdgeElementsFromEdgeDict(self, elType, data, edgeDict, elementDict):
        """ edge要素を取得して返す"""
        #newElmNoを取得
        newElmNo = max(list(elementDict.keys())) + 1
        #edge要素を取得
        elmData = []
        for nodeKey in edgeDict.keys():
            face = edgeDict[nodeKey][0]
            edgeNodes = cm.lineNodeOfFaceDict[elType]
            for i in range(1, len(edgeNodes)):
                nodes = list(map(lambda x: face[x-1], edgeNodes[i])) 
                match = set(nodeKey) - set(nodes)
                if len(match) == 0:
                    edge = nodes
                    break
            elm = [newElmNo] + edge
            elmData.append(elm)
            elementDict[newElmNo] = edge
            newElmNo += 1
        #edgeTypeを取得
        n = len(elmData[0]) - 1
        if n == 2:
            edgeType = "111"
        elif n == 3:
            edgeType = "112"
        #headerData作成
        header = "!ELEMENT, TYPE=" + edgeType + "\n"
        headerData = [header, elmData]
        return headerData, elementDict

    #  addEdgeElements
    def addEdgeElements(self, meshHeaderData, elementDict, nodeDict):
        """ 面要素のedge要素を取得する"""
        #edge不要のformatの場合、直ぐに戻る
        if self.format == "ansys":
            return meshHeaderData, elementDict
        #edge要素を作成
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elType = pyFistr.getValName(header, "TYPE")
                if fistrMeshioDict[elType][1] == "2D":
                    #全edgeを取得
                    edgeDict = self.getEdgeDictFromSurfaceElms(elType, data)
                    #必要なedgeのみ残す
                    edgeDict = self.getNeedEdgeForEdgeElms(edgeDict, nodeDict)
                    #edge要素を取得
                    headerData, elementDict = self.getEdgeElementsFromEdgeDict(elType, data, edgeDict, elementDict)
                    meshHeaderData.append(headerData)
        return meshHeaderData, elementDict

    #  convertToMeshioElement
    def convertToMeshioElement(self, meshHeaderData):
        """ node順をmeshio用に書き換える"""
        for i in range(len(meshHeaderData)):
            header, data = meshHeaderData[i]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elType = pyFistr.getValName(header, "TYPE")
                newData = []
                for ii in range(len(data)):
                    elmNo, nodes = data[ii][0], data[ii][1:]
                    name, newNodes = cm.fistr2meshio_el(elType, nodes)
                    lineData = [elmNo] + newNodes
                    newData.append(lineData)
                meshHeaderData[i] = [header, newData]
                ioType = fistrMeshioDict[elType][0]
                print("  element type " + elType + " -> " + ioType + ":", len(newData))
        return meshHeaderData

    #  renumberNodesElements
    def renumberNodesElements(self, meshHeaderData, nodeDict):
        """ nodeNo, elmNoをゼロ始まりに再設定"""
        #newNoの辞書を作成
        newNodeNoDict = {}
        newNo = 0
        nodeNos = list(nodeDict.keys())
        nodeNos.sort()
        for nodeNo in nodeNos:
            newNodeNoDict[nodeNo] = newNo
            newNo += 1
        #elementのnodeNoをnewNoに置き換える
        for i in range(len(meshHeaderData)):
            header, data = meshHeaderData[i]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                for ii in range(len(data)):
                    elmNo, nodes = data[ii][0], data[ii][1:]
                    nodes = list(map(lambda x: newNodeNoDict[x], nodes))
                    data[ii] = [elmNo] + nodes
                meshHeaderData[i] = [header, data]
        return meshHeaderData        

    #  getPointDictOfDimTags
    def getPointDictOfDimTags(self, meshHeaderData, nodeDict):
        """ gmsh用のdim_tagsを取得する。tag=0で取得"""
        #nodeのdimを取得
        dimDict = {}
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elType = pyFistr.getValName(header, "TYPE")
                n123D = fistrMeshioDict[elType][1]
                dim = int(n123D[0])
                for lineData in data:
                    nodes = lineData[1:]
                    for nodeNo in nodes:
                        if nodeNo in dimDict.keys():
                            dimDict[nodeNo].append(dim)
                        else:
                            dimDict[nodeNo] = [dim]
        #dimTagsを取得
        dimTags = []
        for nodeNo in dimDict.keys():
            dim = max(dimDict[nodeNo])
            dimTags.append([dim, 0])
            #dimTagsDict[nodeNo] = [dim, 0]  #[<dim>, <tag>]
        pointDict = {}
        pointDict["gmsh:dim_tags"] = dimTags
        return pointDict

    #  getCellDictOfGeomPhys
    def getCellDictOfGeomPhys(self, ioDict):
        """ gmsh用のgeometrical, physicalをtag=0で取得する"""
        geoms = []
        physes = []
        for key in ioDict.keys():
            data = ioDict[key]
            vals = [0 for i in range(len(data))]
            geoms.append(vals)
            physes.append(vals)
        cellDict = {}
        cellDict["gmsh:geometrical"] = geoms
        cellDict["gmsh:physical"] = physes
        return cellDict

    #  createMeshioMesh
    def createMeshioMesh(self, meshHeaderData, nodeDict):
        """ meshio形式のmeshを作成"""
        #pointsを取得
        nodeNos = list(nodeDict.keys())
        nodeNos.sort()
        points = []
        for nodeNo in nodeNos:
            loc = nodeDict[nodeNo]
            points.append(loc)
        points = np.array(points)
        #cellsを取得
        dict = {}
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elType = pyFistr.getValName(header, "TYPE")
                ioType = fistrMeshioDict[elType][0]
                cells = []
                for lineData in data:
                    cells.append(lineData[1:])
                if ioType in dict.keys():
                    dict[ioType] += cells
                else:
                    dict[ioType] = cells
        #meshio設定
        if self.format == "gmsh":
            #gmshの場合、point_data, cell_dataを追加する
            pointDict = self.getPointDictOfDimTags(meshHeaderData, nodeDict)
            cellDict = self.getCellDictOfGeomPhys(dict)
            mesh = meshio.Mesh(points=points, cells=dict, point_data=pointDict, cell_data=cellDict)
        else:
            #gmsh以外
            mesh = meshio.Mesh(points=points, cells=dict)
        return mesh

    #  convertToMeshio
    def convertToMeshio(self, meshHeaderData):
        """ meshHeaderDataをmeshio形式に変換する"""
        print("converting FrontISTR mesh to meshio format...")
        print("  -- meshio version-" + meshio.__version__ + " --")
        print("  reading FrontISTR mesh...")
        #nodeDictを取得
        nodeDict = self.getNodeDictFromHeaderData(meshHeaderData)
        #dummyNodeを削除する
        meshHeaderData, nodeDict = self.delDummyNodesFrom641(meshHeaderData, nodeDict)
        meshHeaderData, nodeDict = self.delDummyNodesFrom761(meshHeaderData, nodeDict)
        meshHeaderData, nodeDict = self.delDummyNodesFrom781(meshHeaderData, nodeDict)
        print("  nodes:", len(nodeDict))
        #elementDictを取得
        elementDict = self.getElementDictFromHeaderData(meshHeaderData)
        #solid要素の表面を取得して面要素を作成
        meshHeaderData, elementDict = self.addSurfaceElements(meshHeaderData, elementDict)
        #面要素のedge要素を作成
        meshHeaderData, elementDict = self.addEdgeElements(meshHeaderData, elementDict, nodeDict)
        #meshio形式のnode並び順に変更
        meshHeaderData = self.convertToMeshioElement(meshHeaderData)
        #nodeNoを再設定（ゼロ始まり）
        meshHeaderData = self.renumberNodesElements(meshHeaderData, nodeDict)
        #meshio形式に変換
        mesh = self.createMeshioMesh(meshHeaderData, nodeDict)
        return mesh

    #
    #  write
    #---------
    def write(self, fileName):
        """ meshio.Mesh()を指定されたformatで保存する
        fileNameの拡張子は省略可能。"""
        #fileName確認
        name = os.path.basename(fileName)
        if len(name.split(".")) == 1:
            ext = writeExtDict[self.format]
            fileName += "." + ext
        #変換、書き込み関数を取得
        func = self.writeFuncDict[self.format]
        #書き込み
        func(fileName, self.mesh)
        print("write " + self.format + " mesh file '" + fileName + "'.")


