#!/usr/bin/python3
#  coding:  utf-8
#
#   folderContHddVals.py
#
#       folderContやHDDの容量を取得する
#
#   21/05/09    新規作成
#   23/04/13    "FistrModel.cnt"を使わずにhecmw_ctrl.datから取得する様に修正
#      04/17    "FistrModel.msh"を使わずにhecmw_ctrl.datから取得する様に修正
#      11/03    getNumResults:"FistrModel"をheader名に変更
#   24/07/07    openにencoding="utf-8"を追加
#   25/05/01    getFolderCont:model取得時、pathのbasenameを取得する様に修正。
#      05/02    getFolderCont:nResの取得を廃止。
#      05/03    getNumResults:nResの取得方法を変更して、nResを復活。
#

import os
import shutil
import glob
import abaqus2fistr
import pyFistr

#
#  getFolderCont
#----------------
def getFolderCont(folderDir):
    """ folderの内容を取得する"""

    #
    #  getFolType
    def getFolType(folderDir):
        """ folderTypeを取得する。case, caseLink, folder, foldeLink"""
        if os.path.exists(folderDir + os.sep + "hecmw_ctrl.dat") == True:
            folType = "case"
        else:
            folType = "folder"
        #linkのcheck
        #  windowsのshortCutは、認識しない。
        if os.path.islink(folderDir) == True:
            folType += "Link"
        return folType

    #
    #  getModelName
    def getModelName(folderDir):
        """ mesh変換時のinp or unv ファイル名を取得"""
        model = ""
        fileName = folderDir + os.sep + "easyIstrSub_data"
        if os.path.exists(fileName) == False:
            return model
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        for line in lines:
            if line[:len("sourceMesh")] == "sourceMesh":
                words = line.split()
                if len(words) > 1:
                    model = words[1].split(os.sep)[-1]
                break
        return model

    #
    #  getCaseCont
    def getCaseCont(folderDir):
        """ caseの内容（solType, solver, meshSize, nRes）を取得"""
        solType = ""
        solver = ""
        nRes = ""
        meshSize = ""
        FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(folderDir)
        fileName = folderDir + os.sep + FistrModel_cnt
        if os.path.exists(fileName) == False:
            return (solType, solver, meshSize, nRes)
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        #solutionをチェック
        ist = 0
        for i in range(len(lines)):
            line = lines[i]
            if line[:len("!SOLUTION")].upper() == "!SOLUTION":
                solType = abaqus2fistr.getValName(line, "TYPE").upper()
                ist = i
                break
        if solType == "HEAT":
            #HEATの場合は、定常・非定常を確認
            for i in range(ist, len(lines)):
                line = lines[i]
                if line[:len("!HEAT")].upper() == "!HEAT":
                    words = lines[i+1].replace(" ", "").split(",")
                    try:
                        a = float(words[0])
                        if a != 0.0:
                            #非定常
                            solType = "HEAT_dy"
                    except:
                        pass
                    break
        #solverを取得
        for line in lines:
            if line[:len("!SOLVER")].upper() == "!SOLVER":
                solver = abaqus2fistr.getValName(line, "METHOD")
                break
        #meshSizeを取得
        meshFile = folderDir + os.sep + FistrModel_msh
        if os.path.exists(meshFile):
            meshSize = "%10.3f" % (os.path.getsize(meshFile)/1000000)
        #結果数nResを取得
        nRes = getNumberOfResults(folderDir)
        return (solType, solver, meshSize, nRes)

    folType, model, solType, solver, meshSize, nRes = "folder", "", "", "", "", ""
    folType = getFolType(folderDir)
    if folType == "case" or folType == "caseLink":
        model = getModelName(folderDir)
        (solType, solver, meshSize, nRes) = getCaseCont(folderDir)
        if solType == "":
            folType = ""
    return [folType, solType, solver, model, meshSize, nRes]


#
#  getNumResults
#---------------------
def getNumResults(folderDir):
    """ 結果のflder数を取得"""
    allNames = glob.glob(folderDir + os.sep + "*")
    vtuFiles = []; resFiles = []
    for pathName in allNames:
        name = os.path.basename(pathName)
        if name.find("_vis_psf.") > 0:
            if name.split(".")[-1] == "pvtu":
                vtuFiles.append(name)
        elif name.find(".res.0.") > 0:
            resFiles.append(name)
    n = max([len(vtuFiles), len(resFiles)])
    return n

#
#  getNumberOfResults
def getNumberOfResults(folderDir):
    n = getNumResults(folderDir)
    nRes = ("%4d" % n) + " "
    return nRes

#
#  getDiskSpase
#---------------
def getDiskSpace(folderDir):
    """ HDDの使用状況を取得して返す。
    vals(total, usage, free)
    total: disk.total
    usage: disk.usage
    free :disk.free
    """
    disk = shutil.disk_usage(folderDir)
    total = disk.total
    free = disk.free
    return (total, free)
