#!/usr/bin/python3
#  coding: utf-8
#
#           convertMeshFromOtherMeshDDialog.py
#
#       他形式のメッシュをfistr形式に変換する。
#
#   25/02/24    新規作成
#      04/17    meshioによる他形式のmesh読み込みを追加
#      06/13    第1行目/user/を/usr/に修正
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib

import os
import glob
import shutil
import GtkParts
import universalDialogs

convContDict = {
    #key            表示内容                                pattern
    "unv2fistr":    ["Msh: I-deas Universal  (*.unv)",    ["*.unv"]],
    "abaqus2fistr": ["MshCnt: Calculix  (*.inp)",         ["*.inp"]],
    "readAbaqus":   ["meshio: Abaqus  (*.inp)",           ["*.inp"]],
    "readAnsys":    ["meshio: ANSYS  (*.msh)",            ["*.msh"]],
    "readAvsucd":   ["meshio: AVS-UCD  (*.avs)",          ["*.avs"]],
    "readExodus":   ["meshio: Exodus  (*.e, *.exo)",      ["*.e", "*.exo"]],
    "readGmsh":     ["meshio: Gmsh  (*.msh)",             ["*.msh"]],
    "readMdpa":     ["meshio: Kratos/MDPA  (*.mdpa)",     ["*.mdpa"]],
    "readMed":      ["meshio: Salome/med  (*.med)",       ["*.med"]],
    "readMedit":    ["meshio: Medit  (*.mesh)",           ["*.mesh"]],
    "readNastran":  ["meshio: Nastran  (*.bdf, *.fem, *.nas)", ["*.bdf", "*.fem", "*.nas"]],
    "readNetgen":   ["meshio: Netgen  (*.vol)",           ["*.vol"]],
    "readPermas":   ["meshio: PERMAS  (*.post, *.dato)",  ["*.post", "*.dato"]],
    "readVtk":      ["meshio: VTK  (*.vtk)",              ["*.vtk"]],
    "readVtu":      ["meshio: VTU  (*.vtu)",              ["*.vtu"]],
    "readXdmf":     ["meshio: XDMF  (*.xdmf, *.xmf)",     ["*.xdmf", "*.xmf"]]
}

#--------------------
#  convertMesh class
#--------------------
class convertMesh:

    def __init__(self, currDir, istrData, funcConv=[], parent=None, selKey=""):
        self.builder = Gtk.Builder()
        path = os.getenv("easyIstrPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "convertMeshFromOtherMeshDDialog.glade")
        self.dialog = self.builder.get_object("dialog1")
        self.dialog.set_transient_for(parent)
        self.dialog.connect("delete-event", self.close)
        self.dialog.set_title(_("importMesh"))
        self.builder.connect_signals(self)
        #変数を設定
        self.maskEvent = True
        self.currDir = currDir
        self.istrData = istrData
        self.funcConv = funcConv
        self.selKey = selKey
        self.setGtkObject()
        self.initialize()

    #show
    def show(self):
        """ dialog表示"""
        self.maskEvent = False
        self.dialog.show()

    #close
    def close(self, *args):
        """ dialog閉じる"""
        self.dialog.destroy()

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        """ gtkObjectの定義"""
        self.label_title = self.builder.get_object("label_title")
        self.label_convertCont = self.builder.get_object("label_convertCont")
        self.label_convertMesh = self.builder.get_object("label_convertMesh")
        self.label_otherFormat = self.builder.get_object("label_otherFormat")
        self.label_fileName = self.builder.get_object("label_fileName")
        self.label_afterConvertMesh = self.builder.get_object("label_afterConvertMesh")
        self.label_keepCont = self.builder.get_object("label_keepCont")
        combo_otherFormat = self.builder.get_object("combo_otherFormat")
        self.combo_otherFormat = GtkParts.comboBox(combo_otherFormat)
        self.entry_fileName = self.builder.get_object("entry_fileName")
        self.button_reference = self.builder.get_object("button_reference")
        self.button_cancel = self.builder.get_object("button_cancel")
        self.button_convert = self.builder.get_object("button_convert")
        self.check_addMesh = self.builder.get_object("check_addMesh")
        self.check_keepSetting = self.builder.get_object("check_keepSetting")
        self.label_title.set_text(_("メッシュのimport"))
        msg = _("他形式のメッシュをimportして、FrontISTR形式のメッシュに変換する。") + "\n"
        msg += _("Calculixの(*.inp)は、meshと'*LOAD'等の制御内容もimportする。")
        self.label_convertCont.set_text(_(msg))
        self.label_convertMesh.set_text(_("メッシュのimport"))
        self.label_otherFormat.set_text(_("mesh形式:"))
        self.label_fileName.set_text(_("file名:"))
        self.label_afterConvertMesh.set_text(_("メッシュimport後の処理"))
        self.label_keepCont.set_text(_("（境界条件を保持したまま、メッシュをimportする）"))
        self.button_reference.get_child().set_text(_("参照..."))
        self.button_cancel.get_child().set_text(_("キャンセル"))
        self.button_convert.get_child().set_text(_("mesh変換"))
        self.check_addMesh.get_child().set_text(_("現在のメッシュに追加する"))
        self.check_keepSetting.get_child().set_text(_("設定を保持する"))

    #
    #  initialize
    #-------------
    def initialize(self):
        """ 初期化"""
        #comboを設定
        items = list(map(lambda x: convContDict[x][0], convContDict.keys()))
        self.combo_otherFormat.setItems(items)
        #istrDataを設定
        option = ""; fileName = ""; addMesh = ""; keepCond = ""
        for line in self.istrData:
            words = line.split()
            if len(words) > 1:
                if words[0] == "convertOpt":
                    option = words[1]
                elif words[0] == "sourceMesh":
                    fileName = words[1]
                elif words[0] == "addMesh":
                    addMesh = words[1]
                elif words[0] == "keepCond":
                    keepCond = words[1]
        #comboの設定（istrDataから設定）
        if option in convContDict.keys():
            self.entry_fileName.set_text(fileName)
            selItem = convContDict[option][0]
            self.combo_otherFormat.selectItem(selItem)
        else:
            self.combo_otherFormat.selectItem(0)
        self.check_addMesh.set_active(False)
        self.check_keepSetting.set_active(False)
        if addMesh == "yes":
            self.check_addMesh.set_active(True)
        if keepCond == "yes":
            self.check_keepSetting.set_active(True)
        #comboのitemを選択
        if self.selKey != "":
            #comboの設定
            selItem = convContDict[self.selKey][0]
            self.combo_otherFormat.selectItem(selItem)
            self.combo_otherFormat.comboBox.set_sensitive(False)
            #fileNameをチェック
            if self.selKey != option:
                self.entry_fileName.set_text("")


    #------- event handler -------------------
    def onConvert(self, event):         #変換開始
        self.convert()
        self.close()
    def onCancel(self, event):          #キャンセル
        self.close()
    def onReference(self, event):       #参照...
        self.reference()
    def onChangeItem(self, event):      #comboのitemを変更
        if self.maskEvent == False:
            self.changeItem()

    #
    #  convert
    #----------
    def convert(self):
        """ mesh変換開始"""
        #エラーチェック追加
        comboKey = self.getOtherFormatKey()
        convOpt = comboKey
        sourceMesh = self.entry_fileName.get_text()
        addMesh = self.check_addMesh.get_active()
        keepCond = self.check_keepSetting.get_active()
        funcName = self.funcConv[0]
        self.close()
        funcArgs = [convOpt, sourceMesh, addMesh, keepCond] + self.funcConv[1:]
        funcName(*funcArgs)
        #self.close()

    #  getOtherFormatKey
    def getOtherFormatKey(self):
        """ 選択しているcomboのkeyを取得する"""
        selItem = self.combo_otherFormat.getSelectedItem()
        keys = list(convContDict.keys())
        comboKey = keys[0]
        for key in keys:
            name = convContDict[key][0]
            if name == selItem:
                comboKey = key
                break
        return comboKey

    #
    #  reference
    #------------
    def reference(self):
        """ fileを取得"""
        currDir = self.currDir
        comboKey = self.getOtherFormatKey()
        filterSet = convContDict[comboKey]
        filterSets = [filterSet, [_("全file"), "*.*"]]
        selItem = convContDict[comboKey][0]
        title = selItem + _(" の選択")
        funcOk = [self.setFileToEntry]
        funcCancel = []
        iniName = self.entry_fileName.get_text()
        self.openFileDialog(title, currDir, funcOk, funcCancel, self.dialog, iniName, filterSets)

    #  setFileToEntry
    def setFileToEntry(self, fileName):
        relName = os.path.relpath(fileName, self.currDir)
        self.entry_fileName.set_text(relName)
        
        # name = os.path.basename(fileName)
        # fileDir = os.path.dirname(fileName)
        # self.entry_fileName.set_text(name)
        # if self.currDir != fileDir:
        #     shutil.copy(fileName, self.currDir)

    #
    #  changeItem
    #-------------
    def changeItem(self):
        """ comboの項目を変更した場合"""
        comboKey = self.getOtherFormatKey()
        #entryの内容をclear？
        fileName = self.entry_fileName.get_text()
        fileType = fileName.split(".")[-1]
        filter = convContDict[comboKey]
        patterns = filter[1]
        for pattern in patterns:
            ext = pattern.split(".")[-1]
            if fileType != ext:
                #patternに合致しない時clearする
                self.entry_fileName.set_text("")


    #---universal dialog--------------
    def openFileDialog(self, title, fileDir, funcOk, funcCancel, parent=None, selName="", filterSets=[]):
        dialog = universalDialogs.openFileDialog(title, fileDir, funcOk, funcCancel, parent, selName, filterSets)
        dialog.show()
