#!/usr/bin/python3
#
#       unv2xx.py
#
#   get base unv mesh data
#
#   19/05/07    新規作成
#      06/23    classの内容を整理（余分な内容を削除）
#   24/07/07    openにencoding="utf-8"を追加
#

import os
import sys

startFlag = "    -1"        #unvのデータセットの開始flag
endFlag =   "    -1"        #unvのデータセットの終了flag

#
#  FEM object
#
class FEM:
    """ FEM object structure 
    
    Attributes:
        nodes    list(Node)  :節点データ(Node obj)
        elems list(Element)  :要素データ(element obj)
        nodesets list(Group) :節点setsデータ(Group obj)
        elemsets list(Group) :要素setsデータ(Group obj)
    """
    def __init__(self):
        #self.nodes = []
        self.nodes = {}
        self.elems = []
        self.nodesets = []
        self.elemsets = []

#
#  Node object
#
class Node:
    """ node object 
    
    Attributes:
        coords list(float)  :xyz座標
    """
    def __init__(self, coords):
        self.coords = coords
        self.elmNos = []            #この節点を含む要素No
        self.dummyNode = -1         #対応するdummy節点No

#
#  Element object
#
class Element:
    """ element object 

        Attributes:
            id (int)        :要素No
            type (int)      :要素type
            nnodes (int)    :節点数
            cntvt list(int) :要素を構成する節点
    """
    def __init__(self, id, type, cntvt):
        """ Element 構造を定義 """
        self.id = id
        self.type = type
        self.cntvt = cntvt

#
#  Group object
#
class Group:
    """ group object 
    
    Attributes:
        id (int)        :groupNo
        type (int)      :groupType(nodeGrp:7, elmGrp:8)
        name (str)      :group名
        items list(int) :itemのリスト
    """
    def __init__(self, id, name):
        self.id = id
        self.type = 0
        self.name = name
        self.items = []

#
#  Line2Float
#    行をfloatのlistに変換
def Line2Float(line):
    """ convert a string into a list of float """
    words = line.split()
    try:
        nums = list(map(float, words))
    except:
        words = list(map(lambda x: x.replace("D","e"), words))
        nums = list(map(float, words))
    return nums

#
#  Line2Int
#    行をintのlistに変換
def Line2Int(line):
    """ convert a string into a list of int """
    return list(map(int, line.split()))

#
#  UNV2411Reader
#    2411のデータセット（node）を取得
def UNV2411Reader(file, FEM):
    """ 2411データセット(node)を取得し、FEM objectを返す"""
    print("  2411 data set, reading nodes...")
    loop = True
    while loop:
        #1行読み込み
        line1 = file.readline()
        if len(line1) > 0:
            if line1.startswith(endFlag):
                loop = False
                break
            else:
                line2 = file.readline()
                words = line1.split()
                id = int(words[0])
                coords = Line2Float(line2)
                node = Node(coords)
                FEM.nodes[id] = node
        else:
            loop = False
            break
    return FEM

#
#  UNV2412Reader
#    2412のデータセット（element）を取得
def UNV2412Reader(file, FEM):
    """ 2412データセット(element)を取得し、FEM objectを返す"""
    print("  2412 data set, reading elements...")
    loop = True
    while loop:
        #1行読み込み
        line1 = file.readline()
        if len(line1) > 0:
            if line1.startswith(endFlag) == True:
                loop = False
                break
            else:
                line2 = file.readline()
                words = line1.split()
                id =     int(words[0])
                eltype = int(words[1])
                nnodes = int(words[-1])
                if eltype < 33:
                    #1D elementを取得
                    line3 = file.readline()
                    cntvt = Line2Int(line3)
                else:
                    #2D、3D elementの座標を2行目以降から取得
                    cntvt = Line2Int(line2)
                    if nnodes > 8:
                        cntvt.extend(Line2Int(file.readline()))
                    if nnodes > 16:
                        cntvt.extend(Line2Int(file.readline()))
                    if nnodes > 24:
                        cntvt.extend(Line2Int(file.readline()))
                #elem = Element(id, eltype, nnodes, cntvt)
                elem = Element(id, eltype, cntvt)
                FEM.elems.append(elem)
        else:
            loop = False
    return FEM

#
#  UNV2467Reader
#    2467データセット（group）を取得
def UNV2467Reader(file, FEM):
    """ 2467データセット(group)を取得し、FEM objectを返す"""
    print("  2467 data set, reading groups...")
    loop = True
    while loop:
        #1行読み込み
        line1 = file.readline()
        if len(line1) > 0:
            if line1.startswith(endFlag):
                loop = False
                break
            else:
                #groupの読み込み
                line2 = file.readline()
                groupname = "".join(line2.split())
                words = line1.split()
                id =     int(words[0])
                nitems = int(words[7])
                nlines = (nitems + 1) // 2
                #groupの内容を読み込み
                lst = []
                for i in range(nlines):
                    data = Line2Int(file.readline())
                    lst.append(data[:3])
                    if len(data) > 4:
                        lst.append(data[4:7])
                #nodeとelementのセットに分割
                nset = Group(id, groupname)
                elset = Group(id, groupname)
                nset.type = 7
                elset.type = 8
                for item in lst:
                    if item[0] == 7:
                        nset.items.append(item[1])
                    if item[0] == 8:
                        elset.items.append(item[1])
                #空でないgroupを保存
                if len(nset.items) > 0:
                    FEM.nodesets.append(nset)
                if len(elset.items) > 0:
                    FEM.elemsets.append(elset)
        else:
            loop = False
    return FEM

#
#  UNVParser object
#
class UNVParser:
    """ UNV file 構文解釈、読み込み
    
    Attributes:
        file (file)         :file obj
        filename (str)      :unv file name
        FEM (FEM)           :FEM obj
        datasetsDict (dict) :datasetId VS reader の辞書
        skipIds (func)      :次のdetasetまで読み飛ばし、次のidを返す
        parse (func)        :unvFileを構文解釈し、FEM objを返す
    """
    def __init__(self, filename):
        self.file = None
        self.filename = filename
        self.FEM = FEM()
        # list of supported datasets and corresponding dataset handler functions
        self.datasetsDict = {
            # Id    hundler
            2411: UNV2411Reader,
            2412: UNV2412Reader,
            2467: UNV2467Reader
            } 

    #
    #  skipIds
    #
    def skipIds(self):
        """ skipして次のdatesetIdを返す"""
        id = 0
        loop = True
        while loop:
            line = self.file.readline()
            if len(line) > 0:
                if line.startswith(startFlag):
                    line = self.file.readline()
                    id = int(line)
                    if id in self.datasetsDict.keys():
                        loop = False
                        break
                    #endFlagまでskip
                    while not(line.startswith(endFlag)):
                        line = self.file.readline()
            else:
                loop = False
                break
        return id

    #
    #  parse
    #
    def parse(self):
        """ unvファイルを構文解釈し、FEMデータを取得する"""
        self.file = open(self.filename, "r", encoding="utf-8")
        loop = True
        while loop:
            id = self.skipIds()
            if id in self.datasetsDict.keys():
                func = self.datasetsDict[id]
                self.FEM = func(self.file, self.FEM)
            elif id == 0:
                loop = False
        self.file.close()
        return self.FEM
