#!/usr/bin/python3
#  coding: utf-8
#
#           frd2vtu.py
#
#       calculixのfrdファイルから計算結果をvtu形式に変換する。
#
#   vtu変換対象の結果データ
#       frd内で定義されている結果データが対象
#
#   変換対象のメッシュ
#       solid要素（四面体、五面体、六面体の1次2次要素）が対象
#       frd内で定義されている要素（node,element）を読み取り、変換
#
#------------------------
#   25/01/06    新規作成
#      01/12    convInpFrdToVtu:主応力を追加
#      01/18    getNodeElementDict:node数が3ヶ以下の要素は、無視する事を追加
#               バネ要素を隠すため。
#

import sys, os
import convertMesh as cm
import calcPrincipalStress

#取得する項目名
NODE_RELATED = (
    "NODES",
    "CP3DF",
    "CT3D-MIS",
    "CURR",
    "DEPTH",
    "DISP",
    "DTIMF",
    "ELPOT",
    "EMFB",
    "EMFE",
    "ENER",
    "ERROR",
    "FLUX",
    "FORC",
    "HCRIT",
    "M3DF",
    "MAFLOW",
    "MDISP",
    "MESTRAIN",
    "MSTRAIN",
    "MSTRESS",
    "NDTEMP",
    "PDISP",
    "PE",
    "PFORC",
    "PNDTEMP",
    "PS3DF",
    "PSTRESS",
    "PT3DF",
    "RFL",
    "SDV",
    "SEN",
    "STPRES",
    "STRESS",
    "STRMID",
    "STRNEG",
    "STRPOS",
    "STTEMP",
    "THSTRAIN",
    "TOPRES",
    "TOSTRAIN",
    "TOTEMP",
    "TS3DF",
    "TT3DF",
    "TURB3DF",
    "V3DF",
    "VELO",
    "VSTRES",
    "ZZSTR",
    )
#要素名を取得
NUM_NODES_ELM = {
    4:  "C3D4",     #四面体1次
    10: "C3D10",    #四面体2次
    8:  "C3D8",     #六面体1次
    20: "C3D20",    #六面体2次
    6:  "C3D6",     #五面体1次
    15: "C3D15"     #五面体2次
    }

#
#  getNodeElementDict
#---------------------
def getNodeElementDict(lines):
    #nodeDictを取得
    ip = 0
    ip, nodeDict = getDataOfNodeBlock(lines, ip)
    #  vtkNodeNoを設定
    nodes = list(nodeDict.keys())
    nodes.sort()
    for i in range(len(nodes)):
        nodeNo = nodes[i]
        nodeDict[nodeNo]["vtkNo"] = i
    #elementDictを取得
    ip, elementDict = getDataOfElementBlock(lines, ip)
    #  要素typeを取得
    delElms = []
    for elmNo in elementDict.keys():
        nNodes = len(elementDict[elmNo]["nodes"])
        if nNodes < 4:
            delElms.append(elmNo)
        else:
            type = NUM_NODES_ELM[nNodes]
            elementDict[elmNo]["type"] = type
    for elmNo in delElms:
        dummy = elementDict.pop(elmNo)
    return nodeDict, elementDict

#
#  getMinMaxTimeStep(lines)
#--------------------
def getMinMaxTimeStep(lines):
    """ minMaxのtimeStepを取得する"""
    minTime = 0.0
    for line in lines:
        words = line.split()
        if words[0] == "100CL":
            minTime = float(words[2])
            break
    maxTime = minTime
    for i in range(len(lines)-1, -1, -1):
        words = lines[i].split()
        if words[0] == "100CL":
            maxTime = float(words[2])
            break
    return minTime, maxTime

#
#  getItemNames
#---------------
def getItemNames(lines, minTime):
    """ item名を取得する"""
    items = []
    minTime = -1.0
    for i in range(len(lines)):
        line = lines[i]
        words = line.split()
        if words[0] == "100CL":
            val = float(words[2])
            if minTime < 0.0:
                minTime = val
            elif minTime < val:
                break
            nextLine = lines[i+1]
            wd = nextLine.split()
            newItem = wd[1]
            if newItem in NODE_RELATED:
                items.append(newItem)
    return items

#
#  getTimeStepOfDataBlock
#-------------------------
def getTimeStepOfDataBlock(lines, i):
    timeStep = 0.0
    timeSt = 12; timeEd = 24        #timeを取得
    while i < len(lines):
        line = lines[i]
        if line[:len("  100CL")] == "  100CL":    
            #timeを取得
            timeStep = (line[timeSt:timeEd])
            break
        i += 1
    return i, timeStep

#
#  getDataOfDataBlock
def getDataOfDataBlock(lines, i):
    """ dataBlockのDataを取得"""
    #itemNameを取得
    line = lines[i]
    words = line.split()
    itemName = words[1]
    #data部までskip
    while i < len(lines):
        line = lines[i].strip()
        if line[:2] == "-1":
            break
        i += 1
    #dataを取得
    data = {}        
    count = int(len(lines[i]) / 12)
    indices = [(c * 12, (c + 1) * 12) for c in range(1, count)]
    while i < len(lines):
        line = lines[i].strip()
        if line[:2] != "-1":
            i -= 1              #戻す
            break
        num = int(line[2:12])
        numbers = [float(line[a:b]) for a, b in indices]
        #data.append(tuple(numbers))
        data[num] = tuple(numbers)
        i += 1
    return i, itemName, data

#
#  getDataOfNodeBlock
def getDataOfNodeBlock(lines, i):
    """ NODEのblockDataを取得する"""
    #data部までskip
    for i in range(len(lines)):
        line = lines[i].strip()
        if line[:len("2C")] == "2C":
            i += 1      #次の行に進める
            break
        i += 1
    #dataを取得
    data = {}        
    count = int(len(lines[i]) / 12)
    indices = [(c * 12, (c + 1) * 12) for c in range(1, count)]
    while i < len(lines):
        line = lines[i].strip()
        if line[:2] != "-1":
            i -= 1              #戻す
            break
        num = int(line[2:12])
        numbers = [float(line[a:b]) for a, b in indices]
        loc = tuple(numbers)
        data[num] = {"loc":loc}
        i += 1
    return i, data

#
#  getDataOfElementBlock
def getDataOfElementBlock(lines, i):
    """ ELEMENTのblockDataを取得する"""
    #data部までskip
    while i < len(lines):
        line = lines[i].strip()
        if line[:len("3C")] == "3C":
            i += 1
            break
        i += 1
    #dataを取得
    data = {}
    vals = []
    while i < len(lines):
        line = lines[i].strip()
        words = line.split()
        #elmNoを取得?
        if line[:2] == "-1":
            if len(vals) > 0:
                #valsを取得
                data[elmNo] = {"nodes":tuple(vals)}
            #elmNoを取得
            elmNo = int(words[1])
            vals = []
        #nodeNoを取得?
        elif line[:2] == "-2":
            #valsを取得
            vals += list(map(int, words[1:]))
        #終了?
        else:
            if len(vals) > 0:
                #valsを取得
                data[elmNo] = {"nodes":tuple(vals)}
            break
        i += 1
    return i, data

#
#  getDataTableFromPointer
#--------------------------
def getDataTableFromPointer(lines, currTimeStep, sp):
    """ itemsの結果データをポインタspから取得する"""
    #currTimeStep分のdataを取得
    i = sp
    itemsDataDict = {}
    while i < len(lines):
        i, timeStep = getTimeStepOfDataBlock(lines, i)
        if i >= len(lines) or currTimeStep != timeStep:
            break
        i += 1
        i, item, data = getDataOfDataBlock(lines, i)
        itemsDataDict[item] = data
    return i, timeStep, itemsDataDict

#
#  addMisesStress
#-----------------
def addMisesStress(itemsDataDict):
    """ stressの項目があれば、misesStressを追加する"""
    if "STRESS" in itemsDataDict.keys():
        dataDict = itemsDataDict["STRESS"]
        newData = {}
        for nodeNo in dataDict.keys():
            xx, yy, zz, xy, yz, xz = dataDict[nodeNo]
            C1 = ((xx - yy)**2 + (yy - zz)**2 + (zz - xx)**2) / 2.0
            C2 = 3.0 * (xy**2 + yz**2 + xz**2)
            sm = (C1 + C2)**0.5
            newData[nodeNo] = (sm,)
        itemsDataDict["misesStress"] = newData
    return itemsDataDict

#
#  addPrincipalStress
#---------------------
def addPrincipalStress(itemsDataDict):
    """ stressの項目があれば、主応力を追加する"""
    if "STRESS" in itemsDataDict.keys():
        dataDict = itemsDataDict["STRESS"]
        scalarPrincipalStress = {}
        vectorPrincipalStress1 = {}
        vectorPrincipalStress2 = {}
        vectorPrincipalStress3 = {}
        for nodeNo in dataDict.keys():
            sigmaList = dataDict[nodeNo]
            ans = calcPrincipalStress.principalStress(sigmaList)
            #(SS1, SS2, SS3, SSV1, SSV2, SSV3) = ans
            ((SS1, SS2, SS3), (SSV1, SSV2, SSV3)) = ans
            scalarPrincipalStress[nodeNo] = (SS1,SS2,SS3)
            vectorPrincipalStress1[nodeNo] = tuple(SSV1)
            vectorPrincipalStress2[nodeNo] = tuple(SSV2)
            vectorPrincipalStress3[nodeNo] = tuple(SSV3)
        itemsDataDict["scalarPrincipalStress"] = scalarPrincipalStress
        itemsDataDict["vectorPrincipalStress1"] = vectorPrincipalStress1
        itemsDataDict["vectorPrincipalStress2"] = vectorPrincipalStress2
        itemsDataDict["vectorPrincipalStress3"] = vectorPrincipalStress3
    return itemsDataDict

#
#  getStepNameOfVtk
#-------------------
def getStepNameOfVtk(timeStep, maxTime):
    """ vtkFileのtimeStepを作成する。"""
    if type(timeStep) == str:
        timeStep = float(timeStep)
    maxTimeStr = "%f" % maxTime
    timeStepStr =  "%f" % timeStep
    nMax = len(maxTimeStr.split(".")[0])
    nTime = len(timeStepStr.split(".")[0])
    nAddZero = nMax - nTime
    stepName = "0"*nAddZero + timeStepStr.replace(".", "")
    return stepName

#
#  createVtuLines
#-----------------
def createVtuLines(itemsDataDict, neDicts):
    """ vtu形式に変換する"""
    nodeDict, elementDict = neDicts
    #headerを作成
    lines = ['<?xml version="1.0"?>\n']
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0">\n']
    lines += ["<UnstructuredGrid>\n"]
    #node,element順を取得
    nodes = list(nodeDict.keys())
    nodes.sort()
    elms = list(elementDict.keys())
    elms.sort()
    #node数、cell数を作成
    lines += ['<Piece NumberOfPoints="' + str(len(nodes)) + '" NumberOfCells="' + str(len(elms)) + '">\n']
    #NODEを作成
    lines += ["<Points>\n"]
    lines += ['<DataArray type="Float32" NumberOfComponents="3" format="ascii">\n']
    for nodeNo in nodes:
        loc = nodeDict[nodeNo]["loc"]
        line = " ".join(list(map(str, loc))) + "\n"
        lines += [line]
    lines += ['</DataArray>\n']
    lines += ['</Points>\n']
    #elementを作成
    lines += ['<Cells>\n']
    lines += ['<DataArray type="Int32" Name="connectivity" format="ascii">\n']
    #  connectivity
    for elmNo in elms:
        abaNodes = elementDict[elmNo]["nodes"]
        abaType = elementDict[elmNo]["type"]
        #nodeNoをvtkNodeNoに置き換え
        newNodes = []
        for nodeNo in abaNodes:
            newNo = nodeDict[nodeNo]["vtkNo"]
            newNodes.append(newNo)
        #node順を修正
        name, newNodes = cm.aba2vtk_el(abaType, newNodes)
        line = " ".join(list(map(str, newNodes))) + "\n"
        lines += [line]
    lines += ['</DataArray>\n']
    #  offsets
    lines += ['<DataArray type="Int32" Name="offsets" format="ascii">\n']
    offsets = []
    offset = 0
    for elmNo in elms:
        abaNodes = elementDict[elmNo]["nodes"]
        offset += len(abaNodes)
        offsets.append(offset)
    line = " ".join(list(map(str, offsets))) + "\n"
    lines += [line]
    lines += ['</DataArray>\n']
    #  types
    lines += ['<DataArray type="UInt8" Name="types" format="ascii">\n']
    types = []
    for elmNo in elms:
        elmType = elementDict[elmNo]["type"]
        vtkType, n, func = cm.aba2vtu_elDict[elmType]
        #vtkType = "12"
        types.append(vtkType)
    line = " ".join(types) + "\n"
    lines += [line]
    lines += ['</DataArray>\n']
    lines += ['</Cells>\n']
    #pointDataを作成
    lines += ['<PointData>\n']
    for itemName in itemsDataDict.keys():
        dataDict = itemsDataDict[itemName]
        vals = dataDict[nodes[0]]
        nCompo = str(len(vals))
        lines += ['<DataArray type="Float32" Name="' + itemName + '" NumberOfComponents="' + nCompo + '" format="ascii">\n']
        for nodeNo in nodes:
            vals = dataDict[nodeNo]
            line = " ".join(list(map(str, vals))) + "\n"
            lines += [line]
        lines += ['</DataArray>\n']
    lines += ['</PointData>\n']
    lines += ['</Piece>\n']
    lines += ['</UnstructuredGrid>\n']
    lines += ['</VTKFile>\n']
    return lines

#---------------------
#  convInpFrdToVtu
#---------------------
def convInpFrdToVtu(fileHeader, nConv):
    #frdFileを取得
    frdFile = fileHeader + ".frd"
    f = open(frdFile, encoding="utf-8"); lines = f.readlines(); f.close()
    #node,elmsを取得
    nodeDict, elementDict = getNodeElementDict(lines)
    neDicts = [nodeDict, elementDict]
    #変換する時間を取得
    minTime, maxTime = getMinMaxTimeStep(lines)
    #初期のtimeStepを取得
    ip = 0
    ip, timeStep = getTimeStepOfDataBlock(lines, ip)
    #data変換
    count = 1
    while ip < len(lines):
        ip, tStep, itemsDataDict = getDataTableFromPointer(lines, timeStep, ip)
        print("step:" + timeStep + _("  変換中..."))
        itemsDataDict = addMisesStress(itemsDataDict)
        itemsDataDict = addPrincipalStress(itemsDataDict)
        stepName = getStepNameOfVtk(timeStep, maxTime)
        #vtuファイル作成
        vtuLines = createVtuLines(itemsDataDict, neDicts)
        vtuFile = fileHeader + "_" + stepName + ".vtu"
        f = open(vtuFile, "w", encoding="utf-8"); f.writelines(vtuLines); f.close()
        #終了チェック（全data読み込み後は、tStep=0.0が戻る
        count += 1
        if nConv > 0 and count > nConv:
            #指定数まで変換済み
            break
        if float(tStep) == 0.0:
            #全data変換済み
            break
        #次の計算準備
        timeStep = tStep        

    print(_("calculixの結果をvtuに変換しました。"))

#
#  getArguments
#--------------
def getArguments():
    """ 引数を取得する。"""
    fileHeader = ""
    nConv = -1
    i = 1
    while i < len(sys.argv):
        val = sys.argv[i]
        if val == "-i":
            i += 1
            fileHeader = sys.argv[i]
        elif val == "-n":
            i += 1
            nConv = sys.argv[i]
        i += 1
    return fileHeader, nConv

#
#  printHelp
#------------
def printHelp():
    cont  = "------------ frd2vtu.py ----------------------------\n"
    cont += "calculixの結果file(*.frd)をvtu形式に変換する。\n"
    cont += "*.frdファイル内のdataからnode、elementを取得し、\n"
    cont += "計算結果dataをvtu形式に変換する。\n"
    cont += "変換対象は、solid要素のみ。\n"
    cont += "（四面体、五面体、六面体の1次2次要素が変換の対象）\n"
    cont += "<使い方>\n"
    cont += "  frd2Vtu.py [option]\n"
    cont += "  [option]\n"
    cont += "    -i <headerFile>    inpファイルのheader名\n"
    cont += "    -n <nConv>         変換するvtuファイル数\n"
    cont += "                       省略可（省略時は、全て変換する）\n"
    cont += "    -h, --help:        helpを出力\n"
    cont += "<使用例>\n"
    cont += "  frd2Vtu.py -i model   :model.frdから変換する\n"
    cont += "\n"
    print(cont)



if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    #_ = gettext.gettext

    fileHeader, nConv = getArguments()
    if os.path.exists(fileHeader + ".inp") == False:
        printHelp()
        exit()
    else:
        try:
            nConv = int(nConv)
        except:
            printHelp()
            exit()
    convInpFrdToVtu(fileHeader, nConv)