#!/usr/bin/python3
#
#           fistr2rad.py
#
#       fistr形式のcnt,mshファイルから
#           radioss形式のSTARTER,ENGINEファイルに変換する
#
#   項目    FrontISTR      →       OpenRadiaoss        備考
#   -----------------------------------------------------------------
#   mesh    731,741                 SH3N,SHELL          1次shell
#           761,781                 SH3N,SHELL          1次shell, dummy
#           341,361                 TETRA4,BRICK        1次solid
#           342,362                 TETRA10,BRIC20      2次solid
#           NGRP                    GRNOD/NODE          節点group
#           SGRP                    SURF/SEG            面group
#           EGRP                    GRSH3N,GRSHEL       shell要素group
#                                   GRBRIC              solid要素group
#   材料    ELASTIC                 MAT/PLAS_JOHNS      弾性材料
#           PLASTIC
#             MISES MULTILNEAR      MAT/PLAS_TAB        弾塑性材料
#   拘束    BOUNDARY(0,0,0)         BCS                 変位固定
#           BOUNDARY(変位) (AMP)    IMPDISP (FUNCT)     節点変位
#           CLOAD  (AMP)            CLOAD   (FUNCT)     節点荷重(等分布荷重を含む)
#           DLOAD S  (AMP)          PLOAD   (FUNCT)     面圧力
#           DLOAD GRAV  (AMP)       GRAV    (FUNCT)     重力加速度
#           VELOCITY INITIAL        INIVEL              初速
#           VELOCITY TRANSIT (AMP)  IMPVEL  (FUNCT)     速度
#           ACCELERATION (AMP)      IMPACC  (FUNCT)     加速度
#           CONTACT                 INTER/TYPE7         接触定義 点-面接触のみ
#                                   INTER/TYPE24        接触定義 点-面,面-面接触対応（初期接触あり）
#           CONTACT TIED            INTER/TYPE2         接触ペア結合
#   計算    DYNAMIC                 RUN Tstop           計算終了時間
#             n_step,t_delta
#             ray_m,ray_k           DAMP alpha,beta     Rayleigh減衰αβ
#           WRITE FREQUENCY         ANIM/DT             結果保存間隔
#
#------------------------
#   23/11/15    新規作成
#      12/06    SPHの取得を追加。
#               デフォルトのmsh, cntファイル名をhcemw_ctrl.datから
#               取得するように修正。
#               optionにversionとunitを追加
#      12/07    getSurfaceGroupDict,createGroupSurfaceLines:
#               SGRPを取得して、rad形式変換を追加
#      12/09    rad形式中に、BCS,INIVEL, INTER/TYPE7, MAT, PROP, PART
#               を追加し、そのままで、計算できる様に修正。
#      12/10    sphをstartに追加。
#      12/12    ENGINEファイル作成を追加
#      12/20    SPHのmp算出をegrp全体質量/粒子数で算出する様に修正
#      12/29    塑性材料（PLASTIC, MISES, MULTILINEAR）の変換を追加
#   24/01/06    単位の取得をcurrDir内から取得に変更。
#               option「-w」を追加。（workFolderを取得）
#      01/12    CLOADの変換を追加
#      01/15    PLOADの変換を追加
#      01/17    GRAVの変換を追加
#      01/20    getDload,getCload:AMPが設定されていない場合、一定値のAMP
#               を設定する様に修正。
#      01/24    createContactLines:接触面を結合する「TIED」を追加。
#               getDispBoundary,createDispBoundaryLines:追加。
#               変位を与える拘束条件を追加。
#               getTransitVelocity,createTransitVelocityLines:追加。
#               速度を与える拘束条件を追加。
#      01/25    getValueDirection,getFunctionData:関数を追加してまとめる。
#      01/28    getTransitAcceleration,createTransitAccelerationLines:加速度を追加。
#      01/30    改行コードをlsに変更。lsは、linuxとwindowsで変える。
#      02/03    checkFileNames:「ton_mm_s」を「Mg_mm_s」に変更して変換。
#      02/23    createContactLines:接触定義の「/INTER/TYPE7」と「/INTER/TYPE24」の
#               選択使用を追加。
#               接触type「SURF-SURF」の場合、無条件に「/INTER/TYPE24」に設定。
#               getOption:「-inter」optionを追加
#               printHelp:helpの内容を修正。
#      03/03    getDispBoundary:バグ修正。（name→ampNameに変更）
#      03/05    createElementLines:TETRA10,BRIC20用を追加。
#               createGroupSurfaceLines:2次要素用を追加
#      03/06    getEndTimeFromCntFile:solutionTypeが"dynamic"ではない場合、
#               error発生するように修正。
#      07/05    createElasticMaterialLines,createPlasticMisesMultilinearLines,
#               createPropertyLines:実行時警告が発生していたため修正。
#      07/06    createRadiossStarterFile:等分布荷重を追加
#               getEndTimeFromCntFile:陽解法でない場合、エラー停止する様に修正
#      08/04    getEndTimeFromCntFile:errorMessage内容を修正
#      11/13    getAlphaBeta:rayleigh減衰読み込み追加
#               createAllNodesGroup,createDamperLines:減衰を変換対象に追加
#      11/16    nodeGroup:ALLが使える様に修正。
#               GRAVについては、fistr:egrp, radioss:ngrpに設定する為、
#               GRAV ALLは、全要素を全nodeに置き換え変換する。
#   25/01/10    windowsの改行コードを"\r\n"→"\n"に変更
#

import sys
import os
import datetime
import convertMesh as cm
import geometryFunc as geom
import pyFistr

thisVersion = "1.13-241116"
meshConts = []
incLines = []
radConts = []
engConts = []
latestNodeId = 0
allPartNos = []

ls = "\n"
if os.name == "nt":
    #windows
    #ls = "\r\n"     #改行コード変更
    ls = "\n"

HelpCont = """
------------ fistr2rad.py のヘルプ --------------------------------------
FrontISTR形式のmsh, cntファイルから、
Radioss形式のSTARTER, ENGINEファイルに変換する。

<使い方>
fistr2rad.py [option]
option
    -w <workFolder> :対象のworkFolderを指定する。
                     省略可:省略時はcurrDir。
    -i <header>     :msh, cntファイルのheader名を設定。
                     省略可:省略時はcurrDir内の「hecmw_ctrl.dat」から取得
    -msh <mshFile>  :FistrのmshFile名
                     省略可:省略時はcurrDir内の「hecmw_ctrl.dat」から取得
    -cnt <cntFile>  :FistrのcntFile名
                     省略可:省略時はcurrDir内の「hecmw_ctrl.dat」から取得
    -sph <EGRP>     :指定した要素group名が構成する節点座標を
                     使って、粒子SPHを作成する。
    -o <header>     :inc, radファイルのheader名を設定。
                     省略可:省略時はcntFileのheader名を設定
    -inc <incFile>  :Radioss形式のincludeするfile名
                     省力可:省略時は「<cntFileのheader>_0000.inc」を設定
    -rad <radFile>  :Radioss形式のheader名
                     省略可:省略時は「<cntFileのheader>_0000.rad」を設定
    -vrad <version> :OpenRadiossの年のversionを指定。
                     省略可:省略時は、2022
    -u <unitNo>     :使用する単位系の番号を指定
                     1:MKS, 2:Mg,mm,s, 3:kg,mm,ms, 4:CGS
                     省略可:省略時はcurrDir内の設定unitを読み取る。
                     読み取れない場合は、MKS。
                     「ton,mm,s」は、「Mg,mm,s」に置き換える。
    -inter <type>   :接触定義のtypeを指定。面-面接触は、無条件にtype24が設定。
                     省略可:省略時は「/INTER/TYPE7」ただし面-面接触はtype24。
                     type7:「/INTER/TYPE7」を設定。点-面接触のみ。
                           多用途用、熱接触も可。ただし、初期接触は不可。
                     type24:「/INTER/TYPE24」を設定。点-面、面-面接触対応。
                           構造接触のみだが、初期接触（食込み）も可。
    -v, --version   :fistr2rad.pyのversionを表示。
    -h, --help      :ヘルプを出力

<使用例>
fistr2rad.py
fistr2rad.py -msh FistrModel.msh -inc FistrModel.inc
"""

#
#  要素Typeとgrp名の対応
elmGrpDict = {
  # rad要素    radGroup名
    "SH3N":    "GRSH3N/SH3N",       #三角形shell 1次
    "SHELL":   "GRSHEL/SHEL",       #三角形shell 1次
    "TETRA4":  "GRBRIC/BRIC",       #四面体1次
    "TETRA10": "GRBRIC/BRIC",       #四面体2次
    "BRICK":   "GRBRIC/BRIC",       #6面体1次
    "BRIC20":  "GRBRIC/BRIC"        #6面体2次
    }

#
#  rad要素名と要素typeの辞書
elmTypeDict = {
    "SH3N":    "shell",     #三角形shell 1次
    "SHELL":   "shell",     #三角形shell 1次
    "TETRA4":  "solid",     #四面体1次
    "TETRA10": "solid",     #四面体2次
    "BRICK":   "solid",     #6面体1次
    "BRIC20":  "solid",     #6面体2次
    "SPHCEL":  "sph"        #粒子
    }

#
#  node順入れ替え辞書
fistr2rad_elDict = {
    731: ["SH3N",   [1,2,3], cm.order_3],
    761: ["SH3N",   [1,2,3], cm.order_3],
    741: ["SHELL",  [1,2,3,4], cm.order_4],
    781: ["SHELL",  [1,2,3,4], cm.order_4],
    341: ["TETRA4", [1,2,4,3], cm.order_4],
    342: ["TETRA10", [1,2,3,4,7,5,6,8,9,10], cm.order_10],
    361: ["BRICK",  [1,2,3,4,5,6,7,8], cm.order_8],
    362: ["BRIC20", [1,2,3,4,5,6,7,8,9,10,11,12,17,18,19,20,13,14,15,16], cm.order_20]
    }

#
#  unitNoの辞書
unitDict = {
    #unitNo   unit
    "1": ["kg", "m", "s"],      #MKS
    "2": ["Mg", "mm", "s"],
    "3": ["kg", "mm", "ms"],
    "4": ["g", "cm", "s"]       #CGS
    }

#
#  NGRPのkeyword
ngrpKeywords = ["!BOUNDARY", "!CLOAD", "!TEMPERATURE", "!SPRING",
                "!FLOAD", "!VELOCITY", "!ACCELERATION", "!FIXTEMP",
                "!CFLUX"]

#---- 値の文字数チェック ----------------------------
#
#  checkStrLength10
def checkStrLength10(vals):
    """ 値が20文字以内かチェック"""
    newVals = []
    for val in vals:
        if len(val) >= 10:
            num = float(val)
            newVal = "%10.3e" % num
            newVals.append(newVal)
        else:
            newVals.append(val)
    return newVals

#
#  checkStrLength20
def checkStrLength20(vals):
    """ 値が20文字以内かチェック"""
    newVals = []
    for val in vals:
        if len(val) >= 20:
            num = float(val)
            newVal = "%20.10e" % num
            newVals.append(newVal)
        else:
            newVals.append(val)
    return newVals

#---- meshContsから取得 ----------------------
#
#  getGroupIdFromMeshConts
def getGroupIdFromMeshConts(grpName, grp):
    """ meshContsからgrpとgrpNameに対応するgrpIDを取得する"""
    global meshConts
    for line in meshConts:
        words = line.split()
        ws = words[1].split("/")
        grpID = ""
        if len(ws) >= 2:
            #grp(GRNOD, SURF, GRBRIC, etc)とgrpNameが同じ？
            if ws[1] == grp and words[-1] == grpName:
                grpID = ws[-1]
                break
    return grpID

#
#  getPairIDsFromMeshConts
def getPairIDsFromMeshConts(pairNames):
    """ 接触NODE-SURFのpairNameのIDを取得する。"""
    ngrp, sgrp = pairNames
    ngID = getGroupIdFromMeshConts(ngrp, "GRNOD")
    sgID = getGroupIdFromMeshConts(sgrp, "SURF")
    return ngID, sgID

#
#  getPairSurfIDsFromMeshConts
def getPairSurfIDsFromMeshConts(pairNames):
    """ 接触SURF-SURFのpairNameのIDを取得する。"""
    sgrp0, sgrp = pairNames
    sg0ID = getGroupIdFromMeshConts(sgrp0, "SURF")
    sgID = getGroupIdFromMeshConts(sgrp, "SURF")
    return sg0ID, sgID

#---- radContsから取得 ------------------------
#
#  getGroupIdFromRadConts
def getGroupIdFromRadConts(grpName, item):
    """ radContsからitemとgrpNameに対応するgrpIDを取得する"""
    global radConts
    for line in radConts:
        words = line.split()
        ws = words[0].split("/")
        grpID = ""
        if len(ws) >= 2:
            #item(MAT, PROP, PART, etc)とgrpNameが同じ？
            if ws[1] == item and words[-1] == grpName:
                grpID = ws[-1]
                break
    return grpID

#
#  getPropertyNoFromRadConts
def getPropertyNoFromRadConts(egrp, elmName):
    """ radContsからegrp、elmNameが同じpropIdを取得する。"""
    global radConts
    propNo = ""
    for line in radConts:
        words = line.split()
        ws = words[0].split("/")
        if len(ws) > 1:
            if ws[1] == "PROP":
                egrpProp = words[-1]
                elmProp = words[1]
                if egrpProp == egrp and elmProp == elmName:
                    propNo = ws[-1]
                    break
    return propNo

#
#  getMaterialNoFromRadConts
def getMaterialNoFromRadConts(egrp):
    """ radContsからegrpのmatNoを取得する"""
    global radConts
    matNo = ""
    for line in radConts:
        words = line.split()
        ws = words[0].split("/")
        if len(ws) > 1:
            if ws[1] == "MAT":
                egrpProp = words[-1]
                if egrpProp == egrp:
                    try:
                        a = int(ws[-2])
                        matNo = ws[-2]
                    except:
                        a = int(ws[-1])
                        matNo = ws[-1]
                    break
    return matNo

#
#  getFunctionNoFromRadConts
def getFunctionNoFromRadConts():
    """ radContsからfunctionNoを取得する"""
    global radConts
    funcNos = [0]
    for line in radConts:
        words = line.split()
        ws = words[0].split("/")
        if len(ws) > 1:
            if ws[1] == "FUNCT":
                funcNo = int(ws[2])
                funcNos.append(funcNo)
    funcNos.sort()
    funcNo = funcNos[-1] + 1
    return funcNo

#---- meshConts, radContsの両方から取得 -----------
#
#  getAllGrpID
def getAllGrpID():
    """ ngrp:ALLのgrpIDを取得する"""
    global meshConts, radConts
    grpID = 0
    #meshContsを検索
    for line in meshConts:
        words = line.split()
        ws = words[1].split("/")
        if ws[1] == "GRNOD":
            if words[-1] == "ALL":
                grpID = ws[-1]
                break
    if grpID != 0:
        return grpID
    #radContsを検索
    for line in radConts:
        words = line.split()
        ws = words[0].split("/")
        if len(ws) > 1:
            if ws[1] == "GRNOD":
                if words[-1] == "ALL":
                    grpID = ws[-1]
                    break
    return grpID

#---- その他関数群 -----------------------------------------
#
#  getValueDirection
def getValueDirection(vals):
    """ valsの方向（xyzの文字）とその値を取得し、返す
    vals=[0.0, 0.0, 3.1]の場合、valDir="Z", value=3.1を返す。"""
    xyz = ["X", "Y", "Z"]
    valDir = "X"; value = 0.0
    for i in range(len(vals)):
        if vals[i] != 0.0:
            valDir = xyz[i]
            value = vals[i]
            break
    return valDir, value

#
#  getFunctionData
def getFunctionData(baseVal, ampData):
    """ functionDataを取得（絶対値で取得）する"""
    data = ampData[1]
    funcData = []
    #ABSOLUTE, RELATIVE共、baseValを掛け算する。
    if ampData[0] == "RELATIVE":
        for val, t in data:
            xy = [t, val * baseVal]
            funcData.append(xy)
    else:
        for val, t in data:
            xy = [t, val * baseVal]
            funcData.append(xy)
    return funcData

#
#  getCntMshFileName
def getCntMshFileName(workDir):
    """ hecmw_ctrl.datから、cnt, mshファイル名を取得して返す"""
    cntName = "FistrModel.cnt"
    mshName = "FistrModel.msh"
    fileName = workDir + os.sep + "hecmw_ctrl.dat"
    if os.path.exists(fileName) == False:
        return cntName, mshName
    f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    #cntFile
    i = pyFistr.getNextHeaderName(lines, 0, "!CONTROL", "NAME=fstrCNT")
    i += 1
    if i < len(lines):
        cntName = lines[i].split()[0]
    #mshFile
    i = pyFistr.getNextHeaderName(lines, 0, "!MESH", "TYPE=HECMW-ENTIRE")
    i += 1
    if i < len(lines):
        mshName = lines[i].split()[0]
    return cntName, mshName
#------------------------------------------------------------

#
#  getHeaderDataOfCard
#----------------------
def getHeaderDataOfCard(headerData, cardName):
    """ headerData内から、全てのcardNameのdataを取得する"""
    cardData = []
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == cardName:
            cardData.append([header, data])
    return cardData

#
#  getNodeDataDict
#--------------
def getNodeDataDict(headerData):
    """ nodeDataを取得する。
    ngrp名に「dummy」の文字があるnodeDataは取得しない。
    nodeDict = {<nodeNo>:[x,y,z]}"""
    nodeDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0].upper() == "!NODE":
            for vals in data:
                nodeDict[vals[0]] = vals[1:]
    return nodeDict

#
#  getElementDataDict
#---------------------
def getElementDataDict(headerData):
    """ 要素dataを取得する
    elmDict = {<elmNo>:[[<n1, n2,...], <elmType>]}"""
    elmDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0].upper() == "!ELEMENT":
            elmType = int(pyFistr.getValName(header, "TYPE"))
            for vals in data:
                elmNo = vals[0]
                nodeNos = vals[1:]
                elmDict[elmNo] = [nodeNos, elmType]
    return elmDict

#
#  getNodeElementDict
#----------------------
def getNodeElementDict(elmDict):
    """ nodeが属する要素Noを取得する
    nodeElmDict = {<nodeNo>:[elmNo_1, elmNo_2,..], ..}"""
    nodeElmDict = {}
    for elmNo in elmDict.keys():
        nodeNos = elmDict[elmNo][0]
        for nodeNo in nodeNos:
            if not nodeNo in nodeElmDict.keys():
                nodeElmDict[nodeNo] = set([elmNo])
            else:
                nodeElmDict[nodeNo].add(elmNo)
    return nodeElmDict

#
#  deleteDummyNodesChangeElmType
#--------------------------------
def deleteDummyNodesChangeElmType(nodeDict, elmDict, ngrpDict):
    """ dummyNodeを持っている要素（761、781）のdummyNodeを削除し、
    要素typeを731、741に変更して返す。"""
    dummyNodes = []
    for elmNo in elmDict.keys():
        nodes = elmDict[elmNo][0]
        elmType = elmDict[elmNo][1]
        if elmType == 761:
            dummyNodes += nodes[3:]
            elmDict[elmNo] = [nodes[:3], 731]
        elif elmType == 781:
            dummyNodes += nodes[4:]
            elmDict[elmNo] = [nodes[:4], 741]
    #dummyNodeを削除
    dummyNodes = set(dummyNodes)
    for nodeNo in dummyNodes:
        a = nodeDict.pop(nodeNo)
    #ngrpDicからdummyNodeを削除
    for ngrpName in ngrpDict.keys():
        nodeNos = ngrpDict[ngrpName]
        newNodes = set(nodeNos) - set(dummyNodes)
        ngrpDict[ngrpName] = list(newNodes)
    #ngrpDict内のnode数=0のgrpは削除する
    names = list(ngrpDict.keys())
    for name in names:
        if len(ngrpDict[name]) == 0:
            a = ngrpDict.pop(name)
    return nodeDict, elmDict, ngrpDict

#
#  getShellThicknessDict
#------------------------
def getShellThicknessDict(headerData):
    """ shell要素がある場合は、板圧を取得する
    shellThDict = {<egrp>: [<板圧>, <積分点>], ...}"""
    shellThDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SECTION":
            elmType = pyFistr.getValName(header, "TYPE")
            if elmType == "SHELL":
                egrp = pyFistr.getValName(header, "EGRP")
                ws = pyFistr.deleteSp(data[0]).split(",")
                th = ws[0]      #板圧
                N = ws[1]       #積分点
                shellThDict[egrp] = [th, N]
    return shellThDict

#
#  getContactPair
#-----------------
def getContactPair(headerData):
    """ contactPairを取得する"""
    contactDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!CONTACT PAIR" or words[0] == "!CONTACTPAIR":
            #contactNameを取得
            contactName = pyFistr.getValName(header, "NAME")
            pairNames = pyFistr.deleteSp(data[0]).split(",")
            #contactTypeを取得
            contactType = pyFistr.getValName(header, "TYPE")
            if contactType == "":
                contactType = "NODE-SURF"
            #辞書に保存
            contactDict[contactName] = [pairNames, contactType]
    return contactDict

#
#  getAmplitudeDict
#--------------------
def getAmplitudeDict(headerData):
    """ mshファイルからAMPLITUDEを取得する
    ampDict = {<ampName>:[<valType>, [[value0, time0], [value1, time1],...]]}"""
    ampDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!AMPLITUDE":
            ampName = pyFistr.getValName(header, "NAME")
            valType = pyFistr.getValName(header, "VALUE")
            vals = []
            for line in data:
                ws = pyFistr.deleteSp(line).split(",")
                valTime = list(map(float, ws))          #[value0, time0]
                vals.append(valTime)
            ampDict[ampName] = [valType, vals]
    return ampDict

#
#  getNodeGroupDict
#-------------------
def getNodeGroupDict(headerData):
    """ nodeGroupを取得する。
    group名に「dummy」の文字が入っている場合を除く。
    ngrpDict = {<ngrp名>:{n1, n2,...}}"""
    ngrpDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            name = pyFistr.getValName(header, "NGRP")
            if name != "":
                if not name in ngrpDict.keys():
                    ngrpDict[name] = set([])
                for vals in data:
                    nodeNo = vals[0]
                    ngrpDict[name].add(nodeNo)
        elif words[0] == "!NGROUP":
            name = pyFistr.getValName(header, "NGRP")
            if not name in ngrpDict.keys():
                ngrpDict[name] = set([])
            nodeSet = set(data)
            ngrpDict[name] = ngrpDict[name] | nodeSet
    return ngrpDict

#
#  getSurfaceGroupDict
#----------------------
def getSurfaceGroupDict(headerData, elmDict):
    """ surfaceGroupを取得する
    surfDict = {<SGRP名>:[<nodes0>, <nodes1>,...]}"""
    surfDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SGROUP":
            name = pyFistr.getValName(header, "SGRP")
            if name != "":
                if not name in surfDict.keys():
                    surfDict[name] = []
                for i in range(0, len(data), 2):
                    elmNo = data[i]
                    faceNo = data[i+1]
                    elNodes = elmDict[elmNo][0]
                    elType = str(elmDict[elmNo][1])
                    nodes = cm.fistr_elFaceNode(elType, elNodes, faceNo)
                    surfDict[name] += [nodes]
    return surfDict

#
#  getElementGroupDict
#----------------------
def getElementGroupDict(headerData):
    """ egrpを取得する
    egrpDict = {<egrp名>:{e1, e2,...}}"""
    egrpDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            name = pyFistr.getValName(header, "EGRP")
            if name != "":
                if not name in egrpDict.keys():
                    egrpDict[name] = set([])
                elms = map(lambda x: x[0], data)
                egrpDict[name] = egrpDict[name] | set(elms)
        if words[0] == "!EGROUP":
            name = pyFistr.getValName(header, "EGRP")
            if not name in egrpDict.keys():
                egrpDict[name] = set([])
            elmsSet = set(data)
            egrpDict[name] = egrpDict[name] | elmsSet
    return egrpDict

#
#  renumberNode
#---------------
def renumberNode(nodeDict, elmDict, ngrpDict):
    """ nodeNoをrenumberingする"""
    nodeNos = list(nodeDict.keys())
    nodeNos.sort()
    newNodeNoDict = {}
    newNodeNo = 1
    for nodeNo in nodeNos:
        newNodeNoDict[nodeNo] = newNodeNo
        newNodeNo += 1
    newNodeDict = {}
    for nodeNo in nodeDict.keys():
        newNo = newNodeNoDict[nodeNo]
        newNodeDict[newNo] = nodeDict[nodeNo]
    for elmNo in elmDict.keys():
        nodes = elmDict[elmNo][0]
        newNodes = []
        for nodeNo in nodes:
            newNo = newNodeNoDict[nodeNo]
            newNodes.append(newNo)
        elmDict[elmNo][0] = newNodes
    for ngrpName in ngrpDict.keys():
        nodes = ngrpDict[ngrpName]
        newNodes = []
        for nodeNo in nodes:
            newNo = newNodeNoDict[nodeNo]
            newNodes.append(newNo)
        ngrpDict[ngrpName] = newNodes
    return newNodeDict, elmDict, ngrpDict

#
#  convertElement
#-----------------
def convertElement(elmDict):
    """ 要素の並び順をrad形式に変換する
    radDict = {<elmNo>:[[n1, n2,...], <elmType>]}"""
    radDict = {}
    for elmNo in elmDict.keys():
        nodeNos = elmDict[elmNo][0]
        elmType = elmDict[elmNo][1]
        if elmType in fistr2rad_elDict.keys():
            (radType, n, func) = fistr2rad_elDict[elmType]
            newNos = func(nodeNos, n)
            radDict[elmNo] = [newNos, radType]
        else:
            print()
            print("ERROR: '" + str(elmType) + "' of element type is used. this element type can not conert.")
            exit()
    return radDict

#
#  getEgrpMaterialDictFromCnt
#-----------------------------
def getEgrpMaterialDictFromCnt(cntHeaderData, headerData):
    """ egrpと材料定数をcntFileから取得
    matDict = {<egrp>:[<matName>, E, nu, rho, alpha, []],...}"""
    matValDict = {}
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!MATERIAL":
            matName = pyFistr.getValName(header, "NAME")
            #                      E   nu  rho alpha []
            matValDict[matName] = ["", "", "", "", []]
        elif words[0] == "!ELASTIC":
            ws = pyFistr.deleteSp(data[0]).split(",")
            matValDict[matName][0] = ws[0]
            matValDict[matName][1] = ws[1]
        elif words[0] == "!DENSITY":
            ws = pyFistr.deleteSp(data[0]).split(",")
            matValDict[matName][2] = ws[0]
        elif words[0] == "!EXPANSION_COEFF":
            ws = pyFistr.deleteSp(data[0]).split(",")
            matValDict[matName][3] = ws[0]
        elif words[0] == "!PLASTIC":
            matValDict[matName][4] = [header, data]
    #egrpを取得
    matDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SECTION":
            egrp = pyFistr.getValName(header, "EGRP")
            matName = pyFistr.getValName(header, "MATERIAL")
            if matName in matValDict.keys():
                matDict[egrp] = [matName] + matValDict[matName]
    return matDict

#
#  getEgrpMaterialDict
#------------------------
def getEgrpMaterialDict(headerData):
    """ egrpと材料定数をcntFileから取得
    matDict = {<egrp>:[<matname>, <youngs>, <poisson>, <density>],...}"""
    matDict = {}
    matValDict = {}
    for header, data in headerData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SECTION":
            egrp = pyFistr.getValName(header, "EGRP")
            matName = pyFistr.getValName(header, "MATERIAL")
            matDict[egrp] = [matName]
        elif words[0] == "!MATERIAL":
            matName = pyFistr.getValName(header, "NAME")
        elif words[0] == "!ITEM=1":
            matValDict[matName] = pyFistr.deleteSp(data[0]).split(",")
        elif words[0] == "!ITEM=2":
            matValDict[matName] += pyFistr.deleteSp(data[0]).split(",")
    for egrp in matDict.keys():
        matName = matDict[egrp][0]
        if matName in matValDict.keys():
            matVals = matValDict[matName]
            matDict[egrp] += matVals
        else:
            matVals = ["", "", ""]
            matDict[egrp] += matVals
    return matDict

#
#  createRadiossIncMeshFile
#----------------------------
def createRadiossIncMeshFile(fistrDicts, elmRadDict, sphEgrp, incFile):
    """ RadiossIncMeshファイルを作成する"""
    global meshConts, incLines
    [nodeDict, elmDict, ngrpDict, surfDict, egrpDict,
     matDict, shellThDict, contactDict, nodeElmDict, ampDict] = fistrDicts 
    meshConts = []
    lines = []
    lines += createNodeLines(nodeDict)
    lines += createElementLines(elmRadDict, egrpDict, sphEgrp)
    lines += createGroupNodeLines(ngrpDict)
    lines += createGroupSurfaceLines(surfDict)
    lines += createGroupElementLines(elmRadDict, egrpDict, sphEgrp)
    headerLines = createHeaderLines()
    footerLines = createFooterLines()
    lines = headerLines + lines + footerLines
    incLines = lines
    #file出力
    f = open(incFile, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  createNodeLines
#-------------------
def createNodeLines(nodeDict):
    """ node部のlinesを作成する"""
    global meshConts
    lines = []
    lines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
    lines += ["#  NODES" + ls]
    lines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    lines += ["/NODE" + ls]
    nodeNos = list(nodeDict.keys())
    nodeNos.sort()
    meshConts += ["#   /NODE   Nodes  " + str(len(nodeNos)) + " nodes" + ls]
    for nodeNo in nodeNos:
        line = "%10d" % nodeNo
        x, y, z = nodeDict[nodeNo]
        x, y, z = checkStrLength20([str(x), str(y), str(z)])
        strXYZ = ("%20s" % x) + ("%20s" % y) + ("%20s" % z)
        line += strXYZ + ls
        lines.append(line)
    return lines

#
#  createElementLines
#---------------------
def createElementLines(elmDict, egrpDict, sphEgrp):
    """ elementのlinesを作成する"""
    global meshConts
    radDict = {}
    partNo = 1001
    egrpNames = list(egrpDict.keys())
    egrpNames.sort()
    maxElmNo = 0
    for egrpName in egrpNames:
        elmNos = egrpDict[egrpName]
        for elmNo in elmNos:
            nodeNos = elmDict[elmNo][0]
            elmType = elmDict[elmNo][1]
            key = (partNo, elmType, egrpName)
            if not key in radDict.keys():
                radDict[key] = []
            radDict[key].append([elmNo] + nodeNos)
        partNo += 1
    lines = []
    for key in radDict.keys():
        partNo, elmType, egrpName = key
        if egrpName != sphEgrp:
            #通常の要素を作成
            elms = radDict[key] 
            meshConts += ["#   /" + elmType + "/" + str(partNo) + "  Elements  " + str(len(elms)) + " elms  " + egrpName + ls]
            lines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
            lines += ["#  ELEMENT: partID=" + str(partNo) + "  " + egrpName + ls]
            lines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
            lines += ["/" + elmType + "/" + str(partNo) + ls]
            for i in range(len(elms)):
                elmNo = elms[i][0]
                nodeNos = elms[i][1:]
                if elmType == "TETRA10":
                    line = "%10d" % elmNo + ls
                    lines += [line]
                    line = ""
                    for nodeNo in nodeNos:
                        word = "%10d" % nodeNo
                        line += word
                    line += ls
                    lines += [line]
                elif elmType == "BRIC20":
                    line = "%10d" % elmNo
                    for i in range(0, 8):
                        nodeNo = nodeNos[i]
                        line += "%10d" % nodeNo
                    line += ls
                    lines += [line]
                    line = ""
                    for i in range(8, 16):
                        nodeNo = nodeNos[i]
                        line += "%10d" % nodeNo
                    line += ls
                    lines += [line]
                    line = ""
                    for i in range(16, 20):
                        nodeNo = nodeNos[i]
                        line += "%10d" % nodeNo
                    line += ls
                    lines += [line]
                else:
                    line = ""
                    line += "%10d" % elmNo
                    for nodeNo in nodeNos:
                        word = "%10d" % nodeNo
                        line += word
                    line += ls
                    lines += [line]
        else:
            #SPHを作成
            nodeNosSet = set([])
            elms = radDict[key]
            for i in range(len(elms)):
                nodeNos = elms[i][1:]
                nodeNosSet.update(nodeNos)
            nodeNos = list(nodeNosSet)
            nodeNos.sort()
            meshConts += ["#   /SPHCEL/" + str(partNo) + "  Elements  " + str(len(nodeNos)) + " points   " + egrpName + ls]
            lines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
            lines += ["#  SPH: partID=" + str(partNo) + "  " + egrpName + ls]
            lines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
            lines += ["/SPHCEL/" + str(partNo) + ls]
            for i in range(len(nodeNos)):
                lines += [("%10d" % nodeNos[i]) + ls]
    return lines

#
#  createGroupNodeLines
#------------------------
def createGroupNodeLines(ngrpDict):
    """ groupNode部を作成"""
    global meshConts, latestNodeId
    ngrpNo = 1
    lines = []
    ngrpNames = list(ngrpDict.keys())
    ngrpNames.sort()
    for ngrpName in ngrpNames:
        latestNodeId = ngrpNo
        lines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
        lines += ["#  groupNode: grNodeID=" + str(ngrpNo) + "  " + ngrpName + ls]
        lines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
        lines += ["/GRNOD/NODE/" + str(ngrpNo) + ls]
        lines += ["grNode: " + ngrpName + ls]
        nodeNos = list(ngrpDict[ngrpName])
        meshConts += ["#   /GRNOD/NODE/" + str(ngrpNo) + "  NodeGroup  " + str(len(nodeNos)) + " nodes  " + ngrpName + ls]
        for i in range(0, len(nodeNos), 10):
            nodes = nodeNos[i:i+10]
            line = ""
            for nodeNo in nodes:
                word = "%10d" % nodeNo
                line += word
            line += ls
            lines += [line]
        ngrpNo += 1
    return lines

#
#  createGroupSurfaceLines
def createGroupSurfaceLines(surfDict):
    """ groupSurfaceを作成する"""
    global meshConts
    sgrpNo = 1
    segID = 1
    lines = []
    sgrpNames = list(surfDict.keys())
    sgrpNames.sort()
    for sgrpName in sgrpNames:
        lines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
        lines += ["#  groupSurface: grSurfID=" + str(sgrpNo) + "  " + sgrpName + ls]
        lines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
        lines += ["/SURF/SEG/" + str(sgrpNo) + ls]
        lines += ["grSurface: " + sgrpName + ls]
        segNodes = surfDict[sgrpName]
        meshConts += ["#   /SURF/SEG/" + str(sgrpNo) + "  SurfaceGroup  " + str(len(segNodes)) + " faces  " + sgrpName + ls]
        for nodes in segNodes:
            words = []
            #2次要素？
            if len(nodes) > 4:
                #1個おきにnodeを取得
                for i in range(0, len(nodes), 2):
                    node = nodes[i]
                    words.append(("%10d" % node))
            #1次要素？
            else:
                #全nodeを取得    
                for node in nodes:
                    words.append(("%10d" % node))
            lines += [("%10d" % segID) + "".join(words) + ls]
            segID += 1
        sgrpNo += 1
    return lines

#
#  createGroupElementLines
#--------------------------
def createGroupElementLines(elmDict, egrpDict, sphEgrp):
    """ groupElement部を作成"""
    global meshConts
    egrpNo = 1
    radDict = {}
    egrpNames = list(egrpDict.keys())
    for egrpName in egrpNames:
        elmNos = egrpDict[egrpName]
        for elmNo in elmNos:
            elmType = elmDict[elmNo][1]     #rad要素type
            grpName = elmGrpDict[elmType]   #radGrp名
            key = (egrpNo, grpName, egrpName)
            if not key in radDict.keys():
                radDict[key] = []
            radDict[key].append(elmNo)
        egrpNo += 1
    lines = []
    for key in radDict.keys():
        egrpNo, grpName, egrpName = key
        if egrpName != sphEgrp:
            lines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
            lines += ["#  groupElms: grElmID=" + str(egrpNo) + "  " + egrpName + ls]
            lines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
            lines += ["/" + grpName + "/" + str(egrpNo) + ls]
            lines += ["grElm: " + egrpName + ls]
            elmNos = radDict[key]
            meshConts += ["#   /" + grpName + "/" + str(egrpNo) + "  ElementGroup  " + str(len(elmNos)) + " elms  " + egrpName + ls]
            for i in range(0, len(elmNos), 10):
                line = ""
                elms = elmNos[i:i+10]
                for elmNo in elms:
                    word = "%10d" % elmNo
                    line += word
                line += ls
                lines += [line]
    return lines
         
#
#  createHeaderLines
#--------------------
def createHeaderLines():
    global meshConts
    addLines = []
    addLines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
    addLines += ["#  convert mesh from Fistr to Radioss format" + ls]
    addLines += ["#" + ls]
    addLines += ["#     mesh contents" + ls]
    addLines += meshConts
    addLines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
    addLines += ["/BEGINE" + ls]
    addLines += ["convert mesh" + ls]
    return addLines

#
#  createFooterLines
#--------------------
def createFooterLines():
    """ footerを作成"""
    addLines = ["/END" + ls]
    addLines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
    return addLines

#
#  getAllNameFlag
#------------------
def getAllNameFlag(cntHeaderData):
    """ ngrp:ALLを使っているか確認"""
    NallFlag = 0
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        #ngrp:ALLを確認
        if words[0] in ngrpKeywords:
            ws = pyFistr.deleteSp(data[0]).split(",")
            if ws[0] == "ALL":
                NallFlag = 1
    return NallFlag

#
#  addConstantAmpDict
#---------------------
def addConstantAmpDict(cntHeaderData, ampDict):
    """ 一定値のAMPを作成し、ampDictに追加する"""
    #計算終了時間
    endTime = getEndTimeFromCntFile(cntHeaderData) * 2.0
    ampName = "AMP100"
    valType = "RELATIVE"
    val0 = [1.0, 0.0]
    val1 = [1.0, endTime]
    vals = [val0] + [val1]
    ampDict[ampName] = [valType, vals]
    return ampDict

#
#  getFixBoundary
#-----------------
def getFixBoundary(cntHeaderData):
    """ boundary(0,0,0)を取得する。回転は取得しない。"""
    global meshConts

    def convStr(fix):
        ans = str(fix[0]) + str(fix[1]) + str(fix[2])
        return ans

    fixNameDict = {}
    fix = [0,0,0]
    name = ""
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!BOUNDARY":
            for line in data:
                ws = pyFistr.deleteSp(line).split(",")
                if name != ws[0]:
                    if fix != [0,0,0] and name.find("dummy") < 0:
                        fixNameDict[name] = [convStr(fix)]
                    name = ws[0]
                    fix = [0,0,0]
                if float(ws[3]) == 0.0:     #変位「0」を取得
                    col = int(ws[1]) - 1    #方向を取得
                    fix[col] = 1
    if fix != [0,0,0] and name.find("dummy") < 0:
        fixNameDict[name] = [convStr(fix)]
    #nodeGrpIDを取得
    for name in fixNameDict.keys():
        grpID = getGroupIdFromMeshConts(name, "GRNOD")
        fixNameDict[name].append(grpID)
    return fixNameDict

#
#  getDispBoundary
#--------------------
def getDispBoundary(cntHeaderData, ampDict):
    """ BOUNDARY 変位させるを取得する"""
    global radConts
    dispNameDict = {}
    disp = [0,0,0]
    name = ""
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!BOUNDARY":
            for line in data:
                ws = pyFistr.deleteSp(line).split(",")
                if name != ws[0]:
                    if disp != [0,0,0] and name.find("dummy") < 0:
                        dispNameDict[name] = [disp, ampDict[ampName]]
                    #newNameを登録(初期化)
                    name = ws[0]
                    disp = [0,0,0]
                    #AMPを取得
                    ampName = pyFistr.getValName(header, "AMP")
                    if ampName == "":
                        ampName = "AMP100"
                if float(ws[3]) != 0.0:     #変位「0」以外を取得
                    col = int(ws[1]) - 1
                    disp[col] = float(ws[3])
    #最後のnameを保存
    if disp != [0,0,0] and name.find("dummy") < 0:
        #dispNameDict[name] = [disp, ampDict[name]]
        dispNameDict[name] = [disp, ampDict[ampName]]
    #nodeGrpIDを取得
    for name in dispNameDict.keys():
        grpID = getGroupIdFromMeshConts(name, "GRNOD")
        dispNameDict[name].append(grpID)
    return dispNameDict

#
#  getCload
#------------
def getCload(cntHeaderData, ampDict):
    """ CLOADの荷重とAMPLITUDEを取得"""
    cloadDict = {}
    name = ""
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!CLOAD":
            #AMPが設定されていない場合、一定値のAMPを設定
            ampName = pyFistr.getValName(header, "AMP")
            if ampName == "":
                ampName = "AMP100"
            for line in data:
                ws = pyFistr.deleteSp(line).split(",")
                if ws[0] != "" and ws[0].find("dummy") < 0:
                    name = ws[0]
                    if not name in cloadDict.keys():
                        cloadDict[name] = [[0,0,0], ampDict[ampName]]
                    col = int(ws[1]) - 1
                    cloadDict[name][0][col] = float(ws[2])
    #nodeGrpIDを取得
    for name in cloadDict.keys():
        grpID = getGroupIdFromMeshConts(name, "GRNOD")
        cloadDict[name].append(grpID)
    #ngrpNameをチェック
    ansDict = {}
    undefineDict = {}
    for name in cloadDict.keys():
        try:
            nodeNo = int(name)
            cont = cloadDict[name]
            newName = "node" + name
            ansDict[newName] = cont
            undefineDict[newName] = [nodeNo]
        except:
            ansDict[name] = cloadDict[name]
    return ansDict, undefineDict

#
#  getDlaod
#-----------
def getDload(cntHeaderData, ampDict):
    """ DLOADの圧力とAMPLITUDEを取得"""
    dloadDict = {}
    name = ""
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!DLOAD":
            #AMP設定無い場合、一定値のAMPを使う
            ampName = pyFistr.getValName(header, "AMP")
            if ampName == "":
                ampName = "AMP100"
            ws = pyFistr.deleteSp(data[0]).split(",")
            if ws[1] == "S":
                if ws[0] != "" and ws[0].find("dummy") < 0:
                    name = ws[0]
                    dloadDict[name] = [ws[2], ampDict[ampName]]
    #surfaceGroupIDを取得
    for name in dloadDict.keys():
        grpID = getGroupIdFromMeshConts(name, "SURF")
        dloadDict[name].append(grpID)
    return dloadDict

#
#  getGrav
#-----------
def getGrav(cntHeaderData, ampDict):
    """ GRAV（DLOAD, GRAV）とAMPLITUDEを取得。
    AMPLITUDEは、無くても可"""
    gravDict = {}
    name = ""
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!DLOAD":
            ampName = pyFistr.getValName(header, "AMP")
            if ampName == "":
                ampCont = ""
            else:
                ampCont = ampDict[ampName]
            ws = pyFistr.deleteSp(data[0]).split(",")
            if ws[1] == "GRAV":
                if ws[0] != "" and ws[0].find("dummy") < 0:
                    name = ws[0]
                    gravDict[name] = [ws[2:], ampCont]
    #elementGroupIDを取得
    for name in gravDict.keys():
        if name == "ALL":
            grpID = "0"
        else:
            grpID = getGroupIdFromMeshConts(name, "GRBRIC")
        gravDict[name].append(grpID)
    return gravDict

#
#  getInitialVelocity
#---------------------
def getInitialVelocity(cntHeaderData):
    """ 初期速度を取得"""

    def convStr(iniVel):
        ans = list(map(str, iniVel))
        return ans

    iniVelDict = {}
    iniVel = [0,0,0]
    name = ""
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!VELOCITY":
            velType = pyFistr.getValName(header, "TYPE")
            if velType == "INITIAL":
                for line in data:
                    ws = pyFistr.deleteSp(line).split(",")
                    if name != ws[0]:
                        if iniVel != [0,0,0] and name.find("dummy") < 0:
                            iniVelDict[name] = [convStr(iniVel)]
                        name = ws[0]
                        iniVel = [0,0,0]
                    col = int(ws[1]) - 1
                    iniVel[col] = float(ws[3])
    if iniVel != [0,0,0] and name.find("dummy") < 0:
         iniVelDict[name] = [convStr(iniVel)]
    #nodeGrpIDを取得
    for name in iniVelDict.keys():
        grpID = getGroupIdFromMeshConts(name, "GRNOD")
        iniVelDict[name].append(grpID)
    return iniVelDict

#
#  getTransitVelocity
#---------------------
def getTransitVelocity(cntHeaderData, ampDict):
    """ trinsitVelocity（速度）を取得"""
    tranVelDict = {}
    name = ""
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!VELOCITY":
            velType = pyFistr.getValName(header, "TYPE")
            if velType != "INITIAL":
                for line in data:
                    ws = pyFistr.deleteSp(line).split(",")
                    if ws[0] != name:
                        if (not ws[0] in tranVelDict.keys()) and (ws[0].find("dummy") < 0):
                            ampName = pyFistr.getValName(header, "AMP")
                            if ampName == "":
                                ampName = "AMP100"
                            name = ws[0]
                            tranVelDict[name] = [[0, 0, 0], ampDict[ampName]]
                    col = int(ws[1]) - 1
                    tranVelDict[name][0][col] = float(ws[3])                    
    #nodeGrpIDを取得
    for name in tranVelDict.keys():
        grpID = getGroupIdFromMeshConts(name, "GRNOD")
        tranVelDict[name].append(grpID)
    return tranVelDict

#
#  getTransitAcceleration
#-------------------------
def getTransitAcceleration(cntHeaderData, ampDict):
    """ 加速度を取得"""
    tranAccDict = {}
    name = ""
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ACCELERATION":
            for line in data:
                ws = pyFistr.deleteSp(line).split(",")
                if ws[0] != name:
                    if (not ws[0] in tranAccDict.keys()) and (ws[0].find("dummy") < 0):
                        ampName = pyFistr.getValName(header, "AMP")
                        if ampName == "":
                            ampName = "AMP100"
                        name = ws[0]
                        tranAccDict[name] = [[0,0,0], ampDict[ampName]]
                col = int(ws[1]) - 1
                tranAccDict[name][0][col] = float(ws[3])
    #nodeGroupIDを取得
    for name in tranAccDict.keys():
        grpID = getGroupIdFromMeshConts(name, "GRNOD")
        tranAccDict[name].append(grpID)
    return tranAccDict    

#
#  getAlphaBeta
#----------------
def getAlphaBeta(cntHeaderData):
    """ rayleigh減衰αβを取得"""
    alphaBeta = []
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!DYNAMIC":
            line = data[3]
            ws = pyFistr.deleteSp(line).split(",")
            alpha = ws[2]
            beta = ws[3]
            if float(alpha) == 0.0 and float(beta) == 0.0:
                alphaBeta = []
            else:
                alphaBeta = [alpha, beta]
            break
    return alphaBeta

#
#  createRadiossStarterFile
#---------------------------
def createRadiossStarterFile(fistrFiles, radiossFiles, fistrDicts):
    """ radioss starterファイルを作成"""
    global radConts
    radConts = []
    [meshFile, cntFile, sphEgrp, fistrUnit] = fistrFiles
    [incFile, radFile, yearVersion, unit, interType] = radiossFiles
    [nodeDict, elmDict, ngrpDict, surfDict, egrpDict, 
     matDict, shellThDict, contactDict, nodeElmDict, ampDict] = fistrDicts
    #fistrのcntFileを取得
    f = open(cntFile, encoding="utf-8"); lines = f.readlines(); f.close()
    cntHeaderData = pyFistr.getHeaderData(lines)
    #ALLを使っているかどうか確認
    NallFlag = getAllNameFlag(cntHeaderData)
    #一定値のAMPを作成し、辞書に追加
    ampDict = addConstantAmpDict(cntHeaderData, ampDict)
    #BOUNDARY(0,0,0)を取得
    fixNameDict = getFixBoundary(cntHeaderData)
    #BOUNDARY 変位を取得
    dispNameDict = getDispBoundary(cntHeaderData, ampDict)
    #CLOADを取得（等分布荷重は、undefineDictに入る）
    cloadDict, undefineDict = getCload(cntHeaderData, ampDict)
    #DLOADを取得
    dloadDict = getDload(cntHeaderData, ampDict)
    #GRAVを取得
    gravDict = getGrav(cntHeaderData, ampDict)
    #iniVelを取得
    iniVelDict = getInitialVelocity(cntHeaderData)
    #tranVelocityを取得
    tranVelDict = getTransitVelocity(cntHeaderData, ampDict)
    #tranAccelerationを取得
    tranAccDict = getTransitAcceleration(cntHeaderData, ampDict)
    #damping（減衰）を取得
    alphaBeta = getAlphaBeta(cntHeaderData)
    #incMesh内容を修正
    if len(undefineDict) > 0:
        #meshFile再作成（等分布荷重の場合）
        ngrpDict.update(undefineDict)
        incLines = remakeIncMeshCont(ngrpDict)
        f = open(incFile, "w", encoding="utf-8"); f.writelines(incLines); f.close()
        #cloadDict再作成
        cloadDict = remakeCloadDict(ngrpDict, cloadDict)
    #radFileを作成
    lines  = createRadHeaderLines(cntFile, meshFile)
    lines += createBeginUnitsLines(yearVersion, unit, radFile)
    lines += createMaterialLines(matDict)
    lines += createIncludeMeshLines(incFile)
    lines += createPropertyLines(fistrDicts)
    lines += createPartLines()
    lines += createAllGrpLines(NallFlag, alphaBeta, gravDict)
    if len(alphaBeta) > 0:
        #lines += createAllNodesGroup()
        lines += createDamperLines(alphaBeta)
    lines += createFixBoundaryLines(fixNameDict)
    lines += createDispBoundaryLines(dispNameDict)
    lines += createCloadLines(cloadDict)
    lines += createDloadLines(dloadDict)
    lines += createGravLines(gravDict)
    lines += createInitialVelocityLines(iniVelDict)
    lines += createTransitVelocityLines(tranVelDict)
    lines += createTransitAccelerationLines(tranAccDict)
    lines += createContactLines(cntHeaderData, fistrDicts, interType)
    lines += createRadFooter()
    #fileに出力
    f = open(radFile, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  remakeIncMeshCont
#---------------------
def remakeIncMeshCont(ngrpDict):
    """ 未定義のngrpを定義し直す"""
    global incLines, meshConts
    
    def skipBack(i, incLines):
        """ 最初の#までさかのぼる"""
        si = -1
        while i >= 0:
            i -= 1
            if incLines[i][0] != "#":
                si = i + 1
                break
        return si

    #meshContsを修正
    si = len(meshConts)
    for i in range(len(meshConts)):
        line = meshConts[i]
        if line[:len("#   /GRNOD/")] == "#   /GRNOD/":
            si = i
            break
    ei = len(meshConts)
    for i in range(si, len(meshConts), 1):
        line = meshConts[i]
        if line[:len("#   /GRNOD/")] != "#   /GRNOD/":
            ei = i
            break
    footer = meshConts[ei:]
    meshConts = meshConts[:si]
    lines = createGroupNodeLines(ngrpDict)
    meshConts += footer

    #meshLinesを修正
    for i in range(len(incLines)):
        line = incLines[i]
        if (line[:len("/GRNOD/")] == "/GRNOD/"
            or line[:len("/SURF/")] == "/SURF/"
            or line[:len("/END")] == "/END"):
            si = skipBack(i, incLines)
            break
    for i in range(si, len(incLines), 1):
        line = incLines[i]
        if line[0] == "/":
            if line[:len("/GRNOD/")] != "/GRNOD/":
                ei = skipBack(i, incLines)
                break
    #ngrp行を作成
    header = incLines[:si]
    footer = incLines[ei:]
    newLines = header + lines + footer
    for i in range(len(newLines)):
        line = newLines[i]
        if line[:len("#   /")] == "#   /":
            si = i
            break
    for i in range(si, len(newLines), 1):
        line = newLines[i]
        if line[:len("#   /")] != "#   /":
            ei = i
            break
    newLines = newLines[:si] + meshConts + newLines[ei:]
    return newLines

#
#  remakeCloadDict
#-------------------
def remakeCloadDict(ngrpDict, cloadDict):
    """ cloadDictにgrpIDを追記する"""
    names = list(ngrpDict.keys())
    names.sort()
    grpId = 1
    for name in names:
        if name in cloadDict.keys():
            if cloadDict[name][-1] == "":
                cloadDict[name][-1] = str(grpId)
        grpId += 1
    return cloadDict

#
#  createRadHeaderLines
#-----------------------
def createRadHeaderLines(cntFile, meshFile):
    """ radFileのheaderを作成"""
    now = datetime.datetime.now()
    dateList = [str(now.year), str(now.month), str(now.day)]
    addLines  = ["#RADIOSS STARTER" + ls]
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  generated by fistr2rad.py" + ls]
    addLines += ["#           from " + meshFile + " and " + cntFile + ls]
    addLines += ["#                                                " + "/".join(dateList) + ls]
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    return addLines

#
#  createBeginUnitsLines
#------------------------
def createBeginUnitsLines(yearVersion, unit, radFile):
    global radConts
    runName = "_".join(radFile.split("_")[:-1])
    addLines  = ["/BEGIN" + ls]
    addLines += [runName + ls]
    addLines += [("%10s" % yearVersion) + ls]
    addLines += [("%20s" % unit[0]) + ("%20s" % unit[1]) + ("%20s" % unit[2]) + ls]
    addLines += [("%20s" % unit[0]) + ("%20s" % unit[1]) + ("%20s" % unit[2]) + ls]
    radConts.append("/BEGIN  " + yearVersion + "  unit: " + ",".join(unit) + ls)
    return addLines

#
#  createMaterialLines
#-----------------------
def createMaterialLines(matDict):
    """ 材料物性値を作成"""
    global radConts
    egrpNames = list(matDict.keys())
    if len(egrpNames) == 0:
        return []
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  MATERIAL define" + ls]
    egrpNames.sort()
    matNo = 1
    matFuncNo = getFunctionNoFromRadConts()
    for egrp in egrpNames:
        matName, E, nu, rho, alpha, plastics = matDict[egrp]
        matYield, matHarden = getPlasticYieldHarden(plastics)
        if matYield == "":
            #弾性材料を作成
            addLines += createElasticMaterialLines(matNo, egrp, matDict)
        elif matYield == "MISES" and matHarden == "MULTILINEAR":
            #mises, multilinearの行を作成
            addLines += createPlasticMisesMultilinearLines(matNo, egrp, matDict, matFuncNo)
            addLines += createMultilinearDataLines(egrp, matDict, matFuncNo)
            matFuncNo += 1
        matNo += 1
    return addLines

#
#  getPlasticYieldHarden
def getPlasticYieldHarden(plastics):
    """ !PLASTICの内容からyield、hardenを取得して返す"""
    if len(plastics) == 0:
        return "", ""
    header = plastics[0]
    matYield = pyFistr.getValName(header, "YIELD")
    matHarden = pyFistr.getValName(header, "HARDEN")
    return matYield, matHarden

#
#  createElasticMaterialLines
def createElasticMaterialLines(matNo, egrp, matDict):
    """ 弾性材料の設定行を作成する"""
    global radConts
    matName, E, nu, rho, alpha, plastics = matDict[egrp]
    E, nu, rho = checkStrLength20([E, nu, rho])
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  MAT-" + str(matNo) + "  ElementGroup " + egrp + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/MAT/PLAS_JOHNS/" + str(matNo) + ls]
    addLines += [matName + " of " + egrp + ls]
    addLines += ["#                rho" + ls]
    addLines += [("%20s" % rho) + ls]
    addLines += ["#                  E                  nu     Iflag" + ls]
    addLines += [("%20s" % E) + ("%20s" % nu) + ("%10d" % 1) + ls]
    addLines += ["#            sigma_y                 UTS              epsUTS           eps_p_max            sigmaMax" + ls]
    addLines += [("%20s" % "1e30") + ("%20s" % "2e30") + ls]
    addLines += ["#                  c             eps_DOT       ICC   Fsmooth               F_cut               Chard" + ls]
    addLines += [("%20s" % "0") + ("%20s" % "0") + ("%10s" % "0") + ("%10s" % "0") + ("%20s" % "0") + ("%20s" % "0") + ls]
    addLines += ["#                  m              T_melt              rhoC_P                 T_r" + ls]
    addLines += [("%20s" % "0") + ("%20s" % "0") + ("%20s" % "0") + ("%20s" % "0") + ls]
    radConts.append("/MAT/PLAS_JOHNS/" + str(matNo) + "  ElementGroup " + egrp + ls)
    return addLines

#
#  createPlasticMisesMultilinearLines
def createPlasticMisesMultilinearLines(matNo, egrp, matDict, matFuncNo):
    """ 塑性材料（mises, multiliear)の行を作成"""
    global radConts
    matName, E, nu, rho, alpha, plastics = matDict[egrp]
    E, nu, rho = checkStrLength20([E, nu, rho])
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  MAT-" + str(matNo) + "  ElementGroup " + egrp + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/MAT/PLAS_TAB/" + str(matNo) + ls]
    addLines += [matName + " of " + egrp + ls]
    addLines += ["#                rho" + ls]
    addLines += [("%20s" % rho) + ls]
    addLines += ["#                  E                  nu           eps_p_max               eps_t               eps_m" + ls]
    addLines += [("%20s" % E) + ("%20s" % nu) + ls]
    addLines += ["#    nFunc   Fsmooth              C_hard                Fcut               eps_f                  VP" + ls]
    addLines += ["         1         0                   0" + ls]
    addLines += ["#    Ipfun              Fscale" + ls]
    addLines += ["         0                 1.0" + ls]
    addLines += ["#  FunctID" + ls]
    addLines += [("%10d" % matFuncNo) + ls]   #関数No定義
    addLines += ["#           FscaleID" + ls]
    addLines += ["                 1.0" + ls]
    addLines += ["#          eps_rates" + ls]
    addLines += ["                 0.0" + ls]
    radConts.append("/MAT/PLAS_TAB/" + str(matNo) + "  ElementGroup " + egrp + ls)
    return addLines

#
#  createMultilinearDataLines
def createMultilinearDataLines(egrp, matDict, matFuncNo):
    """ multilinearFunctionDataを作成"""
    global radConts
    matName, E, nu, rho, alpha, plastics = matDict[egrp]
    data = plastics[1]
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  MAT_FUNC-" + str(matFuncNo) + "  ElementGroup " + egrp + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/FUNCT/" + str(matFuncNo) + ls]
    addLines += ["MULTILINEAR function" + ls]
    addLines += ["#   plastic strain:x      yield stress:y" + ls]
    for line in data:
        words = pyFistr.deleteSp(line).split(",")
        stress, strain = words[:2]
        addLines += [("%20s" % strain) + ("%20s" % stress) + ls]
    radConts.append("/FUNCT/" + str(matFuncNo) + "  ElementGroup " + egrp + ls)
    return addLines

#
#  createIncludeMeshLines
#-------------------------
def createIncludeMeshLines(incFile):
    """ include行を作成"""
    global meshConts, radConts
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  convert mesh from Fistr to Radioss format" + ls]
    addLines += ["#" + ls]
    addLines += ["#     mesh contents" + ls]
    addLines += meshConts
    addLines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
    addLines += ["#include " + incFile + ls]
    addLines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
    addLines += ["#" + ls]
    radConts.append("#include " + incFile + ls)
    return addLines

#
#  createPropertyLines
#----------------------
def createPropertyLines(fistrDicts):
    """ 要素のpropertyを作成する"""
    global meshConts, radConts
    [nodeDict, elmDict, ngrpDict, surfDict, egrpDict, 
     matDict, shellThDict, contactDict, nodeElmDict, ampDict] = fistrDicts
    #要素Typeを取得
    egrpElmDict = {}
    for line in meshConts:
        words = line.split()
        if words[2] == "Elements":
            elmType = words[1].split("/")[1]
            egrp = words[-1]
            if not egrp in egrpElmDict.keys():
                egrpElmDict[egrp] = []
            egrpElmDict[egrp].append(elmType)
    egrps = list(egrpElmDict.keys())
    if len(egrps) == 0:
        return []
    egrps.sort()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  PROPERTY define" + ls]
    propNo = 1
    #solidを確認
    for egrp in egrps:
        elmNames = egrpElmDict[egrp]
        for elm in elmNames:
            #solid部を作成
            if elmTypeDict[elm] == "solid":
                addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
                addLines += ["#  PROP-" + str(propNo) + "  ElementGroup " + egrp + ls]
                addLines += ["#--------.---------.---------.---------." + ls]
                addLines += ["/PROP/SOLID/" + str(propNo) + ls]
                addLines += ["solid prperty  ElementGroup " + egrp + ls]
                addLines += ["#   Isolid    Ismstr               Icpre  Itetra10     Inpts   Itetra4    Iframe                  dn" + ls]
                if elm == "BRICK":
                    addLines += ["         1" + ls]
                elif elm == "TETRA4":
                    addLines += ["                                                                  1000" + ls]
                elif elm == "BRIC20":
                    addLines += ["        16" + ls]
                elif elm == "TETRA10":
                    addLines += ["                                                 1" + ls]
                addLines += ["#                 qa                  qb                   h            lambda_v               myu_v" + ls]
                addLines += [ls]
                addLines += ["#              dTmin" + ls]
                addLines += [ls]
                addLines += ["#     Ndir sphpartID" + ls]
                addLines += [ls]
                radConts.append("/PROP/SOLID/" + str(propNo) + "  " + elm + "  ElementGroup " + egrp + ls)
                propNo += 1
            #shellを作成
            elif elmTypeDict[elm] == "shell":
                th, N = shellThDict[egrp]
                [th] = checkStrLength10([th])
                addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
                addLines += ["#  PROP-" + str(propNo) + "  ElementGroup " + egrp + ls]
                addLines += ["#--------.---------.---------.---------." + ls]
                addLines += ["/PROP/SHELL/" + str(propNo) + ls]
                addLines += ["shell prperty  ElementGroup " + egrp + ls]
                elmNames = egrpElmDict[egrp]
                if elm == "SHELL":
                    addLines += ["#   Ishell    Ismstr     Ish3n    Idrill                            P_Thick_Fail" + ls]
                    addLines += ["         2" + ls]
                    addLines += ["#                 Hm                  Hf                  Hr                  Dm                  Dn" + ls]
                    addLines += [ls]
                    addLines += ["#        N   Istrain               Thick              Ashear              Ithick     Iplas" + ls]
                    addLines += [("%10s" % N) + (" " * 20) + ("%10s" % th) + ls]
                elif elm == "SH3N":
                    addLines += ["#   Ishell    Ismstr     Ish3n    Idrill                            P_Thick_Fail" + ls]
                    addLines += ["                             2" + ls]
                    addLines += ["#                 Hm                  Hf                  Hr                  Dm                  Dn" + ls]
                    addLines += [ls]
                    addLines += ["#        N   Istrain               Thick              Ashear              Ithick     Iplas" + ls]
                    addLines += [("%10s" % N) + (" " * 20) + ("%10s" % th) + ls]
                radConts.append("/PROP/SHELL/" + str(propNo) + "  " + elm + "  ElementGroup " + egrp + ls)
                propNo += 1
            elif elmTypeDict[elm] == "sph":
                massP, elmSize = getPointMassElementSizeOfSph(egrp, fistrDicts)
                massP, elmSize = checkStrLength20([str(massP), str(elmSize)])
                addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
                addLines += ["#  PROP-" + str(propNo) + "  ElementGroup " + egrp + ls]
                addLines += ["#--------.---------.---------.---------." + ls]
                addLines += ["#  mp: set calculated element mass from cell volume and density" + ls]
                addLines += ["#   h: set element size from FrontISTR" + ls]
                addLines += ["/PROP/TYPE34/" + str(propNo) + ls]
                addLines += ["SPH property  ElementGroup " + egrp + ls]
                addLines += ["#                 mp                  qa                  qb            alpha_cs   skew_ID      h_ID" + ls]
                addLines += [("%20s" % massP) + "                   2                   1                   0         0         0" + ls]
                addLines += ["#    order                   h             xi_stab" + ls]
                addLines += ["         0" + ("%20s" % elmSize) + "                   0" + ls]
                radConts.append("/PROP/TYPE34/" + str(propNo) + "  " + elm + "  ElementGroup " + egrp + ls)
                propNo += 1
    return addLines

#
#  getPointMassElementSizeOfSph
def getPointMassElementSizeOfSph(egrp, fistrDicts):
    """ egrpの要素の質量と要素サイズを取得して返す"""
    global meshConts
    [nodeDict, elmDict, ngrpDict, surfDict, egrpDict, 
     matDict, shellThDict, contactDict, nodeElmDict, ampDict] = fistrDicts
    #最初の要素を取得
    elmNo = list(egrpDict[egrp])[0]
    #node取得
    nodeNos, elmType = elmDict[elmNo]
    #要素typeを取得
    elmType = str(elmType)
    eType = cm.elmContsDict[elmType][0]
    #egrpの体積を算出
    V = 0.0
    for elmNo in egrpDict[egrp]:
        nodeNos = elmDict[elmNo][0]
        elmName = elmDict[elmNo][1]
        elmConts = cm.elmContsDict[str(elmName)]
        locs = list(map(lambda x: nodeDict[x], nodeNos))
        if elmConts[2] == "tri":
            a = geom.triA(locs)
            t = shellThDict[egrp][0]
            v = float(a) * float(t)
        elif elmConts[2] == "quad":
            a, g = geom.polygonAG(locs)
            t = shellThDict[egrp][0]
            v = float(a) * float(t)
        elif elmConts[2] == "tet":
            v = geom.tetV(locs)
        elif elmConts[2] == "hex":
            v = geom.hexV(locs)
        V += v
    #粒子数を取得
    for line in meshConts:
        words = line.split()
        if words[1][:len("/SPHCEL/")] == "/SPHCEL/":
            if words[-1] == egrp:
                nSph = int(words[3])
                break
    #粒子1個の質量を取得
    rho = float(matDict[egrp][3])
    massP = V * rho / nSph
    #要素sizeを取得
    size = geom.length(locs[0], locs[1])
    return massP, size

#
#  createPartLines
#------------------
def createPartLines():
    """ PARTを作成"""
    global meshConts, radConts, allPartNos
    #partIDを取得
    egrpPartNoElms = []
    elmNames = elmTypeDict.keys()
    for line in meshConts:
        words = line.split()
        ws = words[1].split("/")
        elmName = ws[1]
        if elmName in elmNames:
            partNo = ws[2]
            egrp = words[-1]
            egrpPartNoElms.append([egrp, partNo, elmName])
    if len(egrpPartNoElms) == 0:
        return []
    egrpPartNoElms.sort()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  PART define" + ls]
    for egrp, partNo, elmName in egrpPartNoElms:
        propNo = getPropertyNoFromRadConts(egrp, elmName)
        matNo = getMaterialNoFromRadConts(egrp)
        addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
        addLines += ["#  PART-" + str(partNo) + "  " + elmName + "  ElementGroup " + egrp + ls]
        addLines += ["#--------.---------.---------.---------." + ls]
        addLines += ["/PART/" + partNo + ls]
        addLines += [elmName + ls]
        addLines += ["#  prop_ID    mat_ID" + ls]
        addLines += [("%10s" % propNo) + ("%10s" % matNo) + ls]
        radConts.append("/PART/" + partNo + "  " + elmName + "  ElementGroup " + egrp + ls)
        allPartNos.append(partNo)
    return addLines

#
#  createAllGrpLines
#--------------------
def createAllGrpLines(NallFlag, alphaBeta, gravDict):
    """ ngrp:ALL, egrp:ALLを作成する"""
    global allPartNos, latestNodeId, radConts
    addLines = []
    flag = "ALL" in gravDict.keys()
    if NallFlag == 1 or len(alphaBeta) > 0 or flag == True:
        #ngrp:ALLを作成する
        latestNodeId += 1
        ngrpNo = latestNodeId
        addLines += ["#===================================================================================================" + ls]
        addLines += ["#  NodeGroup of all parts define" + ls]
        addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
        addLines += ["#  groupNode: grNodeID=" + str(ngrpNo) + "  ALL" + ls]
        addLines += ["#--------.---------.---------.---------." + ls]
        addLines += ["/GRNOD/PART/" + str(ngrpNo) + ls]
        addLines += ["nodeGroup of all parts" + ls]
        for i in range(0, len(allPartNos), 10):
            partNos = allPartNos[i:i+10]
            line = ""
            for partNo in partNos:
                line += ("%10s" % partNo)
            addLines += [line + ls]
        radConts.append("/GRNOD/PART/" + str(ngrpNo) + "  GRNOD of ALL" + ls)
    return addLines

# #
# #  createAllNodesGroup
# #----------------------
# def createAllNodesGroup():
#     """ AllNodesのnodeGroupを作成する。"""
#     global allPartNos, latestNodeId, radConts
#     latestNodeId += 1
#     ngrpNo = latestNodeId
#     addLines = []
#     addLines += ["#===================================================================================================" + ls]
#     addLines += ["#  NodeGroup of all parts define" + ls]
#     addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
#     addLines += ["#  groupNode: grNodeID=" + str(ngrpNo) + "  allParts" + ls]
#     addLines += ["#--------.---------.---------.---------." + ls]
#     addLines += ["/GRNOD/PART/" + str(ngrpNo) + ls]
#     addLines += ["nodeGroup of all parts" + ls]
#     for i in range(0, len(allPartNos), 10):
#         partNos = allPartNos[i:i+10]
#         line = ""
#         for partNo in partNos:
#             line += ("%10s" % partNo)
#         addLines += [line + ls]
#     radConts.append("/GRNOD/PART/" + str(ngrpNo) + "  GRNOD of all parts" + ls)
#     return addLines

#
#  createDamperLines
#---------------------
def createDamperLines(alphaBeta):
    """ DAMP（減衰）行を作成"""
    global latestNodeId, radConts
    if len(alphaBeta) == 0:
        return []
    alpha, beta = alphaBeta
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  Rayleigh dampng define" + ls]
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  alpha and beta of rayleigh dampng" + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/DAMP/1" + ls]
    addLines += ["damping of all parts" + ls]
    addLines += ["#              alpha                beta    grndID    skewID              Tstart               Tstop" + ls]
    addLines += [("%20s" % alpha) + ("%20s" % beta) + ("%10d" % latestNodeId) + ("%10s" % "0") + ("%20s" % "0.0") + ls]
    radConts.append("/DAMP/1" + "  rayleigh damping of all parts" + ls)
    latestNodeId += 1

    return addLines

#
#  createFixBoundaryLines
#-------------------------
def createFixBoundaryLines(fixNameDict):
    """ BOUNDARY(0,0,0)の行を作成"""
    global radConts
    names = list(fixNameDict.keys())
    if len(names) == 0:
        return []
    names.sort()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  BOUNDARY conditions" + ls]
    bcNo = 1
    for name in names:
        if name == "ALL":
            grpID = getAllGrpID()
        else:
            grpID = fixNameDict[name][1]
        addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
        addLines += ["#  BC-" + str(bcNo) + "  grpName " + name + ls]
        addLines += ["#--------.---------.---------.---------." + ls]
        addLines += ["/BCS/" + str(bcNo) + ls]
        addLines += ["nodeGroup " + name + ls]
        addLines += ["#  Tra rot   skew_ID  grnod_ID" + ls]
        fix = fixNameDict[name][0]
        addLines += ["   " + fix + " 000" + ("%10s" % "0") + ("%10s" % grpID) + ls]
        radConts.append("/BCS/" + str(bcNo) + "  nodeGroup " + name + ls)
        bcNo += 1
    return addLines

#
#  createDispBoundaryLines
#--------------------------
def createDispBoundaryLines(dispNameDict):
    """ BOUNDARY 変位の行を作成"""
    global radConts
    names = list(dispNameDict.keys())
    if len(names) == 0:
        return []
    names.sort()
    funcNo = getFunctionNoFromRadConts()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  IMPDISP conditions" + ls]
    dispNo = 1
    for name in names:
        addLines += createDispBoundaryNgrpLines(dispNameDict, name, dispNo, funcNo)
        addLines += createDispBoundaryFuncLines(dispNameDict, name, funcNo)
        funcNo += 1
        dispNo += 1
    return addLines

#
#  createDispBoundaryNgrpLines
#-------------------------------
def createDispBoundaryNgrpLines(dispNameDict, name, dispNo, funcNo):
    """ NGRPのIMPDISP行を作成"""
    global radConts
    #変位の方向を取得
    vals = dispNameDict[name][0]
    dispDir, disp = getValueDirection(vals)
    #grpIDを取得
    if name == "ALL":
        grpID = getAllGrpID()
    else:
        grpID = dispNameDict[name][-1]
    #IMPDISP行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  IMPDISP-" + str(dispNo) + "  grpName " + name + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/IMPDISP/" + str(dispNo) + ls]
    addLines += ["nodeGroup " + name + ls]
    addLines += ["#   funcID       DIR    skewID    sensID    grndID     Frame     Icoor" + ls]
    addLines += [("%10d" % funcNo) + ("%10s" % dispDir) + ("%10d" % 0) + ("%10d" % 0) + ("%10s" % grpID) + ("%10d" % 0) + ("%10d" % 0) + ls]
    addLines += ["#            Tscalex             Fscaley              Tstart               Tstop" + ls]
    addLines += [("%20s" % "1.0") + ("%20s" % "1.0") + ("%20s" % "0.0") + ls]
    radConts.append("/IMPDISP/" + str(dispNo) + " dispDir:" + dispDir + "  NodeGroup " + name + ls)
    return addLines

#
#  createDispBoundaryFuncLines
def createDispBoundaryFuncLines(dispNameDict, name, funcNo):
    """ IMPDISPのfunctionDataを作成"""
    global radConts
    #functionDataを取得（絶対値で取得）
    vals = dispNameDict[name][0]
    _dir, baseVal = getValueDirection(vals)
    ampData = dispNameDict[name][1]
    funcData = getFunctionData(baseVal, ampData)
    #func行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  IMPDISP_FUNC-" + str(funcNo) + "  NodeGroup " + name + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/FUNCT/" + str(funcNo) + ls]
    addLines += ["IMPDISP function" + ls]
    addLines += ["#             time:x              disp:y" + ls]
    for x, y in funcData:
        x, y = checkStrLength20([str(x), str(y)])
        addLines += [("%20s" % x) + ("%20s" % y) + ls]
    radConts.append("/FUNCT/" + str(funcNo) + "  NodeGroup " + name + ls)
    return addLines

#
#  createCloadLines
#-------------------
def createCloadLines(cloadDict):
    """ CLOADの行を作成"""
    global radConts
    names = list(cloadDict.keys())
    if len(names) == 0:
        return []
    names.sort()
    funcNo = getFunctionNoFromRadConts()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  CLOAD conditions" + ls]
    cloadNo = 1
    for name in names:
        addLines += createCloadNgrpLines(cloadDict, name, cloadNo, funcNo)
        addLines += createCloadFunctionLines(cloadDict, name, funcNo)
        cloadNo += 1
        funcNo += 1
    return addLines

#
#  createCloadNgrpLines
def createCloadNgrpLines(cloadDict, ngrp, cloadNo, funcNo):
    """ NGRPのCLOAD行を作成"""
    global radConts
    #荷重の方向を取得
    vals = cloadDict[ngrp][0]
    forceDir, force = getValueDirection(vals)
    #grndIDを取得
    if ngrp == "ALL":
        grpID = getAllGrpID()
    else:
        grpID = cloadDict[ngrp][-1]
    #cload行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  CLOAD-" + str(cloadNo) + "  NodeGroup " + ngrp + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/CLOAD/" + str(cloadNo) + ls]
    addLines += ["nodeGroup " + ngrp + ls]
    addLines += ["#   funcID  forceDir    skewID    sensID    grndID                       Tscalex             Fscaley" + ls]
    addLines += [("%10d" % funcNo) + ("%10s" % forceDir) + ("%10d" % 0) + ("%10d" % 0) + ("%10s" % grpID) + ls]
    radConts.append("/CLOAD/" + str(cloadNo) + " forceDir:" + forceDir + "  NodeGroup " + ngrp + ls)
    return addLines

#
#  createCloadFunctionLines
def createCloadFunctionLines(cloadDict, ngrp, funcNo):
    """ CLOADのfunctionDataを作成"""
    global radConts
    #functionDataを取得(絶対値で取得)
    vals = cloadDict[ngrp][0]
    _dir, baseVal = getValueDirection(vals)
    ampData = cloadDict[ngrp][1]
    funcData = getFunctionData(baseVal, ampData)
    #funct行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  CLOAD_FUNC-" + str(funcNo) + "  NodeGroup " + ngrp + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/FUNCT/" + str(funcNo) + ls]
    addLines += ["CLOAD function" + ls]
    addLines += ["#             time:x             force:y" + ls]
    for x, y in funcData:
        x, y = checkStrLength20([str(x), str(y)])
        addLines += [("%20s" % x) + ("%20s" % y) + ls]
    radConts.append("/FUNCT/" + str(funcNo) + "  NodeGroup " + ngrp + ls)
    return addLines

#
#  createDloadLines
#--------------------
def createDloadLines(dloadDict):
    """ DLOADの行を作成"""
    global radConts
    names = list(dloadDict.keys())
    if len(names) == 0:
        return []
    names.sort()
    funcNo = getFunctionNoFromRadConts()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  PLOAD conditions" + ls]
    dloadNo = 1
    for name in names:
        addLines += createDloadSgrpLines(dloadDict, name, dloadNo, funcNo)
        addLines += createDloadFunctionLines(dloadDict, name, funcNo)
        funcNo += 1
        dloadNo += 1
    return addLines

#
#  createDloadSgrpLines
def createDloadSgrpLines(dloadDict, sgrp, dloadNo, funcNo):
    """ SGRPのDLOAD行を作成"""
    global radConts
    #grpIDを取得
    grpID = dloadDict[sgrp][-1]
    #dload行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  PLOAD-" + str(dloadNo) + "  SurfaceGroup " + sgrp + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/PLOAD/" + str(dloadNo) + ls]
    addLines += ["surfaceGroup " + sgrp + ls]
    addLines += ["#   surfID    funcID    sensID    Ipinch      Idel                       Tscalex             Fscaley" + ls]
    addLines += [("%10s" % grpID) + ("%10d" % funcNo) + ("%10d" % 0) + ls]
    radConts.append("/PLOAD/" + str(dloadNo) + "  SurfaceGroup " + sgrp + ls)
    return addLines

#
#  createDloadFunctionLines
def createDloadFunctionLines(dloadDict, sgrp, funcNo):
    """ DLOADのfunctionDataを作成"""
    global radConts
    #functionDataを取得（絶対値で取得）
    #  圧力の方向は、faceの向きと反対方向）
    baseVal = float(dloadDict[sgrp][0]) * (-1.0)
    ampData = dloadDict[sgrp][1]
    funcData = getFunctionData(baseVal, ampData)
    #func行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  PLOAD_FUNC-" + str(funcNo) + "  SuefaceGroup " + sgrp + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/FUNCT/" + str(funcNo) + ls]
    addLines += ["PLOAD function" + ls]
    addLines += ["#             time:x             force:y" + ls]
    for x, y in funcData:
        x, y = checkStrLength20([str(x), str(y)])
        addLines += [("%20s" % x) + ("%20s" % y) + ls]
    radConts.append("/FUNCT/" + str(funcNo) + "  SurfaceGroup " + sgrp + ls)
    return addLines

#
#  createGravLines
#-------------------
def createGravLines(gravDict):
    """ GRAV行を作成。
    時間変化は無しでもOK。
    egrpとしてALL以外を選択した場合、「/GRNOD/PART/」が作成される。
    """
    global radConts, latestNodeId
    names = list(gravDict.keys())
    if len(names) == 0:
        return []
    names.sort()
    funcNo = getFunctionNoFromRadConts()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  GRAV conditions" + ls]
    gravNo = 1
    for name in names:
        grpID = gravDict[name][-1]
        if int(grpID) != 0:
            #grpNoを作り直す
            #grpID = getNewNodeGroupIdFromMeshRadConts()
            latestNodeId += 1
            #print("----", latestNodeId)
            grpID = latestNodeId
            gravDict[name][-1] = grpID
        # else:
        #     #"ALL"の場合
        #     grpID = getAllGrpID()
        if gravDict[name][1] == "":
            #AMPLITUDEを設定しない場合（funcNo = 0）
            addLines += createGravEgrpLines(gravDict, name, gravNo, 0)
        else:
            addLines += createGravEgrpLines(gravDict, name, gravNo, funcNo)
            addLines += createGravFunctionLines(gravDict, name, funcNo)
            funcNo += 1
        if int(grpID) != 0:
            #PART(EGRP)のnodeGroupを作成
            addLines += createNodeGroupOfPartNo(name, grpID)
        gravNo += 1
    return addLines

#
#  createGravEgrpLines
def createGravEgrpLines(gravDict, egrp, gravNo, funcNo):
    """ EGRPのGRAV行を作成"""
    global radConts
    #grpIDを取得
    grpID = str(gravDict[egrp][-1])
    if int(grpID) == 0:
        #grp:ALLのgrpIDを取得する
        grpID = getAllGrpID()
    #GRAVの値と方向を取得
    grav = gravDict[egrp][0][0]
    #  方向を取得
    vals = gravDict[egrp][0][1:]
    vals = list(map(float, vals[:]))
    dirAns, val = getValueDirection(vals)
    #値を再計算
    grav = str(float(grav) * val)
    #Fscaleyの値を取得
    if funcNo == 0:
        scaleY = grav
    else:
        scaleY = "1.0"
    [scaleY] = checkStrLength20([scaleY])
    #grav行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  GRAV-" + str(gravNo) + "  nodeGroup of ElementGroup " + egrp + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/GRAV/" + str(gravNo) + ls]
    addLines += ["nodeGroup of elementGroup " + egrp + ls]
    addLines += ["#   funcID       dir    skewID    sensID    grndID                       Tscalex             Fscaley" + ls]
    addLines += [("%10d" % funcNo) + ("%10s" % dirAns) + ("%10d" % 0) + ("%10d" % 0) + ("%10s" % grpID) + "          " + ("%20s" % "1.0") + ("%20s" % scaleY) + ls]
    radConts.append("/GRAV/" + str(gravNo) + "  nodeGroup of ElementGroup " + egrp + ls)
    return addLines

#
#  createGravFunctionLines
def createGravFunctionLines(gravDict, name, funcNo):
    """ GRAVのfunctionDataを作成する"""
    global radConts
    #方向の正負を取得
    vals = gravDict[name][0][1:]
    vals = list(map(float, vals[:]))
    _dir, baseVal = getValueDirection(vals)
    gravVal = float(gravDict[name][0][0])
    baseVal = baseVal * gravVal
    ampData = gravDict[name][1]
    funcData = getFunctionData(baseVal, ampData)
    #func行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  GRAV_FUNC-" + str(funcNo) + "  nodeGroup of ElementGroup " + name + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/FUNCT/" + str(funcNo) + ls]
    addLines += ["GRAV function" + ls]
    addLines += ["#             time:x           gravity:y" + ls]
    for x, y in funcData:
        x, y = checkStrLength20([str(x), str(y)])
        addLines += [("%20s" % x) + ("%20s" % y) + ls]
    radConts.append("/FUNCT/" + str(funcNo) + "  nodeGroup of ElementGroup " + name + ls)
    return addLines

#
#  createNodeGroupOfPartNo
def createNodeGroupOfPartNo(egrp, grpID):
    """ PART(egrp)のnodeGroupを作成する。"""
    global radConts
    partID = getGroupIdFromRadConts(egrp, "PART")
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  GRNOD-" + str(grpID) + "  nodeGroup of ElementGroup " + egrp + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/GRNOD/PART/" + str(grpID) + ls]
    addLines += ["nodeGroup of ElementGroup " + egrp + ls]
    addLines += [("%10s" % partID) + ls]
    radConts.append("/GRNOD/PART/" + str(grpID) + "  nodeGroup of ElementGroup " + egrp + ls)
    return addLines

#
#  createInitialVelocityLines
#-----------------------------
def createInitialVelocityLines(iniVelDict):
    """ 初速を設定"""
    global radConts
    names = list(iniVelDict.keys())
    if len(names) == 0:
        return []
    names.sort()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  INITIAL VELOCITY" + ls]
    velNo = 1
    for name in names:
        if name == "ALL":
            grpID = getAllGrpID()
        else:
            grpID = iniVelDict[name][1]
        vx, vy, vz = iniVelDict[name][0]
        vx, vy, vz = checkStrLength20([vx, vy, vz])
        addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
        addLines += ["#  INIVEL-" + grpID + "  grpName " + name + ls]
        addLines += ["#--------.---------.---------.---------." + ls]
        addLines += ["/INIVEL/TRA/" + str(velNo) + ls]
        addLines += ["nodeGroup " + name + ls]
        addLines += ["#                 Vx                  Vy                  Vz   grnd_ID   skew_id" + ls]
        addLines += [("%20s" % vx) + ("%20s" % vy) + ("%20s" % vz) + ("%10s" % grpID) + ls]
        radConts.append("/INIVEL/TRA/" + str(velNo) + "  nodeGroup " + name + ls)
        velNo += 1
    return addLines

#
#  createTransitVelocityLines
#------------------------------
def createTransitVelocityLines(tranVelDict):
    """ transitVelocity(速度)を設定"""
    global radConts
    names = list(tranVelDict.keys())
    if len(names) == 0:
        return []
    names.sort()
    funcNo = getFunctionNoFromRadConts()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  IMPVEL conditions" + ls]
    velNo = 1
    for name in names:
        addLines += createTransitVelocityNgrpLines(tranVelDict, name, velNo, funcNo)
        addLines += createTransitVelocityFuncLines(tranVelDict, name, funcNo)
        funcNo += 1
        velNo += 1
    return addLines

#
#  createTransitVelocityNgrpLines
def createTransitVelocityNgrpLines(tranVelDict, name, velNo, funcNo):
    """ NGRPのIMPVEL行を作成"""
    global radConts
    #速度の方向を取得
    vals = tranVelDict[name][0]
    velDir, vel = getValueDirection(vals)
    #grpIDを取得
    if name == "ALL":
        grpID = getAllGrpID()
    else:
        grpID = tranVelDict[name][-1]
    #IMPVEL行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  IMPVEL-" + str(velNo) + "  grpName " + name + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/IMPVEL/" + str(velNo) + ls]
    addLines += ["nodeGroup " + name + ls]
    addLines += ["#   funcID       DIR    skewID    sensID    grndID     Frame     Icoor" + ls]
    addLines += [("%10d" % funcNo) + ("%10s" % velDir) + ("%10d" % 0) + ("%10d" % 0) + ("%10s" % grpID) + ("%10d" % 0) + ("%10d" % 0) + ls]
    addLines += ["#            Tscalex             Fscaley              Tstart               Tstop" + ls]
    addLines += [("%20s" % "1.0") + ("%20s" % "1.0") + ("%20s" % "0.0") + ls]
    radConts.append("/IMPVEL/" + str(velNo) + " velDir:" + velDir + "  NodeGroup " + name + ls)
    return addLines

#
#  createTransitVelocityFuncLines
def createTransitVelocityFuncLines(tranVelDict, name, funcNo):
    """ IMPVELのfunctionDataを作成"""
    global radConts
    #functionDataを取得（絶対値で取得）
    vals = tranVelDict[name][0]
    _dir, baseVal = getValueDirection(vals)
    ampData = tranVelDict[name][1]
    funcData = getFunctionData(baseVal, ampData)
    #func行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  IMPVEL_FUNC-" + str(funcNo) + "  NodeGroup " + name + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/FUNCT/" + str(funcNo) + ls]
    addLines += ["IMPVEL function" + ls]
    addLines += ["#             time:x              disp:y" + ls]
    for x, y in funcData:
        x, y = checkStrLength20([str(x), str(y)])
        addLines += [("%20s" % x) + ("%20s" % y) + ls]
    radConts.append("/FUNCT/" + str(funcNo) + "  NodeGroup " + name + ls)
    return addLines
    
#
#  createTransitAccelerationLines
#--------------------------------
def createTransitAccelerationLines(tranAccDict):
    """ transitAcceleration(加速度)を設定"""
    global radConts
    names = list(tranAccDict.keys())
    if len(names) == 0:
        return []
    names.sort()
    funcNo = getFunctionNoFromRadConts()
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  IMPACC conditions" + ls]
    accNo = 1
    for name in names:
        addLines += createTransitAccelerationNgrpLines(tranAccDict, name, accNo, funcNo)
        addLines += createTransitAccelerationFuncLines(tranAccDict, name, funcNo)
        funcNo += 1
        accNo += 1
    return addLines

#
#  createTransitAccelerationNgrpLines
def createTransitAccelerationNgrpLines(tranAccDict, name, accNo, funcNo):
    """ NGRPのIMPACC行を作成"""
    global radConts
    #加速度の方向を取得
    vals = tranAccDict[name][0]
    accDir, acc = getValueDirection(vals)
    #grpIDを取得
    if name == "ALL":
        grpID = getAllGrpID()
    else:
        grpID = tranAccDict[name][-1]
    #IMPACC行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  IMPACC-" + str(accNo) + "  grpName " + name + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/IMPACC/" + str(accNo) + ls]
    addLines += ["nodeGroup " + name + ls]
    addLines += ["#   funcID       DIR    skewID    sensID    grndID" + ls]
    addLines += [("%10d" % funcNo) + ("%10s" % accDir) + ("%10d" % 0) + ("%10d" % 0) + ("%10s" % grpID) + ls]
    addLines += ["#            Tscalex             Fscaley              Tstart               Tstop" + ls]
    addLines += [("%20s" % "1.0") + ("%20s" % "1.0") + ("%20s" % "0.0") + ls]
    radConts.append("/IMPACC/" + str(accNo) + " accDir:" + accDir + "  NodeGroup " + name + ls)
    return addLines

#
#  createTransitAccelerationFuncLines
def createTransitAccelerationFuncLines(tranAccDict, name, funcNo):
    """ IMPACCのfunctionDataを作成"""
    global radConts
    #functionDataを取得（絶対値で取得）
    vals = tranAccDict[name][0]
    _dir, baseVal = getValueDirection(vals)
    ampData = tranAccDict[name][1]
    funcData = getFunctionData(baseVal, ampData)
    #func行を作成
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  IMPACC_FUNC-" + str(funcNo) + "  NodeGroup " + name + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["/FUNCT/" + str(funcNo) + ls]
    addLines += ["IMPACC function" + ls]
    addLines += ["#             time:x              disp:y" + ls]
    for x, y in funcData:
        x, y = checkStrLength20([str(x), str(y)])
        addLines += [("%20s" % x) + ("%20s" % y) + ls]
    radConts.append("/FUNCT/" + str(funcNo) + "  NodeGroup " + name + ls)
    return addLines

#
#  createContactLines
#---------------------
def createContactLines(cntHeaderData, fistrDicts, interType):
    """ contactの行を作成する。"""
    global meshConts, radConts
    [nodeDict, elmDict, ngrpDict, surfDict, egrpDict, 
     matDict, shellThDict, contactDict, nodeElmDict, ampDict] = fistrDicts
    contacts = list(contactDict.keys())
    if len(contacts) == 0:
        return []
    #contact_algoを取得
    addLines = []
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  CONTACT define" + ls]
    contacts.sort()
    interNo = 1
    for contact in contacts:
        #全てのcontact内から該当するcontactを取得
        cardData = getHeaderDataOfCard(cntHeaderData, "!CONTACT")
        for header, data in cardData:
            words = pyFistr.deleteSp(data[0]).split(",")
            cpName = words[0]
            if contact == cpName:
                interaction = pyFistr.getValName(header, "INTERACTION")
                break
        if interaction == "TIED":
            #接触面を結合
            addLines += createContactTiedLines(contactDict, cpName, interNo)
        else:
            #接触面をペナルティ法で接触
            if contactDict[cpName][1] == "SURF-SURF":
                #SURF-SURFの場合、/INTER/TYPE24で作成
                addLines += createContactPenalty24Lines(fistrDicts, cpName, interNo, header, data)
            else:
                #NODE-SURFの場合、TYPE7 or TYPE24
                if interType == "type7":
                    #/INTER/TYPE7で作成
                    addLines += createContactPenalty7Lines(fistrDicts, cpName, interNo, header, data)
                elif interType == "type24":
                    #/INTER/TYPE24で作成
                    addLines += createContactPenalty24Lines(fistrDicts, cpName, interNo, header, data)
                else:
                    #その他は、/INTER/TYPE7で作成
                    addLines += createContactPenalty7Lines(fistrDicts, cpName, interNo, header, data)
        interNo += 1
    return addLines

#
#  createContactTiedLines
def createContactTiedLines(contactDict, cpName, interNo):
    """ 接触面をTIEDで設定する"""
    global radConts
    #grnodID, surIDを取得
    names = contactDict[cpName][0]
    ngID, sgID = getPairIDsFromMeshConts(names)
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  INTER-" + str(interNo) + "  " + cpName + ": " + ",".join(names) + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["#  contactPair is tied." + ls]
    addLines += ["/INTER/TYPE2/" + str(interNo) + ls]
    addLines += ["tied between " + names[0] + " and " + names[1] + ls]
    addLines += ["#  Slav_id   Surf_id    Ignore  Spotflag     Level   Isearch      Idel                       Dsearch" + ls]
    addLines += [("%10s" % ngID) + ("%10s" % sgID) + ("%10d" % 0) + ("%10d" % 0) + ("%10d" % 0) + ("%10d" % 0) + ("%10d" % 0) + " "*10 + ("%20d" % 0) + ls]
    radConts.append("/INTER/TYPE2/" + str(interNo) + "  tied " + cpName + " " + ",".join(names) + ls)
    return addLines

#
#  createContactPenalty7Lines
def createContactPenalty7Lines(fistrDicts, cpName, interNo, header, data):
    """ 接触定義をペネルティ法をtype7で設定"""
    global radConts
    [nodeDict, elmDict, ngrpDict, surfDict, egrpDict, 
     matDict, shellThDict, contactDict, nodeElmDict, ampDict] = fistrDicts
    words = pyFistr.deleteSp(data[0]).split(",")
    #摩擦係数fcoeffを取得
    fcoeff = words[1]
    #grnodID, surIDを取得
    names = contactDict[cpName][0]
    ngID, sgID = getPairIDsFromMeshConts(names)
    #gapMaxを取得（要素サイズの5倍に設定）
    elmSize = getElementSizeOfNgrp(names[0], fistrDicts)
    gapMax = str(elmSize * 5.0)
    (fcoeff, gapMax) = checkStrLength20([fcoeff, gapMax])
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  INTER-" + str(interNo) + "  " + cpName + ": " + ",".join(names) + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["#     Fric: set from FrontISTR" + ls]
    addLines += ["#  Gap_max: set (element size of nodeGroup) * 5.0" + ls]
    addLines += ["/INTER/TYPE7/" + str(interNo) + ls]
    addLines += ["contact between " + names[0] + " and " + names[1] + ls]
    addLines += ["#  Slav_id   Mast_id      Istf      Ithe      Igap                Ibag      Idel     Icurv      Iadm" + ls]
    addLines += [("%10s" % ngID) + ("%10s" % sgID) + "         0         0         3                   0         1         0         0" + ls]
    addLines += ["#          Fscalegap             Gap_max             Fpenmax                         Itied" + ls]
    addLines += [("%20d" % 1) + ("%20s" % gapMax) + "                 0.0                             0" + ls]
    addLines += ["#              Stmin               Stmax          %mesh_size               dtmin  Irem_gap   Irem_i2" + ls]
    addLines += ["                 0.0                 0.0                 0.0                 0.0         0         0" + ls]
    addLines += ["#              Stfac                Fric              Gapmin              Tstart               Tstop" + ls]
    addLines += ["                 1.0" + ("%20s" % fcoeff) + "                 0.0                 0.0                 0.0" + ls]
    addLines += ["#      IBC                        Inacti                VisS                VisF              Bumult" + ls]
    addLines += ["       000                             5                 0.0                 0.0                 0.0" + ls]
    addLines += ["#    Ifric    Ifiltr               Xfreq     Iform   sens_ID   fct_IDf             AscaleF   fric_ID" + ls]
    addLines += ["         0         0                 0.0         0         0         0                 0.0         0" + ls]
    radConts.append("/INTER/TYPE7/" + str(interNo) + "  contact " + cpName + " " + ",".join(names) + ls)
    return addLines

#
#  createContactPenalty24Lines
def createContactPenalty24Lines(fistrDicts, cpName, interNo, header, data):
    """ 接触定義をペネルティ法をtype7で設定"""
    global radConts
    [nodeDict, elmDict, ngrpDict, surfDict, egrpDict, 
     matDict, shellThDict, contactDict, nodeElmDict, ampDict] = fistrDicts
    words = pyFistr.deleteSp(data[0]).split(",")
    #摩擦係数fcoeffを取得
    fcoeff = words[1]
    #grnodID, surIDを取得
    names = contactDict[cpName][0]
    contactType = contactDict[cpName][1]
    if contactType == "NODE-SURF":
        ngID, sgID = getPairIDsFromMeshConts(names)
        sg0ID = "0"
    elif contactType == "SURF-SURF":
        sg0ID, sgID = getPairSurfIDsFromMeshConts(names)
        ngID = "0"
    #摩擦係数の数値桁数チェック
    [fcoeff] = checkStrLength20([fcoeff])
    addLines = []
    addLines += ["#---1----|----2----|----3----|----4----|----5----|----6----|----7----|----8----|----9----|---10----|" + ls]
    addLines += ["#  INTER-" + str(interNo) + "  " + cpName + ": " + ",".join(names) + ls]
    addLines += ["#--------.---------.---------.---------." + ls]
    addLines += ["#     Fric: set from FrontISTR" + ls]
    addLines += ["/INTER/TYPE24/" + str(interNo) + ls]
    addLines += ["contact between " + names[0] + " and " + names[1] + ls]
    addLines += ["# Surf1_id  Surf2_id      Istf      Ithe                Irem                Idel" + ls]
    addLines += [("%10s" % sg0ID) + ("%10s" % sgID) + ("%10s" % "4") + "                             0                   0" + ls]
    addLines += ["#   grndID                         Iedge          Edge_angle           Gap_max_s           Gap_max_m" + ls]
    addLines += [("%10s" % ngID) + "                             0                                     0.0                 0.0" + ls]
    addLines += ["#              StMin               StMax     Igap0    Iopen0            IopenMax" + ls]
    addLines += ["                 0.0                 0.0         0         0                 0.0" + ls]
    addLines += ["#              Stfac                Fric                                  Tstart               Tstop" + ls]
    addLines += [("%20s" % "0.0") + ("%20s" % fcoeff) + "                                     0.0                 0.0" + ls]
    addLines += ["#      Ibc                        Inacti                VISs                               Tpressfit" + ls]
    addLines += ["       000                             0                 0.0" + ls]
    addLines += ["#    Ifric    Ifiltr               Xfreq              sensID" + ls]
    addLines += ["         0         0                 0.0                   0" + ls]
    radConts.append("/INTER/TYPE24/" + str(interNo) + "  contact " + cpName + " " + ",".join(names) + ls)
    return addLines

#
#  getElementSizeOfNgrp
def getElementSizeOfNgrp(ngrp, fistrDicts):
    """ ngrpが属する要素のサイズを取得する"""
    [nodeDict, elmDict, ngrpDict, surfDict, egrpDict, 
     matDict, shellThDict, contactDict, nodeElmDict, ampDict] = fistrDicts
    #最初のnodeNoを取得
    nodeNo = ngrpDict[ngrp][0]
    elmNo = list(nodeElmDict[nodeNo])[0]
    nodeNos = elmDict[elmNo][0]
    #node0, node1の距離を取得
    loc0 = nodeDict[nodeNos[0]]
    loc1 = nodeDict[nodeNos[1]]
    size = geom.length(loc0, loc1)
    return size

#
#  createRadFooter
#------------------
def createRadFooter():
    """ radFileのfooterを作成"""
    addLines  = ["#===================================================================================================" + ls]
    addLines += ["/END" + ls]
    addLines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
    return addLines

#--------------------------
#  createRadiossEngineFile
#--------------------------
def createRadiossEngineFile(fistrFiles, radiossFiles, fistrDicts):
    """ RadiossのEngineファイルを作成する"""
    global engConts
    engConts = []
    [meshFile, cntFile, sphEgrp, fistrUnit] = fistrFiles
    [incFile, radFile, yearVersion, unit, interType] = radiossFiles
    [nodeDict, elmDict, ngrpDict, surfDict, egrpDict, 
     matDict, shellThDict, contactDict, nodeElmDict, ampDict] = fistrDicts
    #cntFile読み込み
    f = open(cntFile, encoding="utf-8"); lines = f.readlines(); f.close()
    cntHeaderData = pyFistr.getHeaderData(lines)
    #EngineHeader作成
    lines  = createEngineHeaderLines(cntFile, meshFile)
    lines += craeteVersionLines(yearVersion)
    lines += createMonRunTfileLines(radFile, cntHeaderData)
    lines += createDTLines(cntHeaderData)
    lines += createAnimData()
    #fileに出力
    engFile = "_".join(radFile.split("_")[:-1]) + "_0001.rad"
    f = open(engFile, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  createEngineHeaderLines
#---------------------
def createEngineHeaderLines(cntFile, meshFile):
    """ EngineFileのheaderを作成する"""
    now = datetime.datetime.now()
    dateList = [str(now.year), str(now.month), str(now.day)]
    addLines = []
    addLines  = ["#RADIOSS ENGINE" + ls]
    addLines += ["#===================================================================================================" + ls]
    addLines += ["#  generated by fistr2rad.py" + ls]
    addLines += ["#           from " + meshFile + " and " + cntFile + ls]
    addLines += ["#                                                " + "/".join(dateList) + ls]
    addLines += ["#--------.---------.---------.---------.---------.---------.---------.---------.---------.---------." + ls]
    return addLines

#
#  craeteVersionLines
#---------------------
def craeteVersionLines(yearVersion):
    """ version行を作成"""
    global engConts
    addLines = []
    addLines += ["#  VERS" + ls]
    addLines += ["/VERS/" + yearVersion + ls]
    engConts.append("/VERS/" + yearVersion + ls)
    return addLines

#
#  createMonRunTfileLines
#-----------------
def createMonRunTfileLines(radFile, cntHeaderData):
    """ RUN行を作成"""
    global engConts
    runName = "_".join(radFile.split("_")[:-1])
    endTime = getEndTimeFromCntFile(cntHeaderData)
    [endTime] = checkStrLength10([str(endTime)])
    addLines = []
    addLines += ["#  RUN" + ls]
    addLines += ["/RUN/" + runName + "/1" + ls]
    addLines += [("%10s" % endTime) + ls]
    addLines += ["/MON/ON" + ls]
    addLines += ["/TFILE/0" + ls]
    engConts.append("/RUN/" + runName + "/1  " + str(endTime) + ls)
    engConts.append("/MON/ON" + ls)
    engConts.append("/TFILE/0" + ls)
    return addLines

#
#  getEndTimeFromCntFile
def getEndTimeFromCntFile(cntHeaderData):
    """ endTimeを取得する"""
    endTime = ""
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!DYNAMIC":
            line = data[1]
            ws = pyFistr.deleteSp(line).split(",")
            endStep = int(ws[2])
            DT = float(ws[3])
            endTime = endStep * DT
            #陰解法or陽解法チェック
            ws = pyFistr.deleteSp(data[0]).split(",")
            if ws[0] == "1":
                endTime = "implicit"
            break
    if endTime == "":
        print()
        print("ERROR: Solution type is not 'dynamic'.")
        print("  Set 'dynamic' to slution type..")
        exit()
    elif endTime == "implicit":
        print()
        print("ERROR: analysis type is not 'explicit'.")
        print("  Set 'explicit' method to analysis type.")
        exit()
    return endTime

#
#  createDTLines
#----------------
def createDTLines(cntHeaderData):
    """ 時間ステップの設定"""
    global engConts
    DT, animDT = getDeltaTFromCnt(cntHeaderData)
    DT, animDT = checkStrLength10([str(DT), str(animDT)])
    addLines = []
    addLines += ["#  DT" + ls]
    addLines += ["/DT/NODA/CST/0" + ls]
    addLines += [("%10s" % "0.9") + ("%10s" % DT) + ls]
    addLines += ["/ANIM/DT" + ls]
    addLines += [("%10s" % "0.0") + ("%10s" % animDT) + ls]
    engConts.append("/DT/NODA/CST/0  " + str(DT) + ls)
    engConts.append("/ANIM/DT  " + str(animDT) + ls)
    return addLines

#
#  getDeltaTFromCnt
def getDeltaTFromCnt(cntHeaderData):
    """ deltaT（時間増分）をcntFileから取得"""
    DT = 0
    animDT = 1
    endStep = 1
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!DYNAMIC":
            line = data[1]
            ws = pyFistr.deleteSp(line).split(",")
            endStep = int(ws[2])
            DT = float(ws[3])
        elif words[0] == "!WRITE":
            freq = pyFistr.getValName(header, "FREQUENCY")
            if freq == "":
                n = 1
            else:
                n = int(freq)
        if DT > 0 and animDT > 0:
            break
    animDT = DT * n
    if DT == 0:
        #未設定の場合
        return DT, animDT
    nAnim = endStep * DT / animDT
    if nAnim > 1000:
        print()
        print("ERROR: number of animations are too many!!")
        print("  over 1000 animations.")
        print("  check write frequency of FrontISTR.")
        exit()
    return DT, animDT
   
#
#  createAnimData
def createAnimData():
    """ animationに関するデータ行を作成"""
    global engConts
    #要素確認flagを取得
    (shell, solid, sph) = getElementFlag()
    #節点変位
    addLines = []
    addLines += ["/ANIM/VECT/DISP" + ls]
    addLines += ["/ANIM/VECT/VEL" + ls]
    addLines += ["/ANIM/ELEM/EPSP" + ls]
    addLines += ["/ANIM/ELEM/VONM" + ls]
    engConts.append("/ANIM/VECT/DISP" + ls)
    engConts.append("/ANIM/VECT/VEL" + ls)
    engConts.append("/ANIM/ELEM/EPSP" + ls)
    engConts.append("/ANIM/ELEL/VONM" + ls)
    if solid == True:
        addLines += ["/ANIM/BRIC/TENS/DAMA" + ls]
        addLines += ["/ANIM/BRIC/TENS/STRESS" + ls]
        addLines += ["/ANIM/BRIC/TENS/STRAIN" + ls]
        #addLines += ["/ANIM/BRIC/TENS/EPSP" + ls]
        #addLines += ["/ANIM/BRIC/VONM" + ls]
        engConts.append("/ANIM/BRIC/TENS/DAMA" + ls)
        engConts.append("/ANIM/BRIC/TENS/STRESS" + ls)
        engConts.append("/ANIM/BRIC/TENS/STRAIN" + ls)
        #engConts.append("/ANIM/BRIC/TENS/EPSP" + ls)
        #engConts.append("/ANIM/BRIC/VONM" + ls)
    if shell == True:
        addLines += ["/ANIM/SHELL/TENS/UPPER" + ls]
        addLines += ["/ANIM/SHELL/TENS/LOWER" + ls]
        #addLines += ["/ANIM/SHELL/VONM" + ls]
        engConts.append("/ANIM/SHELL/TENS/UPPER" + ls)
        engConts.append("/ANIM/SHELL/TENS/LOWER" + ls)
        #engConts.append("/ANIM/SHELL/VONM" + ls)
    return addLines

#
#  getElementFlag
def getElementFlag():
    """ 使用している要素をチェック"""
    global meshConts
    shell = False; solid = False; sph = False
    for line in meshConts:
        words = line.split()
        if words[2] == "Elements":
            ws = words[1].split("/")
            elm = ws[1]
            elmType = elmTypeDict[elm]
            if elmType == "shell":
                shell = True
            elif elmType == "solid":
                solid = True
            elif elmType == "sph":
                sph = True
    return shell, solid, sph

#
#  printConvertResult
#--------------------
def printConvertResult(meshFile, cntFile, incFile, radFile):
    """ mesh変換結果を出力する"""
    global meshConts, radConts, engConts
    msg = [ls]
    msg += ["Fistr mesh has been converted to Radioss format." + ls]
    msg += [ls]
    msg += [" mesh contents" + ls]
    msg += list(map(lambda x: x[1:], meshConts))
    msg += [ls]
    msg += [" STRATER contents" + ls]
    msg += list(map(lambda x: "   " + x, radConts))
    msg += [ls]
    msg += [" ENGINE contents" + ls]
    msg += list(map(lambda x: "   " + x, engConts))
    msg += [ls]
    msg += ["input files" + ls]
    msg += ["  msh: " + meshFile + ls]
    msg += ["  cnt: " + cntFile + ls]
    msg += ["output files" + ls]
    msg += ["  inc: " + incFile + ls]
    msg += ["  rad: " + radFile + ls]
    engFile = "_".join(radFile.split("_")[:-1]) + "_0001.rad"
    msg += ["       " + engFile + ls]
    msg = "".join(msg)
    print(msg)


#--------------------
#  convertFistr2rad
#--------------------
def convertFistr2rad(fistrFiles, radiossFiles):
    """ fistr.mshをrad形式のメッシュに変換する"""
    global meshConts
    [meshFile, cntFile, sphEgrp, fistrUnit] = fistrFiles
    [incFile, radFile, yearVersion, unit, interType] = radiossFiles
    #fistrのメッシュHeaderDataを取得
    f = open(meshFile, encoding="utf-8"); lines = f.readlines(); f.close()
    headerData = pyFistr.getHeaderNumData(lines)
    f = open(cntFile, encoding="utf-8"); lines = f.readlines(); f.close()
    cntHeaderData = pyFistr.getHeaderData(lines)

    print("getting each setting data...", flush=True)
    #node, element, ngrp, egrpを取得
    nodeDict = getNodeDataDict(headerData)
    elmDict = getElementDataDict(headerData)
    nodeElmDict = getNodeElementDict(elmDict)
    ngrpDict = getNodeGroupDict(headerData)
    surfDict = getSurfaceGroupDict(headerData, elmDict)
    egrpDict = getElementGroupDict(headerData)
    nodeDict, elmDict, ngrpDict = deleteDummyNodesChangeElmType(nodeDict, elmDict, ngrpDict)
    shellThDict = getShellThicknessDict(headerData)    
    contactDict = getContactPair(headerData)
    ampDict = getAmplitudeDict(headerData)
    #nodeDict, elmDict, ngrpDict = renumberNode(nodeDict, elmDict, ngrpDict)
    #rad形式に要素を変換
    elmRadDict = convertElement(elmDict)
    #egrpと材料定数を取得
    matDict = getEgrpMaterialDictFromCnt(cntHeaderData, headerData)
    fistrDicts = [nodeDict, elmDict, ngrpDict, surfDict, egrpDict,
                  matDict, shellThDict, contactDict, nodeElmDict, ampDict]

    print("converting to OpenRadioss files...", flush=True)
    #incFileの内容を作成
    createRadiossIncMeshFile(fistrDicts, elmRadDict, sphEgrp, incFile)
    #starterFileを作成
    createRadiossStarterFile(fistrFiles, radiossFiles, fistrDicts)
    #engineFileを作成
    createRadiossEngineFile(fistrFiles, radiossFiles, fistrDicts)

    #結果を標準出力に出力
    printConvertResult(meshFile, cntFile, incFile, radFile)

#
#  checkFileNames
#----------------
def checkFileNames(fistrFiles, radiossFiles):
    """ file名をチェック"""
    [meshFile, cntFile, sphEgrp] = fistrFiles
    [incFile, radFile, yearVersion, unitNo, interType] = radiossFiles
    #file名をチェック
    if meshFile.split(".")[-1] != "msh":
        meshFile += ".msh"
    if cntFile.split(".")[-1] != "cnt":
        cntFile += ".cnt"
    if incFile.split(".")[-1] != "inc":
        incFile += ".inc"
    if radFile.split("_")[-1] != "0000.rad":
        radFile += "_0000.rad"
    #単位系をチェック
    fistrUnit = getCurrDirUnit()
    if unitNo == "":
        unit = fistrUnit[:]
        #tonは、Mgに変更する。
        if unit[0] == "ton":
            unit[0] = "Mg"
    else:
        unit = unitDict[unitNo]
        fistrUnit = unit
    #接触定義のチェック
    if not interType in ["type7", "type24"]:
        interType = "type7"
    fistrFiles = [meshFile, cntFile, sphEgrp, fistrUnit]
    radiossFiles = [incFile, radFile, yearVersion, unit, interType]
    return fistrFiles, radiossFiles

#
#  getCurrDirUnit
def getCurrDirUnit():
    """ currDir内の単位を取得する"""
    unit = "kg_m_s"
    fileName = "easyIstrSub_data"
    if os.path.exists(fileName) == True:
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        for line in lines:
            words = line.split()
            if len(words) > 1:
                if words[0] == "dimension":
                    unit = words[1]
                    break
    fistrUnit = unit.split("_")
    return fistrUnit

#
#  printHelp
#-------------
def printHelp():
    """ helpを出力する。"""
    global HelpCont
    print(HelpCont)

#
#  printVersion
#---------------
def printVersion():
    """ fistr2rad.pyのversionを表示"""
    lines  = [ls]
    lines += ["Version of fistr2rad.py is" + ls]
    lines += ["      --- " + thisVersion + " ---" + ls]
    cont = "".join(lines)
    print(cont)

#
#  getOption
#-------------
def getOption():
    """ optionを取得"""
    workFolder = ""
    cntFile, meshFile = "", ""
    incFile = ""
    radFile = ""
    sphEgrp = ""
    yearVersion = "2022"
    unit = ""
    interType = "type7"
    help = "off"
    i = 1
    while i < len(sys.argv):
        if sys.argv[i] == "-w":
            i += 1
            workFolder = sys.argv[i]
        elif sys.argv[i] == "-i":
            i += 1
            meshFile = sys.argv[i] + ".msh"
            cntFile = sys.argv[i] + ".cnt"
        elif sys.argv[i] == "-msh":
            i += 1
            meshFile = sys.argv[i]
        elif sys.argv[i] == "-cnt":
            i += 1
            cntFile = sys.argv[i]
        elif sys.argv[i] == "-o":
            i += 1
            incFile = sys.argv[i] + "_0000.inc"
            radFile = sys.argv[i] + "_0000.rad"
        elif sys.argv[i] == "-inc":
            i += 1
            incFile = sys.argv[i] + "_0000.inc"
        elif sys.argv[i] == "-rad":
            i += 1
            radFile = sys.argv[i] + "_0000.rad"
        elif sys.argv[i] == "-sph":
            i += 1
            sphEgrp = sys.argv[i]
        elif sys.argv[i] == "-vrad":
            i += 1
            yearVersion = sys.argv[i]
        elif sys.argv[i] == "-u":
            i += 1
            unit = sys.argv[i]
        elif sys.argv[i] == "-inter":
            i += 1
            interType = sys.argv[i]
        elif sys.argv[i] == "-h" or sys.argv[i] == "--help":
            printHelp()
            help = "on"
            break
        elif sys.argv[i] == "-v" or sys.argv[i] == "--version":
            printVersion()
            help = "on"
        else:
            printHelp()
            help = "on"
        i += 1
    #currDir移動
    if workFolder != "":
        os.chdir(workFolder)
    #meshFile、cntFile名を取得
    if meshFile == "" or cntFile == "":
        cFile, mFile = getCntMshFileName(os.getcwd())
    if meshFile == "":
        meshFile = mFile
    if cntFile == "":
        cntFile = cFile
    #inc, radのファイル名を取得
    if incFile == "" or radFile == "":
        header = ".".join(cntFile.split(".")[:-1])
    if incFile == "":
        incFile = header + "_0000.inc"
    if radFile == "":
        radFile = header + "_0000.rad"
    fistrFiles = [meshFile, cntFile, sphEgrp]
    radiossFiles = [incFile, radFile, yearVersion, unit, interType]
    return fistrFiles, radiossFiles, help


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    #_ = gettext.gettext

    fistrFiles, radiossFiles, help = getOption()
    if help == "on":
        exit()
    fistrFiles, radiossFiles = checkFileNames(fistrFiles, radiossFiles)
    convertFistr2rad(fistrFiles, radiossFiles)
