#!/usr/bin/python3
#
#   fistr2abaqusForBeam.py
#
#       fistr形式のmsh、cntファイルをabaqus形式のinpファイルに変換する
#       beam要素とEQIATION
#
#   21/02/04    新規作成
#      02/07    createLines:座標値の出力を「16.9e」で指定。calculix側で
#               エラー発生する場合がある。
#      02/11    !SPRINGの変換を追加
#      03/19    fistr2abaqus.pyをimportして使用する様に修正。
#   22/03/15    国際化
#   24/07/07    openにencoding="utf-8"を追加
#   25/01/18    change2ndMesh:「*NODE」行にNSET=NALLが存在しない場合の処理追加
#

import sys, os
import geometryFunc as geo
#import convertMesh as cm
import pyFistr
import fistr2abaqus



#
#  changeNLsolution
#-------------------
def changeNLsolution(abaHeaderData):
    """ 非線形への変換"""
    flag = 0
    for i in range(len(abaHeaderData)):
        header = abaHeaderData[i][0]
        data = abaHeaderData[i][1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*STEP":
            header = "*STEP, INC=100, NLGEOM\n"
            abaHeaderData[i][0] = header
            flag += 1
        elif words[0] == "*STATIC":
            data = ["0.1, 1.0\n"]
            abaHeaderData[i][1] = data
            flag += 1
        if flag >= 2:
            break
    return abaHeaderData

#
#  getDictsFromAbaHeader
def getDictsFromAbaHeader(abaHeaderData):
    """ inpファイルのheaderDataからnodeDict、elementDictを作成する"""
    #nodeDictを作成
    nodeDict = {}
    for header, data in abaHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*NODE":
            for node in data:
                nodeNo = node[0]
                loc = node[1:]
                nodeDict[nodeNo] = {"loc":loc, "elms":[]}
    #elementDictを作成
    elementDict = {}
    for header, data in abaHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*ELEMENT":
            elmType = pyFistr.getValName(header, "TYPE")
            for i in range(len(data)):
                elmData = data[i]
                elmNo = elmData[0]
                nodes = elmData[1:]
                elementDict[elmNo] = {"nodes":nodes, "type":elmType}
                for nodeNo in nodes:
                    nodeDict[nodeNo]["elms"].append(elmNo)
    return nodeDict, elementDict

#
#  change2ndMesh
#----------------
def change2ndMesh(abaHeaderData, dicts):
    """ 2次要素に変換する"""
    nodeDict, elementDict = dicts
    addNodeData = []
    maxNo = max(list(nodeDict.keys()))
    for i in range(len(abaHeaderData)):
        header = abaHeaderData[i][0]
        data = abaHeaderData[i][1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*ELEMENT":
            elmType = pyFistr.getValName(header, "TYPE")
            if elmType == "B31" or elmType == "B31R":
                newType = "B32R"
                newHeader = pyFistr.setValName(header, "TYPE=" + newType)
                abaHeaderData[i][0] = newHeader
                for ii in range(len(data)):
                    elmData = data[ii]
                    elmNo = elmData[0]
                    nodes = elmData[1:]
                    elementDict[elmNo] = {"nodes":nodes, "type":elmType}
                    for nodeNo in nodes:
                        nodeDict[nodeNo]["elms"].append(elmNo)
                    loc1 = nodeDict[nodes[0]]["loc"]
                    loc2 = nodeDict[nodes[1]]["loc"]
                    addPointLoc = geo.midPoint(loc1, loc2)
                    maxNo += 1
                    newNo = maxNo
                    nodeDict[newNo] = {"loc":addPointLoc, "elms":[elmNo]}
                    newNodeData = [newNo] + addPointLoc
                    #newNodeLine = ", ".join(list(map(str, newNodeData))) + "\n"
                    #addNodeData.append(newNodeLine)
                    addNodeData.append(newNodeData)
                    newNodes = [nodes[0], newNo, nodes[1]]
                    elmData = [elmData[0]] + newNodes
                    data[ii] = elmData
                abaHeaderData[i][1] = data
    #nodeデータを追加
    flag = 0
    for i in range(len(abaHeaderData)):
        header = abaHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        nset = pyFistr.getValName(header, "NSET")
        if words[0] == "*NODE" and nset == "NALL":
            flag = 1
        else:
            if flag == 1:
                si = i
                break
    if flag == 1:
        addHeaderData = [["*NODE, NSET=NALL\n", addNodeData]]
        abaHeaderData = abaHeaderData[:si] + addHeaderData + abaHeaderData[si:]
    else:
        #「*NODE,NSET=NALL」が見つからない場合
        flag = 0
        for i in range(len(abaHeaderData)):
            header = abaHeaderData[i][0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "*NODE":
                flag = 1
            else:
                if flag == 1:
                    si = i
                    break
        addHeaderData = [["*NODE\n", addNodeData]]
        abaHeaderData = abaHeaderData[:si] + addHeaderData + abaHeaderData[si:]
        #addNodeData(中間節点)をNALLに追加する
        nodes = list(map(lambda x: x[0], addNodeData))
        for i in range(len(abaHeaderData)):
            header, data = abaHeaderData[i]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "*NSET":
                nset = pyFistr.getValName(header, "NSET")
                if nset == "NALL":
                    data += nodes
                    abaHeaderData[i][1] = data
                    break
    return abaHeaderData

#
#  changeSectionPipe
#--------------------
def changeSectionPipe(abaHeaderData):
    """ sectionをpipeに修正"""
    for i in range(len(abaHeaderData)):
        header = abaHeaderData[i][0]
        data = abaHeaderData[i][1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*BEAMSECTION":
            elset = pyFistr.getValName(header, "ELSET")
            matName = pyFistr.getValName(header, "MATERIAL")
            newHeader = "*BEAM GENERAL SECTION, ELSET=" + elset + ", MATERIAL=" + matName + ", SECTION=PIPE\n"
            abaHeaderData[i][0] = newHeader
            if data[0][0] == "*":
                data[0] = "** phi, t\n"
            else:
                data = ["** phi, t\n"] + data
            abaHeaderData[i][1] = data
    return abaHeaderData

#
#  convertToInp
#----------------
def convertToInp(fileHeader, meshFlag):
    """ fistr形式をinp形式に変換する"""
    #fistrのfileを読み込み
    fileName = fileHeader + ".msh"
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    meshHeaderData = pyFistr.getHeaderNumData(lines)
    fileName = fileHeader + ".cnt"
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    contHeaderData = pyFistr.getHeaderData(lines)
    #inpファイルに変換
    #inpFileのheaderを作成
    abaHeaderData = fistr2abaqus.convToInt(meshHeaderData, contHeaderData)
    #非線形への変換
    abaHeaderData = changeNLsolution(abaHeaderData)
    #2次要素へ変換？
    if meshFlag == "change2nd":
        #辞書作成
        #dicts = fistr2abaqus.createNodeElementDict(meshHeaderData)
        dicts = getDictsFromAbaHeader(abaHeaderData)
        #2次へ変換
        abaHeaderData = change2ndMesh(abaHeaderData, dicts)
        #sectionを修正
        abaHeaderData = changeSectionPipe(abaHeaderData)
    #linesを作成し、保存
    lines = fistr2abaqus.createLines(abaHeaderData)
    fileName = fileHeader + ".inp"
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()
    print(_("FrontISTR形式をinp形式に変換しました。"))

#
#  getArgument
#--------------
def getArgument():
    meshFlag = "change2nd"          #1次要素は2次要素に変換する
    fileHeader = "FistrModel"       #FrontISTRのfileHeader名
    if len(sys.argv) == 1:
        return meshFlag, fileHeader
    elif len(sys.argv) == 2:
        if sys.argv[1] == "-n":
            meshFlag = "noChange"
        else:
            fileHeader = sys.argv[1]
        return meshFlag, fileHeader
    else:
        if sys.argv[1] == "-n":
            meshFlag = "noChange"
            fileHeader = sys.argv[2]
        else:
            fileHeader = sys.argv[1]
            if sys.argv[2] == "-n":
                meshFlag = "noChange"
        return meshFlag, fileHeader


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    #_ = gettext.gettext

    meshFlag, fileHeader = getArgument()
    convertToInp(fileHeader, meshFlag)
