#!/usr/bin/python3
#  coding: utf-8
#
#       createShapeVtu.py
#
#       mshファイルから、vtu形式で形状を作成して「shape.vtu」を作成する
#       形状を作成するのみ。（cellData、pointDataは作成しない。）
#
#   20/03/26    新規作成
#

import os, sys
import fistr2vtu
import convertMesh as cm

ls = "\n"


#
#  getLinesWordsN
def getLinesWordsN(words, n):
    """ n words毎に改行して返す"""
    lines = []
    for i in range(0, len(words), n):
        line = " ".join(words[i:i + n]) + ls
        lines.append(line)
    return lines

#
#  createVtkHeader
def createVtkHeader():
    """ vtkファイルのheader部を作成して返す"""
    lines  = ['<?xml version="1.0"?>' + ls]
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0">' + ls]
    lines += ['<UnstructuredGrid>' + ls]
    return lines

#
#  createVtkPieceHeader
def createVtkPieceHeader(vnList, veList):
    """ pieceSectionのheaderを設定"""
    nNodes = len(vnList)
    nCells = len(veList)
    lines = ['<Piece NumberOfPoints="' + str(nNodes) + '" NumberOfCells="' + str(nCells) + '">' + ls]
    return lines

#
#  createVtkNodes
def createVtkNodes(vnList):
    """ node部を作成"""
    lines = ['<Points>' + ls]
    lines += ['<DataArray type="Float32" NumberOfComponents="3" format="ascii">' + ls]
    for nData in vnList:
        loc = nData[0]
        loc = list(map(lambda x: str(x), loc))
        line = " ".join(loc) + ls
        lines.append(line)
    lines += ['</DataArray>' + ls]
    lines += ['</Points>' + ls] 
    return lines

#
#  createVtkCells
def createVtkCells(veList):
    """ cell部を作成する"""
    #node順入れ替え用の辞書
    changeOrderDict = {
     #node数 新order  入れ替え関数名
        3:[[0,1,2],   cm.order_3z],
        4:[[0,1,2,3], cm.order_4z],
        6:[[0,2,4,1,3,5], cm.order_6z],
        8:[[0,2,4,6,1,3,5,7], cm.order_8z]
        }
    #node数からface要素名を取得する辞書
    #  面要素のnode数とelType名
    faceNumToFistrVtkTypeDict = {
        #node数: fistr名、vtk名
            3:   ["231",  "5"],
            4:   ["241",  "9"],
            6:   ["232",  "22"],
            8:   ["242",  "23"]
        }

    lines = ['<Cells>' + ls]
    #elementを作成
    lines += ['<DataArray type="Int32" Name="connectivity" format="ascii">' + ls]
    for eData in veList:
        eNodes = eData[0][1:]
        eNodes = list(map(lambda x: str(x), eNodes))
        line = " ".join(eNodes) + ls
        lines.append(line)
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="Int32" Name="offsets" format="ascii">' + ls]
    #offsetを作成
    offset = 0
    data = []
    #  solid要素
    for eData in veList:
        eNodes = eData[0][1:]
        offset += len(eNodes)
        data.append(str(offset))
    lines += getLinesWordsN(data, 10)      #10 words毎に改行する
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="UInt8" Name="types" format="ascii">' + ls]
    #typesを作成
    data = []
    #  solid要素
    for eData in veList:
        elType = eData[1]
        data.append(elType)
    lines += getLinesWordsN(data, 20)
    lines += ['</DataArray>' + ls]
    lines += ['</Cells>' + ls]
    return lines

#
#  createVtkPieceFooter
def createVtkPieceFooter():
    """ pieceSectionのfooterを設定"""
    lines = ['</Piece>' + ls]
    return lines

#
#  createVtkFooter
def createVtkFooter():
    """ vtkファイルのfooterを作成して返す"""
    lines  = ['</UnstructuredGrid>' + ls]
    lines += ['</VTKFile>' + ls]
    return lines

#
#  createVtkLines
def createVtkLines(vnList, veList):
    newLines = createVtkHeader()
    newLines += createVtkPieceHeader(vnList, veList)
    newLines += createVtkNodes(vnList)
    newLines += createVtkCells(veList)
    newLines += createVtkPieceFooter()
    newLines += createVtkFooter()
    return newLines

#
#  createShape
#--------------
def createShape(fileName):
    fileName = fistr2vtu.getMeshFileName(fileName)
    if fileName == "":
        return
    lines = fistr2vtu.readFileLines(fileName)
    #mesh内容を読み込む
    meshHeaderNumConts = cm.getFistrHeaderNumData(lines)
    #メッシュ内容を取得
    nodeList, elementList = cm.getFistrMeshConts(meshHeaderNumConts)
    #nodeNo振り直し、vnList、veListを取得
    nodeList, vnList = fistr2vtu.getNewNodeNo(nodeList, meshHeaderNumConts)
    elementList, veList = fistr2vtu.renumberNodeElement(nodeList, elementList, meshHeaderNumConts)
    #point, cell行を作成
    vtuLines = createVtkLines(vnList, veList)
    vtuFileName = "shape.vtu"
    fistr2vtu.writeFileLines(vtuFileName, vtuLines)



if __name__ == "__main__":
    fileName = sys.argv[1]
    createShape(fileName)
