#!/usr/bin/python3
#  coding :  utf-8
#
#   createAbaSolidMakuVtu.py
#
#       calculixのinp、datファイルから計算結果をvtu形式に変換する。
#
#   vtk変換対象の結果データ
#       displacements
#       stresses
#       temperatures
#       heat flux
#
#   変換対象のメッシュ
#       solid   C3D4, C3D6, C3D8, C38R, C3D15, C3D20, C3D20R,    
#       shell   S3, S4, S4R, S6, S8, S8R
#       maku    M3D3, M3D4
#       beam    B31, B31R, B32, B32R
#
#   21/03/20    新規作成
#      03/22    temperature, heatFluxを追加
#      05/17    optionとして「-n」を追加
#   22/03/02    createAbaVtk.pyから独立
#               maku要素(M3D3, M3D4)を追加
#               createVtu:節点応力とmisesStressを追加。
#      03/05    createVtuLines:要素No順に並べ変えを追加
#               getAverageData:追加。（積分点の平均を求める）
#               従来は、最後の積分点データが採用されていた。
#      03/15    国際化
#   24/07/07    openにencoding="utf-8"を追加
#

from sre_constants import _NamedIntConstant
import sys, os
import pyFistr
import convertMesh as cm
import geometryFunc as geo

#形状作成する要素名
abaShapeElmsSet = {
    "C3D4", "C3D6", "C3D8", "C38R", "C3D15", "C3D20", "C320R",
    "S3", "S4", "S4R", "S6", "S8", "S8R", "M3D3", "M3D4",
    "B31", "B31R", "B32", "B32R"
    }

#
#  getMultiTransformCont
#-------------------------
def getMultiTransformCont(abaHeaderData):
    """ 座標変換（transform）しているか確認し、全てのtransfoamを取得する"""
    transforms = []
    trans = []
    for header, data in abaHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*TRANSFORM":
            tType = pyFistr.getValName(header, "TYPE")
            nset = pyFistr.getValName(header, "NSET")
            line = data[0]
            ws = pyFistr.deleteSp(line).split(",")
            if len(ws) == 6:
                try:
                    ws = list(map(float, ws))
                    a = ws[:3]
                    b = ws[3:]
                    nodesSet = getTranformNodes(abaHeaderData, nset)
                    trans = [tType, nset, a, b, nodesSet]
                    transforms.append(trans)
                except:
                    pass                        
    return transforms
    
#
#  createNodeElementDict
#------------------------
def createNodeElementDict(abaHeaderData):
    """ nodeDict, elementDictを作成する"""
    nodeDict = {}
    elementDict = {}
    for header, data in abaHeaderData:
        words = cm.deleteSp(header).split(",")
        if words[0] == "*NODE":
            #nset = pyFistr.getValName(header, "NSET")
            for nodeLine in data:
                nodeNo = nodeLine[0]
                loc = nodeLine[1:]
                nodeDict[nodeNo] = {"loc":loc, "elms":set([])}
    for header, data in abaHeaderData:
        words = cm.deleteSp(header).split(",")
        if words[0] == "*ELEMENT":
            elmType = pyFistr.getValName(header, "TYPE")
            for elmLine in data:
                elmNo = elmLine[0]
                nodes = elmLine[1:]
                elementDict[elmNo] = {"nodes":nodes, "type":elmType}
                for nodeNo in nodes:
                    nodeDict[nodeNo]["elms"].add(elmNo)
    return nodeDict, elementDict

#
#  getNewNumberForVtk
#---------------------
def getNewNumberForVtk(neDicts):
    """ vtk用のNoを辞書内に取得する"""
    nodeDict, elementDict = neDicts
    nodes = list(nodeDict.keys())
    nodes.sort()
    n = 0
    for nodeNo in nodes:
        elms = nodeDict[nodeNo]["elms"]
        elmTypes = map(lambda elmNo: elementDict[elmNo]["type"], elms)
        #nodeが使われている要素は、形状作成する要素？
        if len(abaShapeElmsSet & set(elmTypes)) > 0:
            nodeDict[nodeNo]["newNo"] = n
            n += 1
    elms = list(elementDict.keys())
    elms.sort()
    n = 0
    for i in range(len(elms)):
        elmNo = elms[i]
        #要素typeが形状作成する要素？
        if elementDict[elmNo]["type"] in abaShapeElmsSet:
            elementDict[elmNo]["newNo"] = n
            n += 1
    return nodeDict, elementDict

#
#  getMinMaxTimeStep(lines)
#--------------------
def getMinMaxTimeStep(lines):
    """ minMaxのtimeStepを取得する"""
    minTime = 0.0
    for line in lines:
        wordsSet = set(line.split())
        if len(wordsSet & set(["and", "time"])) == 2:
            minTime = float(line.split()[-1])
            break
    maxTime = minTime
    for i in range(len(lines)-1, -1, -1):
        line = lines[i]
        wordsSet = set(line.split())
        if len(wordsSet & set(["and", "time"])) == 2:
            maxTime = float(line.split()[-1])
            break
    return minTime, maxTime

#
#  getItemNames
#---------------
def getItemNames(lines, minTime):
    """ item名とpoint or cellを取得する"""
    itemsSet = set([])
    for line in lines:
        if line.find("time") > 0:
            words = line.split()
            if float(words[-1]) == minTime:
                if words[0] == "displacements":
                    itemsSet.add(("nodes", "displacements"))
                elif words[0] == "stresses":
                    itemsSet.add(("cells", "stresses"))
                elif words[0] == "temperatures":
                    itemsSet.add(("nodes", "temperatures"))
                elif words[0] == "heat" and words[1] == "flux":
                    itemsSet.add(("cells", "heat flux"))
            else:
                break
    #itemsをlistに変換
    items = list(itemsSet)
    items = list(map(lambda x: list(x), items))
    return items

#
#  getDataTableFromPointer
#--------------------------
def getDataTableFromPointer(lines, items, sp):
    """ 結果データをポインタipから取得する"""

    def getDisplacements(lines, si):
        data = []
        for i in range(si, len(lines)):
            line = lines[i]
            words = line.split()
            if len(words) == 4:
                nodeNo = int(words[0])
                disp = list(map(float, words[1:]))
                data.append([nodeNo] + disp)
            elif len(words) == 0:
                if len(data) > 0:
                    break
            else:
                break
        return data, i

    def getTemperature(line, si):
        data = []
        for i in range(si, len(lines)):
            line = lines[i]
            words = line.split()
            if len(words) == 2:
                nodeNo = int(words[0])
                temp = [float(words[1])]
                data.append([nodeNo] + temp)
            elif len(words) == 0:
                if len(data) > 0:
                    break
            else:
                break
        return data, i

    def getStresses(line, si):
        data = []
        for i in range(si, len(lines)):
            line = lines[i]
            words = line.split()
            if len(words) == 8:
                elmNo = int(words[0])
                stress = list(map(float, words[2:]))
                data.append([elmNo] + stress)
            elif len(words) == 0:
                if len(data) > 0:
                    break
            else:
                break
        data = getAverageData(data)
        return data, i

    def getAverageData(data):
        """ 同じelmNoの平均(積分点の平均）を取得して返す"""
        newData = []
        currNo = data[0][0]
        nVers = len(data[0][1:])
        sameData = []
        for lineData in data:
            elmNo = lineData[0]
            if currNo == elmNo:
                sameData.append(lineData[1:])
            else:
                aveData = [0 for i in range(nVers)]
                for vals in sameData:
                    for i in range(len(vals)):
                        aveData[i] += vals[i] / nVers
                newData.append([currNo] + aveData)
                sameData = []
                currNo = elmNo
        if len(sameData) > 0:
            aveData = [0 for i in range(nVers)]
            for vals in sameData:
                for i in range(nVers):
                    aveData[i] += vals[i] / nVers
            newData.append([currNo] + aveData)
        return newData

    def getHeatFlux(line, si):
        data = []
        for i in range(si, len(lines)):
            line = lines[i]
            words = line.split()
            if len(words) == 5:
                elmNo = int(words[0])
                flux = list(map(float, words[2:]))
                data.append([elmNo] + flux)
            elif len(words) == 0:
                if len(data) > 0:
                    break
            else:
                break
        return data, i

    getNamesSet = list(map(lambda x: x[1], items))
    getNamesSet = list(map(lambda x: x.split()[0], getNamesSet))
    timeStep = 0.0
    ip = -1
    i = sp
    while i < len(lines):
        line = lines[i]
        words = line.split()
        if len(words) > 0:
            if words[0] in getNamesSet:
                ip = i
                break
        i += 1
    if ip < 0:
        return ip, timeStep, []
    #tableDataの読み込み
    timeStep = line.split()[-1]
    #print("  step:" + timeStep + "を取得")
    i = ip
    ip = -1
    while i < len(lines):
        line = lines[i]
        words = line.split()
        if len(words) > 0:
            if words[0] in getNamesSet and words[-1] == timeStep:
                #dataを取得
                i += 1
                if words[0] == "displacements":
                    data, i = getDisplacements(lines, i)
                elif words[0] == "stresses":
                    data, i = getStresses(line, i)
                elif words[0] == "temperatures":
                    data, i = getTemperature(line, i)
                elif words[0] == "heat" and words[1] == "flux":
                    data, i = getHeatFlux(line, i)
                ip = i
                #結果データをitemsに付加
                for ii in range(len(items)):
                    if items[ii][1] == words[0]:
                        if len(items[ii]) == 2:
                            items[ii].append(data)
                        else:
                            items[ii][2] += data
                    elif items[ii][1].split() == words[:2]:
                        if len(items[ii]) == 2:
                            items[ii].append(data)
                        else:
                            items[ii][2] += data
            else:
                break
                ip = i
        i += 1
    return ip, timeStep, items

#
#  getFrdDataTableFromPointer
#------------------------------
def getFrdDataTableFromPointer(lines, items, sp):
    """ frdファイルから結果データをポインタipから取得する"""

    def getStresses(line, si):
        data = []
        for i in range(si, len(lines)):
            line = lines[i]
            words = line.split()
            if words[0] == "-1":
                nodeNo = int(line[3:13])
                sigmas = []
                for ii in range(6):
                    ps = 13 + ii * 12
                    sigma = float(line[ps:ps+12])
                    sigmas.append(sigma)
                data.append([nodeNo] + sigmas)
            else:
                break
        return data, i

    getNamesSet = list(map(lambda x: x[1], items))
    getNamesSet = list(map(lambda x: x.split()[0], getNamesSet))
    timeStep = 0.0
    ip = -1
    i = sp
    while i < len(lines):
        line = lines[i]
        words = line.split()
        if len(words) > 1:
            if words[1] in getNamesSet:
                ip = i
                break
        i += 1
    if ip < 0:
        return ip, timeStep, []
    #tableDataの読み込み
    timeStep = float(lines[i-1].split()[2])
    #pointerをデータの頭に進める
    i = ip + 7
    ip = -1
    while i < len(lines):
        line = lines[i]
        words = line.split()
        if len(words) > 0:
            if words[0] == "-1":
                #dataを取得
                data, i = getStresses(line, i)
                ip = i
                #結果データをitemsに付加
                for ii in range(len(items)):
                    if items[ii][1] == "STRESS":
                        items[ii][2] = data
                        break
            else:
                break
        i += 1
    return ip, timeStep, items

#
#  addMisesStress
#------------------
def addMisesStress(itemsData):
    """ stressの項目があれば、misesStressを追加する"""
    addData = []
    for itemData in itemsData:
        sect, item, data = itemData
        if item == "stresses" or item == "STRESS":
            newData = []
            for No, xx, yy, zz, xy, yz, xz in data:
                C1 = ((xx - yy)**2 + (yy - zz)**2 + (zz - xx)**2) / 2.0
                C2 = 3.0 * (xy**2 + yz**2 + xz**2)
                sm = (C1 + C2)**0.5
                newData.append([No, sm])
            addData.append([sect, "misesStresses", newData])
    return addData

#
#  addElementType
#-----------------
def addElementType(neDicts, itemsData):
    """ 要素typeのfieldを作成する"""
    nodeDict, elementDict = neDicts
    data = []
    elmNos = elementDict.keys()
    for elmNo in elmNos:
        elType = elementDict[elmNo]["type"]
        newNo = elementDict[elmNo]["newNo"]
        vtkTypeNo = int(cm.aba2vtk_elDict[elType][0])
        data.append([elmNo, vtkTypeNo])
    addData = [["cells", "cellType", data]]
    return addData

#
#  getStepNameOfVtk
#-------------------
def getStepNameOfVtk(timeStep, maxTime):
    """ vtkFileのtimeStepを作成する。"""
    maxTimeStr = "%f" % maxTime
    timeStepStr =  "%f" % timeStep
    nMax = len(maxTimeStr.split(".")[0])
    nTime = len(timeStepStr.split(".")[0])
    nAddZero = nMax - nTime
    stepName = "0"*nAddZero + timeStepStr.replace(".", "")
    return stepName

#
#  getTranformNodes
#-------------------
def getTranformNodes(meshHeaderData, nset):
    """ transformするnetのnodeNoを取得する"""
    nodeNosSet = set([])
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*NODE":
            name = pyFistr.getValName(header, "NSET")
            if name == nset:
                nodeNos = set(map(lambda x: x[0], data))
                nodeNosSet = nodeNosSet | nodeNos
        elif words[0] == "*NSET":
            name = pyFistr.getValName(header, "NSET")
            if name == nset:
                nodeNosSet = nodeNosSet | set(data)
    return nodeNosSet

#
#  transformDisp
#------------------
def transformDisp(dispTable, transforms, neDicts):
    """ 座標変換（transform）"""
    for transform in transforms:
        tType, nset, a, b, transNodeNos = transform
        nodeDict, _elementDict = neDicts
        if tType == "C":
            print(_("円筒座標の変換 ") + nset)
            #円筒座標を変換する
            for i in range(len(dispTable)):
                vals = dispTable[i]
                nodeNo = vals[0]        #nodeNo
                disp = vals[1:]         #変位
                edgeZ = [a, b]          #中心軸
                #対象のNodeNo？
                if nodeNo in transNodeNos:
                    #座標変換
                    loc = nodeDict[nodeNo]["loc"]
                    #newDisp = transformCylindDisp(edgeZ, loc, disp)
                    newDisp = geo.convDispCtoD(loc, edgeZ, disp)
                    dispTable[i] = [nodeNo] + newDisp
    return dispTable

#
#  checkTransformDisp
#----------------
def checkTransformDisp(itemsData, transforms, neDicts):
    """ 座標変換をチェック修正する"""
    for i in range(len(itemsData)):
        if itemsData[i][1] == "displacements":
            dispTable = itemsData[i][2]
            dispTable = transformDisp(dispTable, transforms, neDicts)
            itemsData[i][2] = dispTable
    return itemsData

#
#  createVtuLines
#-----------------
def createVtuLines(itemsData, neDicts):
    """ vtu形式に変換する"""
    nodeDict, elementDict = neDicts
    #headerを作成
    lines = ['<?xml version="1.0"?>\n']
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0">\n']
    lines += ["<UnstructuredGrid>\n"]
    #必要なnodeNoを取得
    nodes = []
    for nodeNo in nodeDict.keys():
        if "newNo" in nodeDict[nodeNo].keys():
            nodes.append(nodeNo)
    nNodes = len(nodes)
    #必要なelmNoを取得し、vtk要素No（newNo）順に並べ変える
    elms = []
    for elmNo in elementDict.keys():
        if "newNo" in elementDict[elmNo].keys():
            newNo = elementDict[elmNo]["newNo"]
            elms.append([newNo, elmNo])
    elms.sort()
    elms = list(map(lambda x: x[1], elms))
    nElms = len(elms)
    #node数、cell数を作成
    lines += ['<Piece NumberOfPoints="' + str(len(nodes)) + '" NumberOfCells="' + str(len(elms)) + '">\n']
    #NODEを作成
    lines += ["<Points>\n"]
    lines += ['<DataArray type="Float32" NumberOfComponents="3" format="ascii">\n']
    for nodeNo in nodes:
        loc = nodeDict[nodeNo]["loc"]
        line = " ".join(list(map(str, loc))) + "\n"
        lines += [line]
    lines += ['</DataArray>\n']
    lines += ['</Points>\n']
    #elementを作成
    lines += ['<Cells>\n']
    lines += ['<DataArray type="Int32" Name="connectivity" format="ascii">\n']
    #  connectivity
    for elmNo in elms:
        abaNodes = elementDict[elmNo]["nodes"]
        abaType = elementDict[elmNo]["type"]
        newNodes = []
        for nodeNo in abaNodes:
            newNo = nodeDict[nodeNo]["newNo"]
            newNodes.append(newNo)
        #node順を修正
        name, newNodes = cm.aba2vtk_el(abaType, newNodes)
        line = " ".join(list(map(str, newNodes))) + "\n"
        lines += [line]
    lines += ['</DataArray>\n']
    #  offsets
    lines += ['<DataArray type="Int32" Name="offsets" format="ascii">\n']
    offsets = []
    offset = 0
    for elmNo in elms:
        abaNodes = elementDict[elmNo]["nodes"]
        offset += len(abaNodes)
        offsets.append(offset)
    line = " ".join(list(map(str, offsets))) + "\n"
    lines += [line]
    lines += ['</DataArray>\n']
    #  types
    lines += ['<DataArray type="UInt8" Name="types" format="ascii">\n']
    types = []
    for elmNo in elms:
        elmType = elementDict[elmNo]["type"]
        vtkType, n, func = cm.aba2vtk_elDict[elmType]
        types.append(vtkType)
    line = " ".join(types) + "\n"
    lines += [line]
    lines += ['</DataArray>\n']
    lines += ['</Cells>\n']
    #pointDataを作成
    lines += ['<PointData>\n']
    for itemData in itemsData:
        if itemData[0] == "nodes":
            varName = itemData[1]
            data = itemData[2]
            varData = [ [] for i in range(nNodes)]
            for lineData in data:
                nodeNo = lineData[0]
                vals = lineData[1:]
                newNo = nodeDict[nodeNo]["newNo"]
                varData[newNo] = vals
            nCompo = str(len(varData[0]))
            lines += ['<DataArray type="Float32" Name="' + varName + '" NumberOfComponents="' + nCompo + '" format="ascii">\n']
            for vals in varData:
                line = " ".join(list(map(str, vals))) + "\n"
                lines += [line]
            lines += ['</DataArray>\n']
    lines += ['</PointData>\n']
    #cellDataを作成
    lines += ['<CellData>\n']
    for itemData in itemsData:
        if itemData[0] == "cells":
            varName = itemData[1]
            data = itemData[2]
            varData = [ [] for i in range(nElms)]
            for lineData in data:
                elmNo = lineData[0]
                vals = lineData[1:]
                newNo = elementDict[elmNo]["newNo"]
                varData[newNo] = vals
            nCompo = str(len(varData[0]))
            lines += ['<DataArray type="Float32" Name="' + varName + '" NumberOfComponents="' + nCompo + '" format="ascii">\n']
            for vals in varData:
                line = " ".join(list(map(str, vals))) + "\n"
                lines += [line]
            lines += ['</DataArray>\n']
    lines += ['</CellData>\n']
    lines += ['</Piece>\n']
    lines += ['</UnstructuredGrid>\n']
    lines += ['</VTKFile>\n']
    return lines

#------------
#  createVtu
#------------
def createVtu(fileHeader, nConv):
    """ vtuを作成する"""
    #inpFileの読み込み
    inpFile = fileHeader + ".inp"
    f = open(inpFile, encoding="utf-8"); lines = f.readlines(); f.close()
    abaHeaderData = cm.getAbaHeaderNumData(lines)
    #座標変換内容の読み込み（transformの内容）
    transforms = getMultiTransformCont(abaHeaderData)
    #辞書作成
    neDicts = createNodeElementDict(abaHeaderData)
    #辞書にvtkのnewNoを追加
    neDicts = getNewNumberForVtk(neDicts)
    #結果fileを取得
    resFile = fileHeader + ".dat"
    f = open(resFile, encoding="utf-8"); lines = f.readlines(); f.close()
    minTime, maxTime = getMinMaxTimeStep(lines)
    #変換する時間を取得
    if nConv < 0:
        dTime = 0       #全て変換
    else:
        dTime = (maxTime - minTime) / nConv
    #frd fileを取得(nodeDataを取得)
    frdFile = fileHeader + ".frd"
    f = open(frdFile, encoding="utf-8"); fLines = f.readlines(); f.close()
    #項目（displacement等）を取得
    items = getItemNames(lines, minTime)
    fItems = [["nodes", "STRESS", []]]
    #結果データを取得
    cTime = minTime
    ip = 0
    ipf = 0
    loop = True
    while loop:
        newItems = items[:]
        newItemsF = fItems[:]
        ip, timeStep, itemsData = getDataTableFromPointer(lines, newItems, ip)
        ipf, timeStepF, itemsDataF = getFrdDataTableFromPointer(fLines, newItemsF, ipf)
        if ip < 0 or len(itemsData) == 0:
            break
        if float(timeStep) >= cTime - cTime*1e-5:
            cTime += dTime
            print("step:" + timeStep + _("  変換中..."))
            itemsData += itemsDataF
            addData = addMisesStress(itemsData)
            itemsData += addData
            addData = addElementType(neDicts, itemsData)
            itemsData += addData
            stepName = getStepNameOfVtk(float(timeStep), maxTime)
            if len(transforms) > 0:
                #座標変換
                itemsData = checkTransformDisp(itemsData, transforms, neDicts)    
            #vtuファイル変換
            vtuLines = createVtuLines(itemsData, neDicts)
            vtuFile = fileHeader + "_" + stepName + ".vtu"
            f = open(vtuFile, "w", encoding="utf-8"); f.writelines(vtuLines); f.close()
    print(_("calculixの結果をvtuに変換しました。"))

#
#  getArguments
#--------------
def getArguments():
    """ 引数を取得する。"""
    fileHeader = ""
    nConv = -1
    i = 1
    while i < len(sys.argv):
        val = sys.argv[i]
        if val == "-i":
            i += 1
            fileHeader = sys.argv[i]
        elif val == "-n":
            i += 1
            nConv = sys.argv[i]
        i += 1
    return fileHeader, nConv

#
#  printHelp
#------------
def printHelp():
    cont  = "------------ createAbaVtu.py ----------------------------\n"
    cont += "calculixの結果データをvtu形式に変換する。\n"
    cont += "inpファイルとdatファイルからvtu形式に変換する。\n"
    cont += "<使い方>\n"
    cont += "  createAbaVtu.py [option]\n"
    cont += "  [option]\n"
    cont += "    -i <headerFile>    inpファイルのheader名\n"
    cont += "    -n <nConv>         変換するvtuファイル数\n"
    cont += "                       省略可（省略時は、全て変換する）\n"
    cont += "    -h, --help:        helpを出力\n"
    cont += "<使用例>\n"
    cont += "  createAbaVtu.py model  :model.inp, model.datから変換する\n"
    cont += "                         :model.vtuを作り出す。\n"
    print(cont)


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    #_ = gettext.gettext

    fileHeader, nConv = getArguments()
    if os.path.exists(fileHeader + ".inp") == False:
        printHelp()
        exit()
    else:
        try:
            nConv = int(nConv)
        except:
            printHelp()
            exit()
        createVtu(fileHeader, nConv)
