#!/usr/bin/python3
#  coding: utf-8
#
#   createAbaBeamVtu.py
#
#       calculixのbeam要素（B31,B32,B31R,B32R）の形状と変位をvtuに変換する
#
#   21/02/06    新規作成
#      02/12    バネ要素とバネ追加の為に追加した節点を変換しない様に修正
#      03/03    transform,type=C（円筒座標の座標変換を追加)
#               最終のtransformのみ取得して変換する。
#      03/19    createVtkLines:要素定義（nodeの並び順）を修正。
#               （中間節点を最後に修正）
#               getNewNumberForVtk:辞書作成方法を修正
#   24/07/07    openにencoding="utf-8"を追加
#   25/01/03    createBeamVtu:読み込んだinpファイルを大文字に変換を追加
#      01/18    createNodeElementDict:*NSETcardからもNALLを取得するように修正
#

import os, sys
import math
import pyFistr
import convertMesh as cm
import geometryFunc as geo


convAbaTypeToVtkDict = {
    #abaType  vtkType
    "B31":      "3",
    "B31R":     "3",
    "B32":      "21",
    "B32R":     "21"
    }

#
#  getTransformCont
#-------------------
def getTransformCont(meshHeaderData):
    """ 座標変換（transform）しているか確認"""
    trans = ["", "", "", ""]
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*TRANSFORM":
            tType = pyFistr.getValName(header, "TYPE")
            nset = pyFistr.getValName(header, "NSET")
            line = data[0]
            ws = pyFistr.deleteSp(line).split(",")
            if len(ws) == 6:
                try:
                    ws = list(map(float, ws))
                    a = ws[:3]
                    b = ws[3:]
                    trans = [tType, nset, a, b]
                except:
                    pass                        
    if trans[0] == "":
        trans = []
    return trans

#
#  getMultiTransformCont
#-------------------------
def getMultiTransformCont(meshHeaderData):
    """ 座標変換（transform）しているか確認し、全てのtransfoamを取得する"""
    transforms = []
    trans = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*TRANSFORM":
            tType = pyFistr.getValName(header, "TYPE")
            nset = pyFistr.getValName(header, "NSET")
            #nodesSet = getTranformNodes(meshHeaderData, nset)
            line = data[0]
            ws = pyFistr.deleteSp(line).split(",")
            if len(ws) == 6:
                try:
                    ws = list(map(float, ws))
                    a = ws[:3]
                    b = ws[3:]
                    nodesSet = getTranformNodes(meshHeaderData, nset)
                    trans = [tType, nset, a, b, nodesSet]
                    transforms.append(trans)
                except:
                    pass                        
    return transforms

#
#  createNodeElementDict
#------------------------
def createNodeElementDict(meshHeaderData):
    """ nodeDict, elementDictを作成する"""
    nodeDict = {}
    elementDict = {}
    for header, data in meshHeaderData:
        words = cm.deleteSp(header).split(",")
        if words[0] == "*NODE":
            nset = pyFistr.getValName(header, "NSET")
            for nodeLine in data:
                nodeNo = nodeLine[0]
                loc = nodeLine[1:]
                nodeDict[nodeNo] = {"loc":loc, "elms":set([]), "nset":nset}
        elif words[0] == "*NSET":
            nset = pyFistr.getValName(header, "NSET")
            if nset == "NALL":
                for nodeNo in data:
                    nodeDict[nodeNo]["nset"] = nset
    for header, data in meshHeaderData:
        words = cm.deleteSp(header).split(",")
        if words[0] == "*ELEMENT":
            elmType = pyFistr.getValName(header, "TYPE")
            for elmLine in data:
                elmNo = elmLine[0]
                nodes = elmLine[1:]
                elementDict[elmNo] = {"nodes":nodes, "type":elmType}
                for nodeNo in nodes:
                    nodeDict[nodeNo]["elms"].add(elmNo)
    return nodeDict, elementDict

#
#  getNewNumberForVtk
#---------------------
def getNewNumberForVtk(neDicts):
    """ vtk用のNoを辞書内に取得する"""
    nodeDict, elementDict = neDicts
    nodes = list(nodeDict.keys())
    nodes.sort()
    n = 0
    for i in range(len(nodes)):
        nodeNo = nodes[i]
        if nodeDict[nodeNo]["nset"] == "NALL":
            nodeDict[nodeNo]["newNo"] = n
            n += 1
    elms = list(elementDict.keys())
    elms.sort()
    beamsSet = set(convAbaTypeToVtkDict.keys())
    n = 0
    for i in range(len(elms)):
        elmNo = elms[i]
        if elementDict[elmNo]["type"] in beamsSet:
            elementDict[elmNo]["newNo"] = n
            n += 1
    return nodeDict, elementDict

#
#  getMinMaxTimeStep
#--------------------
def getMinMaxTimeStep(lines):
    """ minMaxのtimeStepを取得する"""
    minTime = 0.0
    for line in lines:
        if line.find("displacements") >= 0:
            minTime = float(line.split()[-1])
            break
    maxTime = minTime
    for i in range(len(lines)-1, -1, -1):
        line = lines[i]
        if line.find("displacements") >= 0:
            maxTime = float(line.split()[-1])
            break
    return minTime, maxTime

#
#  getLatestDispTable
#---------------------
def getLatestDispTable(lines, ip):
    timeStep = 0.0
    ip = -1
    for i in range(len(lines)-1, -1, -1):
        line = lines[i]
        if line.find("displacements") >= 0:
            ip = i
            break
    print(_("displacementを取得"))
    print("  " + line, end="")
    timeStep = line.split()[-1]
    disps = []
    for i in range(ip+1, len(lines), 1):
        line = lines[i]
        words = line.split()
        if len(words) == 4:
            try:
                nodeNo = int(words[0])
                disp = list(map(float, words[1:]))
                dispLine = [nodeNo] + disp
                disps.append(dispLine)
            except:
                break
        elif len(words) > 4:
            break
    return i, timeStep, disps

#
#  getDispTableFromPointer(lines, ip)
#--------------------------
def getDispTableFromPointer(lines, sp):
    """ 変位データをポインタipから取得する"""
    timeStep = 0.0
    ip = -1
    i = sp
    while i < len(lines):
        line = lines[i]
        if line.find("displacements") >= 0:
            ip = i
            break
        i += 1
    if ip < 0:
        return ip, timeStep, []
    #tableDataの読み込み
    print(_("displacementを取得"))
    print("  " + line, end="")
    timeStep = line.split()[-1]
    disps = []
    for i in range(ip+1, len(lines), 1):
        line = lines[i]
        words = line.split()
        if len(words) == 4:
            try:
                nodeNo = int(words[0])
                disp = list(map(float, words[1:]))
                dispLine = [nodeNo] + disp
                disps.append(dispLine)
            except:
                break
        elif len(words) > 4:
            break
    return i, timeStep, disps

#
#  getStepNameOfVtk
#-------------------
def getStepNameOfVtk(timeStep, maxTime):
    """ vtkFileのtimeStepを作成する。"""
    maxTimeStr = "%f" % maxTime
    timeStepStr =  "%f" % timeStep
    nMax = len(maxTimeStr.split(".")[0])
    nTime = len(timeStepStr.split(".")[0])
    nAddZero = nMax - nTime
    stepName = "0"*nAddZero + timeStepStr.replace(".", "")
    return stepName

#
#  getTranformNodes
#-------------------
def getTranformNodes(meshHeaderData, nset):
    """ transformするnetのnodeNoを取得する"""
    nodeNosSet = set([])
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*NODE":
            name = pyFistr.getValName(header, "NSET")
            if name == nset:
                nodeNos = set(map(lambda x: x[0], data))
                nodeNosSet = nodeNosSet | nodeNos
        elif words[0] == "*NSET":
            name = pyFistr.getValName(header, "NSET")
            if name == nset:
                nodeNosSet = nodeNosSet | set(data)
    return nodeNosSet

# #
# #  transformCylindDisp
# def transformCylindDisp(edgeZ, loc, disp):
#     """ 円筒座標の座標変換"""
#     #円筒座標のZ軸を取得
#     a, b = edgeZ
#     vecZ = geo.vector(a, b)
#     nVecZ = geo.normal(vecZ)
#     #円筒座標の変位を取得
#     radius, rot, z = disp
#     #変換後の座標を定義
#     newDisp = [0.0, 0.0, 0.0]
#     #半径方向を変換--------------------------
#     edge = [a, b]
#     point = loc
#     #z軸からの方向ベクトル
#     vec = geo.vecEdgeToPoint(edge, point)
#     nVec = geo.normal(vec)
#     #半径方向の変位取得
#     xyz = geo.vecByC(nVec, radius)
#     newDisp = geo.addVec(newDisp, xyz)
#     #円周方向---------------------------------
#     r = geo.mag(vec)
#     #円周方向変位からY方向高さを取得
#     h = r * math.sin(rot)
#     B = r * math.cos(rot)
#     #Y方向高さ,幅方向のベクトルを取得
#     vecHH = geo.crossProduct(vec, nVecZ)
#     nVecH = geo.normal(vecHH)
#     vecH = geo.vecByC(nVecH, h)
#     vecB = geo.vecByC(nVec, B)
#     #円周方向の変位を取得
#     vecBH = geo.addVec(vecB, vecH)
#     xyz = geo.vector(vec, vecBH)
#     newDisp = geo.addVec(newDisp, xyz)
#     #高さ方向-------------------------------
#     xyz = geo.vecByC(nVecZ, z)
#     newDisp = geo.addVec(newDisp, xyz)
#     return newDisp

#
#  transformDisp
#----------------
def transformDisp(dispTable, transforms, neDicts):
    """ 座標変換（transform）"""
    for transform in transforms:
        tType, nset, a, b, transNodeNos = transform
        nodeDict, _elementDict = neDicts
        if tType == "C":
            print(_("円筒座標の変換 ") + nset)
            #円筒座標を変換する
            for i in range(len(dispTable)):
                vals = dispTable[i]
                nodeNo = vals[0]        #nodeNo
                disp = vals[1:]         #変位
                edgeZ = [a, b]          #中心軸
                #対象のNodeNo？
                if nodeNo in transNodeNos:
                    #座標変換
                    loc = nodeDict[nodeNo]["loc"]
                    #newDisp = transformCylindDisp(edgeZ, loc, disp)
                    newDisp = geo.convDispCtoD(loc, edgeZ, disp)
                    dispTable[i] = [nodeNo] + newDisp
            #print("  NSET:" + nset + " node数:" + str(len(transNodeNos)) + " 座標変換しました。")
    return dispTable

#
#  setDispInNodeDict
#---------------------
def setDispInNodeDict(tableData, neDicts):
    """ 変位をnodeDictに追加"""
    nodeDict, elementDict = neDicts
    for line in tableData:
        nodeNo = line[0]
        disp = line[1:]
        nodeDict[nodeNo]["disp"] = disp
    return nodeDict, elementDict

#
#  getElmTypeDict
#-----------------
def getElmTypeDict(neDicts):
    nodeDict, elementDict = neDicts
    elmTypeDict = {}
    for elmNo in elementDict.keys():
        elmType = elementDict[elmNo]["type"]
        if elmType in elmTypeDict.keys():
            elmTypeDict[elmType].append(elmNo)
        else:
            elmTypeDict[elmType] = []
    return elmTypeDict

#
#  createVtkLines
#------------------
def createVtkLines(neDicts):
    nodeDict, elementDict = neDicts
    #headerを作成
    lines = ['<?xml version="1.0"?>\n']
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0">\n']
    lines += ["<UnstructuredGrid>\n"]
    #必要なnodeNoを取得（バネ要素を除く）
    allNodes = list(nodeDict.keys())
    nodes = []
    for nodeNo in allNodes:
        #変位を読み取っていないnode（バネ）を除く
        if "disp" in nodeDict[nodeNo].keys():
            nodes.append(nodeNo)
    nodes.sort()
    #必要な要素を取得（バネ要素を除く）
    allElms = list(elementDict.keys())
    elms = []
    for elm in allElms:
        #typeを確認し、辞書以外の要素は除く
        if elementDict[elm]["type"] in convAbaTypeToVtkDict.keys():
            elms.append(elm)
    elms.sort()
    lines += ['<Piece NumberOfPoints="' + str(len(nodes)) + '" NumberOfCells="' + str(len(elms)) + '">\n']
    #NODEを作成
    lines += ["<Points>\n"]
    lines += ['<DataArray type="Float32" NumberOfComponents="3" format="ascii">\n']
    for nodeNo in nodes:
        loc = nodeDict[nodeNo]["loc"]
        line = " ".join(list(map(str, loc))) + "\n"
        lines += [line]
    lines += ['</DataArray>\n']
    lines += ['</Points>\n']
    #elementを作成
    lines += ['<Cells>\n']
    lines += ['<DataArray type="Int32" Name="connectivity" format="ascii">\n']
    #  connectivity
    for elmNo in elms:
        abaNodes = elementDict[elmNo]["nodes"]
        newNodes = []
        for nodeNo in abaNodes:
            newNo = nodeDict[nodeNo]["newNo"]
            newNodes.append(newNo)
        if len(newNodes) == 3:
            #2次要素の場合、順番を修正（中間要素は最後）
            newNodes = [newNodes[0], newNodes[2], newNodes[1]]
        line = " ".join(list(map(str, newNodes))) + "\n"
        lines += [line]
    lines += ['</DataArray>\n']
    #  offsets
    lines += ['<DataArray type="Int32" Name="offsets" format="ascii">\n']
    offsets = []
    offset = 0
    for elmNo in elms:
        abaNodes = elementDict[elmNo]["nodes"]
        offset += len(abaNodes)
        offsets.append(offset)
    line = " ".join(list(map(str, offsets))) + "\n"
    lines += [line]
    lines += ['</DataArray>\n']
    #  types
    lines += ['<DataArray type="UInt8" Name="types" format="ascii">\n']
    types = []
    for elmNo in elms:
        elmType = elementDict[elmNo]["type"]
        vtkType = convAbaTypeToVtkDict[elmType]
        types.append(vtkType)
    line = " ".join(types) + "\n"
    lines += [line]
    lines += ['</DataArray>\n']
    lines += ['</Cells>\n']
    #disp
    lines += ['<PointData>\n']
    lines += ['<DataArray type="Float32" Name="DISPLACEMENT" NumberOfComponents="3" format="ascii">\n']
    for nodeNo in nodes:
        disp = nodeDict[nodeNo]["disp"]
        line = " ".join(list(map(str, disp))) + "\n"
        lines += [line]
    lines += ['</DataArray>\n']
    lines += ['</PointData>\n']
    lines += ['</Piece>\n']
    lines += ['</UnstructuredGrid>\n']
    lines += ['</VTKFile>\n']
    return lines

#-----------------
#  createBeamVtu
#-----------------
def createBeamVtu(fileHeader):
    """ calculixのbeam要素とその変位をvtu形式に変換する"""
    inpFile = fileHeader + ".inp"
    f = open(inpFile, encoding="utf-8"); lines = f.readlines(); f.close()
    #大文字に変換
    lines = list(map(lambda x: x.upper(), lines))
    meshHeaderData = cm.getAbaHeaderNumData(lines)
    #transform（座標変換）内容を取得
    #transform = getTransformCont(meshHeaderData)
    transforms = getMultiTransformCont(meshHeaderData)
    # if len(transform) != 0:
    #     #座標変換
    #     nset = transform[1]
    #     #変換対象のnodeNoを取得
    #     transNodesSet = getTranformNodes(meshHeaderData, nset)
    #     transform.append(transNodesSet)
    #辞書作成
    neDicts = createNodeElementDict(meshHeaderData)
    neDicts = getNewNumberForVtk(neDicts)
    #結果fileを取得
    resFile = fileHeader + ".dat"
    f = open(resFile, encoding="utf-8"); lines = f.readlines(); f.close()
    #結果データのminMaxTimeを取得する
    minTime, maxTime = getMinMaxTimeStep(lines)
    #結果データを取得
    ip = 0
    #ip, timeStep, dispTable = getLatestDispTable(lines, ip)
    loop = True
    while loop:
        ip, timeStep, dispTable = getDispTableFromPointer(lines, ip)
        if ip < 0 or len(dispTable) == 0:
            break
        #stepName = "%f" % float(timeStep)
        stepName = getStepNameOfVtk(float(timeStep), maxTime)
        if len(transforms) != 0:
            # #座標変換
            # nset = transform[1]
            # #変換対象のnodeNoを取得
            # transNodesSet = getTranformNodes(meshHeaderData, nset)
            # transform.append(transNodesSet)
            #座標変換
            dispTable = transformDisp(dispTable, transforms, neDicts)    
        neDicts = setDispInNodeDict(dispTable, neDicts)
        #vtuファイル作成
        vtuLines = createVtkLines(neDicts)
        vtuFile = fileHeader + "_" + stepName + ".vtu"
        f = open(vtuFile, "w", encoding="utf-8"); f.writelines(vtuLines); f.close()
    print(_("calculixの結果をvtuに変換しました。"))

if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))

    fileHeader = sys.argv[1]
    createBeamVtu(fileHeader)
