#!/usr/bin/python3
#  coding: utf-8
#
#   beam641To611.py
#
#       beam要素641を611に変換する。
#
#   21/01/30    新規作成
#   22/03/15    国際化
#   23/04/13    "FistrModel.cnt"をhecmw_ctrl.datから取得する様に修正
#   24/07/07    openにencoding="utf-8"を追加
#

import os, sys
import pyFistr


#
#  createNodeElementDict
def createNodeElementDict(meshHeaderData):
    """ nodeDict, elementDictを作成する"""
    nodeDict = {}
    elementDict = {}
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            data = headerData[1]
            for node in data:
                nodeNo = node[0]
                loc = node[1:]
                nodeDict[nodeNo] = [loc, []]
        elif words[0] == "!ELEMENT":
            data = headerData[1]
            elmType = pyFistr.getValName(header, "TYPE")
            for elm in data:
                elmNo = elm[0]
                nodes = elm[1:]
                elementDict[elmNo] = [nodes, elmType]
                for nodeNo in nodes:
                    nodeDict[nodeNo][1].append(elmNo)
    return nodeDict, elementDict

#
#  isOnlyBeam641
#----------------
def isOnlyBeam641(meshHeaderData):
    flag = True
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            elmType = pyFistr.getValName(header, "TYPE")
            if elmType != "641":
                flag = False
                break
    return flag

#
#  deleteDummyNodes
#-------------------
def deleteDummyNodes(meshHeaderData, dicts):
    """ meshHeaderData内から、dummy要素を抽出し、削除する。"""
    nodeDict, elementDict = dicts
    #dummyNodeを取得
    dummyNodesSet = set([])
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            elmType = pyFistr.getValName(header, "TYPE")
            if elmType == "641":
                for elm in data:
                    dummyNode = elm[3:]
                    dummyNodesSet.update(dummyNode)
    #dummyNodeを削除
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        data = meshHeaderData[i][1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            newData = []
            for node in data:
                nodeNo = node[0]
                if not (nodeNo in dummyNodesSet):
                    newData.append(node)
            meshHeaderData[i][1] = newData
        elif words[0] == "!ELEMENT":
            newHeader = pyFistr.setValName(header, "TYPE=611")
            newData = []
            for elm in data:
                newElm = elm[:3]
                newData.append(newElm)
            meshHeaderData[i][0] = newHeader
            meshHeaderData[i][1] = newData
        elif words[0] == "!NGROUP":
            newData = []
            for nodeNo in data:
                if not (nodeNo in dummyNodesSet):
                    newData.append(nodeNo)
            meshHeaderData[i][1] = newData
    #headerDataを整形
    newHeaderData = []
    for headerData in meshHeaderData:
        header = headerData[0]
        data = headerData[1]
        words = pyFistr.deleteSp(header).split(",")
        if (words[0] == "!NODE" 
            or words[0] == "!ELEMENT" 
            or words[0] == "!NGROUP"):
            if len(data) != 0:
                newHeaderData.append(headerData)
        else:
            newHeaderData.append(headerData)
    return newHeaderData

#
#  remakeDummyCond
#------------------
def remakeDummyCond(contHeaderData):
    """ dummy_の境界条件を修正する。"""
    for i in range(len(contHeaderData)):
        header = contHeaderData[i][0]
        dataLines = contHeaderData[i][1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!BOUNDARY":
            for ii in range(len(dataLines)):
                line = dataLines[ii]
                if line[:len("dummy_")] == "dummy_":
                    ws = pyFistr.deleteSp(line).split(",")
                    ws[0] = ws[0][len("dummy_"):]
                    ws[1] = str(int(ws[1]) + 3)
                    ws[2] = str(int(ws[2]) + 3)
                    line = ", ".join(ws) + "\n"
                    dataLines[ii] = line
            contHeaderData[i] = [header, dataLines]
        elif words[0] == "!CLOAD":
            for ii in range(len(dataLines)):
                line = dataLines[ii]
                ws = pyFistr.deleteSp(line).split(",")
                if ws[0][:len("dummy_")] == "dummy_":
                    ws[0] = ws[0][len("dummy_"):]
                    ws[1] = str(int(ws[1]) + 3)
                    line = ", ".join(ws) + "\n"
                    dataLines[ii] = line
            contHeaderData[i] = [header, dataLines]
        elif words[0] == "!SPRING":
            for ii in range(len(dataLines)):
                line = dataLines[ii]
                ws = pyFistr.deleteSp(line).split(",")
                if ws[0][:len("dummy_")] == "dummy_":
                    ws[0] = ws[0][len("dummy_"):]
                    ws[1] = str(int(ws[1]) + 3)
                    line = ", ".join(ws) + "\n"
                    dataLines[ii] = line
            contHeaderData[i] = [header, dataLines]
    return contHeaderData

#
#  readFistrModelCnt
#---------------------
def readFistrModelCnt():
    currDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(currDir)
    fileName = currDir + os.sep + FistrModel_cnt
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    return lines 

#
#  writeFistrModelCnt
#---------------------
def writeFistrModelCnt(lines):
    currDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(currDir)
    fileName = currDir + os.sep + FistrModel_cnt
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  writeHeaderDataToCnt
#------------------------
def writeHeaderDataToCnt(headerData):
    lines = []
    for header in headerData:
        lines += [header[0]]
        lines += header[1]
    writeFistrModelCnt(lines)

#-------------------
#  convert641To611
#-------------------
def convert641To611():
    """ beam641を611に変換する"""
    #mshFileを修正
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    nodeDict, elementDict = createNodeElementDict(meshHeaderData)
    dicts = [nodeDict, elementDict]
    if isOnlyBeam641(meshHeaderData) == False:
        print(_("641以外の要素が存在するため、611に変換できません。"))
        print(_("変換できるのは、641のみで構成されたモデルです。"))
        return
    meshHeaderData = deleteDummyNodes(meshHeaderData, dicts)
    lines = pyFistr.createLinesFistrModelMshHeader(meshHeaderData)
    pyFistr.writeFistrModelMsh(lines)
    #cntFileを修正
    lines = readFistrModelCnt()
    contHeaderData = pyFistr.getHeaderData(lines)
    contHeaderData = remakeDummyCond(contHeaderData)
    writeHeaderDataToCnt(contHeaderData)
    print(_("要素Type:641を611に変更しました。"))


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.setenv("LOCALE_DIR"))
    #_ = gettext.gettext

    convert641To611()
