#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#   abaqusinp2fistrcnt.py
#
#       abaqus（inpファイル）の境界条件をfistr（cntファイル）に変換
#       線形静解析、熱伝導静解析が変換できる。
#
#       inpFile     ->          fistr(cntFile)
#   ------------------------------------------------------
#       *BOUNDARY               !BOUNDARY
#       *CLOAD                  !CLOAD
#       *CFLUX                  !CFLUX
#       *DLOAD, Px              !DLOAD, S
#       *FIXTEMP                !BOUNDARY, 11
#       *INITIAL CONDITIONS     !INITIAL CONDITION
#       *MATERIAL               !MATERIAL
#         *ELASTIC                !ELASTIC
#         *DENSITY                !DENSTY
#         *EXPANSION              !EXPANSION_COEFF
#         *SPECIFIC HEAT          !SPECIFIC_HEAT
#         *CONDUCTIVITY           !THERMAL_CONDUCTIVITY
#       *SPRING                 !SPRING
#       *TEMPERATURE            !TEMPERATURE
#       
#--------------------------
#   16/01/12    新規作成
#      01/16    完成
#      02/04    inpファイル読み込み時大文字に変換してファイル変換
#               大文字、小文字関係なくファイル変換可能
#      02/05    小文字のgroup名を残す様に修正
#   18/11/30    convNoToName:境界条件が直接節点No、要素Noで設定されている場合、
#               その数が10以上有る場合は、設定しないように修正。
#   19/05/08    python3用に書き換え
#   22/04/12    log表示修正（import logFileCreaterを追加）
#   23/04/13    getCntFileName:hecmw_ctrl.datからcntファイル名を取得を追加
#      04/17    getCntFileName:mshファイルの取得を追加
#      08/15    addIncludeHeaders:「*INCLUDE」の処理を追加
#   24/07/07    openにencoding="utf-8"を追加
#   25/01/21    abaqusinp2fistrcnt,getMaterialLines,addMatLines,writeMeshFile:
#               cnt,mshファイルに!MATERIAL追加。mshファイルに!SECTION追加。
#      01/24    convNoToName:直接nodeNo or elmNoで指定した条件の場合、50ヶ以下は
#               No毎にGrpを作成し設定する。50ヶを超えると、No毎に設定せず、
#               まとめて、!NGROUP or !EGROUPを作成する様に修正。
#      04/23    writeMeshFile:当分布の荷重設定時、材料が定義されなかった為、修正。
#               入力の引数（caseDir）を追加。
#

import os
import sys
import pyFistr

#  変換内容
def convCont():
    #         abaqus                 fistr
    cont = [["BOUNDARY",          "BOUNDARY"],          #FIXTEMPを含む
            ["CLOAD",             "CLOAD"],
            ["DLOAD",             "DLOAD"],
            ["SPRING",            "SPRING"],
            ["INITIAL CONDITIONS","INITIAL CONDITION"],
            ["TEMPERATURE",       "TEMPERATURE"],
            ["CFLUX",             "CFLUX"]]
    return cont

#  材料
def matCardDict():
    #         abaqus              fistr
    contDict = {"MATERIAL":      "MATERIAL",
                "ELASTIC":       "ELASTIC",
                "DENSITY":       "DENSITY",
                "EXPANSION":     "EXPANSION_COEFF",
                "SPECIFICHEAT":  "SPECIFIC_HEAT",   #SPECIFIC HEAT
                "CONDUCTIVITY":  "THERMAL_CONDUCTIVITY",
                "BEAMGENERALSECTION": "SECTION",     #BEAM GENERAL SECTION
                "BEAMSECTION":   "SECTION",          #BEAM SECTION
                "SHELLSECTION":  "SECTION",          #SHELL SECTION
                "SOLIDSECTION":  "SECTION"}          #SOLID SECTION
    return contDict

#  項目数
def nItemsInBC():
    #          BC名     組数 default値  対象
    cont = [["BOUNDARY", 3,  0.0,       "node"],
            ["CLOAD",    3,  0.0,       "node"],
            ["DLOAD",    1,  0.0,       "element"],
            ["SPRING",   3,  0.0,       "node"],
            ["INITIAL CONDITION", 1, 0.0, "node"],
            ["TEMPERATURE", 1, 0.0,     "node"],
            ["CFLUX",    1,  0.0,       "node"]]
    return cont

#  printMsg
#    print help
def printMsg():
    msg = """
<abaqusinp2fistrcnt.py>
abaqusのinpファイルから境界条件を読み取り、FrontISTRのcntファイルに書き込む。

<使い方>
abaqusinp2fistrcnt.py <inpファイル> <caseDir>

例：
abaqusinp2fistrcnt.py name.inp  test
abaqusinp2fistrcnt.py name  test
"""
    print (msg)

#
#  isConvLine
#    変換対象のlineを取得
def isConvLine(line):
    ans = False
    word = line.split(",")[0][1:].upper()
    commWord = " ".join(word.split())
    comms = convCont()
    for i in range(len(comms)):
        comm = comms[i]
        if commWord == comm[0]:
            ans = True
            break
    return ans

#  deleteSp
#    空白、CR,LFを削除
def deleteSp(line):
    newLine = ""
    for chara in line:
        if chara != " " and chara != "\n" and chara != "\r":
            newLine += chara
    return newLine

#
#  getBoundaryConditions
#    境界条件部を取得
def getBoundaryConditions(lines):
    getLines = []
    flag = 0
    for n in range(len(lines)):
        line = lines[n]
        if line[0] == "*":
            flag = 0
            if isConvLine(line) == True:
                if n+1 < len(lines) and lines[n+1][0] != "*":
                    #header部を取得
                    getLines.append(line)
                    flag = 1
        else:
            if flag == 1:
                #data部を取得
                getLines.append(line)
    #取得したlinesをprint
    i = 0
    while i < len(getLines):
        [bcLines, i] = get1BcLines(getLines, i)
        if len(bcLines) > 0:
            header = bcLines[0]
            dataLines = bcLines[1:]
            headerName = " ".join((header.split(",")[0]).split())
            print (" found " + headerName + " ", end=" ")
            for line in dataLines[:5]:
                print (deleteSp(line.split(",")[0]), end=" ")
            if len(dataLines) > 5:
                print ("...", end=" ")
            print
    return getLines

#
#  checkGrpNameInHeader
#    header内のgrpNameをチェック
#     header内には1ヶのgrpName
def checkGrpNameInHeader(lines):
    newLines = []
    i = 0
    while i < len(lines):
        ist = i
        [bc1Lines, i] = get1BcLines(lines, i)
        header = bc1Lines[0]
        dataLines = bc1Lines[1:]
        numLines = []
        nameLines = []
        grpNames = []
        for line in dataLines:
            words = deleteSp(line).split(",")
            try:
                a = int(words[0])
                numLines.append(line)
            except:
                nameLines.append(line)
                #grp名を取得
                if len(grpNames) == 0:
                    grpNames.append(words[0])
                else:
                    if grpNames[-1] != words[0]:
                        grpNames.append(words[0])
        #複数のgrp名があれば分割する
        if len(grpNames) > 1:
            #headerを含むblockを追加
            #  第1 block を作成
            bcLines = []
            bcLines += [header]
            for line in nameLines:
                if deleteSp(line).split(",")[0] == grpNames[0]:
                    bcLines += [line]
            bcLines += numLines
            #  第2 block 以降を作成
            for name in grpNames[1:]:
                bcLines += [header]
                for line in nameLines:
                    if deleteSp(line).split(",")[0] == name:
                        bcLines += [line]
            #newLinesに追加
            newLines += bcLines
        else:
            #そのままnewLinesに追加
            newLines += bc1Lines
    return newLines

#
#  convBoundaryConditions
#    境界条件名をFistr用に修正する
def convBoundaryConditions(lines):
    newLines = []
    #1文字目を修正
    for i in range(len(lines)):
        if lines[i][0] == "*":
            lines[i] = "!" + lines[i][1:]
    #内容を修正
    for i in range(len(lines)):
        if lines[i][0] == "!":
            words = deleteSp(lines[i]).split(",")
            words[0] = " ".join((lines[i].split(",")[0]).split()).upper()
            comms = convCont()
            for comm in comms:
                if words[0][1:] == comm[0]:
                    words[0] = "!" + comm[1]
            newLine = ", ".join(words) + "\n"
            newLines.append(newLine)
        else:
            newLines.append(lines[i])
    return newLines

#
#  convNoToName
#    node番号で指定しているものをgrpNameに修正する。
def convNoToName(lines):
    #直接nodeNoで設定しているnodeNoを取得
    createGrpNo = 0
    newGrpLines = []
    newBcLines = []
    unDefNodes = []
    unDefElements = []
    bcItems = nItemsInBC()
    i = 0
    while i < len(lines):
        [bc1Lines, i] = get1BcLines(lines, i)
        if len(bc1Lines) > 1:
            header = bc1Lines[0]
            dataLines = bc1Lines[1:]
            [bcName, dummy] = getHeaderGrpName(bc1Lines)
            #node or elementを取得
            cName = ""
            for bcItem in bcItems:
                if bcItem[0] == bcName:
                    cName = bcItem[3]
                    break
            #数字をnode or elementに修正
            if cName != "":
                #dataLinesが50行以下の場合は、処理する
                if len(dataLines) <= 50:
                    #節点No、要素Noを各々のgroup名に変換する
                    for n in range(len(dataLines)):
                        dataLine = dataLines[n]
                        words = deleteSp(dataLine).split(",")
                        try:
                            a = int(words[0])
                            words[0] = cName + "_" + words[0]
                            dataLines[n] = ", ".join(words) + "\n"
                            if cName == "node":
                                unDefNodes.append(a)
                            elif cName == "element":
                                unDefElements.append(a)
                        except:
                            pass
                    newBcLines += [header] + dataLines
                else:
                    #50以上の場合は、NGRPorEGRPを作成する
                    newGrpLines += createNewNEgrpLines(cName, createGrpNo, dataLines) 
                    createGrpNo += 1
            else:
                newBcLines += bc1Lines
        else:
            newBcLines += bc1Lines
    #定義するnodeNoを取得
    nodes = []
    for node in unDefNodes:
        flag = 0
        for getNode in nodes:
            if node == getNode:
                flag = 1
                break
        if flag == 0:
            nodes.append(node)
    #nodeで追加する行を作成
    addLines = []
    if len(nodes) > 0:
        for node in nodes:
            line = "!NGROUP, NGRP=node_" + str(node) + "\n"
            addLines.append(line)
            line = str(node) + "\n"
            addLines.append(line)
    #定義するelementNoを取得
    elements = []
    for element in unDefElements:
        flag = 0
        for getEl in elements:
            if element == getEl:
                flag = 1
                break
        if flag == 0:
            elements.append(element)
    #elementで追加する行を作成
    if len(elements) > 0:
        for el in elements:
            line = "!EGROUP, EGRP=element_" + str(el) + "\n"
            addLines.append(line)
            line = str(el) + "\n"
            addLines.append(line)
    #newGrpLinesを追加
    addLines += newGrpLines
    return [newBcLines, addLines]

#
#  createNewNEgrpLines
def createNewNEgrpLines(ne, createGrpNo, dataLines):
    """ dataLinesからNGRP,EGRP行を作成して返す"""
    lines = []
    #header行を作成
    if ne == "node":
        header = "!NGROUP, NGRP=unsetLoadNgrp_" + str(createGrpNo) + "\n"
    elif ne == "element":
        header = "!EGROUP, EGRP=unsetLoadEgrp_" + str(createGrpNo) + "\n"
    lines.append(header)
    #nodeNo, elementNoを取得
    grps = []
    for n in range(len(dataLines)):
        dataLine = dataLines[n]
        words = deleteSp(dataLine).split(",")
        try:
            neNo = int(words[0])
        except:
            neNo = -1
        if neNo > 0:
            #nodeNo or elmNoを取得
            grps.append(words[0])
    if len(grps) > 0:
        #grp内を整理して、linesに追加
        n = 8
        for i in range(0, len(grps), n):
            line = ", ".join(grps[i:i+n]) + "\n"
            lines.append(line)
    return lines

#
#  combBoundaryConditions
#    変換した境界条件中で同じgroup名を上書きする
def combBoundaryConditions(lines):

    #  mergeLines
    #    両方のlinesを上書きする
    def mergeBoundaryLines(currLines, saveLines):
        header = currLines[0]
        currDataLines = currLines[1:]
        for currLine in currDataLines:
            currWords = deleteSp(currLine).split(",")
            curr = currWords[1:3]
            if len(curr) == 1:
                curr += [curr[0]]
            flag = 0
            saveDataLines = saveLines[1:]
            for i in range(len(saveDataLines)):
                saveLine = saveDataLines[i]
                saveWords = deleteSp(saveLine).split(",")
                save = saveWords[1:3]
                if len(save) == 1:
                    save += [save[0]]
                if curr == save:
                    saveDataLines[i] = ", ".join(currWords) + "\n"
                    flag = 1
                    break
            if flag == 0:
                saveDataLines += [currLine]
        ansLines = [header] + saveDataLines
        return ansLines

    #
    #  addNewBcLines
    #    同じgrp名の境界条件に追加して書き込み
    def addNewBcLines(bcLines, newLines):
        [bcName, grpName] = getHeaderGrpName(bcLines)
        if len(newLines) == 0:
            newLines += bcLines
        else:
            i = 0
            flag = 0
            while i < len(newLines):
                ist = i
                [newBcLines, i] = get1BcLines(newLines, i)
                if len(newBcLines) > 1:
                    [newBcName, newGrpName] = getHeaderGrpName(newBcLines)
                    if bcName == newBcName and grpName== newGrpName:
                        if bcName == "BOUNDARY":
                            bcLines = mergeBoundaryLines(bcLines, newBcLines)
                        newLines = newLines[:ist] + bcLines + newLines[i:]
                        flag = 1
                        break
            if flag == 0:
                newLines += bcLines
        return newLines

    newLines = []
    i = 0
    while i < len(lines):
        #1ヶの境界条件を取得
        [bcLines, i] = get1BcLines(lines, i)
        #同じgrp名中に境界条件を追加する
        newLines = addNewBcLines(bcLines, newLines)
    #各条件内のdataを整形する
    newLines = reformBcLines(newLines)
    return newLines

#
#  refineBoundaryConditions
#    境界条件を微修正
def refineBoundaryConditions(bcLines):
    i = 0
    while i < len(bcLines):
        ist = i
        [bc1Lines, i] = get1BcLines(bcLines, i)
        [bcName, grpName] = getHeaderGrpName(bc1Lines)
        bcName = bcName.upper()
        if bcName == "BOUNDARY":
            coord = deleteSp(bc1Lines[1]).split(",")[1:3]
            if coord == ["11", "11"]:
                #FIXTEMPに修正
                line = bcLines[ist]
                words = deleteSp(line).split(",")
                words[0] = "!FIXTEMP"
                bcLines[ist] = ", ".join(words) + "\n"
                line = bcLines[ist+1]
                words = deleteSp(line).split(",")
                bcLines[ist+1] = ", ".join([words[0], words[3]]) + "\n"
        elif bcName == "CFLUX":
            words = deleteSp(bcLines[ist+1]).split(",")
            if len(words) > 2:
                bcLines[ist+1] = ", ".join([words[0], words[2]]) + "\n"
        elif bcName == "DLOAD":
            line = bcLines[ist+1]
            words = deleteSp(line).split(",")
            if words[1].upper() == "CENTRIF" or words[1].upper == "centrif":
                words[1] = "CENT"
                bcLines[ist+1] = ", ".join(words) + "\n"
    return bcLines

#
#  getMaterialLines(lines)
def getMaterialLines(lines):
    """ 材料定義行を取得する"""
    matLines = []
    matDict = matCardDict()
    anaType = "STATIC"
    flag = 0
    for i in range(len(lines)):
        line = lines[i]
        if line[0] == "*":
            flag = 0
            cWord = deleteSp(line[1:]).split(",")[0]
            if cWord in matDict.keys():
                sWord = cWord[-len("SECTION"):]
                if sWord == "SECTION":
                    #section部を作成
                    if cWord[:len("BEAM")] == "BEAM":
                        newLine = "!SECTION, TYPE=BEAM, "
                    elif cWord[:len("SHELL")] == "SHELL":
                        newLine = "!SECTION, TYPE=SHELL, "
                    elif cWord[:len("SOLID")] == "SOLID":
                        newLine = "!SECTION, TYPE=SOLID, "
                    egrp = getValName(line, "ELSET")
                    newLine += "EGRP=" + egrp + ", "
                    material = getValName(line, "MATERIAL")
                    newLine += "MATERIAL=" + material + "\n"
                else:
                    #section以外
                    words = deleteSp(line[1:]).split(",")
                    newLine = "!" + matDict[cWord]
                    if len(words) > 1:
                        newLine += ", " + ", ".join(words[1:])
                    newLine += "\n"
                matLines.append(newLine)
                flag = 1
            else:
                #matDictの辞書に登録していない行
                if cWord == "STATIC":
                    anaType = "STATIC"
                elif cWord == "HEATTRANSFER":
                    anaType = "HEAT"
        else:
            #*以外の行
            if flag == 1:
                matLines.append(line)
    return matLines, anaType

#
#  seperateBcLines
#    cntとmsh側に分離する
def seperateBcLines(bcLines):
    mshLines = []
    cntLines = []
    i = 0
    while i < len(bcLines):
        [bc1Lines, i] = get1BcLines(bcLines, i)
        if len(bc1Lines) > 0:
            [header, grpName] = getHeaderGrpName(bc1Lines)
            if header == "INITIAL CONDITION":
                mshLines += bc1Lines
            else:
                cntLines += bc1Lines
    return [cntLines, mshLines]


#
#  reformBcLines
#    各境界条件を整形する
def reformBcLines(lines):

    #  getKindBc
    #    境界条件の種類を取得
    def getKindBc(lines):
        kindBc = []
        i = 0
        while i < len(lines):
            [bc1Lines, i] = get1BcLines(lines, i)
            if len(bc1Lines) != 0:
                [header, grpName] = getHeaderGrpName(bc1Lines)
                flag = 0
                for bcName in kindBc:
                    if bcName[:2] == [header, grpName]:
                        flag = 1
                        break
                if flag == 0:
                    kindBc.append([header, grpName, bc1Lines[0], []])
        return kindBc

    #  addBcData
    #    同じ境界条件で同じgroup名内に追加
    def addBcData(dataLines, bcName):
        #組数と初期値を取得
        bcItems = nItemsInBC()
        for bcItem in bcItems:
            if bcItem[0] == header.upper():
                nItems = bcItem[1]
                defValue = bcItem[2]
                break
        #配列を準備
        data = []
        for n in range(nItems):
            data.append([])
        #dataを読み込み（配列内に上書き）
        for line in dataLines:
            words = deleteSp(line).split(",")
            if words != [""]:
                words[0] = " ".join(line.split(",")[0].split())
                for n in range(nItems):
                    if len(data[n]) == 0 or data[n] == "\n":
                        data[n] = words
                        break
                    else:
                        if data[n][:2] == words[:2]:
                            data[n] = words
                            break
        #dataを整形
        newData = []
        for n in range(nItems):
            if len(data[n]) != 0:
                newData.append(", ".join(data[n]) + "\n")
        bcName[3] += newData
        return bcName

    #  checkReduse
    #    省略形を元に戻す
    def checkReduse(kindBc):
        for bcName in kindBc:
            if bcName[0] == "BOUNDARY":
                #初期値を取得
                bcItems = nItemsInBC()
                for bcItem in bcItems:
                    if bcItem[0] == "BOUNDARY":
                        defValue = bcItem[2]
                        break
                data = bcName[3]
                for i in range(len(data)):
                    line = data[i]
                    words = deleteSp(line).split(",")
                    if len(words) == 2:
                        words += [words[1]] + [str(defValue)]
                    elif len(words) == 3:
                        if words[2] == "":
                            words[2] = words[1]
                        words += [str(defValue)]
                    else:
                        if words[2] == "":
                            words[2] = words[1]
                        if words[3] == "":
                            words[3] = str(defValue)
                    data[i] = ", ".join(words) + "\n"
        return kindBc

    #  reorderBc
    #    data部は、group名が最初でnode（element）番号は、後
    def reorderBc(kindBc):
        for bcName in kindBc:
            data = bcName[3]
            num = []
            name = []
            for line in data:
                words = deleteSp(line).split(",")
                try:
                    a=int(words[0])
                    num.append(line)
                except:
                    name.append(line)
            bcName[3] = name + num
        return kindBc

    #境界条件の種類を取得
    kindBc = getKindBc(lines)
    newLines = []
    for bcName in kindBc:
        header = bcName[0]
        grpName = bcName[1]
        i = 0
        while i < len(lines):
            [bc1Lines, i] = get1BcLines(lines, i)
            if len(bc1Lines) != 0:
                [dHeader, dGrpName] = getHeaderGrpName(bc1Lines)
                if header == dHeader and grpName == dGrpName:
                    dataLines = bc1Lines[1:]
                    bcName = addBcData(dataLines, bcName)
    #省略形を修正
    kindBc = checkReduse(kindBc)
    #記述順を修正（nodeGrp名が最初）
    kindBc = reorderBc(kindBc)
    #data行を作成
    for bcName in kindBc:
        line = bcName[2]
        newLines.append(line)
        newLines += bcName[3]
    return newLines

#
#  getHeaderGrpName
#    headerとgrpNameを取得
def getHeaderGrpName(lines):
    ans = []
    if len(lines) >= 2:
        #header = deleteSp(lines[0]).split(",")[0][1:]
        header = " ".join(lines[0].split(",")[0][1:].split())
        grpName = ""
        for line in lines[1:]:
            words = deleteSp(line).split(",")
            try:
                a = int(words[0])
            except:
                grpName = words[0]
                break
        ans = [header, grpName]
    return ans

#
#  getHeaderAllGrpNames
#    header内の全てのgrpNameを取得
def getHeaderAllGrpNames(lines):
    ans = []
    if len(lines) >= 2:
        header = deleteSp(lines[0]).split(",")[0][1:]
        names = []
        for line in lines[1:]:
            words = deleteSp(line).split(",")
            try:
                a = int(words[0])
            except:
                names.append(words[0])
        if len(names) >= 1:
            names.sort()
            grpNames = []
            for name in names:
                flag = 0
                for grpName in grpNames:
                    if name == grpName:
                        flag == 1
                        break
                if flag == 0:
                    grpNames.append(name)
        ans = [header, grpNames]
    return ans        

#
#  deleteSameBcLines
#    同じ境界条件、group名部を削除する
def deleteSameBcLines(cntLines, bcLines):
    i = 0
    while i < len(bcLines):
        [bc1Lines, i] = get1BcLines(bcLines, i)
        header = deleteSp(bc1Lines[0]).split(",")[0]
        grpName = deleteSp(bc1Lines[1]).split(",")[0]
        cntLines = delete1Bc(cntLines, header, grpName)
    #cntLines = topHeader + cntLines
    return cntLines

#
#  addBcLines
#    境界条件をcntファイルに追加
def addBcLines(cntLines, bcLines):
    #挿入場所を取得
    i = getInsertPoint(cntLines, "!BOUNDARY", ["Boundary", "BOUNDARY", "boundary"])
    newLines = cntLines[:i] + bcLines + cntLines[i:]
    return newLines

#
#  addMatLines
def addMatLines(cntLines, matLines, anaType):
    """ cntLinesにmatLines（材料）を追加する。"""
    if len(matLines) == 0:
        return cntLines
    #材料の挿入場所を取得
    i = getInsertPoint(cntLines, "!MATERIAL", ["Material", "MATERIAL", "material"])
    #挿入Linesを取得
    addLines = []
    for line in matLines:
        if line[:len("!SECTION")] != "!SECTION":
            addLines.append(line)
    #挿入
    newLines = cntLines[:i] + addLines + cntLines[i:]
    #anaTypeを設定
    #置き換える場所を取得（!SOLUTIONの次の行）
    ist = getInsertPoint(newLines, "!SOLUTION", ["SOLUTION", "Solution", "solution"])
    if anaType == "STATIC":
        newLines[ist] = pyFistr.setValName(newLines[ist], "TYPE=STATIC")
        newLines[ist+1] = "!STATIC\n"
        if newLines[ist+2][0] != "!" and newLines[ist+2][0] != "#":
            newLines = newLines[:ist+2] + newLines[ist+1:]
    elif anaType == "HEAT":
        newLines[ist] = pyFistr.setValName(newLines[ist], "TYPE=HEAT")
        newLines[ist+1] = "!HEAT\n"
        if newLines[ist+2][0] != "!" and newLines[ist+2][0] != "#":
            newLines[ist+2] = "0.0\n"
        else:
            newLines = newLines[:ist+2] + ["0.0\n"] + newLines[ist+2:]
    return newLines

#
#  writeMeshFile
#    mshファイルの最後にaddLinesを追加する
def writeMeshFile(fileName, addLines, matLines, anaType):
    if len(addLines) == 0 and len(matLines) == 0:
        return
    
    f=open(fileName); meshLines=f.readlines(); f.close()
    if len(addLines) > 0:
        #!ENDの前にaddLinesを追加
        #f=open(fileName); meshLines=f.readlines(); f.close()
        n = len(meshLines) - 1
        ist = 0
        while n >= 0:
            line = meshLines[n]
            if line[:len("!END")] == "!END":
                ist = n
                break
            n -= 1
        meshLines = meshLines[:ist] + addLines + meshLines[ist:]
    if len(matLines) > 0:
        #SECTIONを最後の!ELEMENTの次に追加
        # #構造or熱を確認
        # anaType = "ELASTIC"
        # for line in matLines:
        #     if line[:len("!SPECIFIC_HEAT")] == "!SPECIFIC_HEAT":
        #         anaType = "HEAT"
        #         break
        #     elif line[:len("!THERMAL_CONDUCTIVITY")] == "!THERMAL_CONDUCTIVITY":
        #         anaType = "HEAT"
        #         break
        secLines = []
        nItems = 0
        if anaType == "STATIC":
            #ELASTICの設定
            for line in matLines:
                if line[:len("!MATERIAL")] == "!MATERIAL":
                    secLines.append(line)
                elif line[:len("!ELASTIC")] == "!ELASTIC":
                    newLine = "!ITEM=1, SUBITEM=2\n"
                    secLines.append(newLine)
                    nItems += 1
                elif line[:len("!DENSITY")] == "!DENSITY":
                    newLine = "!ITEM=2, SUBITEM=1\n"
                    secLines.append(newLine)
                    nItems += 1
                elif line[:len("!EXPANSION_COEFF")] == "!EXPANSION_COEFF":
                    newLine = "!ITEM=3, SUBITEM=1\n"
                    secLines.append(newLine)
                    nItems += 1
                elif line[:len("!SECTION")] == "!SECTION":
                    secLines.append(line)
                elif line[0] != "!":
                    secLines.append(line)
        elif anaType == "HEAT":
            #HEATの設定
            for line in matLines:
                if line[:len("!MATERIAL")] == "!MATERIAL":
                    secLines.append(line)
                elif line[:len("!DENSITY")] == "!DENSITY":
                    newLine = "!ITEM=1, SUBITEM=1\n"
                    secLines.append(newLine)
                    nItems += 1
                elif line[:len("!SPECIFIC_HEAT")] == "!SPECIFIC_HEAT":
                    newLine = "!ITEM=2, SUBITEM=1\n"
                    secLines.append(newLine)
                    nItems += 1
                elif line[:len("!THERMAL_CONDUCTIVITY")] == "!THERMAL_CONDUCTIVITY":
                    newLine = "!ITEM=3, SUBITEM=1\n"
                    secLines.append(newLine)
                    nItems += 1
                elif line[:len("!SECTION")] == "!SECTION":
                    secLines.append(line)
                elif line[0] != "!":
                    secLines.append(line)

        if len(secLines) != 0:
            #itemの数を!MATERIAL行に追加
            line = secLines[0].rstrip()
            line += ", ITEM=" + str(nItems) + "\n"
            secLines[0] = line
            #最後の!ELEMENTの次に追加
            flag = 0
            for i in range(len(meshLines)):
                if meshLines[i][0] == "!":
                    if meshLines[i][:len("!ELEMENT")] == "!ELEMENT":
                        flag = 1
                    else:
                        if flag == 1:
                            ist = i
                            break
            #sectLinesを追加
            meshLines = meshLines[:ist] + secLines + meshLines[ist:]
    f=open(fileName, "w")
    for meshLine in meshLines:
        f.write(meshLine)
    f.close()

#  delete1Bc
#    該当する境界条件を削除
def delete1Bc(cntLines, header, grpName):
    i = skipHeader(cntLines, 0)
    newLines = cntLines[:i]
    while i < len(cntLines):
        [cont, i] = get1block(cntLines, i)
        if len(cont) >= 2:
            cntHeader = deleteSp(cont[0]).split(",")[0]
            cntGrpName = deleteSp(cont[1]).split(",")[0]
            if (cntHeader == header and cntGrpName == grpName):
                for line in cont:
                    if line[0] == "#":
                        newLines += [line]
                pass
            else:
                newLines += cont
        else:
            newLines += cont
    return newLines

#  get1BcLines
#    1ヶの境界条件を取得
def get1BcLines(lines, i):
    #ie = i
    ie = len(lines)
    bcLines = []
    flag = 0
    while i < len(lines):
        line = lines[i]
        if ( (line[0] == "!" or line[0] == "*") and
             (lines[i+1][0] != "!" and lines[i+1][0] != "*") ):
            #BCを取得
            bcLines.append(line)
            i += 1
            while i < len(lines):
                if lines[i][0] != "!" and lines[i][0] != "*":
                    bcLines.append(lines[i])
                    flag = 1
                else:
                    break
                i += 1
        if flag == 1:
            ie = i
            break
        i += 1
    return [bcLines, ie]

#  get1block
#    1ヶのコマンド行を取得
def get1block(lines, i):
    sp = skipHeader(lines, i)
    if sp < len(lines):
        ep = skipHeader(lines, sp+1)
        newLines = lines[sp:ep]
    else:
        newLines = []
        ep = len(lines)
    return [newLines, ep]

#  skipEndHeader
#    "!" or "#"文字を検索
def skipEndHeader(lines, i):
    ip = len(lines)
    while i < len(lines):
        if lines[i][0] == "!" or lines[i][0] == "#":
            ip = i
            break
        i += 1
    return ip
    

#  skipHeader
#    "!"文字を検索
def skipHeader(lines, i):
    ip = len(lines)
    while i < len(lines):
        if lines[i][0] == "!":
            ip = i
            break
        i += 1
    return ip

#  getInsertPoint
#    挿入場所を取得
#    keywordsの次の行No、存在しない場合はheaderの行Noが戻る
def getInsertPoint(lines, header, keywords):
    ip = 0
    #keywordを検索
    i = 0
    while i < len(lines):
        line = lines[i]
        if line[0] == "#":
            flag = 0
            for key in keywords:
                if line.find(key) >= 0:
                    flag = 1
                    break
            if flag == 1:
                i += 1
                if lines[i][0] == "#":
                    i += 1
                ip =i
                break
        i += 1
    #keywordに該当したか？
    if ip == 0:
        #headerを検索
        i = 0
        while i < len(lines):
            line = lines[i]
            if line[:len(header)] == header:
                ip = i
                break
            i += 1
    return ip

#  getValName
#    valName(TYPEなど）を取得
def getValName(line, valName):
    valName = valName.upper()
    words = deleteSp(line).split(",")
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        word = deleteSp(word)
        if word.upper().find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    return name

#  deleteSp
#    空白、行末コードを削除
def deleteSp(line):
    newLine = ""
    for chara in line:
        if chara != " " and chara != "\n" and chara != "\r":
            newLine += chara
    return newLine

#
#  deleteComments
#    コメント行を削除
def deleteComments(lines):
    newLines = []
    for line in lines:
        if line[:2] != "**":
            if line.replace(" ", "").replace("\t", "") != "\n":
                newLines.append(line)
    return newLines
 
#
#  readFile
#    ファイル読み込み
def readFile(fileName):
    lines = []
    f=open(fileName)
    for line in f.readlines():
        if line != "\n" and line != "\r\n":
            #空白行を削除
            if line[-2:] == "\r\n":
                #行末を統一
                line = line[:-2] + "\n"
            lines.append(line)
    f.close()
    return lines

#
#  writeFile
#    ファイルに書き込み
def writeFile(fileName, lines):
    fw = open(fileName, "w", encoding="utf-8")
    for line in lines:
        fw.write(line)
    fw.close()

#
#  getCntFileName
#-----------------
def getCntFileName(workDir):
    """ cnt, mshファイル名をhecmw_ctrl.datファイルから取得する"""
    cntName = "FistrModel.cnt"
    mshName = "FistrModel.msh"
    fileName = workDir + os.sep + "hecmw_ctrl.dat"
    if os.path.exists(fileName) == False:
        return cntName, mshName
    lines = readFile(fileName)
    for i in range(len(lines)):
        line = lines[i]
        if line[:len("!CONTROL")] == "!CONTROL":
            n = line.find("NAME=fstrCNT")
            if n >= 0:
                cntName = lines[i+1].split()[0]
                break
    for i in range(len(lines)):
        line = lines[i]
        if line[:len("!MESH")] == "!MESH":
            n = line.find("TYPE=HECMW-ENTIRE")
            if n >= 0:
                mshName = lines[i+1].split()[0]
                break
    return cntName, mshName

#
#  addIncludeHeaders
#--------------------
def addIncludeHeaders(fileDir, lines):
    """ *INCLUDE行を追加する。"""
    newLines = []
    for line in lines:
        words = deleteSp(line).split(",")
        if len(words) >= 2:
            if words[0].upper() == "*INCLUDE":
                name = getValName(line, "INPUT")
                if name[0] == "/":
                    fileName = name
                else:
                    fileName = fileDir + "/../" + name
                f = open(fileName, encoding="utf-8"); addLines = f.readlines(); f.close()
                newLines += addLines
            else:
                newLines += [line]
        else:
            newLines += [line]
    return newLines

#
#  abaqusinp2fistrcnt
#  ------------------
#    inpファイルから境界条件を読み取りcntファイルに書き込む
#def abaqusinp2fistrcnt(fileDir, abaName):
def abaqusinp2fistrcnt(abaName, caseDir):
    print()
    print ("boundary conditions in inp file will be convert to FrontISTR. ")
    #inpファイルの読み込み
    #fileName = fileDir + os.sep + abaName + ".inp"
    fileName = abaName
    fileDir = os.path.dirname(fileName)
    lines = readFile(fileName)
    #コメント削除
    lines = deleteComments(lines)
    #includeを処理
    lines = addIncludeHeaders(fileDir, lines)
    #境界条件を取得
    print ("getting to convert items...")
    bcLines = getBoundaryConditions(lines)
    #境界条件を修正（順番に該当分を全て取得）
    print ("converting above items...")
    bcLines = convBoundaryConditions(bcLines)
    #node番号をnodeGrp名に修正
    print ("changing node or element number to its group name...")
    [bcLines, addMeshLines] = convNoToName(bcLines)
    #境界条件とgrpNameをチェック（heder内には1ヶgrp名にする）
    bcLines = checkGrpNameInHeader(bcLines)
    #修正した境界条件をまとめる（同じgroup名があれば上書きする）
    print ("overwritiing to same keyword and same group name...")
    bcLines = combBoundaryConditions(bcLines)
    #境界条件をrefineする。
    print ("converting detail part...")
    bcLines = refineBoundaryConditions(bcLines)
    #材料を取得
    print ("getting marerial lines...")
    matLines, anaType = getMaterialLines(lines)
    #cntとmsh側を分離する
    [bcLines, newAddMeshLines] = seperateBcLines(bcLines)
    addMeshLines += newAddMeshLines
    #cntファイル読み込み
    #FistrModel_cnt, FistrModel_msh = getCntFileName(fileDir)
    FistrModel_cnt, FistrModel_msh = getCntFileName(caseDir)
    #fileName = fileDir + os.sep + FistrModel_cnt
    fileName = caseDir + os.sep + FistrModel_cnt
    cntLines = readFile(fileName)
    #追加部分を削除（有れば）
    cntLines = deleteSameBcLines(cntLines, bcLines)
    #cntファイルに境界条件を追加
    newLines = addBcLines(cntLines, bcLines)
    #cntファイルに材料（!MATERIAL）を追加
    newLines = addMatLines(newLines, matLines, anaType)
    #cntファイル書き込み
    print ("writing cnt file...")
    writeFile(fileName, newLines)
    #mshファイルに書き込み
    print ("writing msh file...")
    #writeMeshFile(fileDir+os.sep+FistrModel_msh, addMeshLines, matLines, anaType)
    writeMeshFile(caseDir+os.sep+FistrModel_msh, addMeshLines, matLines, anaType)
    print ("above items have be converted.")



if __name__=='__main__':
    #log表示
    import logFileCreater
    logFileCreater.log()

    abaName = sys.argv[1]
    caseDir = sys.argv[2]
    if abaName[:len("-h")] == "-h" or abaName[:len("--help")] == "--help":
        printMsg()
    else:
        if abaName.split(".")[-1] != "inp":
            abaName = abaName + ".inp"
        abaqusinp2fistrcnt(abaName, caseDir)
        # fileDir = os.sep.join(abaNameDir.split(os.sep)[:-1])
        # if fileDir == "":
        #     fileDir = os.getcwd()
        # abaName = abaNameDir.split(os.sep)[-1]
        # if abaName.split(".")[-1] == "inp":
        #     name = abaName[:-4]
        # else:
        #     name = abaName
        # abaqusinp2fistrcnt(fileDir, name)

