#!/usr/bin/python3
# coding: utf-8
#
#       selectOpenFoamResultDDialog.py
#
#           OpenFoamの計算結果から点群fileを作成する
#
#   22/11/21    新規作成
#   23/07/12    selectCaseDir:caseDir取得dialogで表示させるdir
#               自身の親dirに変更
#      07/15    selectCaseDir:currDirを選択表示する様に修正
#   24/05/25    selectOpenFoamResult:クラスを新規追加
#      06/07    universalDialogsに変更
#      06/08    openFolderDialog,errDialog,warningDialog:parentを追加
#      06/09    dialogApp:parentを追加
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk

import os

import pyTreeFoam_ptc as pyTreeFoam
import GtkParts
import universalDialogs

ls = "\n"

#---------------------
#  dialogApp class
#---------------------
class dialogApp:

    def __init__(self, currDir, timeFolder, region, field, selPatches, parent=None):
        self.maskEvent = True
        self.builder = Gtk.Builder()
        path = os.getenv("easyIstrPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "selectOpenFoamResultDDialog.glade")
        self.dialog = self.builder.get_object("dialog1")
        self.dialog.set_transient_for(parent)
        self.dialog.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        self.currDir = currDir
        self.setGtkObject()
        self.timeFolders = []
        self.regions = []
        self.fields = []
        self.patches = []
        self.ans = []
        self.initialize(currDir, timeFolder, region, field, selPatches)

    #  show
    def show(self):
        """ dialogを表示"""
        self.maskEvent = False
        self.dialog.show()

    #  close
    def close(self, *args):
        """ dialogを閉じる"""
        self.dialog.destroy()

    #
    #  setGtkObject
    #----------------
    def setGtkObject(self):
        """ Gtkのobjectを設定する。"""
        self.button_caseDirRef = self.builder.get_object("button_caseDirRef")
        self.button_cancel = self.builder.get_object("button_cancel")
        self.button_apply = self.builder.get_object("button_apply")
        self.entry_caseDir = self.builder.get_object("entry_caseDir")
        self.combo_time_org = self.builder.get_object("combo_time")
        self.combo_region_org = self.builder.get_object("combo_region")
        tree_field = self.builder.get_object("tree_field")
        tree_patch = self.builder.get_object("tree_patch")
        self.combo_region = GtkParts.comboBoxText(self.combo_region_org)
        self.combo_time = GtkParts.comboBoxText(self.combo_time_org)
        self.tree_field = GtkParts.treeList(tree_field)
        self.tree_patch = GtkParts.treeList(tree_patch)
        self.tree_field.create(multi=False)
        self.tree_patch.create(multi=True)
        #文字を設定
        self.dialog.set_title(_("点群file作成"))
        mess  = _("OpenFOAMの計算結果からpatchの点群fileを作成する。") + ls
        mess += _("caseDir, time, regionを確認し、field, patchを選択してください。")
        self.builder.get_object("label_subTitle").set_text(mess)
        self.builder.get_object("label_OFResult").set_text(_("OpenFOAMの計算結果"))
        self.builder.get_object("label_boundary").set_text("boundary (" + _("複数選択可") + ")")
        self.builder.get_object("button_caseDirRef").set_label(_("参照..."))
        self.builder.get_object("button_apply").set_label(_("点群作成"))
        self.builder.get_object("button_cancel").set_label(_("キャンセル"))

    #
    #  initialize
    #--------------
    def initialize(self, currDir, timeFolder, region, field, selPatches):
        """ 全itemを表示する"""
        self.entry_caseDir.set_text(currDir)
        if pyTreeFoam.isCaseDir(currDir) == False:
            #openFoamのcaseではない場合は、直ぐに戻る
            return
        #OpenFoamの内容を取得して設定
        self.currDir = currDir
        case = pyTreeFoam.case(self.currDir)
        self.timeFolders = ["latestTime"] + case.getTimeFolders()
        multi = case.isMultiRegion()
        if multi[0] == True:
            self.regions = multi[2]
        else:
            self.regions = ["(region0)"]
        self.combo_time.setData(self.timeFolders)
        self.combo_region.setData(self.regions)
        if timeFolder == self.timeFolders[-1]:
            selTime = self.timeFolders[0]
        else:
            selTime = timeFolder
        self.combo_time.selectItem(selTime)
        if region == ".":
            selRegion = "(region0)"
        else:
            selRegion = region
        self.combo_region.selectItem(selRegion)
        self.getShowFieldsPatches()
        if field != "":
            self.tree_field.selectItems([field])
            idx = self.fields.index(field)
            path = Gtk.TreePath(idx)
            self.tree_field.treeList.scroll_to_cell(path)
        if selPatches != []:
            self.tree_patch.selectItems(selPatches)
            idx = self.patches.index(selPatches[-1])
            path = Gtk.TreePath(idx)
            self.tree_patch.treeList.scroll_to_cell(path)

    #
    #  warningMess
    def warningMess(self):
        title = _("警告")
        mess = _("field or patch が選択されていません。")
        self.warningDialog(title, mess, parent=self.dialog)

    #
    #  selectcaseDir
    #----------------
    def selectCaseDir(self):
        """ OpenFoamのcaseDirを取得して表示する"""
        currDir = self.entry_caseDir.get_text()
        title = _("OpenFOAMのcaseDirを選択")
        showDir = os.path.dirname(currDir)
        selName = os.path.basename(currDir)
        funcOk = [self.selectCaseDirProcess]
        funcCancel = []
        self.openFolderDialog(title, showDir, funcOk, funcCancel, self.dialog, selName)
        return

    #
    #  selectCaseDirProcess
    def selectCaseDirProcess(self, caseDir):
        """ caseDir内のdataを取得し、表示する。"""
        if pyTreeFoam.isCaseDir(caseDir) == False:
            title = _("エラー")
            mess = _("取得したdirは、OpenFOAMのcaseDirではありません。") + "\n"
            mess += _("再取得してください。")
            self.errDialog(title, mess)
            return
        self.entry_caseDir.set_text(caseDir)
        self.currDir = caseDir
        #timeFolder, reginsを取得してセット
        case = pyTreeFoam.case(self.currDir)
        self.timeFolders = ["latestTime"] + case.getTimeFolders()
        multi = case.isMultiRegion()
        if multi[0] == True:
            self.regions = multi[2]
        else:
            self.regions = ["(region0)"]
        self.combo_time.setData(self.timeFolders)
        self.combo_time.selectItem(0)
        self.combo_region.setData(self.regions)
        self.combo_region.selectItem(0)
        #fieldとpatchを取得し、表示
        self.getShowFieldsPatches()

    #
    #  getShowFieldsPatches
    def getShowFieldsPatches(self):
        """ fieldとpatchを取得して、boxに表示する。"""
        case = pyTreeFoam.case(self.currDir)
        timeFolder = self.getSelectTimeFolder()
        region = self.getSelectRegion()
        self.fields = case.getFieldNames(timeFolder, region)
        self.fields.sort()
        self.tree_field.setItems(self.fields)
        meshDir = case.getCurrMeshDir(timeFolder, region, "boundary")
        patchConts = case.getPatchNameFromBoundary(meshDir)
        self.patches = list(map(lambda x: x[0], patchConts))
        self.patches.sort()
        self.tree_patch.setItems(self.patches)

    #
    def getSelectTimeFolder(self):
        """ combo_time内から選択しているtimeを取得する"""
        selTime = self.combo_time.getSelectedItem()
        if selTime == "latestTime":
            selTime = self.timeFolders[-1]
        return selTime

    #
    def getSelectRegion(self):
        """ combo_region内から選択しているregionを取得する"""
        selRegion = self.combo_region.getSelectedItem()
        if selRegion == "(region0)":
            selRegion = "."
        return selRegion

    #
    #  getAllSelectedItems
    #-----------------------
    def getAllSelectedItems(self):
        """ 選択項目を取得する"""
        ans = []
        caseDir = self.entry_caseDir.get_text()
        selTime = self.getSelectTimeFolder()
        selRegion = self.getSelectRegion()
        selFields = self.tree_field.getSelectedItems()
        if len(selFields) == 0:
            selField = ""
        else:
            selField = selFields[0]        
        selPatches = self.tree_patch.getSelectedItems()
        return caseDir, selTime, selRegion, selField, selPatches


    #----- universal dialog -------------------------
    def openFolderDialog(self, title, fileDir, funcOk, funcCancel, parent=None, selName="", filterSets={}):
        dialog = universalDialogs.openFolderDialog(title, fileDir, funcOk, funcCancel, parent, selName, filterSets)
        dialog.show()

    def errDialog(self, title, message, parent=None):
        dialog = universalDialogs.errDialog(title, message, parent)
        dialog.show()

    def warningDialog(self, title, message, parent=None):
        dialog = universalDialogs.warningDialog(title, message, parent)
        dialog.show()

#------------------------------------------
#  selectOpenFoamResult class
#------------------------------------------
class selectOpenFoamResult:

    def __init__(self, currDir, timeFolder, region, field, selPatches, funcName, funcArgs, parent=None):
        self.currDir = currDir
        self.timeFolder = timeFolder
        self.region = region
        self.field = field
        self.selPatches = selPatches
        self.funcName = funcName
        self.funcArgs = funcArgs
        self.dialog = dialogApp(currDir, timeFolder, region, field, selPatches, parent)
        self.dialog.button_caseDirRef.connect("clicked", self.onSelectCaseDir)
        self.dialog.button_cancel.connect("clicked", self.onCancel)
        self.dialog.button_apply.connect("clicked", self.onApply)
        self.dialog.combo_time_org.connect("changed", self.onChangeTimeFolder)
        self.dialog.combo_region_org.connect("changed", self.onChangeRegion)

    def show(self):
        self.dialog.show()

    def close(self):
        self.dialog.close()

    def onSelectCaseDir(self, event):
        self.dialog.maskEvent = True
        self.dialog.selectCaseDir()
        self.dialog.maskEvent = False
    
    def onCancel(self, event):
        self.close()

    def onApply(self, event):
        ans = self.dialog.getAllSelectedItems()
        #fieldとpatchをチェック
        if ans[3] == "" or len(ans[4]) == 0:
            #field,patchを選択していない場合
            self.dialog.warningMess()
            return
        self.close()
        [sgrp, path, item] = self.funcArgs
        self.funcName(ans, sgrp, path, item)

    def onChangeTimeFolder(self, event):
        if self.dialog.maskEvent == False:
            self.dialog.getShowFieldsPatches()

    def onChangeRegion(self, event):
        if self.dialog.maskEvent == False:
            self.dialog.getShowFieldsPatches()
