#!/usr/bin/python3
# coding: utf-8
#
#   progressBar.py
#
#       進捗状況を表示するdialog
#
#   19/12/07    新規作成
#   20/04/22    多言語化対応
#      09/15    pasteCaseLocalToLocal.__init__:引数にnewNamesを追加。
#               pasteCaseLocalToLocal.getBasePathName:basePathNameを
#               newNames(self.newCaseNames)から取得。
#   22/02/02    pasteLocalToLocal.paste,pasteCaseLocalToLocal.paste:
#               copyできなかったfileがあった場合、errorを返す。
#      02/26    pasteCaseLocalToLocal.paste:case内のcaseもcasePaste
#               できる様に修正。（3段目のcaseは、そのままコピー）
#      03/10    pasteCaseLocalToLocal.paste:case内caseを全てcasePaste
#               できる様に修正。（再帰的呼び出しで実現）
#      03/12    pasteLocalToLocal.paste,pasteCaseLocalToLocal.paste:
#               linkコピーエラー時、link元をコピーする様に修正。
#      07/02    pasteCaseLocalToLocal.paste.getPasteFolderName:
#               caseNameをself.newCaseNames[i]に変更。（バグ修正）
#      10/10    TreeFoam用をeasyIstr用に書き換え
#   23/11/02    resultPathes:計算結果file,folderのpathを修正
#   24/07/07    openにencoding="utf-8"を追加
#

import os
import subprocess
import shutil
import time
import glob
import pyFistr



#
#  resultPathes
#----------------
def resultPathes(currDir):
    """ currDir内の計算結果pathを取得して返す。
    folder、files両方を含む。"""
    #meshHeader名を取得
    FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(currDir)
    FistrModelDict = pyFistr.getCntMshFileNameDict(currDir)
    hm = pyFistr.getHeaderName(FistrModel_msh)
    hr = pyFistr.getHeaderName(FistrModelDict["res"])
    hr = hr.split("_")[0]
    #計算結果
    files = [hr+".res.*", hm+"*.inp", ".restart.*", "*.vtk",
                "conv*", "*.log", "FSTR.dbg.*", "FSTR.msg", "FSTR.sta",
                "*.pvtu", "*.vtu", "*.vis_psf.*"]
    #並列処理用の分割メッシュ
    #files += ["FistrModel_p*"]
    #固有値解析
    #files += ["FistrModel_eigen.res.*", "eigen_log"]
    files += ["*_eigen.res.*", "eigen_log"]
    #周波数応答解析
    #files += ["FistrModel_dyna.res.*", "dyna_*.txt"]
    files += ["*_dyna.res.*", "dyna_*.txt"]
    #熱伝導解析
    #files += ["FistrModel_temp.res.*"]
    files += ["*_temp.res.*"]
    resPathes = []
    for path in files:
        resPathes += glob.glob(currDir + os.sep + path)
    return resPathes


#------------------------------
#  dialog class
#------------------------------
class dialog:
    """ 進捗状況表示dialog"""

    def showSettingDialog(self, title, setting):
        """ dialogを起動する
        """
        #dialogを起動
        path = os.getenv("easyIstrPath") + os.sep + "python" + os.sep
        comm = "python3 " + path + "settingDialog22.py '" + title + "' " + "'" + setting + "'"
        self.settingDialog = subprocess.Popen(
            comm,
            shell=True,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE)
        #dialogに初期値を出力
        self.numSettingDialog(0, "preparing...")

    def numSettingDialog(self, num, cont):
        """ 進捗状況を表示する。
        途中でCANCELまたは、dialogを閉じた場合は、BrokenPipeエラーが発生する。
        この場合、"CANCEL"が戻る。
        """
        stat = "OK"
        try:
            #subprosessのstdinへ(Value, lineCont)を出力
            self.settingDialog.stdin.write((str(num) + "," + cont + "\n").encode())
            self.settingDialog.stdin.flush()
        except:
            stat = "CANCEL"
        return stat

    def closeSettingDialog(self):
        """ dialogを閉じる。"""
        self.numSettingDialog(100, "close")
        self.settingDialog.stdin.write(("close\n").encode())
        if self.settingDialog.poll() == None:
            #BrokenPipeでは無い時にflushする
            self.settingDialog.stdin.flush()


#-------------------------
#  pasteLoacLToLocal class
#-------------------------
class pasteLocalToLocal:
    """ file, folderをpasteする。
    copyDisとpasteDirsからペアの(copy,paste)のdirを取得してpasteする。
    
    Attribute:
        copyDirs (list(str))    :copyするfile, folderのlist
        pasteDirs (list(str))   :pasteするdirのlist
        stat (str)              :"OK" or "CANCEL"
    """

    def __init__(self, copyDirs, pasteDirs):
        self.copyDirs = copyDirs
        self.pasteDirs = pasteDirs
        self.stat = "OK"

    def paste(self):
        """ 貼り付け"""

        def checkDirs():
            """ copyDirとpasteDirが同じ場合は、paste対象から除外する"""
            newCopyDirs = []; newPasteDirs = []
            for i in range(len(self.copyDirs)):
                copyDir = self.copyDirs[i]
                pasteDir = self.pasteDirs[i]
                if copyDir != pasteDir:
                    newCopyDirs.append(copyDir)
                    newPasteDirs.append(pasteDir)
            self.copyDirs = newCopyDirs
            self.pasteDirs = newPasteDirs

        def getBasePathName(copyDir, pasteDir):
            """ コピー先に同じfile名が存在する場合、file名に _copy0を追加する"""
            baseName = copyDir.split(os.sep)[-1]
            basePath = pasteDir + os.sep + baseName
            if os.path.exists(basePath) == True:
                #コピー先に同じfile名が存在する時
                i = 0
                basePath = pasteDir + os.sep + baseName + "_copy" + str(i)
                while os.path.exists(basePath) == True:
                    i += 1
                    basePath = pasteDir + os.sep + baseName + "_copy" + str(i)
            #新しいbaseNameを取得
            baseName = basePath.split(os.sep)[-1]
            return baseName
        
        def getNewPathName(copyDir, pasteDir, copyPathName, copyBaseName):
            """ file, folder名のpaste先pathを取得する
            
            Args:
                copyDir (str)       :self.copyDirsの要素
                pasteDir (str)      :paste先folderのpath
                copyPathName (str)  :copyするfile or folderのpath
                copyBaseName (str)  :copyDirのbaseName
            Returns:
                newCopyName (str)   :pasteするfileのpath"""
            baseName = copyPathName[len(copyDir):]
            newCopyName = pasteDir + os.sep + copyBaseName + baseName
            return newCopyName

        def getCopyFolderFileLinkNames(inNames, names):
            """ dirをコピーする場合。copyNameとpasteNameを取得する"""
            #変数を展開
            (copyDir, copyBaseName, pasteDir) = inNames
            (copyFiles, newFiles, copyFolders, newFolders,
                linkDir, newLinkDir) = names
            #paste側のdirを取得
            newDirName = pasteDir + os.sep + copyBaseName
            newFolders.append(newDirName)
            #copyするfolder内の内容を確認する
            for (root, dirs, files) in os.walk(copyDir):
                if self.stat == "CANCEL":
                    break
                #folderを確認
                for name in dirs:
                    if self.stat == "CANCEL":
                        break
                    dirName = root + os.sep + name
                    newDirName = getNewPathName(copyDir, pasteDir, dirName, copyBaseName)
                    #folderがlinkかどうか
                    if os.path.islink(dirName) == True:
                        #folderLinkの場合
                        linkDir.append(dirName)
                        newLinkDir.append(newDirName)
                    else:
                        #folderの場合
                        copyFolders.append(dirName)
                        newFolders.append(newDirName)
                    #GUIがcloseしたかどうか
                    self.stat = progress.numSettingDialog(0.0, _("コピーの準備中..."))
                #fileを確認
                for name in files:
                    if self.stat == "CANCEL":
                        break
                    copyFile = root + os.sep + name
                    newFile = getNewPathName(copyDir, pasteDir, copyFile, copyBaseName)
                    if os.path.islink(copyFile) == True:
                        linkDir.append(copyFile)
                        newLinkDir.append(newFile)
                    else:
                        copyFiles.append(copyFile)
                        newFiles.append(newFile)
                    #GUIの情報を設定-----------
                    Value = 0.0
                    labelCont = str(len(copyFiles))+" " + _("コピーの準備中...")
                    #GUIの状況を確認
                    self.stat = progress.numSettingDialog(Value, labelCont)
            #変数を格納
            names = (copyFiles, newFiles, copyFolders, newFolders,
                     linkDir, newLinkDir)
            return names

        def getNewFileName(copyFile, pasteDir):
            """ self.copyDirsの要素がfileの場合、
            そのfileと同名のfileがpasteDirに存在する時、file名を修正する"""
            baseName = copyFile.split(os.sep)[-1]
            newPathName = pasteDir + os.sep + baseName
            if os.path.exists(newPathName) == True:
                #同名のfileがpasteDir内に存在する場合
                words = copyFile.split(os.sep)
                baseName = words[-1]
                i = 0
                newPathName = pasteDir + os.sep + baseName + "_copy" + str(i)
                while os.path.exists(newPathName) == True:
                    i += 1
                    newPathName = pasteDir + os.sep + baseName + "_copy" + str(i)
            return newPathName


        #dirを確認する
        checkDirs()
        if len(self.copyDirs) == 0:
            return "OK"
        #progressDialogを表示-----------------
        progress = dialog()
        progress.showSettingDialog(_("folderの貼り付け"), "copying .... ")
        self.stat = progress.numSettingDialog(0, "setting...")
        #paste開始
        copyFiles = []
        newFiles = []
        copyFolders = []
        newFolders = []
        linkDir = []
        newLinkDir = []
        #copyDir毎に確認し、copyName、newPasteNameを取得する
        for i in range(len(self.copyDirs)):
            if self.stat == "CANCEL":
                break
            copyDir = self.copyDirs[i]
            pasteDir = self.pasteDirs[i]
            #paste先のbaseNameを確認（同名が存在する場合、修正）
            copyBaseName = getBasePathName(copyDir, pasteDir)
            if os.path.isdir(copyDir) == True:
                #dirの場合
                inNames = (copyDir, copyBaseName, pasteDir)
                names = (copyFiles, newFiles, copyFolders, newFolders,
                         linkDir, newLinkDir)
                #namesを取得
                names = getCopyFolderFileLinkNames(inNames, names)
            else:
                #ファイルのコピーの場合
                copyFiles.append(copyDir)
                newFile = getNewFileName(copyDir, pasteDir)
                newFiles.append(newFile)
        #増分を取得
        dValue = 1.0 / float(len(copyFiles) + len(linkDir) + 1) * 100.0
        #pasteDirにdirectoryを作成
        for folderDir in newFolders:
            if os.path.exists(folderDir) == False:
                os.makedirs(folderDir)
        #fileのコピー開始
        Value = 0.0
        errorFiles = []
        for i in range(len(copyFiles)):
            if self.stat == "CANCEL":
                break
            print("copy:", newFiles[i])
            try:
                #fileをcopyする
                shutil.copy(copyFiles[i], newFiles[i])
            except:
                errorFiles.append(copyFiles[i])
            #GUIの表示内容を更新-----------------
            Value += dValue
            cpName = copyFiles[i].split(os.sep)[-1]
            labelCont = _(" コピー中 ") + cpName + "..."
            self.stat = progress.numSettingDialog(Value, labelCont)
        #linkのコピー
        errorLinkFiles = []
        for i in range(len(linkDir)):
            if self.stat == "CANCEL":
                break
            linkto = os.readlink(linkDir[i])
            if linkto[0] != ".":
                #絶対pathのリンク
                try:
                    os.symlink(linkto, newLinkDir[i])
                    print(_("絶対参照で「") + newLinkDir[i] + _("」リンクを作成しました。"))
                except:
                    errorLinkFiles.append(linkDir[i])
            else:
                #相対pathのリンク
                try:
                    os.symlink(linkto, newLinkDir[i])
                    print(_("相対pathで「") + linkto + _("」リンクを作成しました。"))
                except:
                    errorLinkFiles.append(linkDir[i])
            #GUIの表示内容を更新------------
            Value += dValue
            cpName = newLinkDir[i].split(os.sep)[-1]
            labelCont = _(" コピー中 ") + cpName + "..."
            self.stat = progress.numSettingDialog(Value, labelCont)
        if len(errorFiles) == 0 and len(errorLinkFiles) != 0:
            #link元をコピー
            for i in range(len(linkDir)):
                copyFileLink = linkDir[i]
                pasteFolder = "/".join(newLinkDir[i].split("/")[:-1])
                try:
                    shutil.copy(copyFileLink, pasteFolder)
                except:
                    errorFiles.append(copyFileLink)
            if len(errorFiles) == 0:
                self.stat = "warningLink"
        #エラーチェック
        if len(errorFiles) != 0:
            print(_("以下のファイルがコピーできませんでした。"))
            for errorFile in errorFiles:
                print("error:", errorFile)
            self.stat = "error"
        #GUIを閉じる------------------
        progress.closeSettingDialog()
        return self.stat


#-----------------------------
#  pasteCaseLocalToLocal class
#-----------------------------
class pasteCaseLocalToLocal:
    """ caseをコピーする。
    caseDirsとpasteDirsからペアの(copy,paste)のdirを取得してコピーする。
    
    Attribute:
        caseDirs (list(str))    :copyするcaseのlist
        pasteDirs (list(str))   :pasteするdirのlist
        newCaseNames (list(str)):renameしてcaseをpasteする場合
        stat (str)              :戻り値（"OK" or "CANCEL"）
    """
    def __init__(self, caseDirs, pasteDirs, newNames=[]):
        self.caseDirs = caseDirs
        self.pasteDirs = pasteDirs
        if len(newNames) == 0:
            #caseDirsのcase名を取得
            self.newCaseNames = list(map(lambda x: x.split(os.sep)[-1], caseDirs))
        else:
            self.newCaseNames = newNames
        self.stat = "OK"
        self.copyFiles = []
        self.nCopyFiles = 0

    def paste(self):
        """ local内のcaseをlocalのpasteDirへコピーする。
        self.caseDirsをself.pasteDirにコピーする。
        コピー先に同じcase名が存在する場合は、<case>_copy0 でコピーする。
        
        Returns:
            stat (str) :"OK" or "CANCEL(中断)"
        """
        
        def checkCaseDirs():
            """ caseDirとpasteDirが同じ場合は、copy対象から削除する"""
            newCaseDirs = []; newPasteDirs = []
            for i in range(len(self.caseDirs)):
                caseDir = self.caseDirs[i]
                pasteDir = self.pasteDirs[i]
                if caseDir != pasteDir:
                    newCaseDirs.append(caseDir)
                    newPasteDirs.append(pasteDir)
            self.caseDirs = newCaseDirs
            self.pasteDirs = newPasteDirs

        def getFilesInFolders(folders):
            """ folder内のfileを取得してself.copyFilesに保存。
            再帰的呼び出しにより、folder内の全てのfolderをチェック。"""            
            subFolders = []
            for folder in folders:
                resPathes = resultPathes(folder)
                allPathes = glob.glob(folder + os.sep + "*")
                needPathes = set(allPathes) - set(resPathes)
                needFolders = list(filter(lambda x: os.path.isdir(x), needPathes))
                needFiles = list(filter(lambda x: os.path.isfile(x), needPathes))
                subFolders += needFolders
                names = list(map(lambda x: os.path.basename(x), needFiles))
                self.copyFiles.append([folder, names])
                self.nCopyFiles += len(names)
                #progressを表示
                labelCont = str(self.nCopyFiles)+" "+_("コピーの準備中...")
                self.stat = progress.numSettingDialog(0.0, labelCont)
                if self.stat == "CANCEL":
                    break
            if len(subFolders) > 0:
                getFilesInFolders(subFolders)

        def getPasteFolderName(copyFolderFiles, pasteDirs):
            """ copyFile名からpasteFolder名を取得して返す"""
            pasteFolders = []
            for i in range(len(copyFolderFiles)):
                #pasteFolderを取得
                pasteDir = pasteDirs[i]
                caseConts = copyFolderFiles[i]
                caseConts.sort()
                caseDir = caseConts[0][0]
                #caseName = os.path.basename(caseDir)
                caseName = self.newCaseNames[i]
                pasteFolder = pasteDir + os.sep + caseName
                #同名のfolderがあるか？
                if os.path.exists(pasteFolder) == True:
                    i = 0
                    pasteFolder = pasteDir + os.sep + caseName + "_copy" + str(i)
                    while os.path.exists(pasteFolder) == True:
                        i += 1
                        pasteFolder = pasteDir + os.sep + caseName + "_copy" + str(i)
                newFolders = []
                for folderDir, names in caseConts:
                    newFolder = pasteFolder + folderDir[len(caseDir):]
                    newFolders.append(newFolder)
                pasteFolders.append(newFolders)
            return pasteFolders

        def copyFilesWithoutLinks(copyFolderFiles, pasteFolders):
            """ linl以外のfileをコピーする。linkを取得して返す。"""
            #増分を取得（progressBarの増分）
            nFiles = self.nCopyFiles
            dValue = 1.0 / float(nFiles + 1) * 100.0
            #貼り付け（linkを除く）
            Value = 0.0
            errorFiles = []
            copyFileLinks = []
            pasteFolderLinks = []
            for i in range(len(copyFolderFiles)):
                caseConts = copyFolderFiles[i]
                caseConts.sort()
                pasteConts = pasteFolders[i]
                for ii in range(len(caseConts)):
                    copyDir, names = caseConts[ii]
                    pasteDir = pasteConts[ii]
                    if os.path.exists(pasteDir) == False:
                        #dirを作成
                        os.mkdir(pasteDir)
                    for name in names:
                        copyFileName = copyDir + os.sep + name
                        if os.path.islink(copyFileName) == False:
                            #fileをコピー
                            print("copy:", copyFileName)
                            try:
                                shutil.copy(copyFileName, pasteDir)
                            except:
                                errorFiles.append(copyFileName)
                            #progressBarを更新
                            Value += dValue
                            labelCont = _(" コピー中 ") + name
                            self.stat = progress.numSettingDialog(Value, labelCont)
                        else:
                            #linkを取得
                            copyFileLinks.append(copyFileName)
                            pasteFolderLinks.append(pasteDir)
                        if self.stat == "CANCEL":
                            break
                    if self.stat == "CANCEL":
                        break
                if self.stat == "CANCEL":
                    break
            return copyFileLinks, pasteFolderLinks, errorFiles

        def copyLinks(copyFileLinks, pasteFolderLinks):
            """ linkをコピー。
            windows共有フォルダでは、linkコピーできないので注意!!"""
            errorFiles = []
            for i in range(len(copyFileLinks)):
                copyFileLink = copyFileLinks[i]
                linkto = os.readlink(copyFileLink)
                baseName = os.path.basename(copyFileLink)
                newLink = pasteFolderLinks[i] + os.sep + baseName
                if linkto[0] != ".":
                    #絶対pathのリンク
                    try:
                        os.symlink(linkto, newLink)
                        print(_("絶対参照で「") + copyFileLink + _("」リンクを作成しました。"))
                    except:
                        errorFiles.append(copyFileLink)
                else:
                    #相対pathのリンク
                    try:
                        os.symlink(linkto, newLink)
                        print(_("相対pathで「") + copyFileLink + _("」リンクを作成しました。"))
                    except:
                        errorFiles.append(copyFileLink)
            return errorFiles

        def copyLinkSource(copyFileLinks, pasteFolderLinks):
            """ link元をコピーする"""
            errorFiles = []
            for i in range(len(copyFileLinks)):
                copyFileLink = copyFileLinks[i]
                pasteFolder = pasteFolderLinks[i]
                try:
                    shutil.copy(copyFileLink, pasteFolder)
                except:
                    errorFiles.append(copyFileLink)
            return errorFiles

        #caseCopy開始--------
        #caseDirsを確認
        checkCaseDirs()
        if len(self.caseDirs) == 0:
            return "OK"
        #progressDialogを表示---------------
        progress = dialog()
        progress.showSettingDialog(_("caseの貼り付け"), "copying .... ")
        self.stat = progress.numSettingDialog(0, "setting...")
        #copyするfileを取得する
        copyFolderFiles = []
        self.nCopyFiles = 0
        for caseDir in self.caseDirs:
            self.copyFiles = []
            getFilesInFolders([caseDir])
            copyFolderFiles.append(self.copyFiles)
        #キャンセルチェック
        if self.stat == "CANCEL":
            return self.stat
        #paste先のfolder名を取得
        pasteFolders = getPasteFolderName(copyFolderFiles, self.pasteDirs)
        #copyFile(link以外)
        resCopy = copyFilesWithoutLinks(copyFolderFiles, pasteFolders)
        (copyFileLinks, pasteFolderLinks, errorFiles) = resCopy
        if self.stat == "CANCEL":
            return self.stat
        #linkをコピー
        errorLinkFiles = copyLinks(copyFileLinks, pasteFolderLinks)
        if len(errorFiles) == 0 and len(errorLinkFiles) != 0:
            #link元をコピー
            errorFiles = copyLinkSource(copyFileLinks, pasteFolderLinks)
            if len(errorFiles) == 0:
                self.stat = "warningLink"
        #エラーチェック
        if len(errorFiles) != 0:
            print(_("以下のファイルがコピーできませんでした。"))
            for errorFile in errorFiles:
                print("error:", errorFile)
            self.stat = "error"
        #progressBarを閉じる
        Value = 1.1
        progress.closeSettingDialog()
        return self.stat

#-------------------------
#  pasteServerLocal class
#-------------------------
class pasteServerLocal:
    """ serverを使ったコピー。
    localToServer, serverToLocal, serverToServerのコピー。
    local-server間はscpでコピー。server内同士はcpでコピーする。
    
    Attrubute:
        runFile (str)   :実行file名
        logFile (str)   :実行結果が記録されるfile
        stat (str)      :戻り値（"OK" or "CANCEL"）
        """

    def __init__(self, runFile, logFile):
        self.runFile = runFile
        self.logFile = logFile
        self.stat = "OK"

    def paste(self):
        """ serverを使ったコピー、caseコピーの進捗表示。
        localToServer, serverToLocal, serverToServerのコピー進捗表示。
        local-server間はscpでコピー、server内同士はcpでコピーしている。
        コピーの内容は、logFile内に記録されているので、この内容を
        確認しながら、進捗を表示させる。
        
        Returns:
            stat (str)      :progressDialogの状態("OK" or "CANCEL")
        """
        #dialogの設定-------------
        #self.progress = progressBar.dialog()
        self.progress = dialog()
        self.progress.showSettingDialog(_("server-local間コピー"), "copying .... ")
        self.stat = self.progress.numSettingDialog(0, _("serverを確認中..."))
        #logfileをopen
        self.logFileObj = open(self.logFile, encoding="utf-8")
        #file数をlogFile中から読み取る
        nFiles = 0
        loop = True
        while loop:
            if self.stat == "CANCEL":
                break
            line = self.get1lineFromLogFile()
            if line != "":
                #GUI側へ表示--------
                labelCont = _("serverから取得中 " + str(nFiles))
                self.stat = self.progress.numSettingDialog(0.0, labelCont)
                if (line[:len("Script started")] == "Script started"):
                    loop = False
                    break
                nFiles += 1
            else:
                loop = False
                break
        #コピーのtype（scp、cp）を確認
        copyType = self.getCopyType_scp_or_cp()
        if copyType == "scp":
            #scp
            self.stat = self.copyFoldersFilesScp(nFiles)
        else:
            #cp
            self.stat = self.copyFoldersFilesCp(nFiles)
        #logFileをclose
        self.logFileObj.close()
        #dialogを閉じる
        self.progress.closeSettingDialog()
        return self.stat

    def get1lineFromLogFile(self):
        """ logFileから1行を取得する。
        logFileの更新が30s待っても更新されない場合は、「""」を返す。"""
        waitTime = 30.0
        count = 0.0
        loop = True
        while loop == True:
            line = self.logFileObj.readline()
            if line == "":
                time.sleep(0.2)
                count += 0.2
                if count >= waitTime:
                    loop = False
                    break
            else:
                loop = False
                break
        return line

    def getCopyType_scp_or_cp(self):
        """ copy方法が「scp」or 「cp」を調べる"""
        currPos = self.logFileObj.tell()
        loop = True
        copyType = ""
        while loop:
            line = self.get1lineFromLogFile()
            if line != "":
                words = line.split()
                if len(words) >= 3:
                    if words[1] == "100%":
                        copyType = "scp"
                        break
                    elif words[1] == "->":
                        copyType = "cp"
                        break
                    elif line[:len("Script done")] == "Script done":
                        copyType = ""
                        break
            else:
                break
        self.logFileObj.seek(currPos)
        return copyType

    #
    #  copyFoldersFilesScp
    #
    def copyFoldersFilesScp(self, nFiles):
        """ scpコピーの場合、進捗状況を取得"""
        nCopyFiles = 0
        loop = True
        while loop == True:
            if self.stat == "CANCEL":
                break
            line = self.get1lineFromLogFile()
            if line != "" and line[:len("Script done")] != "Script done":
                words = line.split()
                if len(words) > 3:
                    if words[1] == "100%":
                        cpName = words[0]
                        self.speed = words[3]
                        #率を計算（値が100以上にならない様に処理）
                        nCopyFiles += 1
                        value = nCopyFiles / nFiles * 100.0
                        if value >= 100.0:
                            value = 99.0
                        Value = value
                        labelCont = _(" コピー中 ") + cpName + " (" + self.speed + ")"
                        self.stat = self.progress.numSettingDialog(Value, labelCont)
            else:
                loop = False
                break
        return self.stat

    #
    #  copyFoldersFilesCp
    #
    def copyFoldersFilesCp(self, nFiles):
        """ cpコピーの場合、進捗状況を取得する"""
        nCopyFiles = 0
        loop = True
        while loop == True:
            if self.stat == "CANCEL":
                break
            line = self.get1lineFromLogFile()
            if line != "" and line[:len("Script done")] != "Script done":
                words = line.split()
                if len(words) > 1:
                    cpName = words[0][1:-1].split("/")[-1]
                    #率を計算（値が100以上にならない様に処理）
                    nCopyFiles += 1
                    value = nCopyFiles / nFiles * 100.0
                    if value >= 100.0:
                        value = 99.0
                    Value = value
                    labelCont = _(" コピー中 ") + cpName
                    self.stat = self.progress.numSettingDialog(Value, labelCont)
            else:
                loop = False
                break
        return self.stat


if __name__ == "__main__":
    import gettext
    #localeDir = os.getenv("easyIstrPath") + os.sep + "data" + os.sep + "locale"
    localeDir = os.getenv("LOCALE_DIR")
    gettext.install("easyistr", localeDir)
    #_ = gettext.gettext
