#!/usr/bin/python3
#  coding :  utf-8
#
#   plotStepMonitorDialog.py
#
#       plotStepMonitor制御用のdialog
#       plotするデータが多いと次第に動作が重くなる為、
#       データを最大1/32まで間引いて圧縮する。
#
#   17/07/29    新規作成
#      08/23    gnuplotコマンドをusingApp内から読み込む様に修正
#      08/24    dataのprintOutを追加。
#      09/12    plotStepMonitor:「os.Popen」を「subprocess.Popen」に変更
#      11/30    plotStepMonitor:端末への出力量を減らす。
#      12/01    plotStepMonitor:端末への出力量をさらに減らす。
#   19/01/08    python3用に書き換え
#      01/14    stdin.flush()を追加
#      02/05    windows用にctrl-cで停止させる事を想定して対応
#   20/04/15    FrontISTR-5用に書き換え
#      04/19    readGnuplotCommand:起動optionを修正。
#               閉じても「gnuplot_qt」が残ってしまう為。
#      07/08    readGnuplotCommand:読み込みを辞書に変更
#   21/05/22    GTK+3でGUIを新規に作成
#               「gnuplot_qt」が残ってしまう事がある為。
#               NLSTATIC, DYNAMIC, HEAT_D, ver4, ver5を統合
#      05/23    checkFullData5:listIndexErrorが発生したので、修正。
#               ver-5.2でエラーが発生
#      05/27    gnuplotの起動オプションに「-slow」を追加。
#               「slow font initialization」の警告が発生し、
#               画面が崩れる為。
#      05/28    threadを投入して、ファイル読込、グラフ表示させる。
#                 thread内では、time.sleep()を実行しない。
#                   GUIがフリーズする。
#                 thread内では、label.set_text()等、GUIを操作しない。
#                   GUIがフリーズする。（正しくセットされない）
#               timerEventをGLib.timeout_add()で発生させ、ここで処理する。
#      05/29    gnuplotCommをpyFistrを通じて取得する様に修正。
#               起動側から引数で取得すると「"」の文字が消える為。
#   22/02/16    ver5のHEAT_Dについて、minTemperatureのplotを追加。
#      03/15    国際化
#      03/22    国際化（文字をpython側でセット）
#      04/12    log表示修正（import logFileCreaterを追加）
#      10/16    plotStepMonitorNL5,plotStepMonitorDY5,plotStepMonitorHDY5:
#               th.setDaemon(True) → th.daemon=True に修正。
#      12/14    readAndPlotData_Hga5:未定義error発生した為、maxVal=""を追加
#      12/22    dataPlotG5,dataPlotGc5,dataHeatPlot5:plotDataをstdinのpipeで
#               gnuplotに渡していたがdataの取りこぼしが発生（windowsのgnuplot-5.4.4）
#               この為、dataFileを作成する様に修正。
#   24/01/30    dataPlotG5,dataPlotGc5,dataHeatPlot5:2回目以降のplotを「replot」
#               のみに変更。（windowsで待ち時間以内にplotできない事が発生する為）
#               setThreadIntervalFromGaCount:dataのプロット間隔をcount数に
#               応じて変化させる様に修正。
#               plotStepMonitorNL5_thread,plotStepMonitorDY5_thread,
#               plotStepMonitorHDY5_thread:data読み込みのlimitTimeを
#               1→self.waitTime*0.2に修正。timeOverで引っかからない様に修正。
#      01/31    setThreadIntervalFromGaCount:global変数のplotIntervalの修正で
#               plot間隔を修正可。
#      07/07    openにencoding="utf-8"を追加
#      09/14    getPlotData5:step=100000以上でエラー停止。修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib

import sys
import os
import subprocess
import time
import threading

import pyFistr

#import locale
localeDir = os.getenv("LOCALE_DIR")
#locale.bindtextdomain("easyistr", localeDir)

ls = b"\n"
if os.name == "nt":
    #windows
    ls = b"\r\n"                #改行コード変更

plotInterval = 5.0              #plot間隔（s）

#------------
#  windowApp
#------------
class windowApp():
    """ GUIの起動と終了"""

    def __init__(self, version, anaType):
        self.builder = Gtk.Builder()
        #self.builder.set_translation_domain("easyistr")
        path = os.getenv("easyIstrPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "plotStepMonitorDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        #self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.mainWindow.set_title("running plotStepMonitor")
        self.builder.connect_signals(self)
        #windowを最上位に表示させる
        self.mainWindow.set_keep_above(True)
        #変数を定義
        self.version = version
        self.anaType = anaType
        self.logFile = "0.log"          #maxMin取得用file
        self.convFile = "FSTR.sta"      #残渣取得用file
        self.waitTime = 5.0             #読込の待ち時間
        self.runFlag = "run"            #起動flag
        self.gaDataFile = "gnuplotDataGa.plt"   #gnuplotのdataFile
        self.gcDataFile = "gnuplotDataGc.plt"   #  ↑

        self.nComp = 10000              #圧縮しないdata数
        self.nCompNest = 6              #圧縮レベル(max:1/32)
        
        self.threadRes = "OK"           #thread実行結果 
        self.ga = ""                    #logFile関連側
        self.fa = ""
        self.gaPlotFlag = 0             #gaの初回plot有無
        self.gaLines = []
        self.gaCount = 0
        self.allData = []
        self.gaFullFlag = 0
        self.gc = ""                    #残渣file関連側
        self.fc = ""
        self.gcPlotFlag = 0             #gcの初回plot有無
        self.gcLines = []
        self.gcCount = 0
        self.fcData = []
        #GUIのobjectの定義
        self.setGtkObject()
        #gnuplotCommを取得する
        self.gnuplotComm = self.getGnuplotComm()
        self.gnuplotComm += " -slow "   #optionを追加


    #
    #  main
    #-------
    def main(self):
        """ GUIの表示"""
        #timerEvent（0.5s後）にplotStepMonitorを起動
        GLib.timeout_add(500.0, self.runPlotStepMonitor)
        #GUI表示
        self.mainWindow.show_all()
        Gtk.main()

    #
    #  close
    #---------
    def close(self, *args):
        """ GUIを閉じる"""
        self.runFlag = "stop"
        Gtk.main_quit(*args)

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        """ objectの定義"""
        self.label_status = self.builder.get_object("label_status")
        #文字をセット
        mess = _("plotStepMonitorが起動中\n停止する場合は、この画面を閉じてください。")
        self.builder.get_object("label_conts").set_text(mess)
        self.builder.get_object("button_stop").get_child().set_text(_("停止"))

    #
    #  getGnuplotComm
    #------------------
    def getGnuplotComm(self):
        """ gnuplotのコマンドを取得する。"""
        appDict = pyFistr.readUsingApplications()
        gnuplotComm = appDict["gnuplot"]
        return gnuplotComm

    #------------ event ---------
    def onClose(self, event):
        self.close()

    #
    #  runPlotStepMonitor
    #-----------------
    def runPlotStepMonitor(self):
        """ 各々のplotStepMonitorを起動する"""
        #fistr-ver5
        if self.version == "5":
            if self.anaType == "NLSTATIC":
                self.plotStepMonitorNL5()       #非線形
            elif self.anaType == "DYNAMIC":
                self.plotStepMonitorDY5()       #動解析（構造）
            elif self.anaType == "HEAT_D":
                self.plotStepMonitorHDY5()      #動解析（熱伝導）
        #fistr-ver4
        elif self.version[0] == "4":
            if self.anaType == "NLSTATIC":
                self.plotStepMonitorNL4()       #非線形
            elif self.anaType == "DYNAMIC":
                self.plotStepMonitorDY4()       #動解析（構造）
            elif self.anaType == "HEAT_D":
                self.plotStepMonitorHDY4()      #動解析（熱伝導）
        return False


    #------------- version 5 用 --------------------
    #
    #  checkFullData5
    #    全てのdataが取得できるまで行が取り込めたかcheck
    #    logFile側
    def checkFullData5(self, line, fullFlag):
        words = line.split()
        if fullFlag == 0:
            if len(words) > 3:
                if words[0] == "#####" and words[1] == "Global" and words[2] == "Summary" and words[3] == "@Node":
                    fullFlag += 1
        elif fullFlag == 1:
            if len(words) > 0:
                if words[0] == "//SMS":
                    fullFlag += 1
        return fullFlag

    #
    #  getPlotData5
    #    plotするデータをlinesから取得
    def getPlotData5(self, lines):

        #  checkError
        #    errorCheckし、dataを修正
        def checkError(data):
            for i in range(len(data)):
                try:
                    float(data[i])
                except:
                    #errorの場合
                    if data[i].find("E") < 0 and data[i].find("e") < 0:
                        #数列に「E」が抜けている場合
                        words = data[i].split("-")
                        if len(words) > 1:
                            #指数が「-」の場合
                            data[i] = "-".join(words[:-1]) + "E-" + words[-1]
                        else:
                            words = data[i].split("+")
                            if len(words) > 1:
                                #指数が「+」の場合
                                data[i] = "+".join(words[:-1]) + "E+" + words[-1]
            return data

        #変数の初期化
        dataList = []
        u1 = ""; u2 = ""; u3 = ""; data = []
        uFlag = 0
        #1行ごとに処理
        for line in lines:
            words = line.split()
            if len(words) >= 3:
                if words[0] == "####" and words[1] == "Result" and words[2][:len("step=")] == "step=":
                    u1 = ""; u2 = ""; u3 = ""; u1min = ""; u2min = ""; u3min = ""; data = []
                    n = line.find("=")
                    step = line[n+1:].split()[0]
                elif words[0] == "#####" and words[1] == "Global" and words[2] == "Summary" and words[3] == "@Node":
                    uFlag = 1
                else:
                    if uFlag == 1:
                        if words[0] == "//U1":
                            #u1 = words[1]; u1min = words[2]
                            u1 = words[1]; u1min = words[3]
                        elif words[0] == "//U2":
                            #u2 = words[1]; u2min = words[2]
                            u2 = words[1]; u2min = words[3]
                        elif words[0] == "//U3":
                            #u3 = words[1]; u3min = words[2]
                            u3 = words[1]; u3min = words[3]
                        elif words[0] == "//SMS":
                            sms = words[1]
                            uFlag = 0
                            data = [step, u1, u2, u3, u1min, u2min, u3min, sms]
                            data = checkError(data)
                            dataList.append(data)
                    else:
                        pass
        return dataList

    #
    #  checkFullFlag_fc5
    #    全てのdataが取得できるまで行が取り込めたかcheck
    #    staFile側
    def checkFullFlag_fc5(self, line):
        flag = 0
        words = line.split()
        if len(words) >= 10 and words[3] == "S":
            flag = 1
        return flag

    #
    #  dataPlotG5
    def dataPlotG5(self, allData, g):
        """ dataGをplot"""
        if len(allData) >= 2:
            #dataFile作成
            f = open(self.gaDataFile, "w", encoding="utf-8")      #gnuplotGa.plt
            for data in allData:
                line = " ".join(data) + ls.decode()
                f.write(line)
            f.close()
            #plot開始
            if self.gaPlotFlag == 0:
                #初回plotの場合
                g.stdin.write(b"set autoscale" + ls)
                name = ("'" + self.gaDataFile + "'").encode()    #dataFile名
                gLine = b"plot "
                gLine += name + b" using 1:2 with line title 'Ux max' linewidth 2, "
                gLine += name + b" using 1:3 with line title 'Uy max' linewidth 2, "
                gLine += name + b" using 1:4 with line title 'Uz max' linewidth 2, "
                gLine += name + b" using 1:5 with line title 'Ux min', "
                gLine += name + b" using 1:6 with line title 'Uy min', "
                gLine += name + b" using 1:7 with line title 'Uz min', "
                gLine += name + b" using 1:8 with line title 'mises stress max' axes x1y2" + ls
                g.stdin.write(gLine)
                self.gaPlotFlag = 1
            else:
                #2回目以降plotの場合
                g.stdin.write(b"replot" + ls)
            g.stdin.flush()         #入力データの吐き出し

    #
    #  dataPlotGc5
    def dataPlotGc5(self, fcData, gc):
        """ dataGcをplot"""
        if len(fcData) >= 2:
            #dataFile作成
            nData = len(fcData)
            f = open(self.gcDataFile, "w", encoding="utf-8")      #gnuplotGc.plt
            for i in range(nData):
                line = " ".join([str(i+1)] + fcData[i]) + ls.decode()
                f.write(line)
            f.close()
            #plot開始
            if self.gcPlotFlag == 0:
                gc.stdin.write(b"set autoscale" + ls)
                name = ("'" + self.gcDataFile + "'").encode()    #dataFile名
                gLine  = b"plot "
                gLine += name + b" using 1:2 with line title 'maxIter' linewidth 2, "
                gLine += name + b" using 1:3 with line title 'sumIter' linewidth 2, "
                gLine += name + b" using 1:4 with line title 'contIter' linewidth 2, "
                gLine += name + b" using 1:5 with line title 'incTime' axes x1y2" + ls
                gc.stdin.write(gLine)
                self.gcPlotFlag = 1
            else:
                gc.stdin.write(b"replot" + ls)
            gc.stdin.flush()

    #
    #  getHeatStepNo5
    def getHeatStepNo5(self, line):
        """ stepNoを取得"""
        step = ""
        words = line.split()
        if words[:2] == ["ISTEP", "="]:
            step = words[2]
        return step

    #
    #  getHeatStepTimeInLine5
    def getHeatStepTimeInLine5(self, line):
        """ stepの時間を取得する"""
        timeVal = ""
        words = line.split()
        if words[:2] == ["Time", "="]:
            timeVal = words[2]
        return timeVal

    #  getHeatStepDataInLine
    #    dataがある行かcheck
    def getHeatStepDataInLine5(self, line):
        stepVal = ""
        words = line.split()
        if words[:3] == ["Maximum", "Temperature(global)", ":"]:
            stepVal = words[3]
        return stepVal

    #
    #  getHeatStepLastDataInLine5(line)
    def getHeatStepLastDataInLine5(self, line):
        """ 最終データがある行かチェック。最終行ならば、値を返す。"""
        lastVal = ""
        words = line.split()
        if words[:3] == ["Minimum", "Temperature(global)", ":"]:
            lastVal = words[3]
        return lastVal

    #
    #  getHeatPlotData5
    #    plotするデータをlinesから取得
    def getHeatPlotData5(self, data):

        #  checkError
        #    errorCheckし、dataを修正
        def checkError(data):
            for i in range(len(data)):
                try:
                    a = float(data[i])
                except:
                    #errorの場合
                    if data[i].find("E") < 0 and data[i].find("e") < 0:
                        #数列に「E」が抜けている場合
                        words = data[i].split("-")
                        if len(words) > 1:
                            #指数が「-」の場合
                            data[i] = "-".join(words[:-1]) + "E-" + words[-1]
                        else:
                            words = data[i].split("+")
                            if len(words) > 1:
                                #指数が「+」の場合
                                data[i] = "+".join(words[:-1]) + "E+" + words[-1]
            return data

        #変数の初期化
        data = checkError(data)
        return data

    #  dataHeatPlot5
    #    dataを描画する
    def dataHeatPlot5(self, allData, g):
        if len(allData) >= 2:
            #dataFile作成
            f = open(self.gaDataFile, "w", encoding="utf-8")      #gnuplotDataGa.plt
            for data in allData:
                #line = " ".join(data) + "\n"
                line = " ".join(data) + ls.decode()
                f.write(line)
            f.close()
            #plot開始
            if self.gaPlotFlag == 0:
                #初回のplot
                g.stdin.write(b"set autoscale" + ls)
                dataFile = ("'" + self.gaDataFile + "'").encode() 
                gLine = b"plot "
                gLine += dataFile + b" using 1:2 with line title 'time', "
                gLine += dataFile + b" using 1:3 with line title 'max temperature' linewidth 2 axes x1y2, "
                gLine += dataFile + b" using 1:4 with line title 'min temperature' linewidth 2 axes x1y2" + ls
                g.stdin.write(gLine)
                self.gaPlotFlag = 1
            else:
                #2回目以降のplot
                g.stdin.write(b"replot" + ls)
            g.stdin.flush()
        return

    #
    #  compressData
    def compressData(self, dataList):
        """ dataList内のnComp分を残し、それ以外を1ヶ毎に削除して圧縮し、戻す。"""
        #圧縮対象の位置
        ns = len(dataList) - self.nComp
        if ns <= 0:
            return dataList
        #圧縮level
        #  nCompNest    最大圧縮level
        #   1            1/1    (10000個)
        #   2            1/2    ( 5000個)
        #   3            1/4    ( 2500個)
        #   4            1/8    ( 1250個)
        #   5            1/16   (  625個)
        #   6            1/32   (  312個)
        #
        #圧縮対象数 n
        n = 0
        for i in range(self.nCompNest):
            n += self.nComp // 2**i
        #圧縮せずkeepする位置
        nk = ns - n
        if nk < 0:
            nk = 0
        #各々データ取得
        keepList = dataList[:nk]        #keep（圧縮後data）
        compList = dataList[nk:ns]      #圧縮
        rowList = dataList[ns:]         #そのまま(最新data)
        #圧縮する
        newComp = []
        for i in range(0, len(compList), 2):
            newComp.append(compList[i])
        compData = keepList + newComp + rowList
        return compData

    #
    #  isMissRead
    def isMissRead(self, vals):
        """ 正しくdataが取得できたかどうか確認する。
        「""」空文字があるかどうか"""
        ans = False
        errVals = list(filter(lambda x: x=="", vals))
        if len(errVals) != 0:
            ans = True          #miss有り
        return ans

    #
    #  writeStatus
    def writeStatus(self, cont):
        """ GUI内のlabel_statusにcontを書き込む。
        通常はthreadから呼び出される為、timerEventで実行する。"""
        self.labelCont = cont
        GLib.timeout_add(100.0, self.writeLabel)

    def writeLabel(self):
        self.label_status.set_text(self.labelCont)

    #
    #  readAndPlotData_ga5
    def readAndPlotData_ga5(self, startTime, limitTime):
        """ 
        logFileを読み込み、plotする。（threadで実行）
        読み込めなくなった時（データが空）、limitTimeを超えた時戻る。
        戻り値:
            データが空:         "OK"
            limitTimeを超えた:  "overTime" 
        """
        #初期化済み？
        if self.ga == "":
            #初期化
            #gnuplot起動
            self.ga = subprocess.Popen(
                self.gnuplotComm,
                shell=True,
                stdin=subprocess.PIPE)
            self.ga.stdin.write(b"set ytics nomirror" + ls)
            self.ga.stdin.write(b"set y2tics" + ls)
            self.ga.stdin.write(b"set grid" + ls)
            self.ga.stdin.write(b"set xlabel 'step'" + ls)
            self.ga.stdin.write(b"set ylabel 'DISPLACEMENT'" + ls)
            self.ga.stdin.write(b"set y2label 'mises stress'" + ls)
            #logFileをopen
            self.fa = open(self.logFile, encoding="utf-8")
            #変数の初期化
            self.allData = []
            self.gaLines = []
            self.gaCount = 0
            self.gaFullFlag = 0
            self.writeStatus("getting data...")
        stTime = startTime
        timeFlag = 0
        plotFlag = 0
        fullFlag = self.gaFullFlag
        line = self.fa.readline()
        while line != "":
            self.gaLines.append(line)
            #全てのdataが取得できる行が取得できたかcheck
            fullFlag = self.checkFullData5(line, fullFlag)
            if fullFlag == 2:
                #dataを取得
                data = self.getPlotData5(self.gaLines)
                if self.isMissRead(data[0]) == False:
                    #missなし
                    self.allData += data
                    latestData = data[0][0]
                    #data数をcountUp
                    self.gaCount += 1
                    #data圧縮？
                    if self.gaCount % self.nComp == 0:
                        #dataを間引いて圧縮する
                        self.allData = self.compressData(self.allData)
                else:
                    print("read error: 0.log file is reading. skip " 
                        + str(self.gaCount+1) + " step.")
                #flagをクリア
                self.gaLines = []
                fullFlag = 0
                plotFlag = 1
                #制限時間オーバ？
                if time.time() > stTime + limitTime:
                    timeFlag = 1
                    break
            line = self.fa.readline()
        self.gaFullFlag = fullFlag
        #行が読み込めなかった場合plot
        if plotFlag == 1:
            if timeFlag == 0:
                #GUIへの出力
                text = "plot until " + str(latestData) + " step."
                self.writeStatus(text)
            #グラフ描画
            self.dataPlotG5(self.allData, self.ga)
        #戻り値を設定
        if timeFlag == 0:
            return "OK"
        else:
            return "overTime"

    #
    #  readAndPlotData_gc5
    def readAndPlotData_gc5(self, startTime, limitTime):
        """ 
        convFile(残渣)を読み込み、plotする。（threadが呼び出す）
        読み込めなくなった時（データが空）、limitTimeを超えた時戻る。
        戻り値:
            データが空:         "OK"
            limitTimeを超えた:  "overTime" 
        """
        #初期化済み？
        if self.gc == "":
            #初期化
            self.gc = subprocess.Popen(
                self.gnuplotComm,
                shell=True,
                stdin=subprocess.PIPE)
            self.gc.stdin.write(b"set ytics nomirror" + ls)
            self.gc.stdin.write(b"set y2tics" + ls)
            self.gc.stdin.write(b"set logscale y" + ls)
            self.gc.stdin.write(b"set logscale y2" + ls)
            self.gc.stdin.write(b"set grid" + ls)
            self.gc.stdin.write(b"set xlabel 'step'" + ls)
            self.gc.stdin.write(b"set ylabel 'number of iteration'" + ls)
            self.gc.stdin.write(b"set y2label 'inc time'" + ls)
            #ファイルをopen
            self.fc = open(self.convFile, encoding="utf-8")
            #変数の初期化
            self.fcData = []
            self.gcLines = []
            self.gcCount = 0
        #file読込
        stTime = startTime
        timeFlag = 0
        plotFlag = 0
        line = self.fc.readline()
        while line != "":
            fullFlag_fc = self.checkFullFlag_fc5(line)
            if fullFlag_fc == 1:
                #iterを読込
                words = line.split()
                contIter = words[4]
                maxIter = words[5]
                totalIter = words[6]
                incTime = words[8]
                data = [maxIter, totalIter, contIter, incTime]
                self.fcData.append(data)
                plotFlag = 1
                #data数をカウントアップ
                self.gcCount += 1
                #data圧縮?
                if self.gcCount % self.nComp == 0:
                    #dataを間引いて圧縮する
                    self.fcData = self.compressData(self.fcData)
                #制限時間オーバ？
                if time.time() > stTime + limitTime:
                    timeFlag = 1
                    break
            line = self.fc.readline()
        #グラフ描画？
        if plotFlag == 1:
            #グラフ表示
            self.dataPlotGc5(self.fcData, self.gc)
        #戻り値を設定
        if timeFlag == 0:
            return "OK"
        else:
            return "overTime"

    #
    #  readAndPlotData_Hga5
    def readAndPlotData_Hga5(self, startTime, limitTime):
        """ 
        HEAT_DのlogFileを読み込み、plotする。
        読み込めなくなった時（データが空）、limitTimeを超えた時戻る。
        戻り値:
            データが空:         "OK"
            limitTimeを超えた:  "overTime" 
        """
        #初期化済み？
        if self.ga == "":
            #初期化
            #gnuplotを起動
            self.ga = subprocess.Popen(
                self.gnuplotComm, 
                shell=True, 
                stdin=subprocess.PIPE)
            self.ga.stdin.write(b"set ytics nomirror" + ls)
            self.ga.stdin.write(b"set y2tics" + ls)
            self.ga.stdin.write(b"set grid" + ls)
            self.ga.stdin.write(b"set xlabel 'step'" + ls)
            self.ga.stdin.write(b"set ylabel 'time'" + ls)
            self.ga.stdin.write(b"set y2label 'temperature'" + ls)
            #logファイルをopen
            self.fa = open(self.logFile, encoding="utf-8")
            #変数の初期化
            self.allData = []
            self.gaLines = []
            self.gaCount = 0
            #self.label_status.set_text("getting data...")
            self.writeStatus("getting data...")
            self.step = ""
            self.timeVal = ""
        #file読込
        stTime = startTime
        timeFlag = 0
        plotFlag = 0
        line = self.fa.readline()
        maxVal = ""                 #未定義エラー発生の為追加
        while line != "":
            self.gaLines.append(line)
            plotFlag = 1
            minVal = ""
            #step数を取得
            _step = self.getHeatStepNo5(line)
            if _step != "":
                self.step = _step
            else:
                _timeVal = self.getHeatStepTimeInLine5(line)
                if _timeVal != "":
                    self.timeVal = _timeVal
                else:
                    #値を取得(maxTemp)
                    _maxVal = self.getHeatStepDataInLine5(line)
                    if _maxVal != "":
                         maxVal = _maxVal
                    else:
                        _minVal = self.getHeatStepLastDataInLine5(line)
                        if _minVal != "":
                            minVal = _minVal
            if minVal != "" and maxVal != "":
                stepData = [self.step, self.timeVal, maxVal, minVal]
                #dataを取得
                stepData = self.getHeatPlotData5(stepData)
                self.allData += [stepData]
                #flagをクリア
                self.gaLines = []
                self.gaCount += 1
                #data圧縮？
                if self.gaCount % self.nComp == 0:
                    #dataを間引いて圧縮する
                    self.allData = self.compressData(self.allData)
                #制限時間オーバ？
                currTime = time.time()
                if currTime > stTime + limitTime:
                    timeFlag = 1
                    break
            line = self.fa.readline() 
        #plot可能？
        if plotFlag == 1:
            #グラフ表示する
            if timeFlag == 0:
                #self.label_status.set_text("plot until " + str(self.gaCount) + " step.")
                cont = "plot until " + str(self.gaCount) + " step."
                self.writeStatus(cont)
            self.dataHeatPlot5(self.allData, self.ga)
        #戻り値を設定
        if timeFlag == 0:
            return "OK"
        else:
            return "overTime"

    #----------------------
    #  plotStepMonitorNL5
    #----------------------
    def plotStepMonitorNL5(self):
        """ ver-5、非線形解析用のplotStepMonitorを起動。
        timerEventで定期的にここが実行される。
        処理は、threadを起動する。"""
        if self.runFlag == "run":
            self.setThreadIntervalFromGaCount()
            th = threading.Thread(target=self.plotStepMonitorNL5_thread)
            #th.setDaemon(True)
            th.daemon = True
            th.start()

    #----------------------
    #  plotStepMonitorDY5
    #----------------------
    def plotStepMonitorDY5(self):
        """ ver-5、動解析用のplotStepMonitorを起動。
        timerEventで定期的にここが実行される。
        処理は、threadを起動する。"""
        if self.runFlag == "run":
            self.setThreadIntervalFromGaCount()
            th = threading.Thread(target=self.plotStepMonitorDY5_thread)
            #th.setDaemon(True)
            th.daemon = True
            th.start()

    #----------------------
    #  plotStepMonitorHDY5
    #----------------------
    def plotStepMonitorHDY5(self):
        """ ver-5、解析用のplotStepMonitorを起動。
        timerEventで定期的にここが実行される。
        処理は、threadを起動する。"""
        if self.runFlag == "run":
            self.setThreadIntervalFromGaCount()
            th = threading.Thread(target=self.plotStepMonitorHDY5_thread)
            #th.setDaemon(True)
            th.daemon = True
            th.start()

    #
    #  setThreadIntervalFromGaCount
    #-------------------------------
    def setThreadIntervalFromGaCount(self):
        """ threadIntervalをself.gaCount数で決定する"""
        if self.gaCount < 1000:
            # 1000回まで: 5s
            self.waitTime = plotInterval
        elif self.gaCount < 5000:
            # 5000回まで: 8s
            self.waitTime = plotInterval * 1.6
        elif self.gaCount < 10000:
            # 10000回まで: 10s
            self.waitTime = plotInterval * 2.0
        elif self.gaCount < 50000:
            # 50000回まで: 20s
            self.waitTime = plotInterval * 4.0
        else:
            # 50000回以上: 30s
            self.waitTime = plotInterval * 6.0

    #
    #  plotStepMonitorNL5_thread
    #----------------------------
    def plotStepMonitorNL5_thread(self):
        """ ver-5、非線形用のplotStepMonitorを起動
        timerEventで定期的にここが実行される。"""
        #logFile側を読み込みplot
        #limitTime = 1.0
        limitTime = self.waitTime * 0.2
        startTime = time.time()
        result = self.readAndPlotData_ga5(startTime, limitTime)
        if result == "OK":
            #convFile側を読み込みplot
            result = self.readAndPlotData_gc5(startTime, limitTime)
        #終了処理
        if self.runFlag == "run":
            waitTime = self.waitTime * 1000
            if result == "overTime":
                waitTime = 0.2 * 1000.0
            GLib.timeout_add(waitTime, self.plotStepMonitorNL5)
        else:
            self.fa.close()
            self.fc.close()
        return False

    #
    #  plotStepMonitorDY5_thread
    #-----------------------------
    def plotStepMonitorDY5_thread(self):
        """ ver-5、動解析用のplotStepMonitorを起動。
        threadで実行する。
        thread内でtime.sleep()の使用、GUIを操作しない。"""
        #limitTime = 1.0
        limitTime = self.waitTime * 0.2
        startTime = time.time()
        self.threadRes = "OK"
        result = self.readAndPlotData_ga5(startTime, limitTime)
        self.threadRes = result
        #終了処理
        if self.runFlag == "run":
            waitTime = self.waitTime * 1000.0
            if result == "overTime":
                waitTime = 0.2 * 1000.0
            GLib.timeout_add(waitTime, self.plotStepMonitorDY5)
        else:
            self.fa.close()
        return False

    #
    #  plotStepMonitorHDY5_thread
    #-----------------------------
    def plotStepMonitorHDY5_thread(self):
        """ ver-5、非定常熱解析用のplotStepMonitorを起動。
        timerEventで定期的にここが実行される。"""
        #fileを読込plot
        #limitTime = 1.0
        limitTime = self.waitTime * 0.2
        startTime = time.time()
        result = self.readAndPlotData_Hga5(startTime, limitTime)
        #終了処理
        if self.runFlag == "run":
            waitTime = self.waitTime * 1000.0
            #if timeFlag == 1:
            if result == "overTime":
                waitTime = 0.2 * 1000.0
            GLib.timeout_add(waitTime, self.plotStepMonitorHDY5)
        else:
            self.fa.close()
        return False


    #------------- version 4 用 --------------------
    #
    #  checkFullData4
    #    全てのdataが取得できるまで行が取り込めたかcheck
    #    logFile側
    def checkFullData4(self, line, fullFlag):
        words = line.split()
        if fullFlag == 0:
            if words[0] == "#####" and words[1] == "Global":
                fullFlag += 1
        elif fullFlag == 1:
            if words[0] == "//SMS":
                fullFlag += 1
        return fullFlag

    #
    #  getPlotData4
    #    plotするデータをlinesから取得
    def getPlotData4(self, lines):

        #  checkError
        #    errorCheckし、dataを修正
        def checkError(data):
            for i in range(len(data)):
                try:
                    a = float(data[i])
                except:
                    #errorの場合
                    if data[i].find("E") < 0 and data[i].find("e") < 0:
                        #数列に「E」が抜けている場合
                        words = data[i].split("-")
                        if len(words) > 1:
                            #指数が「-」の場合
                            data[i] = "-".join(words[:-1]) + "E-" + words[-1]
                        else:
                            words = data[i].split("+")
                            if len(words) > 1:
                                #指数が「+」の場合
                                data[i] = "+".join(words[:-1]) + "E+" + words[-1]
            return data

        #変数の初期化
        dataList = []
        u1 = ""; u2 = ""; u3 = ""; data = []
        uFlag = 0
        #1行ごとに処理
        for line in lines:
            words = line.split()
            if len(words) >= 3:
                if words[0] == "####" and words[1] == "Result" and words[2] == "step=":
                    u1 = ""; u2 = ""; u3 = ""; u1min = ""; u2min = ""; u3min = ""; data = []
                    step = words[3]
                elif words[0] == "#####" and words[1] == "Global" and words[2] == "Summary":
                    uFlag = 1
                else:
                    if uFlag == 1:
                        if words[0] == "//U1":
                            u1 = words[1]; u1min = words[2]
                        elif words[0] == "//U2":
                            u2 = words[1]; u2min = words[2]
                        elif words[0] == "//U3":
                            u3 = words[1]; u3min = words[2]
                        elif words[0] == "//SMS":
                            sms = words[1]
                            uFlag = 0
                            data = [step, u1, u2, u3, u1min, u2min, u3min, sms]
                            data = checkError(data)
                            dataList.append(data)
                    else:
                        pass
        return dataList

    #
    #  checkFullFlag_fc4
    #    全てのdataが取得できるまで行が取り込めたかcheck
    #    staFile側
    def checkFullFlag_fc4(self, line):
        flag = 0
        words = line.split()
        if words[0] == "###" and words[1] == "Converged":
            flag = 1
        return flag

    #
    #  getNIteration4
    #    イタレーション回数を取得する
    def getNIteration4(self, fcLine):
        nIter = ""
        words = fcLine.split()
        for i in range(len(words)):
            if words[i] == "iter=":
                nIter = words[i+1]
                break
        return nIter

    #  dataPlotG4
    #    dataを描画する
    def dataPlotG4(self, allData, g):
        if len(allData) >= 2:
            #g.stdin.write(b'$stepData << EOD\n')
            g.stdin.write(b'$stepData << EOD' + ls)
            for data in allData:
                #line = data[0] + " " + data[1] + " " + data[2] + " " + data[3] + " " + data[4] + " " + data[5] + " " + data[6] + " " + data[7] + "\n"
                line = data[0] + " " + data[1] + " " + data[2] + " " + data[3] + " " + data[4] + " " + data[5] + " " + data[6] + " " + data[7] + ls.decode()
                g.stdin.write(line.encode())
            #g.stdin.write(b"EOD\n")
            g.stdin.write(b"EOD" + ls)
            gLine = b"plot "
            gLine += b"$stepData using 1:2 with line title 'Ux max' linewidth 2, "
            gLine += b"$stepData using 1:3 with line title 'Uy max' linewidth 2, "
            gLine += b"$stepData using 1:4 with line title 'Uz max' linewidth 2, "
            gLine += b"$stepData using 1:5 with line title 'Ux min', "
            gLine += b"$stepData using 1:6 with line title 'Uy min', "
            gLine += b"$stepData using 1:7 with line title 'Uz min', "
            #gLine += b"$stepData using 1:8 with line title 'mises stress max' axes x1y2\n"
            gLine += b"$stepData using 1:8 with line title 'mises stress max' axes x1y2" + ls
            g.stdin.write(gLine)
            g.stdin.flush()         #入力データの吐き出し

    #  dataPlotGc4
    #    dataを描画する
    def dataPlotGc4(self, fcData, gc):
        if len(fcData) >= 2:
            #gc.stdin.write(b"$stepData << EOD\n")
            gc.stdin.write(b"$stepData << EOD" + ls)
            nData = len(fcData)
            for i in range(nData):
                #line = str(i+1) + " " + fcData[i][0] + " " + fcData[i][1] + " " + fcData[i][2] + "\n"
                line = str(i+1) + " " + fcData[i][0] + " " + fcData[i][1] + " " + fcData[i][2] + ls.decode()
                gc.stdin.write(line.encode())
            #gc.stdin.write(b"EOD\n")
            gc.stdin.write(b"EOD" + ls)
            gLine = b"plot "
            gLine += b"$stepData using 1:2 with line title 'residual(1)' linewidth 2, "
            gLine += b"$stepData using 1:3 with line title 'residual(2)' linewidth 2, "
            #gLine += b"$stepData using 1:4 with line title 'number of iteration'\n"
            gLine += b"$stepData using 1:4 with line title 'number of iteration'" + ls
            gc.stdin.write(gLine)
            gc.stdin.flush()        #入力データの吐き出し

    #  getHeatStepDataInLine4
    #    dataがある行かcheck
    def getHeatStepDataInLine4(self, line):
        stepData = []
        words = line.split()
        if len(words) > 5:
            if words[0].upper() == "STEP":
                nStep = words[2]
                timeValue = words[5]
                stepData = [nStep, timeValue]
        return stepData

    #
    #  getHeatPlotData4
    #    plotするデータをlinesから取得
    def getHeatPlotData4(self, data):

        #  checkError
        #    errorCheckし、dataを修正
        def checkError(data):
            for i in range(len(data)):
                try:
                    a = float(data[i])
                except:
                    #errorの場合
                    if data[i].find("E") < 0 and data[i].find("e") < 0:
                        #数列に「E」が抜けている場合
                        words = data[i].split("-")
                        if len(words) > 1:
                            #指数が「-」の場合
                            data[i] = "-".join(words[:-1]) + "E-" + words[-1]
                        else:
                            words = data[i].split("+")
                            if len(words) > 1:
                                #指数が「+」の場合
                                data[i] = "+".join(words[:-1]) + "E+" + words[-1]
            return data

        #変数の初期化
        data = checkError(data)
        return data

    #  dataHeatPlot4
    #    dataを描画する
    def dataHeatPlot4(self, allData, g):
        if len(allData) >= 2:
            #g.stdin.write(b"$stepData << EOD\n")
            g.stdin.write(b"$stepData << EOD" + ls)
            for data in allData:
                #line = " ".join(data) + "\n"
                line = " ".join(data) + ls.decode()
                g.stdin.write(line.encode())
            #g.stdin.write(b"EOD\n")
            g.stdin.write(b"EOD" + ls)
            gLine = b"plot "
            #gLine += b"$stepData using 1:2 with line title 'time'\n"
            gLine += b"$stepData using 1:2 with line title 'time'" + ls
            g.stdin.write(gLine)
            g.stdin.flush()
        return

    #
    #  plotStepMonitorNL4
    #------------------
    def plotStepMonitorNL4(self):
        """ ver-4、非線形用のplotStepMonitorを起動
        timerEventで定期的にここが実行される。"""
        #初期化済み？
        if self.ga == "":
            #初期化
            #gnuplotを起動
            g = subprocess.Popen(
                self.gnuplotComm, 
                shell=True, 
                stdin=subprocess.PIPE)
            # g.stdin.write(b"set ytics nomirror\n")
            # g.stdin.write(b"set y2tics\n")
            # g.stdin.write(b"set grid\n")
            # g.stdin.write(b"set xlabel 'step'\n")
            # g.stdin.write(b"set ylabel 'DISPLACEMENT'\n")
            # g.stdin.write(b"set y2label 'mises stress'\n")
            g.stdin.write(b"set ytics nomirror" + ls)
            g.stdin.write(b"set y2tics" + ls)
            g.stdin.write(b"set grid" + ls)
            g.stdin.write(b"set xlabel 'step'" + ls)
            g.stdin.write(b"set ylabel 'DISPLACEMENT'" + ls)
            g.stdin.write(b"set y2label 'mises stress'" + ls)
            self.ga = g

            gc = subprocess.Popen(
                self.gnuplotComm, 
                shell=True, 
                stdin=subprocess.PIPE)
            # gc.stdin.write(b"set logscale y\n")
            # gc.stdin.write(b"set grid\n")
            # gc.stdin.write(b"set format y '%.1E'\n")
            # gc.stdin.write(b"set xlabel 'step'\n")
            # gc.stdin.write(b"set ylabel 'residual, number of iteration'\n")
            gc.stdin.write(b"set logscale y" + ls)
            gc.stdin.write(b"set grid" + ls)
            gc.stdin.write(b"set format y '%.1E'" + ls)
            gc.stdin.write(b"set xlabel 'step'" + ls)
            gc.stdin.write(b"set ylabel 'residual, number of iteration'" + ls)
            self.gc = gc

            #logファイルをopen
            self.fa = open(self.logFile, encoding="utf-8")
            self.fc = open(self.convFile, encoding="utf-8")
            #変数の初期化
            self.allData = []; self.fcData = []
            self.gaLines = []; self.gcLines = []
            self.gaCount = 0; self.gcCount = 0
            self.gaFullFlag = 0
            self.label_status.set_text("getting data...")

        stTime = time.time()
        timeFlag = 0
        #fa側（allData）を処理
        plotFlag = 0
        fullFlag = self.gaFullFlag
        line = self.fa.readline()
        while line != "":
            self.gaLines.append(line)
            #全てのdataが取得できる行が取得できたかcheck
            fullFlag = self.checkFullData4(line, fullFlag)
            if fullFlag == 2:
                #dataを取得
                data = self.getPlotData4(self.gaLines)
                self.allData += data
                latestData = data[0][0]
                #flagをクリア
                self.gaLines = []
                fullFlag = 0
                plotFlag = 1
                #data数をcountUp
                self.gaCount += 1
                #data圧縮？
                if self.gaCount % self.nComp == 0:
                    #dataを間引いて圧縮する
                    self.allData = self.compressData(self.allData)
                #制限時間オーバ？
                if time.time() > stTime + 1.0:
                    timeFlag = 1
                    break
            line = self.fa.readline()
        self.gaFullFlag = fullFlag
        if plotFlag == 1:
            text = "plot until " + latestData + " step."
            self.label_status.set_text(text)
            #グラフ表示
            self.dataPlotG4(self.allData, self.ga)

        #fc側（fcData）を処理
        plotFlag = 0
        fcLines = []
        line = self.fc.readline()
        while line != "":
            fullFlag_fc = self.checkFullFlag_fc4(line)
            if fullFlag_fc == 1:
                #残渣データを取得
                words = fcLines[-1].split()
                res1 = words[-2]
                res2 = words[-1]
                #fcLineからイタレーション回数を取得
                nIter = self.getNIteration4(line)
                data = [res1, res2, nIter]
                self.fcData.append(data)
                fcLines = []
                plotFlag = 1
                #data数をカウントアップ
                self.gcCount += 1
                #data圧縮?
                if self.gcCount % self.nComp == 0:
                    #dataを間引いて圧縮する
                    self.fcData = self.compressData(self.fcData)
                #制限時間オーバ？
                if time.time() > stTime + 1.0:
                    timeFlag = 1
                    break
            else:
                fcLines.append(line)
            line = self.fc.readline()
        if plotFlag == 1:
            self.dataPlotGc4(self.fcData, self.gc)
            self.label_status.set_text("ploted until " + str(self.gcCount) + " step.")
        #終了処理
        if self.runFlag == "run":
            waitTime = self.waitTime * 1000
            if timeFlag == 1:
                waitTime = 0.2 * 1000.0
            GLib.timeout_add(waitTime, self.plotStepMonitorNL4)
        else:
            self.fa.close()
            self.fc.close()
        return False

    #
    #  plotStepMonitorDY4
    #------------------
    def plotStepMonitorDY4(self):
        """ ver-4、動解析用のplotStepMonitorを起動。
        timerEventで定期的にここが実行される。"""
        #初期化済み？
        if self.ga == "":
            #初期化
            #gnuplot起動
            g = subprocess.Popen(
                self.gnuplotComm,
                shell=True,
                stdin=subprocess.PIPE)
            # g.stdin.write(b"set ytics nomirror\n")
            # g.stdin.write(b"set y2tics\n")
            # g.stdin.write(b"set grid\n")
            # g.stdin.write(b"set xlabel 'step'\n")
            # g.stdin.write(b"set ylabel 'DISPLACEMENT'\n")
            # g.stdin.write(b"set y2label 'mises stress'\n")
            g.stdin.write(b"set ytics nomirror" + ls)
            g.stdin.write(b"set y2tics" + ls)
            g.stdin.write(b"set grid" + ls)
            g.stdin.write(b"set xlabel 'step'" + ls)
            g.stdin.write(b"set ylabel 'DISPLACEMENT'" + ls)
            g.stdin.write(b"set y2label 'mises stress'" + ls)
            self.ga = g
            #logファイルをopen
            self.fa = open(self.logFile, encoding="utf-8")
            #変数の初期化
            self.allData = []
            self.gaLines = []
            self.gaCount = 0
            self.gaFullFlag = 0
            self.label_status.set_text("getting data...")
        stTime = time.time()
        timeFlag = 0
        plotFlag = 0
        fullFlag = self.gaFullFlag
        line = self.fa.readline()
        while line != "":
            self.gaLines.append(line)
            #全てのdataが取得できる行が取得できたかcheck
            fullFlag = self.checkFullData4(line, fullFlag)
            if fullFlag == 2:
                #dataを取得
                data = self.getPlotData4(self.gaLines)
                self.allData += data
                latestData = data[0][0]
                #flagをクリア
                self.gaLines = []
                fullFlag = 0
                plotFlag = 1
                #data数をcountUp
                self.gaCount += 1
                #data圧縮？
                if self.gaCount % self.nComp == 0:
                    #dataを間引いて圧縮する
                    self.allData = self.compressData(self.allData)
                #制限時間オーバ？
                if time.time() > stTime + 1.0:
                    timeFlag = 1
                    break
            line = self.fa.readline()
        self.gaFullFlag = fullFlag
        #グラフ作成可能？
        if plotFlag == 1:
            text = "plot until " + latestData + " step."
            self.label_status.set_text(text)
            self.dataPlotG4(self.allData, self.ga)
        #終了処理
        if self.runFlag == "run":
            waitTime = self.waitTime * 1000.0
            if timeFlag == 1:
                waitTime = 0.2 * 1000.0
            GLib.timeout_add(waitTime, self.plotStepMonitorDY4)
        else:
            self.fa.close()
        return False

    #
    #  plotStepMonitorHDY4
    #------------------
    def plotStepMonitorHDY4(self):
        """ ver-4、非定常熱解析用のplotStepMonitorを起動。
        timerEventで定期的にここが実行される。"""
        #初期化済み？
        if self.ga == "":
            #初期化
            g = subprocess.Popen(
                self.gnuplotComm, 
                shell=True, 
                stdin=subprocess.PIPE)
            # g.stdin.write(b"set grid\n")
            # g.stdin.write(b"set xlabel 'step'\n")
            # g.stdin.write(b"set ylabel 'time'\n")
            g.stdin.write(b"set grid" + ls)
            g.stdin.write(b"set xlabel 'step'" + ls)
            g.stdin.write(b"set ylabel 'time'" + ls)
            self.ga = g
            #logファイルをopen
            self.fa = open(self.logFile, encoding="utf-8")
            #変数の初期化
            self.allData = []
            self.gaLines = []
            self.gaCount = 0
            self.label_status.set_text("getting data...")
        stTime = time.time()
        timeFlag = 0
        plotFlag = 0
        line = self.fa.readline()
        while line != "":
            self.gaLines.append(line)
            plotFlag = 1
            stepData = self.getHeatStepDataInLine4(line)
            if len(stepData) == 2:
                #dataを取得
                data = self.getHeatPlotData4(stepData)
                self.allData += [data]
                #flagをクリア
                self.gaLines = []
                self.gaCount += 1
                #data圧縮？
                if self.gaCount % self.nComp == 0:
                    #dataを間引いて圧縮する
                    self.allData = self.compressData(self.allData)
                #制限時間オーバ？
                if time.time() > stTime + 1.0:
                    timeFlag = 1
                    break
            line = self.fa.readline() 
        #plot可能？
        if plotFlag == 1:
            #グラフ表示する
            self.dataHeatPlot4(self.allData, self.ga)
            self.label_status.set_text("ploted until " + str(self.gaCount) + " step.")
        #終了処理
        if self.runFlag == "run":
            waitTime = self.waitTime * 1000.0
            if timeFlag == 1:
                waitTime = 0.2 * 1000.0
            GLib.timeout_add(waitTime, self.plotStepMonitorHDY4)
        else:
            self.fa.close()
        return False


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", localeDir)
    #gettext.install("app")
    #_ = gettext.gettext

    #log表示
    import logFileCreater
    logFileCreater.log()

    fistrVersion = sys.argv[1]      # "5" or "4"
    anaType = sys.argv[2]           # "NLSTATIC" or "DYNAMIC" or "HEAT_D"

    app = windowApp(fistrVersion, anaType)
    app.main()

