#!/usr/bin/python3
# coding: utf-8
#
#       nodesFacesBondingDDialog.py
#
#       節点結合の設定Dialog
#       node-face, node-nodeの結合
#
#   20/0816     新規作成（nodeGrp, surfaceGrpのみ）
#      12/31    elmGrpの取得を追加。
#               nodesBondingDDialog.py → nodesFacesBondingDDialog.py
#               に名称を変更
#   21/01/11    createEquationNodeToFace:node-face間距離が三角形の3辺の
#               平均値よりも離れていた場合、nodeを結合しない様に修正。
#      01/15    createEquationNodeToFace:node-faceのpairの検索を高速化
#      02/26    getEachLengthForEquation:zero dev error発生の為、修正
#               getNearFacesFromSlaveNodes:minMaxの取得方法修正。
#      07/11    createEquationNodeToFace:shellを追加
#   22/03/14    国際化
#      03/22    国際化（文字をpython側でセット）
#   24/05/23    nodesFacesBonding:新規追加
#      06/06    universalDialogsに変更。
#      06/08    okDialog,errDialog:parentを追加
#      06/09    dialogApp:parentを追加
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib

import os
import GtkParts
import pyFistr
import geometryFunc as geo
import convertMesh
import universalDialogs

#import locale
localeDir = os.getenv("LOCALE_DIR")
#locale.bindtextdomain("easistr", localeDir)

ls = "\n"

#-------------------
#  dialogApp class
#-------------------
class dialogApp:

    def __init__(self, nodeGrps, surGrps, bondNN, bondNF, meshHeaderData, parent=None):
        self.builder = Gtk.Builder()
        path = os.getenv("easyIstrPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "nodesFacesBondingDDialog.glade")
        self.dialog = self.builder.get_object("dialog1")
        self.dialog.set_transient_for(parent)
        self.dialog.connect("delete-event", self.close)
        self.dialog.set_title(_("節点結合"))
        self.builder.connect_signals(self)
        self.setGtkObject()
        #変数の設定
        self.stat = "CANCEL"
        self.data = []              #meshHeaderData
        self.nodeGrps = nodeGrps
        self.surGrps = surGrps
        self.bondNN = bondNN
        self.bondNF = bondNF
        self.meshHeaderData = meshHeaderData
        self.nodeDict = {}
        #初期化
        self.initialize()

    #  show
    def show(self):
        """ dialog表示"""
        self.dialog.show()

    #  close
    def close(self, *args):
        """ dialogを閉じる"""
        self.dialog.destroy()

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        self.tree_grpsSlave = self.builder.get_object("tree_grpsSlave")
        self.tree_grpsMaster = self.builder.get_object("tree_grpsMaster")
        self.tree_bondNF = self.builder.get_object("tree_bondNF")
        self.tree_grpsNN = self.builder.get_object("tree_grpsNN")
        self.tree_bondNN = self.builder.get_object("tree_bondNN")
        self.button_cancel = self.builder.get_object("button_cancel")
        self.button_ok = self.builder.get_object("button_ok")
        self.button_set_p0 = self.builder.get_object("button_set_p0")
        self.button_ret_p0 = self.builder.get_object("button_ret_p0")
        self.button_set_p1 = self.builder.get_object("button_set_p1")
        self.button_ret_p1 = self.builder.get_object("button_ret_p1")
        #文字をセット
        self.builder.get_object("label_subTitle").set_text(_("!EQUATIONによる節点結合（部品の結合）"))
        self.builder.get_object("button_cancel").get_child().set_text(_("キャンセル"))
        #faceNode page (page0)
        self.builder.get_object("label_pageTitle_p0").set_text(_("  <bondNF>\nface面上にnodeを結合"))
        mess = _("NGRP(slave)のnodeをSGRP(master)のface面上に結合する。\nmaster面は、slaveエリアよりも広くする。（master >= slave）\nmaster面からはみ出るnodeは、結合されない。")
        self.builder.get_object("label_method_p0").set_text(mess)
        self.builder.get_object("label_setNodeBond_p0").set_text(_("節点結合の設定"))
        mess = _("結合するペアのslaveとmasterを選択して\n「設定>>」ボタンをクリックして設定する。")
        self.builder.get_object("label_bondConts_p0").set_text(mess)
        self.builder.get_object("label_slave_p0").set_text(_("NGRP名(slave)"))
        self.builder.get_object("label_master_p0").set_text(_("SGRP名(master)"))
        self.builder.get_object("label_setGrp_p0").set_text(_("結合するGroup名"))
        self.builder.get_object("button_set_p0").get_child().set_text(_("設定>>"))
        self.builder.get_object("button_ret_p0").get_child().set_text(_("<<戻す"))
        #nodeNode page (page1)
        self.builder.get_object("label_pageTitle_p1").set_text(_("  <bondNN>\nnodeとnodeを結合"))
        mess = _("nodeGroup同士の節点を結合する。\n節点変位を最も近い相手節点の変位に合わせる（結合する）。\n結合するnodeGroupの節点数は、極力合わせる。")
        self.builder.get_object("label_method_p1").set_text(mess)
        self.builder.get_object("label_setBond_p1").set_text(_("節点結合の設定"))
        mess = _("結合するペアのnodeGroupを選択して\n「設定>>」ボタンをクリックして設定する。")
        self.builder.get_object("label_conts_p1").set_text(mess)
        self.builder.get_object("label_ngrp_p1").set_text(_("nodeGroup名"))
        self.builder.get_object("label_setNgrp_p1").set_text(_("結合するGroup名"))
        self.builder.get_object("button_set_p1").get_child().set_text(_("設定>>"))
        self.builder.get_object("button_ret_p1").get_child().set_text(_("<<戻す"))


    #
    #  initialize
    #--------------
    def initialize(self):
        self.treeList_grpsSlave = GtkParts.treeList(self.tree_grpsSlave)
        self.treeList_grpsSlave.create(multi=False)
        self.treeList_grpsMaster = GtkParts.treeList(self.tree_grpsMaster)
        self.treeList_grpsMaster.create(multi=False)
        self.treeList_bondNF = GtkParts.treeList(self.tree_bondNF)
        self.treeList_bondNF.create()
        self.treeList_grpsNN = GtkParts.treeList(self.tree_grpsNN)
        self.treeList_grpsNN.create()
        self.treeList_bondNN = GtkParts.treeList(self.tree_bondNN)
        self.treeList_bondNN.create()
        self.setItemsToAllTreeList()
        self.nodeDict = self.createNodeLocationDict()

    #
    #  getUnsetGrpNames
    def getUnsetGrpNames(self):
        """ bondNN, bondNFでセットされていないnodeGrps, surGrpsを取得して返す。"""
        setNgrps = []
        #setSgrps = []
        for ngrp, sgrp in self.bondNF:
            setNgrps.append(ngrp)
            #setSgrps.append(sgrp)
        for ngrp1, ngrp2 in self.bondNN:
            setNgrps += [ngrp1, ngrp2]
        unsetNgrps = list(set(self.nodeGrps).difference(setNgrps))
        #unsetSgrps = list(set(self.surGrps).difference(setSgrps))
        #return (unsetNgrps, unsetSgrps)
        return unsetNgrps

    #
    #  getSetGrpNames
    def getSetGrpNames(self):
        """ bondNN, bondNFからsetItemsを取得する"""
        itemsNF = []
        for ngrp, sgrp in self.bondNF:
            name = "bondNF " + ngrp + " " + sgrp
            itemsNF.append(name)
        itemsNN = []
        for ngrp1, ngrp2 in self.bondNN:
            name = "bondNN " + ngrp1 + " " + ngrp2
            itemsNN.append(name)
        return itemsNF, itemsNN

    #
    #  setItemsToAllTreeList
    def setItemsToAllTreeList(self):
        """ 全treeListにitemをセットする。"""
        #(ngrps, sgrps) = self.getUnsetGrpNames()
        ngrps = self.getUnsetGrpNames()
        sgrps = self.surGrps
        ngrps.sort()
        sgrps.sort()
        self.treeList_grpsSlave.setItems(ngrps)
        self.treeList_grpsMaster.setItems(sgrps)
        self.treeList_grpsNN.setItems(ngrps)
        (bondNFs, bondNNs) = self.getSetGrpNames()
        bondNFs.sort()
        bondNNs.sort()
        self.treeList_bondNF.setItems(bondNFs)
        self.treeList_bondNN.setItems(bondNNs)

    #
    #  createNodeLocationDict
    def createNodeLocationDict(self):
        """ 節点Noをキーとして座標が取得できる辞書を作成する"""
        nodeDict = {}
        for headerData in self.meshHeaderData:
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0].upper() == "!NODE":
                for nodeData in headerData[1]:
                    nodeNo = nodeData[0]
                    loc = nodeData[1:]
                    if nodeNo in nodeDict.keys():
                        nodeDict[nodeNo]["loc"] = loc
                    else:
                        nodeDict[nodeNo] = {"loc":loc}
        return nodeDict

    #
    #  setBondNF
    #-------------
    def setBondNF(self):
        """ bondNFの設定側に移動する"""
        ngrps = self.treeList_grpsSlave.getSelectedItems()
        sgrps = self.treeList_grpsMaster.getSelectedItems()
        if len(ngrps) != 1 or len(sgrps) != 1:
            title = _("エラー")
            mess = _("slave, master側とも1ヶづつ選択してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        ngrp = ngrps[0]
        sgrp = sgrps[0]
        self.bondNF += [[ngrp, sgrp]]
        self.setItemsToAllTreeList()
        selItem = "bondNF " + ngrp + " " + sgrp
        self.treeList_bondNF.selectItems([selItem])
        self.treeList_grpsMaster.selectItems([sgrp])

    #
    #  setBondNN
    #------------
    def setBondNN(self):
        """ bondNNの設定側に移動する"""
        ngrps = self.treeList_grpsNN.getSelectedItems()
        if len(ngrps) != 2:
            title = _("エラー")
            mess = _("ペアとなるNGRP名を2ヶ選択してください。")
            self.errDialog(title, mess, parent=self.dialog)
        self.bondNN += [ngrps]
        self.setItemsToAllTreeList()
        selItem = "bondNN " + " ".join(ngrps)
        self.treeList_bondNN.selectItems([selItem])

    #
    #  returnBondNF
    #---------------
    def returnBondNF(self):
        """ 選択されたbondNFの設定を戻す。"""
        selItems = self.treeList_bondNF.getSelectedItems()
        selNgrps = []
        selSgrps = []
        for selItem in selItems:
            words = selItem.split()
            ngrp = words[1]
            sgrp = words[2]
            selNgrps.append(ngrp)
            selSgrps.append(sgrp)
            idx = self.bondNF.index([ngrp, sgrp])
            _item = self.bondNF.pop(idx)
        self.setItemsToAllTreeList()
        self.treeList_grpsSlave.selectItems(selNgrps)
        self.treeList_grpsMaster.selectItems(selSgrps)

    #
    #  returnBondNN
    #---------------
    def returnBondNN(self):
        """ 選択されたbondNNの設定を戻す。"""
        selItems = self.treeList_bondNN.getSelectedItems()
        selNgrps = []
        for selItem in selItems:
            words = selItem.split()
            ngrp1 = words[1]
            ngrp2 = words[2]
            selNgrps += [ngrp1, ngrp2]
            idx = self.bondNN.index([ngrp1, ngrp2])
            _item = self.bondNN.pop(idx)
        self.setItemsToAllTreeList()
        self.treeList_grpsNN.selectItems(selNgrps)
        
    #
    #  createBondingData
    #--------------------
    def createBondingData(self):
        """ bondNF, bondNNのEQUATIONを作成する。"""
        #bondNNのEQUATIONを作成
        headerDataBondNN = self.createHeaderDataBondNN()
        #bondNFのEQUATIONを作成
        headerDataBondNF = self.createHeaderDataBondNF()
        #addHeader作成
        addHeaderData = headerDataBondNN + headerDataBondNF
        #確認用dialog表示
        title = _("部品の結合")
        bonds = []
        for headerData in addHeaderData:
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "#bondNN" or words[0] == "#bondNF":
                cont = " ".join(words)[1:] + ls
                bonds.append(cont)
        if len(bonds) == 0:
            self.deleteAddBondingData(addHeaderData)
            mess = _("nodeの結合データ（!EQUATION)を削除しました。")
            funcOk = [self.close]
            self.okDialog(title, mess, self.dialog, funcOk)
        else:
            self.deleteAddBondingData(addHeaderData)
            mess = _("以下のnode結合データ（!EQUATION)を作成しました。") + ls
            conts = "  " + " ".join(bonds)
            mess += conts
            funcOk = [self.close]
            self.okDialog(title, mess, self.dialog, funcOk)

    #  deleteAddBondingData
    def deleteAddBondingData(self, addHeaderData):
        #bondDataを削除
        self.meshHeaderData = self.deleteBondingData()
        #addHeaderを追加
        self.addBondingData(addHeaderData)

    #
    #  addBondingData
    def addBondingData(self, addHeaderData):
        """ bondNF, bondNNのheaderDataをmeshHeaderDataに追加する"""
        #挿入位置を取得
        st = len(self.meshHeaderData) - 2
        i = len(self.meshHeaderData) - 1
        while i >= 0:
            header = self.meshHeaderData[i][0]
            if header[:len("!END")].upper() == "!END":
                st = i
                break
            i -= 1
        if i < 0:
            self.meshHeaderData += addHeaderData + [["!END", []]]
        else:
            self.meshHeaderData = self.meshHeaderData[:st] + addHeaderData + self.meshHeaderData[st:]

    #
    #  createHeaderDataBondNN
    #-------------------------
    def createHeaderDataBondNN(self):
        """ bondNNのEQUATIONを作成する"""
        bondConts = self.treeList_bondNN.getAllItems()
        addHeader = []
        for bond in bondConts:
            words = bond.split()
            if words[0] == "bondNN":
                addHeader += self.createEquationNodeToNode(words[1:3])
        return addHeader

    #
    #  createEquationNodeToNode
    #---------------------------
    def createEquationNodeToNode(self, grps):
        """ nodeToNodeのEQUATIONを作成する"""
        #nodeGrpのdataを取得
        nodeGrps = []
        count = 0
        for headerData in self.meshHeaderData:
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0].upper() == "!NGROUP":
                name = pyFistr.getValName(header, "NGRP")
                if name in grps:
                    nodeGrps.append([name, headerData[1]])
                    count += 1
                    if count == 2:
                        break
        nodeGrps.sort()
        #EQUATIONを作成
        pairNodes = self.getBondPairNodes(nodeGrps)
        addHeader = []
        headerData = ["#bondNN, " + nodeGrps[0][0] + ", " + nodeGrps[1][0] + ls, []]
        addHeader.append(headerData)
        header = "!EQUATION" + ls
        data = []
        for pairNode in pairNodes:
            node0 = str(pairNode[0])
            node1 = str(pairNode[1])
            data += ["2, 0.0" + ls]
            data += [node0 + ", 1, 1, " + node1 + ", 1, -1" + ls]
            data += ["2, 0.0" + ls]
            data += [node0 + ", 2, 1, " + node1 + ", 2, -1" + ls]
            data += ["2, 0.0" + ls]
            data += [node0 + ", 3, 1, " + node1 + ", 3, -1" + ls]
        addHeader.append([header, data])
        return addHeader

    #
    #  getBondPairNodes
    def getBondPairNodes(self, nodeGrps):
        """ bondするpairの節点を取得する"""
        #nodeGrps[0]側を作成
        pairNodes = []
        for nodeNo in nodeGrps[0][1]:
            disps = []
            for pairNode in nodeGrps[1][1]:
                disp = geo.length(self.nodeDict[nodeNo]["loc"], self.nodeDict[pairNode]["loc"])
                disps.append([disp, pairNode])
            disps.sort()
            pairNodes.append([nodeNo, disps[0][1]])
        #nodeGrps[1]側をチェック
        setNodes = map(lambda x: x[1], pairNodes)
        unsetNodes = set(nodeGrps[1][1]).difference(setNodes)
        for nodeNo in unsetNodes:
            disps = []
            for pairNode in nodeGrps[0][1]:
                disp = geo.length(self.nodeDict[nodeNo]["loc"], self.nodeDict[pairNode]["loc"])
                disps.append([disp, pairNode])
            disps.sort()
            pairNodes.append([nodeNo, disps[0][1]])
        return pairNodes

    #
    #  createHeaderDataBondNF
    #-------------------------
    def createHeaderDataBondNF(self):
        """ bondNFのEQUATIONを作成する"""
        bondConts = self.treeList_bondNF.getAllItems()
        addHeader = []
        for bond in bondConts:
            words = bond.split()
            if words[0] == "bondNF":
                ngrp = words[1]
                sgrp = words[2]
                addHeader += self.createEquationNodeToFace(ngrp, sgrp)
        return addHeader

    #
    #  createEquationNodeToFace
    #----------------------------
    def createEquationNodeToFace(self, ngrp, sgrp):
        """ nodeToFaceのEQUATIONを作成する"""

        def getElementDict(meshHeaderData):
            """ 要素のdictを取得し、戻す"""
            elmDict = {}
            for headerData in meshHeaderData:
                header = headerData[0]
                words = pyFistr.deleteSp(header).split(",")
                if words[0] == "!ELEMENT":
                    elType = pyFistr.getValName(header, "TYPE")
                    for elm in headerData[1]:
                        elmNo = elm[0]
                        nodes = elm[1:]
                        elmDict[elmNo] = [nodes, elType]
            return elmDict

        def getMinLength(point, tri):
            """ pointとtriの頂点間距離のminを取得する"""
            lls = []
            for p in tri:
                l = geo.length(point, p)
                lls.append(l)
            lmin = min(lls)
            return lmin

        def getEachLengthForEquation(p, tri, nodeNo):
            """ pに関して、各長さを取得する"""
            #座標軸を取得（原点：tri[0]）
            vecx = geo.vector(tri[0], tri[1])
            vecx = geo.normal(vecx)
            vec13 = geo.vector(tri[0], tri[2])
            vecz = geo.crossProduct(vecx, vec13)
            vecz = geo.normal(vecz)
            vecy = geo.crossProduct(vecz, vecx)
            vecy = geo.normal(vecy)
            #pの座標を取得
            vecp = geo.vector(tri[0], p)
            apx = geo.dotProduct(vecx, vecp)
            apy = geo.dotProduct(vecy, vecp)
            #cの座標を取得
            if abs(apx) > 1.0e-8:
                ap = apy / apx      #線1-pの勾配
                vec23 = geo.vector(tri[1], tri[2])
                a23x = geo.dotProduct(vecx, vec23)
                a23y = geo.dotProduct(vecy, vec23)
                if abs(a23x) > 1.0e-8:
                    a23 = a23y / a23x   #線2-3の勾配
                    x2 = geo.length(tri[0], tri[1])
                    cx = (-a23*x2)/(ap-a23)
                    cy = ap*cx
                else:
                    #直行する場合(x軸ー線23)
                    x2 = geo.length(tri[0], tri[1])
                    cx = x2
                    cy = ap * x2
            else:
                #直行する場合
                vec23 = geo.vector(tri[1], tri[2])
                a23x = geo.dotProduct(vecx, vec23)
                a23y = geo.dotProduct(vecy, vec23)
                # a23 = a23y / a23x   #線2-3の勾配
                # x2 = geo.length(tri[0], tri[1])
                # cx = 0.0
                # cy = -a23 * x2
                if abs(a23x) > 1.0e-8:
                    a23 = a23y / a23x   #線2-3の勾配
                    x2 = geo.length(tri[0], tri[1])
                    cx = 0.0
                    cy = -a23 * x2
                else:
                    x2 = geo.length(tri[0], tri[1])
                    cx = x2
                    cy = 0.0
            #各線分の長さを取得
            l23 = geo.length(tri[1], tri[2])
            l2 = geo.length2([x2, 0.0], [cx, cy])
            l3 = l23 - l2
            l1c = geo.length2([0.0, 0.0], [cx, cy])
            l1 = geo.length2([0.0, 0.0], [apx, apy])
            lc = l1c - l1
            return l1, l2, l3, lc

        def createTriangleFromFace(faceNodes, elType):
            """ 要素のfaceからtriangleを作成する"""
            
            def getFaces3nds(faceNodes):
                faces = [faceNodes]
                return faces
            
            def getFaces4nds(faceNodes):
                #face1 = faceNodes[:3]
                #face2 = [faceNodes[2], faceNodes[3], faceNodes[0]]
                face1 = [faceNodes[0], faceNodes[1], faceNodes[3]]
                face2 = [faceNodes[1], faceNodes[2], faceNodes[3]]
                faces = [face1, face2]
                return faces

            def getFaces6nds(faceNodes):
                face1 = [faceNodes[0], faceNodes[1], faceNodes[5]]
                face2 = [faceNodes[1], faceNodes[2], faceNodes[3]]
                face3 = [faceNodes[3], faceNodes[4], faceNodes[5]]
                face4 = [faceNodes[1], faceNodes[3], faceNodes[5]]
                faces = [face1, face2, face3, face4]
                return faces

            def getFaces8nds(faceNodes):
                face1 = [faceNodes[0], faceNodes[1], faceNodes[7]]
                face2 = [faceNodes[1], faceNodes[2], faceNodes[3]]
                face3 = [faceNodes[3], faceNodes[4], faceNodes[5]]
                face4 = [faceNodes[5], faceNodes[6], faceNodes[7]]
                face5 = [faceNodes[1], faceNodes[3], faceNodes[5]]
                face6 = [faceNodes[5], faceNodes[7], faceNodes[1]]
                faces = [face1, face2, face3, face4, face5, face6]
                # face1 = [faceNodes[0], faceNodes[2], faceNodes[4]]
                # face2 = [faceNodes[4], faceNodes[6], faceNodes[0]]
                # faces = [face1, face2]
                return faces

            masterFaces = []
            #solid
            if elType == "341":
                masterFaces = getFaces3nds(faceNodes)
            elif elType == "361":
                masterFaces = getFaces4nds(faceNodes)
            elif elType == "351":
                if len(faceNodes) == 3:
                    masterFaces = getFaces3nds(faceNodes)
                elif len(faceNodes) == 4:
                    masterFaces = getFaces4nds(faceNodes)
            elif elType == "342":
                masterFaces = getFaces6nds(faceNodes)
            elif elType == "362":
                masterFaces = getFaces8nds(faceNodes)
            elif elType == "352":
                if len(faceNodes) == 6:
                    masterFaces = getFaces6nds(faceNodes)
                elif len(faceNodes) == 8:
                    masterFaces = getFaces8nds(faceNodes)
            #shell
            elif elType == "761" or elType == "731":
                masterFaces = getFaces3nds(faceNodes)
            elif elType == "781" or elType == "741":
                masterFaces = getFaces4nds(faceNodes)
            elif elType == "732":
                masterFaces = getFaces6nds(faceNodes)
            return masterFaces

        def getNearFacesFromSlaveNodes(slaveNodes, masterFaceNodes, nodeDict, maxLength):
            """ slaveNodesに近いmasterFacesNodesを返す。"""
            #座標のminMaxを取得
            locs = []
            for nodeNo in slaveNodes:
                loc = nodeDict[nodeNo]["loc"]
                locs.append(loc)
            locx = list(map(lambda x: x[0], locs))
            locy = list(map(lambda x: x[1], locs))
            locz = list(map(lambda x: x[2], locs))
            minxyz = [min(locx), min(locy), min(locz)]
            maxxyz = [max(locx), max(locy), max(locz)]
            if abs(minxyz[0] - maxxyz[0]) < maxLength * 2:
                minxyz[0] -= maxLength
                maxxyz[0] += maxLength
            if abs(minxyz[1] - maxxyz[1]) < maxLength * 2:
                minxyz[1] -= maxLength
                maxxyz[1] += maxLength
            if abs(minxyz[2] - maxxyz[2]) < maxLength * 2:
                minxyz[2] -= maxLength
                maxxyz[2] += maxLength
            minMax = [minxyz, maxxyz]
            #nearFacesを取得
            nearFaces = []
            for nodes in masterFaceNodes:
                for nodeNo in nodes:
                    loc = nodeDict[nodeNo]["loc"]
                    if geo.isPointInBlock(minMax, loc) == True:
                        nearFaces.append(nodes)
                        break
            return nearFaces

        print()
        print("connect " + ngrp + " to " + sgrp + ".")
        masterGrp = sgrp
        slaveGrp = ngrp
        meshHeaderData = self.meshHeaderData
        elmDict = getElementDict(meshHeaderData)
        nodeDict = self.nodeDict
        #faceDataを取得
        masterFaces = []
        slaveNodes = []
        for headerData in meshHeaderData:
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!SGROUP":
                grpName = pyFistr.getValName(header, "SGRP")
                if grpName == masterGrp:
                    masterFaces = headerData[1]
            elif words[0] == "!NGROUP":
                grpName = pyFistr.getValName(header, "NGRP")
                if grpName == slaveGrp:
                    slaveNodes = headerData[1]
        #faceDataを　要素No, 面No を nodeNoに置き換え
        masterFaceNodes = []
        for i in range(0, len(masterFaces), 2):
            elmNo = masterFaces[i]
            faceNo = masterFaces[i+1]
            nodes, elType = elmDict[elmNo]
            faceNos = convertMesh.faceNodeOfElementDict[elType][faceNo]
            faceNodes = list(map(lambda x: nodes[x-1], faceNos))
            #triのfaceを取得し、追加する
            masterFaceNodes += createTriangleFromFace(faceNodes, elType)
        #triの3辺のmax値を求める
        print("  getting masterFaces...")
        maxLengthDict = {}
        for faceNodeNo in masterFaceNodes:
            nodes = tuple(faceNodeNo)
            tri = list(map(lambda x: nodeDict[x]["loc"], faceNodeNo))
            #aveLen = sum(geo.tri3length(tri)) / 3.0
            maxLen = max(geo.tri3length(tri))
            maxLengthDict[nodes] = maxLen
        #nodesに近いfacesを取得する
        maxLength = max(maxLengthDict.values()) * 1.5
        nearMasterFaces = getNearFacesFromSlaveNodes(slaveNodes, masterFaceNodes, nodeDict, maxLength)
        #pairのnode,triFfaceを取得
        print("  getting connected pairs of node and face...")
        slaveMasterSets = []
        for nodeNo in slaveNodes:
            #nodeがbox内かどうか確認
            nodeLoc = nodeDict[nodeNo]["loc"]
            minLoc = [nodeLoc[0]-maxLen, nodeLoc[1]-maxLen, nodeLoc[2]-maxLen]
            maxLoc = [nodeLoc[0]+maxLen, nodeLoc[1]+maxLen, nodeLoc[2]+maxLen]
            minMax = [minLoc, maxLoc]
            if geo.isPointInBlock(minMax, nodeLoc) == False:
                #block外の場合
                continue
            #nearMsterFaces内からpairのfaceを取得
            nodeLoc = nodeDict[nodeNo]["loc"]
            flag = 0
            lenFaces = []
            for faceNodeNo in nearMasterFaces:
                tri = list(map(lambda x: nodeDict[x]["loc"], faceNodeNo))
                n, l = geo.isPointInFaceOverValue(tri, nodeLoc, 0.01)
                maxLen = maxLengthDict[tuple(faceNodeNo)]
                if n == 0:
                    #nodeの投影点がface内に存在する場合
                    if l < maxLen:
                        slaveMasterSets.append([nodeNo, faceNodeNo])
                        flag = 1
                        break
                elif n == 1:
                    #nodeの投影点がface近くに存在する場合
                    if l < maxLen:
                        #node-face距離が小さい場合
                        ll = getMinLength(nodeLoc, tri)
                        lenFaces.append([ll, faceNodeNo])
            if flag == 0:
                if len(lenFaces) > 0:
                    lenFaces.sort()
                    faceNodeNo = lenFaces[0][1]
                    slaveMasterSets.append([nodeNo, faceNodeNo])
        print("    " + str(len(slaveMasterSets)) + " pairs were gotten.")
        #各pairで系数を取得
        print("  getting cofficient of Equation...")
        eqCoeffs = []
        for slaveMaster in slaveMasterSets:
            nodeNo, faceNodeNo = slaveMaster
            p = nodeDict[nodeNo]["loc"]
            tri = list(map(lambda x: nodeDict[x]["loc"], faceNodeNo))
            #各寸法を取得
            l1, l2, l3, lc = getEachLengthForEquation(p, tri, nodeNo)
            l1c = l1 + lc
            l23 = l2 + l3
            #系数を計算
            coeff1 = lc / l1c
            coeff2 = l1 * l3 / (l1c * l23)
            coeff3 = l1 * l2 / (l1c * l23)
            coeff4 = -1.0
            coeffs = [coeff1, coeff2, coeff3, coeff4]
            #系数を保存
            eqCoeffs.append(coeffs)
        #equationのheaderDataを作成
        addHeaderData = []
        addHeaderData += [["#bondNF, " + slaveGrp + ", " + masterGrp + ls, []]]
        header = "!EQUATION" + ls
        data = []
        for i in range(len(eqCoeffs)):
            [coeff1, coeff2, coeff3, coeff4] = eqCoeffs[i]
            [nodeNo, faceNodeNo] = slaveMasterSets[i]
            for n in range(1, 4):
                #係数が1e-8以下は無視する
                vals = []
                if coeff1 > 1.0e-8:
                    vals += [faceNodeNo[0], n, coeff1]
                if coeff2 > 1.0e-8:
                    vals += [faceNodeNo[1], n, coeff2]
                if coeff3 > 1.0e-8:
                    vals += [faceNodeNo[2], n, coeff3]
                vals += [nodeNo, n, coeff4]
                #項数を求める
                nItems = len(vals) // 3
                line = str(nItems) + ", 0.0" + ls
                data += [line]
                #式を求める
                line = ", ".join(list(map(str, vals))) + ls
                data += [line]
        addHeaderData += [[header, data]]
        print("    " + str(len(slaveMasterSets)) + " / " + str(len(slaveNodes)) + " nodes will be connected to faces.")
        return addHeaderData

    #
    #  deleteBondingData
    def deleteBondingData(self):
        """ bondingData(EQUATION)を削除する"""
        newHeaderData = []
        flag = 0
        for headerData in self.meshHeaderData:
            header = headerData[0]
            if (header[:len("#bondNN")] == "#bondNN" or
                header[:len("#bondNF")] == "#bondNF"):
                flag = 1
            else:
                if flag == 0:
                    newHeaderData.append(headerData)
                else:
                    flag = 0
        return newHeaderData


    def okDialog(self, title, mess, parent=None, funcOk=[]):
        dialog = universalDialogs.okDialog(title, mess, parent, funcOk)
        dialog.show()

    def errDialog(self, title, message, parent=None, funcOk=[]):
        dialog = universalDialogs.errDialog(title, message, parent, funcOk)
        dialog.show()


#-------------------------------------------
#  nodesFacesBonding class
#-------------------------------------------
class nodesFacesBonding:
    """ nodes、facesをEquationで結合するdialogを表示、処理するdialog。
    
    Args:
        nodeGrps (list): nodeGrp名のlist
        surGrps (list): surfaceGrps名のlist
        bondNN (list):  結合するnodeGrp名のペアのlist
        bondNF (list):  結合するnodeGrp,surGrp名のペアのlist
        meshHeaderData(list):   meshHeaderData
        funcName (object);  結合のペア取得後の処理関数名"""
    
    def __init__(self, nodeGrps, surGrps, bondNN, bondNF, meshHeaderData, funcName, funcArgs, parent=None):
        self.nodeGrps = nodeGrps
        self.surGrps = surGrps
        self.bondNN = bondNN
        self.bondNF = bondNF
        self.meshHeaderData = meshHeaderData
        self.funcName = funcName
        self.funcArgs = funcArgs
        self.stat = "CANCEL"
        self.dialog = dialogApp(nodeGrps, surGrps, bondNN, bondNF, meshHeaderData, parent)
        self.dialog.button_cancel.connect("clicked", self.onCancel)
        self.dialog.button_ok.connect("clicked", self.onOk)
        self.dialog.button_set_p0.connect("clicked", self.onSetNF)
        self.dialog.button_ret_p0.connect("clicked", self.onRetNF)
        self.dialog.button_set_p1.connect("clicked", self.onSetNN)
        self.dialog.button_ret_p1.connect("clicked", self.onRetNN)

    def show(self):
        self.dialog.show()

    def close(self):
        self.dialog.close()

    def onCancel(self, event):
        self.stat = "CANCEL"
        self.close()

    def onOk(self, event):
        self.stat = "OK"
        self.dialog.createBondingData()     #この中でGUIをcloseする
        self.meshHeaderData = self.dialog.meshHeaderData
        path, item = self.funcArgs
        self.funcName(self.stat, self.meshHeaderData, path, item)

    def onSetNF(self, event):
        self.dialog.setBondNF()

    def onRetNF(self, event):
        self.dialog.returnBondNF()

    def onSetNN(self, event):
        self.dialog.setBondNN()

    def onRetNN(self, event):
        self.dialog.returnBondNN()

    