#!/usr/bin/python3
#  coding: utf-8
#
#   getBeamRefDirDDialog.py
#
#       beamのEGRPとその他のベクトルでbeamのz軸を取得する。
#
#   21/02/16    新規作成
#   22/03/14    国際化
#      03/22    国際化（文字をpython側でセット）
#   24/05/24    getBeamRefDir:overload用のクラスを追加
#      06/08    errDialog:parentを追加
#      06/09    dialogApp:parentを追加
#      07/02    settingDataToGui:entryが非表示のままなので、修正。
#      07/07    openにencoding="utf-8"を追加
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib

import os, sys

import universalDialogs

#import locale
localeDir = os.getenv("LOCALE_DIR")
#locale.bindtextdomain("easyistr", localeDir)

ls = "\n"

#-------------------
#  dialogApp class
#-------------------
class dialogApp:

    def __init__(self, parent=None):
        self.builder = Gtk.Builder()
        #self.builder.set_translation_domain("easyistr")
        path = os.getenv("easyIstrPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "getBeamRefDirDDialog.glade")
        self.dialog = self.builder.get_object("dialog1")
        self.dialog.set_transient_for(parent)
        self.dialog.connect("delete-event", self.close)
        self.dialog.set_title(_("参考z軸方向の取得"))
        self.builder.connect_signals(self)
        self.setGtkObject()
        #変数の定義
        self.stat = []
        self.settingFile = os.sep.join([
            os.getenv("easyIstrUserPath"), 
            "data", 
            "getBeamRefDirDDialog_data"])
        self.errFlag = 0
        #self.closeErrDialogFlag = 0
        #初期化
        self.initialize()

    #  show
    def show(self):
        self.dialog.show()

    #  close
    def close(self, *args):
        self.dialog.destroy()

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        self.button_cancel = self.builder.get_object("button_cancel")
        self.button_ok = self.builder.get_object("button_ok")
        self.radio_x = self.builder.get_object("radio_x")
        self.radio_y = self.builder.get_object("radio_y")
        self.radio_z = self.builder.get_object("radio_z")
        self.radio_ngrp = self.builder.get_object("radio_ngrp")
        self.radio_egrp = self.builder.get_object("radio_egrp")
        self.radio_other = self.builder.get_object("radio_other")
        self.entry_x = self.builder.get_object("entry_x")
        self.entry_y = self.builder.get_object("entry_y")
        self.entry_z = self.builder.get_object("entry_z")
        self.entry_ngrp = self.builder.get_object("entry_ngrp")
        self.entry_egrp = self.builder.get_object("entry_egrp")
        #文字をセット
        self.builder.get_object("label_subTitle").set_text(_("EGRPのみでは、平面が取得できません。"))
        self.builder.get_object("lavel_getMethod").set_text(_("以下から追加するベクトルを選択してください。"))
        mess = _("EGRPと追加したベクトル（x, y, z軸, NGRP, EGRP or 任意）で\n面を取得後、その面の垂直方向をbeamのZ軸方向として取得する。")
        self.builder.get_object("label_getMethodConts").set_text(mess)
        self.builder.get_object("label_addVector").set_text(_("追加ベクトル"))
        self.radio_x.get_child().set_text(_("x 軸"))
        self.radio_y.get_child().set_text(_("y 軸"))
        self.radio_z.get_child().set_text(_("z 軸"))
        self.radio_ngrp.get_child().set_text(_("NGRP（NGRPとEGRPのベクトル）NGRP名:"))
        self.radio_egrp.get_child().set_text(_("EGRP（他EGRPのベクトル）EGRP名:"))
        self.radio_other.get_child().set_text(_("任意"))
        self.builder.get_object("button_cancel").get_child().set_text(_("キャンセル"))

    #
    #  initialize
    #-------------
    def initialize(self):
        """ 初期化"""
        #x軸を選択
        self.radio_x.get_active()
        #entryを非アクティブ化
        self.entry_x.set_sensitive(False)
        self.entry_y.set_sensitive(False)
        self.entry_z.set_sensitive(False)
        self.entry_ngrp.set_sensitive(False)
        self.entry_egrp.set_sensitive(False)
        #設定を読み込む
        settingDict = self.getSettingDataFromFile()
        self.settingDataToGui(settingDict)

    #
    #  changeRadio
    def changeRadio(self, event):
        """ radioボタンを変更した時の処理"""
        if event.props.active == True:
            self.entry_x.set_sensitive(False)
            self.entry_y.set_sensitive(False)
            self.entry_z.set_sensitive(False)
            self.entry_ngrp.set_sensitive(False)
            self.entry_egrp.set_sensitive(False)
            if self.radio_other.get_active():
                self.entry_x.set_sensitive(True)
                self.entry_y.set_sensitive(True)
                self.entry_z.set_sensitive(True)
            elif self.radio_ngrp.get_active():
                self.entry_ngrp.set_sensitive(True)
            elif self.radio_egrp.get_active():
                self.entry_egrp.set_sensitive(True)

    #
    #  getRefDir
    def getRefDir(self):
        if self.radio_x.get_active() == True:
            self.stat = ["1.0", "0.0", "0.0"]
        elif self.radio_y.get_active() == True:
            self.stat = ["0.0", "1.0", "0.0"]
        elif self.radio_z.get_active() == True:
            self.stat = ["0.0", "0.0", "1.0"]
        elif self.radio_ngrp.get_active() == True:
            self.stat = ["NGRP", self.entry_ngrp.get_text()]
        elif self.radio_egrp.get_active() == True:
            self.stat = ["EGRP", self.entry_egrp.get_text()]
        elif self.radio_other.get_active() == True:
            x = self.entry_x.get_text()
            y = self.entry_y.get_text()
            z = self.entry_z.get_text()
            self.stat = [x, y, z]
            try:
                float(x); float(y); float(z)
            except:
                title = _("エラー")
                mess = _("任意ベクトルが取得できません。") + "\n"
                mess += _("  任意ベクトル値を入力し直してください。")
                self.errDialog(title, mess, parent=self.dialog, funcOk=[])
                #errFlagをセット
                self.errFlag = 1
                return
        #errFlagをクリア
        self.errFlag = 0

    #
    #  getSettingDataFromFile
    #-----------------------------
    def getSettingDataFromFile(self):
        """ settingFileから設定を読み込む"""
        settingDict = {}
        if os.path.exists(self.settingFile) == False:
            return settingDict
        f = open(self.settingFile, encoding="utf-8"); lines = f.readlines(); f.close()
        #コメント削除
        newLines = []
        for i in range(len(lines)):
            line = lines[i]
            n = line.find("#")
            line = line[:n] + ls
            newLines.append(line)
        #内容を取得
        for line in newLines:
            words = line.split()
            if len(words) >= 2:
                settingDict[words[0]] = words[1]
            elif len(words) == 1 and words != []:
                settingDict[words[0]] = ""
        return settingDict

    #
    #  getSettingDataFromGui
    #------------------------
    def getSettingDataFromGui(self):
        """ settingDataをGUIから取得"""
        settingDict = {}
        if self.radio_x.get_active() == True:
            settingDict["activeRadio"] = "radio_x"
        elif self.radio_y.get_active() == True:
            settingDict["activeRadio"] = "radio_y"
        elif self.radio_z.get_active() == True:
            settingDict["activeRadio"] = "radio_z"
        elif self.radio_other.get_active() == True:
            settingDict["activeRadio"] = "radio_other"
        elif self.radio_ngrp.get_active() == True:
            settingDict["activeRadio"] = "radio_ngrp"
        elif self.radio_egrp.get_active() == True:
            settingDict["activeRadio"] = "radio_egrp"
        settingDict["entry_x"] = self.entry_x.get_text()
        settingDict["entry_y"] = self.entry_y.get_text()
        settingDict["entry_z"] = self.entry_z.get_text()
        settingDict["entry_ngrp"] = self.entry_ngrp.get_text()
        settingDict["entry_egrp"] = self.entry_egrp.get_text()
        return settingDict

    #
    #  settingDataToGui
    #--------------------
    def settingDataToGui(self, settingDict):
        """ dictの内容をセットする"""
        radioDict = {
            "radio_x": self.radio_x,
            "radio_y": self.radio_y,
            "radio_z": self.radio_z,
            "radio_other": self.radio_other,
            "radio_ngrp": self.radio_ngrp,
            "radio_egrp": self.radio_egrp
            }
        entryDict = {
            "entry_x": self.entry_x,
            "entry_y": self.entry_y,
            "entry_z": self.entry_z,
            "entry_ngrp": self.entry_ngrp,
            "entry_egrp": self.entry_egrp
            }
        for item in settingDict.keys():
            if item == "activeRadio":
                radioName = settingDict["activeRadio"]
                radio = radioDict[radioName]
                radio.set_active(True)
                #entryを設定（radioの設定に応じて）
                if self.radio_other.get_active():
                    self.entry_x.set_sensitive(True)
                    self.entry_y.set_sensitive(True)
                    self.entry_z.set_sensitive(True)
                elif self.radio_ngrp.get_active():
                    self.entry_ngrp.set_sensitive(True)
                elif self.radio_egrp.get_active():
                    self.entry_egrp.set_sensitive(True)
            elif (item == "entry_x" or item == "entry_y"
                    or item == "entry_z" or item == "entry_ngrp"
                    or item == "entry_egrp"):
                entry = entryDict[item]
                text = settingDict[item]
                entry.set_text(text)

    #
    #  saveSettingFile
    #------------------
    def saveSettingFile(self, settingDict):
        """ settingFileに設定を保存"""
        if os.path.exists(self.settingFile) == False:
            #内容を作成する
            lines = []
            lines += ["#\n"]
            lines += ["#  getBeamRefDirDDialog.py のsettingData\n"]
            lines += ["#\n"]
            lines += ["\n"]
            lines += ["activeRadio     " + settingDict["activeRadio"] + "\n"]
            lines += ["entry_x         " + settingDict["entry_x"] + "\n"]
            lines += ["entry_y         " + settingDict["entry_y"] + "\n"]
            lines += ["entry_z         " + settingDict["entry_z"] + "\n"]
            lines += ["entry_ngrp      " + settingDict["entry_ngrp"] + "\n"]
            lines += ["entry_egrp      " + settingDict["entry_egrp"] + "\n"]
        else:
            f = open(self.settingFile, encoding="utf-8"); lines= f.readlines(); f.close()
            for item in settingDict.keys():
                flag = 0
                for i in range(len(lines)):
                    line = lines[i]
                    words = line.split()
                    if len(words) >= 2:
                        if words[0] == item:
                            words[1] = settingDict[item]
                            line = " ".join(words) + ls
                            lines[i] = line
                            flag = 1
                            break
                    elif len(words) == 1:
                        if words[0] == item:
                            line = " ".join([words[0], settingDict[item]]) + ls
                            lines[i] = line
                            flag = 1
                            break
                if flag == 0:
                    line = " ".join([item, settingDict[item]]) + ls
                    lines.append(line)
        f = open(self.settingFile, "w", encoding="utf-8"); f.writelines(lines); f.close()

    def errDialog(self, title, message, parent=None, funcOk=[]):
        dialog = universalDialogs.errDialog(title, message, parent, funcOk)
        dialog.show()


#------------------------------------
#  getBeamRefDir class
#------------------------------------
class getBeamRefDir:
    """ beamの参照方向の取得する方法を選択するdialogの表示、参照方向の取得。
    このクラスをoverloadして使う。"""

    def __init__(self, funcName, funcArgs, parent=None):
        self.funcName = funcName
        self.funcArgs = funcArgs
        self.dialog = dialogApp(parent)
        self.dialog.radio_x.connect("toggled", self.onChangeRadio)
        self.dialog.radio_y.connect("toggled", self.onChangeRadio)
        self.dialog.radio_z.connect("toggled", self.onChangeRadio)
        self.dialog.radio_ngrp.connect("toggled", self.onChangeRadio)
        self.dialog.radio_egrp.connect("toggled", self.onChangeRadio)
        self.dialog.radio_other.connect("toggled", self.onChangeRadio)
        self.dialog.button_cancel.connect("clicked", self.onCancel)
        self.dialog.button_ok.connect("clicked", self.onOk)

    def show(self):
        self.dialog.show()

    def close(self):
        self.dialog.close()

    def onChangeRadio(self, event):
        self.dialog.changeRadio(event)

    def onCancel(self, event):
        self.dialog.stat = []
        self.close()

    def onOk(self, event):
        self.dialog.getRefDir()
        if self.dialog.errFlag == 1:
            #error発生時は戻る
            return
        settingDict = self.dialog.getSettingDataFromGui()
        self.dialog.saveSettingFile(settingDict)
        self.close()
        vec2 = self.dialog.stat
        egrp, path, item = self.funcArgs
        self.funcName(vec2, egrp, path, item)



if __name__ == "__main__":
    import gettext
    #gettext.install("app")
    #_ = gettext.gettext
    gettext.install("easyistr", localeDir)

    dialog = dialogApp()
    stat = dialog.show()
