#!/usr/bin/python3
#  coding: utf-8
#
#   editBeamElementDDialog.py
#
#       Beam要素のみで構成されたメッシュで、beam要素を編集する
#
#   21/02/14    新規作成
#   22/03/14    国際化
#      03/22    国際化（文字をpython側でセット）
#   24/05/23    editBeamElement:新規追加
#      06/06    universalDialogsに修正
#      06/08    okDialog,errDialog:parentを追加
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk

import os, sys
import pyFistr
import GtkParts
import setBeamPinNodes
import fistr2abaqusForBeam
import beam611To641
import beam641To611
import universalDialogs

#import locale
localeDir = os.getenv("LOCALE_DIR")
#locale.bindtextdomain("easyistr", localeDir)

ls = "\n"

#--------------
#  dialogApp
#--------------
class dialogApp:

    def __init__(self, parent=None):
        self.builder = Gtk.Builder()
        #self.builder.set_translation_domain("easyistr")
        path = os.getenv("easyIstrPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "editBeamElementDDialog.glade")
        self.dialog = self.builder.get_object("dialog1")
        self.dialog.set_transient_for(parent)
        self.dialog.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        self.dialog.set_title(_("beam要素の編集"))
        self.setGtkObject()
        #変数の定義
        self.stat = ""
        self.meshHeaderData = []
        self.beamTypesSet = set(["611", "641"])
        self.allNgrps = []      #全てのNGRP
        self.pinNgrps = []      #pinNodesにセットされているNGRP
        self.unsetNgrps = []    #unset側のNGRP
        self.setNgrps = []      #set側のNGRP
        #初期化
        self.initialize()

    #  show
    def show(self):
        """ dialogの表示"""
        self.dialog.show()

    #  close
    def close(self, *args):
        self.dialog.destroy()

    #
    #  setGtkObject
    #----------------
    def setGtkObject(self):
        """ GtkObjectを定義する"""
        self.btn_to611 = self.builder.get_object("btn_to611")
        self.btn_to641 = self.builder.get_object("btn_to641")
        self.button_close = self.builder.get_object("button_close")
        self.button_sel = self.builder.get_object("button_sel")
        self.button_ret = self.builder.get_object("button_ret")
        self.button_apply = self.builder.get_object("button_apply")
        self.button_conv = self.builder.get_object("button_conv")
        self.tree_unsetNgrp = self.builder.get_object("tree_unsetNgrp")
        self.tree_setNgrp = self.builder.get_object("tree_setNgrp")
        self.check_2ndBeam = self.builder.get_object("check_2ndBeam")
        self.label_currElmType = self.builder.get_object("label_currElmType")
        #文字をセット
        self.builder.get_object("label_editConts").set_text(_("beam要素のみで構成されたメッシュ上で\n　　　beam要素を編集する"))
        self.builder.get_object("label_convElmType").set_text(_("beamの要素type変換"))
        self.builder.get_object("label_convTypeExp").set_text(_("要素type　611 <-> 641 への変換"))
        self.label_currElmType.set_text(_("現在の要素type:611"))
        self.btn_to611.get_child().set_text(_(" -> 611 に変換\n(6自由度2節点）"))
        self.btn_to641.get_child().set_text(_(" -> 641 に変換\n(3自由度4節点）"))
        self.builder.get_object("label_joinPin").set_text(_("pin接続の設定"))
        self.builder.get_object("label_joinPinExp").set_text(_("指定したNGRPの節点をpin接続に変更する。"))
        self.builder.get_object("label_NGRP").set_text(_("NGRP名"))
        self.builder.get_object("label_setNgrp").set_text(_("設定するNGRP"))
        self.builder.get_object("button_sel").get_child().set_text(_("選択>>"))
        self.builder.get_object("button_ret").get_child().set_text(_("<<戻す"))
        self.builder.get_object("label_addSpring").set_text(_("新しいnode、弱いバネ、(EQUATION)を追加する"))
        self.builder.get_object("button_apply").get_child().set_text(_("適用"))
        self.builder.get_object("label_convToAba").set_text(_("fistr2abaqus変換"))
        mess = _("calculix用のinpファイル「FistrModel.inp」に変換する。\n  非線形解析する時は変換してcalculixで解析する。\n  meshデータ, !BOUNDARY, !CLOAD, !SPRING を変換する。")
        self.builder.get_object("label_convAbaExp").set_text(mess)
        self.check_2ndBeam.get_child().set_text(_("2次要素に変換する"))
        self.builder.get_object("button_conv").get_child().set_text(_("変換"))

    #
    #  initialize
    #--------------
    def initialize(self):
        """ 初期化"""
        #lines = pyFistr.readFistrModelMsh()
        #meshHeaderData = pyFistr.getHeaderNumData(lines)
        meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
        #elmTypeをチェック
        self.elmTypesSet = set([])
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elmType = pyFistr.getValName(header, "TYPE")
                self.elmTypesSet.add(elmType)
        if len(self.elmTypesSet & self.beamTypesSet) == 0:
            self.stat = ""
            title = _("エラー")
            mess =_("beam要素が含まれていません")
            self.errDialog(title, mess, parent=self.dialog)
            self.close()
        #NGRPを取得
        #  全NGRP, pinNGRPを取得
        self.allNgrps = []
        self.pinNgrps = []
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!NODE" or words[0] == "!NGROUP":
                ngrp = pyFistr.getValName(header, "NGRP")
                if (ngrp != "" and ngrp[:len("dummy")] != "dummy"
                    and ngrp != "addedPinNodes" and ngrp != "pinNodes"):
                    self.allNgrps.append(ngrp)
            elif words[0] == "#pinNodes":
                ngrps = words[1:]
                self.pinNgrps += ngrps
        self.unsetNgrps = list(set(self.allNgrps) - set(self.pinNgrps)) 
        self.unsetNgrps.sort()
        self.setNgrps = self.pinNgrps[:]
        self.setNgrps.sort()
        #ボタンの非表示設定, 現在の要素type設定
        self.btn_to611.set_sensitive(False)
        self.btn_to641.set_sensitive(False)
        if len(self.pinNgrps) == 0:
            if "611" in self.elmTypesSet:
                self.btn_to641.set_sensitive(True)
                elmType = "611"
            if "641" in self.elmTypesSet:
                self.btn_to611.set_sensitive(True)
                elmType = "641"
        text = self.label_currElmType.get_text()
        text = text.split(":")[0] + ":" + elmType
        self.label_currElmType.set_text(text)
        #treeListを定義
        self.treeList_unset = GtkParts.treeList(self.tree_unsetNgrp)
        self.treeList_unset.create()
        self.treeList_unset.setItems(self.unsetNgrps)
        self.treeList_set = GtkParts.treeList(self.tree_setNgrp)
        self.treeList_set.create()
        self.treeList_set.setItems(self.setNgrps)
        #checkBoxの設定
        self.check_2ndBeam.set_active(True)

    #
    #  conv641To611
    #----------------
    def conv641To611(self):
        """ 要素type 641 → 611 に変換する"""
        beam641To611.convert641To611()
        title = _("meshType 変換")
        mess = _("meshType　641 -> 611 に変換しました。")
        print(mess)
        self.stat = "reload"
        self.okDialog(title, mess, parent=self.dialog)
        #ボタンの表示
        self.btn_to611.set_sensitive(False)
        self.btn_to641.set_sensitive(True)
        #elmTypesSetを更新
        self.elmTypesSet = self.elmTypesSet - set(["641"])
        self.elmTypesSet.add("611")
        #現在の要素typeを更新
        text = self.label_currElmType.get_text()
        text = text.split(":")[0] + ":611"
        self.label_currElmType.set_text(text)

    #
    #  conv611To641
    #---------------
    def conv611To641(self):
        """ 要素type 611 → 641 に変換する"""
        beam611To641.convertBeamType()
        title = _("meshType 変換")
        mess = _("meshType　611 -> 641 に変換しました。")
        print(mess)
        self.stat = "reload"
        self.okDialog(title, mess, parent=self.dialog)
        #ボタンの表示
        self.btn_to611.set_sensitive(True)
        self.btn_to641.set_sensitive(False)
        #elmTypesSetを更新
        self.elmTypesSet = self.elmTypesSet - set(["611"])
        self.elmTypesSet.add("641")
        #現在の要素typeを更新
        text = self.label_currElmType.get_text()
        text = text.split(":")[0] + ":641"
        self.label_currElmType.set_text(text)

    #
    #  moveToSet
    #------------
    def moveToSet(self):
        """ 選択itemをset側へ移動"""
        selNgrps = self.treeList_unset.getSelectedItems()
        if len(selNgrps) == 0:
            title = _("エラー")
            mess = _("NGRPが選択されていません")
            mess += _("  NGRPを選択してください")
            self.errDialog(title, mess, parent=self.dialog)
            return
        allNgrps = self.treeList_unset.getAllItems()
        unsetNgrps = list(set(allNgrps) - set(selNgrps))
        setNgrps = self.treeList_set.getAllItems() + selNgrps
        #表示
        unsetNgrps.sort()
        setNgrps.sort()
        self.treeList_unset.setItems(unsetNgrps)
        self.treeList_set.setItems(setNgrps)
        self.treeList_set.selectItems(selNgrps)

    #
    #  moveToUnset()
    #---------------
    def moveToUnset(self):
        """ 選択itemをunset側に移動"""
        selNgrps = self.treeList_set.getSelectedItems()
        if len(selNgrps) == 0:
            title = _("エラー")
            mess = _("NGRPが選択されていません")
            mess += _("  NGRPを選択してください")
            self.errDialog(title, mess, parent=self.dialog)
            return
        allNgrps = self.treeList_set.getAllItems()
        setNgrps = list(set(allNgrps) - set(selNgrps))
        unsetNgrps = self.treeList_unset.getAllItems() + selNgrps
        #表示
        setNgrps.sort()
        unsetNgrps.sort()
        self.treeList_unset.setItems(unsetNgrps)
        self.treeList_set.setItems(setNgrps)
        self.treeList_unset.selectItems(selNgrps)

    #
    #  applySetPinNodes()
    #--------------------
    def applySetPinNodes(self):
        """ pinNodesを削除して、設定し直す。"""
        if len(self.pinNgrps) != 0:
            #pinNodesを削除する
            setBeamPinNodes.deletePinNodes()
            self.stat = "reload"
            self.pinNgrps = []
        #pinNodesを設定
        ngrps = self.treeList_set.getAllItems()
        if len(ngrps) != 0:
            setBeamPinNodes.setPinNodes(ngrps)
            self.stat = "reload"
            self.pinNgrps = ngrps
        #ボタンの非表示設定
        self.btn_to611.set_sensitive(False)
        self.btn_to641.set_sensitive(False)
        if len(self.pinNgrps) == 0:
            if "611" in self.elmTypesSet:
                self.btn_to641.set_sensitive(True)
            if "641" in self.elmTypesSet:
                self.btn_to611.set_sensitive(True)
        #message
        if self.stat == "reload":
            if len(self.pinNgrps) == 0:
                title = _("pinNodesの設定")
                mess = _("全てのpinNodesを削除しました。")
                self.okDialog(title, mess, parent=self.dialog)
            else:
                title = _("pinNodesの設定")
                mess = _("以下のNGRPをpinNodesに設定しました。") + ls
                mess += "  " + ", ".join(self.pinNgrps)
                self.okDialog(title, mess, parent=self.dialog)

    #
    #  convertToAbaqus
    #------------------
    def convertToAbaqus(self):
        """ calculix用のinpファイルに変換する。"""
        if "611" in self.elmTypesSet:
            #要素typeが"611"の場合
            if self.check_2ndBeam.get_active() == True:
                meshFlag = "change2nd"
            else:
                meshFlag = "noChange"
            fileHeader = "FistrModel"
            fistrToAba = fistr2abaqusForBeam
            #fistrToAba.meshFlag = meshFlag
            fistrToAba.convertToInp(fileHeader, meshFlag)
            title = _("inpファイルに変換")
            mess = _("calculix用のinpファイルに変換しました。") + ls
            mess += _("  " + fileHeader + ".inp を作成しました。")
            self.okDialog(title, mess, parent=self.dialog)
        else:
            title = _("エラー")
            mess = _("inpファイルに変換できません。") + ls
            mess += _("  変換できる要素typeは、「611」のみです。") + ls
            mess += _("  要素タイプを確認してください。")
            self.errDialog(title, mess, parent=self.dialog)

    def okDialog(self, title, message, parent=None, funcOk=[]):
        dialog = universalDialogs.okDialog(title, message, parent, funcOk)
        dialog.show()

    def errDialog(self, title, message, parent=None, funcOk=[]):
        dialog = universalDialogs.errDialog(title, message, parent, funcOk)
        dialog.show()


#--------------------------------------
#  editBeamElement class
#--------------------------------------
class editBeamElement:
    """ beam要素の編集用dialog表示、編集する。
    
    Args:
        funcName (object):  編集後の処理関数名"""

    def __init__(self, funcName, funcArgs, parent=None):
        self.funcName = funcName
        self.funcArgs = funcArgs
        self.dialog = dialogApp(parent)
        self.dialog.btn_to611.connect("clicked", self.onConvTo611)
        self.dialog.btn_to641.connect("clicked", self.onConvTo641)
        self.dialog.button_close.connect("clicked", self.onClose)
        self.dialog.button_sel.connect("clicked", self.onSelect)
        self.dialog.button_ret.connect("clicked", self.onReturn)
        self.dialog.button_apply.connect("clicked", self.onApply)
        self.dialog.button_conv.connect("clicked", self.onConvToAbaqus)
    
    def show(self):
        self.dialog.dialog.show()

    def close(self):
        self.dialog.dialog.close()

    def onConvTo611(self, event):
        self.dialog.conv641To611()

    def onConvTo641(self, event):
        self.dialog.conv611To641()

    def onClose(self, event):
        stat = self.dialog.stat
        self.close()
        path, item = self.funcArgs
        self.funcName(stat, path, item)

    def onSelect(self, event):
        self.dialog.moveToSet()

    def onReturn(self, event):
        self.dialog.moveToUnset()

    def onApply(self, event):
        self.dialog.applySetPinNodes()

    def onConvToAbaqus(self, event):
        self.dialog.convertToAbaqus()
    


if __name__ == "__main__":
    import gettext
    gettext.install("app")
    _ = gettext.gettext

    dialog = dialogApp()
    stat = dialog.show()
