#!/usr/bin/python3
#  coding: utf-8
#
#       createVtkSourceFiles.py
#
#       meshのelement、surface、node group別にvtk形式で
#       fileを出力する。
#       出力先は、「vtkMeshData」folder内に出力。
#
#   20/06/15    新規作成
#      06/26    !EGROUPのvtkを追加
#      06/27    getVtkNodeData:offsetの値をi→i+1に修正
#               nodeGroupの最後のnodeが表示されない。
#               beamのnodeが表示されない。
#      06/30    path名表記の"/"をos.sepに修正
#      07/11    getSurfaceDataFromFaceNo:faceのnode順が2次要素の場合
#               間違っていたので、修正。
#      08/27    getVtkNodeData, getVtkSurfaceData, getVtkElementData:
#               座標値の出力精度を13.6e形式で出力する様に修正(精度を統一)
#               VTK表示で選択表示がうまく表示できない為、精度を統一               
#      09/02    createElementGrpVtkFiles:vtkFile作成がダブっていたので修正
#      09/04    changeSolidToFaceElement:追加
#               solidの場合、表面要素を抽出してvtkDataを作成する。
#               vtkFileの読み込みを早くする為。
#      09/8     allElement.vtuの作成を廃止。
#               要素Grpをoverlapして定義した場合、要素Grpを選択色に設定
#               すると、重ね書きした裏の要素色が見える事があるため、
#               各要素Grpのvtuファイル中に、overlapするしない事がわかる
#               CellDataを作成。
#               versionを設定
#      09/21    changeSolidToFaceElement:surElmDictにshellやbeamを追加。
#               9/8の修正で、shell、beamでエラー発生した為。
#      09/27    nodeGrpのデータをpointDataに追加。
#      09/28    surfaceGrp, nodeGrpの情報ををelementGrp内に挿入したので
#               これらのvtkFileの作成を止める。
#               vtkFileのversionを「1.2」に更新
#      09/30    clearSaveFolder:rmtreeで削除されるまで待つ事を追加
#               createNodeGrpVtkFiles:ngrpを!NODEからも取得する様に修正
#               dymmy節点grpが取得できなかった為。
#      10/01    changeSolidToFaceElement:elmNoとfaceNoの取得を追加
#               getVtkElementData:elmNoとfaceNoの保存を追加
#      10/09    createNodeGrpNameDict,getVtkElementData:NGRPがoverlapして
#               定義されていた場合、lapした部分が取得できていなかったので
#               修正。
#      10/10    addNodeGrp, addSurfacegrp:外からモジュール関数として使う為
#               に追加。
#      10/16    renameNodeGrp, renameSurfaceGrp, renameElementGrp,
#               deleteNodeGrp, deleteSurfaceGrp, deleteElementGrpを追加。
#               外からモジュール関数として使用する為。
#      10/21    addDeleteNodeGrp, addDeleteSurfaceGrp, addDeleteElementGrp
#               を追加。外からモジュール関数として使用する為。
#      10/25    edgeElementGrpのedgeVtkファイルを追加
#   21/01/06    createVtkFiles:モジュール関数として使える様に修正
#      01/14    createElementGrpVtkFiles:!ELEMENTでEGRPが未定義の場合
#               「--undefine--」名でvtkファイルを作成する様に修正
#      01/17    createElementGrpVtkFiles:バグ修正（vtkデータが正しくない）
#               辞書を作りながらvtk作成 → 辞書を作成後、vtk作成 に変更
#      04/05    shell,beamでもelmNoを書き込むように修正。（ver 1.33）
#      04/09    changeSolidToFaceElement:faceNoでsolid, shell, beamが区別
#               できる様に修正
#      05/03    createVtkFiles:バグ修正。surElmDictの初期化を追加。
#               続けて設定すると、前の設定が残っている為、修正。
#      07/11    changeSolidToFaceElement:solid, shell, beamの区別を
#               faceNoに含ませる。
#                 solid:faceNoが100未満
#                 shell:faceNoが100番台
#                 beam:faceNoが200番台
#               shellのSGRPの定義を追加。faceNoは「1」に固定。
#               versionを1.35に更新
#      04/01    createNodeGrpData,createSurfaceGrpData:高速化の為、ネック
#               部分を修正。（list.index()をdictに変更。）
#   22/04/05    delNodeGrp,delSurfaceGrp:grpからnodeを削除する関数を追加
#      04/08    createVtkFiles:並列処理（2並列）で処理できる様に修正
#      04/12    log表示修正（import logFileCreaterを追加）
#      04/18    createElementGrpVtkFiles:並列処理時のglobal変数をlocalとして
#               引数に追加。（windowsのpython3.8対応）
#   24/07/07    openにencoding="utf-8"を追加
#      09/24    getVtkElementData:elementGrp「ALL」、nodeGrp「ALL」のdataを追加
#               versionを1.35→1.40に変更。
#      12/22    createElementGrpVtkFiles:surfaceのvtk作成を追加（復活）
#      12/30    createElementGrpVtkFiles:surfaceのvtk作成を削除
#   25/02/21    getElementDict,changeSolidToFaceElement:!ENDが無い時、elmData=[]
#               になるので、この時の処理を追加。
#      03/20    createElementGrpVtkFiles,createElementVtkFile,saveVersion:
#               vtkFileの作成は、外から中断される事があり、中断されると
#               file書き込み時にエラーが発生（windowsのみ）するので、例外処理追加。
#      06/07    changeSolidToFaceElement:表面要素取得をSGRPを使うように修正（高速化）
#      06/08    changeSolidToFaceElement:バグ修正。複数の要素groupがある時、取得できない。
#      06/12    vtk作成時、binaryのmeshHeaderDataを使うように修正。
#               大規模メッシュの場合、meshHeaderData取得に時間がかかるため。
#      06/17    vtkFileのdataFormatをがasciiのみだったが、binaryを追加。
#               読み込み速度向上の為。
#               ascii形式、binary形式ともfile内の形式に従って読み込む様に修正。
#      06/20    header_typeを32bitから64bitに修正。
#      06/25    setNumberOfComponents:nCells=0の時、エラー発生の為、修正。    
#               makePointsDataArrayBlock:nPoints=0の時、エラー発生の為、修正
#      06/26    writeSurfaceGrpOtherFiles,writeNodeGrpOtherFiles:dataLineがvtuFormatに
#               従っていなかった為、修正。（バグ）
#      07/27    isCloseFace:otherSを使ってfaceDict作成した結果、閉じていないfaceの場合、
#               全要素をチェックして再作成する様に修正。新規追加。
#      08/20    createElementGrpVtkFiles:vtkFile作成時並列処理を廃止。
#               windowsで極端に遅くなるため。
#

import os
import sys
import shutil
import time
import glob
import multiprocessing as mp
from tokenize import Single
import pickle
import numpy as np
import base64

import pyFistr
import convertMesh as cm


version = "1.40"                    #vtkFileのformat確認用
currDir = ""
SaveFolder = "vtkMeshData"          #vtkFileを保存するfolder名

surElmDict = {}
nodeGrpNameDict = {}

vtkEdgesDict = {
    "3" :[[0,1]],                #111, beam
    "5" :[[0,1],[1,2],[2,0]],    #231, 三角形
    "9" :[[0,1],[1,2],[2,3],[3,0]],  #241, 四角形
    "10":[[0,1],[1,2],[2,0],[0,3],[1,3],[2,3]],     #341, tetra  
    "12":[[0,1],[1,2],[2,3],[3,0],[4,5],[5,6],[6,7],[7,4],[0,4],[1,5],[2,6],[3,7]],  #361, hexa
    "13":[[0,1],[1,2],[2,0],[3,4],[4,5],[5,3],[0,3],[1,4],[2,5]],                   #351, prism
    "21":[[0,2,1]],             #112, beam2次
    "22":[[0,3,1],[1,4,2],[2,5,0]],     #232, 三角形2次
    "23":[[0,4,1],[1,5,2],[2,6,3],[3,7,0]], #242, 四角形1次
    "24":[[0,4,1],[1,5,2],[2,6,0],[0,7,3],[1,8,3],[2,9,3]], #342, tetra2次
    "25":[[0,8,1],[1,9,2],[2,10,3],[3,11,0],[4,12,5],[5,13,6],[6,14,7],[7,15,4],[0,16,4],[1,17,5],[2,18,6],[3,19,7]],    #362, hexa2次
    "26":[[0,6,1],[1,7,2],[2,8,0],[3,9,4],[4,10,5],[5,11,3],[0,12,3],[1,13,4],[2,14,5]],        #352, prism2次
    }

vtuToNumpyTypeDict = {
    #vtuType       numpyType        nBytes
    "Float32": [np.dtype(np.float32), 4],
    "Float64": [np.dtype(np.float64), 8],
    "Int8":    [np.dtype(np.int8), 1],
    "Int16":   [np.dtype(np.int16), 2],
    "Int32":   [np.dtype(np.int32), 4],
    "Int64":   [np.dtype(np.int64), 8],
    "UInt8":   [np.dtype(np.uint8), 1],
    "UInt16":  [np.dtype(np.uint16), 2],
    "UInt32":  [np.dtype(np.uint32), 4],
    "UInt64":  [np.dtype(np.uint64), 8]
    }

vtuToPyTypeDict = {
    #vtuType   pyType
    "Float32": float,
    "Float64": float,
    "Int8":    int,
    "Int16":   int,
    "Int32":   int,
    "Int64":   int,
    "UInt8":   int,
    "UInt16":  int,
    "UInt32":  int,
    "UInt64":  int
    }

#
#  getVtuFormat
#------------------
def getVtuFormat(lines):
    """ 読み込んだvtuのformat（ascii or binary）を調べて返す"""
    vtuFormat = "ascii"
    for line in lines:
        if line.find("<DataArray") >= 0:
            _vtuName, vtuFormat = getVtuTypeFormatInLine(line)
            break
    return vtuFormat

#  getVtuTypeFormatInLine
def getVtuTypeFormatInLine(line):
    """ 行内からvtuのtypeとformatを取得して返す"""
    words = line.split()
    flag = 0
    vtuType = ""; vtuFormat = ""
    for word in words:
        if word[:len("type=")] == "type=":
            vtuType = word.split('"')[1]    #「"」で囲まれた文字を取得
            flag += 1
        elif word[:len("format=")] == "format=":
            vtuFormat = word.split('"')[1]
            flag += 1
        if flag >= 2:
            break
    return vtuType, vtuFormat

#
#  getBinaryData
#-----------------
def getBinaryData(vtuFmt, packVals):
    """ binaryDataをbase64を使って作成する"""
    #byte数をセット
    npType, n = vtuToNumpyTypeDict[vtuFmt]
    nBytes = len(packVals) * n
    nArr = np.array([nBytes], dtype="uint64")
    buffer = nArr.tobytes()
    #packValsをセット
    arr = np.array(packVals, dtype=npType)
    buffer += arr.tobytes()
    #bufferをencode
    data = base64.b64encode(buffer)
    data = data.decode("utf-8")
    return data

#
#  getValsInBinaryData
#----------------------
def getValsInBinaryData(valsData, vtuType):
    """ binaryDataから値を取り出す"""
    #bufferをdecode
    buffer = base64.b64decode(valsData)
    #byte数を取得
    nBytes = np.frombuffer(buffer[:8], dtype="uint64")
    #値を取得
    npType, n = vtuToNumpyTypeDict[vtuType]
    vals = np.frombuffer(buffer[8:], dtype=npType)
    return vals

#
#  getDataFromDataArrayLine
#---------------------------
def getDataFromDataArrayLine(dataLine, vtuFormat, vtuType):
    """ vtuDataLineから値を取得する。（format typeに応じて）"""
    line = dataLine.strip()
    if vtuFormat == "ascii":
        pyType = vtuToPyTypeDict[vtuType]
        data = list(map(pyType, line.split()))
    elif vtuFormat == "binary":
        data = getValsInBinaryData(line, vtuType)
        data = list(data)
    else:
        data = []
    return data

#
#  setDataToDataArrayLine
#-------------------------
def setDataToDataArrayLine(vals, vtuFormat, vtuType):
    """ vtuDataLineを作成する（format typeに応じて）"""
    if vtuFormat == "ascii":
        vals = list(map(str, vals))
        line = " ".join(vals) + "\n"
    elif vtuFormat == "binary":
        dataLine = getBinaryData(vtuType, vals)
        line = dataLine + "\n"
    else:
        line = "\n"
    return line

#
#  readFileLines
#----------------
def readFileLines(fileName):
    f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    return lines

#
#  writeFileLines
#-----------------
def writeFileLines(fileName, lines):
    f=open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()
    return

#
#  getMeshHeaderData
#---------------------
def getMeshHeaderData():
    """ meshDataを取得"""
    global currDir
    meshHeaderData = pyFistr.getHeaderNumDataMsh(currDir)
    return meshHeaderData

#
#  getNodesDict
#---------------
def getNodesDict(meshHeaderData):
    """ 節点Noをkeyとして、座標を求めるnodesDictを作成する。"""
    nodesDict = {}
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            data = headerData[1]
            for nodeData in data:
                nodeNo = nodeData[0]
                loc = nodeData[1:]
                nodesDict[nodeNo] = loc
    return nodesDict

#
#  getElementDict
#------------------
def getElementDict(meshHeaderData):
    """ 要素Noをkeyとして、節点列・要素typeが取得できる
    elementDictを作成する。"""
    elementDict = {}
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            elType = pyFistr.getValName(header, "TYPE")
            data = headerData[1]
            for elmData in data:
                if len(elmData) > 0:
                    elNo = elmData[0]
                    nodes = elmData[1:]
                    elementDict[elNo] = [nodes, elType]
    return elementDict

#
#  clearSaveFolder
#------------------
def clearSaveFolder():
    """ folder内をクリアする"""
    #folderDir = currDir + "/" + SaveFolder
    folderDir = currDir + os.sep + SaveFolder
    if os.path.exists(folderDir) == True:
        shutil.rmtree(folderDir)
        #削除されるまで待つ
        while os.path.exists(folderDir) == True:
            time.sleep(0.1)
        os.mkdir(folderDir)
    else:
        os.mkdir(folderDir)

#
#  createNodeGrpVtkFiles
#-----------------------
def createNodeGrpVtkFiles(meshHeaderData):
    """ 節点group名をmeshHeaderDataから取得。
    vtkFile作成の為の辞書nodeGrpNameDictを作成する。"""
    #nodeGroupを取得
    nodeGrps = []
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            name = pyFistr.getValName(header, "NGRP")
            data = headerData[1]
            nodeGrps.append([name, data])
        elif words[0] == "!NODE":
            name = pyFistr.getValName(header, "NGRP")
            if name != "":
                data = headerData[1]
                data = list(map(lambda x: x[0], data))
                nodeGrps.append([name, data])
    #grp名をチェック。
    nodeGrps = checkSameGroupName(nodeGrps)
    #vtkを作成
    for nodeGrp in nodeGrps:
        #nodeGrpNameDictを作成
        createNodeGrpNameDict(nodeGrp)
    return nodeGrps

#
#  createNodeGrpNameDict
def createNodeGrpNameDict(nodeGrp):
    """ nodeGrpNameDictを作成"""
    global nodeGrpNameDict
    ngrp = nodeGrp[0]
    data = nodeGrp[1]
    for nodeNo in data:
        if nodeNo in nodeGrpNameDict.keys():
            nodeGrpNameDict[nodeNo].append(ngrp)
        else:    
            nodeGrpNameDict[nodeNo] = [ngrp]

#
#  createSurfaceGrpVtkFiles
#----------------------------
def createSurfaceGrpVtkFiles(meshHeaderData, elementDict):
    """ 表面group名毎にvtkFileを作成する。"""
    #surfaceGroupを作成
    surfaceGrps = []
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SGROUP":
            name = pyFistr.getValName(header, "SGRP")
            data = headerData[1]
            surfaceGrps.append([name, data])
    #grp名をチェック。
    surfaceGrps = checkSameGroupName(surfaceGrps)
    #vtkを作成
    for surfaceGrp in surfaceGrps:
        #surface毎にvtkを作成
        name = surfaceGrp[0]
        data = surfaceGrp[1]
        surData = getSurfaceDataFromFaceNo(name, data, elementDict)
    return surfaceGrps

#
#  remakeFaceNodesOrder
def remakeFaceNodesOrder(faceNodes):
    """ 2次要素の場合、連番のnode順をfistrの2次要素の順の置き換える"""
    faceOrderDict = {
        3: [[1,2,3], cm.order_3],
        4: [[1,2,3,4], cm.order_4],
        6: [[1,3,5,4,6,2], cm.order_6],
        8: [[1,3,5,7,2,4,6,8], cm.order_8]
        }
    faceOrder = faceOrderDict[len(faceNodes)][0]
    func = faceOrderDict[len(faceNodes)][1]
    newFaceNodes = func(faceNodes, faceOrder)
    return newFaceNodes

#
#  getSurfaceDataFromFaceNo
def getSurfaceDataFromFaceNo(name, data, elementDict):
    """ 要素のfaceNoから節点No列を取得する。surElmDictを作成する。"""
    global surElmDict
    faceTypeDict = {
      #node数 要素type
        3:  "231",
        4:  "241",
        6:  "232",
        8:  "242"
        }

    surData = []
    for i in range(0, len(data), 2):
        (elNo, faceNo) = data[i:i+2]
        elCont = elementDict[elNo]
        elNodes = elCont[0]
        elType = elCont[1]
        elmCont = cm.elmContsDict[elType][0]
        if elmCont == "solid":
            faceNodes0 = cm.fistr_elFaceNode(elType, elNodes, faceNo)
            #faceNodesを正しく修正する（2次要素の場合、要修正）
            faceNodes = remakeFaceNodesOrder(faceNodes0)
            #surElmDictを作成
            keyNodes = faceNodes[:]
            keyNodes.sort()
            key = tuple(keyNodes)
            if key in surElmDict.keys():
                surElmDict[key]["SGRP"][name] = 1
            else:
                grpDict = {"EGRP":{}, "SGRP":{}}
                grpDict["SGRP"][name] = 1
                surElmDict[key] = grpDict
            #vtkに変換
            faceType = faceTypeDict[len(faceNodes)]
            (vtkName, vtkNodes) = cm.fistr2vtk_el(faceType, faceNodes)
            surData.append([vtkNodes, vtkName])
        elif elmCont == "shell":
            faceNodes = elNodes
            keyNodes = faceNodes[:]
            keyNodes.sort()
            key = tuple(keyNodes)
            if key in surElmDict.keys():
                surElmDict[key]["SGRP"][name] = 1
            else:
                grpDict = {"EGRP":{}, "SGRP":{}}
                grpDict["SGRP"][name] = 1
                surElmDict[key] = grpDict
            #vtkに変換
            faceType = elType
            (vtkName, vtkNodes) = cm.fistr2vtk_el(faceType, faceNodes)
            surData.append([vtkNodes, vtkName])
    return surData

#
#  checkSameGroupName
#---------------------
def checkSameGroupName(Grps):
    """ 同じgroup名があるかチェック。あれば、まとめる。"""
    newGrps = []
    for grp in Grps:
        name = grp[0]
        data = grp[1]
        flag = 0
        for newGrp in newGrps:
            if newGrp[0] == name:
                flag = 1
                #同じgrpNameあり
                newGrp[1] += data
                break
        if flag == 0:
            newGrps.append(grp)
    return newGrps

#
#  createElementGrpVtkFiles
#--------------------------
def createElementGrpVtkFiles(meshHeaderData, nodesDict, elementDict,
                            surfaceGrps, nodeGrps, nCpu, vtkFormat):
    """ 要素group毎にvtkFileを作成する"""
    #elementGroupを作成
    elementGrps = []
    #!ELEMENTを作成
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            name = pyFistr.getValName(header, "EGRP")
            if name == "":
                name = "--undefine--"
            data = headerData[1]
            elementGrps.append([name, data])
    #同じgrpNameをチェック、あればまとめる
    elementGrps = checkSameGroupName(elementGrps)
    #!EGROUPを取得
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!EGROUP":
            name = pyFistr.getValName(header, "EGRP")
            data = []
            for elNo in headerData[1]:
                nodes = elementDict[elNo][0]
                elm = [elNo] + nodes
                data.append(elm)
            elementGrps.append([name, data])

    #solid要素を表面要素に変換、surElmDictを作成
    #  予め、surElmDictを完成させる
    print("getting surface elements of solid...")
    faceElmData = []
    for elementGrp in elementGrps:
        #element毎にvtkを作成
        name = elementGrp[0]
        data = elementGrp[1]
        #solidをsurfaceに変換, surElmDictを作成
        data, elementDict, elmFaceNos = changeSolidToFaceElement(name, data, elementDict, surfaceGrps)
        #変換データを保存
        faceElmData.append([data, elementDict, elmFaceNos])    
    #surElmDictを使ってvtkDataを作成
    print("creating vtk files...")
    for i in range(len(elementGrps)):
        elementGrp = elementGrps[i]
        #element毎にvtkを作成
        name = elementGrp[0]
        #solidのsurfaceDataを取得
        [data, elementDict, elmFaceNos] = faceElmData[i]
        #vtkDataを作成
        elmData = getElementDataForVtk(data, elementDict)
        elmNodeDict, allNodes = getElmNodeDictForVtk(elmData)
        grpDicts = [surElmDict, nodeGrpNameDict]
        #vtkFile作成
        #  elementを作成
        vtkLines = getVtkElementData(name, elmNodeDict, allNodes, elmData, nodesDict, elementGrps, surfaceGrps, nodeGrps, elmFaceNos, grpDicts, vtkFormat)
        fileName = currDir + os.sep + SaveFolder + os.sep + "elementGrp_" + name + ".vtu"
        f = open(fileName, "w", encoding="utf-8"); f.writelines(vtkLines); f.close()
        #  edgeを作成
        vtkLines = getVtkEdgeData(elmNodeDict, allNodes, elmData, nodesDict, vtkFormat)
        fileName = currDir + os.sep + SaveFolder + os.sep + "edgeElementGrp_" + name + ".vtu"
        f = open(fileName, "w", encoding="utf-8"); f.writelines(vtkLines); f.close()

#
#  createElementVtkFile
#-----------------------
def createElementVtkFile(name, elmNodeDict, allNodes, elmData,
                        nodesDict, elementGrps, surfaceGrps,
                        nodeGrps, elmFaceNos, currDir, grpDicts,
                        vtkFormat, q):
    """ elementのvtkファイル作成（subprocessで実行）"""
    #vtkLinesを作成
    vtkLines = getVtkElementData(name, elmNodeDict, allNodes,
        elmData, nodesDict, elementGrps, surfaceGrps, nodeGrps,
        elmFaceNos, grpDicts, vtkFormat)
    #ファイルを保存
    fileName = currDir + os.sep + SaveFolder + os.sep + "elementGrp_" + name + ".vtu"
    try:
        ff = open(fileName, "w", encoding="utf-8"); ff.writelines(vtkLines); ff.close()
    except:
        print("stopped to save egrp vtk data.")
        pass
    q.put(0)

#
#  getVtkEdgeData
def getVtkEdgeData(elmNodeDict, allNodes, elmData, nodesDict, vtkFormat):
    """ edgeのVtkを作成する。"""
    #edgeElmDictを作成
    #  edgeElmDict = {edge:vtkElmNos}
    edgeData = []
    for nodes, elmType in elmData:
        vtkEdgeNos = vtkEdgesDict[elmType]
        edges = []
        for vtkEdge in vtkEdgeNos:
            edge = list(map(lambda x: nodes[x], vtkEdge))
            edge.sort()
            edges.append(tuple(edge))
        edgeData.append(edges)
    edgeElmDict = {}
    for elmNo in range(len(edgeData)):
        edges = edgeData[elmNo]
        for edge in edges:
            if edge in edgeElmDict.keys():
                edgeElmDict[edge].append(elmNo)
            else:
                edgeElmDict[edge] = [elmNo]

    edgeKeys = list(edgeElmDict.keys())
    edgeKeys.sort()

    lines  = ['<?xml version="1.0"?>\n']
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0" byte_order="LittleEndian" header_type="UInt64">\n']
    lines += ['<UnstructuredGrid>\n']
    nPoints = len(allNodes)
    nCells = len(edgeKeys)
    lines += ['<Piece NumberOfPoints="' + str(nPoints) + '" NumberOfCells="' + str(nCells) + '">\n']
    lines += ['<Points>\n']
    #points(座標データ)
    line = '<DataArray type="Float32" NumberOfComponents="3" format="ascii" >\n'
    vals = []
    for nodeNo in allNodes:
        loc = nodesDict[nodeNo]
        vals += loc
    blockLines = makePointsDataArrayBlock(vtkFormat, line, vals, nPoints)
    lines += blockLines
    lines += ['</Points>\n']
    lines += ['<Cells>\n']
    #connectivity
    line = '<DataArray type="Int32" Name="connectivity" format="ascii" >\n'
    nodes = []
    for elmNodes in edgeKeys:
        newNodes = list(map(lambda x: elmNodeDict[x], elmNodes))
        nodes.append(newNodes)
    blockLines = makeConnectivityDataArray(vtkFormat, line, nodes, nCells)
    lines += blockLines
    #offsets
    line = '<DataArray type="Int32" Name="offsets" format="ascii" >\n'
    elms = []
    n = 0
    for elmNodes in edgeKeys:
        n += len(elmNodes)
        elms.append(n)
    blockLines = makeOffsetsTypesDataArray(vtkFormat, line, elms, nCells, name="offsets")
    lines += blockLines
    #types
    line = '<DataArray type="UInt8" Name="types" format="ascii" >\n'
    vtkNames = []
    for elmNodes in edgeKeys:
        if len(elmNodes) == 2:
            vtkName = 3
        else:
            vtkName = 21
        vtkNames.append(vtkName)
    blockLines = makeOffsetsTypesDataArray(vtkFormat, line, vtkNames, nCells, name="types")
    lines += blockLines
    lines += ['</Cells>\n']
    lines += ['<CellData>\n']
    #pairElmNo
    line = '<DataArray type="Int32" Name="pairElmNo" NumberOfComponents="2" format="ascii" >\n'
    values = []
    for elmNodes in edgeKeys:
        elmNos = edgeElmDict[elmNodes]
        if len(elmNos) == 1:
            vals = [str(elmNos[0]), "-1"]
        else:
            vals = [str(elmNos[0]), str(elmNos[1])]
        values += vals
    blockLines = makeDataArrayBlock(vtkFormat, line, values, nCells)
    lines += blockLines
    lines += ['</CellData>\n']
    lines += ['</Piece>\n']
    lines += ['</UnstructuredGrid>\n']
    lines += ['</VTKFile>\n']
    return lines

#
#  getElmNodeDictForVtk
def getElmNodeDictForVtk(elmData):
    """ elmDataのnodeNoをvtkのNodeNoに変換する辞書を作成する"""
    #nodeNo取得
    allNodes = set([])
    for nodes, _name in elmData:
        allNodes.update(set(nodes))
    allNodes = list(allNodes)
    allNodes.sort()
    #辞書作成
    elmNodeDict = {}
    for i in range(len(allNodes)):
        nodeNo = allNodes[i]
        elmNodeDict[nodeNo] = i     #newNodeNoを設定
    return elmNodeDict, allNodes

#
#  changeSolidToFaceElement
def changeSolidToFaceElement(grpName, elmData, elementDict, surfaceGrps):
    """ solid要素をface要素に変換する。shellやbeamはそのまま。
    全要素（solid, shell, beam)について、surElmDictを作成する。"""
    global surElmDict
    solidElmDict = {
        "231":"face",    #三角形1次
        "232":"face",    #三角形2次
        "241":"face",    #四角形1次
        "242":"face",    #四角形2次
        "341":"solid",   #四面体1次
        "342":"solid",   #四面体2次
        "361":"solid",   #六面体1次
        "362":"solid",   #六面体2次
        "351":"solid",   #五面体1次
        "352":"solid",   #五面体2次
        "611":"beam",    #beam要素
        "641":"beam",    #beam要素（3自由度）
        "731":"shell",   #シェル三角形1次
        "732":"shell",   #シェル三角形2次
        "741":"shell",   #シェル四角形1次
        "761":"shell",   #シェル三角形1次（3自由度）
        "781":"shell"    #シェル四角形1次（3自由度）
        }
    nodesToFaceDict = {
        #node数  type    node順
        3:      ["231", [1,2,3]],
        4:      ["241", [1,2,3,4]],
        6:      ["232", [1,3,5,4,6,2]],
        8:      ["242", [1,3,5,7,2,4,6,8]]
        }

    def getFaceDictFromAllElms():
        """ 全要素からfaceDictを求める。
        時間がかかるが、正しいfaceDictが求まる。"""
        faceDict = {}
        for n in range(len(solidElmData)):
            elm = solidElmData[n]
            elmNo = elm[0]
            nodes = elm[1:]
            elmType = elmTypes[n]
            elmFaceNodes = cm.faceNodeOfElementDict[elmType]
            for i in range(1, len(elmFaceNodes)):
                order = elmFaceNodes[i]
                nodeNos = list(map(lambda x: nodes[x-1], order))
                nodeNos.sort()
                key = tuple(nodeNos)
                if key in faceDict.keys():
                    #辞書からkeyを削除(既に定義済みのface)
                    _dummy = faceDict.pop(key)
                else:
                    #辞書に追加
                    faceDict[key] = [elmNo, i]
        return faceDict

    def getFaceDictFromSgrp():
        """ SGRPからfaceDictを取得"""
        faceDict = {}
        #  surfaceGrpから外表面取得
        for surName, surData in surfaceGrps:
            for i in range(0, len(surData), 2):
                elmNo = surData[i]
                faceNo = surData[i+1]
                elNodes, elmType = elementDict[elmNo]
                #対象のelmNo？
                if elmNo in elmNoSets:
                    nodeNos = cm.fistr_elFaceNode(elmType, elNodes, faceNo) 
                    nodeNos.sort()
                    key = tuple(nodeNos)
                    faceDict[key] = [elmNo, faceNo]
        return faceDict

    def isCloseFace(faceDict):
        """ 取得したfaceDictが閉じているか確認"""
        edgeDict = {}
        for key in faceDict.keys():
            elmNo, faceNo = faceDict[key]
            elNodes, elmType = elementDict[elmNo]
            nodes = cm.fistr_elFaceNode(elmType, elNodes, faceNo)
            if len(nodes) == 6:
                nodes = [nodes[0], nodes[2], nodes[4]]
            elif len(nodes) == 8:
                nodes = [nodes[0], nodes[2], nodes[4], nodes[6]]
            for i in range(len(nodes)):
                if i == len(nodes) - 1:
                    edge = [nodes[0], nodes[-1]]
                else:
                    edge = [nodes[i], nodes[i+1]]
                edge.sort()
                edge = tuple(edge)
                if edge in edgeDict.keys():
                    dummy = edgeDict.pop(edge)
                else:
                    edgeDict[edge] = 1
        if len(edgeDict) == 0:
            return True
        return False

    #solidElmを取得する
    solidElmData = []
    noSolidElmData = []
    elmTypes = []
    for elm in elmData:
        if len(elm) > 0:
            elmNo = elm[0]
            nodes = elm[1:]
            elmType = elementDict[elmNo][1]
            if solidElmDict[elmType] == "solid":
                solidElmData.append(elm)
            else:
                noSolidElmData.append(elm)
            elmTypes.append(elmType)

    # #solidElmをfaceElmに変換
    # faceDict = {}
    # for n in range(len(solidElmData)):
    #     elm = solidElmData[n]
    #     elmNo = elm[0]
    #     nodes = elm[1:]
    #     elmType = elmTypes[n]
    #     elmFaceNodes = cm.faceNodeOfElementDict[elmType]
    #     for i in range(1, len(elmFaceNodes)):
    #         order = elmFaceNodes[i]
    #         nodeNos = list(map(lambda x: nodes[x-1], order))
    #         nodeNos.sort()
    #         key = tuple(nodeNos)
    #         if key in faceDict.keys():
    #             #辞書からkeyを削除(既に定義済みのface)
    #             _dummy = faceDict.pop(key)
    #         else:
    #             #辞書に追加
    #             faceDict[key] = [elmNo, i]

    #solid要素？
    faceDict = {}
    if len(solidElmData) > 0:
        #対象のelmNoのみ取得
        elmNoSets = set(map(lambda x: x[0], elmData))
        #対象の要素がotherSに含まれているか？
        flag = 0
        for name, surfaceGrp in surfaceGrps:
            if name[:len("otherS")] == "otherS":
                if surfaceGrp[0] in elmNoSets:
                    flag = 1
                    break
        if flag == 1:
            #SGRPから取得
            faceDict = getFaceDictFromSgrp()
            print("checking gotten surface elements...")
            if isCloseFace(faceDict) == False:
               print("Warning: coud not get vtu file at SGRP 'otherS'.\n  retry to get at all elements.")
               faceDict = getFaceDictFromAllElms()
        else:
            #全要素を検索
            faceDict = getFaceDictFromAllElms()
        #取得できた？
        if len(faceDict) == 0:
            #取得できていない場合は、全要素から再取得
            faceDict = getFaceDictFromAllElms()

    #表面節点を取得
    elmFaceNos = []
    surfaceData = []
    elementNo = max(elementDict.keys()) + 1
    for key in faceDict.keys():
        if key in surElmDict.keys():
            surElmDict[key]["EGRP"][grpName] = 1
        else:
            grpDict = {"EGRP":{}, "SGRP":{}}
            grpDict["EGRP"][grpName] = 1
            surElmDict[key] = grpDict
        elmNo = faceDict[key][0]
        elNodes = elementDict[elmNo][0]
        elmType = elementDict[elmNo][1]
        faceNo = faceDict[key][1]
        elmFaceNos.append([elmNo, faceNo])
        order = cm.faceNodeOfElementDict[elmType][faceNo]
        nodes = list(map(lambda x: elNodes[x-1], order))
        faceType, surOrder = nodesToFaceDict[len(order)]
        nodes = list(map(lambda x: nodes[x-1], surOrder))
        surfaceData.append([elementNo] + nodes)
        elementDict[elementNo] = [nodes, faceType]
        elementNo += 1
    #solid以外のelementのsurElmDictを作成
    for elm in noSolidElmData:
        elmNo = elm[0]
        nodes = elm[1:]
        #elmFaceNos.append([0, 0])
        #elmFaceNos.append([elmNo, 0])
        elmType = elementDict[elmNo][1]
        if solidElmDict[elmType] == "shell":
            #faceNo = 0      #shell
            faceNo = 101
        else:
            #faceNo = 99     #beam
            faceNo = 220
        elmFaceNos.append([elmNo, faceNo])
        nodes.sort()
        key = tuple(nodes)
        if key in surElmDict.keys():
            surElmDict[key]["EGRP"][grpName] = 1
        else:
            grpDict = {"EGRP":{}, "SGRP":{}}
            grpDict["EGRP"][grpName] = 1
            surElmDict[key] = grpDict
    elmData = surfaceData + noSolidElmData
    return elmData, elementDict, elmFaceNos

#
#  getElementDataForVtk
def getElementDataForVtk(data, elementDict):
    """ fistr形式の要素データをvtk形式の要素データに変換する。"""
    elmData = []
    for i in range(len(data)):
        elNo = data[i][0]
        nodes = data[i][1:]
        elCont = elementDict[elNo]
        elType = elCont[1]
        (vtkName, vtkNodes) = cm.fistr2vtk_el(elType, nodes)
        elmData.append([vtkNodes, vtkName])
    return elmData

#  changeFormatToLine
def changeFormatToLine(vtkFormat, line):
    words = line.split()
    for i in range(len(words)):
        if words[i][:len("format=")] == "format=":
            if words[i].find("ascii") >= 0:
                words[i] = words[i].replace('ascii', vtkFormat)
            else:
                words[i] = words[i].replace('binary', vtkFormat)
            line = " ".join(words) + "\n"
            break
    return line

#  setNumberOfComponents
def setNumberOfComponents(line, vals, nCells):
    """ nCompoを設定"""
    if nCells == 0:
        return line
    words = line.split()
    for i in range(len(words)):
        if words[i][:len("NumberOfComponents=")] == "NumberOfComponents=":
            nCompo = str(len(vals) // nCells)
            words[i] = 'NumberOfComponents="' + nCompo + '"'
            line = " ".join(words) + "\n"
            break
    return line

#
#  setNewName
def setNewName(line, name):
    """ Nameを書き換える"""
    words = line.split()
    for i in range(len(words)):
        if words[i][:len("Name=")] == "Name=":
            words[i] = 'Name="' + name + '"'
            line = " ".join(words) + "\n"
            break
    return line

#
#  makePointsDataArrayBlock
def makePointsDataArrayBlock(vtkFormat, line, vals, nPoints):
    """ points(座標)のblockLinesを作成する"""
    vtuType, orgFormat = getVtuTypeFormatInLine(line)
    #formatを修正
    if vtkFormat != orgFormat:
        line = changeFormatToLine(vtkFormat, line)
    #nCompを設定
    line  = setNumberOfComponents(line, vals, nPoints)
    blockLines = [line]
    #DataArrayを作成
    if nPoints > 0:
        if vtkFormat == "ascii":
            nComp = len(vals) // nPoints
            for i in range(0, len(vals), nComp):
                loc = vals[i:i+nComp]
                #座標は、13.6eで出力する
                dataLine = " ".join(map(lambda x: "%13.6e" % x, loc)) + "\n"
                blockLines += [dataLine]
        else:
            dataLine = setDataToDataArrayLine(vals, vtkFormat, vtuType)
            blockLines += [dataLine]
    blockLines += ["</DataArray>\n"]
    return blockLines

#
#  makeConnectivityDataArray
def makeConnectivityDataArray(vtkFormat, line, nodes, nCells):
    """ connectivityを作成する"""
    vtuType, orgFormat = getVtuTypeFormatInLine(line)
    #formatを修正
    if vtkFormat != orgFormat:
        line = changeFormatToLine(vtkFormat, line)
    blockLines = [line]
    #DataArrayを作成
    if vtkFormat == "ascii":
        for nodeNos in nodes:
            dataLine = " ".join(list(map(str, nodeNos))) + "\n"
            blockLines += [dataLine]
    else:
        vals = []
        for nodeNos in nodes:
            vals += nodeNos
        dataLine = setDataToDataArrayLine(vals, vtkFormat, vtuType)
        blockLines += [dataLine]
    blockLines += ["</DataArray>\n"]
    return blockLines

#
#  makeOffsetsTypesDataArray
def makeOffsetsTypesDataArray(vtkFormat, line, elms, nCells, name="offsets"):
    """ offsets or typesを作成する"""
    vtuType, orgFormat = getVtuTypeFormatInLine(line)
    #formatを修正
    if vtkFormat != orgFormat:
        line = changeFormatToLine(vtkFormat, line)
    #Nameを設定
    if name != "":
        line = setNewName(line, name)
    blockLines = [line]
    #DataArrayを作成
    if vtkFormat == "ascii":
        for i in range(0, len(elms), 10):
            elmNos = elms[i:i+10]
            dataLine = " ".join(list(map(str, elmNos))) + "\n"
            blockLines += [dataLine]
    else:
        dataLine = setDataToDataArrayLine(elms, vtkFormat, vtuType)
        blockLines += [dataLine]
    blockLines += ["</DataArray>\n"]
    return blockLines

#
#  makeDataArrayBlock
def makeDataArrayBlock(vtkFormat, line, vals, nCells, name=""):
    """ dataArrayのblock行を作成する。"""
    vtuType, orgFormat = getVtuTypeFormatInLine(line)
    #formatを修正
    if vtkFormat != orgFormat:
        line = changeFormatToLine(vtkFormat, line)
    #nCompを設定
    line  = setNumberOfComponents(line, vals, nCells)
    #Nameを設定
    if name != "":
        line = setNewName(line, name)
    blockLines = [line]
    #DataArrayを作成
    if vtkFormat == "ascii":
        dataLine = " ".join(list(map(str, vals))) + "\n"
    else:
        dataLine = setDataToDataArrayLine(vals, vtkFormat, vtuType)
    blockLines +=[dataLine]
    #footerを作成
    blockLines += ["</DataArray>\n"]
    return blockLines

#
#  getVtkElementData
def getVtkElementData(elmGrpName, elmNodeDict, allNodes, elmData, nodesDict, elementGrps, surfaceGrps, nodeGrps, elmFaceNos, grpDicts, vtkFormat):
    """ 要素group毎のvtkファイルを作成する。"""
    surElmDict, nodeGrpNameDict = grpDicts
    lines  = ['<?xml version="1.0"?>\n']
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0"  byte_order="LittleEndian" header_type="UInt64">\n']
    lines += ['<UnstructuredGrid>\n']
    nPoints = len(allNodes)
    nCells = len(elmData)
    lines += ['<Piece NumberOfPoints="' + str(nPoints) + '" NumberOfCells="' + str(nCells) + '">\n']
    lines += ['<Points>\n']
    #points(座標データ)
    line = '<DataArray type="Float32" NumberOfComponents="3" format="ascii" >\n'
    vals = []
    for nodeNo in allNodes:
        loc = nodesDict[nodeNo]
        vals += loc
    blockLines = makePointsDataArrayBlock(vtkFormat, line, vals, nPoints)
    lines += blockLines
    lines += ['</Points>\n']
    lines += ['<Cells>\n']
    #connectivity
    line = '<DataArray type="Int32" Name="connectivity" format="ascii" >\n'
    nodes = []
    for elmNodes, vtkName in elmData:
        newNodes = list(map(lambda x: elmNodeDict[x], elmNodes))
        nodes.append(newNodes)
    blockLines = makeConnectivityDataArray(vtkFormat, line, nodes, nCells)
    lines += blockLines
    #offsets
    line = '<DataArray type="Int32" Name="offsets" format="' + vtkFormat + '">\n'
    elms = []
    n = 0
    for elmNodes, vtkName in elmData:
        n += len(elmNodes)
        elms.append(n)
    blockLines = makeOffsetsTypesDataArray(vtkFormat, line, elms, nCells, name="offsets")
    lines += blockLines
    #types
    line = '<DataArray type="UInt8" Name="types" format="ascii" >\n'
    vtkNames = []
    for elmNodes, vtkName in elmData:
        vtkNames.append(int(vtkName))
    blockLines = makeOffsetsTypesDataArray(vtkFormat, line, vtkNames, nCells, name="types")
    lines += blockLines
    lines += ['</Cells>\n']
    #grpName
    lines += ['<CellData>\n']
    #  elmNo, faceNo
    if len(elmFaceNos) > 0:
        #elmNo
        line = '<DataArray type="Int32" Name="elmNo" NumberOfComponents="1" format="ascii">\n'
        vals = []
        for elmFaceNo in elmFaceNos:
            elmNo = elmFaceNo[0]
            vals.append(elmNo)
        blockLines = makeDataArrayBlock(vtkFormat, line, vals, nCells)
        lines += blockLines
        #faceNo
        line = '<DataArray type="UInt8" Name="faceNo" NumberOfComponents="1" format="ascii">\n'
        vals = []
        for elmFaceNo in elmFaceNos:
            faceNo = elmFaceNo[1]
            vals.append(faceNo)
        blockLines = makeDataArrayBlock(vtkFormat, line, vals, nCells )
        lines += blockLines
    #  elementGrp
    for egrpName, _data in elementGrps:
        line = '<DataArray type="UInt8" Name="EGRP_xx" NumberOfComponents="1" format="ascii" >\n'
        vals = []
        for elmNodes, vtkName in elmData:
            nodes = elmNodes[:]
            nodes.sort()
            key = tuple(nodes)
            if key in surElmDict.keys():
                if egrpName in surElmDict[key]["EGRP"]:
                    vals.append(1)
                else:
                    vals.append(0)
            else:
                vals.append(0)
        name = "EGRP_" + egrpName
        blockLines = makeDataArrayBlock(vtkFormat, line, vals, nCells, name=name)
        lines += blockLines
    #  elementGrp「ALL」
    egrps = list(map(lambda x: x[0], elementGrps))
    if not "ALL" in egrps:
        #ALLが定義されていない場合、ALLを作成
        line = '<DataArray type="UInt8" Name="EGRP_ALL" NumberOfComponents="1" format="ascii" >\n'
        vals = [1 for i in range(len(elmData))]
        blockLines = makeDataArrayBlock(vtkFormat, line, vals, nCells)
        lines += blockLines
    #  surfaceGrp
    for sgrpName, _data in surfaceGrps:
        line = '<DataArray type="UInt8" Name="SGRP_xx" NumberOfComponents="1" format="ascii" >\n'
        vals = []
        for elmNodes, vtkName in elmData:
            nodes = elmNodes[:]
            nodes.sort()
            key = tuple(nodes)
            if key in surElmDict.keys():
                if sgrpName in surElmDict[key]["SGRP"]:
                    vals.append(1)
                else:
                    vals.append(0)
            else:
                vals.append(0)
        name = "SGRP_" + sgrpName
        blockLines = makeDataArrayBlock(vtkFormat, line, vals, nCells, name=name)
        lines += blockLines
    lines += ['</CellData>\n']
    #  nodeGrp
    lines += ['<PointData>\n']
    line = '<DataArray type="Int32" Name="nodeNo" NumberOfComponents="1" format="ascii" >\n'
    vals = []
    for nodeNo in allNodes:
        vals.append(nodeNo)
    blockLines = makeDataArrayBlock(vtkFormat, line, vals, nPoints)
    lines += blockLines
    for ngrpName, _data in nodeGrps:
        line = '<DataArray type="UInt8" Name="NGRP_xx" NumberOfComponents="1" format="ascii" >\n'
        vals = []
        for key in allNodes:
            if key in nodeGrpNameDict.keys():
                if ngrpName in nodeGrpNameDict[key]:
                    vals.append(1)
                else:
                    vals.append(0)
            else:
                vals.append(0)
        name = "NGRP_" + ngrpName
        blockLines = makeDataArrayBlock(vtkFormat, line, vals, nPoints, name=name)
        lines += blockLines
    #  nodeGrp「ALL」を出力
    ngrps = list(map(lambda x: x[0], nodeGrps))
    if not "ALL" in ngrps:
        #ALLが定義されていない場合、ALLを作成
        line = '<DataArray type="UInt8" Name="NGRP_ALL" NumberOfComponents="1" format="ascii" >\n'
        vals = [1 for key in allNodes]
        blockLines = makeDataArrayBlock(vtkFormat, line, vals, nPoints)
        lines += blockLines
    lines += ['</PointData>\n']
    lines += ['</Piece>\n']
    lines += ['</UnstructuredGrid>\n']
    lines += ['</VTKFile>\n']
    return lines

#
#  saveVersion
#---------------
def saveVersion():
    """ vtkFolder内にversionNoを保存"""
    versionFile = currDir + os.sep + SaveFolder + os.sep + "version_" + version
    cont = "version " + version
    try:
        f = open(versionFile, "w", encoding="utf-8"); f.write(cont); f.close()
    except:
        print("stopped to save version.")
        pass

#-----------------
#  createVtkFiles
#-----------------
def createVtkFiles(tempDir, vtkFormat="ascii"):
    """ 要素、表面、節点group毎にvtkFileを作成する。"""
    global currDir, surElmDict, nodeGrpNameDict
    #初期化
    currDir = tempDir
    os.chdir(currDir)
    surElmDict = {}
    nodeGrpNameDict = {}
    nCpu = mp.cpu_count()
    #mesh読み込み
    meshHeaderData = getMeshHeaderData()
    print("creating node/element dictionary...")
    nodesDict = getNodesDict(meshHeaderData)
    elementDict = getElementDict(meshHeaderData)
    clearSaveFolder()
    print("getting node/element groups...")
    nodeGrps = createNodeGrpVtkFiles(meshHeaderData)
    surfaceGrps = createSurfaceGrpVtkFiles(meshHeaderData, elementDict)
    #vtkFileを作成
    print("creating vtk files at " + vtkFormat + " format...")
    createElementGrpVtkFiles(meshHeaderData, nodesDict, elementDict,
                            surfaceGrps, nodeGrps, nCpu, vtkFormat)
    saveVersion()

#-----------------
#  renameNodeGrp
#-----------------
def renameNodeGrp(vtkDir, ngrpName, newName):
    """ 全egrpファイル内のngrpNameをnewNameに変更する"""
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        lines = readFileLines(fileName)
        si = lines.index("<PointData>\n")
        if si < 0:
            for i in range(len(lines)):
                if lines[i][:len("<PointData>")] == "<PointData>":
                    si = i
                    break
        if si >= 0:
            for i in range(si, len(lines)):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    words = line.split()
                    flag = 0
                    for ii in range(len(words)):
                        if words[ii][:len("Name=")] == "Name=":
                            if words[ii].split("=")[1] == '"NGRP_' + ngrpName + '"':
                                words[ii] = 'Name="NGRP_' + newName + '"'
                                newLine = " ".join(words) + "\n"
                                lines[i] = newLine
                                flag = 1
                                break
                    if flag == 1:
                        break
            writeFileLines(fileName, lines)

#--------------------
#  renameSurfaceGrp
#--------------------
def renameSurfaceGrp(vtkDir, sgrpName, newName):
    """ 全egrpファイル内のsgrpNameをnewNameに変更する"""
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        lines = readFileLines(fileName)
        si = lines.index("<CellData>\n")
        if si < 0:
            for i in range(len(lines)):
                if lines[i][:len("<CellData>")] == "<CellData>":
                    si = i
                    break
        if si >= 0:
            for i in range(si, len(lines)):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    words = line.split()
                    flag = 0
                    for ii in range(len(words)):
                        if words[ii][:len("Name=")] == "Name=":
                            if words[ii].split("=")[1] == '"SGRP_' + sgrpName + '"':
                                words[ii] = 'Name="SGRP_' + newName + '"'
                                newLine = " ".join(words) + "\n"
                                lines[i] = newLine
                                flag = 1
                                break
                    if flag == 1:
                        break
            writeFileLines(fileName, lines)

#-------------------
#  renameElementGrp
#-------------------
def renameElementGrp(vtkDir, egrpName, newName):
    """ 全egrpファイル内のegrpNameをnewNameに変更する"""
    #vtkFile名を変更する
    orgFile = vtkDir + os.sep + "elementGrp_" + egrpName + ".vtu"
    newFile = vtkDir + os.sep + "elementGrp_" + newName + ".vtu"
    orgEdgeFile = vtkDir + os.sep + "edgeElementGrp_" + egrpName + ".vtu"
    newEdgeFile = vtkDir + os.sep + "edgeElementGrp_" + newName + ".vtu"
    os.rename(orgFile, newFile)
    os.rename(orgEdgeFile, newEdgeFile)
    #  renameが完了するまで待つ
    while os.path.exists(newFile) == False and os.path.exists(newEdgeFile):
        time.sleep(0.1)
    #vtkFileの内容を修正
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        lines = readFileLines(fileName)
        si = lines.index("<CellData>\n")
        if si < 0:
            for i in range(len(lines)):
                if lines[i][:len("<CellData>")] == "<CellData>":
                    si = i
                    break
        if si >= 0:
            for i in range(si, len(lines)):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    words = line.split()
                    flag = 0
                    for ii in range(len(words)):
                        if words[ii][:len("Name=")] == "Name=":
                            if words[ii].split("=")[1] == '"EGRP_' + egrpName + '"':
                                words[ii] = 'Name="EGRP_' + newName + '"'
                                newLine = " ".join(words) + "\n"
                                lines[i] = newLine
                                flag = 1
                                break
                    if flag == 1:
                        break
            writeFileLines(fileName, lines)

#----------------
#  deleteNodeGrp
#----------------
def deleteNodeGrp(vtkDir, grpNames):
    """ 全egrpファイル内のngrpNamesを削除する"""
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        lines = readFileLines(fileName)
        si = lines.index("<PointData>\n")
        if si < 0:
            for i in range(len(lines)):
                if lines[i][:len("<PointData>")] == "<PointData>":
                    si = i
                    break
        if si >= 0:
            names = list(map(lambda x: 'Name="NGRP_'+x+'"', grpNames))
            newLines = lines[:si]
            i = si
            while i < len(lines):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    words = line.split()
                    flag = 0
                    for ii in range(len(words)):
                        if words[ii] in names:
                            i += 2      #2行分(data, footer)skipする
                            flag = 1
                            break
                    if flag == 0:
                        newLines.append(line)
                else:
                    newLines.append(line)
                i += 1
            #newLinesを保存               
            writeFileLines(fileName, newLines)

#-------------------
#  deleteSurfaceGrp
#-------------------
def deleteSurfaceGrp(vtkDir, grpNames):
    """ 全egrpファイル内のsgrpNamesを削除する"""
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        lines = readFileLines(fileName)
        si = lines.index("<CellData>\n")
        if si < 0:
            for i in range(len(lines)):
                if lines[i][:len("<CellData>")] == "<CellData>":
                    si = i
                    break
        if si >= 0:
            names = list(map(lambda x: 'Name="SGRP_'+x+'"', grpNames))
            newLines = lines[:si]
            i = si
            while i < len(lines):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    words = line.split()
                    flag = 0
                    for ii in range(len(words)):
                        if words[ii] in names:
                            i += 2      #2行分(data, footer)skipする
                            flag = 1
                            break
                    if flag == 0:
                        newLines.append(line)
                else:
                    newLines.append(line)
                i += 1
            #newLinesを保存               
            writeFileLines(fileName, newLines)

#-------------------
#  deleteElementGrp
#-------------------
def deleteElementGrp(vtkDir, grpNames):
    """ 全egrpファイル内のegrpNamesを削除する"""
    #対象のvtkFileを削除
    delFiles = list(map(lambda x: vtkDir+os.sep+"elementGrp_"+x+".vtu", grpNames))
    delEdgeFiles = list(map(lambda x: vtkDir+os.sep+"edgeElementGrp_"+x+".vtu", grpNames))
    for delFile in delFiles + delEdgeFiles:
        os.remove(delFile)
    #vtkFileの内容を修正
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        #削除対象以外のvtkFile？
        if not fileName in delFiles:
            #file毎に内容を修正        
            lines = readFileLines(fileName)
            si = lines.index("<CellData>\n")
            if si < 0:
                for i in range(len(lines)):
                    if lines[i][:len("<CellData>")] == "<CellData>":
                        si = i
                        break
            if si >= 0:
                names = list(map(lambda x: 'Name="EGRP_'+x+'"', grpNames))
                newLines = lines[:si]
                i = si
                while i < len(lines):
                    line = lines[i]
                    if line.find("<DataArray") >= 0:
                        words = line.split()
                        flag = 0
                        for ii in range(len(words)):
                            if words[ii] in names:
                                i += 2      #2行分(data, footer)skipする
                                flag = 1
                                break
                        if flag == 0:
                            newLines.append(line)
                    else:
                        newLines.append(line)
                    i += 1
                #newLinesを保存               
                writeFileLines(fileName, newLines)

#-------------------
#  addDeleteNodeGrps
#-------------------
def addDeleteNodeGrps(vtkDir, grpNames, newName):
    """ 全egrpファイル内に対し、newNameにgrpNamesを加え、
    grpNamesを削除する"""
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        lines = readFileLines(fileName)
        si = lines.index("<PointData>\n")
        if si < 0:
            #見つからなかった場合、1行ずつ確認する。
            for i in range(len(lines)):
                if lines[i][:len("<PointData>")] == "<PointData>":
                    si = i
                    break
        if si >= 0:
            newLines = lines[:si]
            #grpNamesの値を取得する。
            grpNamesData = []
            repLines = []
            flag = 0
            for i in range(si, len(lines)):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    for grpName in grpNames:
                        if line.find('Name="NGRP_' + grpName + '"') >= 0:
                            vtuType, vtuFormat = getVtuTypeFormatInLine(line)
                            dataLine = lines[i+1]
                            grpData = getDataFromDataArrayLine(dataLine, vtuFormat, vtuType)
                            grpNamesData.append(grpData)
                            repLines.append(i)
                            flag += 1
                            break
                if flag >= len(grpNames):
                    break
            #newNameの値を取得
            for i in range(si, len(lines)):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    if line.find('Name="NGRP_' + newName + '"') >= 0:
                        vtuType, vtuFormat = getVtuTypeFormatInLine(line)
                        dataLine = lines[i+1]
                        newNameData = getDataFromDataArrayLine(dataLine, vtuFormat, vtuType)
                        grpNamesData.append(newNameData)
                        repLines.append(i)
            #grpNames, newNameの値を加算する
            nData = len(grpNamesData[0])
            res = [0 for i in range(nData)]
            for data in grpNamesData:
                res = list(map(lambda x, y: x + y, res, data))
            for i in range(nData):
                if res[i] > 0:
                    res[i] = 1
            #newNameLineを置き換え
            rep_i = repLines[-1]
            header = lines[rep_i]
            words = header.split()
            for i in range(len(words)):
                if words[i][:len("Name=")] == "Name=":
                    vtuType, vtuFormat = getVtuTypeFormatInLine(header)
                    words[i] = 'Name="NGRP_' + newName + '"'
                    break
            header = " ".join(words) + "\n"
            lines[rep_i] = header
            newNameLine = setDataToDataArrayLine(res, vtuFormat, "UInt8")
            lines[rep_i+1] = newNameLine
            repLines = repLines[:-1]
            #grpNameを削除
            newLines = []
            i = si
            while i < len(lines):
                #repLinesから2行を削除
                if i in repLines:
                    i += 2
                else:
                    newLines.append(lines[i])
                i += 1
            newLines = lines[:si] + newLines
            #newLinesを保存
            writeFileLines(fileName, newLines)

#----------------------
#  addDeleteSurfaceGrps
#----------------------
def addDeleteSurfaceGrps(vtkDir, grpNames, newName):
    """ 全egrpファイル内に対し、newNameにgrpNamesを加え、
    grpNameを削除する"""
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        lines = readFileLines(fileName)
        si = lines.index("<CellData>\n")
        if si < 0:
            #見つからなかった場合、1行づつ確認する。
            for i in range(len(lines)):
                if lines[i][:len("<CellData>")] == "<CellData>":
                    si = i
                    break
        if si >= 0:
            newLines = lines[:si]
            #grpNamesの値を取得する
            grpNamesData = []
            repLines = []
            flag = 0
            for i in range(si, len(lines)):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    for grpName in grpNames:
                        if line.find('Name="SGRP_' + grpName + '"') >= 0:
                            vtuType, vtuFormat = getVtuTypeFormatInLine(line)
                            dataLine = lines[i+1]
                            grpData = getDataFromDataArrayLine(dataLine, vtuFormat, vtuType)
                            grpNamesData.append(grpData)
                            repLines.append(i)
                            flag += 1
                            break
                if flag >= len(grpNames):
                    break
            #newNameの値を取得
            for i in range(si, len(lines)):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    if line.find('Name="SGRP_' + newName + '"') >= 0:
                        vtuType, vtuFormat = getVtuTypeFormatInLine(line)
                        dataLine = lines[i+1]
                        newNameData = getDataFromDataArrayLine(dataLine, vtuFormat, vtuType)
                        grpNamesData.append(newNameData)
                        repLines.append(i)
            #grpNames, newNameの値を加算する
            nData = len(grpNamesData[0])
            res = [0 for i in range(nData)]
            for data in grpNamesData:
                res = list(map(lambda x, y: x + y, res, data))
            for i in range(nData):
                if res[i] > 0:
                    res[i] = 1
            #newNameLineを置き換え
            rep_i = repLines[-1]
            header = lines[rep_i]
            words = header.split()
            for i in range(len(words)):
                if words[i][:len("Name=")] == "Name=":
                    vtuType, vtuFormat = getVtuTypeFormatInLine(header)
                    words[i] = 'Name="SGRP_' + newName + '"'
                    break
            header = " ".join(words) + "\n"
            lines[rep_i] = header
            newNameLine = setDataToDataArrayLine(res, vtuFormat, "UInt8")
            lines[rep_i+1] = newNameLine
            repLines = repLines[:-1]
            #grpNameを削除
            newLines = []
            i = si
            while i < len(lines):
                #repLinesから2行を削除
                if i in repLines:
                    i += 2
                else:
                    newLines.append(lines[i])
                i += 1
            newLines = lines[:si] + newLines
            #newLinesを保存
            writeFileLines(fileName, newLines)

#----------------------
#  addDeleteSurfaceGrp
#----------------------
def addDeleteSurfaceGrp(vtkDir, grpName, newName):
    """ 全egrpファイル内に対し、newNameにgrpNameを加え、
    grpNameを削除する"""
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        lines = readFileLines(fileName)
        si = lines.index("<CellData>\n")
        if si < 0:
            for i in range(len(lines)):
                if lines[i][:len("<CellData>")] == "<CellData>":
                    si = i
                    break
        if si >= 0:
            newLines = lines[:si]
            #grpName, newNameのindexを確認
            flag = 0
            for i in range(si, len(lines)):
                line = lines[i]
                if line.find("<DataArray") >= 0:
                    if line.find('Name="SGRP_' + grpName + '"') >= 0:
                        i_grpName = i + 1
                        flag += 1
                    elif line.find('Name="SGRP_' + newName + '"') >= 0:
                        i_newName = i + 1
                        flag += 1
                if flag >= 2:
                    break
            #newNameにgrpNameを加算
            grpNameData = list(map(int, lines[i_grpName].split()))
            newNameData = list(map(int, lines[i_newName].split()))
            for i in range(len(grpNameData)):
                if grpNameData[i] != 0:
                    newNameData[i] = 1
            newNameLine = " ".join(list(map(str, newNameData))) + "\n"
            lines[i_newName] = newNameLine
            #grpNameを削除
            newLines = lines[:i_grpName-1] + lines[i_grpName+2:]
            #newLinesを保存
            writeFileLines(fileName, newLines)

#----------------------
#  addDeleteElementGrp
#----------------------
def addDeleteElementGrp(vtkDir, grpName, newName):
    """ 全egrpファイル内に対し、newNameにgrpNameを加え、
    grpNameを削除する"""
    #対象のvtkFileを削除
    delFile = vtkDir + os.sep + "elementGrp_" + grpName + ".vtu"
    os.remove(delFile)
    #vtkFileの修正
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        #delFile以外のfile？
        if fileName != delFile:
            lines = readFileLines(fileName)
            si = lines.index("<CellData>\n")
            if si < 0:
                for i in range(len(lines)):
                    if lines[i][:len("<CellData>")] == "<CellData>":
                        si = i
                        break
            if si >= 0:
                newLines = lines[:si]
                #grpName, newNameのindexを確認
                flag = 0
                for i in range(si, len(lines)):
                    line = lines[i]
                    if line.find("<DataArray") >= 0:
                        if line.find('Name="EGRP_' + grpName + '"') >= 0:
                            i_grpName = i + 1
                            flag += 1
                        elif line.find('Name="EGRP_' + newName + '"') >= 0:
                            i_newName = i + 1
                            flag += 1
                    if flag >= 2:
                        break
                #newNameにgrpNameを加算
                grpNameData = list(map(int, lines[i_grpName].split()))
                newNameData = list(map(int, lines[i_newName].split()))
                for i in range(len(grpNameData)):
                    if grpNameData[i] != 0:
                        newNameData[i] = 1
                newNameLine = " ".join(list(map(str, newNameData))) + "\n"
                lines[i_newName] = newNameLine
                #grpNameを削除
                newLines = lines[:i_grpName-1] + lines[i_grpName+2:]
                #newLinesを保存
                writeFileLines(fileName, newLines)

#-------------------
#  addNodeGrp
#-------------------
def addNodeGrp(vtkFile, ngrpName, ngrpData):
    """ 指定したegrpファイル（vtkファイル）内にngrpを追加する。
    同名のngrpが存在する場合は、置き換える"""
    lines = readFileLines(vtkFile)
    vtuFormat = getVtuFormat(lines)
    nPoints = getNumberOfPoints(lines)
    nodeNos = getNodeNoOfFistr(lines)
    headerLine = createNodeGrpHeader(ngrpName, vtuFormat)
    stat, si, orgData = getReplaceInsertPoint(ngrpName, lines, nPoints)
    dataLine = createNodeGrpData(nPoints, nodeNos, ngrpData, orgData, vtuFormat)
    footerLine = createNodeGrpFooter()
    lines = insertNgrpGrpLine(headerLine, dataLine, footerLine, ngrpName, lines, stat, si)
    writeFileLines(vtkFile, lines)
    if stat == "insert":
        writeNodeGrpOtherFiles(vtkFile, ngrpName)

#--------------------
#  addSurfaceGrp
#--------------------
def addSurfaceGrp(vtkFile, sgrpName, sgrpData):
    """ 指定したegrpファイル内にsgrpを追加する。
    同名のngrpが存在する場合は、置き換える"""
    lines = readFileLines(vtkFile)
    vtuFormat = getVtuFormat(lines)
    nCells = getNumberOfCells(lines)
    cellNos, faceNos = getCellFaceNoOfFistr(lines)
    headerLine = createSurfaceGrpHeader(sgrpName, vtuFormat)
    stat, si, orgData = getPeplaceInsertCell(sgrpName, lines, nCells)
    dataLine = createSurfaceGrpData(nCells, cellNos, faceNos, sgrpData, orgData, vtuFormat)
    footerLine = createNodeGrpFooter()
    lines = insertSurfaceGrpLine(headerLine, dataLine, footerLine, sgrpName, lines, stat, si)
    writeFileLines(vtkFile, lines)
    if stat == "insert":
        writeSurfaceGrpOtherFiles(vtkFile, sgrpName)

#-------------------
#  delNodeGrp
#-------------------
def delNodeGrp(vtkFile, ngrpName, ngrpData):
    """ 指定したegrpファイル（vtkファイル）内のngrpから
    ngrpDataを削除する"""
    lines = readFileLines(vtkFile)
    vtuFormat = getVtuFormat(lines)
    nPoints = getNumberOfPoints(lines)
    nodeNos = getNodeNoOfFistr(lines)
    headerLine = createNodeGrpHeader(ngrpName, vtuFormat)
    stat, si, orgData = getReplaceInsertPoint(ngrpName, lines, nPoints)
    dataLine = createDelNodeGrpData(nPoints, nodeNos, ngrpData, orgData, vtuFormat)
    footerLine = createNodeGrpFooter()
    lines = insertNgrpGrpLine(headerLine, dataLine, footerLine, ngrpName, lines, stat, si)
    writeFileLines(vtkFile, lines)
    if stat == "insert":
        writeNodeGrpOtherFiles(vtkFile, ngrpName)

#--------------------
#  delSurfaceGrp
#--------------------
def delSurfaceGrp(vtkFile, sgrpName, sgrpData):
    """ 指定したegrpファイル内のsgrpから
    sgrpDataを削除する"""
    lines = readFileLines(vtkFile)
    vtuFormat = getVtuFormat(lines)
    nCells = getNumberOfCells(lines)
    cellNos, faceNos = getCellFaceNoOfFistr(lines)
    headerLine = createSurfaceGrpHeader(sgrpName, vtuFormat)
    stat, si, orgData = getPeplaceInsertCell(sgrpName, lines, nCells)
    dataLine = createDelSurfaceGrpData(nCells, cellNos, faceNos, sgrpData, orgData, vtuFormat)
    footerLine = createNodeGrpFooter()
    lines = insertSurfaceGrpLine(headerLine, dataLine, footerLine, sgrpName, lines, stat, si)
    writeFileLines(vtkFile, lines)
    if stat == "insert":
        writeSurfaceGrpOtherFiles(vtkFile, sgrpName)


#
#  writeSurfaceGrpOtherFiles
def writeSurfaceGrpOtherFiles(vtkFile, sgrpName):
    """ vtkFile以外のfileにngrpNameを追加する"""
    vtkDir = os.path.dirname(vtkFile)
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        if fileName != vtkFile:
            #sgrpを追加
            lines = readFileLines(fileName)
            vtuFormat = getVtuFormat(lines)
            nCells = getNumberOfCells(lines)
            #cellData = ["0" for i in range(nCells)]
            cellData = [0 for i in range(nCells)]
            #dataLine = " ".join(cellData) + "\n"
            dataLine = setDataToDataArrayLine(cellData, vtuFormat, "UInt8")
            stat, si, _orgData = getPeplaceInsertCell(sgrpName, lines, nCells)
            if stat == "insert":
                header = createSurfaceGrpHeader(sgrpName, vtuFormat)
                footer = createNodeGrpFooter()
                lines = lines[:si] + [header] + [dataLine] + [footer] + lines[si:]
                writeFileLines(fileName, lines)

#
#  writeNodeGrpOtherFiles
def writeNodeGrpOtherFiles(vtkFile, ngrpName):
    """ vtkFile以外のfileにngrpNameを追加する"""
    vtkDir = os.path.dirname(vtkFile)
    vtkFiles = glob.glob(vtkDir + os.sep + "elementGrp_*")
    for fileName in vtkFiles:
        if fileName != vtkFile:
            #ngrpを追加
            lines = readFileLines(fileName)
            vtuFormat = getVtuFormat(lines)
            nPoints = getNumberOfPoints(lines)
            #nodeData = ["0" for i in range(nPoints)]
            nodeData = [0 for i in range(nPoints)]
            #dataLine = " ".join(nodeData) + "\n"
            dataLine = setDataToDataArrayLine(nodeData, vtuFormat, "UInt8")
            stat, si, _orgData = getReplaceInsertPoint(ngrpName, lines, nPoints)
            if stat == "insert":
                header = createNodeGrpHeader(ngrpName, vtuFormat)
                footer = createNodeGrpFooter()
                lines = lines[:si] + [header] + [dataLine] + [footer] + lines[si:]
                writeFileLines(fileName, lines)

#
#  getPeplaceInsertCell
def getPeplaceInsertCell(grpName, lines, nCells):
    """ 置き換え、挿入場所を取得する"""
    stat = ""
    for i in range(len(lines)):
        line = lines[i]
        if line.find("<DataArray") >= 0:
            if line.find("SGRP_"+grpName) >= 0:
                stat = "replace"
                vtuType, vtuFormat = getVtuTypeFormatInLine(line)
                si = i
                break
        elif line.find("</CellData>") >= 0:
            si = i
            stat = "insert"
            break
    if stat == "replace":
        dataLine = lines[si+1]
        orgData = getDataFromDataArrayLine(dataLine, vtuFormat, vtuType)
    else:
        orgData = [0 for i in range(nCells)]
    return stat, si, orgData

#
#  getReplaceInsertPoint
def getReplaceInsertPoint(grpName, lines, nPoints):
    """ 置き換え、挿入場所を確認する"""
    stat = ""
    for i in range(len(lines)):
        line = lines[i]
        if line.find("<DataArray") >= 0:
            if line.find("NGRP_"+grpName) >= 0:
                vtuType, vtuFormat = getVtuTypeFormatInLine(line)
                stat = "replace"
                si = i
                break
        elif line.find("</PointData>") >= 0:
            si = i
            stat = "insert"
            break
    if stat == "replace":
        dataLine = lines[si+1]
        orgData = getDataFromDataArrayLine(dataLine, vtuFormat, vtuType)
    else:
        orgData = [0 for i in range(nPoints)] 
    return stat, si, orgData

#
#  insertSurfaceGrpLine
def insertSurfaceGrpLine(header, data, footer, sgrpName, lines, stat, si):
    """ header, dataをlinesに挿入する。"""
    if stat == "insert":
        #挿入する
        lines = lines[:si] + [header] + [data] + [footer] + lines[si:]
    else:
        #置き換える（同名のngrpが存在する場合）
        lines[si] = header
        lines[si+1] = data
        lines[si+2] = footer
    return lines

#
#  createSurfaceGrpData
def createSurfaceGrpData(nCells, cellNos, faceNos, sgrpData, orgData, vtuFormat):
    """ 「0,1」のsurfaceGrpデータを作成する"""
    cellData = orgData
    #grpDataを[cell, face]でまとめる
    grpData = []
    for i in range(0, len(sgrpData), 2):
        cellNo = sgrpData[i]
        faceNo = sgrpData[i+1]
        grpData.append([cellNo, faceNo])
    #cellNosとfaceNosを[cell, face]でまとめる
    faceCellNoDict = {}
    for i in range(len(cellNos)):
        cellNo = cellNos[i]
        faceNo = faceNos[i] % 100
        faceCellNoDict[(cellNo, faceNo)] = i
    #cellDataに値をセット
    for (cellNo, faceNo) in grpData:
        id = faceCellNoDict[(cellNo, faceNo)]
        cellData[id] = 1
    #cellDataを作成
    cellData = list(map(str, cellData))
    line = setDataToDataArrayLine(cellData, vtuFormat, "UInt8")
    return line

#
#  createDelSurfaceGrpData
def createDelSurfaceGrpData(nCells, cellNos, faceNos, sgrpData, orgData, vtuFormat):
    """ 「0,1」のsurfaceGrpデータを作成する"""
    cellData = orgData
    #grpDataを[cell, face]でまとめる
    grpData = []
    for i in range(0, len(sgrpData), 2):
        cellNo = sgrpData[i]
        faceNo = sgrpData[i+1]
        grpData.append([cellNo, faceNo])
    #cellNosとfaceNosを[cell, face]でまとめる
    faceCellNoDict = {}
    for i in range(len(cellNos)):
        cellNo = cellNos[i]
        faceNo = faceNos[i] % 100
        faceCellNoDict[(cellNo, faceNo)] = i
    #cellDataに値をセット
    for (cellNo, faceNo) in grpData:
        id = faceCellNoDict[(cellNo, faceNo)]
        cellData[id] = 0
    #cellDataを作成
    cellData = list(map(str, cellData))
    line = setDataToDataArrayLine(cellData, vtuFormat, "UInt8")
    return line


#
#  createSurfaceGrpHeader
def createSurfaceGrpHeader(sgrpName, vtuFormat):
    """ nodeGrpのheder部を作成する"""
    if vtuFormat == "ascii":
        line = '<DataArray type="UInt8" Name="SGRP_' + sgrpName + '" NumberOfComponents="1" format="ascii">\n'
    else:
        line = '<DataArray type="UInt8" Name="SGRP_' + sgrpName + '" NumberOfComponents="1" format="binary">\n'        
    return line

#
#  getCellFaceNoOfFistr
def getCellFaceNoOfFistr(lines):
    """ FistrのcellNoとfaceNoを取得"""
    cellNos = []
    faceNos = []
    flag = 0
    for i in range(len(lines)):
        line = lines[i]
        if line.find('Name="elmNo"') >= 0:
            elmType, elmFormat = getVtuTypeFormatInLine(line)
            #cellNos = lines[i+1].split()
            elmDataLine = lines[i+1]
            flag += 1
        elif line.find('Name="faceNo"') >= 0:
            faceType, faceFormat = getVtuTypeFormatInLine(line)
            #faceNos = lines[i+1].split()
            faceDataLine = lines[i+1]
            flag += 1
        if flag == 2:
            break
    #値を取得
    cellNos = getDataFromDataArrayLine(elmDataLine, elmFormat, elmType)
    faceNos = getDataFromDataArrayLine(faceDataLine, faceFormat, faceType)
    return cellNos, faceNos       

#
#  insertNgrpGrpLine
def insertNgrpGrpLine(header, data, footer, ngrpName, lines, stat, si):
    """ header, dataをlinesに挿入する。"""
    if stat == "insert":
        #挿入する（同名のngrpが存在しない場合）
        lines = lines[:si] + [header] + [data] + [footer] + lines[si:]
    else:
        #置き換える（同名のngrpが存在する場合）
        lines[si] = header
        lines[si+1] = data
        lines[si+2] = footer
    return lines

#
#  createNodeGrpFooter
def createNodeGrpFooter():
    """ nodeGrpのFooter作成"""
    return "</DataArray>\n"

#
#  createNodeGrpHeader
def createNodeGrpHeader(ngrpName, vtuFormat):
    """ nodeGrpのheder部を作成する"""
    if vtuFormat == "ascii":
        line = '<DataArray type="UInt8" Name="NGRP_' + ngrpName + '" NumberOfComponents="1" format="ascii">\n'
    else:
        line = '<DataArray type="UInt8" Name="NGRP_' + ngrpName + '" NumberOfComponents="1" format="binary">\n'
    return line

#
#  createNodeGrpData
def createNodeGrpData(nPoints, nodeNos, ngrpData, orgData, vtuFormat):
    """ 「0,1」のnodeGrpデータを作成する"""
    #nodeNo -> id 取得の為の辞書作成
    nodeNoDict = {}
    for i in range(len(nodeNos)):
        nodeNo = nodeNos[i]
        nodeNoDict[nodeNo] = i
    #pointDataにセット
    pointData = orgData
    for nodeNo in ngrpData:
        id = nodeNoDict[nodeNo]
        pointData[id] = 1
    #pointData作成
    pointData = list(map(str, pointData))
    line = setDataToDataArrayLine(pointData, vtuFormat, "UInt8")
    return line

#
#  createDelNodeGrpData
def createDelNodeGrpData(nPoints, nodeNos, ngrpData, orgData, vtuFormat):
    """ 「0,1」のnodeGrpデータを作成する"""
    #nodeNo -> id 取得の為の辞書作成
    nodeNoDict = {}
    for i in range(len(nodeNos)):
        nodeNo = nodeNos[i]
        nodeNoDict[nodeNo] = i
    #pointDataにセット
    pointData = orgData
    for nodeNo in ngrpData:
        id = nodeNoDict[nodeNo]
        pointData[id] = 0
    #pointData作成
    pointData = list(map(str, pointData))
    line = setDataToDataArrayLine(pointData, vtuFormat, "UInt8")
    return line

#
#  getNodeNoOfFistr
def getNodeNoOfFistr(lines):
    """ FistrのnodeNoを取得"""
    nodeNos = []
    for i in range(len(lines)):
        line = lines[i]
        n = line.find('Name="nodeNo"')
        if n >= 0:
            vtuType, vtuFormat = getVtuTypeFormatInLine(line)
            dataLine = lines[i+1]
            nodeNos = getDataFromDataArrayLine(dataLine, vtuFormat, vtuType)
            break 
    return nodeNos

#
#  getNumberOfPoints
def getNumberOfPoints(lines):
    nPoints = 0
    for line in lines:
        n = line.find("NumberOfPoints")
        if n >= 0:
            word = line[n:].split()[0]
            nPoints = word.split('"')[1]
            break
    return int(nPoints)

#
#  getNumberOfCells
def getNumberOfCells(lines):
    nCells = 0
    for line in lines:
        n = line.find("NumberOfCells")
        if n >= 0:
            word = line[n:].split()[0]
            nCells = word.split('"')[1]
            break
    return int(nCells)


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    #gettext.install("app")
    #_ = gettext.gettext

    #log表示
    import logFileCreater
    logFileCreater.log()

    if len(sys.argv) == 1:
        currDir = os.getcwd()
        vtkFormat = "ascii"
    elif len(sys.argv) == 2:
        currDir = sys.argv[1]
        vtkFormat = "ascii"
    elif len(sys.argv) >=3:
        currDir = sys.argv[1]
        vtkFormat = sys.argv[2]
    currDir = os.path.abspath(currDir)
    createVtkFiles(currDir, vtkFormat)
