#!/usr/bin/python3
# -*- coding: UTF-8 -*-
#
#       copyFilesFromTempToCurrDir.py
#
#           tempFolder内のfileをcurrDirに転送する。
#           　cntファイル：等分布荷重のセットがあれば、それをセット
#           　mshファイル：data省略形から内容を展開してセット
#
#
#   19/03/07    新規作成
#      03/14    isLatestFile: otherFilesをコピーする時、最新fileのみコピーする様に修正
#               （currDirの内容が旧のfileに置き換わってしまう為。）
#               copyFilesFromTempToCurrDir: timeStamp（tempCaseDate）の設定方法を修正
#               （保存しても、timeStampが変わらなかった為）
#   21/04/13    deleteGrpHeaderFromMsh:NGRP, SGRP, EGRPの削除を追加
#      05/17    os.systemをsubprocessに変更
#      05/27    copyFilesFromTempToCurrDir:「hecmw_part_ctrl.dat」の扱い方を変更。
#               fistr1を実行すると「hecmw_part_ctrl.dat」が消えたり現れたりするので
#               vtu変換時に並列結果がシングル結果として変換される。
#   22/03/31    copyMshFromTempToCurrDir:「!END」有無を確認し、無ければ追加。
#      12/20    getMeshContsFromTempFolder:currDirの読み取りエラーが発生の為、修正
#               command「*initialNames」→「*initempNames」に変更したため。
#      12/27    getMeshContsFromTempFolder:inivelNames, iniaccNamesを削除
#   23/04/13    "FistrModel.cnt"を使わずにhecmw_cnt.dat内から取得する様に修正
#      04/17    "FistrModel.msh"を使わずにhecmw_cnt.dat内から取得する様に修正
#      04/28    createHeaderDataFromTemp:!INITIAL CONDITIONをshrinkの対象にした結果
#               復元ができていなかったので、修正
#      11/18    createHeaderDataFromTemp:initialConditionの設定が正しくできていなかったので修正
#               nodeNo毎に設定した時に正しく設定されなかった。（バグ修正）
#   24/02/27    createHeaderDataFromTemp:AMPLITUDEのdataは、ampのcsvファイルから
#               取得する様に修正。（tempはshrinkしているため）
#      03/06    copyCntFromTempToCurrDir:contFile名を絶対pathで読み込む様に修正。
#      07/07    openにencoding="utf-8"を追加
#      12/31    copyCntFromTempToCurrDir:起動直後に、新規Grpを作成した時、エラー発生。
#               addTrueForceInCntFile.py起動時に、currDirの設定を追加。
#   25/01/23    createHeaderDataFromTemp:"INITIAL CONDITION"cardが検出できていなかった為、修正
#

import sys
import os
import glob
import shutil
import time
import subprocess

import pyFistr

ls = "\n"
FistrModel_cnt = ""
binAppPath = pyFistr.binAppPath
pythonAppPath = pyFistr.pythonAppPath

#tempFolderにコピーする時、shrinkするkeywordを取得（これ以外はそのままコピーする）
shrinkKeywords = pyFistr.shrinkKeywords


#
#  copyCntFromTempToCurrDir
#    temp内のcntファイルに等分布荷重を追加しcurrDirに保存する
def copyCntFromTempToCurrDir():
    tempFile = currDirTemp + os.sep + FistrModel_cnt
    cntFile = currDir + os.sep + FistrModel_cnt
    saveDir = os.getcwd()
    os.chdir(currDir)
    #  等分布荷重,mapping等を設定して保存
    comm = "python3 " + binAppPath + "addTrueForceInCntFile.py " + tempFile + " " + cntFile
    proc = subprocess.run(comm, shell=True)
    os.chdir(saveDir)

#
#  simpleCopyCntFromTempToCurrDir
def simpleCopyCntFromTempToCurrDir():
    tempFile = currDirTemp + os.sep + FistrModel_cnt
    #cntFile = currDir + os.sep + FistrModel_cnt
    shutil.copy(tempFile, currDir)

#
#  deleteGrpHeaderFromMsh
def deleteGrpHeaderFromMsh(lines, grp, grpNames):
    """ lines（mshファイル）から指定のgrpを削除して返す"""
    #headerContを取得
    meshHeaderConts = pyFistr.getHeaderData(lines)
    #grpを削除
    newHeaderConts = deleteGrpHeaderCont(meshHeaderConts, grp, grpNames)
    lines = []
    for header, data in newHeaderConts:
        lines += [header]
        if len(data) > 0:
            lines += data
    return lines

#
#  deleteGrpFromHeaderNumData
def deleteGrpFromHeaderNumData(headerNumData, grp, grpNames):
    """ headerNumData(mshファイル)から指定のgrpを削除して返す"""
    headerNumData = deleteGrpHeaderCont(headerNumData, grp, grpNames)
    return headerNumData

#  deleteGrpHeaderCont
def deleteGrpHeaderCont(meshHeaderConts, grp, grpNames):
    """ meshHeaderContsからgrpを削除する"""
    #checkCardを取得
    if grp == "NGRP":
        cardName = "!NGROUP"
        defCardName = "!NODE"
    elif grp == "SGRP":
        cardName = "!SGROUP"
        defCardName = ""
    elif grp == "EGRP":
        cardName = "!EGROUP"
        defCardName = "!ELEMENT"
    #grpを削除
    newHeaderConts = []
    for header, data in meshHeaderConts:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == defCardName:
            name = pyFistr.getValName(header, grp)
            if name in grpNames:
                newHeader = pyFistr.deleteValName(header, grp)
                newHeaderConts.append([newHeader, data])
            else:
                newHeaderConts.append([header, data])
        elif words[0] == cardName:
            name = pyFistr.getValName(header, grp)
            if name in grpNames:
                pass
            else:
                newHeaderConts.append([header, data])
        else:
            newHeaderConts.append([header, data])
    return newHeaderConts

#
#  copyMshFromTempToCurrDir
#    temp内のmshファイルから実mshを作成しcurrDirに保存する
def copyMshFromTempToCurrDir():
    print("saving mesh data of temp folder to currDir...")
    #FistrModel.mshを転送
    meshHeaderConts = createHeaderDataFromTemp()
    #  最終のheaderを確認
    lastHeader = meshHeaderConts[-1][0]
    words = pyFistr.deleteSp(lastHeader).split(",")
    if words[0].upper() != "!END":
        #最終が!ENDない場合は、!ENDを追加する
        meshHeaderConts.append(["!END", []])
    #linesを作成
    lines = []
    for i in range(len(meshHeaderConts)):
        headerCont = meshHeaderConts[i]
        header = headerCont[0]
        lines += [header]
        if len(headerCont[1]) != 0:
            lines += headerCont[1]
    #linesを保存
    fileName = currDir + os.sep + FistrModel_msh
    f = open(fileName, "w", encoding="utf-8")
    for line in lines:
        f.write(line)
    f.close()

#
#  createHeaderDataFromTemp
#    tempから実meshHeaderDataを作成する
def createHeaderDataFromTemp(numFlag=False):
    """ tempから実meshHeaderDataを作成する。
    !NODE, !ELEMENT, !EGROUP, !SGROUP, !NGROUP, !EQUATION, !AMPLITUDE カードの
    data部を実dataに戻す。
    headerは、tempのheaderに置き換える。
    
    Args:
        numFlag=True:   headerNumDataを返す
        numFlag=False:  headerData(text)を返す"""

    def getNextTempHeader(meshHeaderConts, keyword, j, data):
        cont, j = getNextHeader(meshHeaderConts, keyword, j)
        if len(cont) == 0:
            cont = data
        return [cont, j]                    

    def getNextHeader(meshHeaderConts, keyword, j):
        cont = []
        while j < len(meshHeaderConts):
            header = meshHeaderConts[j][0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == keyword:
                cont = meshHeaderConts[j][1]
                break
            j += 1
        return [cont, j]
    
    def getAmpDataInFile(fileName):
        ampData = []
        contList = pyFistr.readCsvData(fileName)
        for row in contList[2:]:
            data = row[1] + row[2]
            if data != "":
                #timeとvalueを入れ替える
                ampData += [", ".join([row[2], row[1]]) + ls]
        return ampData

    if numFlag==True:
        meshHeaderConts = pyFistr.getHeaderNumDataMsh(currDirTemp)
        trueMeshHeaderConts = pyFistr.getHeaderNumDataMsh(currDir)
    else:
        tempFile = currDirTemp + os.sep + FistrModel_msh
        trueFile = currDir + os.sep + FistrModel_msh
        f=open(tempFile, encoding="utf-8"); lines = f.readlines(); f.close()
        meshHeaderConts = pyFistr.getHeaderData(lines)
        f = open(trueFile, encoding="utf-8"); lines = f.readlines(); f.close()
        trueMeshHeaderConts = pyFistr.getHeaderData(lines)
    #入れ替えるcardを定義（定義されているcardは、実folderの内容を使用する）
    #changeCards = ["!NODE", "!ELEMENT", "!EGROUP", "!SGROUP", "!NGROUP", "!EQUATION"]
    changeCards = shrinkKeywords
    #入れ替え
    j = 0
    for i in range(len(meshHeaderConts)):
        headerCont = meshHeaderConts[i]
        header = headerCont[0].upper()
        words = pyFistr.deleteSp(header).split(",")
        #initialConditionをチェック
        if (words[0] == "!INITIALCONDITION" or 
            words[0] == "!INITIAL_CONDITION" or
            words[0] == "!INITIAL CONDITION"):
            data = headerCont[1]
            char0 = data[0][0]                  #1行目の1文字
            if "0" <= char0 and char0 <= "9":   #数字？
                #変更しない（最初の文字が数字の場合）
                [cont, j] = getNextHeader(trueMeshHeaderConts, words[0].upper(), j)
            else:
                #tempの内容に置き換える（最初の文字が数字以外:変数の場合）
                [cont, j] = getNextTempHeader(meshHeaderConts, words[0].upper(), j, data)
            j += 1
            meshHeaderConts[i][1] = cont
            #break
        #AMPLITUDEをチェック
        elif words[0].upper() == "!AMPLITUDE":
            ampName = pyFistr.getValName(header, "NAME")
            fileName = currDirTemp + os.sep + ampName + "_ampData.csv"
            ampData = getAmpDataInFile(fileName)
            meshHeaderConts[i][1] = ampData
        #INITIAL CONDITION、AMP以外
        else:
            #chageCards内をチェック
            for changeCard in changeCards:
                if words[0].upper() == changeCard:
                    [cont, j] = getNextHeader(trueMeshHeaderConts, changeCard, j)
                    j += 1
                    meshHeaderConts[i][1] = cont
                    break
    return meshHeaderConts

#
#  getMeshContsFromTempFolder
#    tempFolderからその内容を取得する
def getMeshContsFromTempFolder():
    meshConts = {}
    meshConts["currDir"] = os.getenv("HOME")
    fileName = currDirTemp + os.sep + "meshConts"
    f = open(fileName, encoding="utf-8")
    lines = f.readlines()
    f.close()
    flag = ""
    for i in range(len(lines)):
        line = lines[i]
        if line[:len("*currDir")] == "*currDir":
            flag = "currDir"
            meshConts["currDir"] = ""
        elif line[:len("*meshCont")] == "*meshCont":
            flag = "meshCont"
            meshConts["meshCont"] = ""
        elif line[:len("*ampNames")] == "*ampNames":
            flag = "ampNames"
            meshConts["ampNames"] = []
        elif line[:len("*initempNames")] == "*initempNames":
            flag = "initempNames"
            meshConts["initempNames"] = []
        elif line[:len("*initialNames")] == "*initialNames":    #旧のコマンド
            flag = "initempNames"
            meshConts["initempNames"] = []
        elif line[:len("*overlapNGRP")] == "*overlapNGRP":
            flag = "overlapNGRP"
            meshConts["overlapNGRP"] = []
        elif line[:len("*tempCaseDate")] == "*tempCaseDate":
            flag = "tempCaseDate"
            meshConts["tempCaseDate"] = ""
        elif line[0] == "*":
            #その他コマンドは無視
            flag = ""
        else:
            #コマンドの内容を取得
            if flag == "currDir" and len(line.split()) > 0:
                meshConts["currDir"] = line.split()[0]
            elif flag == "meshCont":
                meshConts["meshCont"] += line
            elif flag == "ampNames" and len(line.split()) > 0:
                meshConts["ampNames"] += [line.split()[0]]
            elif flag == "initempNames" and len(line.split()) > 0:
                meshConts["initempNames"] += [line.split()[0]]
            elif flag == "overlapNGRP" and len(line.split()) > 0:
                meshConts["overlapNGRP"] += [line.split()[0]]
            elif flag == "tempCaseDate" and len(line.split()) > 0:
                meshConts["tempCaseDate"] = float(line.split()[0])
    return meshConts

#  saveCaseContDict
#    dict形式のdataを保存する
def saveCaseContDict(dataDict):
    cont = ""
    items = list(dataDict.keys())
    items.sort()
    for item in items:
        value = dataDict[item]
        if type(value) == list:
            value = "\n".join(value)
        elif type(value) == float or type(value) == int:
            value = str(value)
        else:
            if value[-1] == "\n":
                value = value[:-1]
        cont += "*" + item + "\n" + value + "\n"
    #fileに書き込み
    fileName = currDirTemp + os.sep + "meshConts"
    f = open(fileName, "w", encoding="utf-8")
    f.write(cont)
    f.close()

#
#  isLatestFile
#    最新fileかどうかチェック
def isLatestFile(fileName, currDir):
    name = fileName.split(os.sep)[-1]
    fileDate = os.path.getmtime(fileName)
    currFile = currDir + os.sep + name
    #if len(glob.glob(currFile)) == 0:
    if os.path.exists(currFile) == False:
        return True
    currDate = os.path.getmtime(currFile)
    return fileDate > currDate

#
#  remakeTimeStampVtuMeshFile
def remakeTimeStampVtuMeshFile(currDir):
    """ vtuのmeshFileのtimeStampを修正する。"""
    vtuFile = currDir + os.sep + "convFistrModel.msh.vtu"
    if os.path.exists(vtuFile):
        atime = mtime = time.time()
        os.utime(vtuFile, (atime, mtime))
        print("changed time stamp of vtuMeshFile.")

#
#  copyFilesFromTempToCurrDir
#  --------------------------
#    tempFolder内のfileをcurrDirに転送する
def copyFilesFromTempToCurrDir(currDirTemp1, currDir1):
    global currDirTemp, currDir, FistrModel_cnt, FistrModel_msh
    currDirTemp, currDir = currDirTemp1, currDir1
    FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(currDirTemp)
    #その他fileを転送
    copyFiles = []
    files = glob.glob(currDirTemp + os.sep + "*")
    for fileName in files:
        if os.path.isfile(fileName) == True:
            name = fileName.split(os.sep)[-1]
            ext = name.split(".")[-1]
            if (name == FistrModel_cnt
                or name == FistrModel_msh
                or name == "meshConts"
                or name == "run"
                or ext == "bat"
                or ext == "bin"):
                pass
            else:
                if isLatestFile(fileName, currDir) == True:
                    copyFiles.append(fileName)
    #  currTempの「hecmw_part_ctrl.dat」がない場合は、curr側を削除
    fileName = currDirTemp + os.sep + "hecmw_part_ctrl.dat"
    if os.path.exists(fileName) == False:
        delName = currDir + os.sep + "hecmw_part_ctrl.dat"
        if os.path.exists(delName) == True:
            os.remove(delName)
    #  fileを転送
    for fileName in copyFiles:
        shutil.copy(fileName, currDir)
    #FistrModel.cntを転送（等分布荷重をセット）
    print (ls+"Copying cnt file...")
    copyCntFromTempToCurrDir()
    #FistrModel.mshを転送（実メッシュを作成）
    print (ls+"Copying msh file...")
    copyMshFromTempToCurrDir()
    #tempCaseDateを更新（timeStampを更新）
    meshConts = getMeshContsFromTempFolder()
    #tempCaseDate = os.path.getmtime(currDirTemp + os.sep + "FistrModel.cnt") + 1.0
    tempCaseDate = time.time() + 1.0
    meshConts["tempCaseDate"] = tempCaseDate
    saveCaseContDict(meshConts)
    #vtu形式のmeshFileのtimeStampを修正
    remakeTimeStampVtuMeshFile(currDir)
    print (ls+"Copied cnt, msh and other files to currDir.")
    print ("Finished to prepare for FrontISTR.", ls)
    

if __name__ == "__main__":
    currDirTemp = sys.argv[1]
    currDir = sys.argv[2]
    copyFilesFromTempToCurrDir(currDirTemp, currDir)

