#!/usr/bin/python3
#  coding: utf-8
#
#   calcSectionCoeff.py
#
#       はりの断面定数を計算する。
#       X軸方向がはりの方向になる。
#
#   class名                 内容
#   --------------------------------------
#   sectionCircle           丸棒
#   sectionPipe             パイプ
#   sectionRectangle        角棒
#   sectionRectanglePipe    中空角棒
#   sectionH                H型
#   sectionI                I型
#   sectionL                L型
#   sectionC                C型
#   sectionU                U型
#
#   関数名    内容          
#   ------------------------------------
#   Jx      ねじり定数
#   Iz      Z軸回りの断面二次モーメント
#   Iy      Y軸回りの断面二次モーメント
#   Ix      X軸回りの断面二次極モーメント
#   A       断面積
#   Zz      Z軸回りの断面系数
#   Zy      Y軸回りの断面系数
#   k1      系数k1（角棒のみ）
#   k2      系数k2（角棒のみ）
#   k3      系数k3（角棒のみ）
#
#
#   21/04/29    新規作成
#      04/30    sectionC, sectionUを追加
#   22/08/11    YZ軸方向を入れ替え（回転）
#               Z軸が基準の為、これに合わせた。
#


import math

#-----------------------
#  sectionCircle class
#-----------------------
class sectionCircle:
    """ 円形断面の定数を計算する。

    Args:
        d(float)    :外径
    """

    def __init__(self, d):
        self.d = d

    def Jx(self):
        """ ねじり定数Jxを計算"""
        Jx = self.Ix()
        return Jx

    def Iz(self):
        """ Z軸回りの断面2次モーメント"""
        d = self.d
        Iz = math.pi * d**4 / 64.0
        return Iz

    def Iy(self):
        """ Y軸回りの断面2次モーメント"""
        Iy = self.Iz()
        return Iy

    def Ix(self):
        """ X軸回りの断面2次極モーメント"""
        Ix = 2.0 * self.Iz()
        return Ix

    def A(self):
        """ 断面積"""
        A = math.pi * self.d**2 / 4.0
        return A

    def Zz(self):
        """ Z軸の断面系数"""
        Iz = self.Iz()
        r = self.d / 2.0
        Zz = Iz / r
        return Zz

    def Zy(self):
        """ Z軸の断面系数"""
        Iy = self.Iy()
        r = self.d / 2.0
        Zy = Iy / r
        return Zy

#-------------------
#  sectinPipe class
#-------------------
class sectionPipe:
    """ パイプの断面の定数を計算する。

    Args:
        d(float)    :外径
        t(float)    :板厚
    """

    def __init__(self, d, t):
        self.d = d
        self.t = t

    def Jx(self):
        """ ねじり定数Jxを計算"""
        Jx = self.Ix()
        return Jx

    def Iz(self):
        """ Z軸回りの断面2次モーメント"""
        d = self.d
        d2 = self.d - 2.0 * self.t
        I1 = math.pi * d**4 / 64.0
        I2 = math.pi * d2**4 / 64.0
        Iz = I1 - I2
        return Iz

    def Iy(self):
        """ Y軸回りの断面2次モーメント"""
        Iy = self.Iz()
        return Iy

    def Ix(self):
        """ X軸回りの断面2次極モーメント"""
        Ix = 2.0 * self.Iz()
        return Ix

    def A(self):
        """ 断面積"""
        A1 = math.pi * self.d**2 / 4.0
        d2 = self.d - 2.0 * self.t
        A2 = math.pi * d2**2 / 4.0
        A = A1 - A2
        return A

    def Zz(self):
        """ Z軸回りの断面系数"""
        Iz = self.Iz()
        r = self.d / 2.0
        Zz = Iz / r
        return Zz

    def Zy(self):
        """ Y軸回りの断面系数"""
        Iy = self.Iy()
        r = self.d / 2.0
        Zy = Iy / r
        return Zy

#-------------------------
#  sectionRectangle class
#-------------------------
class sectionRectangle:
    """ 長方形断面の定数を計算する。

    Args:
        b(float)    :幅
        h(float)    :高さ

    Form:
                 (Z)
                b |
            -------------
            |     |     | h
          --|-----------|-----(Y)
            |     |     |
            -------------
                  |
    """

    def __init__(self, b, h):
        self.b = b
        self.h = h
        self.tolerance = 1.0e-6     #計算精度（エラー対策）
        self.nLoop = 50             #loop回数

    #
    #  Jx
    #---------
    def Jx(self):
        """ ねじり定数Jxを計算"""
        k2 = self.k2()
        Jx = k2 * self.b * self.h**3
        return Jx

    #
    #  Iy
    #-------
    def Iy(self):
        """ Y軸回りの断面2次モーメント(中立軸:Y）"""
        Iy = self.b * self.h**3 / 12.0
        return Iy

    #
    #  Iz
    #--------
    def Iz(self):
        """ Z軸回りの断面2次モーメント（中立軸:Z）"""
        Iz = self.b**3 * self.h / 12.0
        return Iz

    #
    #  Ix
    #-------
    def Ix(self):
        """ X軸回りの断面2次極モーメント"""
        Ix = self.Iz() + self.Iy()
        return Ix

    #
    #  A
    #------
    def A(self):
        """ 断面積"""
        A = self.b * self.h
        return A

    #
    #  Zz
    #------
    def Zz(self):
        """ Z軸回りの断面系数"""
        Iz = self.Iz()
        r = self.b / 2.0
        Zz = Iz / r
        return Zz

    #
    #  Zy
    #------
    def Zy(self):
        """ Y軸回りの断面系数"""
        Iy = self.Iy()
        r = self.h / 2.0
        Zy = Iy / r
        return Zy

    #
    #  k
    #----------
    def k(self):
        """ 基本の定数kを算出"""

        def loopK(n):
            a = self.b
            b = self.h
            n2 = (2 * n - 1)
            x = n2 * math.pi * a / (2.0 * b)
            try:
                A = n2**2 * math.cosh(x)
                ans = 1.0 / A
            except:
                ans = 0.0
            return ans

        sumK = 0.0
        n = 1
        while n < self.nLoop:
            kn = loopK(n)
            if kn < self.tolerance:
                break
            sumK += kn
            n += 1
        k = 1.0 - 8.0 / math.pi**2 * sumK
        return k

    #
    #  k1
    #----------
    def k1(self):
        """ 定数k1を算出"""
        k2 = self.k2()
        k = self.k()
        k1 = k2 / k
        return k1

    #
    #  k2
    #-----------
    def k2(self):
        """ 定数k2を算出"""

        def loopK2(n):
            a = self.b
            b = self.h
            n2 = (2 * n - 1)
            A = 1.0 / (n2**5)
            B = n2 * math.pi * a / (2 * b)
            ans = A * math.tanh(B)
            return ans

        sumK2 = 0.0
        n = 1
        while n < self.nLoop:
            sumK2 += loopK2(n)
            n += 1
        A = (192.0 / math.pi**5) * (self.h / self.b)
        k2 = (1.0 - A * sumK2) / 3.0
        return k2

    #
    #  k3
    #----------
    def k3(self):
        """ 定数k3を算出"""

        def loopK3(n):
            a = self.b
            b = self.h
            n2 = (2 * n - 1)
            x = n2 * math.pi * a / (2.0 * b)
            ans = -(-1.0)**n / n2**2 * math.tanh(x)
            return ans
        
        sumK3 = 0.0
        n = 1
        while n < self.nLoop:
            sumK3 += loopK3(n)
            n += 1
        k1 = self.k1()
        k2 = self.k2()
        k3 = 8.0 * k1 / (math.pi**2 * k2) * sumK3
        return k3


#-----------------------------
#  sectionRectanglePipe class
#-----------------------------
class sectionRectanglePipe:
    """ 中空の長方形断面の定数を計算する。

    Args:
        b(float)    :幅
        h(float)    :高さ
        t(float)    :板厚

    Form:
                  (Z)
                 b |  t
            ===============
            ||     |     || h
          --||-----------||-----(Y)
            ||     |     ||t
            ===============
                   |
    """

    def __init__(self, b, h, t):
        self.b = b
        self.h = h
        self.t = t

    def Jx(self):
        """ ねじり定数Jxを計算
        薄肉閉断面として計算する。"""
        b = self.b - self.t     #板厚tの中心線上の寸法
        h = self.h - self.t     #   ↑
        t = self.t
        Jx = 2.0 * t**2 * b**2 * h**2 / (t*(b + h))
        return Jx

    def Iy(self):
        """ Y軸回りの断面2次モーメント"""
        I1 = self.b * self.h**3 / 12.0
        I2 = (self.b - 2.0 * self.t) * (self.h - 2.0 * self.t)**3 / 12.0
        Iy = I1 - I2
        return Iy

    def Iz(self):
        """ Z軸回りの断面2次モーメント"""
        b = self.h
        h = self.b
        I1 = b * h**3 / 12.0
        b = self.h - 2.0 * self.t
        h = self.b - 2.0 * self.t
        I2 = b * h**3 / 12.0
        Iz = I1 - I2
        return Iz

    def Ix(self):
        """ X軸回りの断面2次極モーメント"""
        Iz = self.Iz()
        Iy = self.Iy()
        Ix = Iz + Iy
        return Ix

    def A(self):
        """ 断面積"""
        A1 = self.b * self.h
        b = self.b - 2.0 * self.t
        h = self.h - 2.0 * self.t
        A2 = b * h
        A = A1 - A2
        return A

    def Zz(self):
        """ Z軸回りの断面系数"""
        Iz = self.Iz()
        r = self.b / 2.0
        Zz = Iz / r
        return Zz

    def Zy(self):
        """ Y軸回りの断面系数"""
        Iy = self.Iy()
        r = self.h / 2.0
        Zy = Iy / r
        return Zy

#------------------
#  sectionH class
#------------------
class sectionH:
    """ H型の断面定数を計算する。

    Args:
        b(float)    :幅
        h(float)    :高さ
        t1(float)   :中心の板厚
        t2(float)   :左右の板厚

    Form:
                 (Z)
                b |
            --    |    --
            ||    | t1 || h
         ---||=========||-----(Y)
            ||    |    ||t2
            --    |    --
                  |
    """

    def __init__(self, b, h, t1, t2):
        self.b = b
        self.h = h
        self.t1 = t1        #中心の板厚
        self.t2 = t2        #左右の板厚

    def Jx(self):
        """ ねじり定数Jxを計算
        薄肉開断面として計算する。
        ∑bt^3/3"""
        b = self.h
        t = self.t2
        Jh = b * t**3 / 3.0
        b = self.b - 2.0 * self.t2
        t = self.t1
        Jb = b * t**3 / 3.0
        Jx = 2.0 * Jh + Jb
        return Jx

    def Iy(self):
        """ Y軸回りの断面2次モーメント"""
        b = self.t2
        h = self.h
        I1 = sectionRectangle(b, h).Iy()
        b = self.b - 2.0 * self.t2
        h = self.t1
        I2 = sectionRectangle(b, h).Iy()
        Iy = 2.0 * I1 + I2
        return Iy

    def Iz(self):
        """ Z軸回りの断面2次モーメント"""
        b = self.h
        h = self.b
        Iall = sectionRectangle(b, h).Iy()
        b = self.h - self.t1
        h = self.b - 2.0 * self.t2
        Idel = sectionRectangle(b, h).Iy()
        Iz = Iall - Idel
        return Iz

    def Ix(self):
        """ X軸回りの断面2次極モーメント"""
        Iz = self.Iz()
        Iy = self.Iy()
        Ix = Iz + Iy
        return Ix

    def A(self):
        """ 断面積"""
        Ah = self.h * self.t2
        b = self.b - 2.0 * self.t2
        Ab = b * self.t1
        A = 2.0 * Ah + Ab
        return A

    def Zz(self):
        """ Z軸回りの断面系数"""
        Iz = self.Iz()
        r = self.b / 2.0
        Zz = Iz / r
        return Zz

    def Zy(self):
        """ Y軸回りの断面系数"""
        Iy = self.Iy()
        r = self.h / 2.0
        Zy = Iy / r
        return Zy

#-----------------
#  sectionI class
#-----------------
class sectionI:
    """ H型の断面定数を計算する。

    Args:
        b(float)    :幅
        h(float)    :高さ
        t1(float)   :中心の板厚
        t2(float)   :上下の板厚

    Form:
                  (Z)
                b |
              ====|==== -----
                 |||t1      | h
          -------|||-------------(Y)
                 |||        |
              ====|==== -----
                  |  t2
    """

    def __init__(self, b, h, t1, t2):
        self.b = b
        self.h = h
        self.t1 = t1        #中心の板厚
        self.t2 = t2        #上下の板厚
        self.secH = sectionH(self.h, self.b, self.t1, self.t2)

    def Jx(self):
        """ ねじり定数Jxを計算
        薄肉開断面として計算する。"""
        Jx = self.secH.Jx()
        return Jx
    
    def Iz(self):
        """ Z軸回りの断面2次モーメント"""
        Iz = self.secH.Iy()
        return Iz
    
    def Iy(self):
        """ Y軸回りの断面2次モーメント"""
        Iy = self.secH.Iz()
        return Iy

    def Ix(self):
        """ X軸回りの断面2次極モーメント"""
        Ix = self.secH.Ix()
        return Ix

    def A(self):
        """ 断面積"""
        A = self.secH.A()
        return A

    def Zz(self):
        """ Z軸回りの断面系数"""
        Zz = self.secH.Zy()
        return Zz

    def Zy(self):
        """ Y軸回りの断面系数"""
        Zy = self.secH.Zz()
        return Zy

#-----------------
#  sectionL class
#-----------------
class sectionL:
    """ L型の断面定数を計算する。

    Args:
        b(float)    :幅
        h(float)    :高さ
        t1(float)   :垂直板の板厚
        t2(float)   :水平板の板厚

    Form:
                (Z)
            t1   |
            --   |
          h ||   | 
        ----||--------------(Y)
            ||   |
            ============ t2
                 |  b
    """

    def __init__(self, b, h, t1, t2):
        self.b = b
        self.h = h
        self.t1 = t1
        self.t2 = t2

    def Jx(self):
        """ ねじり定数Jxを計算
        薄肉開断面として計算する。
        ∑bt^3/3"""
        b = self.b
        t = self.t2
        Jb = b * t**3 / 3.0
        b = self.h - self.t2
        t = self.t1
        Jh = b * t**3 / 3.0
        Jx = Jb + Jh
        return Jx

    def e12(self, b, h, t1, t2):
        """ 図心の位置

        Returns:
            e1  :t1側
            e2  :t2側
        e1 + e2 = h, e1 > e2"""
        A = h**2 * t1 + t2**2 * (b - t1)
        B = 2.0 * (b * t2 + t1 * (h - t2))
        e1 = h - A / B
        e2 = A / B
        return e1, e2

    def Iy(self):
        """ Y軸回りの断面2次モーメント"""
        b = self.b
        h = self.h
        t1 = self.t1
        t2 = self.t2
        e1, e2 = self.e12(b, h, t1, t2)
        Iy = (t1 * e1**3 + b * e2**3 - (b - t1) * (e2 - t2)**3) / 3.0
        return Iy

    def Iz(self):
        """ Z軸回りの断面2次モーメント"""
        b = self.h
        h = self.b
        t1 = self.t2
        t2 = self.t1
        e1, e2 = self.e12(b, h, t1, t2)
        Iz = (t1 * e1**3 + b * e2**3 - (b - t1) * (e2 - t2)**3) / 3.0
        return Iz

    def Ix(self):
        """ X軸回りの断面2次極モーメント"""
        Ix = self.Iz() + self.Iy()
        return Ix

    def A(self):
        """ 断面積"""
        Ab = self.b * self.t2
        h = self.h - self.t2
        Ah = h * self.t1
        A = Ab + Ah
        return A

    def Zz(self):
        """ Z軸回りの断面系数"""
        Iz = self.Iz()
        b = self.b
        h = self.h
        t1 = self.t1
        t2 = self.t2
        e1, e2 = self.e12(h, b, t2, t1)
        Zz = Iz / e1
        return Zz

    def Zy(self):
        """ Y軸回りの断面系数"""
        Iy = self.Iy()
        b = self.b
        h = self.h
        t1 = self.t1
        t2 = self.t2
        e1, e2 = self.e12(b, h, t1, t2)
        Zy = Iy / e1
        return Zy

#------------------
#  sectionC class
#------------------
class sectionC:
    """ C型の断面定数を計算する。

    Args:
        b(float)    :幅
        h(float)    :高さ
        t1(float)   :垂直板の板厚
        t2(float)   :水平板の板厚

    Form:
                (Z)
            t1   |
            ||==========
          h ||   | 
        ----||---------------(Y)
            ||   |
            ||========== t2
                 |  b
    """

    def __init__(self, b, h, t1, t2):
        self.b = b
        self.h = h
        self.t1 = t1
        self.t2 = t2

    def Jx(self):
        """ ねじり定数Jxを計算
        薄肉開断面として計算する。
        ∑bt^3/3"""
        b = self.b - self.t1
        t = self.t2
        Jb = b * t**3 / 3.0
        b = self.h
        t = self.t1
        Jh = b * t**3 / 3.0
        Jx = Jb + Jh + Jb
        return Jx

    def Iy(self):
        """ Y軸回りの断面2次モーメント"""
        Iall = self.b * self.h**3 / 12.0
        b = self.b - self.t1
        h = self.h - 2.0 * self.t2
        Idel = b * h**3 / 12.0
        Iy = Iall - Idel
        return Iy

    def e12(self, b, h, t1, t2):
        A = 2.0 * b**2 * t2 + t1**2 * (h - 2.0 * t2)
        B = 2.0 * (h * t1 + 2.0 * t2 * (b - t1))
        e1 = b - A / B
        e2 = A / B
        return e1, e2

    def Iz(self):
        """ Z軸回りの断面2次モーメント"""
        b = self.b
        h = self.h
        t1 = self.t1
        t2 = self.t2
        e1, e2 = self.e12(b, h, t1,t2)
        Iz = (2.0 * t2 * e1**3 + h * e2**3 - (h - 2.0 * t2) * (e2 - t1)**3) / 3.0
        return Iz

    def Ix(self):
        """ X軸回りの断面2次極モーメント"""
        Ix = self.Iz() + self.Iy()
        return Ix

    def A(self):
        """ 断面積"""
        Ab = (self.b - self.t1) * self.t2
        Ah = self.h * self.t1
        A = Ab + Ah + Ab
        return A

    def Zy(self):
        """ Z軸回りの断面系数"""
        Iy = self.Iy()
        r = self.h / 2.0
        Zy = Iy / r
        return Zy

    def Zz(self):
        """ Y軸回りの断面系数"""
        Iz = self.Iz()
        b = self.b
        h = self.h
        t1 = self.t1
        t2 = self.t2
        e1, e2 = self.e12(b, h, t1,t2)
        Zz = Iz / e1
        return Zz

#-----------------
#  sectionU class
#-----------------
class sectionU:
    """ U型の断面定数を計算する。

    Args:
        b(float)    :幅
        h(float)    :高さ
        t1(float)   :水平板の板厚
        t2(float)   :垂直板の板厚

    Form:
                 (Z)
            t2    |
            ||    |    ||
          h ||    |    || 
         ---||---------||----(Y)
            ||    |    ||
            ============= t1
                  |  b
    """

    def __init__(self, b, h, t1, t2):
        self.b = b
        self.h = h
        self.t1 = t1
        self.t2 = t2
        #b, hを入れ替えてsectionCを作成する
        self.sect = sectionC(h, b, t1, t2)

    def Jx(self):
        """ ねじり定数Jxを計算
        薄肉開断面として計算する。
        ∑bt^3/3"""
        Jx = self.sect.Jx()
        return Jx

    def Iz(self):
        """ Z軸回りの断面2次モーメント"""
        Iz = self.sect.Iy()
        return Iz

    def Iy(self):
        """ Y軸回りの断面2次モーメント"""
        Iy = self.sect.Iz()
        return Iy

    def Ix(self):
        """ X軸回りの断面2次極モーメント"""
        Ix = self.sect.Ix()
        return Ix

    def A(self):
        """ 断面積"""
        A = self.sect.A()
        return A

    def Zz(self):
        """ Z軸回りの断面系数"""
        Zz = self.sect.Zy()
        return Zz

    def Zy(self):
        """ Y軸回りの断面系数"""
        Zy = self.sect.Zz()
        return Zy

if __name__ == "__main__":
    b = 0.1; h = 0.05; t1 = 4; t2 = 4
    d = 0.01; t = 0.004

    secRect = sectionU(200, 100, 10, 20)
    Ix = secRect.Ix()
    Iz = secRect.Iz()
    Iy = secRect.Iy()
    Jx = secRect.Jx()
    A = secRect.A()
    Zy = secRect.Zy()
    Zz = secRect.Zz()
    print("----", Iz, Iy, Zz)

