#!/usr/bin/python3
#
#       unv2fistrEx.py
#
#   unv形式のメッシュファイルをFrontISTR形式に変換する
#
#   19/05/07    新規作成
#   20/09/21    remakeElements:要素type:91,94を追加
#   21/05/17    os.systemをsubprocessに変更
#      07/06    writeElement:!SECTIONの書き込みをまとめて実施
#               sectionの順番はsolid, beam, shellの順にする必要がある
#               メッシュ変換時は良いが、後でメッシュを追加する場合、
#               sectionの順に注意する必要がある為。
#   22/03/05    setSurfaceGrpOfShell:新規追加。
#               shell変換時にshellのsurGrpを作成する。
#      04/12    log表示修正（import logFileCreaterを追加）
#   24/07/07    openにencoding="utf-8"を追加
#   25/04/17    作成したmeshの保存file名変更
#               
#

import sys
import os
import glob
import subprocess
import unv2xx as unv
import convertMesh as cm
import pyFistr

#
# elementType(unv→fistr)
types = {
    # unvType fistrType
        11: "611",                #beam1  'B31',
        #22: "612",                #beam2
        41: "S341",               #tri1   'STRI35',
        42: "S342",               #tri2   'S6',
        44: "S361",               #quad1  'S4R5',
        45: "S362",               #quad2  'S8',
        91: "S341",               #tri1   'CPS3',
        94: "S361",               #quad1  'CPS4',
        111: "341",               #tetra1 'C3D4',
        112: "351",               #prism1 'C3D6',
        113: "352",               #prism2 'C3D15',
        115: "361",               #hex1   'C3D8',
        116: "362",               #hex2   'C3D20',
        118: "342"                #tetra2 'C3D10'
        }

ls = "\n"

#
#  Node object
#
class Node:
    """ Node object
    
    Attributes:
        id (int)            :節点No
        coords list(float)  :xyz座標値のリスト
        elmNos list(int)    :この節点を含むsolid要素Noのリスト
        dummyNode (int)     :この節点に対応するdummy節点No
    """

    def __init__(self, id, coords):
        self.id = id                #nodeNo
        self.coords = coords        #xyz座標値
        self.elmNos = []            #この節点を含む要素No
        self.dummyNode = -1         #対応するdummy節点No

#
#  Element object
#
class Element:
    """ Element object

    Attributes:
        id (int)        :要素No
        nodes list(int) :要素を構成する節点No
        type (str)      :要素type
        egrp (str)      :この要素が属するgroup名
    """
    
    def __init__(self, nodes, type):
        self.nodes = nodes          #nodeNo
        self.type = type            #要素type
        self.egrp = ""              #要素group名

#
#  fistrFEM object
#
class fistrFEM:
    """ fistrFEM object

    Attributes:
        allNodes (dict)     :全節点の辞書 allNodes[nodeNo] = Node obj
        allElms (dict)      :全要素の辞書 allElms[elNo] = Element obj
        nodes list(list)    :節点のdata [header, 節点Dict]
        elms list(list)     :要素のdata [header, 要素Dict]
        surGrps list(list)  :面groupのdata [header, list(面data)]
        nodeGrps list(list) :節点groupのdata [header, list(nodeNo)]
        elmDict (dict)      :要素type VS [区分,次数,種別,節点数] の辞書
    """
    def __init__(self):
        self.allNodes = {}          #NodeClassのリスト
        self.allElms = {}           #ElementClassのリスト
        self.nodes = []             #[header, nodeDict]のリスト
        self.elms = []              #[header, elmsDict]のリスト
        self.surGrps = []           #[header, surGrps]のリスト
        self.nodeGrps = []          #[header, nodeGrps]のリスト
        #  要素内容を定義
        self.elmDict = {
            #name  区分    次数　種別　　節点数  
            "341":["solid",  1, "tet",   4],
            "342":["solid",  2, "tet",   10],
            "361":["solid",  1, "hex",   8],
            "362":["solid",  2, "hex",   20],
            "351":["solid",  1, "prism", 6],
            "352":["solid",  2, "prism", 15],
            "731":["shell",  1, "tri",   3],
            "732":["shell",  2, "tri",   6],
            "741":["shell",  1, "quad",  4],
            "761":["shell3", 1, "tri",   6],        #3自由度shell
            "781":["shell3", 1, "quad",  8],        #3自由度shell
            "611":["beam",   1, "beam",  2],
            #"612":["beam",   2, "beam",  3],        #追加
            "641":["beam3",  1, "beam",  4],        #3自由度beam
            "S341":["face",  1, "tri",   3],        #tet1の面
            "S342":["face",  2, "tri",   6],        #tet2の面
            "S361":["face",  1, "quad",  4],        #hex1の面
            "S362":["face",  2, "quad",  8]         #hex2の面
            }

#
#  setValName
def setValName(line, val):
    """ 文字列にvalの値をセットする

    Args:
        line (str)  : 文字列
        val (str)   : line内にval("TYPE=341"をセットする
                      設定されている場合は、置き換わる
    Returns:
        (str) : 設定された文字列が戻る
    Example:
        line = setValName(line, "TYPE=342")
    """
    valNames = val.split("=")
    words = deleteSp(line).split(",")
    i = 0
    flag = 0
    for word in words:
        word = deleteSp(word)
        name = word.split("=")[0].upper()
        if name == valNames[0]:
            words[i] = " " + val
            flag = 1
            break
        i += 1
    if flag == 0:
        words.append(" "+val)
    newLine = ",".join(words) + ls
    return newLine

#
#  getValName
def getValName(line, valName):
    """ 行（文字列）から変数の値を取得する
    
    Args:
        line(str)   : 文字列
        valName(str): 取得する変数名("TYPE")
    Returns:
        (str): 変数の値
    Example:
        a = getValName(line, "TYPE")
    """
    valName = valName.upper()
    words = deleteSp(line).split(",")
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        word = deleteSp(word)
        if word.upper().find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    return name

#
#  deleteSp
def deleteSp(line):
    """ 空白、行末コードを削除して返す"""
    newLine = ""
    for chara in line:
        if chara != " " and chara != "\n" and chara != "\r":
            newLine += chara
    return newLine

#
#  isThereSolidElement
def isThereSolidElement(fistrModel, FEM):
    """ モデル内にsolid要素が存在するかどうか"""
    ans = False
    for elem in FEM.elems:
        if elem.type in types.keys():
            fistrType = types[elem.type]
            if fistrModel.elmDict[fistrType][0] == "solid":
                ans = True
                break
    return ans

#
#  setNodes
#
def setNodes(fistrModel):
    """ NODEを取得する"""
    print("getting nodes for FrontISTR format...")
    nodeDict = {}
    for nodeNo in fistrModel.allNodes.keys():
        nodeLoc = fistrModel.allNodes[nodeNo].coords
        nodeDict[nodeNo] = nodeLoc
    header = "!NODE" +ls
    fistrModel.nodes.append([header, nodeDict])
    return fistrModel

#
#  remakeElements
#
def remakeElements(FEM):
    """ 不要な要素を削除し、要素の節点並び順をfistr用に修正する。"""

    #要素の節点並び順を修正する関数群（unv→fistr）
    #  unv節点並び [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20]とした時
    #  1次要素は、修正せずにそのまま
    def changeTri2(n):      #三角形2次
        # [1, 3, 5, 4, 6, 2]
        ns = [n[0], n[2], n[4], n[3], n[5], n[1]]
        return ns
    def changeQuad2(n):     #四角形2次
        # [1, 3, 5, 7, 2, 4, 6, 8]
        ns = [n[0], n[2], n[4], n[6], n[1], n[3], n[5], n[7]]
        return ns
    def changeTet2(n):      #四面体2次
        # [1, 3, 5, 10, 4, 6, 2, 7, 8, 9]
        ns = [n[0], n[2], n[4], n[9], n[3], n[5], n[1], n[6], n[7], n[8]]
        return ns
    def changeHex2(n):      #六面体2次
        # [1, 3, 5, 7, 13, 15, 17, 19, 2, 4, 6, 8, 14, 16, 18, 20, 9, 10, 11, 12]
        ns = [n[0], n[2], n[4], n[6], n[12], n[14], n[16], n[18], n[1], n[3],
            n[5], n[7], n[13], n[15], n[17], n[19], n[8], n[9], n[10], n[11]]
        return ns
    def changePrism2(n):    #五面体2次
        # [1, 3, 5, 10, 12, 14, 4, 6, 2, 13, 15, 11, 7, 8, 9]
        ns = [n[0], n[2], n[4], n[9], n[11], n[13], n[3], n[5], n[1], n[12],
            n[14], n[10], n[6], n[7], n[8]]
        return ns
    def noChange(n):
        return n

    # FEM classから読み込んだelementをelementType別に保存する辞書
    #  これ以外のtypeは取得しない
    elemdic = {     # 22追加
            #11:[],22:[],
            11:[],
            41:[],42:[],44:[],45:[],91:[],94:[],
            111:[],112:[],113:[],115:[],116:[],118:[]
            }

    #節点の並び順を修正する関数（1次要素は修正しない）
    funcDict = {
        11: noChange,           #beam1
        #22: noChange,           #beam2      追加
        41: noChange,           #tri1
        42: changeTri2,         #tri2
        44: noChange,           #quad1
        45: changeQuad2,        #quad2
        91: noChange,
        94: noChange,
        111: noChange,          #tetra1
        112: noChange,          #prism1
        113: changePrism2,      #prism2
        115: noChange,          #hexa1
        116: changeHex2,        #hexa2
        118: changeTet2         #tetra2
        }

    # elements
    #  first delete all elems from X_* groups
    X_Ids = []        
    for group in FEM.elemsets:
        if group.name.startswith('X_'):
            X_Ids.extend(group.items)
            FEM.elemsets.remove(group)     #elSetリストから削除           
    FEM.nelemsets = len(FEM.elemsets)
    #要素の節点並び順をunv→fistr用に修正
    print("renumbering node order of element...")
    for i in range(len(FEM.elems)):
        elm = FEM.elems[i]
        unvType = elm.type
        nodes = elm.cntvt
        if unvType in funcDict.keys():
            func = funcDict[unvType]
            FEM.elems[i].cntvt = func(nodes)
    #elemdicを作成
    for elem in FEM.elems:
        if elem.type in types.keys():
            elemdic[elem.type].append(elem)
    return FEM, elemdic        
#
#  setAllElements
#
def setAllElements(fistrModel, FEM, unvElmsDict):
    "全elementを辞書として取得"
    allElms = {}
    for key in unvElmsDict.keys():
        fistrType = types[key]
        for elem in unvElmsDict[key]:
            elNo = elem.id
            nodes = elem.cntvt
            element = Element(nodes, fistrType)
            allElms[elNo] = element
    #fistrFEM classに保存
    fistrModel.allElms = allElms
    print("all Elements : number " + str(len(allElms)))
    return fistrModel
#
# setAllNodes
#
def setAllNodes(fistrModel, FEM):
    """ 全nodeを辞書として取得"""
    for nd in FEM.nodes:
        node = Node(nd.id, nd.coords)
        fistrModel.allNodes[nd.id] = node
    print("all Nodes : number " + str(len(fistrModel.allNodes)))
    return fistrModel

#
#  setElmNoToAllNodes
#
def setElmNoToAllNodes(fistrModel):
    """ allNodesにそのnodeが属するsolid要素Noを追加する"""
    print("checking relation between element and node...")
    elmDict = fistrModel.elmDict
    for elNo in fistrModel.allElms.keys():
        elType = fistrModel.allElms[elNo].type
        if elmDict[elType][0] == "solid":
            nodes = fistrModel.allElms[elNo].nodes
            for node in nodes:
                fistrModel.allNodes[node].elmNos.append(elNo)
    return fistrModel

#
#  setShellElements
#
def setShellElements(fistrModel, FEM, unvElmsDict, surGrps):
    """ シェル要素（731、741）を取得する"""
    #solid要素が定義されている場合、直ぐに戻る
    if isThereSolidElement(fistrModel, FEM) == True:
        return fistrModel

    #シェル要素を取得（solid要素が存在しない場合）
    print("getting shell elements...")
    #該当するshell要素を取得する
    shellElms = []
    for elType in unvElmsDict.keys():
        fistrType = types[elType]
        fType = ""
        if fistrType == "S341":
            fType = "731"           #三角形1次
        elif fistrType == "S342":
            fType = "732"           #三角形2次
        elif fistrType == "S361":
            fType = "741"           #四角形1次
        elif fistrType == "S362":
            fType = "742"           #四角形2次
        if fType != "" and len(unvElmsDict[elType]) > 0:
            #shell要素Noを取得
            elmNos = []
            for elem in unvElmsDict[elType]:
                elNo = elem.id
                elmNos.append(elNo)
            shellElms.append([fType, elmNos])
    #shellが存在しない場合は、戻る
    if len(shellElms) == 0:
        return fistrModel

    #定義済みのgrp名を取得
    for surGrp in surGrps:
        elms = surGrp.items
        for elNo in elms:
            fistrModel.allElms[elNo].egrp = surGrp.name
    #表面elementを取得
    for elType, elms in shellElms:
        #elType毎に要素を取得
        otherDict = {}
        grps = {}
        for elNo in elms:
            #要素が属するgrp名とnodeNoを取得
            grpName = fistrModel.allElms[elNo].egrp
            nodes = fistrModel.allElms[elNo].nodes
            if grpName == "":
                #grp名が定義されていない場合
                otherDict[elNo] = nodes                
            elif not grpName in grps.keys():
                #最初にgrp名が出現した時：辞書を作成
                elmDict = {elNo:nodes}
                grps[grpName] = elmDict
            else:
                #grp名の中に追加
                grps[grpName][elNo] = nodes
        #grpが定義されているelementを作成
        for grpName in grps.keys():
            header = "!ELEMENT, TYPE=" + elType + " , EGRP=" + grpName + ls
            fistrModel.elms.append([header, grps[grpName]])
        #未定義のelementを作成
        if len(otherDict.keys()) > 0:
            header = "!ELEMENT, TYPE=" + elType + ", EGRP=E" + elType + ls
            fistrModel.elms.append([header, otherDict])
    return fistrModel                

#
#  setSurfaceGrpOfShell
#
def setSurfaceGrpOfShell(fistrModel):
    """ shellのsurfaceGrpを取得して作成する"""
    addSurfaceGrps = []
    for header, data in fistrModel.elms:
        words = deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            fistrElType = getValName(header, "TYPE")
            sect = fistrModel.elmDict[fistrElType][0]
            if sect == "shell" or sect == "shell3":
                surName = "surface_" + getValName(header, "EGRP")
                surHeader = "!SGROUP, SGRP=" + surName + ls
                surData = []
                elmNos = list(data.keys())
                elmNos.sort()
                for elmNo in elmNos:
                    surData.append([elmNo, 1])
                addSurfaceGrps.append([surHeader, surData])
    fistrModel.surGrps += addSurfaceGrps
    return fistrModel

#
#  getOtherSurGroups
#
def getOtherSurGroups(fistrModel, FEM, surGrps, unvElmsDict):
    """ その他の表面要素groupを取得"""
    #全表面要素No
    allSurElms = {}
    for key in unvElmsDict.keys():
        fistrType = types[key]
        if fistrModel.elmDict[fistrType][0] == "face":
            for elm in unvElmsDict[key]:
                allSurElms[elm.id] = 0
    #定義済みの表面要素No
    for elset in surGrps:
        for elNo in elset.items:
            allSurElms[elNo] = 1
    #その他の表面groupを取得
    otherElms = []
    for elNo in allSurElms.keys():
        if allSurElms[elNo] == 0:
            otherElms.append(elNo)
    return otherElms
#
#  isInNodes
#
def isInNodes(nodeLine, nodes):
    """ 要素のnode列内に全nodeが含まれているかチェック
    
    Args:
        nodeLine list(int)  :要素のnode列のリスト
        nodes    list(int)  :チェック対象のnode(例:faceNodes)
    Returns:
        bool    : 含まれている場合、True が戻る  
    """
    ans = set(nodes).issubset(set(nodeLine))
    return ans
    # flag = 0
    # ans = False
    # for node in nodes:
    #     if node in nodeLine:
    #         flag += 1
    #     else:
    #         break
    # if flag == len(nodes):
    #     ans = True
    # return ans

#
#  getFaceNoOfElemen
#
def getFaceNoOfElement(elNodes, faceNodes):
    """ 要素のnode列とfaceのnode列から要素のfaceNoを取得する"""
    faceNo = 0
    faceNo = cm.fistr_elFace(faceNodes, elNodes)
    return faceNo
#
#  fineElementNo
#
def findElementNo(nodes, fistrModel):
    """ faceのnode列から、該当する要素とfaceNoを取得する。"""
    flag = 0
    i = 0
    #最初のnodeNoを含むelementsを取得
    elements = fistrModel.allNodes[nodes[0]].elmNos
    #elements内から該当する要素を探す
    while i < len(elements):
        elementNo = elements[i]
        nodeLine = fistrModel.allElms[elementNo].nodes
        if isInNodes(nodeLine, nodes) == True:
            faceNo = getFaceNoOfElement(nodeLine, nodes)
            flag = 1
            break
        i += 1
    if flag == 1:
        ans = [elementNo, faceNo]
    else:
        ans = ["", ""]
    return ans
#
#  getElementFaceNo
#
def getElementFaceNo(surGrps, otherElms, fistrModel):
    """ SGROUPのデータ（要素NoとfaceNo）とshell要素（761、781）を取得"""
    #要素type変換用の辞書
    convTemp2elmType = {
        "S341":"761",
        "S342":"762",
        "S361":"781"
        }
    elementFaceNoSet = []       #SGROUPのセット
    surElementSet = []          #shellElement
    #全表面要素grpを検索
    for elset in surGrps:
        name = elset.name
        data = []
        surData = {}
        #表面要素grpから検索
        for elmNo in elset.items:
            nodes = fistrModel.allElms[elmNo].nodes
            #faceのnode列から、該当する要素とfaceNoを取得
            [elementNo, faceNo] = findElementNo(nodes, fistrModel)
            if elementNo != "":
                #該当する要素にnode列を保存（SGROUPとして取得）
                data += [[elementNo] + [faceNo]]
            else:
                #要素が見つからない場合（shellとして取得）
                surData[elmNo] = nodes
        if len(data) > 0:
            #SGROUPとして取得
            print(name, end=" ")
            data.sort()
            elementFaceNoSet += [[name] + [data]]
        if len(surData) > 0:
            #shellとして取得
            elmNo = list(surData.keys())[0]
            elType = fistrModel.allElms[elmNo].type
            elType = convTemp2elmType[elType]
            surElementSet += [[[name, elType], surData]]
    #その他（otherElms）を追加する
    if len(otherElms) > 0:
        #未定義のSGROUPを取得
        name = "otherS"
        data = []
        for elmNo in otherElms:
            nodes = fistrModel.allElms[elmNo].nodes
            [elementNo, faceNo] = findElementNo(nodes, fistrModel)
            if elementNo != "":
                data += [[elementNo] + [faceNo]]
        if len(data) > 0:
            print(name, end=" ")
            data.sort()
            elementFaceNoSet += [[name] + [data]]
    #solid要素有無チェック
    if len(elementFaceNoSet) == 0:
        surElementSet = []          #solid要素が無い場合、[]にする
    return [elementFaceNoSet, surElementSet]
#
#  addDummyShellNodes
#
def addDummyShellNodes(surElement, fistrModel):
    """ shell要素（761、781）にdummy節点を追加する"""
    header = surElement[0]
    elmsDict = surElement[1]
    maxNodeNo = max(fistrModel.allNodes.keys())
    addNodeNo = maxNodeNo + 1
    egrp = header[0]
    dummyNodeHeader = "!NODE, NGRP=dummy" + egrp + ls
    dummyNodeDict = {}
    for elNo in elmsDict.keys():
        addNodes = []
        for nodeNo in elmsDict[elNo]:
            #dummyNodeを設定
            dummyNode = fistrModel.allNodes[nodeNo].dummyNode
            if dummyNode < 0:
                #新規にdummyNodeを作成し、設定
                coords = fistrModel.allNodes[nodeNo].coords
                dummyNodeDict[addNodeNo] = coords
                addNodes.append(addNodeNo)
                fistrModel.allNodes[nodeNo].dummyNode = addNodeNo
                fistrModel.allNodes[addNodeNo] = Node(addNodeNo, coords)
                addNodeNo += 1
            else:
                #設定済のdummyNodeを設定
                addNodes.append(dummyNode)
        elmsDict[elNo] += addNodes
    newSurElement = [header, elmsDict]
    fistrModel.nodes.append([dummyNodeHeader, dummyNodeDict])
    return newSurElement, fistrModel

#
#  getSolidSurfaceGrps
#
def getSolidSurfaceGrps(fistrModel, FEM):
    """ elsetをEGROUPとSGROUPに分けて取得する"""
    surGrps = []
    volGrps = []
    for elset in FEM.elemsets:
        elmNo = elset.items[0]
        fistrType = fistrModel.allElms[elmNo].type
        if fistrModel.elmDict[fistrType][0] == "solid":
            volGrps.append(elset)
        elif fistrModel.elmDict[fistrType][0] == "face":
            surGrps.append(elset)
    return volGrps, surGrps

#
#  setSurfaceElementGroup
#
def setSurfaceElementGroup(fistrModel, FEM, unvElmsDict, surGrps):
    """ 表面group取得、shell要素（761、781）と solid要素groupを取得"""
    print("getting shell elements with solid...", end=" ")
    #elsetをEGROUPとSGROUPに分ける
    surGrps = []
    volGrps = []
    for elset in FEM.elemsets:
        elmNo = elset.items[0]
        fistrType = fistrModel.allElms[elmNo].type
        if fistrModel.elmDict[fistrType][0] == "solid":
            volGrps.append(elset)
        elif fistrModel.elmDict[fistrType][0] == "face":
            surGrps.append(elset)
    #その他の表面要素を取得
    otherElms = getOtherSurGroups(fistrModel, FEM, surGrps, unvElmsDict)
    #surGrps（SGROUP）の要素とfaceNoを取得（shell要素も取得）
    [surfaceGroup, surElementSet] = getElementFaceNo(surGrps, otherElms, fistrModel)
    print()
    print("creating SGROUP(surface)...", end=" ")
    if len(surfaceGroup) > 0:
        #SGROUPを作成
        for surGrp in surfaceGroup:
            name = surGrp[0]
            header = "!SGROUP, SGRP=" + name + ls
            surGrp[0] = header
            fistrModel.surGrps.append(surGrp)
            print(name, end=" ")
    print()
    if len(surElementSet) > 0:
        #shell（761、781）を作成、dummyNodeを追加
        for surElement in surElementSet:
            #dummyNodeを追加
            surElement, fistrModel = addDummyShellNodes(surElement, fistrModel)
            #headerを書き換え
            grpName = surElement[0][0]
            elType = surElement[0][1]
            header = "!ELEMENT, TYPE=" + elType + ", EGRP=" + grpName + ls
            element = [header, surElement[1]]
            fistrModel.elms.append(element)
            print("  ELEMENT : " + grpName + " type " + elType + " num " + str(len(surElement[1])))
    return fistrModel
#
#  isBeamInFace
#
def isBeamInFace(nodes, nodeElmDict, fistrModel):
    """ beamのnode列がface要素に含まれているか"""
    ans = False
    elms = []
    for nodeNo in nodes:
        if nodeNo in nodeElmDict.keys():
            elms += nodeElmDict[nodeNo]
        else:
            elms = []
            break
    for elNo in elms:
        nodeLine = fistrModel.allElms[elNo].nodes
        if isInNodes(nodeLine, nodes) == True:
            ans = True
            break
    return ans
#
#  addDummyBeamNodes
#
def addDummyBeamNodes(beamElm, fistrModel):
    """ beam要素（641）にdummy節点を追加"""
    header = beamElm[0]
    elmsDict = beamElm[1]
    maxNodeNo = max(fistrModel.allNodes.keys())
    addNodeNo = maxNodeNo + 1
    egrp = getValName(header, "EGRP")
    dummyNodeHeader = "!NODE, NGRP=dummy" + egrp + ls
    dummyNodeDict = {}
    for elNo in elmsDict.keys():
        addNodes = []
        for nodeNo in elmsDict[elNo]:
            #dummyNodeを設定
            dummyNode = fistrModel.allNodes[nodeNo].dummyNode
            if dummyNode < 0:
                #新規にdummyNodeを作成し、設定
                coords = fistrModel.allNodes[nodeNo].coords
                dummyNodeDict[addNodeNo] = coords
                addNodes.append(addNodeNo)
                fistrModel.allNodes[nodeNo].dummyNode = addNodeNo
                fistrModel.allNodes[addNodeNo] = Node(addNodeNo, coords)
                addNodeNo += 1
            else:
                #設定済のdummynodeを設定
                addNodes.append(dummyNode)
        elmsDict[elNo] += addNodes
    beamElm = [header, elmsDict]
    fistrModel.nodes.append([dummyNodeHeader, dummyNodeDict])
    return beamElm, fistrModel
#
#  setOnlyBeamElement
#
def setOnlyBeamElement(fistrModel, FEM, unvElmsDict):
    """ beam要素(611)の取得　beamが単独で存在するものを取得
    （faceに含まれるbeamは取得しない）"""
    #全beam要素を取得
    tempBeams = []
    faceElms = []
    for unvType in unvElmsDict.keys():
        fistrType = types[unvType]
        if fistrModel.elmDict[fistrType][0][:len("beam")] == "beam":
            tempBeams.append(unvElmsDict[unvType])
        elif fistrModel.elmDict[fistrType][0] == "face":
            faceElms.append(unvElmsDict[unvType])
    if len(tempBeams) == 0:
        #beamが存在しない場合、戻る
        return fistrModel

    #nodeElmDictを作成
    print("getting beam elements...")
    nodeElmDict = {}
    for elms in faceElms:
        for element in elms:
            elNo = element.id
            nodes = element.cntvt
            for nodeNo in nodes:
                if not (nodeNo in nodeElmDict.keys()):
                    nodeElmDict[nodeNo] = [elNo]
                else:
                    if not (elNo in nodeElmDict[nodeNo]):
                        nodeElmDict[nodeNo].append(elNo)
    #全beam要素を取得
    allBeams = {}
    for elms in tempBeams:
        for element in elms:
            elNo = element.id
            nodes = element.cntvt
            if isBeamInFace(nodes, nodeElmDict, fistrModel) == False:
                allBeams[elNo] = 0
    #beamのelsetを取得
    beamGrps = []
    for elset in FEM.elemsets:
        elNo = elset.items[0]
        fistrType = fistrModel.allElms[elNo].type
        if fistrModel.elmDict[fistrType][0][:len("beam")] == "beam":
            beamGrps.append(elset)
    #定義済のbeamを取得
    beamElms = []
    for elset in beamGrps:
        beamDict = {}
        name = elset.name
        elNo = elset.items[0]
        elType = fistrModel.allElms[elNo].type
        for elNo in elset.items:
            allBeams[elNo] = 1
            beamDict[elNo] = fistrModel.allElms[elNo].nodes
        header = "!ELEMENT, TYPE=" + elType + ", EGRP=" + name + ls
        beamElms.append([header, beamDict])
        print("  Element : " + name + " type " + elType + " num " + str(len(beamDict)))
    #その他のbeam要素を取得
    otherBeams = []
    for elNo in allBeams.keys():
        if allBeams[elNo] == 0:
            otherBeams.append(elNo)
    if len(otherBeams) > 0:
        beamDict = {}
        elType = fistrModel.allElms[otherBeams[0]].type
        for elNo in otherBeams:
            beamDict[elNo] = fistrModel.allElms[elNo].nodes
        header = "!ELEMENT, TYPE=" + elType + ", EGRP=" + "E" + elType + ls
        beamElms.append([header, beamDict])
    #solid有無を確認
    if isThereSolidElement(fistrModel, FEM) == True:
        #要素typeを611→641に書き換え、dummyNodeを追加
        for i in range(len(beamElms)):
            beamElms[i], fistrModel = addDummyBeamNodes(beamElms[i], fistrModel)
            header = beamElms[i][0]
            header = setValName(header, "TYPE=641")
            beamElms[i][0] = header
    fistrModel.elms += beamElms
    return fistrModel    
#
#  setSolidElement
#
def setSolidElement(fistrModel, FEM, unvElmsDict, solidGrps):
    """ solid要素（group）を取得"""
    print("getting element groups...", end=" ")
    #solidGroupの要素をセット
    elmGrps = []
    for egrp in solidGrps:
        egrpName = egrp.name
        elms = egrp.items
        elType = fistrModel.allElms[elms[0]].type
        elmGrp = [[egrpName, elType], {}]
        elmGrps.append(elmGrp)
        print(egrpName, end=" ")
        for elNo in elms:
            fistrModel.allElms[elNo].egrp = egrpName
            nodes = fistrModel.allElms[elNo].nodes
            elType = fistrModel.allElms[elNo].type
            flag = 0
            for i in range(len(elmGrps)):
                grp = elmGrps[i]
                if grp[0][0] == egrpName and grp[0][1] == elType:
                    elmGrps[i][1][elNo] = nodes
                    flag = 1
                    break
            if flag == 0:
                elmGrp = [[egrpName, elType], {elNo:nodes}]
                elmGrps.append(elmGrp)
    #その他のsolid要素を作成
    otherGrps = []
    for unvType in unvElmsDict.keys():
        fistrType = types[unvType]
        if fistrModel.elmDict[fistrType][0] == "solid":
            elms = unvElmsDict[unvType]
            for element in elms:
                elNo = element.id
                egrpName = fistrModel.allElms[elNo].egrp
                if egrpName == "":
                    elType = fistrModel.allElms[elNo].type
                    nodes = fistrModel.allElms[elNo].nodes
                    flag = 0
                    for i in range(len(otherGrps)):
                        grp = otherGrps[i]
                        if grp[0][1] == elType:
                            otherGrps[i][1][elNo] = nodes
                            flag = 1
                    if flag == 0:
                        otherGrp = [["E"+elType, elType], {elNo:nodes}]
                        otherGrps.append(otherGrp)
                        print("E"+elType, end=" ") 
    elmGrps += otherGrps
    print()
    #headerを作成
    for i in range(len(elmGrps)):
        egrp = elmGrps[i][0][0]
        elType = elmGrps[i][0][1]
        nElms = len(elmGrps[i][1].keys())
        header = "!ELEMENT, TYPE=" + elType + ", EGRP=" + egrp + ls
        elmGrps[i][0] = header
        print("  Element : " + egrp + " TYPE " + elType + " num " + str(nElms))
    #結果を保存
    fistrModel.elms += elmGrps
    return fistrModel
#
#  setNodeGroup
#
def setNodeGroup(fistrModel, FEM):
    """ 節点groupを取得"""
    print("getting node groups...", end=" ")
    nodeGrps = []
    for nodeset in FEM.nodesets:
        name = nodeset.name
        nodes = nodeset.items
        nodes.sort()
        header = "!NGROUP, NGRP=" + name + ls
        nodeGrps.append([header, nodes])
        print(name, end=" ")
    fistrModel.nodeGrps = nodeGrps
    print()
    return fistrModel

#
#  setDummyNodeGroup
#
def setDummyNodeGroup(fistrModel):
    """ dummy節点のgroupを作成"""
    for nodeGrp in fistrModel.nodeGrps:
        nodes = nodeGrp[1]
        dummyGrp = []
        for nodeNo in nodes:
            #節点に対応するdummy節点が有る場合は、dummyGroupを取得する
            dummyNodeNo = fistrModel.allNodes[nodeNo].dummyNode
            if dummyNodeNo != -1:
                dummyGrp.append(dummyNodeNo)
        if len(dummyGrp) > 0 and len(dummyGrp) == len(nodes):
            header = nodeGrp[0]
            ngrp = getValName(header, "NGRP")
            header = setValName(header, "NGRP=dummy_"+ngrp)
            fistrModel.nodeGrps.append([header, dummyGrp])
    return fistrModel

#
#  setNodeGroups
#
def setNodeGroups(fistrModel, FEM):
    """ 節点groupを作成する"""
    nodeGrps = []
    for group in FEM.nodesets:
        print("Node set: %s"%(group.name))
        header = "!NGROUP,NGRP=%s" % group.name
        nodeNos = group.items
        nodeGrps.append([header, nodeNos])
    fistrModel.nodeGrps = nodeGrps
#
#  convNum2str
#
def convNum2str(num):
    """ 数値を文字列に変換する。指数表記で小数点が無い場合は、少数点を追加する"""
    word = str(num)
    if word.find(".") >= 0:
        return word
    n = word.find("e")
    if n >= 0:
        word = word[:n] + ".0" + word[n:]
    return word
#
#  writeHeader
#
def writeHeader(unvFile, fil):
    """ header行を書き込む"""
    cont = "!HEADER" + ls
    cont += "  generated by unv2fistr.py" + ls
    cont += "    from " + unvFile + ls
    fil.write(cont)
#
#  writeNode
#
def writeNode(fistrModel, fil):
    """ NODE行を作成する"""
    nodeConts = fistrModel.nodes
    for nodeCont in nodeConts:
        header = nodeCont[0]
        fil.write(header)
        lines = []
        nodeDict = nodeCont[1]
        for nodeNo in nodeDict.keys():
            coords = nodeDict[nodeNo]
            words = [str(nodeNo)]
            for coord in coords:
                words.append(convNum2str(coord))
            lines += [", ".join(words)]
            #line = ", ".join(words) + ls
            #fil.write(line)
        cont = ls.join(lines) + ls
        ngrp = getValName(header, "NGRP")
        if ngrp == "":
            ngrpOut = ""
        else:
            ngrpOut = " name " + ngrp
        print("  Node : num " + str(len(nodeDict)) + ngrpOut)
        fil.write(cont)
#
#  writeElement
#
def writeElement(fistrModel, fil):
    """ ELEMENT行を書き込む"""
    #
    #  writeElementCont
    def writeElmCont(header, elmsDict, fil):
        """ elementGroupを書き込む"""
        fil.write(header)
        lines = []
        for elNo in elmsDict.keys():
            nodes = elmsDict[elNo]
            words = list(map(str, ([elNo]+nodes)))
            line = ", ".join(words)
            lines += [line]
        cont = ls.join(lines) + ls
        fil.write(cont)

    #要素データを取得
    elmConts = fistrModel.elms
    sections = []
    #solid要素を出力
    egrpNames = []
    for header, elmsDict in elmConts:
        elType = getValName(header, "TYPE")
        elSect = fistrModel.elmDict[elType][0]
        if elSect[:len("solid")] == "solid":
            writeElmCont(header, elmsDict, fil)
            egrp = getValName(header, "EGRP")
            if not egrp in egrpNames:
                egrpNames.append(egrp)
                footer = "!SECTION, TYPE=SOLID, EGRP=" + egrp + ls
                #fil.write(footer)
                sections.append(footer)
            print("  Element : type " + elType + " num " + str(len(elmsDict)) + " name " + egrp)
    #beam要素を出力
    for header, elmsDict in elmConts:
        elType = getValName(header, "TYPE")
        elSect = fistrModel.elmDict[elType][0]
        if elSect[:len("beam")] == "beam":
            writeElmCont(header, elmsDict, fil)
            egrp = getValName(header, "EGRP")
            if not egrp in egrpNames:
                egrpNames.append(egrp)
                footer  = "!SECTION, TYPE=BEAM, EGRP=" + egrp + ls
                footer += "0.0, 0.0, -1.0, 1.0, 1.0, 1.0, 1.0" + ls
                #fil.write(footer)
                sections.append(footer)
            print("  Element : type " + elType + " num " + str(len(elmsDict)) + " name " + egrp)
    #shell要素を出力
    for header, elmsDict in elmConts:
        elType = getValName(header, "TYPE")
        elSect = fistrModel.elmDict[elType][0]
        if elSect[:len("shell")] == "shell":
            writeElmCont(header, elmsDict, fil)
            egrp = getValName(header, "EGRP")
            if not egrp in egrpNames:
                egrpNames.append(egrp)
                footer  = "!SECTION, TYPE=SHELL, EGRP=" + egrp + ls
                footer += "1.0, 5" + ls
                #fil.write(footer)
                sections.append(footer)
            print("  Element : type " + elType + " num " + str(len(elmsDict)) + " name " + egrp)
    #sectionsを書き込み
    for section in sections:
        fil.write(section)

#
#  writeSurfaceGroup
#
def writeSurfaceGroup(fistrModel, fil):
    """ SGROUPの書込"""
    surGrps = fistrModel.surGrps
    for surGrp in surGrps:
        header = surGrp[0]
        fil.write(header)
        surs = surGrp[1]
        lines = []
        #for sur in surs:
        for i in range(0, len(surs), 5):
            data = []
            sursw = surs[i:i+5]
            for sur in sursw:
                data += [sur[0], sur[1]]
            #line = str(sur[0]) + ", " + str(sur[1]) + ls
            #fil.write(line)
            #line = ", ".join(list(map(str, sur)))
            line = ", ".join(list(map(str, data)))
            lines += [line]
        cont = ls.join(lines) + ls
        fil.write(cont)
        sgrp = getValName(header, "SGRP")
        print("  SGROUP : num " + str(len(lines)) + " name " + sgrp)
#
#  writeNodeGroup
#
def writeNodeGroup(fistrModel, fil):
    """ NGROUPの書込"""
    nodeGrps = fistrModel.nodeGrps
    for nodeGrp in nodeGrps:
        header = nodeGrp[0]
        fil.write(header)
        nodes = nodeGrp[1]
        count = 8
        nums = []
        lines = []
        for i in range(0, len(nodes), count):
            nums = nodes[i:i+count]
            words = list(map(str, nums))
            #line = ", ".join(words) + ls
            #fil.write(line)
            line = ", ".join(words)
            lines += [line]
        cont = ls.join(lines) + ls
        fil.write(cont)
        ngrp = getValName(header, "NGRP")
        print("  NGROUP : num " + str(len(nodes)) + " name " + ngrp)
#
#  writeEnd
#
def writeEnd(fil):
    """ ENDの書込"""
    fil.write("!END" + ls)

#
#  writeFistrModel
#
def writeFistrModel(fistrModel, unvFile, writeFile):
    """ fistrFEM class に保存されているデータをfileに書き込む"""
    print("writing...")
    fil = open(writeFile, "w", encoding="utf-8")
    writeHeader(unvFile, fil)
    writeNode(fistrModel, fil)
    writeElement(fistrModel, fil)
    writeSurfaceGroup(fistrModel, fil)
    writeNodeGroup(fistrModel, fil)
    writeEnd(fil)
    print("converted unv to FrontIstr format!")
    fil.close()

#
#  remakeSolidShellBeamModel
#
def remakeSolidShellBeamModel(fileName):
    """ dummy節点にEQUATIONを追加する"""
    comm = "python3 " + pyFistr.binAppPath + "remakeSolidShellBeamModel.py " + fileName
    #os.system(comm)
    proc = subprocess.run(comm, shell=True)
#
#  convertUnv2Fistr
#
def convertUnv2Fistr(unvFile, prefix):
    """ unv形式のメッシュをfistr形式に変換する"""
    print("reading " + unvFile + " file...")
    if glob.glob(unvFile) == 0:
        print("「" + unvFile + "」が存在しません。")
        exit()
    
    #unvFileの読み込み
    UNV = unv.UNVParser(unvFile)        #UNV objectを作成
    FEM = UNV.parse()                   #読み込み
    #FrontISTRのデータ構造を定義
    fistrModel = fistrFEM()
    #不要なelementを削除し、要素の節点並びを修正
    FEM, unvElmsDict = remakeElements(FEM)

    #全node、全elementを取得
    #fistrModel = setAllNodes(fistrModel, FEM)
    fistrModel.allNodes = FEM.nodes
    print("all nodes : number " + str(len(fistrModel.allNodes)))
    fistrModel = setAllElements(fistrModel, FEM, unvElmsDict)

    #allNodesにnodeNoが属するelmNoを取得)
    fistrModel = setElmNoToAllNodes(fistrModel)
    #NODEを取得
    fistrModel = setNodes(fistrModel)

    #solidGrps, surGrpsを取得
    solidGrps, surGrps = getSolidSurfaceGrps(fistrModel, FEM)

    #表面group取得、shell要素（761、781）取得
    fistrModel = setSurfaceElementGroup(fistrModel, FEM, unvElmsDict, surGrps)
    #シェル要素をチェックし、シェル要素（731、741）があれば取得
    fistrModel = setShellElements(fistrModel, FEM, unvElmsDict, surGrps)
    #shell要素のfaceGroupを作成
    fistrModel = setSurfaceGrpOfShell(fistrModel)

    #beam要素（611）を取得
    fistrModel = setOnlyBeamElement(fistrModel, FEM, unvElmsDict)

    #solid要素を取得
    fistrModel = setSolidElement(fistrModel, FEM, unvElmsDict, solidGrps)
    
    #nodeGroupを取得
    fistrModel = setNodeGroup(fistrModel, FEM)
    
    #dummyNodeGroupを作成
    fistrModel = setDummyNodeGroup(fistrModel)
    #書き出し
    #writeFile = prefix + "_fistr.msh"
    writeFile = prefix
    writeFistrModel(fistrModel, unvFile, writeFile)

    #EQUATIONを追加
    remakeSolidShellBeamModel(writeFile)

if __name__=='__main__':
    helpmsg="""
    ----------- unv2fistrEx.py --------------------------------------
    unv形式のメッシュファイルをFrontISTR形式に変換する。
    <使い方>
    unv2fistrEx.py <unvFile> [fistrFile]
      unvFile       unv形式のメッシュファイル
      fistrFile     FrontISTR形式のメッシュファイル
                    省略時は「FistrModel.msh」に設定される
    <例>
    unv2fistrEx.py mesh_1.unv           mesh_1.mshが出来上がる
    unv2fistrEx.py mesh_1.unv test      test.mshが出来上がる
    """

    #log表示
    import logFileCreater
    logFileCreater.log()

    if len(sys.argv) >= 3:
        unvfile = sys.argv[1]
        prefix = sys.argv[2]
        if prefix.split(".")[-1] != "msh":
            #prefix = ".".join(prefix.split(".")[:-1])
            prefix = prefix + ".msh"
    elif len(sys.argv) == 2:
        unvfile = sys.argv[1]
        prefix = ".".join(unvfile.split(".")[:-1]) + ".msh"
    else:
        print(helpmsg)
        exit()
    convertUnv2Fistr(unvfile, prefix)
