#!/usr/bin/python3
#  coding: utf-8
#
#   setBeamPinNodes.py
#
#       指定されたNGRPの節点をpin構造に変更する。
#       （変位のみ伝達し、モーメントは伝達させない。）
#       611, 641要素に対応する。
#
#   20/01/23    新規作成
#      02/11    addSoftSpringToAddedPinNodes,addSoftSpringToDummyAddedPinNodes:
#               弱いバネのバネ定数を1.0→1.0e5に変更
#      02/18    addSoftSpringToAddedPinNodes,addSoftSpringToDummyAddedPinNodes:
#               弱いバネのバネ定数を1.0e5→1.0に戻す
#   22/03/15    国際化
#   24/07/07    openにencoding="utf-8"を追加
#

import sys, os
import geometryFunc as geo
import pyFistr

beamElms = ["611", "641"]


#
#  isBeamElements(elmType)
def isBeamElements(elmType):
    """ 要素Typeがbeamかどうかチェック"""
    if elmType in beamElms:
        return True
    else:
        return False

#
#  getOnlyBeamElements
def getOnlyBeamElements(nodeNos, nodeDict, elementDict):
    """ beam同士がつながっているnodeを取得する"""
    nodes = []
    for nodeNo in nodeNos:
        elms = nodeDict[nodeNo][1]
        counts = []
        for elmNo in elms:
            elmType = elementDict[elmNo][1]
            if isBeamElements(elmType) == True:
                counts.append(elmNo)
        if len(counts) >= 2:
            nodes.append(nodeNo)
    return nodes

#
#  getElmTypes
def getElmTypes(nodeNos, nodeDict, elementDict):
    """ nodeNosが属しているelmTypeを確認する"""
    elmTypesSet = set([])
    for nodeNo in nodeNos:
        elms = nodeDict[nodeNo][1]
        for elmNo in elms:
            elmType = elementDict[elmNo][1]
            elmTypesSet.add(elmType)
    return list(elmTypesSet)

#
#  getPinNodeNos
def getPinNodeNos(ngrps, meshHeaderData):
    """ pinNodeに変更するnodeNoを取得する。"""
    nodeNos = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp in ngrps:
                nodes = list(map(lambda x: x[0], data))
                nodeNos += nodes
        elif words[0] == "!NGROUP":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp in ngrps:
                nodeNos += data
    return nodeNos

#
#  createNodeElementDict
def createNodeElementDict(meshHeaderData):
    """ nodeDict, elementDictを作成する"""
    nodeDict = {}
    elementDict = {}
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            data = headerData[1]
            for node in data:
                nodeNo = node[0]
                loc = node[1:]
                nodeDict[nodeNo] = [loc, []]
        elif words[0] == "!ELEMENT":
            data = headerData[1]
            elmType = pyFistr.getValName(header, "TYPE")
            for elm in data:
                elmNo = elm[0]
                nodes = elm[1:]
                elementDict[elmNo] = [nodes, elmType]
                for nodeNo in nodes:
                    nodeDict[nodeNo][1].append(elmNo)
    return nodeDict, elementDict

#
#  getPairNodesElements
def getPairNodesElements(nodeNos, nodeDict, elementDict):
    """ baseのnodeNoと同じ座標のnodeを追加する。"""
    pairNodeNos = []
    pairElmNos = []
    maxNodeNo = max(nodeDict.keys())
    for nodeNo in nodeNos:
        elms = nodeDict[nodeNo][1]
        beamElms = []
        for elmNo in elms:
            elmType = elementDict[elmNo][1]
            if isBeamElements(elmType) == True:
                beamElms.append(elmNo)
        pairElmNos.append(beamElms)
        pairNodes = [nodeNo]
        for i in range(1, len(beamElms)):
            maxNodeNo += 1
            newNodeNo = maxNodeNo
            pairNodes.append(newNodeNo)
        pairNodeNos.append(pairNodes)
    return pairNodeNos, pairElmNos

#
#  getDummyPairNodesElements
def getDummyPairNodesElements(nodeNos, nodeDict, elementDict):
    """ baseのnodeNoと同じ座標のnodeを追加する。"""
    
    def getDummyNode(elmNo, nodeNo, elementDict):
        nodes = elementDict[elmNo][0]
        if nodes[0] == nodeNo:
            dummyNodeNo = nodes[2]
        else:
            dummyNodeNo = nodes[3]
        return dummyNodeNo
    
    pairNodeNos = []
    pairElmNos = []
    maxNodeNo = max(nodeDict.keys())
    for nodeNo in nodeNos:
        elms = nodeDict[nodeNo][1]
        beamElms = []
        for elmNo in elms:
            elmType = elementDict[elmNo][1]
            if isBeamElements(elmType) == True:
                beamElms.append(elmNo)
        pairElmNos.append(beamElms)
        dummyNodeNo = getDummyNode(elms[0], nodeNo, elementDict)
        pairNodes = [dummyNodeNo]
        for i in range(1, len(beamElms)):
            maxNodeNo += 1
            newNodeNo = maxNodeNo
            pairNodes.append(newNodeNo)
        pairNodeNos.append(pairNodes)
    return pairNodeNos, pairElmNos


#
#  addNewNodes
def addNewNodes(pairNodes, meshHeaderData, nodeDict):
    """ newNodeを追加する"""
    header = "!NODE, NGRP=addedPinNodes\n"
    lines = []
    egrps = []
    for pair in pairNodes:
        baseNodeNo = pair[0]
        egrps.append(baseNodeNo)
        newNodeNos = pair[1:]
        loc = nodeDict[baseNodeNo][0]
        for newNodeNo in newNodeNos:
            line = [newNodeNo] + loc
            lines.append(line)
    newHeaderData = [[header, lines]]
    header = "!NGROUP, NGRP=pinNodes\n"
    newHeaderData += [[header, egrps]]
    #挿入場所を探す(!ELEMENTの前に挿入)
    flag = 0
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            ngrpName = pyFistr.getValName(header, "NGRP")
            if ngrpName == "addedPinNodes":
                meshHeaderData[i][1] += lines
                flag = 1
                break
    if flag == 0:
        for i in range(len(meshHeaderData)):
            headerData = meshHeaderData[i]
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                si = i
                break
        meshHeaderData = meshHeaderData[:si] + newHeaderData + meshHeaderData[si:]
    return meshHeaderData

#
#  addDummyNewNodes
def addDummyNewNodes(pairNodes, meshHeaderData, nodeDict):
    """ newNodeを追加する"""
    header = "!NODE, NGRP=dummy_addedPinNodes\n"
    lines = []
    baseNodes = []
    for pair in pairNodes:
        baseNodeNo = pair[0]
        baseNodes.append(baseNodeNo)
        newNodeNos = pair[1:]
        loc = nodeDict[baseNodeNo][0]
        for newNodeNo in newNodeNos:
            line = [newNodeNo] + loc
            lines.append(line)
    newHeaderData = [[header, lines]]
    header = "!NGROUP, NGRP=dummy_pinNodes\n"
    newHeaderData += [[header, baseNodes]]
    #挿入場所を探す(!ELEMENTの前に挿入)
    flag = 0
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            ngrpName = pyFistr.getValName(header, "NGRP")
            if ngrpName == "dummyaddedPinNodes":
                meshHeaderData[i][1] += lines
                flag = 1
                break
    if flag == 0:
        for i in range(len(meshHeaderData)):
            headerData = meshHeaderData[i]
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                si = i
                break
        meshHeaderData = meshHeaderData[:si] + newHeaderData + meshHeaderData[si:]
    return meshHeaderData

#
#  replaceNodeNo
def replaceNodeNo(pairNodes, pairElms, meshHeaderData):
    """ 要素内のnodeNoをnewNodeNoに置き換える"""

    def replaceElmNodeNo(elmNo, nodeNo, newNodeNo, meshHeaderData):
        """ elmNo内のnodeNoをnewNodeNoに置き換える"""
        flag = 0
        for i in range(len(meshHeaderData)):
            headerData = meshHeaderData[i]
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elmType = pyFistr.getValName(header, "TYPE")
                if isBeamElements(elmType) == True:
                    data = headerData[1]
                    for ii in range(len(data)):
                        line = data[ii]
                        if line[0] == elmNo:
                            #置き換え
                            nodes = line[1:]
                            idx = nodes.index(nodeNo)
                            nodes[idx] = newNodeNo
                            data[ii] = [line[0]] + nodes
                            flag = 1
                            break
            if flag == 1:
                break
        return meshHeaderData

    for i in range(len(pairNodes)):
        baseNodeNo = pairNodes[i][0]
        newNodeNos = pairNodes[i][1:]
        baseElmNo = pairElms[i][0]
        remakeElmNos = pairElms[i][1:]
        for ii in range(len(remakeElmNos)):
            elmNo = remakeElmNos[ii]
            newNodeNo = newNodeNos[ii]
            meshHeaderData = replaceElmNodeNo(elmNo, baseNodeNo, newNodeNo, meshHeaderData)
    return meshHeaderData

#
#  replaceDummyNodeNo
def replaceDummyNodeNo(pairNodes, pairElms, meshHeaderData):
    """ 要素内のnodeNoをnewNodeNoに置き換える"""

    def replaceElmNodeNo(elmNo, nodeNo, newNodeNo, meshHeaderData):
        """ elmNo内のnodeNoをnewNodeNoに置き換える"""
        flag = 0
        for i in range(len(meshHeaderData)):
            headerData = meshHeaderData[i]
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elmType = pyFistr.getValName(header, "TYPE")
                if isBeamElements(elmType) == True:
                    data = headerData[1]
                    for ii in range(len(data)):
                        line = data[ii]
                        if line[0] == elmNo:
                            #置き換え
                            nodes = line[1:]
                            idx = nodes.index(nodeNo)
                            #nodes[idx+2] = newNodeNo
                            nodes[idx] = newNodeNo
                            data[ii] = [line[0]] + nodes
                            flag = 1
                            break
            if flag == 1:
                break
        return meshHeaderData

    for i in range(len(pairNodes)):
        baseNodeNo = pairNodes[i][0]
        newNodeNos = pairNodes[i][1:]
        baseElmNo = pairElms[i][0]
        remakeElmNos = pairElms[i][1:]
        for ii in range(len(remakeElmNos)):
            elmNo = remakeElmNos[ii]
            newNodeNo = newNodeNos[ii]
            meshHeaderData = replaceElmNodeNo(elmNo, baseNodeNo, newNodeNo, meshHeaderData)
    return meshHeaderData

#
#  createEquation
def createEquation(ngrps, pairNodes, pairElms, meshHeaderData, nodeDict, elementDict):
    """ equationを作成する"""

    def createEqLines(lineVals):
        newVals = []
        for i in range(0, len(lineVals), 3):
            coeff = lineVals[i+2]
            #係数が1.0e-8以下は、無視する。
            if abs(coeff) > 1.0e-8:
                newVals += lineVals[i:i+3]
        n = len(newVals) // 3
        line = str(n) + ", 0.0\n"
        line1 = ", ".join(list(map(str, newVals))) + "\n"
        return [line, line1]

    addHeaderData = []
    header = "#pinNodes, " + ", ".join(ngrps) + "\n"
    addHeaderData += [[header, []]]
    header = "!EQUATION\n"
    lines = []
    for i in range(len(pairNodes)):
        nodeNo = pairNodes[i][0]
        for ii in range(1, len(pairNodes[i])):
            #変位を拘束
            newNodeNo = pairNodes[i][ii]
            lineData = [newNodeNo, 1, 1, nodeNo, 1, -1]
            addLines = createEqLines(lineData)
            lines += addLines
            lineData = [newNodeNo, 2, 1, nodeNo, 2, -1]
            addLines = createEqLines(lineData)
            lines += addLines
            lineData = [newNodeNo, 3, 1, nodeNo, 3, -1]
            addLines = createEqLines(lineData)
            lines += addLines
    headerData = [header, lines]
    addHeaderData.append(headerData)
    #挿入場所は、!ENDの前
    si = -1
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!END":
            si = i
            break
    if si == -1:
        meshHeaderData = meshHeaderData + addHeaderData + [["!END", []]]
    else:
        meshHeaderData = meshHeaderData[:si] + addHeaderData + meshHeaderData[si:]
    return meshHeaderData

#
#  createPinNodesCard
def createPinNodesCard(ngrps, meshHeaderData):
    """ #pinNodesを作成する"""
    header = "#pinNodes, " + ", ".join(ngrps) + "\n"
    addHeaderData = [[header, []]]
    #挿入場所は、「!END」の前
    si = -1
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!END":
            si = i
            break
    if si == -1:
        meshHeaderData = meshHeaderData + addHeaderData + [["!END", []]]
    else:
        meshHeaderData = meshHeaderData[:si] + addHeaderData + meshHeaderData[si:]
    return meshHeaderData

#
#  addSoftSpringToAddedPinNodes
def addSoftSpringToAddedPinNodes(lines):
    """ 弱いバネをaddedPinNodesに追加する"""
    #addLinesを作成
    addLines = []
    addLines.append("!SPRING, GRPID=1\n")
    addLines.append("addedPinNodes, 4, 1.0\n")
    addLines.append("addedPinNodes, 5, 1.0\n")
    addLines.append("addedPinNodes, 6, 1.0\n")
    addLines.append("!SPRING, GRPID=2\n")
    addLines.append("pinNodes, 4, 1.0\n")
    addLines.append("pinNodes, 5, 1.0\n")
    addLines.append("pinNodes, 6, 1.0\n")
    #挿入位置を確認
    nStep = 0    
    ip = pyFistr.getBndInsertPointInCnt(lines, "!SPRING", ["BOUNDARY", "Boundary", "boundary"], nStep)
    #挿入位置を確認
    line = lines[ip]
    words = pyFistr.deleteSp(line).split(",")
    if len(words) > 0:
        if words[0] == "!SPRING":
            line = lines[ip+1]
            words = pyFistr.deleteSp(line).split(",")
            if words[0] == "addedPinNodes":
                #既に!SPRING, addedPinNodesが存在する場合、戻る
                return lines
    #挿入する
    lines = lines[:ip] + addLines + lines[ip:]
    return lines

#
#  addSoftSpringToDummyAddedPinNodes
def addSoftSpringToDummyAddedPinNodes(lines):
    """ 弱いバネをdummy_addedPinNodesに追加する"""
    #addLinesを作成
    addLines = []
    addLines.append("!SPRING, GRPID=1\n")
    addLines.append("dummy_addedPinNodes, 1, 1.0\n")
    addLines.append("dummy_addedPinNodes, 2, 1.0\n")
    addLines.append("dummy_addedPinNodes, 3, 1.0\n")
    addLines.append("!SPRING, GRPID=2\n")
    addLines.append("dummy_pinNodes, 1, 1.0\n")
    addLines.append("dummy_pinNodes, 2, 1.0\n")
    addLines.append("dummy_pinNodes, 3, 1.0\n")
    #挿入位置を確認
    nStep = 0    
    ip = pyFistr.getBndInsertPointInCnt(lines, "!SPRING", ["BOUNDARY", "Boundary", "boundary"], nStep)
    #挿入位置を確認
    line = lines[ip]
    words = pyFistr.deleteSp(line).split(",")
    if len(words) > 0:
        if words[0] == "!SPRING":
            line = lines[ip+1]
            words = pyFistr.deleteSp(line).split(",")
            if words[0] == "dummy_addedPinNodes":
                #既に!SPRING, dummy_addedPinNodesが存在する場合、戻る
                return lines
            elif words[0] == "addedPinNodes":
                #既に!SPRING, addedPinNodesが存在する場合、削除する。
                ipe = ip + 1
                while ipe < len(lines):
                    if lines[ipe][0] == "!":
                        break
                    ipe += 1
                lines = lines[:ip] + lines[ipe:]
    #挿入する
    lines = lines[:ip] + addLines + lines[ip:]
    return lines

#
#  getPinNgrps
#--------------
def getPinNgrps(meshHeaderData):
    """ pinNodesの設定をしたNGRP名を取得する"""
    pinNgrps = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "#pinNodes":
            pinNgrps += words[1:]
    return pinNgrps

#
#  getElmTypeOfPinNodes
#-----------------------
def getElmTypeOfPinNodes(meshHeaderData):
    """ elmTypeを取得する"""
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp == "addedPinNodes":
                elmType = "611"
                break
            elif ngrp == "dummy_addedPinNodes":
                elmType = "641"
                break
    return elmType

#
#  getAllPinNodes
#------------------
def getAllPinNodes(meshHeaderData):
    """ pinNodesとして設定した全nodeNoを取得する"""
    pinNodes = []
    addedPinNodes = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp == "pinNodes":
                pinNodes += data
        elif words[0] == "!NODE":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp == "addedPinNodes":
                addedPinNodes += list(map(lambda x: x[0], data))
    return pinNodes, addedPinNodes

#
#  getAllDummyPinNodes
#-----------------------
def getAllDummyPinNodes(meshHeaderData):
    pinNodes = []
    addedPinNodes = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp == "dummy_pinNodes":
                pinNodes += data
        elif words[0] == "!NODE":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp == "dummy_addedPinNodes":
                addedPinNodes += list(map(lambda x: x[0], data))
    return pinNodes, addedPinNodes

#
#  getPairNodesFromLoc
#----------------------
def getPairNodesFromLoc(pinNodes, addedPinNodes, nodeDict):
    """ pairのnodeをlocから取得する"""
    pairNodes = []
    for pinNode in pinNodes:
        pairNode = [pinNode]
        loc = nodeDict[pinNode][0]
        for addedNode in addedPinNodes:
            if nodeDict[addedNode][0] == loc:
                pairNode.append(addedNode)
        pairNodes.append(pairNode)
    return pairNodes

#
#  replaceAddedPinNodeNo
#-------------------------
def replaceAddedPinNodeNo(pairNodes, meshHeaderData):
    """ 要素内のaddedPinNodeのnodeNoを元のnodeNoに戻す。"""
    for pairNode in pairNodes:
        baseNode = pairNode[0]
        addNodes = pairNode[1:]
        addNodesSet = set(addNodes)
        for i in range(len(meshHeaderData)):
            header = meshHeaderData[i][0]
            data = meshHeaderData[i][1]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                for ii in range(len(data)):
                    elm = data[ii]
                    if len(set(elm[1:]) & addNodesSet) > 0:
                        #置き換える
                        for j in range(1, len(elm)):
                            if elm[j] in addNodesSet:
                                elm[j] = baseNode
                        data[ii] = elm
                meshHeaderData[i][1] = data
    return meshHeaderData

#
#  deleteAddedPinNodes
#----------------------
def deleteAddedPinNodes(meshHeaderData):
    """ addedPinNodesの設定を削除する"""
    newHeaderData = []
    for headerData in meshHeaderData:
        delFlag = 0
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp == "addedPinNodes":
                delFlag = 1
        elif words[0] == "!NGROUP":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp == "pinNodes":
                delFlag = 1
        if delFlag == 0:
            newHeaderData.append(headerData)
    for i in range(len(newHeaderData)):
        header = newHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "#pinNodes":
            ip = i
            break
    meshHeaderData = newHeaderData[:ip] + newHeaderData[ip+2:]
    return meshHeaderData

#
#  deleteDummyAddedPinNodes
#---------------------------
def deleteDummyAddedPinNodes(meshHeaderData):
    """ addedPinNodesの設定を削除する"""
    newHeaderData = []
    for headerData in meshHeaderData:
        delFlag = 0
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp == "dummy_addedPinNodes":
                delFlag = 1
        elif words[0] == "!NGROUP":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp == "dummy_pinNodes":
                delFlag = 1
        if delFlag == 0:
            newHeaderData.append(headerData)
    for i in range(len(newHeaderData)):
        header = newHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "#pinNodes":
            ip = i
            break
    meshHeaderData = newHeaderData[:ip] + newHeaderData[ip+1:]
    return meshHeaderData

#
#  deleteSoftSpring
#-------------------
def deleteSoftSpring(contHeaderData):
    """ 弱いバネを削除する"""
    delNgrps = ["pinNodes", "addedPinNodes"]
    newHeaderData = []
    for i in range(len(contHeaderData)):
        headerData = contHeaderData[i]
        header = headerData[0]
        data = headerData[1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SPRING":
            ngrp = pyFistr.deleteSp(data[0]).split(",")[0]
            if ngrp in delNgrps:
                pass
            else:
                newHeaderData.append(headerData)
        else:
            newHeaderData.append(headerData)
    return newHeaderData

#
#  deleteDummySoftSpring
#------------------------
def deleteDummySoftSpring(contHeaderData):
    """ 弱いバネを削除する"""
    delNgrps = ["dummy_pinNodes", "dummy_addedPinNodes"]
    newHeaderData = []
    for i in range(len(contHeaderData)):
        headerData = contHeaderData[i]
        header = headerData[0]
        data = headerData[1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SPRING":
            ngrp = pyFistr.deleteSp(data[0]).split(",")[0]
            if ngrp in delNgrps:
                pass
            else:
                newHeaderData.append(headerData)
        else:
            newHeaderData.append(headerData)
    return newHeaderData

#
#  readFistrModelCnt
#---------------------
def readFistrModelCnt():
    currDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(currDir)
    fileName = currDir + os.sep + FistrModel_cnt
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    return lines 

#
#  writeFistrModelCnt
#---------------------
def writeFistrModelCnt(lines):
    currDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(currDir)
    fileName = currDir + os.sep + FistrModel_cnt
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

#-----------------
#  setPinNodes
#-----------------
def setPinNodes(ngrps):
    """ 指定されたNGRPの節点をpinNodeに変更する"""
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    nodeDict, elementDict = createNodeElementDict(meshHeaderData)
    nodeNos = getPinNodeNos(ngrps, meshHeaderData)
    nodeNos = getOnlyBeamElements(nodeNos, nodeDict, elementDict)
    nodeNos.sort()
    elmTypes = getElmTypes(nodeNos, nodeDict, elementDict)
    if len(elmTypes) > 1:
        print(_("複数のbeam要素typeが存在します。"))
        print(_("  pinNodeに設定できません。"))
        return
    elif elmTypes[0] == "611":
        pairNodes, pairElms = getPairNodesElements(nodeNos, nodeDict, elementDict)
        meshHeaderData = addNewNodes(pairNodes, meshHeaderData, nodeDict)
        meshHeaderData = replaceNodeNo(pairNodes, pairElms, meshHeaderData)
        meshHeaderData = createEquation(ngrps, pairNodes, pairElms, meshHeaderData, nodeDict, elementDict)
        lines = pyFistr.createLinesFistrModelMshHeader(meshHeaderData)
        pyFistr.writeFistrModelMsh(lines)
        lines = readFistrModelCnt()
        lines = addSoftSpringToAddedPinNodes(lines)
        writeFistrModelCnt(lines)
    elif elmTypes[0] == "641":
        dummyPairNodes, pairElms = getDummyPairNodesElements(nodeNos, nodeDict, elementDict)
        meshHeaderData = addDummyNewNodes(dummyPairNodes, meshHeaderData, nodeDict)
        meshHeaderData = replaceDummyNodeNo(dummyPairNodes, pairElms, meshHeaderData)
        meshHeaderData = createPinNodesCard(ngrps, meshHeaderData)
        lines = pyFistr.createLinesFistrModelMshHeader(meshHeaderData)
        pyFistr.writeFistrModelMsh(lines)
        lines = readFistrModelCnt()
        lines = addSoftSpringToDummyAddedPinNodes(lines)
        writeFistrModelCnt(lines)
    print(_("節点Grp「") + ", ".join(ngrps) + _("」") + _("をpinNodeに設定しました。"))

#------------------
#  deletePinNodes
#------------------
def deletePinNodes():
    """ 全てのpinNode設定を削除する。（設定前の状態に戻す）"""
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    nodeDict, elementDict = createNodeElementDict(meshHeaderData)
    pinNgrps = getPinNgrps(meshHeaderData)
    if len(pinNgrps) == 0:
        print(_("pinNodesの設定がありません。"))
        print(_("  pinNodesの設定が削除できません。"))
        return
    elmType = getElmTypeOfPinNodes(meshHeaderData)
    if elmType == "611":
        pinNodes, addedPinNodes = getAllPinNodes(meshHeaderData)
        pairNodes = getPairNodesFromLoc(pinNodes, addedPinNodes, nodeDict)
        meshHeaderData = replaceAddedPinNodeNo(pairNodes, meshHeaderData)
        meshHeaderData = deleteAddedPinNodes(meshHeaderData)
        lines = pyFistr.createLinesFistrModelMshHeader(meshHeaderData)
        pyFistr.writeFistrModelMsh(lines)
        lines = readFistrModelCnt()
        contHeaderData = pyFistr.getHeaderData(lines)
        contHeaderData = deleteSoftSpring(contHeaderData)
        lines = pyFistr.createLinesFistrModelMshHeader(contHeaderData)
        writeFistrModelCnt(lines)
    elif elmType == "641":
        pinNodes, addedPinNodes = getAllDummyPinNodes(meshHeaderData)
        pairNodes = getPairNodesFromLoc(pinNodes, addedPinNodes, nodeDict)
        meshHeaderData = replaceAddedPinNodeNo(pairNodes, meshHeaderData)
        meshHeaderData = deleteDummyAddedPinNodes(meshHeaderData)
        lines = pyFistr.createLinesFistrModelMshHeader(meshHeaderData)
        pyFistr.writeFistrModelMsh(lines)
        lines = readFistrModelCnt()
        contHeaderData = pyFistr.getHeaderData(lines)
        contHeaderData = deleteDummySoftSpring(contHeaderData)
        lines = pyFistr.createLinesFistrModelMshHeader(contHeaderData)
        writeFistrModelCnt(lines)
    print(_("全pinNodesを削除しました。"))

#
#  printHelp
#------------
def printHelp():
    """ helpを出力"""
    cont  = "------------- setPinNodesOfBeam.py ---------------------\n"
    cont += "指定したNGRP名のnodeをpinNodeに置き換える。\n"
    cont += "（変位は伝達するがモーメントは伝達しないnode）\n"
    cont += "beam要素の611(6自由度)、641(3自由度)に対応。\n"
    cont += "<使い方>\n"
    cont += "  setPinNodesOfBeam.py [option] <NRGP名1> <NGRP名2>...\n"
    cont += "    option\n"
    cont += "      -d, --deletePin  :pinNodeの設定を削除\n"
    cont += "      -h, --help       :helpを表示\n"
    cont += "<使用例>\n"
    cont += "  setPinNodesOfBeam.py linkNodes\n"
    cont += "  setPinNodesOfBeam.py -d                 #pinNodeを削除\n"
    cont += "--------------------------------------------------------\n"
    print(cont)

#
#  getArguments
#---------------
def getArguments():
    """ 引数を取得する"""
    if len(sys.argv) < 2:
        printHelp()
        return []
    ngrps = []
    delFlag = "no"
    for argv in sys.argv[1:]:
        if argv == "-h" or argv == "--help":
            printHelp()
            ngrps = []
            break
        elif argv == "-d" or argv == "--deletePin":
            delFlag = "yes"
            break
        else:
            ngrps.append(argv)
    return ngrps, delFlag


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    _ = gettext.gettext

    ngrps, delFlag = getArguments()
    if len(ngrps) != 0 and delFlag == "no":
        setPinNodes(ngrps)
    elif delFlag == "yes":
        deletePinNodes()
