#!/usr/bin/python3
#  coding : utf-8
#
#           readCadFileCreateVtkFiles_gmsh.py
#
#           gmshでcadFileを読み込みvtkFilesを出力する。
#           gmshは、新たなprocessを起動しないとgmsh.initialize()が通らない。
#           
#
#   25/03/19    新規作成
#      05/11    createVtkSourceFiles:新規追加。vtkDataFile作成を追加。
#      05/14    modelSizeの出力を廃止。
#      05/20    modelSizeの出力を復活。（vtkFolder内にfileで出力）
#      05/22    edges.vtuをedges.vtkに変更。vtkは、gmshがexportして作成。
#      05/28    edges,surfacesをvtuに戻す。
#               gmsh.initialize,finalizeを各々の設定。（2重定義がある為）
#      06/03    edge,surfaceのvtkを1次要素から2次要素に変更。
#      06/10    小さなRがあると、CAD形状が表示できない事がある。形状表示用の
#               surfaceMmeshが作成できていない。edgeMesh作成時に、
#               pointのmeshSizeを取得設定する様に修正
#      06/21    createVtkCurveLines,createVtkSurfaceLines,createVtkPoints:
#               formatをascii or binaryに対応させる。
#      06/30    getGlobalMeshSizeFromEdgeElmLength:globalMeshSizeをmodelSize
#               から決定していたが、edge要素（beam）のmaxSizeから決定する様に修正。
#      07/18    createVtkSurfaceLines:cellのconectivityを取得時、+=で取得していた。
#               これを=で取得に変更。バグ修正。
#      08/12    createVtkSourceFilesFromCad:終了メッセージ出力を追加
#      08/18    settingDialog,numSettingDialog,closeSettingDialog:処理の進捗表示追加。
#               createVtkEdgeMesh,createVtkSurfaceMesh:curNo, surNoの取得方法修正。
#      09/14    createEdgeMeshGetPointMeshSize:meshSizeを「mainVal」→「mainVal*0.8」
#               に変更。
#      09/19    createVtkSurfaceMesh:円柱のモデルでは、surNoが正しく取得できなかった為修正
#               surNoを指定して要素を取得する様に修正。（getElements(2,surTag)を使用）
#               volumeNoを新規追加。
#               createVtkEdgeMesh,createVtkPointsMesh:volumeNoを新規追加。
#      09/21    createVtkPoints:pointNoをvtuファイルに追加。
#               vtkのVersionを1.04→1.05に変更。
#      09/22    createVtkEdgeMesh,createVtkSurfaceMesh,createVtkPointsMesh:
#               2Dモデルの場合、エラー発生する為修正。stlに変更後error発生していた。
#               createVtkSurfaceMesh,createSurfaceMesh:1Dモデルでエラー発生する為修正。
#      09/25    選択部品の非表示設定、EGRPの設定を追加する大幅修正。
#

import os, sys
import gmsh
import glob
import subprocess
import time
import pyFistr
import geometryFunc as geo
import createVtkSourceFiles as createVtkSource

Version = "1.05"        #作成するvtkFileのversion
modelSizeXYZ = ""
settingDialog = ""
globalMeshSize = ""

#
#  getModelSize
#----------------
def getModelSize(fileName, currDir):
    """ modelSizeを取得し、出力する"""
    global modelSizeXYZ, modelDim
    gmsh.initialize()
    dimTag = gmsh.model.occ.importShapes(fileName)
    #modelSizeを取得
    boundingBoxes = []
    for dim, Tag in dimTag:
        boundingBoxes += [gmsh.model.occ.getBoundingBox(dim, Tag)]
    boundingBox = [
        min(list(map(lambda x: x[0], boundingBoxes))),
        min(list(map(lambda x: x[1], boundingBoxes))),
        min(list(map(lambda x: x[2], boundingBoxes))),
        max(list(map(lambda x: x[3], boundingBoxes))),
        max(list(map(lambda x: x[4], boundingBoxes))),
        max(list(map(lambda x: x[5], boundingBoxes)))
        ]
    lx = boundingBox[3] - boundingBox[0]
    ly = boundingBox[4] - boundingBox[1]
    lz = boundingBox[5] - boundingBox[2]
    slx = pyFistr.float2strAuto(lx, prec=3, zeroNum=1e-10, showPoint=False)
    sly = pyFistr.float2strAuto(ly, prec=3, zeroNum=1e-10, showPoint=False)
    slz = pyFistr.float2strAuto(lz, prec=3, zeroNum=1e-10, showPoint=False)
    modelSizeXYZ = " ".join([slx, sly, slz])
    #lines = ["modelSize " + slx + " " + sly + " " + slz + "\n"]
    lines = ["modelSize " + modelSizeXYZ + "\n"]
    #maxDimを取得
    modelDim = "1D"
    maxTag = gmsh.model.occ.getMaxTag(dim=2)
    if maxTag >= 1:
        modelDim = "2D"
        maxTag = gmsh.model.occ.getMaxTag(dim=3)
        if maxTag >= 1:
            modelDim = "3D"
    lines += ["modelDim " + modelDim + "\n"]
    #cadNameを取得
    cadName = os.path.basename(fileName)
    lines += ["cadFile " + cadName + "\n"]
    #結果をを出力
    print(lines[0])
    vtkDir = currDir + os.sep + "vtkCadData"
    fileName = vtkDir + os.sep + "modelSize"
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()
    gmsh.finalize()

#
#  getPointsInCadFile
def getPointsInCadFile(fileName):
    """ cadFile内からpointsの座標を取得"""
    #cadFileをopen
    gmsh.open(fileName)
    #pointsのdimTagsを取得
    points = gmsh.model.getEntities(0)
    #edges = gmsh.model.getEntities(1)
    #pointの座標を取得
    locs = []
    for point in points:
        loc = gmsh.model.getValue(0, point[1], [])
        locs.append(loc)
    return locs

#  createVtkPoints
def createVtkPoints(locs, pointDict, volNos):
    """ vtkLinesを作成する"""
    lines = []
    lines += ['<?xml version="1.0"?>\n']
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0" byte_order="LittleEndian" header_type="UInt64" >\n']
    lines += ['<UnstructuredGrid>\n']
    nPoints = len(locs)
    lines += ['<Piece NumberOfPoints="' + str(nPoints) + '" NumberOfCells="' + str(nPoints) + '">\n']
    lines += ['<Points>\n']
    #座標データ
    line = '<DataArray type="Float32" NumberOfComponents="3" format="ascii">\n'
    vals = []
    for loc in locs:
        vals += list(loc)
    blockLines = createVtkSource.makePointsDataArrayBlock(vtkFormat, line, vals, nPoints)
    lines += blockLines
    lines += ['</Points>\n']
    #cellデータ
    lines += ['<Cells>\n']
    #connectivity
    line = '<DataArray type="Int32" Name="connectivity" format="ascii">\n'
    nodes = []
    for i in range(len(locs)):
        nodes.append([i])
    blockLines = createVtkSource.makeConnectivityDataArray(vtkFormat, line, nodes, nPoints)
    lines += blockLines
    #offsets
    line = '<DataArray type="Int32" Name="offsets" format="ascii">\n'
    elms = []
    for i in range(len(locs)):
        elms.append(i+1)
    blockLines = createVtkSource.makeOffsetsTypesDataArray(vtkFormat, line, elms, nPoints, name="offsets")
    lines += blockLines
    #types
    line = '<DataArray type="UInt8" Name="types" format="ascii">\n'
    vtkNames = [1 for i in range(nPoints)]
    blockLines = createVtkSource.makeOffsetsTypesDataArray(vtkFormat, line, vtkNames, nPoints, name="types")
    lines += blockLines
    lines += ['</Cells>\n']
    #cellData
    lines += ['<CellData>\n']
    line = '<DataArray type="Float32" Name="meshSize" NumberOfComponents="1" format="ascii">\n'
    vals = []
    pointIds = list(pointDict.keys())
    pointIds.sort()
    for pointId in pointIds:
        val = pointDict[pointId]
        if val != "":
            vals.append(float(val))
        else:
            vals.append(-1.0)
    blockLines = createVtkSource.makeDataArrayBlock(vtkFormat, line, vals, nPoints)
    lines += blockLines
    #pointNo
    line = '<DataArray type="Int32" Name="pointNo" NumberOfComponents="1" format="ascii">\n'
    values = pointIds
    blockLines = createVtkSource.makeDataArrayBlock(vtkFormat, line, values, nPoints)
    lines += blockLines
    #volumeNo
    line = '<DataArray type="Int32" Name="volumeNo" NumberOfComponents="1" format="ascii">\n'
    values = volNos
    blockLines = createVtkSource.makeDataArrayBlock(vtkFormat, line, values, nPoints)
    lines += blockLines
    lines += ['</CellData>\n']
    lines += ['</Piece>\n']
    lines += ['</UnstructuredGrid>\n']
    lines += ['</VTKFile>\n']
    return lines

#
#  createVtkEdgeSurfaceMesh
def createVtkEdgeSurfaceMesh(cadFilePath, currDir):
    """ vtkEdge, surfaceのmeshを作成し、保存する"""
    global globalMeshSize
    gmsh.initialize()
    #folderの確認
    vtkFolder = currDir + os.sep + "vtkCadData"
    if os.path.exists(vtkFolder) == False:
        os.mkdir(vtkFolder)
    #pointのlocsを取得
    pntLocs = getPointsInCadFile(cadFilePath)
    #pointのmeshSizeを辞書で取得
    numSettingDialog(20, "getting meshSize of points...")
    pointDict, meshSize = createEdgeMeshGetPointMeshSize(cadFilePath, pntLocs)
    globalMeshSize = pyFistr.float2strAuto(meshSize, prec=3)
    #surfaceMesh作成(stl作成)
    numSettingDialog(40, "creating stl data from CAD data...")
    createSurfaceMesh()
    #volume毎のpoints,edges,surfacesを辞書で取得
    pointVol, curveVol, surfaceVol = getPartsOfEachVolumes()
    #edge,surfaceのvtkを作成
    numSettingDialog(60, "gettting surfaceNo and curveNo form stl data...")
    surNos, curNos = getSurfaceCurveTags()
    numSettingDialog(70, "creating edge vtk data...")
    createVtkEdgeMesh(curNos, currDir, curveVol)
    numSettingDialog(80, "creating surface vtk data...")
    createVtkSurfaceMesh(surNos, currDir, surfaceVol)
    numSettingDialog(90, "creating point vtk data...")
    createVtkPointsMesh(pntLocs, pointDict, pointVol)
    gmsh.finalize()

#
#  createEdgeMeshGetPointMeshSize
def createEdgeMeshGetPointMeshSize(cadFilePath, pntLocs):
    """ edgeMeshを作成し、pointのmeshSizeを取得する"""
    #gmsh.open(cadFilePath)
    #meshSizeを設定
    gmsh.option.setNumber("Mesh.ElementOrder", 2)
    gmsh.option.setNumber("Mesh.MeshSizeFactor", 1.0)
    #edgeのメッシュを作成
    gmsh.model.mesh.generate(dim=1)
    #pointDictを作成する。
    gmType = 8      #beam2次
    #nodeDict, elementDictを作成
    nodeDict, elementDict = getNodeElementDict(gmType, getDim=1)
    #globlMeshSizeを取得
    #mainVal = getGlobalMeshSizeFromModelSize(modelSizeXYZ)
    #mainVal = float(mainVal)
    mainVal = getGlobalMeshSizeFromEdgeElmLength(nodeDict, elementDict)
    meshSize = mainVal * 0.8
    pointDict = getDefaultPointMeshSizeDict(nodeDict, elementDict, pntLocs, mainVal)
    return pointDict, meshSize

#
#  createSurfaceMesh
def createSurfaceMesh():
    """ 表面mesh作成"""
    global modelDim
    #beamモデル？
    if modelDim == "1D":
        #beamモデルの場合
        gmsh.option.setNumber("Mesh.ElementOrder", 1)
        gmsh.option.setNumber("Mesh.MeshSizeFactor", 0.8)
        #edgeのメッシュを作成
        gmsh.model.mesh.generate(dim=1)
    #2D以上モデル？
    else:
        #2D以上のモデル
        # set STL generation options
        gmsh.model.mesh.clear()
        gmsh.option.setNumber('Mesh.StlLinearDeflection', 1)
        gmsh.option.setNumber('Mesh.StlLinearDeflectionRelative', 0)
        gmsh.option.setNumber('Mesh.StlAngularDeflection', 0.5)
        gmsh.model.mesh.importStl()
        gmsh.model.mesh.removeDuplicateNodes()

#
#  createVtkEdgeMesh
def createVtkEdgeMesh(curNos, currDir, curveVol):
    """ vtkEdgeDataを作成。Gmsh上で1Dメッシュを作成し、vtkでexportして作成する。"""
    global modelSizeXYZ
    #curve上のnodeを取得
    curveToNodeSetsDict = {}
    for curNo in curNos:
        tag = curNo
        nodeTags, cood = gmsh.model.mesh.getNodesForPhysicalGroup(1, tag)
        curveToNodeSetsDict[tag] = set(nodeTags)
    #elementDictを作成
    gmType = 1      #beam1次
    #gmType = 8       #beam2次
    #nodeElementDict作成
    nodeDict, elementDict = getNodeElementDict(gmType, getDim=1)
    #elementが属するcurNoを取得
    curNos = []
    for elmNo in elementDict.keys():
        nodes = elementDict[elmNo]
        curNo = -1
        for key in curveToNodeSetsDict.keys():
            curNodeSets = curveToNodeSetsDict[key]
            #nodes[0], nodes[1]がcurNodeSetsに含まれるか？
            if nodes[0] in curNodeSets:
                if nodes[1] in curNodeSets:
                    curNo = key
                    break
        curNos.append(curNo)
    #elementが属するvolumeNoを取得
    shift = min(curNos)
    volNos = []
    if len(curveVol) > 0:
        for curNo in curNos:
            idx = curNo - shift
            volNo = curveVol[idx]
            volNos.append(volNo)
    else:
        #volumeが無い場合（シート等）は、「-1」を設定
        for curNo in curNos:
            volNos.append(-1)
    #vtk用にrenumber
    nodeLocs = []
    newNo = 0
    for nodeNo in nodeDict.keys():
        nodeDict[nodeNo].append(newNo)
        loc = nodeDict[nodeNo][0]
        nodeLocs.append(loc)
        newNo += 1
    newElms = []
    for elmNo in elementDict.keys():
        nodes = elementDict[elmNo]
        newNodes = []
        for nodeNo in nodes:
            newNo = nodeDict[nodeNo][1]
            newNodes.append(newNo)
        newElms.append(newNodes)
    newCurNos = list(map(lambda x: x-1, curNos))
    newVolNos = list(map(lambda x: x-1, volNos))
    #createVtkLines
    lines = createVtkCurveLines(nodeLocs, newElms, newCurNos, newVolNos)
    vtkFolder = currDir + os.sep + "vtkCadData"
    vtkEdgeFile = vtkFolder + os.sep + "edges.vtu"
    f = open(vtkEdgeFile, "w", encoding="utf-8"); f.writelines(lines); f.close()


#  getNodeElementDict
def getNodeElementDict(gmType, getDim):
    """ gmType(elmType)から、nodeDict, elementDictを取得して返す。
    nodeDictは、{<nodeNo>:[[x,y,z]],...}のように、list(list)形式で返す。"""
    #elementDictを取得
    elmTags, elmNodeTags = gmsh.model.mesh.getElementsByType(gmType)
    elementDict = {}
    i = 0
    for elmTag in elmTags:
        elmType, nodeTags, dim, eTag = gmsh.model.mesh.getElement(elmTag)
        #if dim == 1:
        if dim == getDim:
            elementDict[i] = nodeTags
            i += 1
    #nodeDictを取得
    nodeTags, coord, paraCoord = gmsh.model.mesh.getNodesByElementType(gmType)
    nodeDict = {}
    for i in range(len(nodeTags)):
        nodeTag = nodeTags[i]
        j = i * 3
        loc = coord[j:j+3]
        nodeDict[nodeTag] = [loc]
    return nodeDict, elementDict

#  getPartsOfEachVolumes
def getPartsOfEachVolumes():
    """ volume毎のpoint, curves, surfacesの辞書取得して返す
    
    Returns:
        pointVol (list):   volNoのlist
        curveVol (list):    ↑
        surfaceVol (list):  ↑"""
    volPointDict = {}
    volCurveDict = {}
    volSurfaceDict = {}
    #全volumeのtagを取得
    allVolumes = gmsh.model.getEntities(dim=3)
    for volTagList in allVolumes:
        volTag = volTagList[1]
        pointSet = set([])
        curveSet = set([])
        surSet = set([])
        #volume毎のsurTagを取得
        _dummy, surTags = gmsh.model.getAdjacencies(3, volTag)
        for surTag in surTags:
            #surface毎のcurveTagを取得
            _dummy, curveTags = gmsh.model.getAdjacencies(2, surTag)
            surSet.add(surTag)
            for curveTag in curveTags:
                #curve毎のpointTagを取得
                _dummy, pointTags = gmsh.model.getAdjacencies(1, curveTag)
                curveSet.add(curveTag)
                for pointTag in pointTags:
                    pointSet.add(pointTag)
        volPointDict[volTag] = list(pointSet)
        volCurveDict[volTag] = list(curveSet)
        volSurfaceDict[volTag] = list(surSet)
    #patsNo毎にvolNoをまとめ直し
    pointVol = []
    for vol in volPointDict.keys():
        points = volPointDict[vol]
        for point in points:
            pointVol.append([point, vol])
    pointVol.sort()
    pointVol = list(map(lambda x: x[1], pointVol))
    curveVol = []
    for vol in volCurveDict.keys():
        curves = volCurveDict[vol]
        for curve in curves:
            curveVol.append([curve, vol])
    curveVol.sort()
    curveVol = list(map(lambda x: x[1], curveVol))
    surfaceVol = []
    for vol in volSurfaceDict.keys():
        surfaces = volSurfaceDict[vol]
        for surface in surfaces:
            surfaceVol.append([surface, vol])
    surfaceVol.sort()
    surfaceVol = list(map(lambda x: x[1], surfaceVol))
    return pointVol, curveVol, surfaceVol

#  getSurfaceCurveTags
def getSurfaceCurveTags():
    """ surfaceとEdgeのtagを取得"""
    #surfaceTagを取得
    maxTag = gmsh.model.occ.getMaxTag(dim=3)
    surfaceTags = []
    if maxTag < 0:
        #3Dの場合
        for i in range(maxTag):
            tag = i + 1
            loopTags, surTags = gmsh.model.occ.getSurfaceLoops(tag)
            surfaceTags += surTags
    else:
        #2Dの場合
        maxTag = gmsh.model.occ.getMaxTag(dim=2)
        for i in range(maxTag):
            tag = i + 1
            surfaceTags.append([tag])
    #surfaceのphysicalGroup作成とcurveTagを取得
    nextTag = 1
    surNos = []
    if len(surfaceTags) > 0: 
        #surfaceがある場合（2D以上）
        i = 1
        curveTags = []
        for surTag in surfaceTags:
            tags = list(surTag)
            for tag in tags:
                surNos.append(i)
                gmsh.model.addPhysicalGroup(2, [tag], name=str(tag))
                loopTags, curTags = gmsh.model.occ.getCurveLoops(tag)
                curveTags += curTags
                i += 1
        nextTag = i
    else:
        #beamの場合（1Dの場合）
        maxTag = gmsh.model.occ.getMaxTag(dim=1)
        curveTags = []
        for i in range(maxTag):
            tag = i + 1
            curveTags.append([tag])
        nextTag = 1
    #curveTagからphysicalGroupを作成
    i = nextTag
    curveNos = set([])
    curNos = []
    for curTag in curveTags:
        tags = list(curTag)
        for tag in tags:
            if not (tag in curveNos):
                curveNos.add(tag)
                curNos.append(i)
                gmsh.model.addPhysicalGroup(1, [tag], name=str(tag))
                i += 1
    return surNos, curNos

#  getGlobalMeshSizeFromModelSize
def getGlobalMeshSizeFromModelSize(modelSize):
    """ modelSizeからglobalのmeshSizeを取得する"""
    global modelSizeXYZ
    dx, dy, dz = list(map(float, modelSize.split()))
    #maxhV = (dx * dy * dz / 100.0) ** 0.333 
    #maxhS = max([dx, dy, dz]) / 5.0
    maxhV = (dx * dy * dz / 200.0) ** 0.333 
    maxhS = max([dx, dy, dz]) / 10.0
    maxh = max([maxhV, maxhS])
    maxh = pyFistr.float2strAuto(maxh, prec=2, zeroNum=1e-10, showPoint=False)
    return maxh

#  getGlobalMeshSizeFromEdgeElmLength
def getGlobalMeshSizeFromEdgeElmLength(nodeDict, elementDict):
    """ edge要素の最大値からglobalのmeshSizeを設定する"""
    meshSizes = []
    for elmNo in elementDict.keys():
        nodes = elementDict[elmNo]
        node0 = nodes[0]; node1 = nodes[-1]
        loc0 = list(nodeDict[node0][0]); loc1 = list(nodeDict[node1][0])
        l = geo.length(loc0, loc1)
        meshSizes.append(l)
    meshSize = max(meshSizes)
    return meshSize


#  getDefaultPointMeshSizeDict
def getDefaultPointMeshSizeDict(nodeDict, elementDict, pntLocs, mainVal):
    """ pointに設定するmeshSizeを取得する"""

    def isInPointLocs(minLoc, pntLocs, tol):
        """ minLoc点がpoint群の座標と同じか調べる"""
        ans = False
        for pntLoc in pntLocs:
            l = geo.length(pntLoc, minLoc)
            if l < tol:
                ans = True
                break
        return ans

    #pointに接続される要素のlocを取得
    pointDict = {}
    node0 = elementDict[0][0]; node1 = elementDict[0][1]
    tol = geo.length(nodeDict[node0][0], nodeDict[node1][0]) / 1000.0
    for elmNo in elementDict.keys():
        nodes = elementDict[elmNo]
        for pntNo in range(len(pntLocs)):
            loc0 = nodeDict[nodes[0]][0]
            loc1 = nodeDict[nodes[1]][0]
            pntLoc = pntLocs[pntNo]
            l0 = geo.length(loc0, pntLoc)
            l1 = geo.length(loc1, pntLoc)
            if l0 < tol:
                if pntNo in pointDict.keys():
                    pointDict[pntNo] += [loc1]
                else:
                    pointDict[pntNo] = [loc1]
            elif l1 < tol:
                if pntNo in pointDict.keys():
                    pointDict[pntNo] += [loc0]
                else:
                    pointDict[pntNo] = [loc0]
    #pointへの設定値を取得
    for pntNo in pointDict.keys():
        pntLoc = pntLocs[pntNo]
        lls = []
        for loc in pointDict[pntNo]:
            l = geo.length(pntLoc, loc)
            lls.append([l, list(loc)])
        val, minLoc = min(lls)
        #if val < mainVal / 2.0:
        if val < mainVal:
            #minLocがpointLocsに含まれる？
            if isInPointLocs(minLoc, pntLocs, tol) == False:
                #含まれない場合
                if val < mainVal:
                    pointDict[pntNo] = pyFistr.float2strAuto(val, prec=3)
                else:
                    pointDict[pntNo] = ""    
            else:
                #含む場合
                pointDict[pntNo] = ""
        elif len(lls) == 1:
            #val = mainVal / 10.0
            #pointDict[pntNo] = pyFistr.float2strAuto(val, prec=3)
            pointDict[pntNo] = ""
        else:
            pointDict[pntNo] = ""
    return pointDict

#  createVtkCurveLines
def createVtkCurveLines(nodeLocs, elements, curNos, volNos):
    """ curveのvtkLinesを作成する"""
    lines = []
    lines += ['<?xml version="1.0"?>\n']
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0" byte_order="LittleEndian" header_type="UInt64">\n']
    lines += ['<UnstructuredGrid>\n']
    nLocs = len(nodeLocs)
    nElms = len(elements)
    lines += ['<Piece NumberOfPoints="' + str(nLocs) + '" NumberOfCells="' + str(nElms) + '">\n']
    lines += ['<Points>\n']
    #座標データ
    line = '<DataArray type="Float32" NumberOfComponents="3" format="ascii" >\n'
    vals = []
    for loc in nodeLocs:
        vals += list(loc)
    blockLines = createVtkSource.makePointsDataArrayBlock(vtkFormat, line, vals, nLocs)
    lines += blockLines
    lines += ['</Points>\n']
    #cellデータ
    lines += ['<Cells>\n']
    #connectivity
    line = '<DataArray type="Int32" Name="connectivity" format="ascii" >\n'
    nodes = []
    for i in range(len(elements)):
        newNodes = elements[i]
        nodes.append(newNodes)
    blockLines = createVtkSource.makeConnectivityDataArray(vtkFormat, line, nodes, nElms)
    lines += blockLines
    #offsets
    line = '<DataArray type="Int32" Name="offsets" format="ascii">\n'
    elms = []
    n = 0
    for i in range(len(elements)):
        n += len(elements[i])
        elms.append(n)
    blockLines = createVtkSource.makeOffsetsTypesDataArray(vtkFormat, line, elms, nElms, name="offsets")
    lines += blockLines
    #types
    line = '<DataArray type="UInt8" Name="types" format="ascii">\n'
    vtkNames = [3 for i in range(nElms)]        #1次要素
    #vtkNames = [21 for i in range(nElms)]      #2次要素
    blockLines = createVtkSource.makeOffsetsTypesDataArray(vtkFormat, line, vtkNames, nElms, name="types")
    lines += blockLines
    lines += ['</Cells>\n']
    #cellData
    lines += ['<CellData>\n']
    line = '<DataArray type="Int32" Name="curveNo" NumberOfComponents="1" format="ascii">\n'
    values = curNos
    blockLines = createVtkSource.makeDataArrayBlock(vtkFormat, line, values, nElms)
    lines += blockLines
    line = '<DataArray type="Int32" Name="volumeNo" NumberOfComponents="1" format="ascii">\n'
    values = volNos
    blockLines = createVtkSource.makeDataArrayBlock(vtkFormat, line, values, nElms)
    lines += blockLines
    lines += ['</CellData>\n']
    lines += ['</Piece>\n']
    lines += ['</UnstructuredGrid>\n']
    lines += ['</VTKFile>\n']
    return lines

#
#  createVtkSurfaceMesh
def createVtkSurfaceMesh(surNos, currDir, surfaceVol):
    """ edgeに引き続きsurfaceのメッシュを作成する"""
    global modelDim
    if modelDim == "1D":
        #beamモデルの場合
        nodeLocs = []
        newElms = []
        newSurNos = []
        newVolNos = []
        #createVtkLines
        lines = createVtkSurfaceLines(nodeLocs, newElms, newSurNos, newVolNos)
        #保存
        vtkFolder = currDir + os.sep + "vtkCadData"
        vtkSurfaceFile = vtkFolder + os.sep + "surfaces.vtu"
        f = open(vtkSurfaceFile, "w"); f.writelines(lines); f.close()
        return
    
    #elementDictを作成
    gmType = 2      #三角形1次要素
    #辞書を取得
    nodeDict, elementDict = getNodeElementDict(gmType, getDim=2)
    #surfaceに属する要素を辞書で取得
    elmToSurDict = {}
    for surTag in surNos:
        #surTagに属する要素を取得
        elmTypes, npElmTags, nodesTags = gmsh.model.mesh.getElements(2, surTag)
        #要素が取得できているか？
        if len(npElmTags) > 0:
            elmTags = npElmTags[0].tolist()
            for elmTag in elmTags:
                elmToSurDict[elmTag] = surTag
    #要素No順にsurNoを取得
    elms = list(elmToSurDict.keys())
    elms.sort()
    surNos = []
    if len(elms) > 0:
        for elmNo in elms:
            surNo = elmToSurDict[elmNo]
            surNos.append(surNo)
    else:
        for elmNo in elms:
            surNos.append(-1)
    #要素No順にvolNoをsurNoから取得
    shift = min(surNos)
    volNos = []
    if len(surfaceVol) > 0:
        for surNo in surNos:
            idx = surNo - shift
            volNo = surfaceVol[idx]
            volNos.append(volNo)
    else:
        #volumeが無い場合（シート等）は、「-1」を設定
        for surNo in surNos:
            volNos.append(-1)
    #vtk用にrenumber
    nodeLocs = []
    newNo = 0
    for nodeNo in nodeDict.keys():
        nodeDict[nodeNo].append(newNo)
        loc = nodeDict[nodeNo][0]
        nodeLocs.append(loc)
        newNo += 1
    newElms = []
    for elmNo in elementDict.keys():
        nodes = elementDict[elmNo]
        newNodes = []
        for nodeNo in nodes:
            newNo = nodeDict[nodeNo][1]
            newNodes.append(newNo)
        newElms.append(newNodes)
    newSurNos = list(map(lambda x: x-1, surNos))
    newVolNos = list(map(lambda x: x-1, volNos))
    #createVtkLines
    lines = createVtkSurfaceLines(nodeLocs, newElms, newSurNos, newVolNos)
    #保存
    vtkFolder = currDir + os.sep + "vtkCadData"
    vtkSurfaceFile = vtkFolder + os.sep + "surfaces.vtu"
    f = open(vtkSurfaceFile, "w"); f.writelines(lines); f.close()

#
#  createVtkSurfaceLines
def createVtkSurfaceLines(nodeLocs, elements, surNos, surfaceVol):
    lines = []
    lines += ['<?xml version="1.0"?>\n']
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0" byte_order="LittleEndian" header_type="UInt64" >\n']
    lines += ['<UnstructuredGrid>\n']
    nLocs = len(nodeLocs)
    nElms = len(elements)
    lines += ['<Piece NumberOfPoints="' + str(nLocs) + '" NumberOfCells="' + str(nElms) + '">\n']
    lines += ['<Points>\n']
    #座標データ
    line = '<DataArray type="Float32" NumberOfComponents="3" format="ascii">\n'
    vals = []
    for loc in nodeLocs:
        vals += list(loc)
    blockLines = createVtkSource.makePointsDataArrayBlock(vtkFormat, line, vals, nLocs)
    lines += blockLines
    lines += ['</Points>\n']
    #cellデータ
    lines += ['<Cells>\n']
    #connectivity
    line = '<DataArray type="Int32" Name="connectivity" format="ascii">\n'
    nodes = []
    for i in range(len(elements)):
        newNodes = elements[i]
        nodes.append(newNodes)
    #blockLines += createVtkSource.makeConnectivityDataArray(vtkFormat, line, nodes, nElms)
    blockLines = createVtkSource.makeConnectivityDataArray(vtkFormat, line, nodes, nElms)
    lines += blockLines
    #offsets
    line = '<DataArray type="Int32" Name="offsets" format="ascii">\n'
    elms = []
    n = 0
    for i in range(len(elements)):
        n += len(elements[i])
        elms.append(n)
    blockLines = createVtkSource.makeOffsetsTypesDataArray(vtkFormat, line, elms, nElms, name="offsets")
    lines += blockLines
    #types
    line = '<DataArray type="UInt8" Name="types" format="ascii">\n'
    vtkNames = [5 for i in range(nElms)]    #三角形1次
    #vtkNames = [22 for i in range(nElms)]  #三角形2次
    blockLines = createVtkSource.makeOffsetsTypesDataArray(vtkFormat, line, vtkNames, nElms, name="types")
    lines += blockLines
    lines += ['</Cells>\n']
    #cellData
    lines += ['<CellData>\n']
    line = '<DataArray type="Int32" Name="surNo" NumberOfComponents="1" format="ascii">\n'
    values = surNos
    blockLines = createVtkSource.makeDataArrayBlock(vtkFormat, line, values, nElms)
    lines += blockLines
    line = '<DataArray type="Int32" Name="volumeNo" NumberOfComponents="1" format="ascii">\n'
    values = surfaceVol
    blockLines = createVtkSource.makeDataArrayBlock(vtkFormat, line, values, nElms)
    lines += blockLines
    lines += ['</CellData>\n']
    lines += ['</Piece>\n']
    lines += ['</UnstructuredGrid>\n']
    lines += ['</VTKFile>\n']
    return lines

#
#  createVtkPointsMesh
def createVtkPointsMesh(pntLocs, pointDict, pointVol):
    """ pointのvtkを作成"""
    if len(pointVol) > 0:
        newVolNos = list(map(lambda x: x-1, pointVol))
    else:
        newVolNos = [-1 for i in range(len(pntLocs))]
    lines = createVtkPoints(pntLocs, pointDict, newVolNos)
    vtkFolder = currDir + os.sep + "vtkCadData"
    fileName = vtkFolder + os.sep + "points.vtu"
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  saveVersion
def saveVersion(currDir):
    """ vtkFolder内にversionを保存"""
    global Version
    vtkDir = currDir + os.sep + "vtkCadData"
    delFiles = glob.glob(vtkDir + os.sep + "version_*")
    for delFile in delFiles:
        os.remove(delFile)
    versionFile = vtkDir + os.sep + "version_" + Version
    cont = "version " + Version
    f = open(versionFile, "w", encoding="utf-8"); f.write(cont); f.close()

#
#  createVtkSourceFiles
#-----------------------
def createVtkSourceFilesFromCad(fileName, currDir):
    """ vtkDataFileを作成する"""
    global globalMeshSize
    showSettingDialog("reading CAD file...")
    #vtkDataFileを作成
    createVtkEdgeSurfaceMesh(fileName, currDir)
    saveVersion(currDir)
    print("globalMeshSize: " + globalMeshSize + "\n", flush=True)
    print("finish to create vtk files of CAD.", flush=True)
    closeSettingDialog()


#
#  showSettingDialog
#    設定中dialogを表示
def showSettingDialog(title):
    global settingDialog
    pythonAppPath = pyFistr.pythonAppPath
    comm = 'python3 ' + pythonAppPath + 'settingDialog2.py "' + title + '"'
    settingDialog = subprocess.Popen(
        comm,
        shell=True,
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE)
    numSettingDialog(0, "preparing... ")

#
#  numSettingDialog
#    設定中dialogに進行状況を表示
#    （num>100.0の場合は、閉じる）
def numSettingDialog(num, cont):
    settingDialog.stdin.write((str(num) + "," + cont + "\n").encode())
    if settingDialog.poll() == None:
        #BrokenPipeでは無い時にflushする
        settingDialog.stdin.flush()

#
#  closeSettingDialog
#    設定中dialogを閉じる
def closeSettingDialog():
    numSettingDialog(100, "close")
    settingDialog.stdin.write(("close\n").encode())
    if settingDialog.poll() == None:
        #BrokenPipeでは無い時にflushする
        settingDialog.stdin.flush()



if __name__ == "__main__":
    retry = False
    fileName = sys.argv[1]
    currDir = sys.argv[2]
    vtkFormat = sys.argv[3]
    getModelSize(fileName, currDir)
    createVtkSourceFilesFromCad(fileName, currDir)
