#!/usr/bin/env python3
#  coding; utf-8
#
#   fistr2vtu.py
#
#       fistr形式からvtu形式（vtk）に変換する。
#       下表の要素を変換する。
#
#       fistr   node    vtk     note
#       -----------------------------------------
#       341     4       10      四面体1次
#       342     10      24      四面体2次
#       361     8       12      六面体1次
#       362     20      25      六面体2次
#       351     6       13      五面体1次
#       352     15      26      五面体2次
#       611     2       3       beam要素
#       641     4       3       beam要素（3自由度）
#       731     3       5       シェル三角形1次
#       732     6       22      シェル三角形2次
#       741     4       9       シェル四角形1次
#       761     6       5       シェル三角形1次（3自由度）
#       781     8       9       シェル四角形1次（3自由度）
#
#
#   15/05/19    新規作成
#      05/21    faceGroupの値を「1」→「0」からに変更
#               surfaceVtkフォルダ内に表面要素のvtkを作成（表面よvolume要素は別ファイル）
#      05/23    表面要素とvolume要素を１ヶのfileに設定
#               elementGroup 0     :表面要素
#                            1以上 :volume要素のelementGroupNo
#               faceGroup    0     :volume要素
#                            1以上 :表面要素のfaceGroupNo
#      06/08    結果fileにpointデータのみの場合があるので、これに対応
#      09/02    "\n"→os.linesepに修正
#      10/18    print u""形式に全て修正。
#               ファイル読み込み時、"\n"行は、読み飛ばす
#      12/25    行末コードlsを修正
#   16/03/27    シェル要素（731、741）追加
#               createVtkFileResult:DISPLACEMENT内に回転が含まれている場合は、削除
#      04/08    シェル要素（761、781）追加
#      06/11    梁要素（641）追加
#      06/27    checkNanData:nodeDataが「NAN」の場合、「0.0」に修正する事を追加
#      11/08    getNodesAllArrayData:!NODE内のコメント行をskipするように修正
#   17/09/27    高速化の為、ファイルの書き出し方法を修正
#      10/08    getNodeNo:バグ修正（未使用のnodeNoも取得していた。）
#   18/03/28    高速化の為、メッシュデータの取得方法を修正
#      04/01    createVtkNodeNoFields:未使用nodeの削除追加
#      04/05    createVtkCellGrpFields:バグ修正。（要素Grpが正しく表示されない）
#      04/06    getDummyNodes:追加。（dummyNodeを相手nodeの値に置き換え、dummyを隠す。）
#      11/03    getElementalResults:resファイル内の要素Noの並びが、mshファイルの要素並び
#               と異なっているので、要素Noを基準に読み取る様に修正。
#               （要素タイプが異なる要素で構成した場合、並びが異なってくる）
#      12/02    getElementalResults:dummyの要素が有る場合、エラー発生するので修正。
#      12/14    renumberNodeElement:grpName毎にgrpNoを設定
#               同一要素grp名内に異なる要素typeが存在する場合に対応
#   19/01/06    python3用に書き換え
#      03/04    renumberNodeElement:要素GrpNoの設定方法修正。（出現順）
#      03/07    main:旧の結果ファイルが残っていた場合、変換エラーが発生するので、
#               エラー処理を追加
#      05/03    vsCodeによる修正
#   20/03/01    FrontISTR-5用に書き換え（vtu形式）
#      04/24    meshConvertToVtu, cpuConvertVtu:DISPLACEMENTのデータ数を確認・修正
#               shellの場合、DISPLACEMENTに回転を含む場合がある。 
#      07/10    renumberNodeElement:要素データの取得をelementListからに変更
#               取得元をmeshHeaderNumContsからelementListに変更。
#   21/02/25    getNodalResults:ElementalNQMからNodalBeamAbsForce, NodalBeamAbsMoment
#               の作成を追加。
#      05/26    remakeCellFields:Fistr-5.2では、resファイルのcell数とvtuファイル内のcell数が
#               異なっている（solid、shellの様な混合モデルの場合）ので、
#               resファイルに合わせる様に修正。
#      07/08    getNodalResults:バグ修正。戻り値の個数が統一されていなかった。
#   22/02/17    meshのみの変換の場合、nodeNoとelementNoを追加。
#               結果を含む場合は、追加されない。
#      02/18    elementNoのfieldを修正（surfaceのcellにもelNoをセット）
#               fistr2vtk.pyと同じ出力に修正（meshのみ変換の場合）
#      03/15    国際化
#      04/12    log表示修正（import logFileCreaterを追加）
#      04/26    log表示削除
#   23/01/08    multiMeshConvertToVtu:追加。
#               変換fileが多数ある場合、複数fileを並列処理する様に変更。
#      04/18    「-vis」optionを追加。（visのheaderが変更できる様にした為）
#      11/22    option取得時:並列計算結果のnCpuとvtu変換処理のnCpuを同じ値
#               として処理していた為、procNo=0しか変換しない事が発生。（バグ）
#               multiMeshConvertToVtu:getResVisMatchFilesを追加。
#               res,vis両者とも存在するfileのみ取得する。
#   24/07/06    singleCore時にエラー発生する時あり、修正。
#      07/07    openにencoding="utf-8"を追加
#      10/18    convert1file:print内容を修正。複数のprocsが同時にprintする
#               ので、\nの出力が遅れる事がある為、print内容を\nを含める。
#


import os
import sys
import glob
import multiprocessing
import convertMesh as cm


ls = "\n"

localeDir = os.getenv("LOCALE_DIR")

#以下は、bufferサイズを「0」にする設定
#  subprocess等で起動した時、print内容をbufferに溜め込んでrealTimeに出力されない
#  ので、以下の設定でbufferサイズを「0」にする。
#  又は、起動時に「python3 -u」オプション付きで起動する。
#sys.stdout = os.fdopen(sys.stdout.fileno(), 'w', buffering=1)
#sys.stderr = os.fdopen(sys.stderr.fileno(), 'w', buffering=1)
#sys.stdin = os.fdopen(sys.stdin.fileno(), 'r', buffering=1)



#
#  readFileLines
def readFileLines(fileName):
    """ fileをlines(list)形式で読込返す。"""
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    return lines

#
#  writeFileLines
def writeFileLines(fileName, lines):
    """ linesを保存する"""
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  deleteSp
def deleteSp(line):
    """ 空白とCRを削除して返す"""
    line = line.replace(" ", "")
    line = line.replace(ls, "")
    return line

#
#  getValName
def getValName(line, valName, sep=","):
    """ valName(type等）を取得して返す"""
    if sep == " ":
        words = line.split()
    else:
        line = deleteSp(line)
        words = line.split(sep)
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        if word.find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    return name

#
#  getNumberOfSurfaces
def getNumberOfSurfaces(vsuGrps):
    """ surfaceGroupからfaceの数を取得して返す"""
    nFaces = 0
    for grp in vsuGrps:
        nFaces += len(grp[1])
    return nFaces

#----------- vtu作成 ------------------
#
#  getLinesWordsN
def getLinesWordsN(words, n):
    """ n words毎に改行して返す"""
    lines = []
    for i in range(0, len(words), n):
        line = " ".join(words[i:i + n]) + ls
        lines.append(line)
    return lines

#
#  createVtkHeader
def createVtkHeader():
    """ vtkファイルのheader部を作成して返す"""
    lines  = ['<?xml version="1.0"?>' + ls]
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0">' + ls]
    lines += ['<UnstructuredGrid>' + ls]
    return lines

#
#  createVtkFooter
def createVtkFooter():
    """ vtkファイルのfooterを作成して返す"""
    lines  = ['</UnstructuredGrid>' + ls]
    lines += ['</VTKFile>' + ls]
    return lines

#
#  createVtkPieceHeader
def createVtkPieceHeader(vnList, veList, vsuGrps):
    """ pieceSectionのheaderを設定"""
    nNodes = len(vnList)
    nCells = len(veList)
    nFaces = getNumberOfSurfaces(vsuGrps)
    lines = ['<Piece NumberOfPoints="' + str(nNodes) + '" NumberOfCells="' + str(nCells + nFaces) + '">' + ls]
    return lines

#
#  createVtkPieceFooter
def createVtkPieceFooter():
    """ pieceSectionのfooterを設定"""
    lines = ['</Piece>' + ls]
    return lines

#
#  createVtkNodes
def createVtkNodes(vnList):
    """ node部を作成"""
    lines = ['<Points>' + ls]
    lines += ['<DataArray type="Float32" NumberOfComponents="3" format="ascii">' + ls]
    for nData in vnList:
        loc = nData[0]
        loc = list(map(lambda x: str(x), loc))
        line = " ".join(loc) + ls
        lines.append(line)
    lines += ['</DataArray>' + ls]
    lines += ['</Points>' + ls] 
    return lines

#
#  createVtkCells
def createVtkCells(veList, vsuGrps):
    """ cell部を作成する"""
    #node順入れ替え用の辞書
    changeOrderDict = {
     #node数 新order  入れ替え関数名
        3:[[0,1,2],   cm.order_3z],
        4:[[0,1,2,3], cm.order_4z],
        6:[[0,2,4,1,3,5], cm.order_6z],
        8:[[0,2,4,6,1,3,5,7], cm.order_8z]
        }
    #node数からface要素名を取得する辞書
    #  面要素のnode数とelType名
    faceNumToFistrVtkTypeDict = {
        #node数: fistr名、vtk名
            3:   ["231",  "5"],
            4:   ["241",  "9"],
            6:   ["232",  "22"],
            8:   ["242",  "23"]
        }

    lines = ['<Cells>' + ls]
    #elementを作成
    lines += ['<DataArray type="Int32" Name="connectivity" format="ascii">' + ls]
    for eData in veList:
        eNodes = eData[0][1:]
        eNodes = list(map(lambda x: str(x), eNodes))
        line = " ".join(eNodes) + ls
        lines.append(line)
    #surface要素を作成
    cells = []
    for grpData in vsuGrps:
        suNodes = grpData[1]
        for nodes in suNodes:
            #2次要素の場合、node順を入れ替え
            #  左回り順　→　2次要素順（頂点→中間点）
            [newOrder, func] = changeOrderDict[len(nodes)]
            cell = func(nodes, newOrder)
            cells.append(cell)
            words = list(map(str, cell))    #cellをstrのlistに変換
            line = " ".join(words) + ls
            lines.append(line)
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="Int32" Name="offsets" format="ascii">' + ls]
    #offsetを作成
    offset = 0
    data = []
    #  solid要素
    for eData in veList:
        eNodes = eData[0][1:]
        offset += len(eNodes)
        data.append(str(offset))
    #  surfaceを作成
    for cell in cells:
        offset += len(cell)
        data.append(str(offset))
    #lines += [" ".join(data) + ls]
    lines += getLinesWordsN(data, 10)      #10 words毎に改行する
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="UInt8" Name="types" format="ascii">' + ls]
    #typesを作成
    data = []
    #  solid要素
    for eData in veList:
        elType = eData[1]
        data.append(elType)
    #  surface要素
    for cell in cells:
        elType = faceNumToFistrVtkTypeDict[len(cell)][1]
        data.append(elType)
    #lines += [" ".join(data) + ls]
    lines += getLinesWordsN(data, 20)
    lines += ['</DataArray>' + ls]
    lines += ['</Cells>' + ls]
    return lines

#
#  createVtkNodesData
def createVtkNodesData(vnList, vndGrps):
    """ nodesData(nodeGroupNo)を作成"""
    #nodeNo
    nodeNos = list(map(lambda x: str(x[2]), vnList))
    #ndGrpNoを設定
    nNodes = len(vnList)
    ndNos = [ 0 for i in range(nNodes) ]
    for i in range(len(vndGrps)):
        ndGrpNo = i + 1
        nodes = vndGrps[i][1]
        for ndNo in nodes:
            ndNos[ndNo] = ndGrpNo
    #linesを作成
    lines = ['<PointData>' + ls]
    lines += ['<DataArray type="Int32" Name="nodeNo" NumberOfComponents="1" format="ascii">' + ls]
    lines += getLinesWordsN(nodeNos, 10)
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="Int16" Name="nodeGroup" NumberOfComponents="1" format="ascii">' + ls]
    ndGrpNos = list(map(lambda x:str(x), ndNos))
    lines += getLinesWordsN(ndGrpNos, 20)
    lines += ['</DataArray>' + ls]
    lines += ['</PointData>' + ls]
    return lines

#
#  createVtkCellsData
def createVtkCellsData(veList, vsuGrps, surGrps):
    """ cellData(elGroupNo, suGroupNo)を作成"""
    #faceを含めた全cell数を取得
    nFaces = 0
    for grpData in vsuGrps:
        nFaces += len(grpData[1])
    nCells = len(veList) + nFaces
    #elemtntNoをセット
    elNos = [ 0 for i in range(nCells)]
    #  elementのcellにelNoをセット
    for newElNo in range(len(veList)):
        elNo = veList[newElNo][2]
        elNos[newElNo] = elNo
    #  surfaceのcellにelNoをセットする
    newElNo += 1
    for i in range(len(surGrps)):
        grpData = surGrps[i]
        elmNos = grpData[1]
        for elNo in elmNos:
            elNos[newElNo] = elNo
            newElNo += 1
    #solid要素の要素grpNoをセット
    elGrpNos = [ 0 for i in range(nCells)]
    for elNo in range(len(veList)):
        elGrpNo = veList[elNo][0][0]
        elGrpNos[elNo] = elGrpNo
    #surfaceのgrpNoをセット
    suGrpNos = [ 0 for i in range(nCells)]
    faceNo = len(veList) - 1
    for i in range(len(vsuGrps)):
        grpData = vsuGrps[i]
        faces = grpData[1]
        suGrpNo = i + 1
        for _ii in range(len(faces)):
            faceNo += 1
            suGrpNos[faceNo] = suGrpNo
    #linesを作成
    lines = ['<CellData>' + ls]
    lines += ['<DataArray type="Int32" Name="elementNo" NumberOfComponents="1" format="ascii">' + ls]
    elNos = list(map(lambda x: str(x), elNos))
    lines += getLinesWordsN(elNos, 10)
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="Int16" Name="elementGroup" NumberOfComponents="1" format="ascii">' + ls]
    elGrpNos = list(map(lambda x: str(x), elGrpNos))
    lines += getLinesWordsN(elGrpNos, 20)
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="Int16" Name="surfaceGroup" NumberOfComponents="1" format="ascii">' + ls]
    suGrpNos = list(map(lambda x: str(x), suGrpNos))
    lines += getLinesWordsN(suGrpNos, 20)
    lines += ['</DataArray>' + ls]
    lines += ['</CellData>' + ls]
    return lines

def getDtypeFromName(name):
    """ Nameからdtypeを返す。"""
    dtype = "Float64"
    if name == "elementGroup":
        dtype = "Int32"
    return dtype

#
#  createVtkAddValue
def createVtkAddValue(fields):
    """ 追加するvtkDataArrayを作成する"""
    lines = []
    for field in fields:
        name = field[0]
        data = field[1]
        dtype = getDtypeFromName(name)
        nComp = len(data[0])
        lines += ['<DataArray type="' + dtype + '" Name="' + name + '" NumberOfComponents="' + str(nComp) + '" format="ascii">' + ls]
        for val in data:
            lines += [" ".join(val) + ls]
        lines += ['</DataArray>' + ls]
    return lines

#
#  insertVtkPointData
def insertVtkPointData(lines, addLines):
    """ pointDataにaddLinesを追加する"""
    for i in range(len(lines)):
        line = lines[i]
        if line.find("</PointData>") >= 0:
            lines = lines[:i] + addLines + lines[i:]
            break
    return lines

#
#  insertVtkCellData
def insertVtkCellData(lines, addLines):
    """ cellDataにaddLinesを追加する"""
    for i in range(len(lines)):
        line = lines[i]
        if line.find("</CellData>") >= 0:
            lines = lines[:i] + addLines + lines[i:]
            break
    return lines


#--------------------------------------

#
#  getNewNodeNo
#----------------
def getNewNodeNo(nodeList, meshHeaderNumConts):
    """ 未使用のnodeNoを調べ、新しいnodeNoを設定（「0」始まり）。
    nodeNoは、出現順に設定する。（結果dataが出現順の為）
    
    Args:
        nodeList (list)             :旧nodeList
        meshHeaderNumConts (list)   :header毎のnumData
    Returns:
        nodeList (list) :使用有無、新nodeNo追加
        vnList (list)   :vtk用のnodeList"""
    vnList = []
    newNo = 0
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!NODE":
            nodeData = headerCont[1]
            for nodeCont in nodeData:
                nodeNo = nodeCont[0]
                if len(nodeList[nodeNo][1]) > 0:    #使用有無チェック
                    loc = nodeCont[1:]
                    nodeList[nodeNo].append(newNo)
                    vnList.append([loc, newNo, nodeNo])    #[座標, newNodeNo, 旧nodeNo]
                    newNo += 1
    return [nodeList, vnList]

#
#  renumberNodeElement
#----------------------
def renumberNodeElement(nodeList, elementList, meshHeaderNumConts):
    """ nodeNoとelNoを書き換え。
    elNoは、出現順に設定する。（結果dateが出現順の為）
    
    Args:
        nodeList (list)             :旧nodeList
        elementList (list)          :旧elementList
        meshHeaderNumConts (list)   :header毎のnumData
    Returns:
        elementList (list)  :新elNoを追加
        veList (list)       :vtk用のelNo
                            :[[<grpNo>, nodeNos], <elType>, <旧elNo>],..."""
    vtkNodesToElmNoDict = {}
    #elNoとgrpNoを設定
    veList = []
    newNo = 0
    for elNo in range(len(elementList)):
        if len(elementList[elNo]) > 0:
            elm = elementList[elNo]
            grpNo = elm[0][0]
            elNodes = elm[0][1:]
            elType = elm[1]
            newNodes = []
            for elNode in elNodes:
                newNodeNo = nodeList[elNode][-1]
                newNodes.append(newNodeNo)
            #3自由度のbeam、シェル（三角形、四角形）の場合
            #  dummyNodeを削除して変換
            if elType == "641":
                newNodes = newNodes[:len(newNodes)//2]
                elType = "611"
            elif elType == "761":
                newNodes = newNodes[:len(newNodes)//2]
                elType = "731"
            elif elType == "781":
                newNodes = newNodes[:len(newNodes)//2]
                elType = "741"
            [vtkName, newNodes] = cm.fistr2vtk_el(elType, newNodes)
            newElCont = [[grpNo] + newNodes] + [vtkName] + [elNo]
            veList.append(newElCont)
            elementList[elNo] += [newNo]
            newNo += 1
            key = tuple(newNodes)
            vtkNodesToElmNoDict[key] = elNo
    return [elementList, veList, vtkNodesToElmNoDict]

#
#  getDummyNodes
#----------------
def getDummyNodes(elementList, nodeList, meshHeaderNumConts):
    """ dummy節点とその相手nodeNoを取得"""
    dummyNodes = []
    for meshCont in meshHeaderNumConts:
        header = meshCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            elType = getValName(header, "TYPE")
            #dummyNodeを取得
            if elType == "761":
                #三角形
                elmData = meshCont[1]
                for elm in elmData:
                    nodes = elm[1:]
                    dummyNodes.append([nodeList[nodes[3]][2], nodeList[nodes[0]][2]])
                    dummyNodes.append([nodeList[nodes[4]][2], nodeList[nodes[1]][2]])
                    dummyNodes.append([nodeList[nodes[5]][2], nodeList[nodes[2]][2]])
            elif elType == "781":
                #四角形
                elmData = meshCont[1]
                for elm in elmData:
                    nodes = elm[1:]
                    dummyNodes.append([nodeList[nodes[4]][2], nodeList[nodes[0]][2]])
                    dummyNodes.append([nodeList[nodes[5]][2], nodeList[nodes[1]][2]])
                    dummyNodes.append([nodeList[nodes[6]][2], nodeList[nodes[2]][2]])
                    dummyNodes.append([nodeList[nodes[7]][2], nodeList[nodes[3]][2]])
            elif elType == "641":
                #beam
                elmData = meshCont[1]
                for elm in elmData:
                    nodes = elm[1:]
                    dummyNodes.append([nodeList[nodes[2]][2], nodeList[nodes[0]][2]])
                    dummyNodes.append([nodeList[nodes[3]][2], nodeList[nodes[1]][2]])
    return dummyNodes

#
#  renumberGrps
#--------------
def renumberGrps(meshHeaderNumConts, nodeList, elementList):
    """ node, surfaceのgroupを取得する
    
    Args:
        meshHeaderNumConts  (list)  :meshのheaderとdataのペアlist
        nodeList (list)             :nodeNoをindexとした座標のlist
        elementList (list)          :elementNoをindexとしたnodeNoのlist
    Returns:
        vndGrps (list)  :[<grpName>, [nodeNoのlist]]
        vsuGrps (list)  :[<grpName>, [[面のnodeNos], ..]]
        surGrps (list)  :[<grpName>, [elementNoのlist]]"""
    vndGrps = []
    vsuGrps = []
    surGrps = []
    for headerData in meshHeaderNumConts:
        header = headerData[0]
        words = deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            #nodeGrpを取得
            grpName = getValName(header, "NGRP")
            nodes = headerData[1]
            newNodes = []
            for node in nodes:
                newNo = nodeList[node][-1]
                newNodes.append(newNo)
            grp = [grpName, newNodes]
            vndGrps.append(grp)
        elif words[0] == "!SGROUP":
            #surfaceGrpを取得
            grpName = getValName(header, "SGRP")
            suData = headerData[1]
            grpData = []
            grpElData = []
            for i in range(0, len(suData), 2):
                elNo = suData[i]
                faceNo = suData[i+1]
                newNo = elementList[elNo][-1]
                elType = elementList[elNo][1]
                elNodes = elementList[elNo][0][1:]
                #faceNodeを左回りで連続して取得
                nodes = cm.fistr_elFaceNode(elType, elNodes, faceNo)
                #nodeNoをvtkのnodeNoに置き換え
                newNodes = []
                for node in nodes:
                    newNo = nodeList[node][-1]
                    newNodes.append(newNo)
                #vtkNodeNoを保存
                grpData.append(newNodes)
                grpElData.append(elNo)
            #grpNameと共にnodeNoを保存
            grp = [grpName, grpData]
            vsuGrps.append(grp)
            surGrps.append([grpName, grpElData])
    return [vndGrps, vsuGrps, surGrps]

#
#  createVtkFileMesh
#---------------------
def createVtkFileMesh(vnList, veList, vndGrps, vsuGrps, surGrps):
    """ vtkファイルのmeshデータを作成"""
    #vtk行を作成
    newLines = createVtkHeader()
    newLines += createVtkPieceHeader(vnList, veList, vsuGrps)
    newLines += createVtkNodes(vnList)
    newLines += createVtkCells(veList, vsuGrps)
    newLines += createVtkNodesData(vnList, vndGrps)
    newLines += createVtkCellsData(veList, vsuGrps, surGrps)
    newLines += createVtkPieceFooter()
    newLines += createVtkFooter()
    return newLines

#
#  getNqmNordalField
def getNqmNordalField(nqmField, elementList, nodeToOrderDict, orderToElmDict):
    """ nqmのnodalFieldを作成する"""
    nqmData = nqmField[1]
    #nodalForceを取得
    nNodes = len(nodeToOrderDict.keys())
    nqmNodalForce = [["0.0", "0.0", "0.0"] for i in range(nNodes)]
    moment = [["0.0", "0.0", "0.0"] for i in range(nNodes)]
    for i in range(len(nqmData)):
        elmNo = orderToElmDict[i]
        if elmNo < len(elementList):
            #実際のelementのみ確認（EQUATIONで要素が追加される）
            nodesNos = elementList[elmNo][0][1:]
            node1 = nodesNos[0]
            node2 = nodesNos[1]
            idx1 = nodeToOrderDict[node1]
            idx2 = nodeToOrderDict[node2]
            vals = nqmData[i]
            #軸力
            val1 = abs(float(vals[0]))
            val2 = abs(float(vals[3]))
            if val1 > float(nqmNodalForce[idx1][0]):
                nqmNodalForce[idx1][0] = str(val1)
            if val2 > float(nqmNodalForce[idx2][0]):
                nqmNodalForce[idx2][0] = str(val2)
            #Y方向せん断
            val1 = abs(float(vals[1]))
            val2 = abs(float(vals[4]))
            if val1 > float(nqmNodalForce[idx1][1]):
                nqmNodalForce[idx1][1] = str(val1)
            if val2 > float(nqmNodalForce[idx2][1]):
                nqmNodalForce[idx2][1] = str(val2)
            #Z方向せん断
            val1 = abs(float(vals[2]))
            val2 = abs(float(vals[5]))
            if val1 > float(nqmNodalForce[idx1][2]):
                nqmNodalForce[idx1][2] = str(val1)
            if val2 > float(nqmNodalForce[idx2][2]):
                nqmNodalForce[idx2][2] = str(val2)
            #ねじり
            val1 = abs(float(vals[6]))
            val2 = abs(float(vals[9]))
            if val1 > float(moment[idx1][0]):
                moment[idx1][0] = str(val1)
            if val2 > float(moment[idx2][0]):
                moment[idx2][0] = str(val2)
            #Y軸モーメント
            val1 = abs(float(vals[7]))
            val2 = abs(float(vals[10]))
            if val1 > float(moment[idx1][1]):
                moment[idx1][1] = str(val1)
            if val2 > float(moment[idx2][1]):
                moment[idx2][1] = str(val2)
            #Z軸モーメント
            val1 = abs(float(vals[8]))
            val2 = abs(float(vals[11]))
            if val1 > float(moment[idx1][2]):
                moment[idx1][2] = str(val1)
            if val2 > float(moment[idx2][2]):
                moment[idx2][2] = str(val2)
    nqmNodalForceField = ["NodalBeamAbsForce", nqmNodalForce]
    nqmNodalMomentField = ["NodalBeamAbsMoment", moment]   
    nodalFields = [nqmNodalForceField, nqmNodalMomentField]
    return nodalFields

#
#  getNodalResults
def getNodalResults(lines, elementList):
    """ nodeのdataを取得する"""

    def getGrpNoFromElmNo(elmNo):
        """ elmNoから要素grpNoを取得する"""
        if elmNo >= len(elementList):
            grpNo = "0"
        elif len(elementList[elmNo]) == 0:
            grpNo = "0"
        else:
            grpNo = str(elementList[elmNo][0][0])
        return grpNo

    nodalFields = []
    elmFields = []
    #*dataを検索
    flag = 0
    for i in range(len(lines)):
        if lines[i][:len("*data")] == "*data":
            flag = 1
            break
    if flag == 0:
        print("error: nodalResult が読み取れません")
        return [nodalFields, elmFields, {}]

    #node数、要素数を取得
    i += 1
    words = lines[i].split()
    (nNodes, nElms) = (int(words[0]), int(words[1]))
    #変数の数を取得
    i += 1
    words = lines[i].split()
    (nNodalVals, nElmVals) = (int(words[0]), int(words[1]))

    #nodal側を取得-------------------
    #nodal変数の桁数を取得
    i += 1
    words = lines[i].split()
    comps = list(map(lambda x: int(x), words))
    nFloat = sum(comps)
    #nodal変数名を取得
    i += 1
    valNames = []
    for _ii in range(nNodalVals):
        valName = lines[i].split()[0]
        valNames.append(valName)
        i += 1
    #nodal変数の配列を準備
    nodalVals = [ [] for ii in range(nNodalVals)]
    #各nodal変数の値を取得
    nodeToOrderDict = {}
    for ii in range(nNodes):
        nodeNo = int(lines[i])
        nodeToOrderDict[nodeNo] = ii
        i += 1
        vals = []
        loop = True
        while loop:
            # NANをチェックし、NANがあれば、0.0に置き換える
            line = lines[i].replace(" NAN ", " 0.0 ")
            words = line.split()
            vals += words
            i += 1
            if len(vals) >= nFloat:
                break
        #値を取得
        st = 0
        for jj in range(len(comps)):
            ed = st + comps[jj]
            val = vals[st:ed]
            nodalVals[jj].append(val)
            st += comps[jj]
    #field名をセット
    for ii in range(nNodalVals):
        valName = valNames[ii]
        vals = nodalVals[ii]
        nodalFields.append([valName, vals])

    #要素側を取得----------------------
    if i >= len(lines):
        return [nodalFields, elmFields, {}]
    grpVals = []
    #桁数を取得
    words = lines[i].split()
    comps = list(map(lambda x: int(x), words))
    nFloat = sum(comps)
    #変数名を取得
    i += 1
    valNames = []
    for _ii in range(nElmVals):
        valName = lines[i].split()[0]
        valNames.append(valName)
        i += 1
    #変数の配列を準備
    elmVals = [ [] for ii in range(nElmVals)]
    #各変数の値を取得
    orderToElmDict = {}         #順位Noをkeyとして要素Noを返すDict
    elmToOrderDict = {}         #要素Noをkeyとして順位Noを返すDict
    for ii in range(nElms):
        #grpNoを取得
        elmNo = int(lines[i])
        orderToElmDict[ii] = elmNo
        elmToOrderDict[elmNo] = ii
        grpNo = getGrpNoFromElmNo(elmNo)
        grpVals.append([grpNo])
        #要素データを取得
        i += 1
        vals = []
        loop = True
        while loop:
            # NANをチェックし、NANがあれば、0.0に置き換える
            line = lines[i].replace(" NAN ", " 0.0 ")
            words = line.split()
            vals += words
            i += 1
            if len(vals) >= nFloat:
                break
        #値を取得
        st = 0
        for jj in range(len(comps)):
            ed = st + comps[jj]
            val = vals[st:ed]
            elmVals[jj].append(val)
            st += comps[jj]
    #field名をセット
    for ii in range(nElmVals):
        valName = valNames[ii]
        vals = elmVals[ii]
        elmFields.append([valName, vals])
    elmFields.append(["elementGroup", grpVals])
    #ElementalNQMを確認
    if "ElementalNQM" in valNames:
        if not("NodalBeamAbsForce" in valNames):
            nqmField = list(filter(lambda x: x[0] == "ElementalNQM", elmFields))[0]
            nqmNodalFields = getNqmNordalField(nqmField, elementList, nodeToOrderDict, orderToElmDict)
            nodalFields += nqmNodalFields
    return [nodalFields, elmFields, elmToOrderDict]

#
#  getResultFields
#------------------
def getResultFields(fileName, elementList):
    """ 結果fieldをnodeとcellに分けて取得する"""
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    #nodeFieldを作成
    (nodeFields, elmFields, elmToOrderDict) = getNodalResults(lines, elementList)
    return (nodeFields, elmFields, elmToOrderDict)

#
#  getSinglePvtuFileName
def getSinglePvtuFileName(resFileName):
    step = resFileName.split(".")[-1]
    newStep = "0"*(4-len(step)) + step
    #vtuFile = resFile + ".vis_psf." + newStep + ".pvtu"
    #vtuFile = meshFile + ".vis_psf." + newStep + ".pvtu"
    #vtuFile = "FistrModel.vis_psf." + newStep + ".pvtu"
    vtuFile = visFile + "_psf." + newStep + ".pvtu"
    return vtuFile

#
#  getAddFields
def getAddFields(lines, nodeFields, cellFields):
    """ 足りないfieldを確認し、返す"""
    #pointData部を取得する
    nodeNames = []
    cellNames = []
    flag = 0
    for i in range(len(lines)):
        line = lines[i]
        if line.find("<PPointData") >= 0:
            flag = 1
        elif line.find("<PCellData") >= 0:
            flag = 2
        if flag == 1:
            if line.find("<PDataArray") >= 0:
                valName = getValName(line, 'Name', sep=" ")
                if valName != "":
                    valName = valName[1:-1]
                    nodeNames.append(valName)
            elif line.find("</PPointData") >= 0:
                flag = 0
        elif flag == 2:
            if line.find("<PDataArray") >= 0:
                valName = getValName(line, "Name", sep=" ")
                if valName != "":
                    valName = valName[1:-1]
                    cellNames.append(valName)
            elif line.find("</PCellData") >= 0:
                flag = 1
    #追加するfield名を取得
    #  node側
    setNames = list(map(lambda x: x[0], nodeFields))
    addFields = list(set(setNames).difference(nodeNames))
    addNodeFields = []
    for fieldName, fieldData in nodeFields:
        if fieldName in addFields:
            addNodeFields.append([fieldName, fieldData])
    #  cell側
    setNames = list(map(lambda x: x[0], cellFields))
    addFields = list(set(setNames).difference(cellNames))
    addCellFields = []
    for name, data in cellFields:
        if name in addFields:
            addCellFields.append([name, data])
    #field名にdataを追加
    return (addNodeFields, addCellFields)

#
#  getSingleVtuFileName
def getSingleVtuFileName(lines):
    """ vtuファイル名を取得する"""
    vtuFile = ""
    for line in lines:
        if line.find("<Piece") >= 0:
            vtuFile = getValName(line, "Source", sep=" ")
            sep = vtuFile[0]
            ed = vtuFile.find(sep, 1)
            vtuFile = vtuFile[1:ed]
            break
    return vtuFile

#
#  addFieldsToPVtuFile
def addFieldsToPVtuFile(lines, addFields):
    """ pvtuFileにfield名を追加する"""
    (addNodeFields, addCellFields) = addFields
    #node側を確認
    #  addLinesを作成
    addLines = []
    for field in addNodeFields:
        name = field[0]
        dtype = getDtypeFromName(name)
        nComp = str(len(field[1][0]))
        addLines += ['<PDataArray type="' + dtype + '" Name="' + name + '" NumberOfComponents="' + nComp + '" format="ascii"/>' + ls]
    #  addLinesを挿入
    for i in range(len(lines)):
        line = lines[i]
        if line.find("</PPointData>") >= 0:
            lines = lines[:i] + addLines + lines[i:]
            break
    #cell側を確認
    #  addLinesを作成
    addLines = []
    for field in addCellFields:
        name = field[0]
        dtype = getDtypeFromName(name)
        nComp = str(len(field[1][0]))
        addLines += ['<PDataArray type="' + dtype + '" Name="' + name + '" NumberOfComponents="' + nComp + '" format="ascii"/>' + ls]
    #  addLinesを挿入
    for i in range(len(lines)):
        line = lines[i]
        if line.find("</PCellData>") >= 0:
            lines = lines[:i] + addLines + lines[i:]
            break
    return lines

#
#  remakeCellFields
def remakeCellFields(lines, addCellFields, elmToOrderDict, vtkNodesElmNoDict):
    """ cellFieldの内容を修正する。
    resファイル内のcell数とvtuファイル内のcell数が合致していれば、そのまま。"""
    #  vtu側のcell数(nCells)を取得
    for line in lines:
        n = line.find("NumberOfCells") 
        if n > 0:
            st = line[n:].find('"') + n + 1
            ed = line[st:].find('"') + st
            nCells = int(line[st:ed])
            break
    #cell数を比較
    resCellField = addCellFields[0]
    if len(resCellField[1]) == nCells:
        #cell数がresファイルとvtuファイルで同じ場合は戻る。
        return addCellFields

    #cellFieldの内容を修正
    #  cellのconnectivityを取得(elConnectListを作成)
    for i in range(len(lines)):
        line = lines[i]
        n = line.find('Name="connectivity"')
        if n > 0:
            st = i + 1
            break
    for i in range(st, len(lines)):
        line = lines[i]
        n = line.find("</DataArray>")
        if n >= 0:
            ed = i
            break
    dataLines = lines[st:ed]
    elConnectList = []
    for i in range(len(dataLines)):
        line = dataLines[i]
        nodes = tuple(map(int, line.split()))
        elConnectList.append(nodes)
    #  値を再セット
    for nn in range(len(addCellFields)):
        name, cellField = addCellFields[nn]
        nCompo = len(cellField[0])
        dtype = getDtypeFromName(name)
        if dtype[:len("Float")] == "Float":
            val = [ "0.0" for i in range(nCompo) ]
        else:
            val = [ "0" for i in range(nCompo) ]
        #fieldを作成
        data = [ val for i in range(nCells) ]
        for i in range(nCells):
            nodes = elConnectList[i]
            if nodes in vtkNodesElmNoDict.keys():
                elNo = vtkNodesElmNoDict[nodes]
                ii = elmToOrderDict[elNo]
                fieldVal = cellField[ii]
                data[i] = fieldVal
        addCellFields[nn][1] = data
    return addCellFields

#
#  addFieldsToVtuFile
def addFieldsToVtuFile(vtuFile, addFields, elmToOrderDict, vtkNodesElmNoDict):
    """ vtuFileにfieldを追加する。"""
    (addNodeFields, addCellFields) = addFields
    lines = readFileLines(vtuFile)
    #node側を確認
    addLines = createVtkAddValue(addNodeFields)
    lines = insertVtkPointData(lines, addLines)
    #cell側を確認
    if len(addCellFields) > 0:
        addCellFields = remakeCellFields(lines, addCellFields, 
                            elmToOrderDict, vtkNodesElmNoDict)
    #addLine作成
    addLines = createVtkAddValue(addCellFields)
    lines = insertVtkCellData(lines, addLines)
    #書き込み
    writeFileLines(vtuFile, lines)

#
#  addFieldsToVtkFile
#---------------------
def addFieldsToVtkFile(fileName, nodeFields, cellFields, elmToOrderDict, vtkNodesElmNoDict):
    """ vtuファイルに足りないfieldを追加する"""
    pvtuFile = getSinglePvtuFileName(fileName)
    lines = readFileLines(pvtuFile)
    addFields = getAddFields(lines, nodeFields, cellFields)
    #(addNodeFields, addCellFields) = addFields
    lines = addFieldsToPVtuFile(lines, addFields)
    writeFileLines(pvtuFile, lines)
    vtuFile = getSingleVtuFileName(lines)
    addFieldsToVtuFile(vtuFile, addFields, elmToOrderDict, vtkNodesElmNoDict)

#
#  getNumOfDispplacementData
def getNumOfDisplacementData(lines):
    """ pvtuファイル内のDISPLACEMENTのデータ数を確認修正"""
    nData = 3
    for i in range(len(lines)):
        line = lines[i]
        if line.find("<PDataArray") >= 0:
            words = line.split()
            flag = 0
            for word in words:
                nameVal = word.split("=")
                if nameVal[0] == "Name":
                    if nameVal[1] == '"DISPLACEMENT"':
                        flag = 1
                        break
                    else:
                        break
            if flag == 1:
                for ii in range(len(words)):
                    nameVal = words[ii].split("=")
                    if nameVal[0] == "NumberOfComponents":
                        if nameVal[1] == '"3"':
                            nData = 3
                            break
                        else:
                            nData = int(nameVal[1][1:-1])
                            words[ii] = 'NumberOfComponents="3"'
                            newLine = " ".join(words) + ls
                            lines[i] = newLine
                            break
                break
    return (nData, lines)

#
#  remakeDisplacement
def remakeDisplacement(lines):
    """ vtuファイル内のDISPLACEMENTのデータ数を3ヶに修正する。"""
    flag = 0
    for i in range(len(lines)):
        line = lines[i]
        if line.find("<DataArray") >= 0:
            flag = 0
            if line.find("DISPLACEMENT") >= 0:
                words = line.split()
                for ii in range(len(words)):
                    nameVal = words[ii].split("=")
                    if nameVal[0] == "NumberOfComponents":
                        words[ii] = 'NumberOfComponents="3"'
                        newLine = " ".join(words) + ls
                        lines[i] = newLine
                        flag = 1
                        break
        else:
            if flag == 1:
                words = line.split()
                newLine = " ".join(words[:3]) + ls
                lines[i] = newLine
    return lines

#
#  checkRemakeDisplacement
#---------------------------
def checkRemakeDisplacement(fileName):
    """ DISPLACEMENTのデータ数を確認し、
    データ数が3以上の場合は、3ヶに縮小する。
    変位のみ残し、回転は削除。（shellの場合、回転が含まれる）"""
    pvtuFile = getSinglePvtuFileName(fileName)
    lines = readFileLines(pvtuFile)
    nData, lines = getNumOfDisplacementData(lines)
    if nData == 3:
        return
    #pvtuFileを修正
    writeFileLines(pvtuFile, lines)
    vtuFile = getSingleVtuFileName(lines)
    lines = readFileLines(vtuFile)
    lines = remakeDisplacement(lines)
    writeFileLines(vtuFile, lines)

#
#  getVtuMeshType
def getVtuMeshTypeGrp(lines):
    """ vtuファイル中のMesh_Typeを取得する。"""
    #Mesh_Typeの行を取得
    flag = 0
    typeLines = []
    for i in range(len(lines)):
        line = lines[i]
        if line.find('"Mesh_Type"') >= 0:
            flag = 1
        else:
            if flag == 1:
                if line.find("</DataArray>") >= 0:
                    break
                else:
                    typeLines.append(line)
    #要素typeをlistで取得
    meshTypes = []
    for line in typeLines:
        words = line.split()
        meshTypes += words
    return meshTypes

#
#  getMeshFileName
def getMeshFileName(headerName):
    """ headerNameからmshファイルを取得して返す。
    ファイルが存在しない場合は、「""」を返す"""
    if headerName.split(".")[-1] == "msh":
        fileName = headerName
    else:
        fileName = headerName + ".msh"
    if os.path.exists(fileName) == False:
        print("error: " + "「" + headerName + "」が存在しません。")
        fileName = ""
    return fileName

#
#  getParallelResultFileNames
def getParallelResultFileNames():
    """ parallel処理の場合、resFileから結果fileNameを取得して返す"""
    names = names = glob.glob(resFile + ".res.*")
    fileNames = []
    for name in names:
        if name[-1] != "~" and name.split(".")[-1] != "inp":
            fileNames.append(name)
    return fileNames

#
#  getNcpuFromResFileNames
def getNcpuFromResFileNames(names):
    """ fileNameからnCpu数を取得する"""
    cpuNos = []
    for name in names:
        words = name.split(".")
        try:
            cpuNo = int(words[-2])
            cpuNos.append(cpuNo)
        except:
            pass
    nCpus = max(cpuNos) + 1
    return nCpus

#
#  getSingleResultFileNames
def getSingleResultFileNames():
    """ singleCoreの場合、resFileから結果fileNameを取得して返す"""
    names = glob.glob(resFile + ".res.0.*")
    fileNames = []
    for name in names:
        if name[-1] != "~" and name.split(".")[-1] != "inp":
            fileNames.append(name)
    return fileNames

#
#  sortStepFileName
def sortStepFileName(names):
    """ fileNameをstepでsortする。"""
    steps = []
    for name in names:
        step = int(name.split(".")[-1])
        steps.append(step)
    steps.sort()
    nameHeader = ".".join(names[0].split(".")[:-1])
    names = []
    for step in steps:
        fileName = nameHeader + "." + str(step)
        names.append(fileName)
    return names

#
#  renumberVeList
#-----------------
def renumberVeList(veList):
    """ veListをsolid、シェル、ビームの順に並べ替えて返す"""
    elDict = {
        "3": 2,     #"beam"
        "4": 2,     #"beam"
        "5": 1,     #"shell"
        "7": 1,     #"shell"
        "9": 1,     #"shell"
        "10": 0,    #"solid"
        "12": 0,    #"solid"
        "13": 0,    #"solid"
        "14": 0,    #"solid"
        "21": 2,    #"beam"
        "22": 1,    #"shell"
        "23": 1,    #"shell"
        "24": 0,    #"solid"
        "25": 0,    #"solid"
        "26": 0     #"solid"
        }
    newList = [[], [], []]
    for i in range(len(veList)):
        elm = veList[i]
        #newList[elDict[elm[-1]]].append(elm)
        newList[elDict[elm[1]]].append(elm)
    ve = newList[0] + newList[1] + newList[2]
    return ve

#
#  getAddFieldNames
def getAddFieldNames(paraFileNames, elementList):
    """ 追加するfield名を取得する"""
    paraFileNames.sort()
    fileName = paraFileNames[0]
    #resFile中のfield名を取得
    lines = readFileLines(fileName)
    (nodeFields, elmFields, elmToOrderDict) = getNodalResults(lines, elementList)
    #pvtu中のfield名を取得
    pvtuFile = getSinglePvtuFileName(fileName)
    lines = readFileLines(pvtuFile)
    #追加するnode、elm名を取得する
    addFields = getAddFields(lines, nodeFields, elmFields)
    return addFields

#
#  getvtuFiles
def getvtuFiles(pvtuFile):
    """ pvtuFileからvtuFileを取得する"""
    lines = readFileLines(pvtuFile)
    vtuFiles = []
    for line in lines:
        if line.find("<Piece") >= 0:
            vtuFile = getValName(line, "Source", sep=" ")
            sep = vtuFile[0]
            ed = vtuFile.find(sep, 1)
            vtuFile = vtuFile[1:ed]
            vtuFiles.append(vtuFile)
    #cpu順に並べ替え
    newNames = []
    for name in vtuFiles:
        cpu = int(name.split(".")[-2])
        newNames.append([cpu, name])
    newNames.sort()
    vtuFiles = list(map(lambda x: x[1], newNames))
    return vtuFiles

#
#  deleteDummyNodes
#----------------------
def deleteDummyNodes(vnList, veList, vndGrps, dummyNodes):
    """ dummyNodeGroupを削除する。"""
    #dummyNodesを削除する
    for i in range(len(dummyNodes)):
        (dummy, _node) = dummyNodes[i]  # (dmmyNodeNo, 相手NodeNo)
        vnList[dummy] = []      #vnListを空に設定
    #newNoを設定
    newNodeList = []
    n = 0
    for i in range(len(vnList)):
        if len(vnList[i]) > 0:
            newNodeList.append(vnList[i])
            #vnList[i].append(n)
            vnList[i][1] = n
            n += 1
    #veListのnodeNoを更新
    for i in range(len(veList)):
        for ii in range(len(veList[i][0][1:])):
            nodeNo = veList[i][0][ii+1]
            newNo = vnList[nodeNo][1]
            veList[i][0][ii+1] = newNo
    #vndGrpsのnodeNoを更新、dummyのGroupを削除
    newGrps = []
    for i in range(len(vndGrps)):
        grpName = vndGrps[i][0]
        if grpName[:len("dummy")] != "dummy":
            for ii in range(len(vndGrps[i][1])):
                nodeNo = vndGrps[i][1][ii]
                newNo = vnList[nodeNo][1]
                vndGrps[i][1][ii] = newNo
            newGrps.append(vndGrps[i])
    return (newNodeList, veList, newGrps)

#------------------------
#  multiMeshConvertToVtu
#------------------------
def multiMeshConvertToVtu():
    """ fistr形式のmshファイルををvtu形式に変換する。
    複数のfileをparalleで処理する。"""
    meshFileName = getMeshFileName(meshFile)
    if meshFileName == "":
        return
    lines = readFileLines(meshFileName)
    #mesh内容を読込
    meshHeaderNumConts = cm.getFistrHeaderNumData(lines)
    #メッシュ内容を取得
    [nodeList, elementList] = cm.getFistrMeshConts(meshHeaderNumConts)
    print (_("meshファイルを変換中..."))
    #node番号の振り直し（未使用nodeを確認）vtk用のvnList, veListを取得
    [nodeList, vnList] = getNewNodeNo(nodeList, meshHeaderNumConts)
    [elementList, veList, vtkNodesElmNoDict] = renumberNodeElement(nodeList, elementList, meshHeaderNumConts)
    #veListをsolid、シェル、ビーム要素順に並べ替える
    veList = renumberVeList(veList)
    #dummyNodeを取得(dummyと相手nodeNoを取得）
    dummyNodes = getDummyNodes(elementList, nodeList, meshHeaderNumConts)
    #groupを取得
    [vndGrps, vsuGrps, surGrps] = renumberGrps(meshHeaderNumConts, nodeList, elementList)
    if resFile == "":
        #resFileが存在しない場合
        #dummyのnodeGrpを削除する
        (vnList, veList, vndGrps) = deleteDummyNodes(vnList, veList, vndGrps, dummyNodes)
        #point, cell定義行を作成(nodeNo, elementNoを追加)
        defineLines = createVtkFileMesh(vnList, veList, vndGrps, vsuGrps, surGrps)
        fileName = "conv" + meshFileName + ".vtu"
        writeFileLines(fileName, defineLines)
        print(_("  変換しました"))
        return

    #resFileの処理
    fileNames = getSingleResultFileNames()
    if len(fileNames) == 0:
        print(_("error: 結果fileが存在しません"))
        return
    fileNames = sortStepFileName(fileNames)
    fileNames = getResVisMatchFiles(fileNames)
    np = int(nCpu)
    if np > len(fileNames):
        np = len(fileNames)
    print(_("全file数:") + str(len(fileNames)) + "  " + _("並列数:") + str(np) + " --> " + _("vtu変換します。"))
    valueList = list(map(lambda x: [x, elementList, vtkNodesElmNoDict, resFile, visFile], fileNames))
    p = multiprocessing.Pool(np)
    res = p.map(convert1file, valueList)

#
#  getResVisMatchFiles
def getResVisMatchFiles(fileNames):
    """ resFileとpvtuファイルの有無を確認し、両者が対応していない場合は
    除いて返す。"""
    newNames = []
    for fileName in fileNames:
        step = fileName.split(".")[-1]
        pvtuFile = visFile + "_psf." + ("%04d" % int(step)) + ".pvtu"
        if os.path.exists(pvtuFile) == True:
            newNames.append(fileName)
    return newNames

#
#  convert1file
#---------------
def convert1file(values):
    """ 並列処理用の関数。（global変数を使用しない）
    指定されたfileをvtu形式に変換する。"""

    def getSinglePvtuFileName(resFileName, visFile):
        step = resFileName.split(".")[-1]
        newStep = "0"*(4-len(step)) + step
        vtuFile = visFile + "_psf." + newStep + ".pvtu"
        return vtuFile

    def addFieldsToVtkFile(fileName, nodeFields, cellFields, elmToOrderDict, vtkNodesElmNoDict, visFile):
        """ vtuファイルに足りないfieldを追加する"""
        pvtuFile = getSinglePvtuFileName(fileName, visFile)
        lines = readFileLines(pvtuFile)
        addFields = getAddFields(lines, nodeFields, cellFields)
        lines = addFieldsToPVtuFile(lines, addFields)
        writeFileLines(pvtuFile, lines)
        vtuFile = getSingleVtuFileName(lines)
        addFieldsToVtuFile(vtuFile, addFields, elmToOrderDict, vtkNodesElmNoDict)

    def checkRemakeDisplacement(fileName, visFile):
        """ DISPLACEMENTのデータ数を確認し、
        データ数が3以上の場合は、3ヶに縮小する。
        変位のみ残し、回転は削除。（shellの場合、回転が含まれる）"""
        pvtuFile = getSinglePvtuFileName(fileName, visFile)
        lines = readFileLines(pvtuFile)
        nData, lines = getNumOfDisplacementData(lines)
        if nData == 3:
            return
        #pvtuFileを修正
        writeFileLines(pvtuFile, lines)
        vtuFile = getSingleVtuFileName(lines)
        lines = readFileLines(vtuFile)
        lines = remakeDisplacement(lines)
        writeFileLines(vtuFile, lines)

    [fileName, elementList, vtkNodesElmNoDict, resFile, visFile] = values
    name = os.path.basename(fileName)
    print("  converting " + name + "...\n", end="", flush=True)
    #計算結果のfieldを取得
    (nodeFields, cellFields, elmToOrderDict) = getResultFields(fileName, elementList)
    #vtkファイルに追加
    addFieldsToVtkFile(fileName, nodeFields, cellFields, elmToOrderDict, vtkNodesElmNoDict, visFile)
    #DISPLACEMENTのデータ数を確認、修正（shellの場合、回転を含んでいる事がある）
    checkRemakeDisplacement(fileName, visFile)
    return name

#-------------------
#  singleMeshConvertToVtu
#-------------------
def singleMeshConvertToVtu():
    """ fistr形式のmshファイルををvtu形式に変換する。
    1個のfileをsingle or paralleで処理する。"""
    meshFileName = getMeshFileName(meshFile)
    if meshFileName == "":
        return
    lines = readFileLines(meshFileName)
    #mesh内容を読込
    meshHeaderNumConts = cm.getFistrHeaderNumData(lines)
    #メッシュ内容を取得
    [nodeList, elementList] = cm.getFistrMeshConts(meshHeaderNumConts)
    print (_("meshファイルを変換中..."))
    #node番号の振り直し（未使用nodeを確認）vtk用のvnList, veListを取得
    [nodeList, vnList] = getNewNodeNo(nodeList, meshHeaderNumConts)
    [elementList, veList, vtkNodesElmNoDict] = renumberNodeElement(nodeList, elementList, meshHeaderNumConts)
    #veListをsolid、シェル、ビーム要素順に並べ替える
    veList = renumberVeList(veList)
    #dummyNodeを取得(dummyと相手nodeNoを取得）
    dummyNodes = getDummyNodes(elementList, nodeList, meshHeaderNumConts)
    #groupを取得
    [vndGrps, vsuGrps, surGrps] = renumberGrps(meshHeaderNumConts, nodeList, elementList)
    if resFile == "":
        #dummyのnodeGrpを削除する
        (vnList, veList, vndGrps) = deleteDummyNodes(vnList, veList, vndGrps, dummyNodes)
        #point, cell定義行を作成(nodeNo, elementNoを追加)
        defineLines = createVtkFileMesh(vnList, veList, vndGrps, vsuGrps, surGrps)
        fileName = "conv" + meshFileName + ".vtu"
        writeFileLines(fileName, defineLines)
        print(_("  変換しました"))
        return

    #resFileの処理
    #  結果ファイルの取得
    if parallel == False:
        #singleCoreの場合
        fileNames = getSingleResultFileNames()
        if len(fileNames) == 0:
            print(_("error: 結果fileが存在しません"))
            return
        fileNames = sortStepFileName(fileNames)
        names = list(map(lambda x: os.path.basename(x), fileNames))
        i = 0
        for fileName in fileNames:
            print("  converting " + names[i] + "...")
            #計算結果のfieldを取得
            (nodeFields, cellFields, elmToOrderDict) = getResultFields(fileName, elementList)
            #vtkファイルに追加
            addFieldsToVtkFile(fileName, nodeFields, cellFields, elmToOrderDict, vtkNodesElmNoDict)
            #DISPLACEMENTのデータ数を確認、修正（shellの場合、回転を含んでいる事がある）
            checkRemakeDisplacement(fileName)
            i += 1
            
    else:
        #parallel処理の場合
        paraFileNames = getParallelResultFileNames()
        nProcs = getNcpuFromResFileNames(paraFileNames)
        #追加するfield名を取得
        addFields = getAddFieldNames(paraFileNames, elementList)
        #pvtuファイルにfieldを追加する。
        pvtuFiles = glob.glob(visFile + "_psf.*.pvtu")
        #pvtuFiles = glob.glob("FistrModel.vis_psf.*.pvtu")
        for pvtuFile in pvtuFiles:
            lines = readFileLines(pvtuFile)
            lines = addFieldsToPVtuFile(lines, addFields)
            #DISPLACEMENTのデータ数を確認修正する
            nDispData, lines = getNumOfDisplacementData(lines)
            writeFileLines(pvtuFile, lines)
        nodeNames = list(map(lambda x: x[0], addFields[0]))
        cellNames = list(map(lambda x: x[0], addFields[1]))
        addFieldNames = [nodeNames, cellNames]
        #各step毎にfieldを追加
        #全stepNoを取得
        steps = []
        for fileName in paraFileNames:
            step = int(fileName.split(".")[-1])
            steps.append(step)
        steps = list(set(steps))
        steps.sort()

        if int(nCpu) > 1:
            print(_("並列数:") + nCpu + " --> " + _("分散結果を並列で処理します。"))
        #step毎に処理
        for step in steps:
            #vtuFile名を取得
            stepName = "0"*(4-len(str(step))) + str(step)
            pvtuFile = visFile + "_psf." + stepName + ".pvtu"
            #pvtuFile = "FistrModel.vis_psf." + stepName + ".pvtu"
            if os.path.exists(pvtuFile) == True:
                vtuFiles = getvtuFiles(pvtuFile)
                #singleCore?
                if nCpu == "1":
                    for i in range(nProcs):
                        vtuFile = vtuFiles[i]
                        fileName = resFile + ".res." + str(i) + "." + str(step)
                        cpuConvertVtu(i, fileName, vtuFile, addFieldNames, elementList, nDispData, vtkNodesElmNoDict)
                #フルparallel?
                elif int(nCpu) >= nProcs:
                    #jobを投入
                    queues = []
                    processes = []
                    for i in range(nProcs):
                        vtuFile = vtuFiles[i]
                        fileName = resFile + ".res." + str(i) + "." + str(step)
                        queue = multiprocessing.Queue()
                        process = multiprocessing.Process(
                            target=paraConvertVtu,
                            args=(i, fileName, vtuFile, addFieldNames, elementList, nDispData, vtkNodesElmNoDict, queue)
                            )
                        queues.append(queue)
                        processes.append(process)
                        process.start()
                    #結果を取得
                    for i in range(nProcs):
                        dummy = queues[i].get()
                #準parallelで処理
                else:
                    i = 0
                    while i < nProcs:
                        queues = []
                        for _n in range(int(nCpu)):
                            if i >= nProcs:
                                break
                            vtuFile = vtuFiles[i]
                            fileName = resFile + ".res." + str(i) + "." + str(step)
                            queue = multiprocessing.Queue()
                            process = multiprocessing.Process(
                                target=paraConvertVtu,
                                args=(i, fileName, vtuFile, addFieldNames, elementList, nDispData, vtkNodesElmNoDict, queue)
                                )
                            queues.append(queue)
                            process.start()
                            i += 1
                        for queue in queues:
                            dummy = queue.get()

#
#  paraConvertVtu
#-----------------
def paraConvertVtu(procNo, fileName, vtuFile, addFieldNames,
            elementList, nDispData, vtkNodesElmNoDict, queue):
    """ 並列処理で変換する"""
    cpuConvertVtu(procNo, fileName, vtuFile, addFieldNames,
            elementList, nDispData, vtkNodesElmNoDict)
    queue.put("end")

#
#  cpuConvertVtu
#----------------
def cpuConvertVtu(procNo, fileName, vtuFile, addFieldNames,
            elementList, nDispData, vtkNodesElmNoDict):
    """ 各cpu毎の結果をvtu変換する"""
    if procNo == 0:
        name = os.path.basename(fileName)
        print("  converting " + name + "...")
    #計算結果のfieldを取得
    (nodeFields, cellFields, elmToOrderDict) = getResultFields(fileName, elementList)
    #addFieldを取得
    addNodeFields = []
    for field in nodeFields:
        if field[0] in addFieldNames[0]:
            addNodeFields.append(field)
    addCellFields = []
    for field in cellFields:
        if field[0] in addFieldNames[1]:
            addCellFields.append(field)
    addFields = [addNodeFields, addCellFields]
    #vtuファイルに追加し、書き込み
    addFieldsToVtuFile(vtuFile, addFields, elmToOrderDict, vtkNodesElmNoDict)
    #DISPLACEMENTのデータ数を確認・修正(回転を削除)
    if nDispData != 3:
        #データ数を修正
        lines = readFileLines(vtuFile)
        lines = remakeDisplacement(lines)
        writeFileLines(vtuFile, lines)

#
#  printMsg
#-----------
def printMsg():
    """ helpを出力する。"""
    msg = """

<fistr2vtu.py>---------------------------------------------------------------
FrontISTRフォーマットをvtuフォーマットに変換する。
FrontISTR側のmshファイルとresファイルを読み込みファイル変換する。
「-res」を省略した場合は、meshファイルのみvtu変換する。
この場合、nodeNo、elementNoのdata（field）が追加される。

<使い方>
fistr2vtk.py -mesh <meshFileのheader> [-res <resFileのheader>] [-para yes/no]

<option>
 -mesh fileHeader   meshのfileHeader名を設定
 -res fileHeader    結果のfileHeader名を設定
 -vis fileHeader    vtu形式のfileHeaderを設定
 -para yes/no       並列計算の場合（省略時は、自動判別）
 -np <n>            並列処理で変換する場合のcpu数（省略時は、singleCore）
 -h                 ヘルプ
 -help              ヘルプ

例:
fistr2vtk.py -mesh FistrModel
fistr2vtk.py -mesh FistrModel -res FistrModel -vis FistrModel
fistr2vtk.py -mesh FistrModel -res FistrModel_eigen -para yes
"""
    print (msg)

#
#  getOption
#-------------
def getOption(meshFile, resFile, visFile, para, nCpu):
    """ optionを取得する"""
    i = 0
    while i < len(sys.argv):
        if sys.argv[i] == "-mesh":
            i += 1
            meshFile = sys.argv[i]
        elif sys.argv[i] == "-res":
            i += 1
            resFile = sys.argv[i]
        elif sys.argv[i] == "-vis":
            i += 1
            visFile = sys.argv[i]
        elif sys.argv[i] == "-para":
            i += 1
            para = sys.argv[i]
        elif sys.argv[i] == "-np":
            i += 1
            nCpu = sys.argv[i]
        i += 1
    return (meshFile, resFile, visFile, para, nCpu)


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", localeDir)
    #log表示
    #import logFileCreater
    #logFileCreater.log()

    meshFile = ""
    resFile = ""
    visFile = ""
    para = ""
    nCpu = "1"
    (meshFile, resFile, visFile, para, nCpu) = getOption(meshFile, resFile, visFile, para, nCpu)
    if meshFile == "":
        print (_("error:meshファイルが入力されていません。"))
        printMsg()
        exit()

    #single or parallelを判断
    parallel = False
    if para == "yes":
        parallel = True
    # else:
    #     if len(glob.glob(resFile + ".res.1.*")) > 0:
    #         parallel = True
    #並列処理方法を決定
    method = "singleFile"
    if parallel == True:
        if (len(glob.glob(resFile + ".res.0.*")) >= 2 and
            len(glob.glob(resFile + ".res.1.*")) == 0):
            method = "multiFiles"
    #処理開始
    if method == "singleFile":
        #1個のfileを並列 or single処理
        singleMeshConvertToVtu()
    else:
        #複数のfileを並列処理
        multiMeshConvertToVtu()
        