#!/usr/bin/python3
#  coding: utf-8
#
#           fistr2abaqus.py
#
#       FrontISTRのcnt, mshファイルをinpファイルに変換する
#           静的な線形、大変形の構造解析
#           静的な熱伝導解析
#
#       FrontISTR     ->        inpFile
#   ------------------------------------------------------
#       !NODE                   *NODE
#       !ELEMENT                *ELEMENT
#       !EGROUP                 *ELSET
#       !SGROUP                 *SURFACE
#       !NGROUP                 *NSET
#       !SECTION                *SOLID, shell or beam SECTION
#       !MATERIAL               *MATERIAL
#         !ELASTIC                *ELASTIC
#         !DENSITY                *DENSITY
#         !EXPANSION_COEFF        *EXPANSION
#         !SPECIFIC_HEAT          *SPECIFIC HEAT
#         !THERMAL_CONDUCTIVITY   *CONDUCTIVITY
#       !EQUATION               *EQUATION
#       !BOUNDARY               *BOUNDARY
#       !CLOAD                  *CLOAD
#       !CFLUX                  *CFLUX
#       !DLOAD,S                *DLOAD, Px
#       !FIXTEMP                *BOUNDARY, 11
#
#   <meshType>
#       341    solid            C3D4
#       342    solid            C3D10
#       351    solid            C3D6
#       352    solid            C3D15
#       361    solid            C3D8
#       362    solid            C3D20
#       611    beam             B31     #変換後、断面形状を設定要
#       731    shell            S3
#       732    shell            S6
#       741    shell            S4
#
#-----------------------
#   21/03/17    新規作成
#      03/20    getEquationCard:系数の値をチェックして、1e-8以下は
#               項を省略する様に修正。
#      03/22    静的な熱伝導解析を追加
#   22/03/01    getNodeCard,getNsetCard:NGRP=dummyを除く様に修正。
#      03/15    国際化
#   24/07/07    openにencoding="utf-8"を追加
#   25/01/03    getNsetCard:ngrpの"NALL"作成を追加。
#      01/04    remakeSurfaceCard,createLines:SURFACEの定義を
#               prepomaxに合わせる様に修正。
#      01/07    createLines:*ELEMENTcardの行当りのbode数が16を超えると
#               エラー発生のため、修正。（六面体2次要素が該当する）
#      01/13    remakeSurfaceCard:SGRP定義無い場合、エラー発生のため、修正。
#      04/20    remakeSurfaceCard:NGRP定義無い場合、エラー発生のため、修正。
#

import os, sys
import pyFistr
import convertMesh as cm


SolType = ""

elmTypeDict = {
    "341":"solid",
    "342":"solid",
    "351":"solid",
    "352":"solid",
    "361":"solid",
    "362":"solid",
    "611":"beam",
    "641":"beam",
    "731":"shell",
    "732":"shell",
    "741":"shell",
    "761":"shell",
    "781":"shell"
    }

#
#  createNodeElementDict
def createNodeElementDict(meshHeaderData):
    """ node, elementDictを作成する"""
    #nodeDictを作成
    nodeDict = {}
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            for node in data:
                nodeNo = node[0]
                loc = node[1:]
                nodeDict[nodeNo] = {"loc":loc, "elms":[]}
    #elementDictを作成
    elementDict = {}
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            elmType = pyFistr.getValName(header, "TYPE")
            for i in range(len(data)):
                elmData = data[i]
                elmNo = elmData[0]
                nodes = elmData[1:]
                elementDict[elmNo] = {"nodes":nodes, "type":elmType}
                for nodeNo in nodes:
                    nodeDict[nodeNo]["elms"].append(elmNo)
    return nodeDict, elementDict

#
#  getNodeCard
def getNodeCard(meshHeaderData):
    """ *NODE cardを作成する。NGRP=dummy_の場合は、取得しない。"""
    addHeaderData = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp[:len("dummy")] != "dummy":
                #newHeader = "*NODE, NSET=NALL\n"
                newHeader = "*NODE\n"
                addHeaderData.append([newHeader, data])
    return addHeaderData

#
#  getElementCard
def getElementCard(meshHeaderData):
    """ *ELEMENT cardを作成する"""
    addHeaderData = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            elmType = pyFistr.getValName(header, "TYPE")
            abaType = cm.fistr2aba_elDict[elmType][0]
            egrp = pyFistr.getValName(header, "EGRP")
            headerWords = ["*ELEMENT", "TYPE="+abaType]
            if egrp != "":
                headerWords += ["ELSET="+egrp]
            newHeader = ", ".join(headerWords) + "\n"
            newData = []
            for vals in data:
                elmNo = vals[0]
                nodes = vals[1:]
                newNodes = cm.fistr2aba_el(elmType, nodes)[1]
                elmData = [elmNo] + newNodes
                newData.append(elmData)
            addHeaderData.append([newHeader, newData])
    return addHeaderData            

#
#  getElsetCard
def getElsetCard(meshHeaderData):
    """ *ELSET cardを作成する"""
    addHeaderData = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!EGROUP":
            egrp = pyFistr.getValName(header, "EGRP")
            newHeader = "*ELSET, ELSET=" + egrp + "\n"
            addHeaderData.append([newHeader, data])
    return addHeaderData

#
#  getMaterialCard
def getMaterialCard(contHeaderData):
    """ *MATERIAL cardを作成"""
    #matCardを定義（これ以外は作成しない）
    matCardsSet = {
        "!Material", "!ELASTIC", "!DENSITY", "!EXPANSION_COEFF",
        "!SPECIFIC_HEAT", "!THERMAL_CONDUCTIVITY"        
        }
    addHeaderData = []
    matHeaders = []
    for header, data in contHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!MATERIAL":
            if len(matHeaders) != 0:
                addHeaderData += matHeaders
                matHeaders = []
            matName = pyFistr.getValName(header, "NAME")
            newHeader = "*MATERIAL, NAME=" + matName + "\n"
            matHeaders.append([newHeader, []])
        elif words[0] == "!ELASTIC":
            newHeader = "*ELASTIC\n"
            matHeaders.append([newHeader, data])
        elif words[0] == "!DENSITY":
            newHeader = "*DENSITY\n"
            matHeaders.append([newHeader, data])
        elif words[0] == "!EXPANSION_COEFF":
            newHeader = "*EXPANSION\n"
            matHeaders.append([newHeader, data])
        elif words[0] == "!SPECIFIC_HEAT":
            newHeader = "*SPECIFIC HEAT\n"
            matHeaders.append([newHeader, data])
        elif words[0] == "!THERMAL_CONDUCTIVITY":
            newHeader = "*CONDUCTIVITY\n"
            matHeaders.append([newHeader, data])
        elif header[0] == "#" or not (words[0] in matCardsSet):
            if len(matHeaders) != 0:
                addHeaderData += matHeaders
                matHeaders = []
    return addHeaderData

#
#  getSectionCard
def getSectionCard(meshHeaderData, dicts):
    """ *SECTION cardを作成する"""

    def getElmType(egrp, meshHeaderData, elementDict):
        elmType = ""
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                name = pyFistr.getValName(header, "EGRP")
                eType = pyFistr.getValName(header, "TYPE")
                if name == egrp:
                    elmType = eType
                    break
        if elmType == "":
            for header, data in meshHeaderData:
                words = pyFistr.deleteSp(header).split(",")
                if words[0] == "!EGROUP":
                    name = pyFistr.getValName(header, "EGRP")
                    if name == egrp:
                        elmNo = data[0]
                        elmType = elementDict[elmNo]["type"]
                        break
        return elmType

    nodeDict, elementDict = dicts
    addHeaderData = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SECTION":
            egrp = pyFistr.getValName(header, "EGRP")
            matName = pyFistr.getValName(header, "MATERIAL")
            elmType = getElmType(egrp, meshHeaderData, elementDict)
            if elmTypeDict[elmType] == "beam":
                #beam section
                newHeader = "*BEAM SECTION, ELSET=" + egrp + ", MATERIAL=" + matName + ", SECTION=RECT\n"
                newData = ["** w, h\n"]
                newData += ["0.1, 0.05\n"]
                refDir = ", ".join(pyFistr.deleteSp(data[0]).split(",")[:3]) + "\n"
                newData += [refDir]
                addHeaderData.append([newHeader, newData])
            elif elmTypeDict[elmType] == "shell":
                #shell section
                newHeader = "*SHELL SECTION, ELSET=" + egrp + ", MATERIAL=" + matName + "\n"
                t = pyFistr.deleteSp(data[0]).split(",")[0]
                newData = [t + "\n"]
                addHeaderData.append([newHeader, newData])
            elif elmTypeDict[elmType] == "solid":
                #solid section
                newHeader = "*SOLID SECTION, ELSET=" + egrp + ", MATERIAL=" + matName + "\n"
                addHeaderData.append([newHeader, []])
    return addHeaderData

#
#  remakeSurfaceCard
def remakeSurfaceCard(meshHeaderData):
    """ SGROUPカードの内容をinp形式に修正"""
    newEgrpDict = {}
    insertPoint = ""
    flag = 0
    for i in range(len(meshHeaderData)):
        header, data = meshHeaderData[i]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SGROUP":
            if flag == 0:
                insertPoint = i
                flag = 1
            sgrp = pyFistr.getValName(header, "SGRP")
            faceElmDict = {}
            for ii in range(0, len(data), 2):
                elm = data[ii]
                faceNo = data[ii+1]
                if faceNo in faceElmDict.keys():
                    faceElmDict[faceNo].append(elm)
                else:
                    faceElmDict[faceNo] = [elm]
            faceNos = list(faceElmDict.keys())
            faceNos.sort()
            newData = []
            for faceNo in faceNos:
                egrp = sgrp + "_elmsS" + str(faceNo)
                newEgrpDict[egrp] = faceElmDict[faceNo]
                newData.append(egrp)
                newData.append(faceNo)
            meshHeaderData[i][1] = newData
    #egrpを追加
    addEgrp = []
    for egrp in newEgrpDict.keys():
        header = "!EGROUP, EGRP=" + egrp + "\n"
        data = newEgrpDict[egrp]
        addEgrp.append([header, data])
    #insertPointを確認
    if insertPoint == "":
        #挿入位置を調査
        flag = 0
        for i in range(len(meshHeaderData)):
            header, data = meshHeaderData[i]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!NGROUP":
                insertPoint = i
                break
    if insertPoint != "":
        i = insertPoint
    newHeaderData = meshHeaderData[:i] + addEgrp + meshHeaderData[i:]
    return newHeaderData

#
#  getSurfaceCard
def getSurfaceCard(meshHeaderData):
    """ *SURFACE cardを作成"""
    addHeaderData = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SGROUP":
            sgrp = pyFistr.getValName(header, "SGRP")
            newHeader = "*SURFACE, NAME=" + sgrp + ", TYPE=ELEMENT\n"
            addHeaderData.append([newHeader, data])
    return addHeaderData

#
#  getNsetCard
def getNsetCard(meshHeaderData, dicts):
    """ *NSET cardを作成。ngrp=dummyは除く。"""
    nodeDict, elementDict = dicts
    addHeaderData = []
    ngrps = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            ngrp = pyFistr.getValName(header, "NGRP")
            ngrps.append(ngrps)
            if ngrp[:len("dummy")] != "dummy":
                newHeader = "*NSET, NSET=" + ngrp + "\n"
                addHeaderData.append([newHeader, data])
    if not "NALL" in ngrps:
        allNodes = []
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!NODE":
                ngrp = pyFistr.getValName(header, "NGRP")
                if ngrp[:len("dummy")] != "dummy":
                    allNodes += list(map(lambda x: x[0], data))
        allNodes.sort()
        newHeader = "*NSET, NSET=NALL\n"
        addHeaderData = [[newHeader, allNodes]] + addHeaderData
        #addHeaderData.append([newHeader, allNodes])
    return addHeaderData

#
#  getBoundaryCard
def getBoundaryCard(contHeaderData):
    """ *BOUNDARY cardを作成"""
    addHeaderData = []
    for header, data in contHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!BOUNDARY":
            newHeader = "*BOUNDARY\n"
            addHeaderData.append([newHeader, data])
        elif words[0] == "!FIXTEMP":
            newHeader = "*BOUNDARY\n"
            ws = pyFistr.deleteSp(data[0]).split(",")
            ngrp = ws[0]
            temp = ws[1]
            newData = [ngrp + ", 11, 11, " + temp + "\n"]
            addHeaderData.append([newHeader, newData])
    return addHeaderData

#
#  getSpringCard
def getSpringCard(contHeaderData, meshHeaderData, dicts):
    """ *SPRING cardとその為のNODE, ELEMENTを作成する。
    xyz各方向に長さ1.0mのバネを作成する。"""

    def getNgrpData(meshHeaderData, ngrpName):
        """ nodeGrpのnodeNoを取得"""
        ngrpData = []
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!NGROUP":
                ngrp = pyFistr.getValName(header, "NGRP")
                if ngrp == ngrpName:
                    ngrpData += data
            elif words[0] == "!NODE":
                ngrp = pyFistr.getValName(header, "NGRP")
                if ngrp == ngrpName:
                    nodeNos = list(map(lambda x: x[0], data))
                    ngrpData += nodeNos
        return ngrpData

    nodeDict, elementDict = dicts
    spHeaderData = []
    addElmsData = []
    addNodesData = []
    addBoundData = []
    maxNo = max(list(nodeDict.keys()))
    maxElmNo = max(list(elementDict.keys()))
    for header, data in contHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SPRING":
            ws = pyFistr.deleteSp(data[0]).split(",")
            ngrpName = ws[0]
            spValue = ws[-1]
            ngrpNodes = getNgrpData(meshHeaderData, ngrpName)
            newNodes = []
            newElms = []
            for nodeNo in ngrpNodes:
                loc = nodeDict[nodeNo]["loc"]
                for i in range(3):
                    newLoc = loc[:]
                    newLoc[i] += 1.0
                    maxNo += 1
                    newNodeNo = maxNo
                    node = [newNodeNo] + newLoc
                    newNodes.append(node)
                    maxElmNo += 1
                    elmNo = maxElmNo
                    elm = [elmNo, nodeNo, newNodeNo]
                    newElms.append(elm)
            #*NODE card
            newHeader = "*NODE, NSET=" + "springFix_" + ngrpName + "\n"
            addNodesData.append([newHeader, newNodes])
            #*BOUNDARY card
            newHeader = "*BOUNDARY\n"
            newData  = ["springFix_" + ngrpName + ", 1, 1, 0.0\n"]
            newData += ["springFix_" + ngrpName + ", 2, 2, 0.0\n"]
            newData += ["springFix_" + ngrpName + ", 3, 3, 0.0\n"]
            addBoundData.append([newHeader, newData])
            #*ELEMENT card
            elsetName = "elm_" + ngrpName
            newHeader = "*ELEMENT, TYPE=SPRINGA, ELSET=" + elsetName + "\n"
            addElmsData.append([newHeader, newElms])
            #*SPRING card
            newHeader = "*SPRING, ELSET=" + elsetName + "\n"
            spHeaderData.append([newHeader, [spValue + "\n"]])
    return addNodesData, addBoundData, addElmsData, spHeaderData

#
#  getEquationCard
def getEquationCard(meshHeaderData):
    """ *EQUATION cardを作成
    係数が1e-8以下は、その項を省略する"""
    addHeaderData = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!EQUATION":
            newData = []
            for i in range(0, len(data), 2):
                line1 = data[i]
                line2 = data[i+1]
                ws = pyFistr.deleteSp(line2).split(",")
                eqWords = []
                for ii in range(0, len(ws), 3):
                    #系数の値をチェック
                    coeff = float(ws[ii+2])
                    if abs(coeff) > 1.0e-8:
                        eqWords += ws[ii:ii+3]
                    else:
                        eqWords += ws[ii:ii+2] + ["0.0"]
                nEq = len(eqWords) // 3
                line1 = str(nEq) + "\n"
                line2 = ", ".join(eqWords) + "\n"
                newData.append(line1)
                newData.append(line2)
            newHeader = "*EQUATION\n"
            addHeaderData.append([newHeader, newData])
    return addHeaderData

#
#  getCloadCard
def getCloadCard(contHeaderData):
    """ *CLOAD cardの作成"""
    addHeaderData = []
    for header, data in contHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!CLOAD":
            newHeader = "*CLOAD\n"
            newData = []
            for valLine in data:
                ws = pyFistr.deleteSp(valLine).split(",")
                if float(ws[2]) != 0.0:
                    newData.append(valLine)
            addHeaderData.append([newHeader, newData])
    return addHeaderData

#
#  getDloadCard
def getDloadCard(contHeaderData, surfaceHeaderData):
    """ *DLOAD cardを作成"""

    def getSurfaceData(sgrp, press, surfaceHeaderData):
        """ dloadDataを作成する"""
        vals = []
        for header, data in surfaceHeaderData:
            name = pyFistr.getValName(header, "NAME")
            if name == sgrp:
                newData = list(map(str, data))
                for i in range(0, len(newData), 2):
                    line = ", ".join([newData[i], "P" + newData[i+1], press]) + "\n"
                    vals.append(line)
        return vals

    addHeaderData = []
    for header, data in contHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!DLOAD":
            vals = []
            for line in data:
                ws = pyFistr.deleteSp(line).split(",")
                if ws[1] == "S":
                    sgrp = ws[0]
                    press = ws[2]
                    vals += getSurfaceData(sgrp, press, surfaceHeaderData)
            if len(vals) > 0:
                newHeader = "*DLOAD\n"
                addHeaderData.append([newHeader, vals])
    return addHeaderData

#
#  getCfluxCard
#---------------
def getCfluxCard(contHeaderData):
    """ *CFLUX cardを作成"""
    addHeaderData = []
    for header, data in contHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!CFLUX":
            newHeader = "*CFLUX\n"
            newData = []
            for line in data:
                ws = pyFistr.deleteSp(line).split(",")
                if float(ws[1]) != 0.0:
                    newLine = ", ".join([ws[0], "11", ws[1]]) + "\n"
                    newData.append(newLine)
            addHeaderData.append([newHeader, newData])
    return addHeaderData

#
#  getFilmCard
#--------------
def getFilmCard(contHeaderData, surfaceHeaderData):
    """ FILM cardを作成"""

    def getSurfaceData(sgrp, temp, alpha, surfaceHeaderData):
        """ filmDataを作成する"""
        vals = []
        for header, data in surfaceHeaderData:
            name = pyFistr.getValName(header, "NAME")
            if name == sgrp:
                newData = list(map(str, data))
                for i in range(0, len(data), 2):
                    line = ", ".join([newData[i], "F" + newData[i+1], temp, alpha]) + "\n"
                    vals.append(line)
        return vals

    addHeaderData = []
    for header, data in contHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SFILM":
            newHeader = "*FILM\n"
            ws = pyFistr.deleteSp(data[0]).split(",")
            sgrp = ws[0]
            temp = ws[1]
            alpha = ws[2]
            vals = getSurfaceData(sgrp, temp, alpha, surfaceHeaderData)
            addHeaderData.append([newHeader, vals])
    return addHeaderData

#
#  getAnaTypeCard
def getAnaTypeCard(contHeaderData):
    """ *STEP cardを作成"""
    global SolType
    addHeaderData = []
    for header, data in contHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SOLUTION":
            solType = pyFistr.getValName(header, "TYPE")
            if solType == "STATIC":
                addHeaderData.append(["*STEP\n", []])
                addHeaderData.append(["*STATIC\n", []])
            elif solType == "NLSTATIC":
                addHeaderData.append(["*STEP, INC=100, NLGEOM\n", []])
                addHeaderData.append(["*STATIC\n", ["0.1, 1.0\n"]])
            elif solType == "HEAT":
                addHeaderData.append(["*STEP\n", []])
                addHeaderData.append(["*HEAT TRANSFER, STEADY STATE\n", []])
            else:
                addHeaderData.append(["*STEP\n", []])
                addHeaderData.append(["*STATIC\n", []])
            break
    SolType = solType
    return addHeaderData

#
#  getOutputCard
def getOutputCard(sectionHeaderData):
    """ 出力のcardを作成"""
    global SolType
    addHeaderData = []
    if SolType == "HEAT":
        #熱解析
        addHeaderData.append(["*NODE PRINT, NSET=NALL\n", ["NT\n"]])
        for header, _data in sectionHeaderData:
            name = pyFistr.getValName(header, "ELSET")
            addHeaderData.append(["*EL PRINT, ELSET=" + name + "\n", ["HFL\n"]])
        addHeaderData.append(["*NODE FILE\n", ["NT\n"]])
        addHeaderData.append(["*EL FILE\n", ["HFL\n"]])
        addHeaderData.append(["*END STEP\n", []])
    else:
        #構造解析
        addHeaderData.append(["*NODE PRINT, NSET=NALL\n", ["U\n"]])
        for header, _data in sectionHeaderData:
            name = pyFistr.getValName(header, "ELSET")
            addHeaderData.append(["*EL PRINT, ELSET=" + name + "\n", ["S\n"]])
        addHeaderData.append(["*NODE FILE\n", ["U\n"]])
        addHeaderData.append(["*EL FILE\n", ["S\n"]])
        addHeaderData.append(["*END STEP\n", []])
    return addHeaderData

#
#  convToInt
#------------
def convToInt(meshHeaderData, contHeaderData):
    """ abaqus形式のinpFileのheaderDataを作成する"""
    #node, elementDictを作成
    dicts = createNodeElementDict(meshHeaderData)
    #*NODEを作成 (NGRP=dummyは、除く)
    nodeHeaderData = getNodeCard(meshHeaderData)
    #*ELEMENTを作成
    elmHeaderData = getElementCard(meshHeaderData)
    #*ELSETを作成
    #elsetHeaderData = getElsetCard(meshHeaderData)
    #*MATERIALを作成
    matHeaderData = getMaterialCard(contHeaderData)
    #*SECTIONを作成
    sectionHeaderData = getSectionCard(meshHeaderData, dicts)
    #*SURFACEを作成
    meshHeaderData = remakeSurfaceCard(meshHeaderData)
    surfaceHeaderData = getSurfaceCard(meshHeaderData)
    #*ELSETを作成
    elsetHeaderData = getElsetCard(meshHeaderData)
    #*NSETを作成（NGRP=dummyは除く）
    nsetHeaderData = getNsetCard(meshHeaderData, dicts)
    #*EQUATIONを作成
    equationHeaderData = getEquationCard(meshHeaderData)
    #*BOUNDARYを作成
    boundHeaderData = getBoundaryCard(contHeaderData)
    #*SPRINGを作成
    headers = getSpringCard(contHeaderData, meshHeaderData, dicts)
    addNodesData, addBoundData, addElmsData, springHeaderData = headers
    nodeHeaderData += addNodesData
    elmHeaderData += addElmsData
    boundHeaderData += addBoundData
    #*CLOADを作成
    cloadHeaderData = getCloadCard(contHeaderData)
    #*DLOADを作成
    dloadHeaderData = getDloadCard(contHeaderData, surfaceHeaderData)
    #*CFLUXを作成
    cfluxHeaderData = getCfluxCard(contHeaderData)
    #*FILMを作成
    filmHeaderData = getFilmCard(contHeaderData, surfaceHeaderData)
    #*STEPを作成（線形or非線形）
    anaTypeHeaderData = getAnaTypeCard(contHeaderData)
    #出力の設定
    outHeaderData = getOutputCard(sectionHeaderData)
    #全体のheaderDataを作成
    headerData = nodeHeaderData + elmHeaderData + equationHeaderData
    headerData += elsetHeaderData + surfaceHeaderData + nsetHeaderData
    headerData += matHeaderData + sectionHeaderData + springHeaderData
    #headerData += boundHeaderData + anaTypeHeaderData
    headerData += anaTypeHeaderData + boundHeaderData
    headerData += cloadHeaderData + dloadHeaderData + cfluxHeaderData
    headerData += filmHeaderData
    headerData += outHeaderData
    return headerData

#
#  createLines
#--------------
def createLines(abaHeaderData):
    """ headerDataをlinesに変換する"""
    lines = []
    for header, data in abaHeaderData:
        lines.append(header)
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "*NODE":
            for node in data:
                #座標値を整形する。
                #  整形しないと、実行時エラー発生する場合がある
                strList = [str(node[0])]
                for i in range(1, len(node)):
                    val = "{:16.9e}".format(node[i])
                    strList.append(val)
                line = ", ".join(strList) + "\n"
                lines.append(line)
        elif words[0] == "*ELEMENT":
            for elm in data:
                if len(elm) > 16:
                    line = ", ".join(list(map(str, elm[:16]))) + ",\n"
                    lines.append(line)
                    line = ", ".join(list(map(str, elm[16:]))) + "\n"
                    lines.append(line)
                else:
                    line = ", ".join(list(map(str, elm))) + "\n"
                    lines.append(line)
        elif words[0] == "*NSET" or words[0] == "*ELSET":
            vals = list(map(str, data))
            for i in range(0, len(vals), 5):
                line = ", ".join(vals[i:i+5]) + "\n"
                lines.append(line)
        elif words[0] == "*SURFACE":
            vals = list(map(str, data))
            for i in range(0, len(vals), 2):
                vals[i+1] = "S" + vals[i+1]
            for i in range(0, len(vals), 2):
                line = ", ".join(vals[i:i+2]) + "\n"
                lines.append(line)
            # for i in range(0, len(vals), 6):
            #     line = ", ".join(vals[i:i+6]) + "\n"
            #     lines.append(line)
        else:
            for line in data:
                lines.append(line)
    return lines

#
#  convertToInp
#-----------------
def convertToInp(inHeader, outHeader):
    """ fistrのデータをinpファイルに変換する"""
    #fistrのfileを読み込み
    fileName = inHeader + ".msh"
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    meshHeaderData = pyFistr.getHeaderNumData(lines)
    fileName = inHeader + ".cnt"
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    contHeaderData = pyFistr.getHeaderNumData(lines)
    #inpFileのheaderを作成
    abaHeaderData = convToInt(meshHeaderData, contHeaderData)
    #linesを作成し、保存
    lines = createLines(abaHeaderData)
    fileName = outHeader + ".inp"
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()
    print(_("FrontISTR形式をinp形式に変換しました。"))

#
#  printHelp
#-------------
def printHelp():
    cont  = "----------- fistr2abaqus.py -------------------------------\n"
    cont += "fistrのcnt,mshファイルからabaqus用のinpファイルに変換する。\n"
    cont += "入出力のfileHeader名のdefaultは、「FistrModel」。\n"
    cont += "<使い方>\n"
    cont += "  fistr2abaqus.py [-i FistrModel] [-o abaqus]\n"
    cont += "<使用例>\n"
    cont += "  fistr2abaqus.py\n"
    cont += "  fistr2abaqus.py -o aba\n"
    print(cont)

#
#  getArgument
#--------------
def getArgument():
    fistrHeader = "FistrModel"      #fistrのfileHeader
    abaHeader = "FistrModel"        #abaqusのfileHeader
    i = 1
    while i < len(sys.argv):
        val = sys.argv[i]
        if val == "-i":
            i += 1
            fistrHeader = sys.argv[i]
        elif val == "-o":
            i += 1
            abaHeader = sys.argv[i]
        else:
            printHelp()
            fistrHeader = ""
            abaHeader = ""
            break
        i += 1
    return [fistrHeader, abaHeader]


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    #_ = gettext.gettext

    inHeaderName, outHeaderName = getArgument()
    if inHeaderName == "":
        exit()
    convertToInp(inHeaderName, outHeaderName)
