#!/usr/bin/python3  
# -*- coding: utf-8 -*-
#
#   convertMesh.py
#
#       メッシュの変換
#
#   18/02/03    新規作成
#      03/23    fist2vtk分を追加
#      04/06    fistr_elFace:faceNo検索用のdictのkeyを修正。（バグ）
#      07/10    getNeutralMeshConts:neutral形式のsurfaceList取得方法修正（バグ）
#               getMinMaxLoc:min、max値が同じ値になった場合にエラー発生するので修正
#               各軸に平行な平面の場合、min、maxが同じ値になる。
#      09/20    neutral要素のnode順を修正（バグ）
#   19/01/04    python3用に書き換え
#      02/22    fistr_elFace:2次要素の面取得時エラー発生のため、対応。
#               nFace/2 → int(nFace/2) に修正。
#      04/20    vsCodeで修正
#      05/03    vsCodeで再修正
#   20/06/16    faceNodeOfElementDict:面の向きが誤っており修正
#      07/10    getFistrMeshConts:elementListのgrpNoの取得を修正。
#               !EGROUPがあれば、!SECTIONからEGRPを取得してgrpNoを設定。
#               inpファイルの場合、ELSETが出来上がり、これにSECTIONを設定する為。
#      08/16    getFistrHeaderData:commentが取得できる様に修正。
#               pyFistrのgetHeaderDataは、commentが取得できる。
#      08/27    fistr2vtk_elDict:五面体1次2次要素のfistr→vtk変換を修正
#      09/10    lineNodeOfFaceDict:faceからlineに変更するDictを追加
#               fistr2vtk_elDict:要素typeとして、111, 112を追加。
#   21/02/06    getAbaHeaderData,getAbaHeaderNumData:abaqus用の関数を追加
#      03/12    getAbaHeaderData:バグ修正（コメント文があるとエラーが発生）
#      03/21    aba2vtk_elを追加。
#      07/11    elmContsDict:fistrのメッシュ内容を取得するDictを追加
#   22/03/01    aba2vtk_elDict:M3D4を追加
#   23/07/24    fistr2aba_elDict:四面体2次要素のorder順修正（バグ）
#   25/01/03    getAbaHeaderNumData:*SURFACE取得時にerror発生する時あるため
#               errorTrapを追加
#      01/04    getAbaHeaderNumData:*SURFACE取得をelsetで指定している場合
#               でも、取得できるように修正。
#      01/07    getAbaHeaderNumData:*ELEMENT内の要素行が2行に渡る場合がある為、
#               これに対応。
#               fistr2vtu_elDict,aba2vtu_elDict:新規追加。vtuの六面体2次要素名が
#               vtkと異なっているので追加。
#      04/05    meshio2fistr_el:meshioからfistr形式に変換を新規追加。
#

import geometryFunc as geo

ls = "\n"

#
#  getValName
#    lineからvalName(TYPEなど）を取得
def getValName(line, valName):
    words = deleteSp(line).split(",")
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        if word.find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    return name

#
#  deleteSp
#    空白とCRを削除して返す
def deleteSp(line):
    line = line.replace(" ", "")
    line = line.replace("\n", "")
    return line

#
#  skipSp
#    空白以下をskipする
def skipSp(cont, p):
    while p < len(cont) and cont[p] <= " ":
        p += 1
    return p

#
#  nodeNoの並び順を変更する関数群
def order_2(el, n):
    element = [el[n[0]-1], el[n[1]-1]]
    return element

def order_3(el, n):
    element = [el[n[0]-1], el[n[1]-1], el[n[2]-1]]
    return element

def order_4(el, n):
    element = [el[n[0]-1], el[n[1]-1], el[n[2]-1], el[n[3]-1]]
    return element

def order_6(el,n):
    element = [el[n[0]-1], el[n[1]-1], el[n[2]-1], el[n[3]-1], el[n[4]-1],
               el[n[5]-1]]
    return element

def order_8(el, n):
    element = [el[n[0]-1], el[n[1]-1], el[n[2]-1], el[n[3]-1], el[n[4]-1],
               el[n[5]-1], el[n[6]-1], el[n[7]-1]]
    return element

def order_10(el, n):
    element = [el[n[0]-1], el[n[1]-1], el[n[2]-1], el[n[3]-1], el[n[4]-1],
               el[n[5]-1], el[n[6]-1], el[n[7]-1], el[n[8]-1], el[n[9]-1]]
    return element

def order_15(el, n):
    element = [el[n[0]-1], el[n[1]-1], el[n[2]-1], el[n[3]-1], el[n[4]-1], el[n[5]-1], el[n[6]-1], el[n[7]-1],
               el[n[8]-1], el[n[9]-1], el[n[10]-1], el[n[11]-1], el[n[12]-1], el[n[13]-1], el[n[14]-1]]
    return element

def order_20(el, n):
    element = [el[n[0]-1], el[n[1]-1], el[n[2]-1], el[n[3]-1], el[n[4]-1], el[n[5]-1], el[n[6]-1], el[n[7]-1],
               el[n[8]-1], el[n[9]-1], el[n[10]-1], el[n[11]-1], el[n[12]-1], el[n[13]-1], el[n[14]-1],
               el[n[15]-1], el[n[16]-1], el[n[17]-1], el[n[18]-1], el[n[19]-1]]
    return element

def order_27(el, n):
    element = [el[n[0]-1], el[n[1]-1], el[n[2]-1], el[n[3]-1], el[n[4]-1], el[n[5]-1], el[n[6]-1], el[n[7]-1],
               el[n[8]-1], el[n[9]-1], el[n[10]-1], el[n[11]-1], el[n[12]-1], el[n[13]-1], el[n[14]-1],
               el[n[15]-1], el[n[16]-1], el[n[17]-1], el[n[18]-1], el[n[19]-1], el[n[20]-1], el[n[21]-1],
               el[n[22]-1], el[n[23]-1], el[n[24]-1], el[n[25]-1], el[n[26]-1] ]
    return element

def order_2z(el, n):
    element = [el[n[0]], el[n[1]]]
    return element

def order_3z(el, n):
    element = [el[n[0]], el[n[1]], el[n[2]]]
    return element

def order_4z(el, n):
    element = [el[n[0]], el[n[1]], el[n[2]], el[n[3]]]
    return element

def order_6z(el, n):
    element = [el[n[0]], el[n[1]], el[n[2]], el[n[3]], el[n[4]], el[n[5]]]
    return element

def order_8z(el, n):
    element = [el[n[0]], el[n[1]], el[n[2]], el[n[3]], el[n[4]], el[n[5]], el[n[6]], el[n[7]]]
    return element

def order_10z(el, n):
    element = [el[n[0]], el[n[1]], el[n[2]], el[n[3]], el[n[4]], el[n[5]], el[n[6]], el[n[7]],
               el[n[8]], el[n[9]]]
    return element

def order_15z(el, n):
    element = [el[n[0]], el[n[1]], el[n[2]], el[n[3]], el[n[4]], el[n[5]], el[n[6]], el[n[7]],
               el[n[8]], el[n[9]], el[n[10]], el[n[11]], el[n[12]], el[n[13]], el[n[14]]]
    return element

def order_20z(el, n):
    element = [el[n[0]], el[n[1]], el[n[2]], el[n[3]], el[n[4]], el[n[5]], el[n[6]], el[n[7]],
               el[n[8]], el[n[9]], el[n[10]], el[n[11]], el[n[12]], el[n[13]], el[n[14]],
               el[n[15]], el[n[16]], el[n[17]], el[n[18]], el[n[19]]]
    return element


#
#  辞書作成
neu2fistr_elDict = {
    #type node数 要素名 node順      関数名
    ("3", 3):   ["731", [1, 2, 3],    order_3],                      #三角形1次 (tri1)
    ("3", 6):   ["732", [1, 2, 3, 4, 5, 6], order_6],                #三角形2次 (tri2)
    ("6", 4):   ["341", [1, 3, 2, 4], order_4],                      #四面体1次 (tet1)
    #("6", 10):  ["342", [1, 6,10, 3, 9, 7, 4, 2, 5, 8], order_10],   #四面体2次 (tet2)
    ("6", 10):  ["342", [1, 2, 3, 4, 6, 7, 5, 8,10, 9], order_10],   #四面体2次 (tet2)?
    ("4", 8):   ["361", [1, 2, 4, 3, 5, 6, 8, 7], order_8],          #六面体1次（hex1）
    #("4", 20):  ["362", [1,3,8,6,13,15,20,18,2,5,7,4,14,17,19,16,9,10,12,11], order_20]     #六面体2次（hex2）
    }

vol2fistr_elDict = {
    #type node数 要素名  node順  関数名
    ("1D", 2):   ["611", [1, 2], order_2],                  #beam1次
    ("2D", 3):   ["731", [1, 2, 3], order_3],               #三角形1次
    ("2D", 6):   ["732", [1, 2, 3, 4, 5, 6], order_6],      #三角形2次
    ("3D", 4):   ["341", [1, 3, 2, 4], order_4],            #四面体1次
    ("3D", 10):  ["342", [1,3,2,4,8,5,6,7,10,9], order_10]  #四面体2次
    }

gmsh2fistr_elDict = {
    #type 要素名 node順   関数名
    1: ["611", [1, 2], order_2],                            #beam1次
    2: ["731", [1, 2, 3], order_3],                         #三角形1次
    9: ["732", [1, 2, 3, 5 ,6, 4], order_6],                #三角形2次
    3: ["741", [1, 2, 3, 4], order_4],                      #四角形1次
    4: ["341", [1, 2, 3, 4], order_4],                      #四面体1次
    11:["342", [1, 2, 3, 4, 6, 7, 5, 8, 10, 9], order_10],  #四面体2次
    5: ["361", [1, 2, 3, 4, 5, 6, 7, 8], order_8],          #六面体1次
    17:["362", [1,2,3,4,5,6,7,8,9,12,14,10,17,19,20,18,11,13,15,16], order_20],     #六面体2次
    }

fistr2neu_elDict = {
     #3:["", [1, 3, 2],    order_3],                         #三角形1次 (tri1)
     3:["", [1, 2, 3],    order_3],                         #三角形1次 (tri1)
     4:["", [1, 3, 2, 4], order_4],                         #四面体1次 (tet1)
     #6:["", [1, 3, 2, 4, 6, 5], order_6],                   #三角形2次 (tri2)
     6:["", [1, 2, 3, 4, 5, 6], order_6],                   #三角形2次 (tri2)
    10:["", [1, 3, 2, 4, 6, 7, 8, 5, 10, 9], order_10]      #四面体2次 (tet2)
    }
#fistr2vol_elDict = fistr2neu_elDict
fistr2vol_elDict = {
    3:["", [1, 3, 2],    order_3],
    4:["", [1, 3, 2, 4], order_4]
    }
fistr2vtk_elDict = {
    #    vtkType fistr順をvtkNoで記述
    "111":["3",  [0,1], order_2z],                          #線要素1次
    "112":["21", [0,2,1], order_3z],                        #線要素2次
    "231":["5",  [0,1,2], order_3z],                        #三角形1次
    "232":["22", [0,1,2,5,3,4], order_6z],                  #三角形2次
    "241":["9",  [0,1,2,3], order_4z],                      #四角形1次
    "242":["23", [0,1,2,3,4,5,6,7], order_8z],              #四角形2次
    "341":["10", [0,1,2,3], order_4z],                      #四面体1次
    "342":["24", [0,1,2,3,6,4,5,7,8,9], order_10z],         #四面体2次
    #"342":["24", [0,1,2,3,5,6,4,7,8,9], order_10z],         #四面体2次
    "361":["12", [0,1,2,3,4,5,6,7], order_8z],              #六面体1次
    "362":["25", [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19], order_20z],   #六面体2次
    #"351":["13", [0,1,2,3,4,5], order_6z],                  #五面体1次
    "351":["13", [0,2,1,3,5,4], order_6z],                  #五面体1次
    #"352":["26", [0,1,2,3,4,5,8,6,7,11,9,10,12,13,14], order_15z],      #五面体2次
    "352":["26", [0,2,1,3,5,4,7,6,8,10,9,11,12,14,13], order_15z],      #五面体2次
    "611":["3",  [0,1], order_2z],                          #beam要素
    "641":["9", [0,1,2,3], order_4z],                       #beam要素（3自由度）
    "731":["5",  [0,1,2], order_3z],                        #シェル三角形1次
    "732":["22", [0,1,2,5,3,4], order_6z],                  #シェル三角形2次
    "741":["9",  [0,1,2,3], order_4z],                      #シェル四角形1次
    "761":["13", [0,1,2,3,4,5], order_6z],                  #シェル三角形1次（3自由度）
    "781":["12", [0,1,2,3,4,5,6,7], order_8z]               #シェル四角形1次（3自由度）
    }

# vtuは、六面体2次,五面体2次要素のvtk名が異なっている
fistr2vtu_elDict = {
    #    vtkType fistr順をvtkNoで記述
    "111":["3",  [0,1], order_2z],                          #線要素1次
    "112":["21", [0,2,1], order_3z],                        #線要素2次
    "231":["5",  [0,1,2], order_3z],                        #三角形1次
    "232":["22", [0,1,2,5,3,4], order_6z],                  #三角形2次
    "241":["9",  [0,1,2,3], order_4z],                      #四角形1次
    "242":["23", [0,1,2,3,4,5,6,7], order_8z],              #四角形2次
    "341":["10", [0,1,2,3], order_4z],                      #四面体1次
    "342":["24", [0,1,2,3,6,4,5,7,8,9], order_10z],         #四面体2次
    "361":["12", [0,1,2,3,4,5,6,7], order_8z],              #六面体1次
    #"362":["25", [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19], order_20z],   #六面体2次
    "362":["12", [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19], order_20z],   #六面体2次
    "351":["13", [0,2,1,3,5,4], order_6z],                  #五面体1次
    #"352":["26", [0,2,1,3,5,4,7,6,8,10,9,11,12,14,13], order_15z],      #五面体2次
    "352":["13", [0,2,1,3,5,4,7,6,8,10,9,11,12,14,13], order_15z],      #五面体2次
    "611":["3",  [0,1], order_2z],                          #beam要素
    "641":["9", [0,1,2,3], order_4z],                       #beam要素（3自由度）
    "731":["5",  [0,1,2], order_3z],                        #シェル三角形1次
    "732":["22", [0,1,2,5,3,4], order_6z],                  #シェル三角形2次
    "741":["9",  [0,1,2,3], order_4z],                      #シェル四角形1次
    "761":["13", [0,1,2,3,4,5], order_6z],                  #シェル三角形1次（3自由度）
    "781":["12", [0,1,2,3,4,5,6,7], order_8z]               #シェル四角形1次（3自由度）
    }

fistr2aba_elDict = {
    "341":["C3D4", [1,2,3,4], order_4],                     #四面体1次
    #"342":["C3D10", [1,2,3,4,6,7,5, 8,9,10], order_10],     #四面体2次
    "342":["C3D10", [1,2,3,4,7,5,6,8,9,10], order_10],     #四面体2次
    "351":["C3D6", [1,2,3,4,5,6], order_6],                 #五面体1次
    #"352":["C3D15", [1,2,3,4,5,6,8,9,7,11,12,10,13,14,15], order_15],   #五面体2次
    "352":["C3D15", [1,2,3,4,5,6,9,7,8,12,10,11,13,14,15], order_15],   #五面体2次
    "361":["C3D8", [1,2,3,4,5,6,7,8], order_8],             #六面体1次
    "362":["C3D20", [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20], order_20],    #六面体2次
    "611":["B31", [1,2], order_2],                          #beam1次
    "641":["B31", [1,2], order_2],                          #beam1次（3自由度）
    "731":["S3", [1,2,3], order_3],                         #三角形シェル1次
    "732":["S6", [1,2,3,5,6,4], order_6],                   #三角形シェル2次
    "741":["S4", [1,2,3,4], order_4],                       #四角形シェル1次
    "761":["S3", [1,2,3], order_3],                         #三角形シェル1次（3自由度）
    "781":["S4", [1,2,3,4], order_4]                        #四角形シェル1次（3自由度）
    }

aba2neu_elDict = {
    #3:["", [2, 3, 1], order_3],
    3:["", [1, 2, 3], order_3],
    4:["", [1, 3, 2, 4], order_4],
   10:["", [1, 3, 2, 4, 6, 8, 5, 7, 10, 9], order_10]
   }
aba2vol_elDict = {
    3:["", [1, 2, 3],    order_3],
    4:["", [1, 2, 3, 4], order_4]
    }

vol2elmer_elDict = {
    3:["731", [1, 2, 3],    order_3],                       #三角形1次
    4:["341", [1, 3, 2, 4], order_4]                        #四面体1次
    }

aba2vtk_elDict = {
    "C3D4":["10", [0,1,2,3], order_4z],
    "C3D10":["24", [0,1,2,3,4,5,6,7,8,9], order_10z],
    "C3D6":["13", [0,2,1,3,5,4], order_6z],
    "C3D15":["26", [0,2,1,3,5,4,8,7,6,11,10,9,12,14,13], order_15z],
    "C3D8":["12", [0,1,2,3,4,5,6,7], order_8z],
    "C3D8R":["12", [0,1,2,3,4,5,6,7], order_8z],
    "C3D20":["25", [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19], order_20z],
    "C3D20R":["25", [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19], order_20z],
    "S3":["5", [0,1,2], order_3z],
    "S4":["9", [0,1,2,3], order_4z],
    "S4R":["9", [0,1,2,3], order_4z],
    "S6":["22", [0,1,2,3,4,5], order_6z],
    "S8":["23", [0,1,2,3,4,5,6,7], order_8z],
    "S8R":["23", [0,1,2,3,4,5,6,7], order_8z],
    "M3D3":["5", [0,1,2], order_3z],
    "M3D4":["9", [0,1,2,3], order_4z],
    "B31":["3", [0,1], order_2z],
    "B31R":["3", [0,1], order_2z],
    "B32":["21", [0,2,1], order_3z],
    "B32R":["21", [0,2,1], order_3z]
    }

# vtuの場合、六面体2次,五面体2次のvtk名が異なっている。
aba2vtu_elDict = {
    "C3D4":["10", [0,1,2,3], order_4z],
    "C3D10":["24", [0,1,2,3,4,5,6,7,8,9], order_10z],
    "C3D6":["13", [0,2,1,3,5,4], order_6z],
    #"C3D15":["26", [0,2,1,3,5,4,8,7,6,11,10,9,12,14,13], order_15z],
    "C3D15":["13", [0,2,1,3,5,4,8,7,6,11,10,9,12,14,13], order_15z],
    "C3D8":["12", [0,1,2,3,4,5,6,7], order_8z],
    "C3D8R":["12", [0,1,2,3,4,5,6,7], order_8z],
    #"C3D20":["25", [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19], order_20z],
    "C3D20":["12", [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19], order_20z],
    #"C3D20R":["25", [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19], order_20z],
    "C3D20R":["12", [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19], order_20z],
    "S3":["5", [0,1,2], order_3z],
    "S4":["9", [0,1,2,3], order_4z],
    "S4R":["9", [0,1,2,3], order_4z],
    "S6":["22", [0,1,2,3,4,5], order_6z],
    "S8":["23", [0,1,2,3,4,5,6,7], order_8z],
    "S8R":["23", [0,1,2,3,4,5,6,7], order_8z],
    "M3D3":["5", [0,1,2], order_3z],
    "M3D4":["9", [0,1,2,3], order_4z],
    "B31":["3", [0,1], order_2z],
    "B31R":["3", [0,1], order_2z],
    "B32":["21", [0,2,1], order_3z],
    "B32R":["21", [0,2,1], order_3z]
    }

meshio2fistr_elDict = {
    #meshioName     fistr   node順
    "line":         ["611", [1,2], order_2],
    "triangle":     ["731", [1,2,3], order_3],
    "quad":         ["741", [1,2,3,4], order_4],
    #"tetra":        ["341", [1,2,3,4], order_4],
    "tetra":        ["341", [1,3,2,4], order_4],
    #"tetra10":      ["342", [1,2,3,4,6,7,5,8,9,10], order_10],
    "tetra10":      ["342", [1,3,2,4,6,5,7,8,10,9], order_10],
    #"hexahedron":   ["361", [1,2,3,4,5,6,7,8], order_8],
    "hexahedron":   ["361", [1,4,3,2,5,8,7,6], order_8],
    #"hexahedron20": ["362", [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20], order_20],
    "hexahedron20": ["362", [1,4,3,2,5,8,7,6,12,11,10,9,16,15,14,13,17,20,19,18], order_20],
    #"wedge":        ["351", [1,2,3,4,5,6], order_6]
    "wedge":        ["351", [1,3,2,4,6,5], order_6],
    "wedge15":      ["352", [1,3,2,4,6,5,8,7,9,11,10,12,13,15,14], order_15]
    }

fistr2meshio_elDict = {
    #fistr   meshioName      node順
    "111": ["line",         [1,2], order_2],
    "611": ["line",         [1,2], order_2],
    "112": ["line3",        [1,2,3], order_3],
    "731": ["triangle",     [1,2,3], order_3],
    "231": ["triangle",     [1,2,3], order_3],
    "232": ["triangle6",    [1,2,3,6,4,5], order_6],
    "741": ["quad",         [1,2,3,4], order_4],
    "241": ["quad",         [1,2,3,4], order_4],
    "242": ["quad8",        [1,2,3,4,5,6,7,8], order_8],
    "341": ["tetra",        [1,3,2,4], order_4],
    "342": ["tetra10",      [1,3,2,4,6,5,7,8,10,9], order_10],
    "361": ["hexahedron",   [1,4,3,2,5,8,7,6], order_8],
    "362": ["hexahedron20", [1,4,3,2,5,8,7,6,12,11,10,9,16,15,14,13,17,20,19,18], order_20],
    "351": ["wedge",        [1,3,2,4,6,5], order_6],
    "352": ["wedge15",      [1,3,2,4,6,5,8,7,9,11,10,12,13,15,14], order_15]    
    }


#
#  vol2elmer_el
#    vol -> elmer 変換
#    節点数から要素typeを判断し、node順を入れ替える
def vol2elmer_el(element):
    nNode = len(element)
    [name, n, func] = vol2elmer_elDict[nNode]
    newElement = func(element, n)
    return [name, newElement]

#
#  neu2fistr_el
#    neutral -> FrontISTR　変換
#    節点数から要素typeを判断し、node順を入れ替える
def neu2fistr_el(elType, element):
    nNodes = len(element)
    key = (elType, nNodes)
    [name, n, func] = neu2fistr_elDict[key]
    newElement = func(element, n)
    return [name, newElement]

#
#  fistr2neu_el
#    FrontISTR -> neutral 変換
#    節点数から要素typeを判断し、node順を入れ替える
def fistr2neu_el(element):
    nNode = len(element)
    [name, n, func] = fistr2neu_elDict[nNode]
    newElement = func(element, n)
    return [name, newElement]

#
#  fistr2vol_el
#    FrontISTR -> vol 変換
#    節点数から要素typeを判断し、node順を入れ替える
def fistr2vol_el(element):
    nNode = len(element)
    [name, n, func] = fistr2vol_elDict[nNode]
    newElement = func(element, n)
    return [name, newElement]
    #return fistr2neu_el(element)

#
#  fistr2vtk_el
#    FrontISTR -> vtk 変換
#    要素typeから、node順を入れ替える
def fistr2vtk_el(elType, element):
    [name, n, func] = fistr2vtk_elDict[elType]
    newElement = func(element, n)
    return [name, newElement]

#
#  fistr2vtu_el
#    FrontISTR -> vtu 変換
#    要素typeから、node順を入れ替える（六面体2次要素のvtu名がvtkと異なっている)
def fistr2vtu_el(elType, element):
    [name, n, func] = fistr2vtu_elDict[elType]
    newElement = func(element, n)
    return [name, newElement]

#
#  fistr2aba_el
#    FrontISTR -> abaqus 変換
#    fistrの要素typeから、node順を入れ替える
def fistr2aba_el(elType, element):
    [name, n, func] = fistr2aba_elDict[elType]
    abaElement = func(element, n)
    return [name, abaElement]

#
#  vol2fistr_el
#    vol -> FrontISTR 変換
#    nD(1D,2D,3D)と節点数から要素typeを判断し、node順を入れ替える
def vol2fistr_el(nD, element):
    nNode = len(element)
    key = (nD, nNode)
    [name, n, func] = vol2fistr_elDict[key]
    newElement = func(element, n)
    return [name, newElement]
    #return neu2fistr_el(element)

#
#  gmsh2fistr_el
#    vol -> FrontISTR 変換
def gmsh2fistr_el(elType, element):
    [name, n, func] = gmsh2fistr_elDict[elType]
    newElement = func(element, n)
    return [name, newElement]

#  aba2neu_el
#    abaqus -> neutral 変換
#    節点数から要素typeを判断し、node順を入れ替える
def aba2neu_el(element):
    nNode = len(element)
    [name, n, func] = aba2neu_elDict[nNode]
    newElement = func(element, n)
    return [name, newElement]

#  aba2vol_el
#    abaqus -> vol 変換
def aba2vol_el(element):
    nNode = len(element)
    [name, n, func] = aba2vol_elDict[nNode]
    newElement = func(element, n)
    return [name, newElement]
    #return aba2neu_el(element)

#
#  aba2vtk_el
#    abaqus -> vtk 変換
def aba2vtk_el(elType, element):
    (name, n, func) = aba2vtk_elDict[elType]
    newElement = func(element, n)
    return name, newElement

#
#  aba2vtu_el
#    abaqus -> vtu 変換 (六面体2次要素のvtu名がvtkと異なっている)
def aba2vtu_el(elType, element):
    (name, n, func) = aba2vtu_elDict[elType]
    newElement = func(element, n)
    return name, newElement

#
#  meshio2fistr_el
def meshio2fistr_el(elType, element):
    """ meshio -> fistr 変換"""
    (name, n, func) = meshio2fistr_elDict[elType]
    newElement = func(element, n)
    return name, newElement

#
#  fistr2meshio_el
def fistr2meshio_el(elType, element):
    """ fistr -> meshio 変換"""
    (name, n, func) = fistr2meshio_elDict[elType]
    newElement = func(element, n)
    return name, newElement

#  ----------------- FrontISTR専用 ----------------------------

faceElmDict = {
    "tet":[[1, [1,2,3]],
           [2, [1,2,4]],
           [3, [2,3,4]],
           [4, [3,1,4]]],
    "hex":[[1, [1,2,3,4]],
           [2, [5,6,7,8]],
           [3, [1,2,6,5]],
           [4, [2,3,7,6]],
           [5, [3,4,8,7]],
           [6, [4,1,5,8]]],
    "prs":[[1, [1,2,3]],
           [2, [4,5,6]],
           [3, [1,2,5,4]],
           [4, [2,3,6,5]],
           [5, [3,1,4,6]]]
    }

#  要素のnode数からfaceNoを取得するdictを取得する
#    faceNoのkeyは、「sum(faceNodeNo) * min(faceNodeNo)」としている。
faceElmFromNodesDict = {
    #tetra 1次
    #node数 nElm faceNoDict
    #          key faceNo
     4:[4,   {  6:1,        #[1,2,3], No.1  key=(1+2+3)*min()=6
                7:2,        #[1,2,4], No.2
               18:3,        #[2,3,4], No.3
                8:4 }],     #[3,1,4], No.4
    #hexa 1次
    8:[8,    { 10:1,        #[1,2,3,4], 1
              130:2,        #[5,6,7,8], 2
               14:3,        #[1,2,6,5], 3
               36:4,        #[2,3,7,6], 4
               66:5,        #[3,4,8,7], 5
               18:6 }],     #[4,1,5,8], 6
    #tetra 2次
    10:[4,   {  6:1,        #[1,2,3], No.1
                7:2,        #[1,2,4], No.2
               18:3,        #[2,3,4], No.3
                8:4 }],     #[3,1,4], No.4
    #hexa 2次
    20:[8,   { 10:1,        #[1,2,3,4], 1
              130:2,        #[5,6,7,8], 2
               14:3,        #[1,2,6,5], 3
               36:4,        #[2,3,7,6], 4
               66:5,        #[3,4,8,7], 5
               18:6 }],     #[4,1,5,8], 6
    #prism 1次
    6:[6,    {  6:1,        #[1,2,3], 1
               60:2,        #[4,5,6], 2
               12:3,        #[1,2,5,4], 3
               32:4,        #[2,3,6,5], 4
               14:5 }],     #[3,1,4,6], 5
    #prism 2次
    15:[6,   {  6:1,        #[1,2,3], 1
               60:2,        #[4,5,6], 2
               12:3,        #[1,2,5,4], 3
               32:4,        #[2,3,6,5], 4
               14:5 }]      #[3,1,4,6], 5
    }


faceNodeOfElementDict = {
    "341":[[], [1,3,2], [1,2,4], [2,3,4], [3,1,4]],
    "342":[[], [1,6,3,5,2,7], [1,7,2,9,4,8], [2,5,3,10,4,9], [3,6,1,8,4,10]],
    "361":[[], [1,4,3,2], [5,6,7,8], [1,2,6,5], [2,3,7,6], [3,4,8,7], [4,1,5,8]],
    "362":[[], [1,12,4,11,3,10,2,9], [5,13,6,14,7,15,8,16], [1,9,2,18,6,13,5,17],
           [2,10,3,19,7,14,6,18], [3,11,4,20,8,15,7,19], [4,12,1,17,5,16,8,20]],
    "351":[[], [1,3,2], [4,5,6], [1,2,5,4], [2,3,6,5], [3,1,4,6]],
    "352":[[], [1,8,3,7,2,9], [4,12,5,10,6,11], [1,9,2,14,5,12,4,13],
           [2,7,3,15,6,10,5,14], [3,8,1,13,4,11,6,15]],
    "731":[[], [1,2,3]],                #三角形
    "732":[[], [1,6,2,4,3,5]],          #三角形2次
    "741":[[], [1,2,3,4]],              #四角形
    "781":[[], [1,2,3,4]],              #四角形dummy付き
    "761":[[], [1,2,3]],                #三角形dummy付き
    #"781":[[], [1,2,3,4,5,6,7,8]],
    #"761":[[], [1,2,3,4,5,6]],
    "611":[],
    "641":[]
    }

lineNodeOfFaceDict = {
    "231":[[], [1,2], [2,3],[3,1]],
    "731":[[], [1,2], [2,3],[3,1]],
    "232":[[], [1,6,2], [2,4,3], [3,5,1]],
    "732":[[], [1,6,2], [2,4,3], [3,5,1]],
    "241":[[], [1,2], [2,3], [3,4], [4,1]],
    "741":[[], [1,2], [2,3], [3,4], [4,1]],
    "242":[[], [1,5,2], [2,6,3], [3,7,4], [4,8,1]]
    }

funcNameDict = {
    3:order_3,
    4:order_4,
    6:order_6,
    8:order_8,
    10:order_10
    }

elmContsDict = {
    #name  type    次数　種別　　節点数  
    "341":["solid",  1, "tet",   4],
    "342":["solid",  2, "tet",   10],
    "361":["solid",  1, "hex",   8],
    "362":["solid",  2, "hex",   20],
    "351":["solid",  1, "prism", 6],
    "352":["solid",  2, "prism", 15],
    "731":["shell",  1, "tri",   3],
    "732":["shell",  2, "tri",   6],
    "741":["shell",  1, "quad",  4],
    "761":["shell",  1, "tri",   6],        #3自由度shell
    "781":["shell",  1, "quad",  8],        #3自由度shell
    "611":["beam",   1, "beam",  2],
    "641":["beam",   1, "beam",  4]         #3自由度beam
    }

#
#  normalNodeNo1
#    faceNodesを「1」始まりの標準のnodeNoに置き換える
def normalNodeNo1(elNodes, faceNodes):
    newNodes = []
    for node in faceNodes:
        i = elNodes.index(node) + 1
        newNodes.append(i)
    return newNodes

#
#  fistr_elFaceNode
#    fistr要素の面Noから節点Noを取得
#    節点順は、2次要素の場合、連番なので注意
def fistr_elFaceNode(elType, elNodes, faceNo):
    nList = faceNodeOfElementDict[elType][faceNo]
    func = funcNameDict[len(nList)]
    nodes = func(elNodes, nList)
    return nodes

#
#  fistr_elFace
#    fistr要素のfaceNoを取得
#    （表面NodeNoから要素の面Noを取得）
def fistr_elFace(suNodes, elNodes):
    nEl = len(elNodes)
    nFace = len(suNodes)
    if nFace == 6 or nFace == 8:
        #2次要素の場合、主nodeのみに修正
        suNodes = suNodes[:int(nFace/2)]
    #要素の主node数とdictを取得
    [nElm, faceDict] = faceElmFromNodesDict[nEl]
    #主nodeのみに変換
    elNodes = elNodes[:nElm]
    #faceのnodeNoを標準Noに変更
    normalFace = normalNodeNo1(elNodes, suNodes)
    #標準nodeNoの 合計＊min(faceNodeNo) を求める
    sumNode = sum(normalFace) * min(normalFace)
    #標準nodeNoの合計からfaceNoを取得
    faceNo = faceDict[sumNode]
    return faceNo

#
#  getFistrHeaderData
#    fistr形式のlinesからheaderとその内容をstr形式で取得
def getFistrHeaderData(lines):
    flag = 0
    headerData = []
    data = []
    header = ""
    for i in range(len(lines)):
        line = lines[i]
        if line == "":
            #空文の処理
            flag = 1
        elif line[:1] == "!" and line[:2] != "!!":
            #headerの処理
            if header != "":
                headerData.append([header, data])
            header = lines[i]
            data = []
            flag = 1
        elif line[:2] == "!!" or line[0] == "#":
            #commentの処理
            headerData.append([header, data])
            header = line
            data = []
            flag = 1
        else:
            #header, comment以外の処理
            flag = 0
        #data部を作成
        if flag == 0:
            data.append(lines[i])
    headerData.append([header, data])
    return headerData

#
#  getFistrHeaderNumData
#    headerとそのnumDataを取得
#    NODE:      [nodeNo, locx, locy, locz]形式で数値dataを取得
#    ELEMENT:   [elNo, node1, node2, node3, ...]形式で数値データを取得
#    EGROUP:    [ele1, ...]形式で数値データを取得
#    SGROUP:    [sur1, ...]形式で数値データを取得
#    NGROUP:    [node1, ...]形式で数値データを取得
def getFistrHeaderNumData(lines):
    headerData = getFistrHeaderData(lines)
    i = 0
    for hData in headerData:
        header = hData[0]
        if header[:len("!NODE")] == "!NODE":
            #print "  getting NODE..."
            dLines = hData[1]
            data = []
            for line in dLines:
                words = line.split(",")
                dataLine = [int(words[0]), float(words[1]), float(words[2]), float(words[3])]
                data.append(dataLine)
            headerData[i][1] = data
        elif header[:len("!ELEMENT")] == "!ELEMENT":
            #typeName = getValName(header, "TYPE")
            #print "  getting ELEMENT " + typeName + "..."
            dLines = hData[1]
            data = []
            for line in dLines:
                words = line.split(",")
                elData = list(map(int, words))      #wordsをintListに変換
                # elData = []
                # for word in words:
                #     elData.append(int(word))
                data.append(elData)
            headerData[i][1] = data
        elif (header[:len("!EGROUP")] == "!EGROUP" or header[:len("!SGROUP")] == "!SGROUP"
            or header[:len("!NGROUP")] == "!NGROUP"):
            item = deleteSp(header).split(",")[0][1:]
            if item == "EGROUP":
                typeName = getValName(header, "EGRP")
            elif item == "SGROUP":
                typeName = getValName(header, "SGRP")
            elif item == "NGROUP":
                typeName = getValName(header, "NGRP")
            #print "  getting " + item + " " + typeName + "..."
            dLines = hData[1]
            data = []
            for line in dLines:
                words = line.split(",")
                elData = list(map(int, words[:-1]))      #wordsをintListに変換
                # elData = []
                # for word in words[:-1]:
                #     elData.append(int(word))
                if skipSp(words[-1], 0) != len(words[-1]):
                    elData.append(int(words[-1]))
                data += elData
            headerData[i][1] = data
        i += 1
    return headerData

#
#  getFistrMeshConts
#    fistr形式のheaderNumhDataからnodeList, elementListを取得する
def getFistrMeshConts(headerNumData):

    def isEgroup():
        """ !EGROUPが存在するかチェック"""
        ans = False
        for headerData in headerNumData:
            header = headerData[0]
            words = deleteSp(header).split(",")
            if words[0] == "!EGROUP":
                ans = True
                break
        return ans

    #nodeList作成  nodeList[i] -> [[x,y,z], [eli, elj...]]
    #                             座標      要素No
    #  listの準備
    maxNo = -1
    for headerData in headerNumData:
        words = deleteSp(headerData[0]).split(",")
        if words[0] == "!NODE":
            nodeData = headerData[1]
            for data in nodeData:
                if data[0] > maxNo:
                    maxNo = data[0]
    nodeList = [ [] for i1 in range(maxNo + 1) ]
    #  nodeDataを取得
    for headerData in headerNumData:
        words = deleteSp(headerData[0]).split(",")
        if words[0] == "!NODE":
            nodeData = headerData[1]
            for data in nodeData:
                nodeNo = data[0]
                loc = data[1:]
                nodeList[nodeNo] = [loc, []]
    
    #elementList作成  elementList -> [[grp, node1, node2,...]]
    #  listの準備
    maxNo = -1
    for headerData in headerNumData:
        words = deleteSp(headerData[0]).split(",")
        if words[0] == "!ELEMENT":
            elData = headerData[1]
            for data in elData:
                if data[0] > maxNo:
                    maxNo = data[0]
    elementList = [ [] for i1 in range(maxNo+1) ]
    #!EGROUPが定義されているか
    if isEgroup() == False:
        #そのままelementDataを取得
        grpNo = 1
        for headerData in headerNumData:
            words = deleteSp(headerData[0]).split(",")
            if words[0] == "!ELEMENT":
                elType = getValName(headerData[0], "TYPE")
                elData = headerData[1]
                for data in elData:
                    #要素Dataを保存
                    elNo = data[0]
                    elm = data[1:]
                    element = [grpNo] + elm
                    elementList[elNo] = [element] + [elType]
                    #nodeListに要素Noを保存
                    for node in elm:
                        nodeList[node][1] += [elNo]
                grpNo += 1
    else:
        #SECTION行からEGRPを取得する
        #  !ELEMENTのEGRPを取得
        elmDict = {}
        for headerData in headerNumData:
            header = headerData[0]
            words = deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                egrpName = getValName(header, "EGRP")
                for elm in headerData[1]:
                    elNo = elm[0]
                    elmDict[elNo] = [egrpName]
        #  !EGROUPからEGRPを取得
        for headerData in headerNumData:
            header = headerData[0]
            words = deleteSp(header).split(",")
            if words[0] == "!EGROUP":
                egrpName = getValName(header, "EGRP")
                for elNo in headerData[1]:
                    elmDict[elNo] += [egrpName]
        #  !SECTIONのEGRPを取得
        egrpNames = []
        egrpNo = 1
        for headerData in headerNumData:
            header = headerData[0]
            words = deleteSp(header).split(",")
            if words[0] == "!SECTION":
                egrpName = getValName(header, "EGRP")
                egrpNames.append([egrpName, egrpNo])
                egrpNo += 1
        #  elementDataを取得
        for headerData in headerNumData:
            header = headerData[0]
            words = deleteSp(header).split(",")
            if words[0] == "!ELEMENT":
                elType = getValName(headerData[0], "TYPE")
                elData = headerData[1]
                for data in elData:
                    elNo = data[0]
                    elm = data[1:]
                    grpNo = 0
                    for egrpName, egrpNo in egrpNames:
                        if egrpName in elmDict[elNo]:
                            grpNo = egrpNo
                            break
                    element = [grpNo] + elm
                    elementList[elNo] = [element] + [elType]
                    #nodeListに要素Noを保存
                    for node in elm:
                        nodeList[node][1] += [elNo]
    return [nodeList, elementList]


#---------------- neutral形式専用 -------------------------

#
#  createNeutralMeshLines
#    nodeList, surfaceList, elementListから、neutral形式のmeshLinesを作成
def createNeutralMeshLines(nodeList, surfaceList, elementList):

    def createLines(dataList):
        dataLines = []
        nMax = 0
        for data in dataList:
            if len(data) > 0:
                items = data[0]
                words = list(map(str, items))   #itemsをstrListに変換
                # words = []
                # for item in items:
                #     words.append(str(item))
                line = " ".join(words) + ls
                dataLines += [line]
                nMax += 1
        lines = [str(nMax) + ls] + dataLines
        return lines

    #node行を作成
    lines = createLines(nodeList)
    #element行を作成
    lines += createLines(elementList)
    #surface行を作成
    lines += createLines(surfaceList)
    return lines    

#
#  getNeutralMeshConts
#    neutral形式のlinesから、nodeList, elementList, surfaceListを取得
def getNeutralMeshConts(lines):
    #各linesを取得
    i = 0
    nNode = int(lines[i])
    i += 1
    nodeLines = lines[i:i+nNode]
    i += nNode
    nElm = int(lines[i])
    i += 1
    elLines = lines[i:i+nElm]
    i += nElm
    nSur = int(lines[i])
    i += 1
    surLines = lines[i:i+nSur]

    #nodeList作成 nodeList[i] -> [[x,y,z], [eli, elj...], [sui, suj...]]
    #                             座標      要素No         surfaceNo
    #  要素No:    surfaceが属する要素の面番号を取得するため
    #  surfaceNo: 表面の節点を取得する為
    nodeList = [ [] for i1 in range(len(nodeLines) + 1) ]
    for i in range(len(nodeLines)):
        line = nodeLines[i]
        words = line.split()
        loc = [float(words[0]), float(words[1]), float(words[2])]
        nodeList[i+1] = [loc, [], []]
    #elementList作成 elementList -> [[grp, node1, node2,...]]
    elementList = [ [] for i1 in range(len(elLines) + 1) ]
    for i in range(len(elLines)):
        line = elLines[i]
        words = line.split()
        grp = words[0]
        nEl = len(words[1:])
        element = [ [] for i in range(nEl+1) ]        #要素1ヶ分のlistを作成
        element[0] = int(grp)           #grpNo
        n = 1
        for word in words[1:]:
            nodeNo = int(word)
            element[n] = nodeNo
            nodeList[nodeNo][1] += [i+1]     #nodeListに要素Noを保存
            n += 1
        elementList[i+1] = [element]
    #surfaceList作成  surfaceList -> [[grp, node1, node2,...], [eli, elj...]]
    #                                   表面要素内容             要素No
    #  要素No: surfaceが属する面番号を取得するため
    surfaceList = [ [] for i1 in range(len(surLines) + 1) ]
    for i in range(len(surLines)):
        line = surLines[i]
        words = line.split()
        grp = words[0]
        nP = len(words[1:])
        #surface = [[]] * (nP+1)
        surface = [ [] for i in range(nP+1) ]
        surface[0] = int(grp)
        n = 1
        for word in words[1:]:
            nodeNo = int(word)
            surface[n] = nodeNo
            nodeList[nodeNo][2] += [i+1]   #nodeListにsurfaceNoを保存
            n += 1
        surfaceList[i+1] = [surface, []]
    #surfaceListに要素Noを追加
    for i in range(len(surfaceList)):
        surface = surfaceList[i]
        if len(surface) > 0:
            #nodeNoを取得
            nodes = surface[0][1:]
            if len(nodes) == 3 or len(nodes) == 6:
                #三角形
                nodes = nodes[:3]
            else:
                #四角形
                nodes = nodes[:4]
            #nodeから属する要素Noを取得
            elms = []
            for node in nodes:
                elms += nodeList[node][1]
            #同じ要素がnode数分ある要素を取得
            elms.sort()
            before = elms[0]
            n = 1
            for elm in elms[1:]:
                if elm == before:
                    n += 1
                else:
                    if n >= len(nodes):
                        surfaceList[i][1] += [before]
                        #break
                    n = 1
                    before = elm
            if n >= len(nodes):
                surfaceList[i][1] += [before]
    return [nodeList, elementList, surfaceList]

#--------------- stl 専用 -------------------------------------------

#
#  getStlNodeSurfaceEdgeList
#    stl三角形からnodeList、surfaceList、edgeListを作成する
#    入力:三角形座標のlist（[tri0],[tri1],[tri2], ...]
#                           tri = [loc0,loc1,loc2]
#                           loc = [x,y,z]
#    戻り:[nodeList, surfaceList, edgeList]
#         点：nodeList = [ [[x,y,z],[sur0,sur1...]], ...]
#         面：surfaceList = [ [[n0,n1,n2]], ...]
#         線：edgeList = [ [[n0, n1],[sur1,sur2...]], ...]
def getStlNodeSurfaceEdgeList(stlLocs):

    #  setNodeNos
    #    node座標を1列で取得
    def setNodeNos(stlLocs):
        locNos = [ [] for i in range(len(stlLocs) * 3) ]
        n = 0
        for locs in stlLocs:
            for loc in locs:
                locNos[n] = [loc]
                n += 1
        return locNos

    #  getMinMaxLoc
    #    minMaxの座標を取得
    def getMinMaxLoc(locNos):
        loc = locNos[0][0]
        minx = loc[0]; maxx = loc[0]
        miny = loc[1]; maxy = loc[1]
        minz = loc[2]; maxz = loc[2]
        for loc in locNos:
            x = loc[0][0]; y = loc[0][1]; z = loc[0][2]
            if minx > x:
                minx = x
            if miny > y:
                miny = y
            if minz > z:
                minz = z
            if maxx < x:
                maxx = x
            if maxy < y:
                maxy = y
            if maxz < z:
                maxz = z
        if (maxx == minx):
            dx = ((maxy-miny) + (maxz-minz))/2.0/10.0
            maxx += dx; minx -= dx
        if (maxy == miny):
            dy = ((maxx-minx) + (maxz-minz))/2.0/10.0
            maxy += dy; miny -= dy
        if (maxz == minz):
            dz = ((maxx-minx) + (maxy-miny))/2.0/10.0
            maxz += dz; minz -= dz
        return [[minx, miny, minz], [maxx, maxy, maxz]]

    #  getLocAddress
    #    座標のaddressを取得
    def getLocAddress(loc, minMaxLoc):
        [x,y,z] = loc
        [[minx, miny, minz], [maxx, maxy, maxz]] = minMaxLoc
        ax = int((x - minx) / (maxx - minx) * 100.0 + 0.5)
        ay = int((y - miny) / (maxy - miny) * 100.0 + 0.5)
        az = int((z - minz) / (maxz - minz) * 100.0 + 0.5)
        return [ax, ay, az]    

    #  getNodeAddress
    #    node座標のaddressを作成
    def getNodeAddress(locNos, minMaxLoc):
        #3次元配列を作成
        nodeAdd = nodeAdd = [[[ [] for i3 in range(101)] for i2 in range(101)] for i1 in range(101)]
        #座標をaddressにセット
        #[[minx, mixy, mixz], [maxx, maxy, maxz]] = minMaxLoc
        for i in range(len(locNos)):
            loc = locNos[i][0]
            [ax, ay, az] = getLocAddress(loc, minMaxLoc)
            nodeAdd[ax][ay][az] += [i]
        return nodeAdd

    #  getNewNodeNo
    #    nodeNoを取得
    def getNewNodeNo(locNos, locNo, nearLocs, tol, lastNodeNo):
        loc = locNos[locNo][0]
        #既に設定されているか
        if len(locNos[locNo]) > 1:
            #nodeNoを読み取り戻る
            nodeNo = locNos[locNo][1]
            return [nodeNo, locNos, lastNodeNo]
        #同じlocationのnodeNoを取得（近隣のnodeNoから）
        toll = geo.mag(loc) * tol + tol     #toleranceを設定
        sameLocNos = []
        for nearNo in nearLocs:
            nearLoc = locNos[nearNo][0]
            l = geo.length(loc, nearLoc)
            if l < toll:
                #同じlocationのnodeを保存
                sameLocNos.append(nearNo)
        #同じlocationでnodeNoが設定されていれば、読み取り戻る
        if len(locNos[sameLocNos[0]]) > 1:
            nodeNo = locNos[sameLocNos[0]][1]
            return [nodeNo, locNos, lastNodeNo]
        #新規にnodeNoを設定する
        lastNodeNo += 1
        nodeNo = lastNodeNo
        #同じlocation内のnodeにnodeNoを設定
        for sameLocNo in sameLocNos:
            if len(locNos[sameLocNo]) > 1:
                locNos[sameLocNo][1] = nodeNo
            else:
                locNos[sameLocNo] += [nodeNo]
        return [-1, locNos, lastNodeNo]

    #  getEdgeNo
    #    edgeAddressからedgeNoを取得
    def getEdgeNo(edge, edgeList, edgeAdd):
        if edgeAdd[edge[0]][edge[1]] != 0:
            edgeNo = edgeAdd[edge[0]][edge[1]]
        elif edgeAdd[edge[1]][edge[0]] != 0:
            edgeNo = edgeAdd[edge[1]][edge[0]]
        else:
            edgeNo = len(edgeList)
            edgeList.append([edge, []])
            edgeAdd[edge[0]][edge[1]] = edgeNo
        return [edgeNo, edgeList, edgeAdd]

    #
    #toleranceを設定（有効数字6桁）
    tol = 1.0e-6

    #nodeList, surfaceListを作成
    #  minMaxLocを取得
    locNos = setNodeNos(stlLocs)
    minMaxLoc = getMinMaxLoc(locNos)
    #  nodeAddを作成
    nodeAdd = getNodeAddress(locNos, minMaxLoc)
    #  nodeNoを取得
    nodeList = [[]]
    surfaceList = [ [] for i in range(len(stlLocs)+1)]
    lastNodeNo = 0
    n = 0
    for i in range(len(stlLocs)):
        surNo = i + 1
        locs = stlLocs[i]
        surNodeNos = []
        for loc in locs:
            [ax, ay, az] = getLocAddress(loc, minMaxLoc)
            nearLocs = nodeAdd[ax][ay][az]
            #nodeNoを取得、locNos[i]の最後にnodeNoを書き込む、lastNodeを更新
            #新規にnodeNoを追加した場合、nodeNoは「-1」が戻る
            [nodeNo, locNos, lastNodeNo] = getNewNodeNo(locNos, n, nearLocs, tol, lastNodeNo)
            if nodeNo < 0:
                #新規にnodeNoが設定された場合
                nodeNo = lastNodeNo
                nodeList.append([loc, []])
            nodeList[nodeNo][1] += [surNo]
            surNodeNos.append(nodeNo)
            n += 1
        surfaceList[surNo] = [surNodeNos]
    #メモリクリア
    locNos = []
    nodeAdd = []

    #edgeListを作成
    #  edgeのaddress用配列を作成
    edgeAdd = [[0 for i1 in range(len(nodeList)+1)] for i2 in range(len(nodeList)+1)]
    #  最初のedgeを設定
    edge0 = [surfaceList[1][0][0], surfaceList[1][0][1]]
    edgeAdd[edge0[0]][edge0[1]] = 1
    edgeList = [[], [edge0, []]]
    #  2番目以降のedgeListを作成
    for i in range(1, len(surfaceList)):
        surface = surfaceList[i]
        surNo = i
        edge = [surface[0][0], surface[0][1]]
        [edgeNo, edgeList, edgeAdd] = getEdgeNo(edge, edgeList, edgeAdd)
        edgeList[edgeNo][1] += [surNo]
        edge = [surface[0][1], surface[0][2]]
        [edgeNo, edgeList, edgeAdd] = getEdgeNo(edge, edgeList, edgeAdd)
        edgeList[edgeNo][1] += [surNo]
        edge = [surface[0][2], surface[0][0]]
        [edgeNo, edgeList, edgeAdd] = getEdgeNo(edge, edgeList, edgeAdd)
        edgeList[edgeNo][1] += [surNo]
    return [nodeList, surfaceList, edgeList]        
        

#  ----------------- abaqus専用 ----------------------------

#
#  getAbaHeaderData
#-------------------
def getAbaHeaderData(lines):
    """ abaqus形式のlinesからheaderとその内容をstr形式で取得"""
    flag = 0
    headerData = []
    data = []
    header = ""
    for i in range(len(lines)):
        line = lines[i]
        #空文？
        if line == "":
            #空文の処理
            flag = 1
        #header？
        elif line[:1] == "*" and line[:2] != "**":
            #headerの処理
            if header != "":
                headerData.append([header, data])
            header = lines[i]
            data = []
            flag = 1
        #comment?
        elif line[:2] == "**":
            flag = 1
            #commentの処理
            # headerData.append([header, data])
            # header = line
            # data = []
            # flag = 1
        else:
            #header, comment以外の処理
            flag = 0
        #data部を作成
        if flag == 0:
            data.append(lines[i])
    headerData.append([header, data])
    return headerData

#
#  getAbaHeaderNumData
#----------------------
def getAbaHeaderNumData(lines):
    """
    headerとそのnumDataを取得
    NODE:      [nodeNo, locx, locy, locz]形式で数値dataを取得
    ELEMENT:   [elNo, node1, node2, node3, ...]形式で数値データを取得
    ELSET:     [ele1, ...]形式で数値データを取得
    SURFACE:   [sur1, ...]のFistr形式で数値データを取得
    NSET:      [node1, ...]形式で数値データを取得
    """
    headerData = getAbaHeaderData(lines)
    egrpDict = {}
    for i in range(len(headerData)):
        hData = headerData[i]
        header = hData[0]
        hWords = deleteSp(header).upper().split(",")
        if hWords[0] == "*NODE":
            dLines = hData[1]
            data = []
            for line in dLines:
                words = line.split(",")
                dataLine = [int(words[0]), float(words[1]), float(words[2]), float(words[3])]
                data.append(dataLine)
            headerData[i][1] = data
        elif hWords[0] == "*ELEMENT":
            dLines = hData[1]
            data = []
            allWords = []
            for line in dLines:
                words = deleteSp(line).split(",")
                if words[-1] == "":
                    allWords += words[:-1]
                else:
                    allWords += words
                    elData = list(map(int, allWords))
                    data.append(elData)
                    allWords = []
            headerData[i][1] = data
            egrp = getValName(header, "ELSET")
            if egrp != "":
                egrpDict[egrp] = data
        elif (hWords[0] == "*ELSET" or hWords[0] == "*NSET"):
            dLines = hData[1]
            data = []
            for line in dLines:
                words = line.split(",")
                elData = list(map(int, words[:-1]))      #wordsをintListに変換
                if skipSp(words[-1], 0) != len(words[-1]):
                    elData.append(int(words[-1]))
                data += elData
            headerData[i][1] = data
            egrp = getValName(header, "ELSET")
            egrpDict[egrp] = data
        elif hWords[0] == "*SURFACE":
            dLines = hData[1]
            data = []
            newData = []
            for line in dLines:
                words = deleteSp(line).split(",")
                if words[-1] == "":
                    words = words[:-1]
                data += words
            for ii in range(0, len(data), 2):
                try:
                    #そのまま取得
                    elmNo = int(data[ii])
                    faceNo = int(data[ii+1][1:])
                    newData.append(elmNo)
                    newData.append(faceNo)
                except:
                    egrpName = data[ii]
                    faceNo = int(data[ii+1][1:])
                    if egrpName in egrpDict.keys():
                        elms = egrpDict[egrpName]
                        for elm in elms:
                            newData.append(elm)
                            newData.append(faceNo)
                    else:
                        newData = []                        
                        break
            headerData[i][1] = newData
    return headerData

