#!/usr/bin/python3
#
#          convDimensionMatCsv.py
#
#       材料DBのdimensionを変換したfileを作成する。
#
#   23/12/12    新規作成
#      12/22    単位変換の辞書作成追加
#      12/24    単位変換辞書に、EasyISTRが使っている全単位を追加
#      12/29    convDimensionOtherMatFiles:workFolder内のその他matFile
#               の単位変換を追加（MULTILINEARのfileのみ追加）
#   24/01/01    convertAllValues:非線形dataの単位変換を追加
#               createNewLinesOfMatDB:非線形dataの単位表示を追加
#      01/02    全非線形材料dataの単位系変換を追加
#      03/26    unitDictに"1/press"を追加
#               convertAllValues:全非線形dataの単位の取得を追加
#               unitKey_db:新規追加。これで材料DBの単位を取得する。
#   24/07/07    openにencoding="utf-8"を追加
#      09/28    definedUnitDict:定義済みの単位系が外からアクセス
#               できる様に独立させる。
#      10/03    createNewLinesOfMatDB:matDB保存時、値を有効桁数8桁で丸めて
#               保存する様に修正。（変換を繰り返すと、誤差が累積するため）
#      11/11    unitKey_db:#22の単位表示を変更。
#

import os, sys
import glob
import pyFistr

#
def unitKey_db():
    """ 材料DB内の単位取得の為のkeyListを取得"""
    keys = ["", "press", "-", "rho", "",                    #0,1,2,3,4,
            "thermCond", "specHeat", "press", "press", "-", #5,6,7,8,9,
            "press", "-", "-", "1/press", "-",              #10,11,12,13,14,
            "", "press", "press", "1/press", "press",       #15,16,17,18,19,
            "press", "1/press", "-", "-", "1/press",        #20,21,22,23,24,
            "press", "s", "", "", ""]                       #25,26,27,28,29
    return keys

#
def definedUnitDict():
    """ 定義済み単位系の辞書を返す"""
    unitDict = {
        "kg_m_s": {
            "mass":  "kg",
            "len":   "m",
            "time":  "s",
            "press": "Pa",
            "1/press": "1/Pa",
            "force": "N",
            "moment": "N.m",
            "forceB": "N/m3",           #体積力
            "spring": "N/m",            #バネ定数
            "springR": "N.m/rad",       #バネ定数（回転）
            "rho":   "kg/m3",           #密度
            "area":  "m2",              #面積
            "velo":  "m/s",             #速度
            "accel": "m/s2",            #加速度
            "intMoment": "m4",          #断面二次モーメント
            "thermCond": "W/m.K",       #熱伝導率
            "specHeat":  "J/kg.K",      #比熱
            "watt": "J/s",              #ワット
            "wattF": "J/m2.s",          #ワット/面積
            "wattV": "J/m3.s",          #ワット/体積
            "heatTr": "J/m2.s.K",       #熱伝達率
            "Hz":   "1/s",              #Hz
            "omega": "rad/s"            #角速度
            },
        "ton_mm_s": {
            "mass":  "ton",
            "len":   "mm",
            "time":  "s",
            "press": "MPa",
            "1/press": "1/MPa",
            "force": "N",
            "moment": "N.mm",
            "forceB": "N/mm3",          #体積力
            "spring": "N/mm",           #バネ定数
            "springR": "N.mm/rad",      #バネ定数（回転）
            "rho":   "ton/mm3",         #密度
            "area":  "mm2",             #面積
            "velo":  "mm/s",            #速度
            "accel": "mm/s2",           #加速度
            "intMoment": "mm4",         #断面二次モーメント
            "thermCond": "mW/mm.K",     #熱伝導率
            "specHeat":  "mJ/ton.K",    #比熱
            "watt": "mJ/s",             #ワット
            "wattF": "mJ/mm2.s",        #ワット/面積
            "wattV": "mJ/mm3.s",        #ワット/体積
            "heatTr": "mJ/mm2.s.K",     #熱伝達率
            "Hz":   "1/s",              #Hz
            "omega": "rad/s"            #角速度
            },
        "Mg_mm_s": {
            "mass":  "Mg",
            "len":   "mm",
            "time":  "s",
            "press": "MPa",
            "1/press": "1/MPa",
            "force": "N",
            "moment": "N.mm",
            "forceB": "N/mm3",          #体積力
            "spring": "N/mm",           #バネ定数
            "springR": "N.mm/rad",      #バネ定数（回転）
            "rho":   "Mg/mm3",          #密度
            "area":  "mm2",             #面積
            "velo":  "mm/s",            #速度
            "accel": "mm/s2",           #加速度
            "intMoment": "mm4",         #断面二次モーメント
            "thermCond": "mW/mm.K",     #熱伝導率
            "specHeat":  "mJ/Mg.K",     #比熱
            "watt": "mJ/s",             #ワット
            "wattF": "mJ/mm2.s",        #ワット/面積
            "wattV": "mJ/mm3.s",        #ワット/体積
            "heatTr": "mJ/mm2.s.K",     #熱伝達率
            "Hz":   "1/s",              #Hz
            "omega": "rad/s"            #角速度
            },
        "kg_mm_ms": {
            "mass":  "kg",
            "len":   "mm",
            "time":  "ms",
            "press": "GPa",
            "1/press": "1/GPa",
            "force": "uN",
            "moment": "uN.mm",
            "forceB": "uN/mm3",         #体積力
            "spring": "uN/mm",          #バネ定数
            "springR": "uN.mm/rad",     #バネ定数（回転）
            "rho":   "kg/mm3",          #密度
            "area":  "mm2",             #面積
            "velo":  "mm/ms",           #速度
            "accel": "mm/ms2",          #加速度
            "intMoment": "mm4",         #断面二次モーメント
            "thermCond": "MW/mm.K",     #熱伝導率
            "specHeat":  "J/kg.K",      #比熱
            "watt": "J/ms",             #ワット
            "wattF": "J/mm2.ms",        #ワット/面積
            "wattV": "J/mm3.ms",        #ワット/体積
            "heatTr": "J/mm2.ms.K",     #熱伝達率
            "Hz":   "1/ms",             #Hz
            "omega": "rad/ms"           #角速度
            },
        "g_cm_s": {
            "mass":  "g",
            "len":   "cm",
            "time":  "s",
            "press": "dyn/cm2",
            "1/press": "cm2/dyn",
            "force": "dyn",
            "moment": "dyn.cm",
            "forceB": "dyn/cm3",        #体積力
            "spring": "dyn/cm",         #バネ定数
            "springR": "dyn.cm/rad",    #バネ定数（回転）
            "rho":   "g/cm3",           #密度
            "area":  "cm2",             #面積
            "velo":  "cm/s",            #速度
            "accel": "cm/s2",           #加速度
            "intMoment": "cm4",         #断面二次モーメント
            "thermCond": "erg/s.cm.K",  #熱伝導率
            "specHeat":  "erg/g.K",     #比熱
            "watt": "erg/s",            #ワット
            "wattF": "erg/cm2.s",       #ワット/面積
            "wattV": "erg/cm3.s",       #ワット/体積
            "heatTr": "erg/cm2.s.K",    #熱伝達率
            "Hz":   "1/s",              #Hz
            "omega": "rad/s"            #角速度
            },
        }
    return unitDict

#
def createUnitDict(dim):
    """ 単位の辞書作成"""
    #定義済みの辞書を取得
    unitDict = definedUnitDict()
    #unitDictに「any」（任意）を追加。
    dims = dim.split("_")
    unitDict["any"] = {
        "mass":  unit_g(dims),
        "len":   unit_m(dims),
        "time":  unit_s(dims),
        "press": unit_g_ms2(dims),
        "1/press": unit_ms2_g(dims),
        "force": unit_gm_s2(dims),
        "moment": unit_gm2_s2(dims),
        "forceB": unit_g_m2s2(dims),       #体積力
        "spring": unit_g_s2(dims),         #バネ定数
        "springR": unit_gm2_rs2(dims),     #バネ定数（回転）
        "rho":   unit_g_m3(dims),          #密度
        "area":  unit_m2(dims),            #面積
        "velo":  unit_m_s(dims),           #速度
        "accel": unit_m_s2(dims),          #加速度
        "intMoment": unit_m4(dims),        #断面二次モーメント
        "thermCond": unit_gm_ks3(dims),    #熱伝導率
        "specHeat":  unit_m2_ks2(dims),    #比熱
        "watt": unit_gm2_s2(dims),         #ワット
        "wattF": unit_g_s2(dims),          #ワット/面積
        "wattV": unit_g_ms2(dims),         #ワット/体積
        "heatTr": unit_g_ks3(dims),        #熱伝達率
        "Hz":   unit_1_s(dims),            #Hz
        "omega": unit_r_s(dims)            #角速度
        }
    #該当辞書を取得
    if dim in unitDict.keys():
        dimDict = unitDict[dim]
    else:
        #任意dimension
        dimDict = unitDict["any"]
    return dimDict

def mlsDicts():
    """ dimension変換の為の基本単位辞書を作成"""
    #質量
    mDict = {
        "Gg":  1.0e9,
        "Mg":  1.0e6,
        "ton": 1.0e6,
        "kg":  1.0e3,
        "g":   1.0,
        "mg":  1.0e-3
        }
    #長さ
    lDict = {
        "km": 1.0e3,
        "m":  1.0,
        "cm": 1.0e-2,
        "mm": 1.0e-3
        }
    #時間
    sDict = {
        "s":  1.0,
        "ms": 1.0e-3
        }
    dicts = (mDict, lDict, sDict)
    return dicts

#
#  任意のunit作成
def unit_g(dims):
    """ 質量の単位を取得"""
    return dims[0]

def unit_m(dims):
    """ 長さの単位を取得"""
    return dims[1]

def unit_s(dims):
    """ 時間の単位を取得"""
    return dims[2]

def unit_g_ms2(dims):
    """ pressの単位を取得"""
    unit = dims[0] + "/" + dims[1] + "." + dims[2] + "2"
    return unit

def unit_ms2_g(dims):
    """ 1/pressの単位を取得"""
    unit = dims[1] + "." + dims[2] + "2/" + dims[0]
    return unit

def unit_gm_s2(dims):
    """ forceの単位を取得"""
    unit = dims[0] + "." + dims[1] + "/" + dims[2] + "2"
    return unit

def unit_gm2_s2(dims):
    """ モーメントの単位"""
    unit = dims[0] + "." + dims[1] + "2/" + dims[2] + "2"
    return unit

def unit_g_m2s2(dims):
    """ 体積力の単位"""
    unit = dims[0] + "/" + dims[1] + "2." + dims[2] + "2"
    return unit

def unit_g_s2(dims):
    """ ばね定数の単位"""
    unit = dims[0] + "/" + dims[2] + "2"
    return unit

def unit_gm2_rs2(dims):
    unit = dims[0] + "." + dims[1] + "2/rad." + dims[2] + "2"
    return unit

def unit_g_m3(dims):
    """ rhoの単位を出力"""
    unit = dims[0] + "/" + dims[1] + "3"
    return unit

def unit_m2(dims):
    """ areaの単位"""
    unit = dims[1] + "2"
    return unit

def unit_m_s(dims):
    """ 速度の単位"""
    unit = dims[1] + "/" + dims[2]
    return unit

def unit_m_s2(dims):
    """ 加速度の単位"""
    unit = dims[1] + "/" + dims[2] + "2"
    return unit

def unit_m4(dims):
    """ 断面二次モーメントの単位"""
    unit = dims[1] + "4"
    return unit

def unit_gm_ks3(dims):
    """ 熱伝導率thermCondの単位を取得"""
    unit = dims[0] + "." + dims[1] + "/K." + dims[2] + "3"
    return unit

def unit_m2_ks2(dims):
    """ 比熱specHeatの単位を取得"""
    unit = dims[1] + "2" + "/K." + dims[2] + "2"
    return unit

def unit_g_ks3(dims):
    """ 熱伝達率の単位"""
    unit = dims[0] + "/K." + dims[2] + "3"
    return unit

def unit_1_s(dims):
    """ 周波数の単位を取得"""
    unit = "1/" + dims[2]
    return unit

def unit_r_s(dims):
    """ 角速度の単位"""
    unit = "rad/" + dims[2]
    return unit

#
#  getMatDBName
#---------------
def getMatDBName():
    tempDir = os.getenv("easyIstrUserPath") + os.sep + "data"
    fileName = tempDir + os.sep + "easyIstr_data"
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    for line in lines:
        words = line.split()
        if len(words) > 1:
            if words[0] == "materialDB":
                matFile = words[1]
                break
    return matFile

#
#  getDimension
#---------------
def getDimension(matFile):
    """ matDBの現在のdimensionを取得する"""
    print()
    print("dimension of source matDB file will be get.")
    print("  source file is '" + matFile + "'.")
    f = open(matFile, encoding="utf-8"); lines = f.readlines(); f.close()
    words = pyFistr.deleteSp(lines[0]).split(",")
    dim = words[0]
    dims = dim.split("_")
    if len(dims) == 3:
        error = checkDimension(dim)
        if error == 1:
            print()
            print("error: wrong dimension in source file!!")
            print()
            exit()
        return dim, lines
    else:
        print("dimension does not write in its file.")
        print("  so, 'kg_m_s' has gotton as the dimension.")
        print("  if the dimension is wrong, write the righit dimension in the first colum of first line.")
        dim = "kg_m_s"
        return dim, lines

#
#  checkDimension
def checkDimension(dim):
    """ 取得したdimensionをチェック"""
    error = 1
    dicts = mlsDicts()
    if type(dim) == str:
        dim = dim.split("_")
    if not dim[0] in dicts[0].keys():
        return error
    if not dim[1] in dicts[1].keys():
        return error
    if not dim[2] in dicts[2].keys():
        return error
    return 0

#
#  convDimension
#---------------
def convDimension(newDim):
    matFile = getMatDBName()
    currDim, lines = getDimension(matFile)
    dbData = getMatData(lines)
    #値を変換
    newData = convertMatData(currDim, newDim, dbData)
    #変換した値のlinesを作成
    newLines = createNewLinesOfMatDB(lines, newDim, newData)
    #fileに保存
    saveNewMatFile(newLines, newDim, matFile, currDim)
    
#
#  getMatData
def getMatData(lines):
    """ matDataをlist形式で取得する"""
    data = []
    for i in range(2, len(lines)):
        line = lines[i]
        words = pyFistr.deleteSp(line).split(",")
        nums = []
        nums.append(words[0])
        for ii in range(1, len(words)):
            word = words[ii]
            if word != "":
                num = float(word)
                nums.append(num)
            else:
                nums.append(word)
        data.append(nums)
    return data

#
#  convertMatData
def convertMatData(currDim, newDim, dbData):
    """ dimension変換"""
    cm, cl, cs = getConvCoeff(currDim, newDim)
    coeffs = [cm, cl, cs]
    data = convertAllValues(coeffs, dbData)
    return data

#
#  getConvCoeff
def getConvCoeff(currDim, newDim):
    """ 変換の為の基礎係数（基本単位の係数）を取得する"""
    #基本単位の辞書を取得
    mDict, lDict, sDict = mlsDicts()
    currDim = currDim.split("_")
    newDim = newDim.split("_")
    #基本単位への係数を算出
    cm = mDict[currDim[0]] / mDict[newDim[0]]
    cl = lDict[currDim[1]] / lDict[newDim[1]]
    cs = sDict[currDim[2]] / sDict[newDim[2]]
    return cm, cl, cs

#
#  convertAllValues
def convertAllValues(coeffs, dbData):
    """ DB内の全変数（ヤング率等）を変換する"""
    cm, cl, cs = coeffs         #基本単位の係数(mass,length,second)
    #ヤング率
    c = cm / (cl * cs * cs)     #係数
    #     B
    col = 1                     #場所
    dbData = convertValues(c, col, dbData)
    #密度の係数
    c = cm / (cl * cl * cl)
    #     D
    col = 3
    dbData = convertValues(c, col, dbData)
    #熱伝導率
    c = (cm * cl) / (cs * cs * cs)
    #     F
    col = 5
    dbData = convertValues(c, col, dbData)
    #比熱
    c = (cl * cl) / (cs * cs)
    #     G
    col = 6
    dbData = convertValues(c, col, dbData)
    #非線形data(Pa)
    #       H  I  K   Q   R   T   U   W   Z
    cols = [7, 8, 10, 16, 17, 19, 20, 22, 25]
    c = cm / (cl * cs * cs)
    for col in cols:
        dbData = convertValues(c, col, dbData)
    #非線形data(1/Pa)
    #       N   S   V   Y
    cols = [13, 18, 21, 24]
    c = (cl * cs * cs) / cm
    for col in cols:
        dbData = convertValues(c, col, dbData)
    #時間（s）
    #     AA
    col = 26
    c = cs
    dbData = convertValues(c, col, dbData)
    return dbData
    
#
#  convertValues
def convertValues(coeff, col, dbData):
    """ 指定カラムの値を変換する。"""
    for row in range(len(dbData)):
        val = dbData[row][col]
        if val != "":
            newVal = val * coeff
            dbData[row][col] = newVal
    return dbData

#
#  createNewLinesOfMatDB
def createNewLinesOfMatDB(lines, newDim, newData):
    """ 単位変換したfileを作成する"""
    newLines = []
    #第1行
    words = pyFistr.deleteSp(lines[0]).split(",")
    words[0] = newDim
    line = ",".join(words) + "\n"
    newLines.append(line)
    #第2行
    words = pyFistr.deleteSp(lines[1]).split(",")
    for i in range(1, len(words)):
        words[i] = ""
    uDict = createUnitDict(newDim)
    #youngs
    words[1] = uDict["press"]
    #密度
    words[3] = uDict["rho"]
    #線膨張係数
    words[4] = "1/K"
    #熱伝導率
    words[5] = uDict["thermCond"]
    #比熱
    words[6] = uDict["specHeat"]
    #非線形応力
    words[7] = uDict["press"]
    words[8] = uDict["press"]
    words[16] = uDict["press"]
    words[19] = uDict["press"]
    words[20] = uDict["press"]
    words[21] = uDict["1/press"]
    line = ",".join(words) + "\n"
    newLines.append(line)
    #data行を作成
    for vals in newData:
        name = vals[0]
        #words = list(map(str, vals[1:]))
        words = list(map(pyFistr.float2strAuto, vals[1:]))
        line = name + "," + ",".join(words) + "\n"
        newLines.append(line)
    return newLines

#
#  saveNewMatFile
def saveNewMatFile(newLines, newDim, matFile, currDim):
    """ newMatFileを保存する"""
    matDir = os.path.dirname(matFile)
    name = os.path.basename(matFile)
    words = name.split(".")
    if len(words[-2].split("_")) == 3:
        newName = ".".join(words[:-2]) + "." + newDim + ".csv"
    else:
        newName = ".".join(words[:-1]) + "." + newDim + ".csv"
    newMatFile = matDir + os.sep + newName
    f = open(newMatFile, "w", encoding="utf-8"); f.writelines(newLines); f.close()
    print()
    print("new dimension was converted from '" + currDim + "' to '" + newDim + "'.")
    print("newDimensionFile was created at '" + newName + "'.")
    print()

#
#  convDimensionOtherMatFile
#----------------------------
def convDimensionOtherMatFiles(matDir, dim):
    """ その他のmatFileの単位変換"""
    #MULTILINEARのファイルの変換
    files = getOtherMatFile_multilinear(matDir)
    convDimensionOtherMatFile_multilinear(files, dim)

#
#  convDimensionOtherMatFile_multilinear
def convDimensionOtherMatFile_multilinear(files, newDim):
    """ MULTILINEARのfileをdimension変換"""
    for fileName in files:
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        currDim = pyFistr.deleteSp(lines[0]).split(",")[0]
        if currDim == "":
            currDim = "kg_m_s"
        #基本単位の係数を算出
        cm, cl, cs = getConvCoeff(currDim, newDim)
        #ヤング率の係数を算出
        coeff = cm / (cl * cs * cs)
        row = 2; col = 2
        for n in range(row, len(lines)):
            words = pyFistr.deleteSp(lines[n]).split(",")
            if words[col] != "":
                E = float(words[col]) * coeff
                words[col] = str(E)
                lines[n] = ",".join(words) + "\n"
        #単位を修正
        unitDict = createUnitDict(newDim)
        newUnit = unitDict["press"]
        words = pyFistr.deleteSp(lines[0]).split(",")
        words[0] = newDim
        words[2] = newUnit
        lines[0] = ",".join(words) + "\n"
        #書き込み
        f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  getOtherMatFile_multilinear
def getOtherMatFile_multilinear(matDir):
    """ hardenがmultilinearの全fileを取得"""
    pattern = matDir + os.sep + "*_PlasticSSdata*.csv"
    files = glob.glob(pattern)
    pattern = matDir + os.sep + "*_PlasticDruckerSVdata*.csv"
    files += glob.glob(pattern)
    pattern = matDir + os.sep + "*_PlasticMohrSVdata*.csv"
    files += glob.glob(pattern)
    return files

#
#  showCurrDimension
#--------------------
def showCurrDimension():
    matFile = getMatDBName()
    currDim, lines = getDimension(matFile)
    print()
    print("dimendion: " + currDim)
    print()

#
#  printHelp
#------------
def printHelp():
    """ helpを表示する"""
    cont = """
----------------- convDimensionMatCsv.py ----------------------------
EasyISTRで使用している材料DB（mat.csv）のdimensionを変換する。
材料DBのdimensionは、csvファイル内の最初のcell内容から判断する。
変換後は、ファイル名にdimensionが追加されたcsvファイルを作成し、
そのcsvファイルの最初のcellに変換後のdimensionを追記する。
dimensionを入力しなかった場合は、現在のdimensionを表示する。

<使い方>
convDimensionMatCsv.py [option] <dimension>

option
    -h, --help   :ヘルプを表示

dimension例
    kg_m_s, ton_mm_s, g_cm_s, kg_mm_ms

<使用例>
convDimensionMatCsv.py              現在のdimensionを表示
convDimensionMatCsv.py ton_mm_s     ton,mm,sの単位系に変換する。
"""
    print(cont)

#
#  getOption
#------------
def getOption():
    """ optionを取得する"""
    dim = ""
    if len(sys.argv) == 1:
        return dim
    dim = sys.argv[1]
    return dim


if __name__ == "__main__":
    dim = getOption()
    if dim == "":
        showCurrDimension()
        exit()
    dims = dim.split("_")
    if len(dims) == 3:
        error = checkDimension(dims)
        if error == 1:
            print()
            print("error:newDimension is wrong!!")
            print()
            exit()
        convDimension(dim)
        currDir = os.path.getcwd()
        convDimensionOtherMatFile(currDir, dim)
    else:
        printHelp()
        

