#!/usr/bin/python3
#
#   checkCorrectMesh.py
#
#       mshファイルをチェック修正する。
#       未使用nodeを削除する
#       2重定義のnodeを結合する。
#         2重定義かどうかの判定は、節点間距離が要素サイス/20以下の場合、
#         2重定義と判断する。
#       対象は、要素type=611の場合のみ。
#
#   21/01/28    新規作成
#

import os, sys
import geometryFunc as geo
import pyFistr



#
#  createNodeElementDict
def createNodeElementDict(meshHeaderData):
    """ nodeDict, elementDictを作成する"""
    nodeDict = {}
    elementDict = {}
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            data = headerData[1]
            for node in data:
                nodeNo = node[0]
                loc = node[1:]
                nodeDict[nodeNo] = [loc, []]
        elif words[0] == "!ELEMENT":
            data = headerData[1]
            elmType = pyFistr.getValName(header, "TYPE")
            for elm in data:
                elmNo = elm[0]
                nodes = elm[1:]
                elementDict[elmNo] = [nodes, elmType]
                for nodeNo in nodes:
                    nodeDict[nodeNo][1].append(elmNo)
    return nodeDict, elementDict

#
#  deleteUnuseNodes
def deleteUnuseNodes(delNodes, meshHeaderData):
    """ 未使用nodeを削除する"""
    delNodesSet = set(delNodes)
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            data = meshHeaderData[i][1]
            newData = []
            for node in data:
                if not node[0] in delNodesSet:
                    newData.append(node)
            meshHeaderData[i][1] = newData
        elif words[0] == "!NGROUP":
            data = meshHeaderData[i][1]
            newData = (list(set(data) - set(delNodes)))
            meshHeaderData[i][1] = newData
    return meshHeaderData

#
#  mergePairNodes
def mergePairNodes(mgNodes, meshHeaderData):
    """ mgNodes= [<親node>, <被結合node1>, <被結合node1>,...]
    を結合する。"""
    #不要なnodeのみ取得
    delNodes = []
    for mgNode in mgNodes:
        delNodes += mgNode[1:]
    delNodesSet = set(delNodes)
    #meshデータを修正
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        #!NODEから被結合nodeを削除する
        if words[0] == "!NODE":
            data = meshHeaderData[i][1]
            newData = []
            for node in data:
                if not node[0] in delNodesSet:
                    newData.append(node)
            meshHeaderData[i][1] = newData
        #!ELEMENTのnodeNoを修正
        elif words[0] == "!ELEMENT":
            data = meshHeaderData[i][1]
            newData = []
            for elm in data:
                nodes = elm[1:]
                if len(set(nodes) & delNodesSet) > 0:
                    #修正する
                    flag = 0
                    for mgNode in mgNodes:
                        idx = 0
                        for node in nodes:
                            if node in mgNode[1:]:
                                pairNode = mgNode[0]
                                nodes[idx] = pairNode
                                elm = [elm[0]] + nodes
                                flag = 1
                                break
                            idx += 1
                        if flag == 1:
                            break
                    newData.append(elm)
                else:
                    newData.append(elm)
            meshHeaderData[i][1] = newData
        #!EGROUPのnodeNoを修正
        elif words[0] == "!EGROUP":
            data = meshHeaderData[i][1]
            newData = []
            dataSet = set(data)
            if len(dataSet & delNodesSet) > 0:
                for mgNode in mgNodes:
                    pairNode = mgNode[0]
                    nodes = mgNode[1:]
                    if len(dataSet & set(nodes)) > 0:
                        for node in nodes:
                            if node in dataSet:
                                dataSet.remove(node)
                                dataSet.add(pairNode)
                newData = list(dataSet)
                newData.sort()
                meshHeaderData[i][1] = newData
    return meshHeaderData

#
#  replaceSameLocationNodes
def replaceSameLocationNodes(meshHeaderData, dicts):
    nodeDict, elementDict = dicts
    allNodes = list(nodeDict.keys())
    allNodes.sort()
    for nodeNo in allNodes:
        nodeDict[nodeNo].append([])

    #未使用nodeを取得
    print("deleting unused nodes...")
    delNodes = []
    for nodeNo in allNodes:
        if len(nodeDict[nodeNo][1]) == 0:
            delNodes.append(nodeNo)
    #  削除
    meshHeaderData = deleteUnuseNodes(delNodes, meshHeaderData)
    print("  delete " + str(len(delNodes)) + " nodes.")

    #近接nodeを結合
    print("finding overlap nodes...")
    ip = 0
    for nodeNo in allNodes:
        elmNos = nodeDict[nodeNo][1]
        if len(elmNos) > 0:
            #toleranceを設定（要素サイズの1/20）
            nodes = elementDict[elmNos[0]][0]
            loc1 = nodeDict[nodes[0]][0]
            loc2 = nodeDict[nodes[1]][0]
            tol = geo.length(loc1, loc2) / 20.0
            loc = nodeDict[nodeNo][0]
            #check用のblockを作成
            minX = loc[0] - tol
            minY = loc[1] - tol
            minZ = loc[2] - tol
            maxX = loc[0] + tol
            maxY = loc[1] + tol
            maxZ = loc[2] + tol
            minMax = [[minX, minY, minZ], [maxX, maxY, maxZ]]
            for i in range(ip+1, len(allNodes)):
                checkNode = allNodes[i]
                if len(nodeDict[checkNode][1]) > 0:
                    loc = nodeDict[checkNode][0]
                    if geo.isPointInBlock(minMax, loc) == True:
                        nodeDict[nodeNo][2].append(checkNode)
        ip += 1
    #mergeするnode
    mgNodes = []
    for nodeNo in allNodes:
        if len(nodeDict[nodeNo][2]) != 0:
            mgNode = [nodeNo] + nodeDict[nodeNo][2]
            mgNodes.append(mgNode)
    meshHeaderData = mergePairNodes(mgNodes, meshHeaderData)
    print("  " + str(len(mgNodes)) + " nodes are overlaped.")
    return meshHeaderData

#
#  checkMesh
#-------------
def checkMesh():
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    print("creating nodeDict and elementDict...")
    nodeDict, elementDict = createNodeElementDict(meshHeaderData)
    dicts = (nodeDict, elementDict)
    meshHeaderData = replaceSameLocationNodes(meshHeaderData, dicts)
    lines = pyFistr.createLinesFistrModelMshHeader(meshHeaderData)
    pyFistr.writeFistrModelMsh(lines)
    print("mesh file was corrected and saved.")


if __name__ == "__main__":
    import gettext
    gettext.install("app")
    _ = gettext.gettext

    checkMesh()
