#!/usr/bin/python3
# coding:  utf-8
#
#   beam611To641.py
#
#       beam要素611を641に変換する。
#
#   21/01/28    新規作成
#   22/03/15    国際化
#   23/04/13    "FistrModel.cnt"をhecmw_ctrl.datから取得する様に修正
#   24/07/07    openにencoding="utf-8"を追加
#


import os, sys
import pyFistr

#
#  createNodeElementDict
def createNodeElementDict(meshHeaderData):
    """ nodeDict, elementDictを作成する"""
    nodeDict = {}
    elementDict = {}
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            data = headerData[1]
            for node in data:
                nodeNo = node[0]
                loc = node[1:]
                nodeDict[nodeNo] = [loc, []]
        elif words[0] == "!ELEMENT":
            data = headerData[1]
            elmType = pyFistr.getValName(header, "TYPE")
            for elm in data:
                elmNo = elm[0]
                nodes = elm[1:]
                elementDict[elmNo] = [nodes, elmType]
                for nodeNo in nodes:
                    nodeDict[nodeNo][1].append(elmNo)
    return nodeDict, elementDict

#
#  isOnlyBeam611
#----------------
def isOnlyBeam611(meshHeaderData):
    flag = True
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            elmType = pyFistr.getValName(header, "TYPE")
            if elmType != "611":
                flag = False
                break
    return flag

#
#  addDummyNodes
#----------------
def addDummyNodes(meshHeaderData, dicts):
    """ dummyNodeを追加する"""
    nodeDict, elementDict = dicts
    maxNo = max(list(nodeDict.keys()))
    #dummyDict作成
    dummyNodeDict = {}
    dummyNodeNo = maxNo + 1
    for nodeNo in nodeDict.keys():
        dummyNodeDict[nodeNo] = dummyNodeNo
        dummyNodeNo += 1
    #dummyの!ELEMENTを作成
    addHeaderData = []
    nodesSet = set([])
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        data = meshHeaderData[i][1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            header = pyFistr.setValName(header, "TYPE=641")
            meshHeaderData[i][0] = header
            egrpName = pyFistr.getValName(header, "EGRP")
            #nodesSet = set([])
            for ii in range(len(data)):
                elm = data[ii]
                elmNo = elm[0]
                nodes = elm[1:]
                for nodeNo in nodes:
                    nodesSet.add(nodeNo)
                    loc = nodeDict[nodeNo][0]
                    dummyNodeNo = dummyNodeDict[nodeNo]
                    #nodeData = [dummyNodeNo] + loc
                    nodeDict[dummyNodeNo] = [[loc], [elmNo]]
                    elm += [dummyNodeNo]
                data[ii] = elm
                elementDict[elmNo][0] = elm[1:]
    #!dummyの!NODEを作成
    nodes = list(nodesSet)
    nodes.sort()
    dummyHeader = "!NODE, NGRP=dummyNodes\n"
    dummyData = []
    for nodeNo in nodes:
        dummyNo = dummyNodeDict[nodeNo]
        loc = nodeDict[nodeNo][0]
        dummyData.append([dummyNo] + loc)
    addHeaderData.append([dummyHeader, dummyData])

    #挿入(!ELEMENTの前)
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        data = meshHeaderData[i][1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            si = i
            break
    meshHeaderData = meshHeaderData[:si] + addHeaderData + meshHeaderData[si:]
    return meshHeaderData

#
#  addDummyNgrp
#---------------
def addDummyNgrp(meshHeaderData, dicts):
    """ dummyのNGRPを追加する"""
    nodeDict, elementDict = dicts
    addHeaderData = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            ngrpName = pyFistr.getValName(header, "NGRP")
            addHeader = "!NGROUP, NGRP=dummy_" + ngrpName + "\n"
            addData = []
            for nodeNo in data:
                if len(nodeDict[nodeNo][1]) > 0:
                    elmNos = nodeDict[nodeNo][1]
                    elmNo = elmNos[0]
                    nodes = elementDict[elmNo][0]
                    idx = nodes.index(nodeNo)
                    dummyNodeNo = nodes[idx+2]
                    addData.append(dummyNodeNo)
            addHeaderData.append([addHeader, addData])
    #挿入
    flag = 0
    ip = len(meshHeaderData)
    for i in range(len(meshHeaderData)):
        header = meshHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        if flag == 0:
            if words[0] == "!NGROUP":
                flag = 1
        else:
            if words[0] != "!NGROUP":
                ip = i
                break
    meshHeaderData = meshHeaderData[:ip] + addHeaderData + meshHeaderData[ip:]
    return meshHeaderData

#
#  createDummyCond
#-------------------
def createDummyCond(contHeaderData):
    """ cntFileを641用に修正"""

    def checkRotation(ws):
        if int(ws[1]) >= 4:
            ws[0] = "dummy_" + ws[0]
            ws[1] = str(int(ws[1]) - 3)
            ws[2] = str(int(ws[2]) - 3)
        return ws

    def checkCloadRotation(ws):
        if int(ws[1]) >= 4:
            ws[0] = "dummy_" + ws[0]
            ws[1] = str(int(ws[1]) - 3)
        return ws

    for i in range(len(contHeaderData)):
        header = contHeaderData[i][0]
        dataLines = contHeaderData[i][1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!BOUNDARY":
            for ii in range(len(dataLines)):
                line = dataLines[ii]
                ws = pyFistr.deleteSp(line).split(",")
                ws = checkRotation(ws)
                line = ", ".join(ws) + "\n"
                dataLines[ii] = line
            contHeaderData[i] = [header, dataLines]
        elif words[0] == "!CLOAD" or words[0] == "!SPRING":
            for ii in range(len(dataLines)):
                line = dataLines[ii]
                ws = pyFistr.deleteSp(line).split(",")
                ws = checkCloadRotation(ws)
                line = ", ".join(ws) + "\n"
                dataLines[ii] = line
            contHeaderData[i] = [header, dataLines]
    return contHeaderData

#
#  readFistrModelCnt
#---------------------
def readFistrModelCnt():
    currDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(currDir)
    fileName = currDir + os.sep + FistrModel_cnt
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    return lines 

#
#  writeFistrModelCnt
#---------------------
def writeFistrModelCnt(lines):
    currDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(currDir)
    fileName = currDir + os.sep + FistrModel_cnt
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  writeHeaderDataToCnt
#------------------------
def writeHeaderDataToCnt(headerData):
    lines = []
    for header in headerData:
        lines += [header[0]]
        lines += header[1]
    writeFistrModelCnt(lines)

#
#  convertBeamType
#------------------
def convertBeamType():
    """ beam611を641に変換する"""
    #mshFileを修正
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    nodeDict, elementDict = createNodeElementDict(meshHeaderData)
    dicts = [nodeDict, elementDict]
    if isOnlyBeam611(meshHeaderData) == False:
        print(_("611以外の要素が存在するため、641に変換できません。"))
        print(_("変換できるのは、611のみで構成されたモデルです。"))
        return
    meshHeaderData = addDummyNodes(meshHeaderData, dicts)
    meshHeaderData = addDummyNgrp(meshHeaderData, dicts)
    lines = pyFistr.createLinesFistrModelMshHeader(meshHeaderData)
    pyFistr.writeFistrModelMsh(lines)
    print(_("要素Type:611を641に変更しました。"))
    #cntFileを修正
    lines = readFistrModelCnt()
    contHeaderData = pyFistr.getHeaderData(lines)
    contHeaderData = createDummyCond(contHeaderData)
    writeHeaderDataToCnt(contHeaderData)

#
#  convert
#----------
def convert(meshHeaderData):
    """ meshHeaderDataを641に変換して、返す。
    モジュール専用"""
    nodeDict, elementDict = createNodeElementDict(meshHeaderData)
    dicts = [nodeDict, elementDict]
    meshHeaderData = addDummyNodes(meshHeaderData, dicts)
    meshHeaderData = addDummyNgrp(meshHeaderData, dicts)
    return meshHeaderData

if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    #_ = gettext.gettext

    convertBeamType()