#!/usr/bin/python3
#  coding: utf-8
#
#           remakeCntMeshDataForEasyIstr.py
#
#       cntData,meshDataがEasyISTR用に合致しているかチェック、修正する。
#         cnt:境界条件の整合性
#         msh:EGRP,NGRPのGENERATEを展開する。
#
#   24/03/18    新規作成
#      06/06    universalDialogsに変更
#      07/07    openにencoding="utf-8"を追加
#      09/14    remakeMsh:同じsgrp,ngrp名の複数定義をまとめる事を追加
#      09/25    remakeCnt:vtu出力が可能になるように修正する。
#      12/18    remakeCnt,remakeMsh:bakファイルある場合、削除する。
#   25/01/15    checkDummyGrp:新規追加。dummy節点が存在する時、
#               dummyGrpが独立して定義される不具合発生のため修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk
import os
import time
import pyFistr
import universalDialogs

ls = "\n"

#
#  warningDialog
#-----------------
def warningDialog(title, message):
    dialog = universalDialogs.warningDialog(title, message)
    dialog.show()

#
#  isNumber
def isNumber(name):
    """ 最初の文字が数字かどうかチェックする"""
    if "0" <= name[0] and name[0] <= "9":
        return True
    return False

#
#  run
#--------
def run(currDir, FistrModel_msh, lines, meshHeaderData, FistrModel_cnt):
    """ cnt,meshDataがEasyISTR用になっているかどうかチェック修正する。"""
    flagCnt = remakeCnt(currDir, FistrModel_cnt)
    flagMsh, lines, meshHeaderData = remakeMsh(currDir, FistrModel_msh, lines, meshHeaderData)
    if flagCnt != 0 or flagMsh != 0:
        title = _("cnt,mshファイルのチェック")
        mess = _("修正して以下のbackupFileを作成しました。") + ls
        if flagCnt != 0:
            mess += "cnt: " + FistrModel_cnt + ".bak" + ls
        if flagMsh != 0:
            mess += "msh: " + FistrModel_msh + ".bak" + ls
        #warningDialog(title, mess)
        print(ls + "<" + title + ">")
        print(mess)
    return lines, meshHeaderData

#
#  remakeMsh
#-------------
def remakeMsh(currDir, FistrModel_msh, lines, meshHeaderData):
    """ meshDataがEasyISTR用になっているかどうかチェック修正する。
    GENERATEを解釈して展開する。同じgrp名の複数定義があればまとめる。"""
    #GENERATEを修正
    flag = 0
    for i in range(len(meshHeaderData)):
        header, data = meshHeaderData[i]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!EGROUP" or words[0] == "!NGROUP":
            if "GENERATE" in words:
                #GENERATEを展開する
                flag = 1
                data = createDataFromGenerate(data)
                words.remove("GENERATE")
                newHeader = ", ".join(words) + ls
                meshHeaderData[i] = [newHeader, data]
    #同じsgrp,ngrp名の複数定義があれば、まとめる。
    ngrpDict = {}
    sgrpDict = {}
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            ngrp = pyFistr.getValName(header, "NGRP")
            if not ngrp in ngrpDict.keys():
                ngrpDict[ngrp] = data
            else:
                ngrpDict[ngrp] += data
                flag = 2
        elif words[0] == "!SGROUP":
            sgrp = pyFistr.getValName(header, "SGRP")
            if not sgrp in sgrpDict.keys():
                sgrpDict[sgrp] = data
            else:
                sgrpDict[sgrp] += data
                flag = 2
    if flag == 2:
        newHeaderData = []
        for header, data in meshHeaderData:
            words = pyFistr.deleteSp(header).split(",")
            if words[0] != "!NGROUP" and words[0] != "!SGROUP" and words[0] != "!END":
                newHeaderData.append([header, data])
        for sgrp in sgrpDict.keys():
            header = "!SGROUP, SGRP=" + sgrp + ls
            data = sgrpDict[sgrp]
            newHeaderData.append([header, data])
        for ngrp in ngrpDict.keys():
            header = "!NGROUP, NGRP=" + ngrp + ls
            data = ngrpDict[ngrp]
            newHeaderData.append([header, data])
        newHeaderData.append(["!END", []])
        meshHeaderData = newHeaderData
    #修正ファイルを保存
    if flag != 0:
        #ファイル作成、書き込み
        sourceFile = currDir + os.sep + FistrModel_msh
        newFile = sourceFile + ".bak"
        #同名のnewFileがあれば削除
        if os.path.exists(newFile) == True:
            os.remove(newFile)
        os.rename(sourceFile, newFile)
        lines = pyFistr.createLinesFistrModelMshHeader(meshHeaderData)
        f = open(sourceFile, "w", encoding="utf-8"); f.writelines(lines); f.close()
    return flag, lines, meshHeaderData

#
#  createDataFromGenerate
def createDataFromGenerate(data):
    """ GENERATEからdataを作成する"""
    newData = []
    for i in range(0, len(data), 3):
        st, ed, inc = data[i], data[i+1], data[i+2]
        nData = [ii for ii in range(st, ed+1, inc)]
        newData += nData
    return newData

#
#  remakeCnt
#------------
def remakeCnt(currDir, FistrModel_cnt):
    """ cntFileがEasyISTR用になっているかチェック修正する"""
    boundaryEtc = ["!BOUNDARY", "!VELOCITY", "!ACCELERATION"]
    #cloadEtc = ["!CLOAD", "!FLOAD"]
    boundaryAll = ["!BOUNDARY", "!CLOAD", "!DLOAD", "!SPRING", 
                   "!FLOAD", "!VELOCITY", "!ACCELERATION", 
                   "!FIXTEMP", "!CFLUX", "!SFLUX", "!DFLUX", 
                   "!SFILM", "!SRADIATE"]
    cntFile = currDir + os.sep + FistrModel_cnt
    f = open(cntFile, encoding="utf-8"); lines = f.readlines(); f.close()
    cntHeaderData = pyFistr.getHeaderData(lines)
    #keyword内に複数のgrp名が存在する場合、分割する
    newHeaderData = []
    flag = 0
    for i in range(len(cntHeaderData)):
        header, data = cntHeaderData[i]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] in boundaryAll:
            flag, addHeaderData = separateHeaderIfMulti(header, data)
        else:
            addHeaderData = [[header, data]]
        newHeaderData += addHeaderData
    #keyword内のgrp名の数、順番を修正
    cntHeaderData = newHeaderData
    nStep = 0
    #BOUNDARY,VELOCITY,ACCELERATION----
    newHeaderData = []
    for header, data in cntHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] in boundaryEtc:
            newData = []
            for line in data:
                ws = pyFistr.deleteSp(line).split(",")
                st, ed = int(ws[1]), int(ws[2])
                if st == ed:
                    newData += [line]
                else:
                    for i in range(st, ed+1, 1):
                        newLine = ", ".join([ws[0], str(i), str(i), ws[3]]) + ls
                        newData += [newLine]
                    flag = 1
        #CLOAD,FLOAD ---------
        elif words[0] in ["!CLOAD", "!FLOAD"]:
            wsList = list(map(lambda x: pyFistr.deleteSp(x).split(","), data))
            numList = list(map(lambda x: int(x[1]), wsList))
            nMax = max(numList)
            grp = wsList[0][0]
            if nMax > 3:
                #モーメント含む
                dataList = [[grp, str(i), "0.0"] for i in range(1, 7, 1)]
                if len(data) != 6:
                    flag = 1
            else:
                #荷重のみ
                dataList = [[grp, str(i), "0.0"] for i in range(1, 4, 1)]
                if len(data) != 3:
                    flag = 1
            for i in range(len(wsList)):
                n = numList[i]
                dataList[n-1][-1] = wsList[i][-1]
            newData = list(map(lambda x: ", ".join(x) + ls, dataList))
        #DLOAD>BXYZ --------------
        elif words[0] == "!DLOAD":
            wsList = list(map(lambda x: pyFistr.deleteSp(x).split(","), data))
            bdir = wsList[0][1]
            if bdir == "BX" or bdir == "BY" or bdir == "BZ":
                if len(wsList) < 3:
                    grp = wsList[0][0]
                    xyz = []
                    BList = list(map(lambda x: x[1], wsList))
                    if not "BX" in BList:
                        xyz += [[grp,"BX", "0.0"]]
                    if not "BY" in BList:
                        xyz += [[grp,"BY", "0.0"]]
                    if not "BZ" in BList:
                        xyz += [[grp,"BZ", "0.0"]]
                    wsList += xyz
                    wsList = sorted(wsList, key=lambda x: x[1])
                    newData = map(lambda x: ", ".join(x) + ls, wsList)
                    flag = 1
                else:
                    newData = data
            else:
                newData = data
        #STEP ----
        elif words[0] == "!STEP":
            stepName = pyFistr.getValName(header, "stepName")
            if stepName == "":
                header = pyFistr.setValName(header, "stepName=STEP"+str(nStep))
                nStep += 1
                flag = 1
            newData = data
        else:
            newData = data
        newHeaderData += [[header, newData]]
    #出力方法,出力頻度のチェック(以下をチェック)
    #  !WRITE, VISUAL
    si = -1
    addLine = "!WRITE, VISUAL, FREQUENCY=1\n"
    for i in range(len(newHeaderData)):
        header = newHeaderData[i][0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!WRITE":
            if words[1] == "VISUAL":
                break
            else:
                freq = pyFistr.getValName(header, "FREQUENCY")
                if freq != "":
                    addLine = pyFistr.setValName(addLine, "FREQUENCY="+freq)
                    si = i
                    break
        elif words[0] == "!SOLUTION":
            si = i
            break
    if si > 0:
        newHeaderData = newHeaderData[:si] + [[addLine, []]] + newHeaderData[si:]
        flag = 1
    #出力方法のチェック(以下をチェック)
    #  !VISUAL, method=PSR
    #  !surface_num=1
    #  !surface 1
    #  !output_type=VTK
    addHeaderData = []
    addFlags = [0,0,0,0]
    for i in range(len(newHeaderData)):
        header = newHeaderData[i][0]
        words = header.split()
        delHeader = pyFistr.deleteSp(header)
        if delHeader[:len("!VISUAL,")].upper() == "!VISUAL,":
            addFlags[0] = 1
            method = pyFistr.getValName(header, "method")
            if method == "PSR":
                pass
            else:
                newHeaderData[i][0] = "!VISUAL, method=PSR\n"
                flag = 1
        elif delHeader[:len("!surface_num=")].upper() == "!SURFACE_NUM=":
            addFlags[1] = 1
            words = pyFistr.deleteSp(header).split("=")
            if words[1] == "1":
                pass
            else:
                newHeaderData[i][0] = "!surface_num=1\n"
                flag = 1
        elif words[0].upper() == "!SURFACE":
            addFlags[2] = 1
            if len(words) == 2:
                if words[1] == "1":
                    pass
                else:
                    newHeaderData[i][0] = "!surface 1\n"
                    flag = 1
            else:
                newHeaderData[i][0] = "!surface 1\n"
                flag = 1
        elif delHeader[:len("!output_type=")].upper() == "!OUTPUT_TYPE=":
            addFlags[3] = 1
            words = pyFistr.deleteSp(header).split("=")
            if words[1] == "VTK":
                pass
            else:
                newHeaderData[i][0] = "!output_type=VTK\n"
                flag = 1
    #addHeaderを確認
    if addFlags[0] == 0:
        addHeaderData.append(["!VISUAL, method=PSR\n", []])
    if addFlags[1] == 0:
        addHeaderData.append(["!surface_num=1\n", []])
    if addFlags[2] == 0:
        addHeaderData.append(["!surface 1\n", []])
    if addFlags[3] == 0:
        addHeaderData.append(["!output_type=VTK\n", []])
    #addheaderを追加する
    if len(addHeaderData) > 0:
        flag = 1
        for i in range(len(newHeaderData)):
            header = newHeaderData[i][0]
            words = pyFistr.deleteSp(header).split(",")
            if words[0] == "!END":
                newHeaderData = newHeaderData[:i] + addHeaderData + newHeaderData[i:]
                break
    #dummyGrpの設定をチェック
    flag, newHeaderData = checkDummyGrp(flag, newHeaderData)
    #修正有無を確認
    if flag == 1:
        #newHeaderDataを書き込み
        lines = []
        for header, data in newHeaderData:
            lines += [header]
            lines += data
        sourceName = currDir + os.sep + FistrModel_cnt
        backName = sourceName + ".bak"
        #同名のbackNameがあれば削除
        if os.path.exists(backName) == True:
            os.remove(backName)
        os.rename(sourceName, backName)
        f = open(sourceName, "w", encoding="utf-8"); f.writelines(lines); f.close()
    return flag

#
#  separateHeaderIfMulti
def separateHeaderIfMulti(header, data):
    """ 1ヶのkeyword内に複数のgrpNameが含まれている場合、分割する
    （dummyは除く）"""
    addHeaderData = []
    bfGrp = ""
    spData = []
    flag = 0
    for line in data:
        words = pyFistr.deleteSp(line).split(",")
        currGrp = words[0]
        if isNumber(currGrp) == False:
            if currGrp != bfGrp:
                if len(spData) > 0:
                    addHeaderData += [[header, spData]]
                if bfGrp != "":
                    flag = 1
                bfGrp = currGrp
                spData = [line]
            else:
                spData += [line]
    if len(spData) > 0:
        addHeaderData += [[header, spData]]
    return flag, addHeaderData

#
#  checkDummyGrp
def checkDummyGrp(flag, cntHeaderData):
    """ dummyGrpの設定がある場合は、headerが追加されているので、headerを削除し、
    直前のheader内に加える.
    修正した場合は、flag=1を返す"""
    newHeaderData = []
    for i in range(len(cntHeaderData)):
        header, data = cntHeaderData[i]
        if len(data) > 0:
            line = data[0]
            words = pyFistr.deleteSp(line).split(",")
            if words[0][:len("dummy_")] == "dummy_":
                newHeaderData[-1][1] += data
                flag = 1
            else:
                newHeaderData += [[header, data]]
        else:
            newHeaderData += [[header, data]]
    return flag, newHeaderData


if __name__ == "__main__":
    import gettext
    gettext.install("app") # replace with the appropriate catalog name
    _ = gettext.gettext

    fileName = "cylinder.msh"
    cntFileName = "cylinder.cnt"
    currDir = os.getcwd()
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = run(currDir, fileName, meshHeaderData, cntFileName)
    

