#!/usr/bin/python3
#  coding: utf-8
#
#           meshConverterForMeshType.py
#
#       solid要素のmeshTypeの変換する
#       ・1次要素を2次要素に変換
#       ・2次要素を1次要素に変換
#       ・tetra1次要素をhexa1次要素に変換
#
#   25/05/04    新規作成
#


import os
import pyFistr
import convertMesh as cm
import geometryFunc as geo


NodeDict = {}
ElementDict = {}
NgrpDict = {}
SgrpDict = {}
EgrpDict = {}
EdgeDict = {}

#edgeの節点を取得する辞書
edgeNodeOfElementDict = {
    "341":[[1,2],[2,3],[1,3],[1,4],[2,4],[3,4]],    #四面体1次
    "342":[[1,7,2],[2,5,3],[1,6,3],[1,8,4],[2,9,4],[3,10,4]],   #四面体2次
    "361":[[1,2],[2,3],[3,4],[1,4],[5,6],[6,7],[7,8],[5,8],[1,5],[2,6],[3,7],[4,8]],    #六面体1次
    "362":[[1,9,2],[2,10,3],[3,11,4],[1,12,4],[5,13,6],[6,14,7],[7,15,8],[5,16,8],[1,17,5],[2,18,6],[3,19,7],[4,20,8]],   #六面体2次
    "351":[[1,2],[2,3],[1,3],[4,5],[5,6],[4,6],[1,4],[2,5],[3,6]],   #五面体1次
    "352":[[1,9,2],[2,7,3],[1,8,3],[4,12,5],[5,10,6],[4,11,6],[1,13,4],[2,14,5],[3,15,6]],   #五面体2次
    "731":[[1,2],[2,3],[1,3]],  #三角形1次
    "732":[[1,6,2],[2,4,3],[1,5,3]],    #三角形2次
    "741":[[1,2],[2,3],[3,4],[1,4]],    #四角形1次
    "781":[[1,2],[2,3],[3,4],[1,4]],    #四角形dummy付き
    "761":[[1,2],[2,3],[1,3]],  #三角形dummy付き
    "611":[[1,2]],      #beam1次
    "641":[[1,2]]       #beam dummy付き
    }

#
#  getNodeElementDict
#---------------------
def getNodeElementDict(meshHeaderData, targetTypes):
    """ nodeDictを取得して返す"""
    nodeDict = {}; elementDict = {}
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            for vals in data:
                nodeNo = vals[0]
                loc = vals[1:4]
                nodeDict[nodeNo] = loc
        elif words[0] == "!ELEMENT":
            elType = pyFistr.getValName(header, "TYPE")
            if (elType in targetTypes) == True:
                egrpName = pyFistr.getValName(header, "EGRP")
                for vals in data:
                    elNo = vals[0]
                    nodes = vals[1:]
                    elData = [nodes, elType, egrpName]
                    elementDict[elNo] = elData
    return nodeDict, elementDict

#
#  getNodeSurfaceElementGroup
#-----------------------------
def getNodeSurfaceElementGroup(meshHeaderData):
    """ node, surface, elementのgroupを取得
    !NGROUP, !SGROUP, !EGROUPから取得する"""
    ngrpDict = {}; sgrpDict = {}; egrpDict = {}
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            ngrp = pyFistr.getValName(header, "NGRP")
            if ngrp in ngrpDict.keys():
                ngrpDict[ngrp] += data
            else:
                ngrpDict[ngrp] = data
        elif words[0] == "!SGROUP":
            sgrp = pyFistr.getValName(header, "SGRP")
            if sgrp in sgrpDict.keys():
                sgrpDict[sgrp] += data
            else:
                sgrpDict[sgrp] = data
        elif words[0] == "!EGROUP":
            egrp = pyFistr.getValName(header, "egrp")
            if egrp in egrpDict.keys():
                egrpDict[egrp] += data
            else:
                egrpDict[egrp] = data
    return ngrpDict, sgrpDict, egrpDict

#
#  addMiddleNodes
#----------------
def addMiddleNodes():
    """ 中間節点を追加"""
    global NodeDict, EdgeDict, ElementDict
    #対象egrpのedgeDictを取得
    EdgeDict = {}
    for elNo in ElementDict.keys():
        elData = ElementDict[elNo]
        nodes = elData[0]
        elType = elData[1]
        edges = edgeNodeOfElementDict[elType]
        for edge in edges:
            node0 = nodes[edge[0] - 1]
            node1 = nodes[edge[-1] - 1]
            newEdge = [node0, node1]
            newEdge.sort()
            newEdge = tuple(newEdge)
            if not newEdge in EdgeDict.keys():
                EdgeDict[newEdge] = -1
    #newNodeを取得
    maxNodeNo = max(list(NodeDict.keys()))
    newNodeNo = maxNodeNo + 1
    allEdges = list(EdgeDict.keys())
    allEdges.sort()
    for edge in allEdges:
        EdgeDict[edge] = newNodeNo
        node0, node1 = edge
        loc0 = NodeDict[node0]
        loc1 = NodeDict[node1]
        midLoc = geo.midPoint(loc0, loc1)
        NodeDict[newNodeNo] = midLoc
        newNodeNo += 1
    #要素にnodeを追加
    for elNo in ElementDict.keys():
        elData = ElementDict[elNo]
        nodes = elData[0]
        elType = elData[1]
        if elType == "341":
            newElType = "342"
        elif elType == "361":
            newElType = "362"
        else:
            newElType = "352"
        elData[1] = newElType
        edges = edgeNodeOfElementDict[newElType]
        midDict = {}
        for edge in edges:
            node0 = nodes[edge[0] - 1]
            node1 = nodes[edge[-1] - 1]
            midKey = edge[1]
            newEdge = [node0, node1]
            newEdge.sort()
            newEdge = tuple(newEdge)
            midNodeNo = EdgeDict[newEdge]
            midDict[midKey] = midNodeNo
        midNodes = list(midDict.keys())
        midNodes.sort()
        for midNode in midNodes:
           nodes.append(midDict[midNode])
        elData[0] = nodes
        ElementDict[elNo] = elData
    #ngrpを修正
    for edge in EdgeDict.keys():
        node0 = edge[0]
        node1 = edge[1]
        for ngrp in NgrpDict.keys():
            ngrpSets = set(NgrpDict[ngrp])
            if node0 in ngrpSets:
                if node1 in ngrpSets:
                    midNode = EdgeDict[edge]
                    NgrpDict[ngrp].append(midNode)

#
#  getElement
def getElement(elmType, egrp):
    """ elementDict内からelmTypeのみを取得する"""
    global ElementDict
    data = []
    for elmNo in ElementDict.keys():
        elData = ElementDict[elmNo]
        if len(elData) == 3:
            nodes = elData[0]
            elType = elData[1]
            elName = elData[2]
            if elType == elmType:
                if egrp == "" or egrp == elName:
                    lineData = [elmNo] + nodes
                    data.append(lineData)
                    ElementDict[elmNo].append(1)    #flag書込
    return data

#
#  createMeshHeaderDataFor2nd
#-----------------------
def createMeshHeaderDataFor2nd(meshHeaderData):
    """ headerDataを作り直す"""
    newHeaderData = []
    for i in range(len(meshHeaderData)):
        header, data = meshHeaderData[i]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            newData = []
            for nodeNo in NodeDict.keys():
                nodeLoc = [nodeNo] + NodeDict[nodeNo]
                newData.append(nodeLoc)
            #meshHeaderData[i][1] = newData
            newHeaderData.append([header, newData])
        elif words[0] == "!ELEMENT":
            elType = pyFistr.getValName(header, "TYPE")
            egrp = pyFistr.getValName(header, "EGRP")
            if elType == "341":
                newType = "342"
            elif elType == "361":
                newType = "362"
            else:
                newType = "352"
            newData = getElement(newType, egrp)
            if len(newData) > 0:
                newHeader = pyFistr.setValName(header, "TYPE="+newType)
                newHeaderData.append([newHeader, newData])
        else:
            newHeaderData.append([header, data])
    return newHeaderData

#
#  deleteMiddleNodes
#--------------------
def deleteMiddleNodes():
    """ 中間節点を削除"""
    global NodeDict, EdgeDict, ElementDict, NgrpDict
    delNodes = []
    for elNo in ElementDict.keys():
        elData = ElementDict[elNo]
        nodes = elData[0]
        elType = elData[1]
        edges = edgeNodeOfElementDict[elType]
        for edge in edges:
            node0 = nodes[edge[0] - 1]
            node1 = nodes[edge[-1] - 1]
            midNode = nodes[edge[1] - 1]
            delNodes.append(midNode)
    #要素から中間節点を削除
    for elNo in ElementDict.keys():
        elData = ElementDict[elNo]
        nodes = elData[0]
        elType = elData[1]
        if elType == "342":
            newElType = "341"
        elif elType == "362":
            newElType = "361"
        else:
            newElType = "351"
        elData[1] = newElType
        nNodes = cm.elmContsDict[newElType][3]
        newNodes = nodes[:nNodes]
        elData[0] = newNodes
        ElementDict[elNo] = elData
    #nodeDictからnodeを削除
    delNodeSets = set(delNodes)
    nodeDict = {}
    for nodeNo in NodeDict.keys():
        if (nodeNo in delNodeSets) == False:
            nodeDict[nodeNo] = NodeDict[nodeNo]
    NodeDict = nodeDict
    #ngrpを修正
    for ngrp in NgrpDict.keys():
        nodes = NgrpDict[ngrp]
        newNodes = []
        for nodeNo in nodes:
            if not(nodeNo in delNodeSets):
                newNodes.append(nodeNo)
        NgrpDict[ngrp] = newNodes

#
#  createMeshHeaderDataFor1st
#-----------------------
def createMeshHeaderDataFor1st(meshHeaderData):
    """ headerDataを作り直す"""
    global NgrpDict
    newHeaderData = []
    for i in range(len(meshHeaderData)):
        header, data = meshHeaderData[i]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            newData = []
            for nodeNo in NodeDict.keys():
                nodeLoc = [nodeNo] + NodeDict[nodeNo]
                newData.append(nodeLoc)
            newHeaderData.append([header, newData])
        elif words[0] == "!ELEMENT":
            elType = pyFistr.getValName(header, "TYPE")
            egrp = pyFistr.getValName(header, "EGRP")
            if elType == "342":
                newType = "341"
            elif elType == "362":
                newType = "361"
            else:
                newType = "351"
            newData = getElement(newType, egrp)
            newHeader = pyFistr.setValName(header, "TYPE="+newType)
            newHeaderData.append([newHeader, newData])
        elif words[0] == "!NGROUP":
            ngrp = pyFistr.getValName(header, "NGRP")
            newData = NgrpDict[ngrp]
            newHeaderData.append([header, newData])
        else:
            newHeaderData.append([header, data])
    return newHeaderData

#
#  create2ndOrderMesh
#-----------------------
def create2ndOrderMesh(meshHeaderData):
    """ tetraの2次要素を作成する"""
    global NodeDict, ElementDict, NgrpDict, SgrpDict, EgrpDict
    #node, elementDictを取得
    targetTypes = ["341"]
    NodeDict, ElementDict = getNodeElementDict(meshHeaderData, targetTypes)
    NgrpDict, SgrpDict, EgrpDict = getNodeSurfaceElementGroup(meshHeaderData)
    #2次要素に変換(NodeDict, ElementDictを書き換える)
    addMiddleNodes()
    meshHeaderData = createMeshHeaderDataFor2nd(meshHeaderData)
    return meshHeaderData

#
#  getfaceCenterLoc
#------------------
def getFaceCenterLoc():
    global NodeDict, ElementDict
    """ faceの中心座標を辞書形式で取得する"""

    def getTriCelterLoc(nodes):
        loc0 = NodeDict[nodes[0]]
        loc1 = NodeDict[nodes[1]]
        loc2 = NodeDict[nodes[2]]
        locG = geo.triG([loc0, loc1, loc2])
        return locG
    
    def getTetraCenterLoc(nodes):
        loc0 = NodeDict[nodes[0]]
        loc1 = NodeDict[nodes[1]]
        loc2 = NodeDict[nodes[2]]
        loc3 = NodeDict[nodes[3]]
        locVG = geo.tetG([loc0, loc1, loc2, loc3])
        return locVG       

    maxNodeNo = max(list(NodeDict.keys()))
    nodeNo = maxNodeNo + 1
    faceLocDict = {}
    faceNodeDict = {} 
    for elmNo in ElementDict.keys():
        elData = ElementDict[elmNo]
        aNodes = elData[0]
        elType = elData[1]
        #face中心座標
        faceLocs = []
        for faceNo in range(1, 5, 1):
            faceNodes = cm.fistr_elFaceNode(elType, aNodes, faceNo)
            nodes = (faceNodes[0], faceNodes[2], faceNodes[4])
            words = list(map(str, nodes))
            nodes = list(map(int, words))
            nodes.sort()
            faceKey = tuple(nodes)
            if not faceKey in faceLocDict.keys():
                nodes = (faceNodes[1], faceNodes[3], faceNodes[5])
                loc = getTriCelterLoc(nodes)
                NodeDict[nodeNo] = loc
                faceLocDict[faceKey] = [nodeNo, loc]
                faceLocs.append(nodeNo)
                nodeNo += 1
            else:
                nNo, loc = faceLocDict[faceKey]
                faceLocs.append(nNo)
        #elm中心座標
        loc = getTetraCenterLoc(aNodes[:4])
        NodeDict[nodeNo] = loc
        faceLocs.append(nodeNo)
        nodeNo += 1
        faceNodeDict[elmNo] = faceLocs
    return faceNodeDict


#
#  createHex
#------------
def createHex():
    """ tetra2次要素からhexを作成する"""
    global NodeDict, ElementDict
    #faceの辞書を作成
    faceNodeDict = getFaceCenterLoc()
    #六面体作成
    newElmDict = {}
    nodeNos = NodeDict.values()
    maxNodeNo = len(nodeNos)
    nodeNo = maxNodeNo + 1
    hexNo = 1
    for elmNo in ElementDict.keys():
        elData = ElementDict[elmNo]
        nodes = elData[0]
        elType = elData[1]
        egrpName = elData[2]
        cNodes = faceNodeDict[elmNo]
        #六面体要素4個を作成
        #面1-7-2-5, tetG-面2-9-面3
        hexNodes = [cNodes[0], nodes[6], nodes[1], nodes[4],       #面1-7-2-5
                    cNodes[4], cNodes[1], nodes[8], cNodes[2]]   #tetG, 面2, 9, 面3
        newElmDict[hexNo] = [hexNodes, "361", egrpName]
        hexNo += 1
        #面1-5-3-6 ,tetG-面3-10-面4
        hexNodes = [cNodes[0], nodes[4], nodes[2], nodes[5],
                    cNodes[4], cNodes[2], nodes[9], cNodes[3]]
        newElmDict[hexNo] = [hexNodes, "361", egrpName]
        hexNo += 1
        #面1-6-1-7, tetG-面4-8-面2
        hexNodes = [cNodes[0], nodes[5], nodes[0], nodes[6],
                    cNodes[4], cNodes[3], nodes[7], cNodes[1]]
        newElmDict[hexNo] = [hexNodes, "361", egrpName]
        hexNo += 1
        #tetG-面4-8-面2, 面3-10-4-9
        hexNodes = [cNodes[4], cNodes[3], nodes[7], cNodes[1],
                    cNodes[2], nodes[9], nodes[3], nodes[8]]
        newElmDict[hexNo] = [hexNodes, "361", egrpName]
        hexNo += 1
    ElementDict = newElmDict

#
#  createMeshHeaderDataForHex
#-----------------------------
def createMeshHeaderDataForHex(meshHeaderData):
    """ headerDataを作り直す"""
    newHeaderData = []
    for i in range(len(meshHeaderData)):
        header, data = meshHeaderData[i]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            newData = []
            for nodeNo in NodeDict.keys():
                nodeLoc = [nodeNo] + NodeDict[nodeNo]
                newData.append(nodeLoc)
            newHeaderData.append([header, newData])
        elif words[0] == "!ELEMENT":
            elType = pyFistr.getValName(header, "TYPE")
            egrp = pyFistr.getValName(header, "EGRP")
            if elType == "342":
                newType = "361"
                newData = getElement(newType, egrp)
                if len(newData) > 0:
                    newHeader = pyFistr.setValName(header, "TYPE="+newType)
                    newHeader = pyFistr.setValName(newHeader, "EGRP=E"+newType)
                    newHeaderData.append([newHeader, newData])
        elif words[0] == "!HEADER":
            newHeaderData.append([header, data])
    newHeaderData.append(["!END", []])
    return newHeaderData

#--------------------------
#  convert1stTo2ndOrder
#--------------------------
def convert1stTo2ndOrder(meshHeaderData):
    """ solid要素の1次要素を2次要素に変換する。"""
    global NodeDict, ElementDict, NgrpDict, SgrpDict, EgrpDict
    #node, elementDictを取得
    targetTypes = ["341", "361", "351"]
    NodeDict, ElementDict = getNodeElementDict(meshHeaderData, targetTypes)
    NgrpDict, SgrpDict, EgrpDict = getNodeSurfaceElementGroup(meshHeaderData)
    #2次要素に変換(NodeDict, ElementDictを書き換える)
    addMiddleNodes()
    meshHeaderData = createMeshHeaderDataFor2nd(meshHeaderData)
    return meshHeaderData

#-----------------------
#  convert2ndTo1stOrder
#-----------------------
def convert2ndTo1stOrder(meshHeaderData):
    """ solid要素の2次要素を1次要素に変換する"""
    global NodeDict, ElementDict, NgrpDict, SgrpDict, EgrpDict
    #node, elementDictを取得
    targetTypes = ["342", "362", "352"]
    NodeDict, ElementDict = getNodeElementDict(meshHeaderData, targetTypes)
    NgrpDict, SgrpDict, EgrpDict = getNodeSurfaceElementGroup(meshHeaderData)
    #1次要素に変換（nodeDict, ElementDictを書き換える）
    deleteMiddleNodes()
    meshHeaderData = createMeshHeaderDataFor1st(meshHeaderData)
    return meshHeaderData

#-------------------
#  convertTetToHex
#-------------------
def convertTetToHex(meshHeaderData):
    """ solid要素のtetraをhexaに変換する。"""
    global NodeDict, ElementDict, NgrpDict, SgrpDict, EgrpDict
    #node, elementDictを取得
    targetTypes = ["341"]
    NodeDict, ElementDict = getNodeElementDict(meshHeaderData, targetTypes)
    NgrpDict, SgrpDict, EgrpDict = getNodeSurfaceElementGroup(meshHeaderData)
    #tetraの2次要素を作成する
    meshHeaderData = create2ndOrderMesh(meshHeaderData)
    #Hexaに変換（nodeDict, ElementDictを書き換える）
    createHex()
    meshHeaderData = createMeshHeaderDataForHex(meshHeaderData)
    return meshHeaderData


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))

    fileName = "FistrModel.msh"
    f = open(fileName); lines = f.readlines(); f.close()
    headerNumData = pyFistr.getHeaderNumData(lines)
    
    #meshHeaderData = convert1stTo2ndOrder(headerNumData)
    #meshHeaderData = convert2ndTo1stOrder(headerNumData)
    meshHeaderData = convertTetToHex(headerNumData)

    lines = pyFistr.createLinesFistrModelMshHeader(meshHeaderData)
    fileName = "FistrModel_test.msh"
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()
