#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#   getNodeForceAtArea.py
#
#       nodeForceが等分布荷重となるように設定する。
#       nodeGroupは、face、edge、pointsに対して適用できる。
#       （nodeForceの合計＝1.0で取得する）
#       要素の種類は、以下の種類に対応。
#           四面体（tetra）  1次2次
#           六面体（hexa）   1次2次
#           五面体（prism）  1次2次
#           シェル三角形     1次（731、761）
#           シェル四角形     1次（741、781）
#
#   15/12/28    新規作成
#   16/01/04    2次要素の場合、中間節点のみに設定
#      03/27    シェル要素1次（731、741）を追加
#      04/08    シェル要素1次（761、781）を追加
#      12/17    シェル要素の面積、ビーム要素の長さの取得を追加
#      12/21    getElementWithNode:高速化の為修正
#      12/25    setAllNodesList:バグ修正（nodeListの取りこぼしがあった。）
#   17/08/30    calcNodeForce,calcNodeForceLine:高速化の為、修正。
#      08/31    getNodeForceWM:追加。（実行状況をdialogでモニタしながら処理）
#      09/01    setAllElementList,setAllNodeElmList:高速化のため修正
#      09/14    getHeaderData:meshFileの内容をheader毎のblockを読み込み処理する様に修正
#      09/15    高速化の為、必要なnodeデータのみ取得する様に修正
#      09/22    getNodeForceWM:高速化の為、meshHeaderDataから設定する様に修正。
#   18/12/06    calcNodeForce:print分を修正。
#   19/01/07    python3用に書き換え
#      04/20    vsCodeにより修正
#   22/03/15    国際化
#   23/04/17    mshファイル名をhecmw_ctrl.datファイルから取得に変更
#   24/07/07    openにencoding="utf-8"を追加
#


import os
import pyFistr

nodeList = []
elementList = []
areaSize = []
nodeElmList = []
needNodeList = []

#
#  getHeaderData
#    headerとそのdataを取得
def getHeaderData(lines):
    flag = 0
    headerData = []
    data = []
    header = ""
    for i in range(len(lines)):
        line = lines[i]
        if line[:1] == "!" and line[:2] != "!!":
            if header != "":
                headerData.append([header, data])
            header = lines[i]
            data = []
            flag = 1
        else:
            flag = 0
        if flag == 0:
            data.append(lines[i])
    headerData.append([header, data])
    return headerData

#
#  setAllNodesList
#    nodeのdataを取得
def setAllNodesList(headerData):
    global nodeList

    def getNodeData(nodeLines):
        #nodeのdataを取得
        nodes = []
        i = 0
        while i < len(nodeLines):
            words = nodeLines[i].split(",")
            node = int(words[0])
            if node < len(needNodeList):
                if len(needNodeList[node]) != 0:
                    data = [node, float(words[1]), float(words[2]), float(words[3])]
                    nodes.append(data)
            i += 1
        return nodes

    #nodeのdataを取得
    nodesData = []
    for hData in headerData:
        header = hData[0]
        if header[:len("!NODE")] == "!NODE":
            nodeLines = hData[1]
            nodes = getNodeData(nodeLines)
            nodesData += nodes
    #nodeDataをlistに保存
    nodesData.sort()
    maxNo = nodesData[-1][0]
    # nodeList = []
    # for i in range(maxNo+1):
    #     nodeList.append([])
    nodeList = [ [] for i in range(maxNo+1) ]
    for data in nodesData:
        node = data[0]       
        loc = data[1:]
        if node < len(needNodeList):
            if len(needNodeList[node]) != 0:
                nodeList[node] = loc

#
#  setAllNodesListHeader
#    nodeのdataを取得
def setAllNodesListHeader(meshHeaderData):
    global nodeList
    #nodeのdataを取得
    nodesData = []
    for hData in meshHeaderData:
        header = hData[0]
        if header[:len("!NODE")] == "!NODE":
            nodesData += hData[1]
    #nodeDataをlistに保存
    nodesData.sort()
    maxNo = nodesData[-1][0]
    # nodeList = []
    # for i in range(maxNo+1):
    #     nodeList.append([])
    nodeList = [ [] for i in range(maxNo+1) ]
    for data in nodesData:
        node = data[0]       
        loc = data[1:]
        if node < len(needNodeList):
            if len(needNodeList[node]) != 0:
                nodeList[node] = loc

#  isIncludeNeedNode
#    必要なnodeを含んでいるかどうか確認する
def isIncludeNeedNode(nodes):
    ans = False
    for node in nodes:
        if node < len(needNodeList):
            if needNodeList[node] != 0:
                ans = True
                break
    return ans

#
#  setAllNodeElmList
#    nodeNoをindexとしてelementのデータを保存（nodeElmList）
def setAllNodeElmList(headerData):
    global nodeElmList, elementList

    def getElementData(elLines):
        elements = []
        i = 0
        while i < len(elLines):
            line = elLines[i]
            words = line.split(",")
            data = []
            for word in words:
                try:
                    data.append(int(word))
                except:
                    pass
            #必要なnodeを含むelementのみ取得する
            if isIncludeNeedNode(data[1:]) == True:
                elements.append(data)
            i += 1
        return elements

    #配列を準備
    elementsData = []
    # nodeElmList = []
    # for i in range(len(nodeList) + 1):
    #     nodeElmList.append([])
    nodeElmList = [ [] for i in range(len(nodeList) + 1) ]
    #要素の内容を取得
    elements = []
    for hData in headerData:
        header = hData[0]
        if header[:len("!ELEMENT")] == "!ELEMENT":
            elType = pyFistr.getValName(header, "TYPE")
            elLines = hData[1]
            elements = getElementData(elLines)
            for element in elements:
                elNo = element[0]
                nodes = element[1:]
                n = 0
                for node in nodes:
                    #必要なnodeのみ取得する
                    if node < len(needNodeList):
                        if len(needNodeList[node]) != 0:
                            nodeElmList[node] += [[elType, elNo, n]]
                    n += 1
            #elemrntsTypeを保存（elementList作成用）
            elementsData.append([elType, elements])
    return elementsData                

#
#  setAllNodeElmListHeader
#    nodeNoをindexとしてelementのデータを保存（nodeElmList）
def setAllNodeElmListHeader(meshHeaderData):
    global nodeElmList, elementList

    def getElementData(elements):
        elms = []
        for elm in elements:
            if isIncludeNeedNode(elm[1:]) == True:
                elms.append(elm)
        return elms

    #配列を準備
    elementsData = []
    # nodeElmList = []
    # for i in range(len(nodeList) + 1):
    #     nodeElmList.append([])
    nodeElmList = [ [] for i in range(len(nodeList) + 1) ]
    #要素の内容を取得
    elements = []
    for hData in meshHeaderData:
        header = hData[0]
        if header[:len("!ELEMENT")] == "!ELEMENT":
            elType = pyFistr.getValName(header, "TYPE")
            elements = hData[1]
            elements = getElementData(elements)
            for element in elements:
                elNo = element[0]
                nodes = element[1:]
                n = 0
                for node in nodes:
                    #必要なnodeのみ取得する
                    if node < len(needNodeList):
                        if len(needNodeList[node]) != 0:
                            nodeElmList[node] += [[elType, elNo, n]]
                    n += 1
            #elemrntsTypeを保存（elementList作成用）
            elementsData.append([elType, elements])
    return elementsData                


#
#  setAllElementList
#    elementNoをindexとしてelementDataを保存（elementList）
def setAllElementList(elementsData):
    global elementList
    #elementsDataからelementListに保存
    elNums = []
    for elements in elementsData:
        for elNo in elements[1]:
            elNums.append(elNo[0])
    maxNo = max(elNums)
    # elementList = []
    # for i in range(maxNo+1):
    #     elementList.append([])
    elementList = [ [] for i in range(maxNo+1) ]
    #elementList作成
    for elements in elementsData:
        elType = elements[0]
        for data in elements[1]:
            elNo = data[0]
            nodes = data[1:]
            elementList[elNo] = [elType, nodes]

#
#  getNodeNo
#    nodeNoを取得
def getNodeNo(headerData, ngrpName):
    nodeNo = []
    for hData in headerData:
        header = hData[0]
        if header[:len("!NGROUP")] == "!NGROUP":
            name = pyFistr.getValName(header, "NGRP")
            if name == ngrpName:
                for line in hData[1]:
                    words = pyFistr.deleteSp(line).split(",")
                    for word in words:
                        if word != "":
                            nodeNo.append(int(word))
    return nodeNo

#
#  getNodeNoHeader
#    nodeNoをmeshHeaderDataから取得
def getNodeNoHeader(meshHeaderData, ngrpName):
    nodeNo = []
    for hData in meshHeaderData:
        header = hData[0]
        if header[:len("!NGROUP")] == "!NGROUP":
            name = pyFistr.getValName(header, "NGRP")
            if name == ngrpName:
                nodeNo = hData[1]
    return nodeNo
    
#
#  setNeedNodeNoToList
#    必要なnodeNoのみnodeListを作成し、マークを記入
def setNeedNodeNoToList(nodeNo):
    global needNodeList
    maxNodeNo = max(nodeNo)
    #nodeList = []
    # needNodeList = []
    # for i in range(maxNodeNo+1):
    #     needNodeList.append([])
    needNodeList = [ [] for i in range(maxNodeNo+1) ]
    for node in nodeNo:
        needNodeList[node] = [0, 0.0, 0.0, 0.0]      #マークを追加

#
#  getElementWithNode
#    nodeを含むelementを取得
#    [[<elType>, <elNo>], [nodeOrder]]を取得
#    [["761", 2942],      [0, 1, 2]]
def getElementWithNode(nodeNo):
    #配列を準備
    elementNodes = []
    for i in range(len(elementList)):
        elementNodes.append([])
    #elementを取得
    for node in nodeNo:
        elms = nodeElmList[node]
        for elm in elms:
            [elType, elNo, n] = elm
            elementNodes[elNo] += [n]
    #データ整形
    elNodes = []
    for i in range(len(elementNodes)):
        nodes = elementNodes[i]
        if len(nodes) > 0:
            nodes.sort()
            data = [[elementList[i][0], i], nodes]
            elNodes.append(data)
    return elNodes

#
#  lengthPP
#    線分の長さを取得
def lengthPP(loc1, loc2):
    locx = loc1[0] - loc2[0]
    locy = loc1[1] - loc2[1]
    locz = loc1[2] - loc2[2]
    l = pow((locx*locx + locy*locy + locz*locz), 0.5)
    return l    

#
#  calcTriangleArea
#    三角形の面積を計算
def calcTriangleArea(loc1, loc2, loc3):
    #各辺の長さを計算
    a = lengthPP(loc1, loc2)
    b = lengthPP(loc2, loc3)
    c = lengthPP(loc3, loc1)
    s = (a + b + c)/2.0
    area = pow((s*(s - a)*(s - b)*(s - c)), 0.5)
    return area

#
#  calcArea341
#    三角形の面積を取得
def calcArea341(elNo, nodes):
    element = elementList[elNo]
    elNodes = element[1]
    #node番号を取得
    nodeNo1 = elNodes[nodes[0]]
    nodeNo2 = elNodes[nodes[1]]
    nodeNo3 = elNodes[nodes[2]]
    #座標を取得
    loc1 = nodeList[nodeNo1]
    loc2 = nodeList[nodeNo2]
    loc3 = nodeList[nodeNo3]
    area = calcTriangleArea(loc1, loc2, loc3)
    return [area, [nodeNo1, nodeNo2, nodeNo3]]

#
#  calcArea342
#    三角形の面積を取得
def calcArea342(elNo, nodes):
    #エラーチェック
    if len(nodes) < 6:
        area = 0.0
        return [area, []]

    element = elementList[elNo]
    elNodes = element[1]
    #node番号を取得
    nodeNo1 = elNodes[nodes[0]]
    nodeNo2 = elNodes[nodes[1]]
    nodeNo3 = elNodes[nodes[2]]
    nodeNo4 = elNodes[nodes[3]]
    nodeNo5 = elNodes[nodes[4]]
    nodeNo6 = elNodes[nodes[5]]
    #座標を取得
    loc1 = nodeList[nodeNo1]
    loc2 = nodeList[nodeNo2]
    loc3 = nodeList[nodeNo3]
    #面積計算
    area = calcTriangleArea(loc1, loc2, loc3)
    return [area, [nodeNo1, nodeNo2, nodeNo3, nodeNo4, nodeNo5, nodeNo6]]

#
#  calcArea361
#    faceの面積を取得
def calcArea361(elNo, nodes):
    if len(nodes) < 4:
        area = 0.0
        return [area, []]
    element = elementList[elNo]
    elNodes = element[1]
    #node番号を取得
    nodeNo1 = elNodes[nodes[0]]
    nodeNo2 = elNodes[nodes[1]]
    nodeNo3 = elNodes[nodes[2]]
    nodeNo4 = elNodes[nodes[3]]
    #座標を取得
    loc1 = nodeList[nodeNo1]
    loc2 = nodeList[nodeNo2]
    loc3 = nodeList[nodeNo3]
    loc4 = nodeList[nodeNo4]
    #面積計算
    area  = calcTriangleArea(loc1, loc2, loc3)
    area += calcTriangleArea(loc3, loc4, loc1)        
    return [area, [nodeNo1, nodeNo2, nodeNo3, nodeNo4]]

#
#  calcArea362
#    faceの面積を取得
def calcArea362(elNo, nodes):
    if len(nodes) < 8:
        area = 0.0
        return [area, []]
    element = elementList[elNo]
    elNodes = element[1]
    #node番号を取得
    nodeNo1 = elNodes[nodes[0]]
    nodeNo2 = elNodes[nodes[1]]
    nodeNo3 = elNodes[nodes[2]]
    nodeNo4 = elNodes[nodes[3]]
    nodeNo5 = elNodes[nodes[4]]
    nodeNo6 = elNodes[nodes[5]]
    nodeNo7 = elNodes[nodes[6]]
    nodeNo8 = elNodes[nodes[7]]
    #座標を取得
    loc1 = nodeList[nodeNo1]
    loc2 = nodeList[nodeNo2]
    loc3 = nodeList[nodeNo3]
    loc4 = nodeList[nodeNo4]
    #面積計算
    area  = calcTriangleArea(loc1, loc2, loc3)
    area += calcTriangleArea(loc3, loc4, loc1)        
    return [area, [nodeNo1, nodeNo2, nodeNo3, nodeNo4, nodeNo5, nodeNo6, nodeNo7, nodeNo8]]

#
#  calcArea351
#    faceの面積を計算
def calcArea351(elNo, nodes):
    if len(nodes) == 3:
        sNodes = calcArea341(elNo, nodes)
    elif len(nodes) == 4:
        sNodes = calcArea361(elNo, nodes)
    else:
        area = 0.0
        sNodes = [area, []]
    return sNodes

#
#  calcArea352
#    faceの面積を計算
def calcArea352(elNo, nodes):
    if len(nodes) == 6:
        sNodes = calcArea342(elNo, nodes)
    elif len(nodes) == 8:
        sNodes = calcArea362(elNo, nodes)
    else:
        area = 0.0
        sNodes = [area, []]
    return sNodes

#
#  calcArea731
#    shellFaceの面積を計算
def calcArea731(elNo, nodes):
    if len(nodes) == 3:
        sNodes = calcArea341(elNo, nodes)
    elif len(nodes) == 6:
        sNodes = calcArea341(elNo, nodes[:3])
    return sNodes

#
#  calcArea741
#    shellFaceの面積を計算
def calcArea741(elNo, nodes):
    if len(nodes) == 4:
        sNodes = calcArea361(elNo, nodes)
    elif len(nodes) == 8:
        sNodes = calcArea361(elNo, nodes[:4])
    return sNodes

#
#  calcArea732
#    shell2Faceの面積を計算
def calcArea732(elNo, nodes):
    #sNodes = calcArea342(elNo, nodes)
    return calcArea342(elNo, nodes)

#
#  calcArea742
#    shell2Faceの面積を計算
def calcArea742(elNo, nodes):
    #sNodes = calcArea362(elNo, nodes)
    return calcArea362(elNo, nodes)

#
#  getAreaWithNodes
#    nodeを含む領域の面積を取得
def getAreaWithNodes(elNodes):
    areaNodes = []
    for elNode in elNodes:
        if len(elNode[1]) >= 3:
            elType = elNode[0][0]       #要素タイプ
            elNo = elNode[0][1]         #要素No
            node = elNode[1]            #nodeNoの組
            if elType == "341":
                sNodes = calcArea341(elNo, node)
            elif elType == "342":
                sNodes = calcArea342(elNo, node)
            elif elType == "361":
                sNodes = calcArea361(elNo, node)
            elif elType == "362":
                sNodes = calcArea362(elNo, node)
            elif elType == "351":
                sNodes = calcArea351(elNo, node)
            elif elType == "352":
                sNodes = calcArea352(elNo, node)
            elif elType == "732":
                sNodes = calcArea732(elNo, node)
            elif elType == "742":
                sNodes = calcArea742(elNo, node)
            elif elType == "731" or elType == "761":
                sNodes = calcArea731(elNo, node)
            elif elType == "741" or elType == "781":
                sNodes = calcArea741(elNo, node)
            area = sNodes[0]
            if area != 0.0:
                areaNodes.append(sNodes)
    return areaNodes

#
#  calcNodeForce
#    節点荷重をセット（荷重合計＝1.0）
def calcNodeForce(areaNodes, nodeNo):
    #全面積を取得
    area = 0.0
    for [s, nodes] in areaNodes:
        area += s
    #配列を準備
    maxNodeNo = max(nodeNo)
    # nodeForce = []
    # for i in range(maxNodeNo+1):
    #     nodeForce.append(0.0)
    nodeForce = [ 0.0 for i in range(maxNodeNo+1) ]
    #1次要素の等分布荷重を算出
    if len(nodes) == 3:
        #三角形1次要素
        for [s, nodes] in areaNodes:
            force = s / area / len(nodes)
            for setNode in nodes:
                nodeForce[setNode] += force
    elif len(nodes) == 4:
        #四角形1次要素
        for [s, nodes] in areaNodes:
            force = s / area / len(nodes)
            #nodesに荷重を配分
            for setNode in nodes:
                nodeForce[setNode] += force
    #2次要素の場合
    elif len(nodes) == 6:
        #三角形2次要素（コーナ:0、中間:1/3）
        for [s, nodes] in areaNodes:
            force = s / area / 3.0
            #中間節点のみへ配分
            for n in [3,4,5]:
                setNode = nodes[n]
                nodeForce[setNode] += force
    elif len(nodes) == 8:
        #四角形2次要素（コーナ:-1/12、中間:1/3）
        for [s, nodes] in areaNodes:
            force = s / area / 12.0
            #コーナ節点への配分
            for n in [0,1,2,3]:
                setNode = nodes[n]
                nodeForce[setNode] += -1.0 * force
            #中間節点への配分
            for n in [4,5,6,7]:
                setNode = nodes[n]
                nodeForce[setNode] += 4.0 * force
    #nodeForceを取得保存
    setForce = []
    for node in nodeNo:
        force = nodeForce[node]
        setForce.append(force)
    #面積算出
    a = area
    #荷重係数算出（合計＝1.0）
    tForce = 0.0
    for force in setForce:
        tForce += force
    print (_(u"面に等分布荷重設定　面の面積:") + str(a) + _(u"  荷重係数:") + str(tForce))
    return setForce

#
#  calcLine341
#    lineの長さを取得
def calcLine341(elNo, nodes):
    element = elementList[elNo]
    elNodes = element[1]
    #node番号を取得
    nodeNo1 = elNodes[nodes[0]]
    nodeNo2 = elNodes[nodes[1]]
    #座標を取得
    loc1 = nodeList[nodeNo1]
    loc2 = nodeList[nodeNo2]
    #線分の長さを取得
    lx = (loc1[0]-loc2[0])
    ly = (loc1[1]-loc2[1])
    lz = (loc1[2]-loc2[2])
    line = pow((lx*lx + ly*ly + lz*lz), 0.5)
    return [line, [nodeNo1, nodeNo2]]

#
#  calcLine342
#    lineの長さを取得
def calcLine342(elNo, nodes):
    if len(nodes) != 3:
        line = 0.0
        return [line, []]
    element = elementList[elNo]
    elNodes = element[1]
    #node番号を取得
    nodeNo1 = elNodes[nodes[0]]
    nodeNo2 = elNodes[nodes[1]]
    nodeNo3 = elNodes[nodes[2]]
    #座標を取得
    loc1 = nodeList[nodeNo1]
    loc2 = nodeList[nodeNo2]
    #線分の長さを取得
    lx = (loc1[0]-loc2[0])
    ly = (loc1[1]-loc2[1])
    lz = (loc1[2]-loc2[2])
    line = pow((lx*lx + ly*ly + lz*lz), 0.5)
    return [line, [nodeNo1, nodeNo2, nodeNo3]]    

#
#  calcLine361
#    線分の長さを取得
def calcLine361(elNo, nodes):
    lNodes = calcLine341(elNo, nodes)
    return lNodes

#
#  calcLine362
#    線分の長さを取得
def calcLine362(elNo, nodes):
    lNodes = calcLine342(elNo, nodes)
    return lNodes

#
#  calcLine351
#    線分の長さを取得
def calcLine351(elNo, nodes):
    lNodes = calcLine341(elNo, nodes)
    return lNodes

#
#  calcLine352
#    線分の長さを取得
def calcLine352(elNo, nodes):
    lNodes = calcLine342(elNo, nodes)
    return lNodes

#
#  calcLine731
#    線分の長さを取得
def calcLine731(elNo, nodes):
    lNodes = calcLine341(elNo, nodes)
    return lNodes

#
#  calcLine741
#    線分の長さを取得
def calcLine741(elNo, nodes):
    lNodes = calcLine341(elNo, nodes)
    return lNodes

#
#  calcLine611
#    beamの線分の長さを取得
def calcLine611(elNo, nodes):
    lNodes = calcLine341(elNo, nodes)
    return lNodes

#
#  getLineWithNodes
#    nodeを含む線分の長さを取得
def getLineWithNodes(elNodes):
    lineNodes = []
    for elNode in elNodes:
        if len(elNode[1]) >= 2:
            elType = elNode[0][0]       #要素タイプ
            elNo = elNode[0][1]         #要素No
            node = elNode[1]            #nodeNoの組
            if elType == "341":
                lNodes = calcLine341(elNo, node)
            elif elType == "342":
                lNodes = calcLine342(elNo, node)
            elif elType == "361":
                lNodes = calcLine361(elNo, node)
            elif elType == "362":
                lNodes = calcLine362(elNo, node)
            elif elType == "351":
                lNodes = calcLine351(elNo, node)
            elif elType == "352":
                lNodes = calcLine352(elNo, node)
            elif elType == "731" or elType == "761":
                lNodes = calcLine731(elNo, node)
            elif elType == "741" or elType == "781":
                lNodes = calcLine741(elNo, node)
            elif elType == "611" or elType == "641":
                lNodes = calcLine611(elNo, node)
            line = lNodes[0]
            if line != 0.0:
                lineNodes.append(lNodes)
    if len(lineNodes) != 0:
        #取得した線分をチェックする（違う要素で同じ線分が含まれている場合がある。）
        #  check用の配列checkNodeにコピー
        checkNodes = []
        for i in range(len(lineNodes)):
            lineNode = lineNodes[i]
            checkNodes.append([lineNode, i])
            checkNodes[-1][0][1].sort()         #節点Noをsortする
        #　データ全体をsortする。
        checkNodes.sort()
        newNodes = []
        addNode = []
        for i in range(len(checkNodes)):
            checkNode = checkNodes[i][0]
            if checkNode != addNode:
                #節点をsortする前のデータを保存
                n = checkNodes[i][1]
                newNodes.append(lineNodes[n])
                addNode = checkNode
        return newNodes
    else:
        return lineNodes
    
#
#  calcNodeForceLine
#    線分の荷重を計算
def calcNodeForceLine(lineNodes, nodeNo):
    #全長を取得
    line = 0.0
    for [l, nodes] in lineNodes:
        line += l
    #配列を準備
    maxNodeNo = max(nodeNo)
    # nodeForce = []
    # for i in range(maxNodeNo+1):
    #     nodeForce.append(0.0)
    nodeForce = [ 0.0 for i in range(maxNodeNo+1) ]

    #nodeForceを計算
    if len(nodes) == 2:
        #1次要素（均等に分配）
        for [l, nodes] in lineNodes:
            force = l / line / len(nodes)
            for setNode in nodes:
                nodeForce[setNode] += force
    elif len(nodes) == 3:
        #2次要素（コーナ節点:1、中間節点:4）
        for [l, nodes] in lineNodes:
            force = l / line / 6.0
            for n in range(len(nodes)):
                setNode = nodes[n]
                if n < 2:
                    #コーナ節点
                    nodeForce[setNode] += force
                else:
                    #中間節点
                    nodeForce[setNode] += 4.0 * force
    #nodeForceを取得保存
    setForce = []
    for node in nodeNo:
        force = nodeForce[node]
        setForce.append(force)
    #荷重系数算出
    tForce = 0.0
    for force in setForce:
        tForce += force
    #print _(u"線分に等分布荷重設定　線分の全長:") + str(l) + "  荷重係数:" + str(tForce)
    print (_(u"線分に等分布荷重設定　線分の全長:") + str(line) + "  荷重係数:" + str(tForce))
    return setForce

#
#  calcNodeForcePoint
#    各pointの均等に荷重をセット
def calcNodeForcePoint(nodeNo):
    force = 1.0/float(len(nodeNo))
    # setForce = []
    # for i in range(len(nodeNo)):
    #     setForce.append(force)
    setForce = [ force for i in range(len(nodeNo)) ]
    print (_(u"点に等分布荷重設定　点の数:") + str(len(nodeNo)))
    return setForce

#
#  getNodeForce
#  ------------
#
def getNodeForce(lines, ngrpName):
    #全てのnode、elementのdataを取得
    print (_(u"メッシュデータを取得中..."))
    headerData = getHeaderData(lines)
    setAllNodesList(headerData)
    elementsData = setAllNodeElmList(headerData)
    setAllElementList(elementsData)
    #nodeNoを取得
    print (_(u"節点座標を取得中..."))
    nodeNo = getNodeNo(headerData, ngrpName)
    #nodeを含むelementを取得
    elNodes = getElementWithNode(nodeNo)
    #領域の面積と領域のnodeNoを取得
    areaNodes = getAreaWithNodes(elNodes)
    if len(areaNodes) > 0:
        #面積に応じた荷重を計算
        nodeForce = calcNodeForce(areaNodes, nodeNo)
    else:
        #線分の長さに応じた荷重を計算
        lineNodes = getLineWithNodes(elNodes)
        if len(lineNodes) > 0:
            nodeForce = calcNodeForceLine(lineNodes, nodeNo)
        else:
            #節点に均等に荷重をかける
            nodeForce = calcNodeForcePoint(nodeNo)
    return [nodeNo, nodeForce]

#
#  getNodeForceWM
#  ------------
#    実行状況をdialog上にモニタしながら処理する
#    呼び出し専用のプロセス
def getNodeForceWM(meshHeaderData, ngrpName, settingDialog):

    global nodeList, elementList, areaSize, nodeElmList, needNodeList

    def numSettingDialog(num, setCont):
        line = str(num) + "," + setCont + "\n"
        settingDialog.stdin.write(line)

    #全てのnode、elementのdataを取得
    print (_(u"メッシュデータを取得中..."))
    numSettingDialog(20, "reading node No. of " + ngrpName + "...")
    nodeNo = getNodeNoHeader(meshHeaderData, ngrpName)
    numSettingDialog(25, "setting node No. of " + ngrpName + "...")
    setNeedNodeNoToList(nodeNo)
    numSettingDialog(30, "reading node locations...")
    setAllNodesListHeader(meshHeaderData)
    numSettingDialog(40, "reading element data...")
    elementsData = setAllNodeElmListHeader(meshHeaderData)
    numSettingDialog(50, "setting elements and nodes data...")
    setAllElementList(elementsData)
    #nodeNoを取得
    print (_(u"節点座標を取得中..."))
    #nodeを含むelementを取得
    numSettingDialog(60, "getting element No. of " + ngrpName + "..." )
    elNodes = getElementWithNode(nodeNo)
    #領域の面積と領域のnodeNoを取得
    numSettingDialog(70, "calculating area...")
    areaNodes = getAreaWithNodes(elNodes)
    numSettingDialog(80, "calculating each force...")
    if len(areaNodes) > 0:
        #面積に応じた荷重を計算
        nodeForce = calcNodeForce(areaNodes, nodeNo)
    else:
        #線分の長さに応じた荷重を計算
        lineNodes = getLineWithNodes(elNodes)
        if len(lineNodes) > 0:
            nodeForce = calcNodeForceLine(lineNodes, nodeNo)
        else:
            #節点に均等に荷重をかける
            nodeForce = calcNodeForcePoint(nodeNo)
    #配列をクリア
    nodeList = []
    elementList = []
    areaSize = []
    nodeElmList = []
    #headerData = []
    needNodeList = []
    return [nodeNo, nodeForce]

#
#  getNodeForceWOM
#  ------------
#    実行状況をdialog上にモニタせずに処理する
#    呼び出し専用のプロセス
def getNodeForceWOM(meshHeaderData, ngrpName):

    global nodeList, elementList, areaSize, nodeElmList, needNodeList

    #全てのnode、elementのdataを取得
    print (_(u"メッシュデータを取得中..."))
    nodeNo = getNodeNoHeader(meshHeaderData, ngrpName)
    setNeedNodeNoToList(nodeNo)
    setAllNodesListHeader(meshHeaderData)
    elementsData = setAllNodeElmListHeader(meshHeaderData)
    setAllElementList(elementsData)
    #nodeNoを取得
    print (_(u"節点座標を取得中..."))
    #nodeを含むelementを取得
    elNodes = getElementWithNode(nodeNo)
    #領域の面積と領域のnodeNoを取得
    areaNodes = getAreaWithNodes(elNodes)
    if len(areaNodes) > 0:
        #面積に応じた荷重を計算
        nodeForce = calcNodeForce(areaNodes, nodeNo)
    else:
        #線分の長さに応じた荷重を計算
        lineNodes = getLineWithNodes(elNodes)
        if len(lineNodes) > 0:
            nodeForce = calcNodeForceLine(lineNodes, nodeNo)
        else:
            #節点に均等に荷重をかける
            nodeForce = calcNodeForcePoint(nodeNo)
    #配列をクリア
    nodeList = []
    elementList = []
    areaSize = []
    nodeElmList = []
    #headerData = []
    needNodeList = []
    return [nodeNo, nodeForce]



if __name__=="__main__":
    import gettext
    localeDir = os.getenv("LOCALE_DIR")
    gettext.install("easyistr", localeDir)
    #_ = gettext.gettext

    workDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(workDir)
    #fileName = "FistrModel.msh"
    fileName = FistrModel_msh
    lines = []
    f = open(fileName, encoding="utf-8")
    for line in f.readlines():
        if line != "\n":
            lines.append(line)
    f.close()
    ngrpName = "point"
    [nodeNo, nodeForce] = getNodeForce(lines, ngrpName)
    print (nodeNo)
    print (nodeForce)

