#!/usr/bin/python3
#  coding: utf-8
#
#       focusJobControlDialog.py
#
#       FOCUSのJob管理用
#       JobFileの選択、作成、投入。slurmコマンドは自動作成される。
#
#   16/10/20    新規作成
#      10/28    「#sbatch」を大文字「#SBATCH」に変更。logファイルが設定されないため。
#               「#SBATCH -J」の場所を変更。jib名をjobFile名と同じ設定に修正。
#      11/06    caseDirを引数から取得する様に修正。
#      12/05    runJob:submit時にjobIdファイル作成を追加
#               setQueueCpu:logFile名をfocus_data内データから設定
#      12/10    showLog、getLogFile追加。tail-fコマンドでlogを表示。
#   17/01/15    plotWatcherを追加
#      02/26    「nCPUs」→「nPs」に修正。
#      03/06    scancel:停止確認のdialogの表示を追加。
#      05/20    sshHeaderをssh実行前に追加
#      05/28    runFocusCommandReturnCont:sshHeader追加時、「" "」の追加をしない様に修正。
#      07/01    setCurrDir:追加（caseDirを表示させる。）
#      07/13    runTerminal:バグ修正（login端末起動時のdir設定を修正）
#      08/05    showLog, plotWatcher:jobを表示させていない場合、currentDirでlogFileを検索
#      08/09    setCurrDir:currDirの表示がおかしくなる事があり、修正。
#   18/06/05    gnome-terminalのオプション「-x」を「--」に変更
#   19/11/07    GTK+3, python3用に大幅修正
#   20/04/21    多言語化対応
#   22/10/07    squeues:コマンドを修正。
#               "squeues"→"env LD_LIBRARY_PATH=/opt/munge/lib64 /home1/share/bin/squeues"
#      10/15    easyIstr用に書き換え
#   23/04/13    "FistrModel.cnt"を使わずにhecmw_ctrl.datから取得する様に修正
#   24/06/06    universalDialogsに修正
#      06/08    okDialog,errDialog,okCancelDialog,openFileDialog:parentを追加
#      06/12    windowApp:window位置をmouseに合わせる様修正
#      07/07    openにencoding="utf-8"を追加
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk
import os, sys
import glob
import shutil

import pyFistr
import universalDialogs

#import locale
#localeDir = os.getenv("easyIstrPath") + os.sep + "data" + os.sep + "locale"
localeDir = os.getenv("LOCALE_DIR")
#locale.bindtextdomain("easyistr", localeDir)

import logFileCreater
logFileCreater.log()

FistrModel_cnt = ""
serverName = ""
caseDir = ""
jobFile = ""
queue = ""
nCpu = ""
nThread = ""
nNode = ""
queueNames = []         #queue名のlist

ls = "\n"

#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("easyistr")
        path = os.getenv("easyIstrPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "focusJobControlDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに」合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #変数の設定
        self.caseDir = caseDir
        self.serverName = ""            #接続するserver名
        self.queueNames = []            #comboにセットする内容
        #GUIのobject名を取得
        self.setGtkObject()
        #初期化
        self.initialize()
        self.setComboBox()              #comboにデータセット
        #self.hideSbatchOption()         #sbatchOptionを隠す
        self.setSqueueLabel()           #squeue実行時のtitle行を設定
        self.setCurrDir()               #currDirをlabelに設定
        self.iniTreeView()              #treeView（実行状況のlist）初期化

    def setGtkObject(self):
        """ glade内のobject名を取得する"""
        self.entry_jobFile = self.builder.get_object("entry_jobFile")
        self.entry_nProcs = self.builder.get_object("entry_nProcs")
        self.entry_nThreads = self.builder.get_object("entry_nThreads")
        self.entry_nNodes = self.builder.get_object("entry_nNodes")
        self.entry_otherComm = self.builder.get_object("entry_otherComm")
        self.combo_queue = self.builder.get_object("combo_queue")
        self.text_commList = self.builder.get_object("text_commList")
        self.tree_jobList = self.builder.get_object("tree_jobList")
        self.button_setJobCont = self.builder.get_object("button_setJobCont")
        self.button_editQueueCont = self.builder.get_object("button_editQueueCont")
        self.button_runJob = self.builder.get_object("button_runJob")
        self.label_squeueTitle = self.builder.get_object("label_squeueTitle")
        self.label_currDir = self.builder.get_object("label_currDir")
        #labelに文字をセット
        self.mainWindow.set_title(_("FOCUS Job管理"))
        self.builder.get_object("label_title").set_label(_("FOCUS Job管理"))
        self.builder.get_object("label_submitJob").set_label(_("Job投入"))
        mess  = _("Jobファイルを選択、作成して、Jobを投入する。") + ls
        mess += _("sbatchオプション（キュー名、プロセス数等）は、右画面上で再設定できる。")
        self.builder.get_object("label_setConts").set_label(mess)
        self.builder.get_object("label_jobFile").set_label(_("Job file名"))
        self.builder.get_object("label_ref").set_label(_("参照..."))
        self.builder.get_object("label_editJob").set_label(_("Job編集"))
        self.builder.get_object("label_queue").set_label(_("キュー"))
        self.button_setJobCont.set_label(_("設定"))
        self.button_editQueueCont.set_label(_("FOCUS設定file編集"))
        self.button_runJob.set_label(_("Job投入\n実行..."))
        self.builder.get_object("label_statNodes").set_label(_("ノードの状況確認"))
        self.builder.get_object("label_otherCommand").set_label(_("その他コマンド"))
        self.builder.get_object("label_jobStat").set_label(_("Job実行状況の確認"))
        self.builder.get_object("label_readJobs").set_label(_("投入Job読込・表示"))
        self.builder.get_object("label_selJob").set_label(_("Jobを選択後、クリック"))
        self.builder.get_object("label_showLog").set_label(_("log表示"))
        self.builder.get_object("label_showZansa").set_label(_("残渣の表示"))
        self.builder.get_object("label_cancelJob").set_label(_("Job停止"))
        self.builder.get_object("button_openFolder").set_label(_("folder開く"))
        self.builder.get_object("button_runTerm").set_label(_("login端末起動"))
        self.builder.get_object("button_close").set_label(_("閉じる"))


    #
    #  main
    #  ----
    def main(self):
        """ GUIを表示する"""
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    #-------- event handler ------------------
    #参照...
    def onRefJobFile(self, event):
        self.selectJob()
    #Job名入力
    def onSetJobFile(self, event):
        self.showSbatchOption()
    #Job編集
    def onEditJobFile(self, event):
        self.editJob()
    #設定
    def onSetJobCont(self, event):
        self.setJobCont()
    #queue設定file編集
    def onEditQueueCont(self, event):
        self.editQueueCont()
    #Job投入実行
    def onRunJob(self, event):
        self.runJob()
    #freenodes
    def onFreenodes(self, event):
        self.freenodes()
    #squeues
    def onSqueues(self, event):
        self.squeues()
    #sinfo
    def onSinfo(self, event):
        self.sinfo()
    #uacct
    def onUacct(self, event):
        self.uacct()
    #enterOtherComm
    def onEnterOtherComm(self, event):
        self.runOtherComm()
    #run
    def onRunOtherComm(self, event):
        self.runOtherComm()
    #squeue
    def onSqueue(self, event):
        self.squeue()
    #tail_f
    def onTail_f(self, event):
        self.showLog()
    #plotWatcher
    def onPlotStepMonitor(self, event):
        self.plotStepMonitor()
    #scancel
    def onScancel(self, event):
        self.scancel()
    #folder開く
    def onOpenFolder(self, event):
        self.openFolder()
    #login端末起動
    def onRunLoginTerm(self, event):
        self.runLoginTerm()
    #閉じる
    def onClose(self, event):
        self.close()
    #-------------------------------------------

    #
    #  initialize
    #  ----------
    def initialize(self):
        """ 初期化"""
        global jobFile, queue, nCpu, nThread, nNode, FistrModel_cnt
        FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(self.caseDir)
        (conts, _logFiles) = self.readFocusData()
        self.queueNames = conts
        self.serverName = pyFistr.servers().getServerName(self.caseDir)
        #optionを隠す
        self.hideSbatchOption()
        #jobFileNameを取得
        fileName = self.caseDir + os.sep + ".jobFileName"
        if os.path.exists(fileName):
            #jobFile有無を確認
            f = open(fileName, encoding="utf-8"); cont = f.read(); f.close()
            jobFile = cont.split()[0]
            if os.path.exists(self.caseDir + os.sep + jobFile):            
                #jobFileをentryに設定
                self.entry_jobFile.set_text(jobFile)
                (queue, nCpu, nThread, nNode) = self.readJobFile(jobFile)
                self.setSbatchOption(queue, nCpu, nThread, nNode)
                self.showSbatchOption()

    #
    #  setComboBox
    #  -----------
    def setComboBox(self):
        """ comboBoxにqueue名をセットする"""
        self.combo_queue.clear()
        listModel = Gtk.ListStore(str)
        for name in self.queueNames:
            listModel.append([name])
        cell = Gtk.CellRendererText()
        self.combo_queue.pack_start(cell, True)
        self.combo_queue.add_attribute(cell, "text", 0)
        self.combo_queue.set_model(listModel)
        self.combo_queue.set_entry_text_column(0)

    #
    #  hideSbatchOption
    #  ----------------
    def hideSbatchOption(self):
        """ sbatchオプション関連を隠す"""
        self.combo_queue.set_sensitive(False)
        self.entry_nProcs.set_sensitive(False)
        self.entry_nThreads.set_sensitive(False)
        self.entry_nNodes.set_sensitive(False)
        self.button_setJobCont.set_sensitive(False)
        self.button_editQueueCont.set_sensitive(False)
        self.button_runJob.set_sensitive(False)

    #
    #  showSbatchOption
    #  ----------------
    def showSbatchOption(self):
        """ sbatchオプション関連を表示させる"""
        self.combo_queue.set_sensitive(True)
        self.entry_nProcs.set_sensitive(True)
        self.entry_nThreads.set_sensitive(True)
        self.entry_nNodes.set_sensitive(True)
        self.button_setJobCont.set_sensitive(True)
        self.button_editQueueCont.set_sensitive(True)
        self.button_runJob.set_sensitive(True)

    #
    #  setSqueueLabel
    #  --------------
    def setSqueueLabel(self):
        """ squeue実行時のtitle行を設定"""
        titleLine = "   JOBID PARTITION     NAME     USER ST       TIME  NODES NODELIST(REASON)"
        self.label_squeueTitle.set_label(titleLine)

    #
    #  setCurrDir
    #  ----------
    def setCurrDir(self):
        """ currDirをlabelにセットして表示"""
        caseDir = self.caseDir
        if len(caseDir) > 73:
            caseDir = "..." + caseDir[-70:]
        currDirCont = "currDir: " + caseDir
        self.label_currDir.set_label(currDirCont)

    #
    #  iniTreeView
    #  -----------
    def iniTreeView(self):
        """ treeViewを初期化する"""
        #title行を非表示
        self.tree_jobList.set_headers_visible(False)
        #treeColumnの設定
        cell = Gtk.CellRendererText()
        col = Gtk.TreeViewColumn()
        col.pack_start(cell, True)
        col.add_attribute(cell, "text", 0)
        self.tree_jobList.append_column(col)

    #
    #  readJobfile
    def readJobFile(self, jobName):
        """ JobFileを読み込み、queue, nCpu, nThread, nNodeを返す。
        定義されていない場合は、「""」を返す。
        """
        fileName = self.caseDir + "/" + jobName
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        queue = ""; nCpu = ""; nThread = ""; nNode = ""
        for line in lines:
            if line[:len("#SBATCH ")].upper() == "#SBATCH ":
                words = line.split()
                if len(words) >= 3:
                    if words[1] == "-n":
                        nCpu = words[2]
                    elif words[1] == "-N":
                        nNode = words[2]
                    elif words[1] == "-p":
                        queue = words[2]
                    elif words[1] == "-c":
                        nThread = words[2]
        return (queue, nCpu, nThread, nNode)

    #
    #  setSbatchOption
    def setSbatchOption(self, queue, nCpu, nThread, nNode):
        """ sbatchのオプションをtextBoxに設定する。"""
        entry = self.combo_queue.get_child()
        entry.set_text(queue)
        self.entry_nProcs.set_text(nCpu)
        self.entry_nThreads.set_text(nThread)
        self.entry_nNodes.set_text(nNode)

    #
    #  selectJob
    #  ---------
    def selectJob(self):
        global jobFile, queue, nCpu, nNode
        title = _("Jobファイルの選択")
        #mess = _("編集するJobファイルを選択する。")
        filterSets = [["all","*"]]
        funcOk = [self.selectJobProcess]
        funcCancel = []
        self.openFileDialog(title, self.caseDir, funcOk, funcCancel, self.mainWindow, jobFile, filterSets)
        return
        # fileName = self.fileDialog(title, self.caseDir, [["all","*"]])
        # if fileName != "":
        #     self.selectJobProcess(fileName)

    #  selectJobProcess
    def selectJobProcess(self, fileName):
        jobFile = fileName.split(os.sep)[-1]
        self.entry_jobFile.set_text(jobFile)
        (queue, nCpu, nThread, nNode) = self.readJobFile(jobFile)
        self.setSbatchOption(queue, nCpu, nThread, nNode)
        self.showSbatchOption()
        #jobFileNameを保存
        jobFileName = self.caseDir + os.sep + ".jobFileName"
        f = open(jobFileName, "w", encoding="utf-8"); f.write(jobFile); f.close()

    #
    #  editJob
    #  -------
    def editJob(self):
        """ JobFileの編集"""
        jobName = self.entry_jobFile.get_text()
        if jobName == "":
            title = _("エラー")
            mess = _("job名が入力されていません。") + "\n"
            mess += _("  job名を入力してください。")
            self.errDialog(title, mess, parent=self.mainWindow)
            return
        self.showSbatchOption()
        fileName = self.caseDir + "/" + jobName
        appDict = pyFistr.readUsingApplications()
        editor = appDict["editor"]
        comm = editor + " " + fileName
        pyFistr.run(self.caseDir).commandWithLogBG(comm)

    #
    #  setJobCont
    #  ----------
    def setJobCont(self):
        """ sbatchのoptionを書き込む"""
        jobFile = self.entry_jobFile.get_text()
        job = jobFile
        queue = self.combo_queue.get_child().get_text()
        nCpu = self.entry_nProcs.get_text()
        nThread = self.entry_nThreads.get_text()
        nNode = self.entry_nNodes.get_text()
        (oFile, eFile) = self.getFocusOutputLogFileNames(job)
        error = self.writeJobFile(jobFile, job, queue, nCpu, nThread, nNode, oFile, eFile)
        if error != "":
            title = _("エラー")
            self.errDialog(title, error, parent=self.mainWindow)
            return
        title = _("Jobファイルの修正")
        mess = _("以下の内容を書き込みました。\n")
        mess += ("  #SBATCH -p " + queue + "\n")
        mess += ("  #SBATCH -n " + nCpu + "\n")
        mess += ("  #SBATCH -c " + nThread + "\n")
        mess += ("  #SBATCH -N " + nNode + "\n")
        mess += ("  #SBATCH -J " + job + "\n")
        mess += ("  #SBATCH -e " + eFile + "\n")
        mess += ("  #SBATCH -o " + oFile + "\n")
        self.okDialog(title, mess, parent=self.mainWindow)

    #
    #  editQueueCont
    #  -------------
    def editQueueCont(self):
        """ focusの設定file編集"""
        fileName = os.getenv("easyIstrUserPath") + os.sep + "data" + os.sep + "focus_data"
        appDict = pyFistr.readUsingApplications()
        editor = appDict["editor"]
        comm = editor + " " + fileName
        pyFistr.run(self.caseDir).commandWithLog(comm)
        #pyFistr.run(self.caseDir).editorWait([fileName])
        (data, _logFiles) = self.readFocusData()
        self.queueNames = data
        selQueue = self.combo_queue.get_child().get_text()
        self.setComboBox()
        if selQueue in self.queueNames:
            idNo = self.queueNames.index(selQueue)
        else:
            idNo = 0
        self.combo_queue.set_active(idNo)

    #
    #  runJob
    #  ------
    def runJob(self):
        """ job投入・実行"""
        jobFile = self.entry_jobFile.get_text()
        (queue, nCpu, nThread, nNode) = self.readJobFile(jobFile)
        error = ""
        if queue == "":
            error = _("キューが設定されていません。")
        elif nCpu == "":
            error = _("nPs（プロセス数）が設定されていません。")
        elif nThread == "":
            error = _("nThrs（スレッド数）が設定されていません。")
        if error != "":
            title = _("エラー")
            self.errDialog(title, error, parent=self.mainWindow)
            return

        title = _("Jobの投入")
        mess = _("以下の条件でJobを投入します。\n")
        mess += _("  キュー名:  ") + queue + "\n"
        mess += _("  process数: ") + nCpu + "\n"
        mess += _("  thread数:  ") + nThread + "\n"
        mess += _("  node数:    ") + nNode + "\n"
        funcOk = [self.runJobProcess, jobFile]
        funcCancel = []
        self.okCancelDialog2(title, mess, funcOk, funcCancel, parent=self.mainWindow)
        return
        # stat = self.okCancelDialog(title, mess)
        # if stat != "OK":
        #     return
        # self.runJobProcess(jobFile)

    #  runJobProcess
    def runJobProcess(self, jobFile):
        #今回の結果を書き込む
        cont = self.runAddThisJobToJobListCont(jobFile)
        #jobFileNameを保存
        fileName = self.caseDir + os.sep + ".jobFileName"
        f = open(fileName, "w", encoding="utf-8"); f.write(jobFile); f.close()
        #実行の状況を表示
        title = _("Jobの投入")
        mess = _("Jobを投入しました。") + "\n\n"
        mess += cont
        self.okDialog(title, mess, parent=self.mainWindow)

    #
    #  runAddThisJobToJobListCont
    def runAddThisJobToJobListCont(self, jobFile):
        """ jobを投入し、その結果を返す。
        jobListContに今回のjobを追加し、古いjobは削除する。"""
        script = os.getenv("easyIstrPath") + os.sep + "python" + os.sep + "focus_addJobList.py"
        server = pyFistr.server(self.serverName)
        hostDir = server.sshfsDict["hostDir"]
        mountDir = server.sshfsDict["mountPoint"]
        jobDir = hostDir + self.caseDir[len(mountDir):]
        args = jobDir + " " + jobFile
        #スクリプトを転送し、実行
        (_stat, res, err) = server.moveRunPythonReturnCont(script, args)
        cont = res + err
        return cont

    #
    #  getJobListCont
    def getJobListCont(self, server):
        """ ~/.jobListContの内容を取得"""
        mountDir = server.sshfsDict["mountPoint"]
        fileName = mountDir + os.sep + ".jobList"
        #if len(glob.glob(fileName)) == 0:
        if os.path.exists(fileName) == False:
            return ""
        else:
            f = open(fileName, encoding="utf-8"); jobListCont = f.read(); f.close()
            return jobListCont

    #
    #  freenodes
    #  ---------
    def freenodes(self):
        """ freenodesコマンド実行"""
        comm = "freenodes"
        cont = self.runFocusCommandReturnCont("~", comm)
        buffer = self.text_commList.get_buffer()
        buffer.set_text(cont)

    #
    #  squeues
    #  -------
    def squeues(self):
        """ squeuesコマンド実行"""
        #comm = "squeues"
        comm = "env LD_LIBRARY_PATH=/opt/munge/lib64 /home1/share/bin/squeues"
        cont = self.runFocusCommandReturnCont("~", comm)
        buffer = self.text_commList.get_buffer()
        buffer.set_text(cont)

    #
    #  sinfo
    #  -----
    def sinfo(self):
        """ sinfoコマンド実行"""
        comm = "sinfo"
        cont = self.runFocusCommandReturnCont("~", comm)
        buffer = self.text_commList.get_buffer()
        buffer.set_text(cont)

    #
    #  uacct
    #--------
    def uacct(self):
        """ uacctコマンド実行"""
        comm = "uacct"
        cont = self.runFocusCommandReturnCont("~", comm)
        buffer = self.text_commList.get_buffer()
        buffer.set_text(cont)

    #
    #  runOtherComm
    #  ------------
    def runOtherComm(self):
        """ その他コマンド"""
        server = pyFistr.server(self.serverName)
        hostDir = server.sshfsDict["hostDir"]
        mountPoint = server.sshfsDict["mountPoint"]
        workDir = hostDir + self.caseDir[len(mountPoint):]
        comm = self.entry_otherComm.get_text()
        cont = self.runFocusCommandReturnCont(workDir, comm)
        buffer = self.text_commList.get_buffer()
        buffer.set_text(cont)

    #
    #  squeue
    #  ------
    def squeue(self):
        """ squeueコマンドの実行（実行状況のリスト）"""
        cont = self.getClearJobList()
        lines = cont.split("\n")[:-1]
        items = []
        for line in lines[1:]:
            i = 0
            while i <= 10:
                chara = line[i]
                if chara == " ":
                    i += 1
                else:
                    break
            items.append(line[i:])
        #treeに値をセット
        listStore = Gtk.ListStore(str)
        for name in items:
            listStore.append([name])
        self.tree_jobList.set_model(listStore)

    #
    #  getClearJobList
    def getClearJobList(self):
        """ jobListを取得し現状に合わせる"""
        script = os.getenv("easyIstrPath") + os.sep + "python" + os.sep + "focus_clearJobList.py"
        server = pyFistr.server(self.serverName)
        args = ""
        #スクリプトを転送し、実行
        (_stat, cont, _err) = server.moveRunPythonReturnCont(script, args)
        return cont

    #
    #  showLog
    #  -------
    def showLog(self):
        """ 選択されたJobのlogを表示（tail-f）"""

        def dialogOk():
            #currentDir内でlogFileを取得
            FistrModelDict = pyFistr.getCntMshFileNameDict(self.caseDir)
            header = pyFistr.getHeaderName(FistrModelDict["res"])
            logFile = self.caseDir + os.sep + header + ".log"
            self.showLogProcess(logFile)

        #選択しているjobを取得
        listSelection = self.tree_jobList.get_selection()
        (listStore, pathes) = listSelection.get_selected_rows()
        if len(pathes) == 0:
            #itemを選択していない場合
            if listStore == None:
                n = 0
            else:
                n = len(listStore)
            if n == 0:
                #listBox内が空の場合
                title = _("log表示")
                mess = _("currentDirectoryのlogを表示します。")
                funcOk = [dialogOk]
                funcCancel = []
                self.okCancelDialog2(title, mess, funcOk, funcCancel, parent=self.mainWindow)
                return
                # stat = self.okCancelDialog(title, mess)
                # if stat != "OK":
                #     return
                # #currentDir内でlogFileを取得
                # FistrModelDict = pyFistr.getCntMshFileNameDict(self.caseDir)
                # header = pyFistr.getHeaderName(FistrModelDict["res"])
                # logFile = self.caseDir + os.sep + header + ".log"
            else:
                #listBox内にjobが表示されている状態でjobを選択していない場合
                title = _("エラー")
                mess = _("jobが選択されていません。")
                self.errDialog(title, mess, parent=self.mainWindow)
                return
        else:
            iter = listStore.get_iter(pathes[0])
            item = listStore.get_value(iter, 0)
            jobId = item.split()[0]
            serverLogFile = self.getLogFileFromjobId(jobId)
            server = pyFistr.server(self.serverName)
            hostDir = server.sshfsDict["hostDir"]
            mountDir = server.sshfsDict["mountPoint"]
            logFile = mountDir + serverLogFile[len(hostDir):]
        self.showLogProcess(logFile)

    #  showLogProcess
    def showLogProcess(self, logFile):
        comm = "tail -n 100 -f " + logFile
        fileName = os.getenv("easyIstrUserPath") + os.sep + "runBash"
        f = open(fileName, "w", encoding="utf-8"); f.write(comm); f.close()
        pyFistr.run().command("chmod a+x " + fileName)
        appDict = pyFistr.readUsingApplications()
        terminalRun = appDict["terminalRun"]
        commLine = terminalRun + " " + fileName
        pyFistr.run().command(commLine)

    #
    #  plotStepMonitor
    #  -----------
    def plotStepMonitor(self):
        """ plotStepMonitorの起動"""

        def dialogOk(server):
            (logFile, statFile,
             cntFile, subDataFile) = self.getEachFileNames(server)
            self.plotStepMonitorProcess(server, logFile, statFile, cntFile, subDataFile)

        #logFileを取得する
        server = pyFistr.server(self.serverName)
        #選択しているjobを取得
        listSelection = self.tree_jobList.get_selection()
        (listStore, pathes) = listSelection.get_selected_rows()
        if len(pathes) == 0:
            #itemを選択していない場合
            if listStore == None:
                n = 0
            else:
                n = len(listStore)
            if n == 0:
                #listBox内が空の場合
                title = _("log表示")
                mess = _("currentDirectoryのlogを表示します。")
                funcOk = [dialogOk, server]
                funcCancel = []
                self.okCancelDialog2(title, mess, funcOk, funcCancel, parent=self.mainWindow)
                return
                # stat = self.okCancelDialog(title, mess)
                # if stat != "OK":
                #     return
                # #logFile名を取得
                # (logFile, statFile,
                #   cntFile, subDataFile) = self.getEachFileNames(server)
            else:
                #listBox内にjobが表示されている状態でjobを選択していない場合
                title = _("エラー")
                mess = _("jobが選択されていません。")
                self.errDialog(title, mess, parent=self.mainWindow)
                return
        else:
            iter = listStore.get_iter(pathes[0])
            item = listStore.get_value(iter, 0)
            jobId = item.split()[0]
            jobDir = self.getJobDirFromJobId(jobId)
            logFile = jobDir + os.sep + "0.log"
            statFile = jobDir + os.sep + "FSTR.sta"
            cntFile = jobDir + os.sep + FistrModel_cnt
            subDataFile = jobDir + os.sep + "easyIstrSub_data"
        #plotStepMonitor起動
        self.plotStepMonitorProcess(server, logFile, statFile, cntFile, subDataFile)

    #  getEachFileNames
    def getEachFileNames(self, server):
        #currentDir内でlogFileを取得
        logFileLocal = self.caseDir + os.sep + "0.log"
        statFileLocal = self.caseDir + os.sep + "FSTR.sta"
        cntFileLocal = self.caseDir + os.sep + FistrModel_cnt
        subDataFileLocal = self.caseDir + os.sep + "easyIstrSub_data"
        #server側のlogFileに変換
        mountDir = server.sshfsDict["mountPoint"]
        hostDir = server.sshfsDict["hostDir"]
        logFile = hostDir + logFileLocal[len(mountDir):]
        statFile = hostDir + statFileLocal[len(mountDir):]
        cntFile = hostDir + cntFileLocal[len(mountDir):]
        subDataFile = hostDir + subDataFileLocal[len(mountDir):]
        return logFile, statFile, cntFile, subDataFile

    #
    #  plotStepMonitorProcess
    def plotStepMonitorProcess(self, server, logFile, statFile, cntFile, subDataFile):
        #local側に必要なfileを転送してplotStepMonitorを起動する
        #  plotStepMonitorは、local側で起動する
        #  転送先のdirを確認する
        tempCase = os.getenv("easyIstrUserPath") + os.sep + "temp"
        if os.path.exists(tempCase) == False:
            os.mkdir(tempCase)
        #  local側に転送する
        server.scpCommandNoTerm(logFile, tempCase)
        server.scpCommandNoTerm(statFile, tempCase)
        server.scpCommandNoTerm(cntFile, tempCase)
        server.scpCommandNoTerm(subDataFile, tempCase)
        #FrontISTRのversionをsubDataから取得
        version = "5.0"
        fileName = tempCase + os.sep + os.path.basename(subDataFile)
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        for line in lines:
            words = line.split()
            if len(words) > 1:
                if words[0] == "fistrVersion":
                    version = words[1]
                    break
        ver = version.split(".")[0]
        #anaTypeをcntFileから取得
        fileName = tempCase + os.sep + os.path.basename(cntFile)
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        anaType = "STATIC"
        i = 0
        while i < len(lines):
            line = lines[i]
            if line[:len("!SOLUTION")] == "!SOLUTION":
                anaType = pyFistr.getValName(line, "TYPE")
                break
            i += 1
        if anaType == "DYNAMIC":
            if lines[i+1][:len("!EIGENREAD")] == "!EIGENREAD":
                anaType = "DYNAMIC_F"
        elif anaType == "HEAT":
            if lines[i+1][:len("!HEAT")] == "!HEAT":
                words = pyFistr.deleteSp(lines[i+2]).split(",")
                if len(words) > 3:
                    try:
                        if float(words[0]) > 0.0:
                            anaType = "HEAT_D"
                    except:
                        pass
        saveDir = os.getcwd()
        os.chdir(tempCase) 
        #plotMonitor開始
        if os.name == "nt":
            #windows
            runComm  = "python3 " + pythonAppPath + "plotStepMonitorDialog.py "
            runComm += " ".join([ver, anaType])
            pyFistr.run().commandBG(runComm)
        else:
            #linux
            runComm  = "python3 " + pyFistr.pythonAppPath + "plotStepMonitorDialog.py "
            runComm += " ".join([ver, anaType])
            pyFistr.run().commandBG(runComm)
        #環境を戻す
        os.chdir(saveDir)
        return

    #
    #  scancel
    #  -------
    def scancel(self):
        """ 選択Jobをキャンセルする"""
        #選択しているjobを取得
        listSelection = self.tree_jobList.get_selection()
        (listStore, pathes) = listSelection.get_selected_rows()
        if len(pathes) == 0:
            title = _("エラー")
            mess = _("停止させるjobが選択されていません。")
            self.errDialog(title, mess, parent=self.mainWindow)
            return
        #確認message
        title = _("jobの停止")
        mess = _("選択しているJobを停止させますか？")
        funcOk = [self.scancelProcess, listStore, pathes]
        funcCancel = []
        self.okCancelDialog2(title, mess, funcOk, funcCancel, parent=self.mainWindow)
        return
        # stat = self.okCancelDialog(title, mess)
        # if stat != "OK":
        #     return
        # self.scancelProcess(listStore, pathes)

    #
    #  scancelProcess
    def scancelProcess(self, listStore, pathes):
        #jobをキャンセル
        iter = listStore.get_iter(pathes[0])
        item = listStore.get_value(iter, 0)
        jobId = item.split()[0]
        comm = "scancel " + jobId
        _cont = self.runFocusCommandReturnCont("~", comm)
        title = _("jobの停止")
        mess = _("以下のjobIdに対し、停止コマンド（scancel）を送出しました。") + "\n\n"
        mess += "jobId: " + jobId
        self.okDialog(title, mess, parent=self.mainWindow)

    #
    #  getLogFileFromjobId
    def getLogFileFromjobId(self, jobId):
        """ jobIdからlogfile(serverのpath）を取得する"""
        jobDir = self.getJobDirFromJobId(jobId)
        #cntName, mshName = pyFistr.cntMshFileName(jobDir)
        FistrModelDict = pyFistr.getCntMshFileNameDict(jobDir)
        header = pyFistr.getHeaderName(FistrModelDict["res"])
        #logFile = jobDir + os.sep + "FistrModel.log"
        logFile = jobDir + os.sep + header + ".log"
        return logFile

    #
    #  getJobDirFromJobId
    def getJobDirFromJobId(self, jobId):
        """ jobIdからserverのjobDirを取得する"""
        server = pyFistr.server(self.serverName)
        jobList = self.getJobListCont(server)
        jobLines = jobList.split("\n")
        jobDir = ""
        for line in jobLines:
            words = line.split()
            if len(words) > 0:
                if jobId == words[0]:
                    jobDir = words[2]
                    break
        return jobDir

    #
    #  openFolder
    #  ----------
    def openFolder(self):
        """ folderを開く"""
        appDict = pyFistr.readUsingApplications()
        fileManager = appDict["fileManager"]
        comm = fileManager + " " + self.caseDir
        pyFistr.run(self.caseDir).commandWithLogBG(comm)

    #
    #  runLoginTerm
    #  ------------
    def runLoginTerm(self):
        """ login端末を開く"""
        server = pyFistr.server(self.serverName)
        hostDir = server.sshfsDict["hostDir"]
        mountDir = server.sshfsDict["mountPoint"]
        setEnviron = server.sshfsDict["setEnviron"]
        #loginDir, setEnvironを作成
        if mountDir == self.caseDir[:len(mountDir)]:
            openDir = hostDir + self.caseDir[len(mountDir):]
            lines = setEnviron.split("\n")
            i = len(lines) - 1
            while i >= 0:
                if lines[i][:len("cd")] == "cd":
                    #words = lines[i].split()
                    lines[i] = "cd " + openDir
                    break
                i -= 1
            setEnviron = "\n".join(lines)
        else:
            setEnviron = ""
        #tempFile作成
        name = "setEnviron"
        fileName = os.getenv("easyIstrUserPath") + os.sep + name
        f=open(fileName, "w"); f.write(setEnviron); f.close()
        #environを転送
        server.scpCommandNoTerm(fileName, "~")
        #login端末を起動 
        server.runLoginTerm()

    #
    #  writeJobFile
    def writeJobFile(self, jobFile, job, queue, nCpu, nThread, nNode, oFile, eFile):
        """ jobFileに書き込む"""
        error = ""
        #errorチェック
        if job == "":
            error = _("Jobファイル名が設定されていません。")
            return error
        if queue == "" or nCpu == "" or nThread == "" or nNode == "":
            error = _("queue, nCpu, nNodeが入力されていません。")
            return error

        jobFile = self.caseDir + os.sep + jobFile
        #ファイルが存在しない場合、作成する
        #if len(glob.glob(jobFile)) == 0:
        if os.path.exists(jobFile) == False:
            f=open(jobFile, "w"); f.write("#!/bin/bash\n"); f.close()
        #ファイルを読み込む
        f = open(jobFile, encoding="utf-8"); lines = f.readlines(); f.close()
        #関係する行を削除
        newLines = []
        for line in lines:
            if line[:len("#SBATCH ")].upper() == "#SBATCH ":
                words = line.split()
                if not (words[1] == "-J" or words[1] == "-n"
                    or words[1] == "-c" or words[1] == "-N"  
                    or words[1] == "-p" or words[1] == "-e" 
                    or words[1] == "-o"):
                    newLines.append(line)
            else:
                newLines.append(line)
        #追加する行を作成
        addLines  = ["#SBATCH -p " + queue + "\n"]
        addLines += ["#SBATCH -N " + nNode + "\n"]
        addLines += ["#SBATCH -n " + nCpu + "\n"]
        addLines += ["#SBATCH -c " + nThread + "\n"]
        addLines += ["#SBATCH -J " + job + "\n"]
        addLines += ["#SBATCH -e " + eFile + "\n"]
        addLines += ["#SBATCH -o " + oFile + "\n"]
        #行を追加
        newLines = newLines[:1] + addLines + newLines[1:]
        #書き込み
        f = open(jobFile, "w", encoding="utf-8")
        for line in newLines:
            f.write(line)
        f.close()
        return error

    #
    #  getFocusOutputLogFileNames
    def getFocusOutputLogFileNames(self, job):
        """ logFile名を取得"""
        (_data, logFiles) = self.readFocusData()
        (logFile, errLogFile) = logFiles
        logFile = self.getStringNameFromEnv(logFile, job)
        errLogFile = self.getStringNameFromEnv(errLogFile, job)
        return (logFile, errLogFile)

    #
    #  getStringNameFromEnv
    def getStringNameFromEnv(self, name, job):
        """ 環境変数(name)を文字列に変換する"""

        def getEnvName(name, j):
            if name[j] == "{":
                n = name[j:].find("}")
                if n >= 0:
                    jj = j + n
                    strName = name[j+1:jj]
                else:
                    strName = name[j+1:]
                    jj = len(name)
            else:
                strName = name[j:]
                jj = len(name)
            return (strName, jj)

        newName = ""
        i = 0
        while i < len(name):
            chara = name[i]
            if chara == "$":
                (envName, i) = getEnvName(name, i+1)
                if envName == "jobName":
                    strName = job
                else:
                    strName = os.getenv(envName)
                    if strName == None:
                        strName = ""
                newName += strName
            else:
                newName += chara
            i += 1
        return newName

    #
    #  readFocusData
    def readFocusData(self):
        """ FOCUSのデータを読み込む"""
        fileName = os.getenv("easyIstrUserPath") + os.sep + "data" + os.sep + "focus_data"
        #if len(glob.glob(fileName)) == 0:
        if os.path.exists(fileName) == False:
            orgName = os.getenv("easyIstrPath") + os.sep + "data" + os.sep + "focus_data"
            shutil.copy(orgName, os.getenv("easyIstrUserPath") + os.sep + "data")
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        lines = self.deleteComment(lines)
        flag = 0; logFlag = 0; eLogFlag = 0
        queueNames = []; logFile = "solve.log"; errLogFile = "${jobName}.e%J"
        for line in lines:
            if line[0] != " ":
                words = line.split()
                if len(words) > 0:
                    if words[0] == "queueNames":
                        flag = 1
                    elif words[0] == "logFileName":
                        flag = 0
                        if len(words) >= 2:
                            logFile = words[1]
                            logFlag = 1
                    elif words[0] == "errLogFileName":
                        flag = 0
                        if len(words) >= 2:
                            errLogFile = words[1]
                            eLogFlag = 1
                    else:
                        flag = 0
            else:
                if flag == 1:
                    words = line.split()
                    if len(words) > 0:
                        queueNames.append(words[0])
        queueNames.sort()
        logFiles = [logFile, errLogFile]
        if logFlag == 1 and eLogFlag == 1:
            return (queueNames, logFiles)

        #fileを読み込み再作成する
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        if logFlag == 0 and eLogFlag == 0:
            #logFile errLogFile共、再作成
            addLines = ["#  Log file names\n"]
            addLines += ["#    These names are able to change at following lines.\n"]
            addLines += ["logFileName     " + logFile + "\n"]
            addLines += ["errLogFileName  " + errLogFile + "\n"]
            lines += addLines
        elif logFlag == 0:
            #logFileを追加
            addLines = ["logFileName     " + logFile + "\n"]
            for i in range(len(lines)):
                line = lines[i]
                if line[:len("errLogFileName ")] == "errLogFileName ":
                    break
            lines = lines[:i] + addLines + lines[i:]
        elif eLogFlag == 0:
            #errLogFileを追加
            addLines = ["errLogFileName  " + errLogFile + "\n"]
            for i in range(len(lines)):
                line = lines[i]
                if line[:len("logFileName ")] == "logFileName ":
                    break
            lines = lines[:i] + addLines + lines[i:]
        #書き込み
        f = open(fileName, "w", encoding="utf-8")
        for line in lines:
            f.write(line)
        f.close()
        #戻る
        return (queueNames, logFiles)

    #
    #  runFocusCommandReturnCont
    def runFocusCommandReturnCont(self, workDir, comm):
        """ FOCUSコマンドを実行し、その結果を返す。"""
        server = pyFistr.server(self.serverName)
        hostName = server.sshfsDict["hostName"]
        sshHeader = server.sshfsDict["sshHeader"]
        userName = server.sshDict["user"]
        #コマンド作成
        cont = "cd " + workDir + "; "
        cont += comm
        #コマンド実行
        commLine = "ssh " + userName + "@" + hostName + " '" + cont + "'" 
        print(commLine)
        commLine = sshHeader + commLine
        result = self.runShellCommand(commLine)
        return result

    #
    #  runShellCommand
    def runShellCommand(self, comm):
        result = ""
        (_stat, res, err) = pyFistr.run().commandReturnCont(comm, isOut=False)
        result = res + err
        return result

    #
    #  deleteComment
    def deleteComment(self, lines):
        """ コメント文を削除"""
        newLines = []
        for line in lines:
            n = line.find("#")
            if n >= 0:
                newLine = line[:n] + "\n"
            else:
                newLine= line
            newLines.append(newLine)
        return newLines

    #---universal dialog--------------
    def okDialog(self, title, message, parent=None, funcOk=[]):
        dialog = universalDialogs.okDialog(title, message, parent, funcOk)
        dialog.show()

    def errDialog(self, title, message, parent=None, funcOk=[]):
        dialog = universalDialogs.errDialog(title, message, parent, funcOk)
        dialog.show()

    def okCancelDialog2(self, title, mess, funcOk, funcCancel, parent=None):
        dialog = universalDialogs.okCancelDialog(title, mess, funcOk, funcCancel, parent)
        dialog.show()

    def openFileDialog(self, title, fileDir, funcOk, funcCancel, parent=None, selName="", filterSets=[]):
        dialog = universalDialogs.openFileDialog(title, fileDir, funcOk, funcCancel, parent, selName, filterSets)
        dialog.show()




if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    winApp = windowApp(caseDir)
    winApp.main()
