#!/usr/bin/python3
#  coding: utf-8
#
#           createMeshFromCadFileDialog.py
#
#       cadFile（brep, step, stp）からメッシュを作成する
#       メッシュは、
#           beam 1次
#           三角形,四角形shell 1次
#           四面体tetra,六面体hexa 1次2次
#       を作成する。
#
#   25/02/20    新規作成
#      03/01    setModelSizeToLabel:maxhの算出方法変更。
#      04/17    createMeshGmsh_run:不要addMeshを削除。
#      05/20    moduleとして実行をprocessとして実行する様に修正
#      05/21    部分的にmeshサイズが変更できる様に修正。
#               cadのvtk表示を追加。
#      05/22    createVtkActorDict:edges.vtuをedges.vtkに変更し、
#               これに伴い、readerも変更。
#      05/23    dummyRunDialog:windows対応で追加。
#               windowsの場合、これを追加しないと、entryがactiveにならない。
#      06/03    部分的なmeshSizeの設定対象にsurface、edgeを追加。
#               NGRP,SGRPの設定を追加。
#      06/04    setInitialValue:起動時、cadFile名と異なるvtkが表示される
#               事がある為、修正。
#      06/06    createResultFile:書込内容に「!END」を追加。
#               pyFistr.readFistrFileUntilEndで読み込める様にするため。
#      06/21    createVtkSourceFromCad_runGmsh:process起動時にvtkFormatの引数を追加
#      06/26    vtkInitialize:versionが異なった場合、vtkFolder内をクリアを追加。
#               versionが異なっていた場合、エラー発生し起動しない。（バグ）
#      06/29    saveCadFileToEasyIstrSub_data:新規追加。runGmsh前にcadFile名を
#               easyIstrSub_dataに保存
#      06/30    getGlobalMeshSizeFromEdgeElmLength:globalMeshSizeをmodelSize
#               から決定していたが、edge要素（beam）のmaxSizeから決定する様に修正。
#      07/18    setButtonSensitive_localMesh,setButtonSensitive_ngrpSgrp:新規追加
#               各ボタンの表示、非表示を設定する。
#               getSurfaceAverageLocalMeshSize,getEdgeAverageLocalMeshSize,
#               getPointLocalMeshSize:新規追加。選択itemのmeshSizeを取得する。
#      07/19    getPickedCellId:localMeshでvtkの面線点を選択した時、選択場所の
#               設定値をentryに表示する様に修正。
#               setMeshSizeToEntry:localMeshの設定表が非選択の時、entryに書き込む
#               様に修正。
#      08/10    選択モードに応じてvtk画面のbackgroundColorを変更する様に修正。
#      08/13    createVtkSourceFromCad_runGmsh:gmshを裏で起動し、終了をチェックする
#               方法に変更。
#               setStatusBar:statusBarにメッセージの表示を追加。
#      08/17    getPickedEdge:edgeのpick方法をworldPointPickerを使ってpickする様に修正
#      08/18    settingDialog,numSettingDialog,closeSettingDialog:削除
#               これらの処理は、readCadFileCreateVtkFiles_gmshに移動
#               isPickOnLine:新規追加。pickLocがline上か否かを確認し距離l算出。
#               getGlobalMeshSizeFromEdgeElmLength:削除。meshSizeは、別の方法で取得。
#               meshSizeは、readCadFileCreateVtkFiles_gmsh.pyの出力から検出して取得
#      08/21    viewFromXP,viewFromXM,viewFromYP,viewFromYM,viewFromZP,viewFromZM:
#               vtk表示を+-XYZ方向の6種類に追加。
#      09/05    setModelSizeMeshSizeFromVtk:modelSizeファイルが出来上がる前に読みに行く
#               事があるため、修正。
#               createVtkSourceFromCad_runGmsh:gmsh起動前に、logFile削除を追加。
#               cadFileを変更したときに、cadVtkが正常に表示されないため。バグ修正。
#      09/20    createVtkActorDict,addPointCellsActors:reader基準→actor基準でcolorを設定。
#      09/21    checkGmshLog:CADファイル再読込後、vtk画面がresetされなかった為、修正。
#      09/24    addPointCellsActors:「self.pickedMode="no"」の時、モデルのgree色変更。
#      09/25    選択部品の非表示設定、EGRPの設定を追加する大幅修正。
#               addPointCellsActors:モデル表示を「RGB」→「RGBA」に変更（透明化追加）
#               edgeが選択しにくいので、線幅を「4」→「6」に変更。
#               surfaceの表裏表示設定を追加。
#

import vtk
import GtkVtkRenderWindowInteractor as gtkVtk
from vtk.util import colors

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib

import os, sys
import glob
import shutil
import math
import time
import subprocess
import GtkParts
import pyFistr
import geometryFunc as geo
import universalDialogs
import readCadFileCreateVtkFiles_gmsh as createVtk
import createMesh_gmsh

localeDir = os.getenv("LOCALE_DIR")

#AppのPathを定義
binAppPath = pyFistr.binAppPath             #binFolder
pythonAppPath = pyFistr.pythonAppPath       #pythonFolder
#numSetting = 0                              #settingDialogの進捗

#-------------------
#  createMesh class
#-------------------
class createMesh:

    def __init__(self, currDir):
        self.builder = Gtk.Builder()
        path = os.getenv("easyIstrPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "createMeshFromCadFileDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.mainWindow.set_title(_("CADファイルからmesh作成"))
        self.builder.connect_signals(self)
        self.maskEvent = True
        self.setGtkObject()
        #event設定
        #変数の設定
        self.currDir = currDir
        #self.funcAfterMesh = funcAfterMesh
        self.meshFileName = ""
        self.closeFlag = False          #timerEvent停止Flag
        self.runningTimer = False       #timerEvent実行有無Flag
        istrLines = pyFistr.readSavedDataForEasyistr()
        self.istrLines = istrLines
        appDict = pyFistr.readUsingApplications()
        self.vtkFormat = appDict["vtkFormat"]
        self.gmshLogFileObj = ""
        #vtk関連
        self.vtkWinSize = (250,250)     #vtkWindowのサイズ
        self.vtkObj = gtkVtk.GtkVtkRenderWindowInteractor(self)
        self.vtkFolder = currDir + os.sep + "vtkCadData"
        self.renderer = ""
        self.hideVolumes = []           #非表示volumNoのlist
        self.hideSurfaces = []
        self.hideEdges = []
        self.hidePoints = []
        self.actorDict = {}
        self.readerDict = {}
        self.vtkInitialRun = True
        self.originAxes = None
        self.modelDim = "3D"            #modelの次元
        self.vtkSelectMode = "surface"  #選択モード
        self.selPoints = []             #選択したpointNo(red)
        self.allPoints = []             #cad内の全points
        self.selEdges = []              #選択したedgeNo(red)
        self.selSurfaces = []           #選択したsurfaceNo(red)
        self.selVolumes = []            #選択したvolume
        self.edgeDict = {}              #edgeのcellData保存用
        self.surfaceDict = {}           #surfaceのcellData保存用
        self.showPickedMode = "no"      #vtkの表示がpickモードか否か
        self.pickedData = []            #pickしたdata[actor, cells, nodes]
        self.localMeshItemConts = []    #localMeshのlistData
        self.vtkInitialize()
        #初期化
        self.initialize()
        self.setInitialValue(istrLines)      #初期設定（istrDataをセット）
        self.setVtkArea()
        self.clearSelPoints()           #vtk,各表の選択をクリア

    #
    #  main
    #-------
    def main(self):
        """ GUIを表示する"""
        self.setWindowSize()
        self.maskEvent = False
        self.mainWindow.show_all()
        self.setVtkModel()
        self.dummyRunDialog()
        Gtk.main()

    #
    #  close
    def close(self, *args):
        """ 閉じる"""
        self.endindApp()
        self.closeFlag = True
        Gtk.main_quit()

    #  dummyRunDialog
    def dummyRunDialog(self):
        """ Gtk.main()の前に、dilog or windowを表示させる。
        windows特有。これをしないと、mainWindow内のentryがactiveにならない"""
        dialog = universalDialogs.okDialog("", "")
        dialog.show()
        dialog.close()

    #
    #  setWindowSize
    def setWindowSize(self):
        """ windowSizeを設定する"""
        #画面の設定
        winDict = pyFistr.readWindowSize()
        if "createMeshWinSize" in winDict.keys():
            cont = winDict["createMeshWinSize"]
            words = cont.split()
            x, y = list(map(int, words[:2]))
            self.mainWindow.set_default_size(x, y)
        #splitterの設定
        if "createMeshWinSize_splitter" in winDict.keys():
            cont = winDict["createMeshWinSize_splitter"]
            words = cont.split()
            splitPos = int(words[0])
            self.paned_win.set_position(splitPos)

    #
    #  endindApp
    def endindApp(self):
        """ 終了処理"""
        #windowSizeを設定
        if self.mainWindow.is_maximized() == False:
            #fullScreen以外は、windowSizeを保存
            (x, y) = self.mainWindow.get_size()
            winDict = {}
            winDict["createMeshWinSize"] = str(x) + " " + str(y)
            #paned
            splitPos = self.paned_win.get_position()
            winDict["createMeshWinSize_splitter"] = str(splitPos)
            pyFistr.writeWindowSize(winDict)

    #---- vtk関連 ---------------------
    
    #  vtkInitialize
    def vtkInitialize(self):
        """ vtk関連のinitialize"""
        #vtkFolderのチェック
        if os.path.exists(self.vtkFolder) == False:
            os.mkdir(self.vtkFolder)
        else:
            #Versionのチェック
            versionFile = self.vtkFolder + os.sep + "version_" + createVtk.Version
            if os.path.exists(versionFile) == False:
                fileNames = glob.glob(self.vtkFolder + os.sep + "*")
                for fileName in fileNames:
                    os.remove(fileName)
        #notebookのpageを設定
        self.note_createMesh_gmsh.set_current_page(0)
        self.radio_point.set_active(True)
        return
    
    #  setVtkArea
    def setVtkArea(self):
        """ vtkAreaの作成"""
        #vtkAreaを設定
        vtkObj = self.vtkObj
        vtkObj.set_size_request(*self.vtkWinSize)
        self.box_vtk.pack_start(vtkObj, True, True, 0)

    #  setVtkModel
    def setVtkModel(self):
        """ vtkを表示する"""
        #actorを作成        
        self.createVtkActorDict()
        self.addPointCellsActors()
        #rendererをチェック
        if self.renderer == "":
            self.renderer = vtk.vtkRenderer()
        else:
            self.renderer.NewInstance()
        #assyを定義
        self.assy = vtk.vtkAssembly()
        self.setShowVtkItems()
        #backgroundColorを設定
        #self.renderer.SetBackground2(0.5, 0.5, 1)
        #self.renderer.SetBackground(0.8, 0.8, 1)
        self.setBackgroundColor(0)   #defaultColor
        self.renderer.GradientBackgroundOn()
        self.renderer.SetTexturedBackground(True)
        self.renWin = self.vtkObj.GetRenderWindow()
        self.renWin.NewInstance()
        #新しいrendererを追加
        self.renWin.AddRenderer(self.renderer)
        self.addCornerAxis()
        #vtk表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.GlobalWarningDisplayOff()   #windowsのエラー対策
        self.vtkObj.Initialize()
        self.vtkObj.Start()
        return
    
    #
    #  setBackgroundColor
    def setBackgroundColor(self, colorNo):
        """ backgroundColorを設定する
        colorNo=0: default (newSelect)
        colorNo=1: editSelect用"""
        if self.renderer == "":
            return
        #backgropundColorを設定
        if colorNo == 0:
            self.renderer.SetBackground2(0.5, 0.5, 1)
            self.renderer.SetBackground(0.8, 0.8, 1)
            #self.label_selectMode.set_text("newSelect")
        else:
            self.renderer.SetBackground2(0.5, 0.5, 0.9)
            self.renderer.SetBackground(0.8, 0.8, 0.9)
            #self.label_selectMode.set_text("editSelect")
        #モード表示
        pageNum = self.note_createMesh_gmsh.get_current_page()
        self.setModeFromPage(pageNum, colorNo)

    #  setModeFromPage
    def setModeFromPage(self, pageNum, colorNo):
        """ pageNumからmode表示を設定する"""
        if pageNum == 0:
            mode = "showParts"
            cont = _("CADのsurface, edge, pointを表示")
        else:
            if colorNo == 0:
                mode = "newSelect"
                cont = "mode: newSelect"
            else:
                mode = "editSelect"
                cont = "mode: editSelect"
        self.label_selectMode.set_text(mode)
        self.setStatusBar(cont)

    #  setStatusBar
    def setStatusBar(self, mess):
        """ statusBarにメッセージを表示する"""
        self.label_vtkComm.set_text(mess)

    #
    #  createVtkActorDict
    def createVtkActorDict(self):
        """ vtkActorの辞書作成。self.allPointsを取得する。"""
        self.readerDict = {}
        self.actorDict = {}
        self.allPoints = []
        #edgesActor作成
        edgesFile = self.vtkFolder + os.sep + "edges.vtu"
        if os.path.exists(edgesFile) == True:
            reader = vtk.vtkXMLUnstructuredGridReader()
            reader.SetFileName(edgesFile)
            reader.Update()
            self.readerDict["edges"] = reader
            #表示設定のみでactorを作成
            actor = self.createShowEdgesActor()
            prop = actor.GetProperty()
            prop.SetLineWidth(2.0)
            self.actorDict["edges"] = actor
            #全edgesを取得する
            self.edgeDict = self.getAllEdges(actor)
        #pointsActor作成
        pointsFile =  self.vtkFolder + os.sep + "points.vtu"
        if os.path.exists(pointsFile) == True:
            reader = vtk.vtkXMLUnstructuredGridReader()
            reader.SetFileName(pointsFile)
            reader.Update()
            self.readerDict["points"] = reader
            actor = self.createShowPointsActor()
            self.actorDict["points"] = actor
            #全pointsを取得する
            mapper = actor.GetMapper()
            polyData = mapper.GetInput()
            nData = polyData.GetNumberOfPoints()
            if nData == 0:
                self.allPoints = []
            else:
                self.allPoints = polyData.GetPoints().GetData()
        #surfacesActor作成
        surfacesFile = self.vtkFolder + os.sep + "surfaces.vtu"
        if os.path.exists(surfacesFile) == True:
            reader = vtk.vtkXMLUnstructuredGridReader()
            reader.SetFileName(surfacesFile)
            reader.Update()
            self.readerDict["surfaces"] = reader
            actor = self.createShowSurfacesActor()
            self.actorDict["surfaces"] = actor
            #全surfacesを取得する
            self.surfaceDict = self.getAllSurfaces(actor)

    #  createShowPointsActor
    def createShowPointsActor(self):
        """ 非表示部品を除いた部品のみのactorを作成して戻す"""
        #表示のcellIdを取得する
        reader = self.readerDict["points"]
        polyData = reader.GetOutput()
        nCells = polyData.GetNumberOfCells()
        cellData = polyData.GetCellData()
        surData = cellData.GetArray("pointNo")
        volData = cellData.GetArray("volumeNo")
        showIds = []
        for i in range(nCells):
            volNo = volData.GetValue(i)
            surNo = surData.GetValue(i)
            if not (volNo in self.hideVolumes or surNo in self.hidePoints):
                showIds.append(i)
        #cellDataを抽出して新しく作成する
        newIdTypeArray = vtk.vtkIdTypeArray()
        for i in showIds:
            newIdTypeArray.InsertNextValue(i)
        selectionCells = vtk.vtkSelectionNode()
        selectionCells.SetFieldType(vtk.vtkSelectionNode.CELL)
        selectionCells.SetContentType(vtk.vtkSelectionNode.INDICES)
        selectionCells.SetSelectionList(newIdTypeArray)
        selection = vtk.vtkSelection()
        selection.AddNode(selectionCells)
        extractSelection = vtk.vtkExtractSelection()
        extractSelection.SetInputData(0, polyData)
        extractSelection.SetInputData(1, selection)
        extractSelection.Update()
        #curveNo, volumeNoを取得
        newArrayVals = []
        newVolumeNos = []
        newSurNoCells = cellData.GetArray("pointNo")
        volumeNoCells = cellData.GetScalars("volumeNo")
        for i in showIds:
            surNo = newSurNoCells.GetValue(i)
            volumeNo = volumeNoCells.GetValue(i)
            newArrayVals.append(surNo)
            newVolumeNos.append(volumeNo)
        #newArrayを作成
        newOutput = extractSelection.GetOutput()
        newCellData = newOutput.GetCellData()
        nCells = newOutput.GetNumberOfCells()
        newArray = vtk.vtkIntArray()
        newArray.SetNumberOfComponents(1)
        newArray.SetNumberOfTuples(nCells)
        newVolNosArray = vtk.vtkIntArray()
        newVolNosArray.SetNumberOfComponents(1)
        newVolNosArray.SetNumberOfTuples(nCells)
        for i in range(nCells):
            newArray.SetValue(i, newArrayVals[i])
            newVolNosArray.SetValue(i, newVolumeNos[i])
        newArray.SetName("pointNo")
        newVolNosArray.SetName("volumeNo")
        newCellData.AddArray(newArray)
        newCellData.AddArray(newVolNosArray)
        #mapper
        mapper = vtk.vtkDataSetMapper()
        mapper.SetInputData(newOutput)
        #actor
        actor = vtk.vtkActor()
        actor.SetMapper(mapper)
        return actor

    #  createShowEdgesActor
    def createShowEdgesActor(self):
        """ 非表示部品を除いた部品のみのactorを作成して戻す"""
        #表示のcellIdを取得する
        reader = self.readerDict["edges"]
        polyData = reader.GetOutput()
        nCells = polyData.GetNumberOfCells()
        cellData = polyData.GetCellData()
        surData = cellData.GetArray("curveNo")
        volData = cellData.GetArray("volumeNo")
        showIds = []
        for i in range(nCells):
            volNo = volData.GetValue(i)
            surNo = surData.GetValue(i)
            if not (volNo in self.hideVolumes or surNo in self.hideEdges):
                showIds.append(i)
        #cellDataを抽出して新しく作成する
        newIdTypeArray = vtk.vtkIdTypeArray()
        for i in showIds:
            newIdTypeArray.InsertNextValue(i)
        selectionCells = vtk.vtkSelectionNode()
        selectionCells.SetFieldType(vtk.vtkSelectionNode.CELL)
        selectionCells.SetContentType(vtk.vtkSelectionNode.INDICES)
        selectionCells.SetSelectionList(newIdTypeArray)
        selection = vtk.vtkSelection()
        selection.AddNode(selectionCells)
        extractSelection = vtk.vtkExtractSelection()
        extractSelection.SetInputData(0, polyData)
        extractSelection.SetInputData(1, selection)
        extractSelection.Update()
        #curveNo, volumeNoを取得
        newArrayVals = []
        newVolumeNos = []
        newSurNoCells = cellData.GetArray("curveNo")
        volumeNoCells = cellData.GetScalars("volumeNo")
        for i in showIds:
            surNo = newSurNoCells.GetValue(i)
            volumeNo = volumeNoCells.GetValue(i)
            newArrayVals.append(surNo)
            newVolumeNos.append(volumeNo)
        #newArrayを作成
        newOutput = extractSelection.GetOutput()
        newCellData = newOutput.GetCellData()
        nCells = newOutput.GetNumberOfCells()
        newArray = vtk.vtkIntArray()
        newArray.SetNumberOfComponents(1)
        newArray.SetNumberOfTuples(nCells)
        newVolNosArray = vtk.vtkIntArray()
        newVolNosArray.SetNumberOfComponents(1)
        newVolNosArray.SetNumberOfTuples(nCells)
        for i in range(nCells):
            newArray.SetValue(i, newArrayVals[i])
            newVolNosArray.SetValue(i, newVolumeNos[i])
        newArray.SetName("curveNo")
        newVolNosArray.SetName("volumeNo")
        newCellData.AddArray(newArray)
        newCellData.AddArray(newVolNosArray)
        #mapper
        mapper = vtk.vtkDataSetMapper()
        mapper.SetInputData(newOutput)
        #actor
        actor = vtk.vtkActor()
        actor.SetMapper(mapper)
        return actor

    #  createShowSurfacesActor
    def createShowSurfacesActor(self):
        """ 非表示部品を除いた部品のみのactorを作成して戻す"""
        #表示のcellIdを取得する
        reader = self.readerDict["surfaces"]
        polyData = reader.GetOutput()
        nCells = polyData.GetNumberOfCells()
        cellData = polyData.GetCellData()
        surData = cellData.GetArray("surNo")
        volData = cellData.GetArray("volumeNo")
        showIds = []
        for i in range(nCells):
            volNo = volData.GetValue(i)
            surNo = surData.GetValue(i)
            if not (volNo in self.hideVolumes or surNo in self.hideSurfaces):
                showIds.append(i)
        #cellDataを抽出して新しく作成する
        newIdTypeArray = vtk.vtkIdTypeArray()
        for i in showIds:
            newIdTypeArray.InsertNextValue(i)
        selectionCells = vtk.vtkSelectionNode()
        selectionCells.SetFieldType(vtk.vtkSelectionNode.CELL)
        selectionCells.SetContentType(vtk.vtkSelectionNode.INDICES)
        selectionCells.SetSelectionList(newIdTypeArray)
        selection = vtk.vtkSelection()
        selection.AddNode(selectionCells)
        extractSelection = vtk.vtkExtractSelection()
        extractSelection.SetInputData(0, polyData)
        extractSelection.SetInputData(1, selection)
        extractSelection.Update()
        #curveNo, volumeNoを取得
        newArrayVals = []
        newVolumeNos = []
        newSurNoCells = cellData.GetArray("surNo")
        volumeNoCells = cellData.GetScalars("volumeNo")
        for i in showIds:
            surNo = newSurNoCells.GetValue(i)
            volumeNo = volumeNoCells.GetValue(i)
            newArrayVals.append(surNo)
            newVolumeNos.append(volumeNo)
        #newArrayを作成
        newOutput = extractSelection.GetOutput()
        newCellData = newOutput.GetCellData()
        nCells = newOutput.GetNumberOfCells()
        newArray = vtk.vtkIntArray()
        newArray.SetNumberOfComponents(1)
        newArray.SetNumberOfTuples(nCells)
        newVolNosArray = vtk.vtkIntArray()
        newVolNosArray.SetNumberOfComponents(1)
        newVolNosArray.SetNumberOfTuples(nCells)
        for i in range(nCells):
            newArray.SetValue(i, newArrayVals[i])
            newVolNosArray.SetValue(i, newVolumeNos[i])
        newArray.SetName("surNo")
        newVolNosArray.SetName("volumeNo")
        newCellData.AddArray(newArray)
        newCellData.AddArray(newVolNosArray)
        #mapper
        mapper = vtk.vtkDataSetMapper()
        mapper.SetInputData(newOutput)
        #actor
        actor = vtk.vtkActor()
        actor.SetMapper(mapper)
        return actor


    #  getAllSurfaces
    #def getAllSurfaces(self, reader):
    def getAllSurfaces(self, actor):
        """ surfaceActor作成の為のdataを取得して返す。"""
        #cellData = reader.GetOutput().GetCellData().GetArray("surNo")
        mapper = actor.GetMapper()
        polyData = mapper.GetInput()
        cellData = polyData.GetCellData().GetArray("surNo")
        nData = cellData.GetSize()
        surNoDict = {}
        for i in range(nData):
            surNo = cellData.GetValue(i)
            if surNo in surNoDict.keys():
                surNoDict[surNo].append(i)
            else:
                surNoDict[surNo] = [i]
        surfaceDict = {}
        surfaceDict["surNoDict"] = surNoDict
        #cellのdataを取得
        #cellNodeData = self.getVtkCellConts(reader)
        cellNodeData = self.getVtkCellConts(actor)
        surfaceDict["cellNodeData"] = cellNodeData
        #allPointsを取得
        #allPoints = reader.GetOutput().GetPoints().GetData()
        nData = polyData.GetNumberOfPoints()
        if nData == 0:
            allPoints = []
        else:
            allPoints = polyData.GetPoints().GetData()
        surfaceDict["allPoints"] = allPoints
        return surfaceDict

    #
    #  getVtkCellConts
    #def getVtkCellConts(self, reader):
    def getVtkCellConts(self, actor):
        """ readerからcell内容のcellCont(nNodesとnodes)のlistを取得"""
        mapper = actor.GetMapper()
        polyData = mapper.GetInput()
        #cells = reader.GetOutput().GetCells().GetData()
        nCells = polyData.GetNumberOfCells()
        if nCells == 0:
            return []
        cells = polyData.GetCells().GetData()
        nData = cells.GetSize()
        cellNodeData = []
        nNodes = 0
        data = []
        for i in range(nData):
            if nNodes == 0:
                if len(data) > 0:
                    cellNodeData.append(data)
                #cellのnode数を保存
                data = []
                #vtk-8.2対応
                try:
                    nNodes = cells.GetValue(i)
                    data.append(nNodes)
                except:
                    break
            else:
                #nodeNoを保存
                nodeNo = cells.GetValue(i)
                data.append(nodeNo)
                nNodes -= 1
        if len(data) > 0:
            cellNodeData.append(data)
        return cellNodeData

    #  getAllEdges
    #def getAllEdges(self, reader):
    def getAllEdges(self, actor):
        """ edgeActor作成のためのdataを取得して返す"""
        mapper = actor.GetMapper()
        polyData = mapper.GetInput()
        #edgeNoを取得
        #cellData = reader.GetOutput().GetCellData().GetArray("curveNo")
        cellData = polyData.GetCellData().GetArray("curveNo")
        nData = cellData.GetSize()
        edgeNoDict = {}
        for i in range(nData):
            edgeNo = cellData.GetValue(i)
            if edgeNo in edgeNoDict.keys():
                edgeNoDict[edgeNo].append(i)
            else:
                edgeNoDict[edgeNo] = [i]
        edgeDict = {}
        edgeDict["edgeNoDict"] = edgeNoDict
        #cellの内容を取得
        #cellNodeData = self.getVtkCellConts(reader)
        cellNodeData = self.getVtkCellConts(actor)
        edgeDict["cellNodeData"] = cellNodeData
        #edgeNo毎のnodeNoをsurNoPointDictを作成
        edgePointDict = {}
        for i in range(len(cellNodeData)):
            cellCont = cellNodeData[i]
            nodes = cellCont[1:]
            edgeNo = cellData.GetValue(i)
            if edgeNo in edgePointDict.keys():
                edgePointDict[edgeNo].update(set(nodes))
            else:
                edgePointDict[edgeNo] = set(nodes)
        edgeDict["edgePointDict"] = edgePointDict
        #allPointsを取得
        if nData == 0:
            allPoints = []
        else:
            allPoints = polyData.GetPoints().GetData()
        edgeDict["allPoints"] = allPoints
        return edgeDict

    #  addPointCellsActors
    def addPointCellsActors(self):
        """ 選択ponits、非選択pointsのactorを作成、辞書に追加する"""
        #greeを設定
        green = colors.green
        if self.showPickedMode == "no":
            green = colors.green_yellow
        #透明化をセット
        opacity = 255
        if self.check_opacity_gmsh.get_active() == True:
            opacity = 25
        #volume, surfaceを設定
        if "surfaces" in self.actorDict.keys():  
            actor = self.actorDict["surfaces"]
            prop = actor.GetProperty()
            mapper = actor.GetMapper()
            polyData = mapper.GetInput()
            nCells = polyData.GetNumberOfCells()
            #colorの設定
            colorCells = [0 for i in range(nCells)]
            selColor = colors.red
            selColor = list(map(int, [x*255 for x in selColor]))
            selColor += [255]
            unColor = green
            unColor = list(map(int, [x*255 for x in unColor]))
            unColor += [opacity]
            #volume?
            if self.radio_volume.get_active() == True:
                #表面を表示
                prop.BackfaceCullingOn()
                prop.FrontfaceCullingOff()
                #色設定
                cellData = polyData.GetCellData().GetArray("volumeNo")
                for i in range(nCells):
                    volNo = cellData.GetValue(i)
                    if volNo in self.selVolumes:
                        colorCells[i] = selColor
                    else:
                        colorCells[i] = unColor
            #surface?
            else:
                #表裏両面を表示
                prop.BackfaceCullingOff()
                prop.FrontfaceCullingOff()
                #色設定
                cellData = polyData.GetCellData().GetArray("surNo")
                for i in range(nCells):
                    surNo = cellData.GetValue(i)
                    if surNo in self.selSurfaces:
                        colorCells[i] = selColor
                    else:
                        colorCells[i] = unColor
            colorData = vtk.vtkUnsignedCharArray()
            colorData.SetName("colors")
            #colorData.SetNumberOfComponents(3)
            colorData.SetNumberOfComponents(4)
            for ucrgb in colorCells:
                #colorData.InsertNextTuple3(*ucrgb)
                colorData.InsertNextTuple4(*ucrgb)
            polyData.GetCellData().SetScalars(colorData)
        #edgeを設定
        if "edges" in self.actorDict.keys():
            #actorからcellDataを取得
            actor = self.actorDict["edges"]
            mapper = actor.GetMapper()
            polyData = mapper.GetInput()
            nCells = polyData.GetNumberOfCells()
            cellData = polyData.GetCellData().GetArray("curveNo")
            #colorを設定
            colorCells = [0 for i in range(nCells)]
            if self.vtkSelectMode == "edge":
                selColor = colors.red
                unColor = green
                lineWidth = 5.0
            else:
                selColor = colors.white
                unColor = colors.white
                lineWidth = 2.0
            selColor = list(map(int, [x*255 for x in selColor]))
            selColor += [255]
            unColor = list(map(int, [x*255 for x in unColor]))
            unColor += [255]
            for i in range(nCells):
                curveNo = cellData.GetValue(i)
                if curveNo in self.selEdges:
                    colorCells[i] = selColor
                else:
                    colorCells[i] = unColor
            colorData = vtk.vtkUnsignedCharArray()
            colorData.SetName("colors")
            #colorData.SetNumberOfComponents(3)
            colorData.SetNumberOfComponents(4)
            for ucrgb in colorCells:
                #colorData.InsertNextTuple3(*ucrgb)
                colorData.InsertNextTuple4(*ucrgb)
            polyData.GetCellData().SetScalars(colorData)
            prop = actor.GetProperty()
            prop.SetLineWidth(lineWidth)
        #selPointsのactorを作成追加
        if "points" in self.actorDict.keys():
            actor = self.actorDict["points"]
            mapper = actor.GetMapper()
            polyData = mapper.GetInput()
            nCells = polyData.GetNumberOfCells()
            cellData = polyData.GetCellData().GetArray("pointNo")
            #選択対象からcolorを作成
            prop = actor.GetProperty()
            if self.radio_volume.get_active() == True:
                selColor = colors.white
                unColor = colors.white
                prop.SetPointSize(2)
            elif self.radio_surface.get_active() == True:
                selColor = colors.white
                unColor = colors.white
                prop.SetPointSize(10)
            elif self.radio_edge.get_active() == True:
                selColor = colors.white
                unColor = colors.white
                prop.SetPointSize(10)
            elif self.radio_point.get_active() == True:
                selColor = colors.red
                unColor = green
                prop.SetPointSize(12)
            colorCells = [0 for i in range(nCells)]
            selColor = list(map(int, [x*255 for x in selColor]))
            selColor += [255]
            unColor = list(map(int, [x*255 for x in unColor]))
            unColor += [255]
            #pointにcolorを設定
            for i in range(nCells):
                pointNo = cellData.GetValue(i)
                if pointNo in self.selPoints:
                    colorCells[i] = selColor
                else:
                    colorCells[i] = unColor
            colorData = vtk.vtkUnsignedCharArray()
            colorData.SetName("colors")
            #colorData.SetNumberOfComponents(3)
            colorData.SetNumberOfComponents(4)
            for ucrgb in colorCells:
                #colorData.InsertNextTuple3(*ucrgb)
                colorData.InsertNextTuple4(*ucrgb)
            polyData.GetCellData().SetScalars(colorData)

    #  addCornerAxis
    def addCornerAxis(self):
        """ 最初の1回のみ、XYZの軸（compas）をrendererに追加する。"""
        if self.vtkInitialRun == True:
            #add axis(最初の起動時のみ)
            interactor = self.renWin.GetInteractor()
            axesActor = vtk.vtkAxesActor()
            self.axes = vtk.vtkOrientationMarkerWidget()
            self.axes.SetOrientationMarker(axesActor)
            self.axes.SetInteractor(interactor)
            self.axes.EnabledOn()
            self.axes.InteractiveOn()
            #flagをセット
            self.vtkInitialRun = False
        return

    #  getCameraBounds
    def getCameraBounds(self):
        """ 表示させるactorのboundsを取得する。
        原点、マージンも含めて取得する。"""
        bounds = self.getOriginBounds()
        if len(bounds) == 0:
            return bounds
        #マージンを確保
        xmin = bounds[0]
        xmax = bounds[1]
        ymin = bounds[2]
        ymax = bounds[3]
        zmin = bounds[4]
        zmax = bounds[5]
        a = 0.1
        xmin = xmin - (xmax-xmin) * a
        xmax = xmax + (xmax-xmin) * a
        ymin = ymin - (ymax-ymin) * a
        ymax = ymax + (ymax-ymin) * a
        zmin = zmin - (zmax-xmin) * a
        zmax = zmax + (zmax-xmin) * a
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #  getOriginBounds
    def getOriginBounds(self):
        """ 表示させるactorと原点のboundsを取得する。"""
        bounds = self.getBoundsActors()
        if self.originAxes == None:
            #原点無しの場合
            return bounds
        #原点を追加する
        xmin = min([bounds[0], 0.0])
        xmax = max([bounds[1], 0.0])
        ymin = min([bounds[2], 0.0])
        ymax = max([bounds[3], 0.0])
        zmin = min([bounds[4], 0.0])
        zmax = max([bounds[5], 0.0])
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #  getBoundsActors
    def getBoundsActors(self):
        """ 全actorのboundsを取得して返す。"""
        actors = self.renderer.GetActors()
        bounds = []
        for actor in actors:
            bound = actor.GetBounds()
            bounds.append(bound)
        if len(bounds) == 0:
            return ()
        Xmin = min(map(lambda x: x[0], bounds))
        Xmax = max(map(lambda x: x[1], bounds))
        Ymin = min(map(lambda x: x[2], bounds))
        Ymax = max(map(lambda x: x[3], bounds))
        Zmin = min(map(lambda x: x[4], bounds))
        Zmax = max(map(lambda x: x[5], bounds))
        return (Xmin, Xmax, Ymin, Ymax, Zmin, Zmax)

    #
    #  setShowVtkItems
    def setShowVtkItems(self):
        """ 表示するactorを設定する"""
        if len(self.actorDict) == 0:
            return
        #actorをrendererに追加
        self.renderer.RemoveActor(self.assy)
        self.renderer.SetErase(True)
        self.assy = vtk.vtkAssembly()
        if self.vtkSelectMode == "volume":
            actor = self.actorDict["surfaces"]
            self.assy.AddPart(actor)
        if self.vtkSelectMode == "surface":
            actor = self.actorDict["surfaces"]
            self.assy.AddPart(actor)
        #actor = self.actorDict["edges"]
        #self.assy.AddPart(actor)
        if "edges" in self.actorDict.keys():
            actor = self.actorDict["edges"]
            self.assy.AddPart(actor)
        actor = self.actorDict["points"]
        self.assy.AddPart(actor)
        # if "selPoints" in self.actorDict.keys():
        #     actor = self.actorDict["selPoints"]
        #     self.assy.AddPart(actor)
        # if "unselPoints" in self.actorDict.keys():
        #     actor = self.actorDict["unselPoints"]
        #     self.assy.AddPart(actor)
        self.renderer.AddActor(self.assy)
        #原点表示？
        if self.check_showOrgAxis_gmsh.get_active() == True:
            if self.originAxes == None:
                #原点を表示する
                bounds = self.getBoundsActors()
                lenX = (bounds[1] - bounds[0]) / 2.0 * 1.2
                lenY = (bounds[3] - bounds[2]) / 2.0 * 1.2
                lenZ = (bounds[5] - bounds[4]) / 2.0 * 1.2
                length = max([lenX, lenY, lenZ])
                axes = vtk.vtkAxesActor()
                axes.SetTotalLength(length, length, length)
                #axes.SetTotalLength(lenX, lenY, lenZ)
                axes.GetXAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetXAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetYAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetYAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetZAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetZAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                self.renderer.AddActor(axes)
                self.originAxes = axes
        #modelSizeをentryに表示
        self.setModelSizeMeshSizeFromVtk()

    #
    #  setModelSizeMeshSizeFromVtk
    def setModelSizeMeshSizeFromVtk(self):
        """ modeSizeと全体のmeshSizeをvtkから取得し設定する"""

        def waitUntilCreateFile(fileName):
            """ ffileNameが出来上がりるまで待つ"""
            count = 10
            while os.path.exists(fileName) == False:
                time.sleep(0.1)
                count -= 1
                if count < 0.0:
                    break
            return

        fileName = self.vtkFolder + os.sep + "modelSize"
        waitUntilCreateFile(fileName)
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        modelSize = " ".join(lines[0].split()[1:])
        self.modelDim = lines[1].split()[1]
        self.label_modelSize_gmsh.set_text(modelSize)

    #
    #  showVtkModel
    def showVtkModel(self):
        self.setShowVtkItems()
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  showAllCellActors
    def showAllCellActors(self):
        """ 全actorを表示させる。
        pick直後にGtkVtkRenderWindowInteractor.pyから呼び出される"""
        #pickedDataをselPointsに加算する
        #if len(self.pickedData) > 0:
        #    nodeNo = self.pickedData[2]
            #self.selPoints.append(nodeNo)
        #pointsのactorDictをクリアする
        self.deletePointsActors()
        # if "selPoints" in self.actorDict.keys():
        #     dummy = self.actorDict.pop("selPoints")
        # if "unselPoints" in self.actorDict.keys():
        #     dummy = self.actorDict.pop("unselPoints")
        #pointsのactorを作成
        #self.addPointCellsActors()
        #self.setShowVtkItems()

    #  deletePointsActors
    def deletePointsActors(self):
        """ selPoints, unselPointsActor, orgActorを削除する"""
        # if "selPoints" in self.actorDict.keys():        #red
        #     dummy = self.actorDict.pop("selPoints")
        # if "unselPoints" in self.actorDict.keys():      #white
        #     dummy = self.actorDict.pop("unselPoints")
        if self.originAxes != None:
            self.renderer.RemoveActor(self.originAxes)
            self.originAxes = None

    #
    #  addPickedActors
    #-------------------
    def addPickedActors(self, pickedData):
        """ GtkVtkRenderWindowInteractorから呼び出される。
        何もせず、戻る"""
        return
    
    #
    #  getPickedCellId
    #-------------------
    def getPickedCellId(self, pickedData):
        """ pickしたpointを選択表示する"""
        
        def clearMeshSizeToEntry():
            """ entry内のmeshSizeをクリアする"""
            meshSize = ""
            self.entry_localMeshSize_gmsh.set_text(meshSize)

        def setMeshSizeToEntry(meshSize):
            """ meshSizeをentryにセットする"""
            selItems = self.treeList_localMesh_gmsh.getSelectedItems()
            if len(selItems) == 0:
                #localMeshの表内が非選択状態の時、entryに書き込む
                self.entry_localMeshSize_gmsh.set_text(meshSize)

        cellId, actor, pointId, loc = pickedData
        if self.vtkSelectMode == "volume":
            volNo = self.getPickedVolume(cellId)
            if volNo < 0:
                return
            print("volumeNo", volNo, "was picked.")
            if volNo in self.selVolumes:
                self.selVolumes.remove(volNo)
                clearMeshSizeToEntry()
            else:
                localSize = self.getVolumeMaxLoaclMeshSize(volNo)
                setMeshSizeToEntry(localSize)
                self.selVolumes.append(volNo)
        #surface選択モード？
        elif self.vtkSelectMode == "surface":
            surNo = self.getPickedSurface(cellId)
            if surNo < 0:
                return
            print("surfaceNo", surNo, "was picked.")
            if surNo in self.selSurfaces:
                self.selSurfaces.remove(surNo)
                clearMeshSizeToEntry()
            else:
                localSize = self.getSurfaceAverageLocalMeshSize(surNo)
                setMeshSizeToEntry(localSize)
                self.selSurfaces.append(surNo)
        #edge選択モード？
        elif self.vtkSelectMode == "edge":
            edgeNo = self.getPickedEdge()
            if edgeNo < 0:
                return
            print("edgeNo", edgeNo, "was picked.")
            if edgeNo in self.selEdges:
                self.selEdges.remove(edgeNo)
            else:
                localSize = self.getEdgeAverageLocalMeshSize(edgeNo)
                setMeshSizeToEntry(localSize)
                self.selEdges.append(edgeNo)
        #point選択モード
        else:
            pointNos = self.getPointNoFromLocation(loc)
            if len(pointNos) == 0:
                return
            print("pointNo", pointNos, "was picked.")
            self.pickedData = [actor, cellId, pointId, loc]
            for pointNo in pointNos:
                if pointNo in self.selPoints:
                    dummy = self.selPoints.remove(pointNo)
                else:
                    localSize = self.getPointLocalMeshSize(pointNo)
                    setMeshSizeToEntry(localSize)
                    self.selPoints.append(pointNo)
        self.addPointCellsActors()
        self.setShowVtkItems()

    #  getPickedVolume
    def getPickedVolume(self, pickedCellId):
        """ pickしたcellが属するvolumeNoを取得する。属さないcellの場合、-1を返す"""
        pickedSurNo = self.getPickedSurface(pickedCellId)
        if pickedSurNo < 0:
            return pickedSurNo
        actor = self.actorDict["surfaces"]
        mapper = actor.GetMapper()
        polyData = mapper.GetInput()
        nCells = polyData.GetNumberOfCells()
        surCellData = polyData.GetCellData().GetArray("surNo")
        volCellData = polyData.GetCellData().GetArray("volumeNo")
        pickedVolNo = -1
        for i in range(nCells):
            surNo = surCellData.GetValue(i)
            if surNo == pickedSurNo:
                pickedVolNo = volCellData.GetValue(i)
                break
        return pickedVolNo

    #  getPickedSurface
    def getPickedSurface(self, pickedCellId):
        """ pickしたcellが属するsurfaceNoを取得する。属さないcellの場合、-1を返す"""
        surNoDict = self.surfaceDict["surNoDict"]
        pickedSurface = -1
        for surNo in surNoDict.keys():
            cellSets = set(surNoDict[surNo])
            if pickedCellId in cellSets:
                pickedSurface = surNo
                break
        return pickedSurface

    #  getPickedEdge
    def getPickedEdge(self):
        """ pickLoc（world座標）に最も近いedgeを探して返す"""
        #全edgeを取得
        #edgePointDict = self.edgeDict["edgePointDict"]
        #edges = list(edgePointDict.keys())
        #nodeDictを作成
        nodeDict = {}
        #allPoints = self.edgeDict["allPoints"]
        actor = self.actorDict["edges"]
        mapper = actor.GetMapper()
        polyData = mapper.GetInput()
        allPoints = polyData.GetPoints().GetData()
        nPoints = allPoints.GetSize()
        for i in range(0, nPoints, 3):
            nodes = []
            nodes.append(allPoints.GetValue(i))
            nodes.append(allPoints.GetValue(i+1))
            nodes.append(allPoints.GetValue(i+2))
            nodeDict[i//3] = nodes
        #pickしたworld座標を取得
        pickLoc = self.vtkObj.clickWorldPos
        #全edgeを取得
        edgePointDict = self.edgeDict["edgePointDict"]
        edges = list(edgePointDict.keys())
        #locに近接するedgeを取得
        closeEdges = []
        for edgeNo in edges:
            #edge内でpickLocに近い2個のnodeを取得
            lenNodes = []
            nodes = edgePointDict[edgeNo]
            for nodeNo in nodes:
                nodeLoc = nodeDict[nodeNo]
                l = geo.length(pickLoc, nodeLoc)
                lenNodes.append([l, nodeNo])
            lenNodes.sort()
            #line（2個のnode）とpickLocの距離算出
            l0, node0 = lenNodes[0]; l1, node1 = lenNodes[1]
            line = [nodeDict[node0], nodeDict[node1]]
            #pickLocがline上にあるか？
            if self.isPickOnLine(line, pickLoc) == True:
                l = geo.lenLineToPoint(line, pickLoc)
                closeEdges.append([l, edgeNo])
        #最も近接するedgeを取得
        closeEdges.sort()
        minl = closeEdges[0][0]
        closestEdgeNo = closeEdges[0][1]
        meshSize = self.entry_meshSize_gmsh.get_text()
        if minl > float(meshSize):
            #pickLocとedgeの距離(l)がmeshSize以上の場合、pickせず
            closestEdgeNo = -1
        return closestEdgeNo

    #  isPickOnLine
    def isPickOnLine(self, line, pickLoc):
        """ pickLocがonLine上にあるか？"""
        #line[0]基準で角度算出
        vec0 = geo.vector(line[0], line[1])
        vecP = geo.vector(line[0], pickLoc)
        cosAng = geo.vecsAngle(vec0, vecP)
        if cosAng < 0.0:
            return False
        #line[1]基準で角度算出
        vec1 = geo.vector(line[1], line[0])
        vecP = geo.vector(line[1], pickLoc)
        cosAng = geo.vecsAngle(vec1, vecP)
        if cosAng < -1.0:
            return False
        return True

    #  getPointNoFromLocation
    def getPointNoFromLocation(self, loc0):
        """ 座標が最も近いpointNoを取得"""
        #modelSizeからtolを取得
        words = self.label_modelSize_gmsh.get_text().split()
        maxModelSize = max(list(map(float, words)))
        tol = maxModelSize / 50.0
        #camera位置からtolを取得        
        camera = self.renderer.GetActiveCamera()
        camPos = camera.GetPosition()                
        viewAng = camera.GetViewAngle() * math.pi /180.0
        orgPt = camera.GetFocalPoint()      #原点
        l = geo.length(camPos, orgPt)      #視点-原点間長さ
        #world座標の幅（camera位置とviewAngleから算出）
        width = l * math.tan(viewAng/2.0) * 2.0
        #tolを決定
        if width < maxModelSize:
            tol = width / 50.0
        #全pointを調べる
        nPoints = self.allPoints.GetSize()
        pointIds = []
        for i in range(0, nPoints, 3):
            x = self.allPoints.GetValue(i)
            y = self.allPoints.GetValue(i+1)
            z = self.allPoints.GetValue(i+2)
            loc = [x,y,z]
            l = geo.length(loc, loc0)
            if l < tol:
                pointNo = i // 3
                pointIds.append(pointNo)
        actor = self.actorDict["points"]
        mapper = actor.GetMapper()
        polyData = mapper.GetInput()
        pointNoCells = polyData.GetCellData().GetArray("pointNo")
        pointNos = []
        for pointId in pointIds:
            pointNo = pointNoCells.GetValue(pointId)
            pointNos.append(pointNo)
        return pointNos

    #  getVolumeMaxLoaclMeshSize
    def getVolumeMaxLoaclMeshSize(self, volNo):
        """ volumeのmaxlocalMeshSizeを取得して返す"""
        createMesh_gmsh.CurrDir = self.currDir
        surReader = createMesh_gmsh.getSurfaceReader()
        edgeReader = createMesh_gmsh.getEdgeReader()
        pntReader = createMesh_gmsh.getPointReader()
        pntLocs = createMesh_gmsh.getPointLocsFromVtuFile(pntReader)
        #pointNosを取得
        actor = self.actorDict["points"]
        mapper = actor.GetMapper()
        polyData = mapper.GetInput()
        nCells = polyData.GetNumberOfCells()
        volCellData = polyData.GetCellData().GetArray("volumeNo")
        pointNos = []
        for i in range(nCells):
            vNo = volCellData.GetValue(i)
            if vNo == volNo:
                pointNo = i + 1     #gmshTagに変換
                pointNos.append(pointNo)
        #pointの設定値を取得
        pointIdDict = self.getDefaultLocalMeshSize(surReader, edgeReader, pntLocs)
        #maxの設定値を取得
        vals = []
        for pointNo in pointNos:
            defVal = pointIdDict[pointNo]
            vals.append(defVal)
        maxVal = str(max(vals))
        return maxVal

    #  getSurfaceAverageLocalMeshSize
    def getSurfaceAverageLocalMeshSize(self, surNo):
        """ surfaceの平均localMeshSizeを取得して返す"""
        createMesh_gmsh.CurrDir = self.currDir
        surReader = createMesh_gmsh.getSurfaceReader()
        edgeReader = createMesh_gmsh.getEdgeReader()
        pntReader = createMesh_gmsh.getPointReader()
        #surface内のpointNo(gmshTag)を取得
        pntLocs = createMesh_gmsh.getPointLocsFromVtuFile(pntReader)
        locs = createMesh_gmsh.getSurfaceLocsFromVtuFile([surNo], surReader)
        pointNos = createMesh_gmsh.getIncludePointsInSurface(pntLocs, locs)
        #pointの設定値を取得
        pointIdDict = self.getDefaultLocalMeshSize(surReader, edgeReader, pntLocs)
        #平均の設定値を取得
        vals = []
        for pNo in pointNos:
            val = pointIdDict[pNo]
            vals.append(val)
        aveSize = sum(vals) / len(vals)
        aveSize = pyFistr.float2strAuto(aveSize, prec=3)
        return aveSize

    #  getEdgeAverageLocalMeshSize(edgeNo)
    def getEdgeAverageLocalMeshSize(self, edgeNo):
        """ edgeの平均localMeshSizeを取得して返す"""
        createMesh_gmsh.CurrDir = self.currDir
        surReader = createMesh_gmsh.getSurfaceReader()
        edgeReader = createMesh_gmsh.getEdgeReader()
        pntReader = createMesh_gmsh.getPointReader()
        #edge内のpointNo(gmshTag)を取得
        pntLocs = createMesh_gmsh.getPointLocsFromVtuFile(pntReader)
        locs = createMesh_gmsh.getEdgeLocsFromVtuFile([edgeNo], edgeReader)
        pointNos = createMesh_gmsh.getIncludePointsInSurface(pntLocs, locs)
        #pointの設定値を取得
        pointIdDict = self.getDefaultLocalMeshSize(surReader, edgeReader, pntLocs)
        #平均の設定値を取得
        vals = []
        for pNo in pointNos:
            val = pointIdDict[pNo]
            vals.append(val)
        aveSize = sum(vals) / len(vals)
        aveSize = pyFistr.float2strAuto(aveSize, prec=3)
        return aveSize

    #  getPointLocalMeshSize
    def getPointLocalMeshSize(self, pointNo):
        """ pointのlocalMeshSizeを取得して返す"""
        pointNo += 1
        createMesh_gmsh.CurrDir = self.currDir
        surReader = createMesh_gmsh.getSurfaceReader()
        edgeReader = createMesh_gmsh.getEdgeReader()
        pntReader = createMesh_gmsh.getPointReader()
        #edge内のpointNo(gmshTag)を取得
        pntLocs = createMesh_gmsh.getPointLocsFromVtuFile(pntReader)
        #pointの設定値を取得
        pointIdDict = self.getDefaultLocalMeshSize(surReader, edgeReader, pntLocs)
        aveSize = pointIdDict[pointNo]
        aveSize = pyFistr.float2strAuto(aveSize, prec=3)
        return aveSize

    #  getDefaultLocalMeshSize
    def getDefaultLocalMeshSize(self, surReader, edgeReader, pntLocs):
        """ 設定済みのdefault,localのmeshSizeを取得して返す"""
        #defaultのmeshSizeを取得
        maxSizeStr = self.entry_meshSize_gmsh.get_text()
        maxSize = float(maxSizeStr)
        pointIdDict = createMesh_gmsh.setDefaultPointMeshSize(maxSize)
        #treeListで設定したsurfaceの設定値を取り込む
        localMeshSizes = self.treeList_localMesh_gmsh.getAllItemNames()
        for localMeshSize in localMeshSizes:
            localSize = localMeshSize[0]    #meshSize
            target = localMeshSize[1]       #対象（surface, edge or points）
            conts = localMeshSize[2]        #vtkの対象No（surNo, curveNo or pointNo）
            contNos = conts.split(",")
            contNos = list(map(int, contNos))
            if target == "surface":
                surNos = contNos
                locs = createMesh_gmsh.getSurfaceLocsFromVtuFile(surNos, surReader)
                setPointNos = createMesh_gmsh.getIncludePointsInSurface(pntLocs, locs)
                for pointNo in setPointNos:
                    pointIdDict[pointNo] = float(localSize)
            elif target == "edge":
                curNos = contNos
                locs = createMesh_gmsh.getEdgeLocsFromVtuFile(curNos, edgeReader)
                setPointNos = createMesh_gmsh.getIncludePointsInSurface(pntLocs, locs)
                for pointNo in setPointNos:
                    pointIdDict[pointNo] = float(localSize)
            else:
                setPointNos = contNos
                setPointNos = list(map(lambda x: x+1, setPointNos))
                for pointNo in setPointNos:
                    pointIdDict[pointNo] = float(localSize)
        return pointIdDict
        # vals = []
        # for pNo in pointNos:
        #     val = pointIdDict[pNo]
        #     vals.append(val)
        # aveSize = sum(vals) / len(vals)
        # aveSize = pyFistr.float2strAuto(aveSize, prec=3)
        # print("----", aveSize)

    #
    #  setLocalMeshAtPoints
    #-----------------------
    def setLocalMeshAtPoints(self):
        """ 選択部品にmeshSizeを適用させる。
        treeListに追加する"""
        #設定値を取得
        meshSize = self.entry_localMeshSize_gmsh.get_text()
        if self.radio_volume.get_active() == True:
            selType = "volume"
            vtkNos = self.selVolumes
        elif self.radio_surface.get_active() == True:
            selType = "surface"
            vtkNos = self.selSurfaces
        elif self.radio_edge.get_active() == True:
            selType = "edge"
            vtkNos = self.selEdges
        else:
            selType = "point"
            vtkNos = self.selPoints
        vtkConts = ",".join(map(str, vtkNos))
        gmshNos = map(lambda x: x+1, vtkNos)
        gmshConts = ",".join(map(str, gmshNos))
        #エラーチェック
        error = self.errorCheck_meshSizePickedCells(meshSize, vtkConts)
        if error != "OK":
            return
        #treeListに設定を追加
        item = [meshSize, selType, vtkConts, gmshConts]
        self.localMeshItemConts.append(item)
        items = self.localMeshItemConts
        self.treeList_localMesh_gmsh.setItems(items)
        #vtk画面修正
        self.pickedData = []
        idx = items.index(item)
        self.treeList_localMesh_gmsh.selectItemNos([idx])
        self.setButtonSensitive_localMesh()
        self.setBackgroundColor(1)      #editSelectColor
        self.vtkObj.Initialize()

    #
    #  errorCheck_meshSizePickedCells
    def errorCheck_meshSizePickedCells(self, meshSize, selPoints):
        """ meshSizeと選択pointsのエラーチェック"""
        flag = 0
        try:
            a = float(meshSize)
        except:
            flag = 1
        if flag == 1:
            title = _("エラー")
            mess = _("meshSizeが数値に変換できません。") + "\n"
            mess += "  " + _("再入力してください。")
            self.errDialog(title, mess)
            return "error"
        if len(selPoints) == 0:
            title = _("エラー")
            mess = _("モデル内の対象部品が選択されていません。") + "\n"
            mess += "  " + _("マウスで部品（surface, edge, point）を選択してください。")
            self.errDialog(title, mess)
            return "error"
        return "OK"

    #
    #  deleteLocalMesh
    #------------------
    def deleteLocalMesh(self):
        """ treeList内の選択行を削除"""
        delNos = self.treeList_localMesh_gmsh.getSelectedItemNos()
        if len(delNos) == 0:
            pass
            return
        newItems = []
        for i in range(len(self.localMeshItemConts)):
            if i in delNos:
                pass
            else:
                newItems.append(self.localMeshItemConts[i])
        self.treeList_localMesh_gmsh.setItems(newItems)
        self.localMeshItemConts = newItems
        self.clearSelPoints()

    #
    #  selectRowOfLocalMesh
    #-----------------------
    def selectRowOfLocalMesh(self):
        """ 設定済のlocalMeshのpointsをvtk上にbrownで表示する"""
        GLib.timeout_add(100, self.selectRowOfLocalMesh_run)

    def selectRowOfLocalMesh_run(self):
        self.setButtonSensitive_localMesh()
        #treeListから選択pointsNoを取得
        selItems = self.treeList_localMesh_gmsh.getSelectedItemNames()
        if len(selItems) == 0:
            return
        item = selItems[0]
        meshSize = item[0]
        vtkNos = item[2].split(",")
        vtkNos = list(map(int, vtkNos))
        self.entry_localMeshSize_gmsh.set_text(meshSize)
        selType = item[1]
        if selType == "volume":
            self.radio_volume.set_active(True)
            self.selVolumes = vtkNos
        elif selType == "surface":
            self.radio_surface.set_active(True)
            self.selSurfaces = vtkNos
        elif selType == "edge":
            self.radio_edge.set_active(True)
            self.selEdges = vtkNos
        else:
            self.radio_point.set_active(True)
            self.selPoints = vtkNos
        self.setBackgroundColor(1)      #editSelect用
        self.deletePointsActors()
        self.addPointCellsActors()
        self.setShowVtkItems()
        self.vtkObj.Initialize()

    #
    #  defineNgrp
    #-------------
    def defineNgrp(self):
        """ 選択対象をngrpに追加する"""
        #errorCheck
        if self.radio_volume.get_active() == True:
            title = _("エラー")
            mess = _("volumeが選択されています。") + "\n"
            mess += _("surface, edge, pointのいずれかを選択してください。") + "\n"
            mess += _("  volumeは、設定できません。")
            self.errDialog(title, mess)
            return
        #ngrpを取得
        ngrpName = self.entry_ngrpName_gmsh.get_text()
        if self.radio_surface.get_active() == True:
            selItem = "surface"
            selNos = self.selSurfaces
        elif self.radio_edge.get_active() == True:
            selItem = "edge"
            selNos = self.selEdges
        elif self.radio_point.get_active() == True:
            selItem = "point"
            selNos = self.selPoints
        else:
            selItem = ""
            selNos = []
        gmshNos = list(map(lambda x: x+1, selNos))
        #エラーチェック
        error = self.errorCheck_groupNamePickedCells(ngrpName, selNos)
        if error != "OK":
            return
        #treeListに内容を設定
        selNos = ",".join(list(map(str, selNos)))
        gmshNos = ",".join(list(map(str, gmshNos)))
        item = [ngrpName, selItem, selNos, gmshNos]
        items = self.treeList_ngrp_gmsh.getAllItemNames()
        items.append(item)
        self.treeList_ngrp_gmsh.setItems(items)
        selNos = [len(items)-1]
        self.treeList_ngrp_gmsh.selectItemNos(selNos)
        self.setButtonSensitive_ngrpSgrp()
        self.setBackgroundColor(1)
        self.vtkObj.Initialize()
        
    #
    #  errorCheck_groupNamePickedCells
    def errorCheck_groupNamePickedCells(self, grpName, selNos):
        """ group名とpickした部品のエラーチェック"""
        a = grpName.split()
        if len(a) == 0:
            title = _("エラー")
            mess = _("group名が入力されていません。") + "\n"
            mess += "  " + _("再入力してください。")
            self.errDialog(title, mess)
            return "error"
        if len(selNos) == 0:
            title = _("エラー")
            mess = _("モデル内の対象部品が選択されていません。") + "\n"
            mess += "  " + _("マウスで部品（surface, edge, point）を選択してください。")
            self.errDialog(title, mess)
            return "error"
        return "OK"

    #
    #  overSetDefineNgrp
    #--------------------
    def overSetDefineNgrp(self):
        """ 設定済のngrpの選択行に上書きする"""
        ngrpName = self.entry_ngrpName_gmsh.get_text()
        if self.radio_surface.get_active() == True:
            selItem = "surface"
            vtkNos = self.selSurfaces
        elif self.radio_edge.get_active() == True:
            selItem = "edge"
            vtkNos = self.selEdges
        elif self.radio_point.get_active() == True:
            selItem = "point"
            vtkNos = self.selPoints
        else:
            selItem = ""
            vtkNos = []
        gmshNos = list(map(lambda x: x+1, vtkNos))
        #エラーチェック
        error = self.errorCheck_groupNamePickedCells(ngrpName, vtkNos)
        if error != "OK":
            return
        #treeListのItemを作成
        vtkNos = ",".join(list(map(str, vtkNos)))
        gmshNos = ",".join(list(map(str, gmshNos)))
        item = [ngrpName, selItem, vtkNos, gmshNos]
        #itemを上書き
        selNos = self.treeList_ngrp_gmsh.getSelectedItemNos()
        #  selNosのエラーチェック
        error = self.errorCheck_groupNamePickedCells(ngrpName, selNos)
        if error != "OK":
            return
        idx = selNos[0]
        newItem = [ngrpName, selItem, vtkNos, gmshNos]
        allItems = self.treeList_ngrp_gmsh.getAllItemNames()
        allItems[idx] = newItem
        self.treeList_ngrp_gmsh.setItems(allItems)
        self.treeList_ngrp_gmsh.selectItemNos([idx])

    #
    #  deleteDefineNgrp
    #-------------------
    def deleteDefineNgrp(self):
        """ 選択行を削除する"""
        delNos = self.treeList_ngrp_gmsh.getSelectedItemNos()
        if len(delNos) == 0:
            return
        allItems = self.treeList_ngrp_gmsh.getAllItemNames()
        newItems = []
        for i in range(len(allItems)):
            if i in delNos:
                pass
            else:
                newItems.append(allItems[i])
        self.treeList_ngrp_gmsh.setItems(newItems)
        self.clearSelPoints()

    #
    #  selectNgrpRow
    #----------------
    def selectNgrpRow(self):
        """ treeList_NGRPの行を選択したときのevent"""
        GLib.timeout_add(100, self.selectNgrpRow_run)

    def selectNgrpRow_run(self):
        #SGRP側をunselect
        self.treeList_sgrp_gmsh.unselectAllItems()
        #treeListからname、nameTypeを取得
        selItems = self.treeList_ngrp_gmsh.getSelectedItemNames()
        if len(selItems) <= 0:
            return
        item = selItems[0]
        name = item[0]
        nameType = item[1]
        vtkConts = item[2].split(",")
        vtkNos = list(map(int, vtkConts))
        #画面に設定
        self.entry_ngrpName_gmsh.set_text(name)
        if nameType == "surface":
            self.radio_surface.set_active(True)
            self.selSurfaces = vtkNos
        elif nameType == "edge":
            self.radio_edge.set_active(True)
            self.selEdges = vtkNos
        else:
            self.radio_point.set_active(True)
            self.selPoints = vtkNos
        #vtk表示
        self.setBackgroundColor(1)  #editSelectColor
        self.deletePointsActors()
        self.addPointCellsActors()
        self.setShowVtkItems()
        self.vtkObj.Initialize()
        self.setButtonSensitive_ngrpSgrp()

    #
    #  defineSgrp
    #--------------
    def defineSgrp(self):
        """ 選択対象をsgrpに追加する"""
        #errorCheck
        if self.radio_volume.get_active() == True:
            title = _("エラー")
            mess = _("volumeが選択されています。") + "\n"
            mess += _("surface, edge, pointのいずれかを選択してください。") + "\n"
            mess += _("  volumeは、設定できません。")
            self.errDialog(title, mess)
            return
        #sgrpを取得
        sgrpName = self.entry_sgrpName_gmsh.get_text()
        if self.radio_surface.get_active() == True:
            selItem = "surface"
            selNos = self.selSurfaces
        elif self.radio_edge.get_active() == True:
            selItem = "edge"
            selNos = self.selEdges
        elif self.radio_point.get_active() == True:
            selItem = "point"
            selNos = self.selPoints
        else:
            selItem = ""
            selNos = []
        gmshNos = list(map(lambda x: x+1, selNos))
        #エラーチェック
        error = self.errorCheck_groupNamePickedCells(sgrpName, selNos)
        if error != "OK":
            return
        #treeListに内容を設定
        selNos = ",".join(list(map(str, selNos)))
        gmshNos = ",".join(list(map(str, gmshNos)))
        item = [sgrpName, selItem, selNos, gmshNos]
        items = self.treeList_sgrp_gmsh.getAllItemNames()
        items.append(item)
        self.treeList_sgrp_gmsh.setItems(items)
        selNos = [len(items)-1]
        self.treeList_sgrp_gmsh.selectItemNos(selNos)
        self.setButtonSensitive_ngrpSgrp()
        self.setBackgroundColor(1)
        self.vtkObj.Initialize()
        
    #
    #  overSetDefineSgrp
    #--------------------
    def overSetDefineSgrp(self):
        """ 設定済のsgrpの選択行に上書きする"""
        sgrpName = self.entry_sgrpName_gmsh.get_text()
        if self.radio_surface.get_active() == True:
            selItem = "surface"
            vtkNos = self.selSurfaces
        elif self.radio_edge.get_active() == True:
            selItem = "edge"
            vtkNos = self.selEdges
        elif self.radio_point.get_active() == True:
            selItem = "point"
            vtkNos = self.selPoints
        else:
            selItem = ""
            vtkNos = []
        gmshNos = list(map(lambda x: x+1, vtkNos))
        #エラーチェック
        error = self.errorCheck_groupNamePickedCells(sgrpName, vtkNos)
        if error != "OK":
            return
        #treeListのItemを作成
        vtkNos = ",".join(list(map(str, vtkNos)))
        gmshNos = ",".join(list(map(str, gmshNos)))
        item = [sgrpName, selItem, vtkNos, gmshNos]
        #itemを上書き
        selNos = self.treeList_sgrp_gmsh.getSelectedItemNos()
        #  selNosのエラーチェック
        error = self.errorCheck_groupNamePickedCells(sgrpName, selNos)
        if error != "OK":
            return
        idx = selNos[0]
        newItem = [sgrpName, selItem, vtkNos, gmshNos]
        allItems = self.treeList_sgrp_gmsh.getAllItemNames()
        allItems[idx] = newItem
        self.treeList_sgrp_gmsh.setItems(allItems)
        self.treeList_sgrp_gmsh.selectItemNos([idx])

    #
    #  delDefineSgrp
    #----------------
    def delDefineSgrp(self):
        """ 選択行を削除する"""
        delNos = self.treeList_sgrp_gmsh.getSelectedItemNos()
        if len(delNos) == 0:
            return
        allItems = self.treeList_sgrp_gmsh.getAllItemNames()
        newItems = []
        for i in range(len(allItems)):
            if i in delNos:
                pass
            else:
                newItems.append(allItems[i])
        self.treeList_sgrp_gmsh.setItems(newItems)
        self.clearSelPoints()

    #
    #  selectSgrpRow
    #----------------
    def selectSgrpRow(self):
        """ treeList_NGRPの行を選択したときのevent"""
        GLib.timeout_add(100, self.selectSgrpRow_run)

    #  selectSgrpRow_run
    def selectSgrpRow_run(self):
        self.treeList_ngrp_gmsh.unselectAllItems()
        #treeListからname、nameTypeを取得
        selItems = self.treeList_sgrp_gmsh.getSelectedItemNames()
        if len(selItems) <= 0:
            return
        item = selItems[0]
        name = item[0]
        nameType = item[1]
        vtkConts = item[2].split(",")
        vtkNos = list(map(int, vtkConts))
        #画面に設定
        self.entry_sgrpName_gmsh.set_text(name)
        if nameType == "surface":
            self.radio_surface.set_active(True)
            self.selSurfaces = vtkNos
        elif nameType == "edge":
            self.radio_edge.set_active(True)
            self.selEdges = vtkNos
        else:
            self.radio_point.set_active(True)
            self.selPoints = vtkNos
        #vtk表示
        self.setBackgroundColor(1)      #editSelectColor
        self.deletePointsActors()
        self.addPointCellsActors()
        self.setShowVtkItems()
        self.vtkObj.Initialize()
        self.setButtonSensitive_ngrpSgrp()

    #
    #  defineEgrp
    #-------------
    def defineEgrp(self):
        """ 選択対象をEGRPに追加する"""
        #errorCheck
        mess = ""
        if self.modelDim == "3D":
            if self.radio_volume.get_active() == False:
                mess = _("volume が選択されていません。") + "\n"
                mess += _("3Dモデルの為、volumeを選択してください。")
        elif self.modelDim == "2D":
            if self.radio_surface.get_active() == False:
                mess = _("surface が選択されていません。") + "\n"
                mess += _("2Dモデルの為、surfaceを選択してください。")
        elif self.modelDim == "1D":
            if self.radio_edge.get_active() == False:
                mess = _("edge が選択されていません。") + "\n"
                mess +=("1Dモデルの為、edgeを選択してください。")
        if mess != "":
            self.errDialog(_("エラー"), mess)
            return
        #sgrpを取得
        egrpName = self.entry_egrpName_gmsh.get_text()
        if self.modelDim == "3D":
            selItem = "volume"
            selNos = self.selVolumes
        elif self.modelDim == "2D":
            selItem = "surface"
            selNos = self.selSurfaces
        elif self.modelDim == "1D":
            selItem = "edge"
            selNos = self.selEdges
        else:
            selItem = ""
            selNos = []
        gmshNos = list(map(lambda x: x+1, selNos))
        #errorCheck
        error = self.errorCheck_egrpNamePickedCells(egrpName, selNos)
        if error != "OK":
            return
        #treeList内に内容を設定
        selNos = ",".join(list(map(str, selNos)))
        gmshNos = ",".join(list(map(str, gmshNos)))
        item = [egrpName, selItem, selNos, gmshNos]
        items = self.treeList_egrp_gmsh.getAllItemNames()
        items.append(item)
        self.treeList_egrp_gmsh.setItems(items)
        selNos = [len(items)-1]
        self.treeList_egrp_gmsh.selectItemNos(selNos)
        self.setButtonSensitive_ngrpSgrp()
        self.setBackgroundColor(1)
        self.vtkObj.Initialize()

    #
    #  overSetDefineEgrp
    #--------------------
    def overSetDefineEgrp(self):
        """ 設定済みのegrpの選択行に上書きする"""
        egrpName = self.entry_egrpName_gmsh.get_text()
        if self.modelDim == "3D":
            selItem = "volume"
            vtkNos = self.selVolumes
        elif self.modelDim == "2D":
            selItem = "surface"
            vtkNos = self.selSurfaces
        elif self.modelDim == "1D":
            selItem = "edge"
            vtkNos = self.selEdges
        else:
            selItem = ""
            vtkNos = []
        gmshNos = list(map(lambda x: x+1, vtkNos))
        #errorCheck
        error = self.errorCheck_egrpNamePickedCells(egrpName, vtkNos)
        if error != "OK":
            return
        #treeListのitemを作成
        vtkNos = ",".join(list(map(str, vtkNos)))
        gmshNos = ",".join(list(map(str, gmshNos)))
        selNos = self.treeList_egrp_gmsh.getSelectedItemNos()
        error = self.errorCheck_egrpNamePickedCells(egrpName, selNos)
        if error != "OK":
            return
        idx = selNos[0]
        newItem = [egrpName, selItem, vtkNos, gmshNos]
        allItems = self.treeList_egrp_gmsh.getAllItemNames()
        allItems[idx] = newItem
        self.treeList_egrp_gmsh.setItems(allItems)
        self.treeList_egrp_gmsh.selectItemNos([idx])

    #
    #  delDefineEgrp
    #----------------
    def delDefineEgrp(self):
        """ 選択業を削除する"""
        delNos = self.treeList_egrp_gmsh.getSelectedItemNos()
        if len(delNos) == 0:
            return
        allItems = self.treeList_egrp_gmsh.getAllItemNames()
        newItems = []
        for i in range(len(allItems)):
            if i in delNos:
                pass
            else:
                newItems.append(allItems[i])
        self.treeList_egrp_gmsh.setItems(newItems)
        self.clearSelPoints()

    #
    #  selectEgrpRow
    #----------------
    def selectEgrpRow(self):
        """ treeList_egrp内の行を選択したときのevent"""
        GLib.timeout_add(100, self.selectEgrpRow_run)

    #  selectEgrpRow_run
    def selectEgrpRow_run(self):
        #選択行を取得
        selItems = self.treeList_egrp_gmsh.getSelectedItemNames()
        if len(selItems) <= 0:
            return
        #選択行の内容を取得
        item = selItems[0]
        name = item[0]
        nameType = item[1]
        vtkConts = item[2].split(",")
        vtkNos = list(map(int, vtkConts))
        #画面に設定
        self.entry_egrpName_gmsh.set_text(name)
        if nameType == "volume":
            self.radio_volume.set_active(True)
            self.selVolumes = vtkNos
        elif nameType == "surface":
            self.radio_surface.set_active(True)
            self.selSurfaces = vtkNos
        elif nameType == "edge":
            self.radio_edge.set_active(True)
            self.selEdges = vtkNos
        #vtk表示
        self.setBackgroundColor(1)      #editSelectColor
        self.deletePointsActors()
        self.addPointCellsActors()
        self.setShowVtkItems()
        self.vtkObj.Initialize()
        self.setButtonSensitive_ngrpSgrp()

    #    
    #  errorCheck_egrpNamePickedCells
    def errorCheck_egrpNamePickedCells(self, egrpName, selNos):
        if len(egrpName.split()) == 0:
            title = _("エラー")
            mess = _("group名が入力されていません。") + "\n"
            mess += "  " + _("再入力してください。")
            self.errDialog(title, mess)
            return "error"
        if len(selNos) == 0:
            title = _("エラー")
            mess = _("モデル内の対象部品が選択されていません。") + "\n"
            mess += "  " + _("マウスで部品（volume, surface, edge）を選択してください。")
            self.errDialog(title, mess)
            return "error"
        return "OK"


    #
    #  changePage
    #-------------
    def changePage(self, pageNum):
        """ notebookのpage変更した時のevent"""
        mode = "newSelect"
        if pageNum == 0:
            #基本設定
            self.setBasicConditions()
            mode = "showParts"
        elif pageNum == 1:
            #localMeshSize
            self.setLocalMeshSize()
            mode = "newSelect"
        elif pageNum == 2:
            #group定義
            self.setDefineGroup()
            mode = "newSelect"
        elif pageNum == 3:
            #EGRP定義
            page = self.note_createMesh_gmsh.get_current_page()
            self.setDefineEgrp()
            page = self.note_createMesh_gmsh.get_current_page()
            mode = "newSelect"
        self.setModeFromPage(pageNum, 0)

    #
    #  setBasicConditions
    def setBasicConditions(self):
        """ 基本設定画面の設定"""
        self.radio_point.set_active(True)
        self.showPickedMode = "no"
        self.clearSelPoints()

    #
    #  setLocalMeshSize
    def setLocalMeshSize(self):
        """ localMeshSizeの設定"""
        self.setDefineGroup()

    #
    #  setDefineGroup
    def setDefineGroup(self):
        """ group定義画面の設定"""
        #選択対象を設定
        if self.modelDim == "3D" or self.modelDim == "2D":
            self.radio_surface.set_active(True)
        else:
            self.radio_point.set_active(True)
        #pick可能に設定
        self.showPickedMode = "yes"
        self.clearSelPoints()

    #
    #  setDefineEgrp
    def setDefineEgrp(self):
        #選択対象を設定
        if self.modelDim == "3D":
            self.radio_volume.set_active(True)
        elif self.modelDim == "2D":
            self.radio_surface.set_active(True)
        elif self.modelDim == "1D":
            self.radio_edge.set_active(True)
        else:
            self.radio_point.set_active(True)
        #pick可能に設定
        self.showPickedMode = "yes"
        self.clearSelPoints()


    #
    #  overSetLocalMesh
    #-------------------
    def overSetLocalMesh(self):
        """ 設定を選択行に上書きする"""
        #設定値を取得
        meshSize = self.entry_localMeshSize_gmsh.get_text()
        if self.radio_volume.get_active() == True:
            selType = "volume"
            vtkNos = self.selVolumes
        elif self.radio_surface.get_active() == True:
            selType = "surface"
            vtkNos = self.selSurfaces
        elif self.radio_edge.get_active() == True:
            selType = "edge"
            vtkNos = self.selEdges
        else:
            selType = "point"
            vtkNos = self.selPoints
        #エラーチェック
        error = self.errorCheck_meshSizePickedCells(meshSize, vtkNos)
        if error != "OK":
            return
        vtkConts = ",".join(map(str, vtkNos))
        gmshNos = map(lambda x: x+1, vtkNos)
        gmshConts = ",".join(map(str, gmshNos))
        selNos = self.treeList_localMesh_gmsh.getSelectedItemNos()
        if len(selNos) == 0:
            title = _("エラー")
            mess = _("表内の対象が選択されていません。") + "\n"
            mess += _("  上書き対象の行を選択してください。")
            self.errDialog(title, mess)
            return
        idx = selNos[0]
        newItem = [meshSize, selType, vtkConts, gmshConts]
        self.localMeshItemConts[idx] = newItem
        items = self.localMeshItemConts
        self.treeList_localMesh_gmsh.setItems(items)
        self.treeList_localMesh_gmsh.selectItemNos([idx])

    #
    #  fitToWindow
    #--------------
    def fitToWindow(self):
        """ 表示をvtk画面にfitさせる"""
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromXP
    #------------
    def viewFromXP(self):
        """ X軸視（Y-Z面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (-(Xcenter+maxLeng*2.0), Ycenter, Zcenter)
        #X軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの焦点を中心に合わせる
        camera.SetFocalPoint(Xcenter, Ycenter, Zcenter)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 0, 1)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()
        
    #
    #  viewFromXM
    #-------------
    def viewFromXM(self):
        """ -X軸視（Y-Z面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = ((Xcenter+maxLeng*2.0), Ycenter, Zcenter)
        #X軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの焦点を中心に合わせる
        camera.SetFocalPoint(Xcenter, Ycenter, Zcenter)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 0, 1)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromYP
    #------------
    def viewFromYP(self):
        """ Y軸視（Z-X面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, -(Ycenter+maxLeng*2.0), Zcenter)
        #Y軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの焦点を中心に合わせる
        camera.SetFocalPoint(Xcenter, Ycenter, Zcenter)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 0, 1)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromYM
    #-------------
    def viewFromYM(self):
        """ -Y軸視（Z-X面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, (Ycenter+maxLeng*2.0), Zcenter)
        #Y軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの焦点を中心に合わせる
        camera.SetFocalPoint(Xcenter, Ycenter, Zcenter)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 0, 1)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromZP
    #------------
    def viewFromZP(self):
        """ Z軸視（X-Y面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter, -(Zcenter+maxLeng*2.0))
        #Z軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの焦点を中心に合わせる
        camera.SetFocalPoint(Xcenter, Ycenter, Zcenter)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 1, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromZM
    #-------------
    def viewFromZM(self):
        """ -Z軸視（X-Y面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        if len(bounds) == 0:
            return
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter, (Zcenter+maxLeng*2.0))
        #Z軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの焦点を中心に合わせる
        camera.SetFocalPoint(Xcenter, Ycenter, Zcenter)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 1, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  setReverseCamera
    #-------------------
    def setReverseCamera(self):
        """ camera位置を反転（左右反転）させる"""
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        center = (Xcenter, Ycenter, Zcenter)
        camera = self.renderer.GetActiveCamera()
        pos = camera.GetPosition()
        relPos = (pos[0]-center[0], pos[1]-center[1], pos[2]-center[2])
        newRelPos = list(map(lambda x: -x, relPos))
        newPos = (newRelPos[0]+center[0], newRelPos[1]+center[1], newRelPos[2]+center[2])
        camera.SetPosition(*newPos)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  rolling90
    #------------
    def rolling90(self):
        """ 視点を右回転（90°）させる。"""
        camera = self.renderer.GetActiveCamera()
        camera.Roll(-90.0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  showAllParts
    #---------------
    def showAllParts(self):
        """ 全volumeを表示する"""
        self.hideVolumes = []
        self.hideSurfaces = []
        self.hideEdges = []
        self.hidePoints = []
        self.createVtkActorDict()
        self.addPointCellsActors()
        self.setShowVtkItems()
        self.vtkObj.Initialize()

    #
    #  hideSelParts
    #----------------
    def hideSelParts(self):
        """ 選択したvolumeを非表示設定する"""
        #volume?
        if self.radio_volume.get_active() == True:
            self.hideVolumes += self.selVolumes[:]
            self.selVolumes = []
        #surface?
        elif self.radio_surface.get_active() == True:
            self.hideSurfaces += self.selSurfaces[:]
            self.selSurfaces = []
        #edge?
        elif self.radio_edge.get_active() == True:
            self.hideEdges += self.selEdges[:]
            self.selEdges = []
        #point?
        else:
            self.hidePoints += self.selPoints[:]
            self.selPoints = []
        self.createVtkActorDict()
        self.addPointCellsActors()
        self.setShowVtkItems()
        self.vtkObj.Initialize()

    #
    #  newReloadVtkModel
    #--------------------
    def newReloadVtkModel(self):
        """ vtkFileを再作成し、表示する。"""
        cadName = self.entry_fileName_gmsh.get_text()
        cadFile = self.currDir + os.sep + cadName
        print(_("vtkDataFileを再作成し、vtkModelを再表示します。"))
        #vtkFile作成
        stat = self.createVtkSourceFromCad_runGmsh(cadFile, self.currDir)
        if stat != "OK":
            return
        #vtk表示
        self.createVtkActorDict()
        self.addPointCellsActors()
        self.showVtkModel()
        self.viewFromZM()

    #
    #  reloadVtkModel
    #-----------------
    def reloadVtkModel(self):
        """ vtkを再表示。vtkFileが存在しない場合は、再作成する。"""
        vtkDir = self.currDir + os.sep + "vtkCadData"
        edgesVtk = vtkDir + os.sep + "edges.vtu"
        #edgesVtk = vtkDir + os.sep + "edges.vtk"
        pointsVtk = vtkDir + os.sep + "points.vtu"
        verFile = vtkDir + os.sep + "version_" + createVtk.Version
        reloadFlag = 0
        if os.path.exists(edgesVtk) == True and os.path.exists(pointsVtk) == True:
            if os.path.exists(verFile) == True:
                reloadFlag = 1
                #vtkを表示
                print(_("vtkCadDataを再読込します。"))
                self.createVtkActorDict()
                self.addPointCellsActors()
                self.viewFromZM()
        if reloadFlag == 0:
            #vtkを再作成
            self.newReloadVtkModel()

    #
    #  selectVolume
    #---------------
    def selectVolume(self):
        """ volume選択モード"""
        if self.radio_volume.get_active() == True:
            self.selPoints = []
            self.selEdges = []
            self.selSurfaces = []
            self.selVolumes = []
            self.vtkSelectMode = "volume"
            self.deletePointsActors()
            self.addPointCellsActors()
            self.setShowVtkItems()
            self.vtkObj.Initialize()

    #
    #  selectSurface
    #----------------
    def selectSurface(self):
        """ surface選択モード"""
        if self.radio_surface.get_active() == True:
            self.selPoints = []
            self.selEdges = []
            self.selSurfaces = []
            self.selVolumes = []
            self.vtkSelectMode = "surface"
            self.deletePointsActors()
            self.addPointCellsActors()
            self.setShowVtkItems()
            self.vtkObj.Initialize()

    #
    #  selectEdge
    #-------------
    def selectEdge(self):
        """ edge選択モード"""
        if self.radio_edge.get_active() == True:
            self.selPoints = []
            self.selEdges = []
            self.selSurfaces = []
            self.selVolumes = []
            self.vtkSelectMode = "edge"
            self.deletePointsActors()
            self.addPointCellsActors()
            self.setShowVtkItems()
            self.vtkObj.Initialize()

    #
    #  selectPoint
    #--------------
    def selectPoint(self):
        """ point選択モード"""
        if self.radio_point.get_active() == True:
            self.selPoints = []
            self.selEdges = []
            self.selSurfaces = []
            self.selVolumes = []
            self.vtkSelectMode = "point"
            self.deletePointsActors()
            self.addPointCellsActors()
            self.setShowVtkItems()
            self.vtkObj.Initialize()

    #
    #  clearSelPoints
    #-----------------
    def clearSelPoints(self):
        """ 選択pointsをクリアする。"""
        self.selPoints = []
        self.selSurfaces = []
        self.selEdges = []
        self.selVolumes = []
        self.entry_localMeshSize_gmsh.set_text("")
        self.entry_ngrpName_gmsh.set_text("")
        self.entry_sgrpName_gmsh.set_text("")
        self.entry_egrpName_gmsh.set_text("")
        self.treeList_localMesh_gmsh.unselectAllItems()
        self.treeList_ngrp_gmsh.unselectAllItems()
        self.treeList_sgrp_gmsh.unselectAllItems()
        self.treeList_egrp_gmsh.unselectAllItems()
        self.setBackgroundColor(0)
        self.deletePointsActors()
        self.addPointCellsActors()
        self.setShowVtkItems()
        self.vtkObj.Initialize()
        self.setButtonSensitive_localMesh()
        self.setButtonSensitive_ngrpSgrp()

    #
    #  showOrgAxis
    #----------------
    def showOrgAxis(self):
        """ 原点を表示する"""
        self.deletePointsActors()
        self.addPointCellsActors()
        self.setShowVtkItems()
        self.vtkObj.Initialize()

    #
    #  setShowOpacity
    #-----------------
    def setShowOpacity(self):
        """ 透明化の設定"""
        self.showOrgAxis()

    #---- end vtk ---------------------


    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        """ objectの定義"""
        self.paned_win = self.builder.get_object("paned_win")
        self.note_createMesh_gmsh = self.builder.get_object("note_createMesh_gmsh")
        #----- 基本設定 -------------
        self.entry_fileName_gmsh = self.builder.get_object("entry_fileName_gmsh")
        self.entry_meshSize_gmsh = self.builder.get_object("entry_meshSize_gmsh")
        self.entry_localMeshSize_gmsh = self.builder.get_object("entry_localMeshSize_gmsh")
        self.entry_nThread_gmsh = self.builder.get_object("entry_nThread_gmsh")
        self.label_modelSize_gmsh = self.builder.get_object("label_modelSize_gmsh")
        self.label_localMeshSize_gmsh = self.builder.get_object("label_localMeshSize_gmsh")
        self.check_showOrgAxis_gmsh = self.builder.get_object("check_showOrgAxis_gmsh")
        self.check_showSurface_gmsh = self.builder.get_object("check_showSurface_gmsh")
        self.check_opacity_gmsh = self.builder.get_object("check_opacity_gmsh")
        #self.check_setLocalMesh_gmsh = self.builder.get_object("check_setLocalMesh_gmsh")
        self.check_allHexa_gmsh = self.builder.get_object("check_allHexa_gmsh")
        self.check_reformHex_gmsh = self.builder.get_object("check_reformHex_gmsh")
        self.check_2ndOrder_gmsh = self.builder.get_object("check_2ndOrder_gmsh")
        self.check_thread_gmsh = self.builder.get_object("check_thread_gmsh")
        self.check_addMesh_gmsh = self.builder.get_object("check_addMesh_gmsh")
        self.check_keepCond_gmsh = self.builder.get_object("check_keepCond_gmsh")
        self.label_nThread_gmsh = self.builder.get_object("label_nThread_gmsh")
        self.box_vtk = self.builder.get_object("box_vtk")
        self.button_setLocalMesh_gmsh = self.builder.get_object("button_setLocalMesh_gmsh")
        self.button_overSetLocalMesh_gmsh = self.builder.get_object("button_overSetLocalMesh_gmsh")
        self.button_delLocalMesh_gmsh = self.builder.get_object("button_delLocalMesh_gmsh")
        self.button_defineNgrp_gmsh = self.builder.get_object("button_defineNgrp_gmsh")
        self.button_overSetDefineNgrp_gmsh = self.builder.get_object("button_overSetDefineNgrp_gmsh")
        self.button_delDefineNgrp_gmsh = self.builder.get_object("button_delDefineNgrp_gmsh")
        self.button_defineSgrp_gmsh = self.builder.get_object("button_defineSgrp_gmsh")
        self.button_overSetDefineSgrp_gmsh = self.builder.get_object("button_overSetDefineSgrp_gmsh")
        self.button_delDefineSgrp_gmsh = self.builder.get_object("button_delDefineSgrp_gmsh")
        treeList_localMesh_gmsh = self.builder.get_object("treeList_localMesh_gmsh")
        self.treeList_localMesh_gmsh = GtkParts.treeListCols(treeList_localMesh_gmsh)
        self.label_selectCadFile_gmsh = self.builder.get_object("label_selectCadFile_gmsh")
        self.label_gmshConts = self.builder.get_object("label_gmshConts")
        self.label_afterMesh_gmsh = self.builder.get_object("label_afterMesh_gmsh")
        self.label_selectMode = self.builder.get_object("label_selectMode")
        self.label_vtkComm = self.builder.get_object("label_vtkComm")
        self.radio_volume = self.builder.get_object("radio_volume")
        self.radio_surface = self.builder.get_object("radio_surface")
        self.radio_edge = self.builder.get_object("radio_edge")
        self.radio_point = self.builder.get_object("radio_point")
        #-------group定義 -------------------
        treeList_ngrp_gmsh = self.builder.get_object("treeList_ngrp_gmsh")
        self.treeList_ngrp_gmsh = GtkParts.treeListCols(treeList_ngrp_gmsh)
        treeList_sgrp_gmsh = self.builder.get_object("treeList_sgrp_gmsh")
        self.treeList_sgrp_gmsh = GtkParts.treeListCols(treeList_sgrp_gmsh)
        self.entry_ngrpName_gmsh = self.builder.get_object("entry_ngrpName_gmsh")
        self.entry_sgrpName_gmsh = self.builder.get_object("entry_sgrpName_gmsh")
        #------egrp定義 ------------------------
        treeList_egrp_gmsh = self.builder.get_object("treeList_egrp_gmsh")
        self.treeList_egrp_gmsh = GtkParts.treeListCols(treeList_egrp_gmsh)
        self.entry_egrpName_gmsh = self.builder.get_object("entry_egrpName_gmsh")
        self.button_defineEgrp_gmsh = self.builder.get_object("button_defineEgrp_gmsh")
        self.button_overSetDefineEgrp_gmsh = self.builder.get_object("button_overSetDefineEgrp_gmsh")
        self.button_delDefineEgrp_gmsh = self.builder.get_object("button_delDefineEgrp_gmsh")

        #文字設定
        self.builder.get_object("label_selTarget").set_text(_("選択対象"))
        self.builder.get_object("button_clearSelect").get_child().set_text(_("選択clear"))
        self.builder.get_object("check_showOrgAxis_gmsh").get_child().set_text(_("原点表示"))
        self.builder.get_object("check_opacity_gmsh").get_child().set_text(_("透明化"))
        msg = "     " + _("1D, 2D, 3Dのメッシュを作成する。") + "\n"
        msg += _(">1D:(beam)-1次  >2D:(三角形, 四角形)-1次") + _("  >3D:(四面体, 六面体)-1次2次")
        self.builder.get_object("label_conts_gmsh").set_text(msg)
        msg = _("CADファイルを選択 (step, stp, brepファイル)")
        self.builder.get_object("label_selectCadFile_gmsh").set_text(_(msg))
        self.builder.get_object("label_fileName_gmsh").set_text(_("file名:"))
        self.builder.get_object("page_setMesh").set_text(_("meshの設定"))
        self.builder.get_object("page_setLocalMesh").set_text(_("localMeshの設定"))
        self.builder.get_object("page_defineGroup").set_text(_("NGRP,SGRPの定義"))
        self.builder.get_object("page_defineEgrp").set_text(_("EGRPの定義"))
        msg = _("mesh作成  Gmsh 1D-2D-3D")
        self.builder.get_object("label_gmshConts").set_text(_(msg))
        self.builder.get_object("label_meshSize").set_text(_("メッシュサイズ"))
        msg = _("CADモデルのサイズ")
        self.builder.get_object("label_cadModelSize").set_text(msg)
        msg = _("全体のメッシュサイズを設定")
        self.builder.get_object("label_globalMeshSize").set_text(msg)
        self.builder.get_object("label_nThread_gmsh").set_text(_("thread数:"))
        self.builder.get_object("label_afterMesh_gmsh").set_text(_("メッシュ作成後の処理"))
        self.check_allHexa_gmsh.get_child().set_text(_("全て六面体(subdivision)"))        
        self.check_reformHex_gmsh.get_child().set_text(_("六面体の組み換え(recombine)"))
        self.check_2ndOrder_gmsh.get_child().set_text(_("2次要素"))
        self.check_thread_gmsh.get_child().set_text(_("thread並列する(四面体のみ)"))
        self.builder.get_object("label_thread").set_text(_("（多数のsolidを含む場合、thread並列するとエラー発生する事がある。）"))
        self.check_addMesh_gmsh.get_child().set_text(_("現在のメッシュに追加する"))
        self.check_keepCond_gmsh.get_child().set_text(_("設定を保持する"))
        self.builder.get_object("label_keepCond_gmsh").set_text(_("（境界条件を保持したまま、メッシュをimportする）"))
        self.builder.get_object("button_reference_gmsh").get_child().set_text(_("参照..."))
        self.builder.get_object("button_close").get_child().set_text(_("閉じる"))
        self.builder.get_object("button_createMesh").get_child().set_text(_("mesh作成(Gmsh実行)"))
        msg = _("部分的なmeshSiseを設定する。") + "\n"
        msg += _("マウスで選択した volume, surface, edge, point のmeshSizeを設定する。")
        self.builder.get_object("label_localMeshConts").set_text(msg)
        self.builder.get_object("button_setLocalMesh_gmsh").get_child().set_text(_("設定追加"))
        self.builder.get_object("button_overSetLocalMesh_gmsh").get_child().set_text(_("選択行に上書き"))
        self.builder.get_object("button_delLocalMesh_gmsh").get_child().set_text(_("設定削除"))
        msg = _("マウスで選択した surface, edge, point のgroupを作成する。") + "\n"
        msg += _("volumeは、選択できません。")
        self.builder.get_object("label_groupConts").set_text(msg)
        self.builder.get_object("label_setNgrp").set_text(_("NGRPの設定（nodeGroup）"))
        self.builder.get_object("label_ngrpName").set_text(_("NGRP名:"))
        self.builder.get_object("button_defineNgrp_gmsh").get_child().set_text(_("設定追加"))
        self.builder.get_object("button_overSetDefineNgrp_gmsh").get_child().set_text(_("選択行に上書き"))
        self.builder.get_object("button_delDefineNgrp_gmsh").get_child().set_text(_("設定削除"))
        self.builder.get_object("label_setSgrp").set_text(_("SGRPの設定（surfaceGroup）"))
        self.builder.get_object("label_sgrpName").set_text(_("SGRP名:"))
        self.builder.get_object("button_defineSgrp_gmsh").get_child().set_text(_("設定追加"))
        self.builder.get_object("button_overSetDefineSgrp_gmsh").get_child().set_text(_("選択行に上書き"))
        self.builder.get_object("button_delDefineSgrp_gmsh").get_child().set_text(_("設定削除"))
        self.builder.get_object("button_defineEgrp_gmsh").get_child().set_text(_("設定追加"))
        self.builder.get_object("button_overSetDefineEgrp_gmsh").get_child().set_text(_("選択行に上書き"))
        self.builder.get_object("button_delDefineEgrp_gmsh").get_child().set_text(_("設定削除"))
        msg = _("マウスで選択した volume のEGRPを作成する。")
        self.builder.get_object("label_defineEgrp").set_text(msg)
        self.builder.get_object("label_setEGRP").set_text(_("EGRPの設定 (elementGroup)"))

        #tooltipを設定
        resetCamera = _("モデルを画面にフィットさせる")
        viewXP = _("X軸視(Y-Z平面)を表示")
        viewXM = _("-X軸視(Y-Z平面)を表示")
        viewYP = _("Y軸視(X-Z平面)を表示")
        viewYM = _("-Y軸視(X-Z平面)を表示")
        viewZP = _("Z軸視(X-Y平面)を表示")
        viewZM = _("-Z軸視(X-Y平面)を表示")
        #rightToLeft = _("視点を反転する(左右反転)")
        #rolling90 = _("時計回りに90度回転する")
        showAllParts = _("全部品を表示させる")
        hideSelParts = _("選択している部品（volume,surface,edge,point）を非表示する。")
        newReloadVtkMesh = _("vtkデータを再作成し、読み込む")
        reloadVtk = _("vtkデータを再読込する")
        self.builder.get_object("button_resetCamera").get_child().props.tooltip_text = resetCamera
        self.builder.get_object("button_viewXP").get_child().props.tooltip_text = viewXP
        self.builder.get_object("button_viewXM").get_child().props.tooltip_text = viewXM
        self.builder.get_object("button_viewYP").get_child().props.tooltip_text = viewYP
        self.builder.get_object("button_viewYM").get_child().props.tooltip_text = viewYM
        self.builder.get_object("button_viewZP").get_child().props.tooltip_text = viewZP
        self.builder.get_object("button_viewZM").get_child().props.tooltip_text = viewZM
        self.builder.get_object("button_newReloadVtkMesh").get_child().props.tooltip_text = newReloadVtkMesh
        #self.builder.get_object("button_rightToLeft").get_child().props.tooltip_text = rightToLeft
        #self.builder.get_object("button_rolling90").get_child().props.tooltip_text = rolling90
        self.builder.get_object("button_showAllParts").get_child().props.tooltip_text = showAllParts
        self.builder.get_object("button_hideSelParts").get_child().props.tooltip_text = hideSelParts
        self.builder.get_object("button_reloadVtkMesh").get_child().props.tooltip_text = reloadVtk
        clearSelect = _("viewer内の選択している部品をclearして、非選択状態に設定する。")
        self.builder.get_object("button_clearSelect").get_child().props.tooltip_text = clearSelect

        cadFile = _("複数のgeometryを同じファイルに含める事ができる。") + "\n"
        cadFile += _("（次元の異なるgeometryは、同じfileに含める事ができない。）")
        meshSize = _("CADモデルのサイズを参考にして、モデル全体のmeshSizeを設定する。")
        localMeshSize = _("localMeshのmeshSizeを設定する。")
        #setLocalMesh = _("部分的にmeshSizeを設定する場合は、ここをcheckする。") + "\n"
        #setLocalMesh += _("この後、meshSizeと設定するpointを指定し、「設定追加」「設定上書き」ボタンをクリックして設定する。")
        addMesh =_("現在のmeshに、このmeshを追加する。")
        keepCond = _("材料物性値や境界条件の設定を保持した状態で、このmeshに入れ替える。")
        setLocalMeshButton = _("viewer内の選択している部品のmeshSizeの設定が下表に追加される。") 
        overSetLocalMeshButton = _("viewer内の選択している部品のmeshSizeの設定が下表の選択行に置き換わる。")
        delLocalMeshButton = _("下表の選択行を削除する。")
        treeList = _("meshSizeと部品Noの表。") + "\n"
        treeList += _("行を選択すると、選択行の部品がviewerに表示される。")
        allHexa = _("tetra要素を4分割して、hexa要素を作成する。") + "\n"
        allHexa += _("直方体モデルの場合、「全て六面体」「六面体の組み換え」を選択。")
        reformHex = _("tetra要素を分割して作成したhexa要素をまとめ直して成形する。") + "\n"
        reformHex += _("直方体モデルの場合、ここをcheckする。") + "\n"
        reformHex += _("直方体モデル以外の場合、ここをcheckすると、要素品質が極端に悪くなる為、checkしない。")
        self.entry_fileName_gmsh.props.tooltip_text = cadFile
        self.entry_meshSize_gmsh.props.tooltip_text = meshSize
        self.entry_localMeshSize_gmsh.props.tooltip_text = localMeshSize
        self.check_addMesh_gmsh.props.tooltip_text = addMesh
        self.check_keepCond_gmsh.props.tooltip_text = keepCond
        self.check_allHexa_gmsh.props.tooltip_text = allHexa
        self.check_reformHex_gmsh.props.tooltip_text = reformHex
        self.button_setLocalMesh_gmsh.props.tooltip_text = setLocalMeshButton
        self.button_overSetLocalMesh_gmsh.props.tooltip_text = overSetLocalMeshButton
        self.button_delLocalMesh_gmsh.props.tooltip_text = delLocalMeshButton
        self.treeList_localMesh_gmsh.treeList.props.tooltip_text = treeList
        setMesh = _("mesh作成の為の基本的な設定値を入力する")
        self.builder.get_object("page_setMesh").props.tooltip_text = setMesh
        setLocalMesh = _("部分的にメッシュサイズを変更したい場合、ここで設定する。")
        self.builder.get_object("page_setLocalMesh").props.tooltip_text = setLocalMesh
        defineGroup = _("メッシュ作成時にNGRP、SGRPのgroupが定義できる。")
        self.builder.get_object("page_defineGroup").props.tooltip_text = defineGroup
        defineNgrp = _("viewer内の選択している部品のNGRPの定義が下表に追加される。")
        self.button_defineNgrp_gmsh.props.tooltip_text = defineNgrp
        overSetDefineNgrp = _("viewer内の選択している部品のNGRPの定義が下表の選択行に置き換わる。")
        self.button_overSetDefineNgrp_gmsh.props.tooltip_text = overSetDefineNgrp
        delDefineNgrp = _("下表の選択行を削除する。")
        self.button_delDefineNgrp_gmsh.props.tooltip_text = delDefineNgrp
        treeListNgrp = _("NGRP名と部品Noの表。") + "\n"
        treeListNgrp += _("行を選択すると、選択行の部品がviewerに表示される。")
        treeList_ngrp_gmsh.props.tooltip_text = treeListNgrp
        defineSgrp = _("viewer内の選択している部品のSGRPの定義が下表に追加される。")
        self.button_defineSgrp_gmsh.props.tooltip_text = defineSgrp
        overSetDefineSgrp = _("viewer内の選択している部品のSGRPの定義が下表の選択行に置き換わる。")
        self.button_overSetDefineSgrp_gmsh.props.tooltip_text = overSetDefineSgrp
        delDefineSgrp = _("下表の選択行を削除する。")
        self.button_delDefineSgrp_gmsh.props.tooltip_text = delDefineSgrp
        treeListSgrp = _("SGRP名と部品Noの表。") + "\n"
        treeListSgrp += _("行を選択すると、選択行の部品がviewerに表示される。")
        treeList_sgrp_gmsh.props.tooltip_text = treeListSgrp


    #
    #  initialize
    #-------------
    def initialize(self):
        """ 初期化"""
        #modelSizeをクリア
        self.label_modelSize_gmsh.set_text("")
        #checkを非表示
        self.check_reformHex_gmsh.set_sensitive(False)
        #treeViewを設定
        #  treeList_localMesh
        model = (str, str, str, str)
        headerConts = ["meshSize", _("選択対象"), _("選択No"), _("gmshTag")]
        self.treeList_localMesh_gmsh.create(model, multi=False, header=True, headerConts=headerConts)
        #  treeList_ngrp
        model = (str, str, str, str)
        headerConts = ["ngrpName", _("選択対象"), _("選択No"), _("gmshTag")]
        self.treeList_ngrp_gmsh.create(model, multi=False, header=True, headerConts=headerConts)
        #  treeList_sgrp
        model = (str, str, str, str)
        headerConts = ["ngrpName", _("選択対象"), _("選択No"), _("gmshTag")]
        self.treeList_sgrp_gmsh.create(model, multi=False, header=True, headerConts=headerConts)
        #  treeList_egrp
        model = (str, str, str, str)
        headerConts = ["egrpName", _("選択対象"), _("選択No"), _("gmshTag")]
        self.treeList_egrp_gmsh.create(model, multi=False, header=True, headerConts=headerConts)
        #frameのlabelをゴシックに設定
        bold_css = Gtk.CssProvider.new()
        cont  = b"label\n"
        cont += b"{\n"
        cont += b"    font-weight: bold;\n"
        cont += b"}\n"
        bold_css.load_from_data(cont)
        style_action = self.label_selectCadFile_gmsh.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_action = self.label_gmshConts.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_action = self.label_afterMesh_gmsh.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        #meshSizeButtonの設定
        self.setButtonSensitive_localMesh()
        self.setButtonSensitive_ngrpSgrp()

    #  setButtonSensitive_localMesh
    def setButtonSensitive_localMesh(self):
        """ localMesh項目の選択内容によってbuttonのsensitiveを決定する"""
        selItems = self.treeList_localMesh_gmsh.getSelectedItems()
        if len(selItems) == 0:
            #非選択の場合
            self.button_setLocalMesh_gmsh.set_sensitive(True)
            self.button_overSetLocalMesh_gmsh.set_sensitive(False)
            self.button_delLocalMesh_gmsh.set_sensitive(False)
        else:
            #選択している場合
            self.button_setLocalMesh_gmsh.set_sensitive(False)
            self.button_overSetLocalMesh_gmsh.set_sensitive(True)
            self.button_delLocalMesh_gmsh.set_sensitive(True)

    #  setButtonSensitive_ngrpSgrp
    def setButtonSensitive_ngrpSgrp(self):
        """ ngrp関連の選択内容によってbuttonのsensitiveを設定する"""
        #ngrp側
        selItems = self.treeList_ngrp_gmsh.getSelectedItems()
        if len(selItems) == 0:
            #非選択の場合
            self.button_defineNgrp_gmsh.set_sensitive(True)
            self.button_overSetDefineNgrp_gmsh.set_sensitive(False)
            self.button_delDefineNgrp_gmsh.set_sensitive(False)
        else:
            #選択している場合
            self.button_defineNgrp_gmsh.set_sensitive(False)
            self.button_overSetDefineNgrp_gmsh.set_sensitive(True)
            self.button_delDefineNgrp_gmsh.set_sensitive(True)
        #sgrp側
        selItems = self.treeList_sgrp_gmsh.getSelectedItems()
        if len(selItems) == 0:
            #非選択の場合
            self.button_defineSgrp_gmsh.set_sensitive(True)
            self.button_overSetDefineSgrp_gmsh.set_sensitive(False)
            self.button_delDefineSgrp_gmsh.set_sensitive(False)
        else:
            #選択している場合
            self.button_defineSgrp_gmsh.set_sensitive(False)
            self.button_overSetDefineSgrp_gmsh.set_sensitive(True)
            self.button_delDefineSgrp_gmsh.set_sensitive(True)
        #egrp側
        selItems = self.treeList_egrp_gmsh.getSelectedItems()
        if len(selItems) == 0:
            #非選択の場合
            self.button_defineEgrp_gmsh.set_sensitive(True)
            self.button_overSetDefineEgrp_gmsh.set_sensitive(False)
            self.button_delDefineEgrp_gmsh.set_sensitive(False)
        else:
            #選択している場合
            self.button_defineEgrp_gmsh.set_sensitive(False)
            self.button_overSetDefineEgrp_gmsh.set_sensitive(True)
            self.button_delDefineEgrp_gmsh.set_sensitive(True)

    #
    #  setInitialValue
    #------------------
    def setInitialValue(self, istrLines):
        """ istrDataを読み込みその値をセットする"""
        meshSize = ""; mesh2nd = "no"
        option = ""; cadFile = ""; fileName = ""; addMesh = ""; keepCond = ""
        thread = ""; nThread = "4"; allHexa = "no"; reformHex = "no"
        localMeshSizes = []; ngrpNames = []; sgrpNames = []
        egrpNames = []
        #istrLinesから取得
        for line in istrLines:
            words = line.split()
            if len(words) > 1:
                if words[0] == "convertOpt":
                    option = words[1]
                elif words[0] == "sourceMesh":
                    cadFile = words[1]
                elif words[0] == "addMesh":
                    addMesh = words[1]
                elif words[0] == "keepCond":
                    keepCond = words[1]
        #createMeshData.gmshから取得
        dataFile = self.currDir + os.sep + "createMeshData.gmsh"
        if os.path.exists(dataFile) == True:
            f = open(dataFile, encoding="utf-8"); lines = f.readlines(); f.close()
            #cadFile名をチェック
            cad = ""
            for line in lines:
                words = line.split()
                if words[0] == "cadFile":
                    cad = os.path.basename(words[1])
                    break
            #設定file内のcad名とistrDataのcad名が同じ？
            vtkCadFile = self.getVtkCadFileName()
            if cad == os.path.basename(cadFile) and cad == vtkCadFile:
                #同じ場合、設定ファイルのdataを読み込み設定
                for line in lines:
                    words = line.split()
                    if words[0] == "meshSize":
                        meshSize = words[1]
                    elif words[0] == "localMeshSize":
                        ws = words[1].split(":")
                        if len(ws) >= 3:
                            mSize = ws[0]
                            selType = ws[1]
                            nums = ws[2].split(",")
                            localMeshSizes.append([mSize, selType, nums])
                    elif words[0] == "2ndOrder":
                        mesh2nd = words[1]
                    elif words[0] == "allHexa":
                        allHexa = words[1]
                    elif words[0] == "thread":
                        thread = words[1]
                    elif words[0] == "nThread":
                        nThread = words[1]
                    elif words[0] == "reformHex":
                        reformHex = words[1]
                    elif words[0] == "ngrp":
                        ws = words[1].split(":")
                        if len(ws) >= 3:
                            name = ws[0]
                            nameType = ws[1]
                            nums = ws[2].split(",")
                            ngrpNames.append([name, nameType, nums])
                    elif words[0] == "sgrp":
                        ws = words[1].split(":")
                        if len(ws) >= 3:
                            name = ws[0]
                            nameType = ws[1]
                            nums = ws[2].split(",")
                            sgrpNames.append([name, nameType, nums])
                    elif words[0] == "egrp":
                        ws = words[1].split(":")
                        if len(ws) >= 3:
                            name = ws[0]
                            nameType = ws[1]
                            nums = ws[2].split(",")
                            egrpNames.append([name, nameType, nums])
            else:
                #異なっている場合、vtkFileをクリアする
                files = glob.glob(self.vtkFolder + os.sep + "*")
                for name in files:
                    os.remove(name)
            #Versionが合っている？
            verFile = self.vtkFolder + os.sep + "version_" + createVtk.Version
            if os.path.exists(verFile) == False:
                #vtkFileをクリアする
                files = glob.glob(self.vtkFolder + os.sep + "*")
                for name in files:
                    os.remove(name)
        #gmsh
        if option == "gmsh2fistr":
            #gmshの設定データを設定
            self.setSettingData(meshSize, mesh2nd, cadFile,
                                addMesh, keepCond, thread, nThread,
                                allHexa, reformHex, localMeshSizes,
                                ngrpNames, sgrpNames, egrpNames)

    #  getVtkCadFileName
    def getVtkCadFileName(self):
        """ cadのvtkFileを作成した元のcadFile名を取得して返す"""
        cadName = ""
        modelFile = self.currDir + os.sep + "vtkCadData" + os.sep + "modelSize"
        if os.path.exists(modelFile) == False:
            return cadName
        #vtkFileを作成したときのcadFile名を取得する
        f = open(modelFile, encoding="utf-8"); lines = f.readlines(); f.close()
        for line in lines:
            words = line.split()
            if words[0] == "cadFile":
                cadName = words[1]
                break
        return cadName

    #
    #  setSettingData
    def setSettingData(self, meshSize, mesh2nd, cadFile,
                       addMesh, keepCond, thread, nThread, 
                       allHexa, reformHex, localMeshSizes,
                       ngrpNames, sgrpNames, egrpNames):
        """ setDataをentry, label, checkに追加する。"""
        #cadFile
        self.entry_fileName_gmsh.set_text(cadFile)
        #meshSize
        self.entry_meshSize_gmsh.set_text(meshSize)
        #localMeshSizes
        if len(localMeshSizes) > 0:
            #self.check_setLocalMesh_gmsh.set_active(True)
            self.showPickedMode = "yes"            
            #self.setActivePartsOfLocalMesh()
            items = []
            for localSizes in localMeshSizes:
                mSize = localSizes[0]
                selType = localSizes[1]
                gmshConts = ",".join(localSizes[2])
                intConts = map(int, localSizes[2])
                conts = list(map(str, map(lambda x: x-1, intConts)))
                vtkConts = ",".join(conts)
                items.append([mSize, selType, vtkConts, gmshConts])
            self.localMeshItemConts = items
            self.treeList_localMesh_gmsh.setItems(items)
        else:
            self.showPickedMode = "no"
            items = []
            self.localMeshItemConts = items
            self.treeList_localMesh_gmsh.setItems(items)
        #ngrpNames
        if len(ngrpNames) > 0:
            self.showPickedMode = "yes"
            items = []
            for ngrp in ngrpNames:
                name = ngrp[0]
                nameType = ngrp[1]
                gmshConts = ",".join(ngrp[2])
                intConts = map(int, ngrp[2])
                conts = list(map(str, map(lambda x: x-1, intConts)))
                vtkConts = ",".join(conts)
                items.append([name, nameType, vtkConts, gmshConts])
            self.treeList_ngrp_gmsh.setItems(items)
        else:
            items = []
            self.treeList_ngrp_gmsh.setItems(items)
        #sgrpNames
        if len(sgrpNames) > 0:
            self.showPickedMode = "yes"
            items = []
            for sgrp in sgrpNames:
                name = sgrp[0]
                nameType = sgrp[1]
                gmshConts = ",".join(sgrp[2])
                intConts = map(int, sgrp[2])
                conts = list(map(str, map(lambda x: x-1, intConts)))
                vtkConts = ",".join(conts)
                items.append([name, nameType, vtkConts, gmshConts])
            self.treeList_sgrp_gmsh.setItems(items)
        else:
            items = []
            self.treeList_sgrp_gmsh.setItems(items)
        #egrpNames
        if len(egrpNames) > 0:
            self.showPickedMode = "yes"
            items = []
            for egrp in egrpNames:
                name = egrp[0]
                nameType = egrp[1]
                gmshConts = ",".join(egrp[2])
                intConts = map(int, egrp[2])
                conts = list(map(str, map(lambda x: x-1, intConts)))
                vtkConts = ",".join(conts)
                items.append([name, nameType, vtkConts, gmshConts])
            self.treeList_egrp_gmsh.setItems(items)
        else:
            items = []
            self.treeList_egrp_gmsh.setItems(items)
        #showPickedModeの設定(page=0は「no」に設定する)
        self.showPickedMode = "yes"
        if self.note_createMesh_gmsh.get_current_page() == 0:
            self.showPickedMode = "no"
        #allHex
        self.check_allHexa_gmsh.set_active(False)
        if allHexa == "yes":
            self.check_allHexa_gmsh.set_active(True)
        #2ndOrder
        self.check_2ndOrder_gmsh.set_active(False)
        if mesh2nd == "yes":
            self.check_2ndOrder_gmsh.set_active(True)
        #addMesh
        self.check_addMesh_gmsh.set_active(False)
        if addMesh == "yes":
            self.check_addMesh_gmsh.set_active(True)
        #keepCond
        self.check_keepCond_gmsh.set_active(False)
        if keepCond == "yes":
            self.check_keepCond_gmsh.set_active(True)
        #thread
        if thread == "yes":
            self.check_thread_gmsh.set_active(True)
            self.entry_nThread_gmsh.set_sensitive(True)
            self.label_nThread_gmsh.set_sensitive(True)
        else:
            self.check_thread_gmsh.set_active(False)
            self.entry_nThread_gmsh.set_sensitive(False)
            self.label_nThread_gmsh.set_sensitive(False)
        self.entry_nThread_gmsh.set_text(nThread)
        #reformHex
        self.check_reformHex_gmsh.set_active(False)
        if reformHex == "yes":
            self.check_reformHex_gmsh.set_active(True)


    #---- event handler ----------------
    #----- vtk画面 ---------
    def onFitToWindow(self, event):         #画面にfit
        self.fitToWindow()
    def onViewFromXP(self, event):          #X軸視
        self.viewFromXP()
    def onViewFromXM(self, event):          #-X軸視
        self.viewFromXM()
    def onViewFromYP(self, event):          #Y軸視
        self.viewFromYP()
    def onViewFromYM(self, event):          #-Y軸視
        self.viewFromYM()
    def onViewFromZP(self, event):          #Z軸視
        self.viewFromZP()
    def onViewFromZM(self, event):          #-Z軸視
        self.viewFromZM()
    # def onSetReverseCamera(self, event):    #左右反転
    #     self.setReverseCamera()
    # def onRolling90(self, event):           #90°回転
    #     self.rolling90()
    def onShowAllParts(self, event):        #全部品表示
        self.showAllParts()
    def onHideSelParts(self, event):        #選択部品を非表示
        self.hideSelParts()
    def onNewReloadVtkModel(self, event):   #新reload
        self.newReloadVtkModel()
    def onReloadVtkModel(self, event):      #reload
        self.reloadVtkModel()
    def onSelectVolume(self, event):        #volume選択
        self.selectVolume()
    def onSelectSurface(self, event):       #surface選択
        self.selectSurface()
    def onSelectEdge(self, event):          #edge選択
        self.selectEdge()
    def onSelectPoint(self, event):         #point選択
        self.selectPoint()
    def onClearSelPoints(self, event):      #選択clear
        self.clearSelPoints()
    def onShowOrgAxis(self, event):         #原点表示
        self.showOrgAxis()
    def onSetShowOpacity(self, event):      #透明化
        self.setShowOpacity()
    #----- 全体 -----------
    def onCloseWindow(self, event):         #閉じる
        self.close()
    def onCreateMesh(self, event):          #mesh作成
        self.createMesh()
    #----- meshの設定---------
    def onChangeAllHexaGmsh(self, event):   #allHexのcheck
        self.changeAllHexaGmsh()
    def onThreadMeshGmsh(self, event):      #threadのcheck
        self.threadMeshGmsh()
    def onReferenceGmsh(self, event):       #参照...
        self.getCadFileGmsh()
    def onSetFileNameGmsh(self, event):     #filename入力
        self.setFileNameGmsh()
    #---- localMeshの設定 ------
    def onSetLocalMesh(self, event):        #設定追加
        self.setLocalMeshAtPoints()
    def onOverSetLocalMesh(self, event):    #設定を上書き
        self.overSetLocalMesh()
    def onDelLocalMesh(self, event):        #設定削除
        self.deleteLocalMesh()
    def onSelectRow(self, event):           #localMeshの行を選択
        if self.maskEvent == False:
            self.selectRowOfLocalMesh()
    #----- NGRP,SGRP定義 ----------
    def onDefineNgrp(self, event):          #設定追加
        self.defineNgrp()
    def onOverSetDefineNgrp(self, event):   #選択行に上書き
        self.overSetDefineNgrp()
    def onDelDefineNgrp(self, event):       #設定削除
        self.deleteDefineNgrp()
    def onSelectNgrpRow(self, event):       #NGRPの行を選択
        self.selectNgrpRow()
    def onDefineSgrp(self, event):          #SGRP設定追加
        self.defineSgrp()
    def onOverSetDefineSgrp(self, event):   #SGRP選択行に上書き
        self.overSetDefineSgrp()
    def onDelDefineSgrp(self, event):       #SGRP選択行を削除
        self.delDefineSgrp()
    def onSelectSgrpRow(self, event):       #SGRPの行を選択
        self.selectSgrpRow()
    #----- EGRP定義 ----------
    def onDefineEgrp(self, event):          #EGRP設定追加
        self.defineEgrp()
    def onOverSetDefineEgrp(self, event):   #EGRP選択行に上書き
        self.overSetDefineEgrp()
    def onDelDefineEgrp(self, event):       #EGRP選択業を削除
        self.delDefineEgrp()
    def onSelectEgrpRow(self, event):       #EGRPの行を選択
        self.selectEgrpRow()
    #----- notebook -------------------
    def onChangePage(self, note, widget, pageNum):  #page変更
        self.changePage(pageNum)
    #----- vtkObjから ----------------
    def onGetPickedCellId(self, event):     #pointをpickした時
        if self.showPickedMode == "yes":
            self.getPickedCellId(event)
    def onGetDoublePickedActor(self, event): #doubleClickした時
        return

    #
    #  getCadFileGmsh
    #-----------------
    def getCadFileGmsh(self):
        """ 「参照」ボタンでgmsh用用のcadFileを取得する。"""
        currDir = self.currDir
        fileTypes = ["*.step", "*.stp", "*.brep", "*.STEP", "*.STP", "*.BREP"]
        filterSets = [[_("CADファイル  (*.step, *.stp, *.brep)"), fileTypes], [_("全ファイル"), ["*"]]]
        title = _("CADファイルの選択")
        funcOk = [self.setCadFileToEntryGmsh, currDir]
        funcCancel = []
        iniName = self.entry_fileName_gmsh.get_text()
        self.openFileDialog(title, currDir, funcOk, funcCancel, self.mainWindow, iniName, filterSets)
    
    #
    #  setCadFileToEntryGmsh
    def setCadFileToEntryGmsh(self, fileName, currDir):
        fileName = os.path.relpath(fileName, currDir)
        self.entry_fileName_gmsh.set_text(fileName)
        cadFile = currDir + os.sep + fileName
        self.createVtkSourceFromCad(cadFile, currDir)

    #  createVtkSourceFromCad
    def createVtkSourceFromCad(self, fileName, currDir):
        """ file名からgmshを起動して、modelSize取得、vtkDataFile作成する。"""
        if os.path.exists(fileName) == False:
            title = _("エラー")
            mess = _("入力したCADファイルは、存在しません。")
            self.errDialog(title, mess)
            return
        #Gmshを起動
        self.createVtkSourceFromCad_runGmsh(fileName, currDir)
        self.saveCadFileToEasyIstrSub_data(fileName)

    #  createVtkSourceFromCad_runGmsh
    def createVtkSourceFromCad_runGmsh(self, fileName, currDir):
        """ Gmshを起動して、vtkを作成し、modelSizeを取得する。"""
        global numSetting
        logFile = os.getenv("easyIstrUserPath") + os.sep + "gmshLog"
        #logFileを削除
        if os.path.exists(logFile) == True:
            os.remove(logFile)
        #comm作成
        comm  = "python3 "
        comm += os.getenv("easyIstrPath") + os.sep 
        comm += "bin" + os.sep
        comm += "readCadFileCreateVtkFiles_gmsh.py "
        comm += fileName + " "
        comm += currDir + " "
        comm += self.vtkFormat
        #logFile = os.getenv("easyIstrUserPath") + os.sep + "gmshLog"
        print(comm)
        #teeComm作成
        teeComm = "python3 "
        teeComm += os.getenv("easyIstrPath") + os.sep
        teeComm += "python" + os.sep
        teeComm += "commTee.py "
        teeComm += comm + " "
        teeComm += logFile
        pyFistr.run().commandBG(teeComm)
        #logFileが出来上がるまで待つ
        i = 0
        while i < 10:
            if os.path.exists(logFile) == True:
                self.gmshLogFileObj = open(logFile)
                break
            time.sleep(0.1)
            i += 1
        #logFileをチェック（timerEventをセット）
        GLib.timeout_add(200, self.checkGmshLog)
        self.setStatusBar("reading CAD file...")
        return

    #  checkGmshLog
    def checkGmshLog(self):
        """ gmshLogの内容をcheck
        gmshの処理が終了したかどうかをチェック。
        終了していれば、終了処理。"""
        global numSetting
        lines = self.gmshLogFileObj.readlines()
        if len(lines) == 0:
            return True
        flag = 0
        globalMeshSize = ""
        for line in lines:
            if line[:len("globalMeshSize:")] == "globalMeshSize:":
                globalMeshSize = line[len("globalMeshSize:"):].split()[0]
            elif line[:len("finish to create ")] == "finish to create ":
                flag = 1
                break
        if flag == 0:
            return True
        
        #logFileをclose
        self.gmshLogFileObj.close()
        #エラーチェック(vtkFileをcheck)
        vtkFolder = currDir + os.sep + "vtkCadData"
        flag = 0
        for name in ["points.vtu", "edges.vtu", "surfaces.vtu"]:
            vtkFile = vtkFolder + os.sep + name
            if os.path.exists(vtkFile) == False:
                flag = 1
                break
        if flag == 1:
            #error停止
            title = _("エラー")
            mess = _("CAD形状表示用のdataが作成できませんでした。") + "\n"
            mess += _("CAD形状を再作成してください。")
            self.errDialog(title, mess)
        #vtkを表示
        self.vtkInitialize()
        self.viewFromZM()
        self.createVtkActorDict()
        self.addPointCellsActors()
        self.showVtkModel()
        self.clearSettingData(globalMeshSize)
        #statusBarを元に戻す
        self.setBackgroundColor(0)
        return False

    #
    #  clearSettingData
    def clearSettingData(self, meshSize=""):
        """ 設定データをクリアする"""
        cadFile = self.entry_fileName_gmsh.get_text()
        #meshSize = ""; mesh2nd = "no"
        mesh2nd = "no"
        addMesh = ""; keepCond = ""
        thread = ""; nThread = "4"; allHexa = "no"; 
        reformHex = "no"; localMeshSizes = []; ngrpNames = []
        sgrpNames = []; egrpNames = []
        self.setSettingData(meshSize, mesh2nd, cadFile,
                            addMesh, keepCond, thread, nThread,
                            allHexa, reformHex, localMeshSizes,
                            ngrpNames, sgrpNames, egrpNames)
        #modelSize, meshSizeを設定する
        self.setModelSizeMeshSizeFromVtk()

    #
    #  setFileNameGmsh
    #------------------
    def setFileNameGmsh(self):
        """ fileNameを入力後、「enter」ボタンで処理"""
        fileName = self.entry_fileName_gmsh.get_text()
        if fileName == "":
            return
        cadFile = self.currDir + os.sep + fileName
        self.createVtkSourceFromCad(cadFile, self.currDir)

    #
    #  changeAllHexaGmsh
    #----------------
    def changeAllHexaGmsh(self):
        """ gmsh側で要素Typeを変更した時のevent"""
        if self.check_allHexa_gmsh.get_active() == True:
            self.check_reformHex_gmsh.set_sensitive(True)
        else:
            self.check_reformHex_gmsh.set_sensitive(False)
            self.check_reformHex_gmsh.set_active(False)

    #
    #  threadMeshGmsh
    def threadMeshGmsh(self):
        """ gmsh側でnThreadsを変更した時のevent"""
        if self.check_thread_gmsh.get_active() == True:
            self.label_nThread_gmsh.set_sensitive(True)
            self.entry_nThread_gmsh.set_sensitive(True)
        else:
            self.label_nThread_gmsh.set_sensitive(False)
            self.entry_nThread_gmsh.set_sensitive(False)

    #  afterCreatedMesh
    def afterCreatedMesh(self, condMeshDict):
        """ mesh作成後の処理"""
        funcName = self.funcAfterMesh[0]
        funcArgs = [condMeshDict] + self.funcAfterMesh[1:]
        funcName(*funcArgs)

    #
    #  createMesh
    #------------------
    def createMesh(self):
        """ gmshによるmesh作成"""
        #初期化
        #  currを設定
        os.chdir(self.currDir)
        #設定値読み込み
        fileName = self.entry_fileName_gmsh.get_text()
        meshSize = self.entry_meshSize_gmsh.get_text()
        maxSize = meshSize
        minSize = "0.0"
        allHexa = self.check_allHexa_gmsh.get_active()
        mesh2nd = self.check_2ndOrder_gmsh.get_active()
        thread = self.check_thread_gmsh.get_active()
        nThread = self.entry_nThread_gmsh.get_text()
        reform = self.check_reformHex_gmsh.get_active()
        addMesh = self.check_addMesh_gmsh.get_active()
        keepCond = self.check_keepCond_gmsh.get_active()
        localMeshSizes = self.treeList_localMesh_gmsh.getAllItemNames()
        ngrpNames = self.treeList_ngrp_gmsh.getAllItemNames()
        sgrpNames = self.treeList_sgrp_gmsh.getAllItemNames()
        egrpNames = self.treeList_egrp_gmsh.getAllItemNames()
        #エラーチェック
        error = ""
        eTitle = _("エラー")
        #cadFile
        if fileName == "":
            error = _("CADファイル名が設定されていません")
            self.errDialog(eTitle, error)
            return
        else:
            filePath = self.currDir + os.sep + fileName
            if os.path.exists(filePath) == False:
                error = _("CADファイルが存在しません")
                self.errDialog(eTitle, error)
                return
        
        #meshFileを削除
        self.meshFileName = ".".join(os.path.basename(fileName).split(".")[:-1]) + "_fistr.msh"
        meshFile = self.currDir + os.sep + self.meshFileName
        if os.path.exists(meshFile) == True:
            os.remove(meshFile)
        #maxSize
        if maxSize == "":
            #maxSizeのdefault値
            maxSize = "1e22"
        else:
            try:
                a = float(maxSize)
            except:
                error = _("maxhが数値に変換できません")
                self.errDialog(eTitle, error)
                return
        #minSize
        if minSize == "":
            #minSizeのdefault値
            minSize = "0.0"
        else:
            try:
                a = float(minSize)
            except:
                error = _("minhが数値に変換できません")
                self.errDialog(eTitle, error)
                return
        #allHexa
        if allHexa == True:
            allHexa = "yes"
            if reform == True:
                reformHex = "yes"
            else:
                reformHex = "no"
        else:
            allHexa = "no"
            reformHex = "no"
        #2ndorder
        if mesh2nd == True:
            mesh2nd = "yes"
        else:
            mesh2nd = "no"
        #thread
        if thread == True:
            thread = "yes"
        else:
            thread = "no"
        if nThread == "":
            nThread = "1"
        #addMesh
        if addMesh == True:
            addMesh = "yes"
        else:
            addMesh = "no"
        #keepCond
        if keepCond == True:
            keepCond = "yes"
        else:
            keepCond = "no"
        #gmshOptファイル作成
        self.createMeshDataForGmsh(
            fileName, meshSize, localMeshSizes, mesh2nd, 
            allHexa, thread, nThread, reformHex, ngrpNames,
            sgrpNames, egrpNames)
        #mesh作成
        self.setStatusBar("creating mesh...")
        self.createMeshGmsh_run(fileName, addMesh, keepCond)        
        return

    #
    #  createMeshDataForGmsh
    def createMeshDataForGmsh(self, 
            cadFile, meshSize, localMeshSizes, mesh2nd, 
            allHexa, thread, nThread, reformHex, ngrpNames,
            sgrpNames, egrpNames):
        """ mesh作成用data（Gmsh）をfileに保存する。"""
        cadFile = os.path.abspath(cadFile)
        lines = []
        lines += ["#creating mesh data for Gmsh\n"]
        lines += ["cadFile " + cadFile + "\n"]
        lines += ["meshSize " + meshSize + "\n"]
        for meshSize in localMeshSizes:
            #      meshSize            対象                gmshTag
            cont = meshSize[0] + ":" + meshSize[1] + ":" + meshSize[3]
            lines += ["localMeshSize " + cont + "\n"]
        lines += ["2ndOrder " + mesh2nd + "\n"]
        lines += ["allHexa " + allHexa + "\n"]
        lines += ["thread " + thread + "\n"]
        lines += ["nThread " + nThread + "\n"]
        lines += ["reformHex " + reformHex + "\n"]
        for ngrp in ngrpNames:
            #      ngrp            対象            gmshTag
            cont = ngrp[0] + ":" + ngrp[1] + ":" + ngrp[3]
            lines += ["ngrp " + cont + "\n"]
        for sgrp in sgrpNames:
            #      sgrp            対象            gmshTag
            cont = sgrp[0] + ":" + sgrp[1] + ":" + sgrp[3]
            lines += ["sgrp " + cont + "\n"]
        for egrp in egrpNames:
            #      egrp            対象            gmshTag
            cont = egrp[0] + ":" + egrp[1] + ":" + egrp[3]
            lines += ["egrp " + cont + "\n"]
        gmshOptFile = self.currDir + os.sep + "createMeshData.gmsh"
        f = open(gmshOptFile, "w", encoding="utf-8"); f.writelines(lines); f.close()

    #
    #  saveCadFileToEasyIstrSub_data
    def saveCadFileToEasyIstrSub_data(self, cadFilePath):
        """ 相対pathのcadFileをeasyIstrSub_dataファイルに保存する"""
        cadFileRelPath = os.path.relpath(cadFilePath, self.currDir)
        fileName = self.currDir + os.sep + "easyIstrSub_data"
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        for i in range(len(lines)):
            line = lines[i]
            words = line.split()
            if len(words) >= 2:
                if words[0] == "sourceMesh":
                    words[1] = cadFileRelPath
                    line = " ".join(words) + "\n"
                    lines[i] = line
                    break
        f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

    #
    #  createMeshGmsh_run
    #---------------------
    def createMeshGmsh_run(self, fileName, addMesh, keepCond):
        """ gmshによるmesh作成を実行する"""
        #保存用の辞書作成
        condMeshDict = {
            "convertOpt": "gmsh2fistr", 
            "sourceMesh": fileName,         #cadFileName
            "addMesh": addMesh, "keepCond": keepCond}
        #実行コマンド作成
        cadFilePath = self.currDir + os.sep + fileName
        comm  = "python3 "
        comm += os.getenv("easyIstrPath") + os.sep 
        comm += "bin" + os.sep + "createMesh_gmsh.py "
        comm += self.currDir
        print(comm)
        #実行
        if os.name == "nt":
            #windows
            runFile = self.currDir + os.sep + "runGmsh.bat"
            logFile = self.currDir + os.sep + "gmsh.log"
            comm = "python3 " + os.getenv("pythonApp") + os.sep + "commTee.py " + comm + " " + logFile
            f = open(runFile, "w", encoding="utf-8"); f.write(comm); f.close()
            appDict = pyFistr.readUsingApplications()
            terminalRun = appDict["terminalRun"]
            comm = terminalRun + " cmd /c " + runFile
            pyFistr.run().command(comm)
        else:
            #linux
            comm += " 2>&1 | tee " + self.currDir + os.sep + "gmsh.log"
            runFile = self.currDir + os.sep + "runGmsh"
            f = open(runFile, "w", encoding="utf-8"); f.write(comm); f.close()
            pyFistr.run().command("chmod a+x " + runFile)
            appDict = pyFistr.readUsingApplications()
            terminalRun = appDict["terminalRun"]
            comm = terminalRun + " " + runFile
            pyFistr.run().command(comm)
        # if self.runningTimer == False:
        #     #timerEventでmesh完成を監視
        #     GLib.timeout_add(200, self.waitUntilGmshHalt, condMeshDict)
        #timerEventでmesh完成を監視
        GLib.timeout_add(200, self.waitUntilGmshHalt, condMeshDict)


    #  waitUntilGmshHalt
    def waitUntilGmshHalt(self, condMeshDict):
        """ gmshの終了を待つtimerEvent"""
        #gmsh終了?
        meshFile = self.currDir + os.sep + self.meshFileName
        #meshFileが出来上がっているか？
        if os.path.exists(meshFile) == True:
            #window閉じる
            self.close()
            #timerEvent停止
            self.runningTimer = False
            #resultFileを作成
            self.createResultFile("OK", condMeshDict)
            return False
        else:
            if self.closeFlag == True:
                #window閉じている場合、timer停止
                self.runningTimer = False
                #resultFileを作成
                self.createResultFile("CANCEL", condMeshDict)
                return False
            #timerEvent継続
            self.runningTimer = True
            return True

    #
    #  createResultFile
    def createResultFile(self, stat, condMeshDict):
        """ createMeshの結果を「createMeshResult」に保存する"""
        resultFile = self.currDir + os.sep + "createMeshResult"
        if stat == "OK":
            lines = ["OK\n"]
        else:
            lines = ["CANCELl\n"]
        lines += ["convertOpt " + condMeshDict["convertOpt"] + "\n"]
        lines += ["sourceMesh " + condMeshDict["sourceMesh"] + "\n"]
        lines += ["addMesh " + condMeshDict["addMesh"] + "\n"]
        lines += ["keepCond " + condMeshDict["keepCond"] + "\n"]
        lines += ["!END\n"]
        f = open(resultFile, "w", encoding="utf-8")
        f.writelines(lines)
        f.close()


    #---universal dialog--------------
    def openFileDialog(self, title, fileDir, funcOk, funcCancel, parent=None, selName="", filterSets=[]):
        dialog = universalDialogs.openFileDialog(title, fileDir, funcOk, funcCancel, parent, selName, filterSets)
        dialog.show()

    def okDialog(self, title, message, parent=None):
        dialog = universalDialogs.okDialog(title, message, parent)
        dialog.show()

    def errDialog(self, title, message, parent=None):
        dialog = universalDialogs.errDialog(title, message, parent)
        dialog.show()




if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", localeDir)

    currDir = sys.argv[1]
    winApp = createMesh(currDir)
    winApp.main()
