#!/usr/bin/python3
#  coding: utf-8
#
#   calcSectionCoeffDDialog.py
#
#       beamの断面形状から断面の定数を計算する。
#       断面積、断面2次モーメント、ねじり定数
#
#   21/02/17    新規作成
#      02/25    断面系数Zyy, Zzzの計算を追加
#      04/29    定数の計算をモジュール化
#               （calcSectionCoeff.py）
#      04/30    断面形状がC型、U型を追加
#   22/03/14    国際化
#      03/22    国際化（文字をpython側でセット）
#   24/05/24    calcSectionCoeffClass:新規追加
#      06/06    universalDialogsに変更
#      06/08    errDialog:parentを追加
#      06/09    parentを追加
#      07/02    setSettingDataToGui:起動時にcomboとnoteBookの内容が
#               合わなくなったので、起動時には、combo、noteの両方をセット
#               する様に修正。
#      07/07    openにencoding="utf-8"を追加
#      07/25    setSettingDataToGui:未定義の設定項目があると、エラー表示して
#               いたが、デフォルトの値をセットして保存するように修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk

import os, sys
import pyFistr
import GtkParts
import math
import calcSectionCoeff
import universalDialogs

#import locale
localeDir = os.getenv("LOCALE_DIR")
#locale.bindtextdomain("easyistr", localeDir)

ls = "\n"

#---------------------
#  dialogApp class
#---------------------
class dialogApp:

    def __init__(self, parent=None):
        self.builder = Gtk.Builder()
        #self.builder.set_translation_domain("easyistr")
        path = os.getenv("easyIstrPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "calcSectionCoeffDDialog.glade")
        self.dialog = self.builder.get_object("dialog1")
        self.dialog.set_transient_for(parent)
        self.dialog.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        self.dialog.set_title(_("断面の定数を算出"))
        self.setGtkObject()
        #変数を定義
        self.stat = []
        self.comboItems = (
            _("丸棒"), _("パイプ"), _("角棒"), _("中空角棒"),
            _("H型"), _("I型"), _("L型"), _("C型"), _("U型"))
        self.materials = (
            "round", "pipe", "square", "squarePipe",
             "H", "I", "L", "C", "U")
        self.resItems = ("material", "area", "Iyy", "Izz", "Jx", "Zyy", "Zzz")
        self.settingFile = os.sep.join([
            os.getenv("easyIstrUserPath"), 
            "data", 
            "calcSectionCoeffDDialog_data"])
        #初期化
        self.initialize()

    #  show
    def show(self):
        self.dialog.show()

    #  close
    def close(self, *args):
        self.dialog.destroy()

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        self.entry_area = self.builder.get_object("entry_area")
        self.entry_Iyy = self.builder.get_object("entry_Iyy")
        self.entry_Izz = self.builder.get_object("entry_Izz")
        self.entry_Jx = self.builder.get_object("entry_Jx")
        self.entryMaru_dia = self.builder.get_object("entryMaru_dia")
        self.entryPipe_d = self.builder.get_object("entryPipe_d")
        self.entryPipe_t = self.builder.get_object("entryPipe_t")
        self.entryKaku_B = self.builder.get_object("entryKaku_B")
        self.entryKaku_h = self.builder.get_object("entryKaku_h")
        self.entryChuKaku_B = self.builder.get_object("entryChuKaku_B")
        self.entryChuKaku_h = self.builder.get_object("entryChuKaku_h")
        self.entryChuKaku_t = self.builder.get_object("entryChuKaku_t")
        self.entryH_B = self.builder.get_object("entryH_B")
        self.entryH_h = self.builder.get_object("entryH_h")
        self.entryH_t1 = self.builder.get_object("entryH_t1")
        self.entryH_t2 = self.builder.get_object("entryH_t2")
        self.entryI_B = self.builder.get_object("entryI_B")
        self.entryI_h = self.builder.get_object("entryI_h")
        self.entryI_t1 = self.builder.get_object("entryI_t1")
        self.entryI_t2 = self.builder.get_object("entryI_t2")
        self.entryL_B = self.builder.get_object("entryL_B")
        self.entryL_h = self.builder.get_object("entryL_h")
        self.entryL_t1 = self.builder.get_object("entryL_t1")
        self.entryL_t2 = self.builder.get_object("entryL_t2")
        self.entryC_B = self.builder.get_object("entryC_B")
        self.entryC_h = self.builder.get_object("entryC_h")
        self.entryC_t1 = self.builder.get_object("entryC_t1")
        self.entryC_t2 = self.builder.get_object("entryC_t2")
        self.entryU_B = self.builder.get_object("entryU_B")
        self.entryU_h = self.builder.get_object("entryU_h")
        self.entryU_t1 = self.builder.get_object("entryU_t1")
        self.entryU_t2 = self.builder.get_object("entryU_t2")
        self.label_Zyy = self.builder.get_object("label_Zyy")
        self.label_Zzz = self.builder.get_object("label_Zzz")
        self.note_mat = self.builder.get_object("note_mat")
        self.combo_mat_org = self.builder.get_object("combo_mat")
        self.combo_mat = GtkParts.comboBox(self.combo_mat_org)
        self.button_cancel = self.builder.get_object("button_cancel")
        self.button_apply = self.builder.get_object("button_apply")
        self.button_maru = self.builder.get_object("button_calc_p0")
        self.button_pipe = self.builder.get_object("button_calc_p1")
        self.button_kaku = self.builder.get_object("button_calc_p2")
        self.button_chuKaku = self.builder.get_object("button_calc_p3")
        self.button_H = self.builder.get_object("button_calc_p4")
        self.button_I = self.builder.get_object("button_calc_p5")
        self.button_L = self.builder.get_object("button_calc_p6")
        self.button_C = self.builder.get_object("button_calc_p7")
        self.button_U = self.builder.get_object("button_calc_p8")
        #文字をセット
        self.setStringToFirstPage()
        self.setStringToNotebook()

    #
    #  setStringToFirstPage
    def setStringToFirstPage(self):
        """ 表紙（notebookを除く）に文字をセット"""
        self.builder.get_object("label_subTitle").set_text(_("断面2次モーメント等の定数を取得"))
        mess = _("プルダウンボックスから材料を選択。\n材料（形状）に応じた定数をを入力し、「計算」ボタンをクリック。\n「適用」ボタンクリックして計算結果を戻す。")
        self.builder.get_object("label_calcMethod").set_text(mess)
        self.builder.get_object("label_material").set_text(_("材料:"))
        self.builder.get_object("label_calcResult").set_text(_("計算結果、設定内容"))
        self.builder.get_object("label_area").set_text(_("断面積"))
        self.builder.get_object("label_moment").set_text(_("断面2次モーメント"))
        self.builder.get_object("label_nejiri").set_text(_("ねじり定数"))
        self.builder.get_object("label_IzzZzzConts").set_text(_("Izz,Zzz: z軸が中立軸の方向（z軸回りモーメントで変形）"))
        self.builder.get_object("button_cancel").get_child().set_text(_("キャンセル"))
        self.builder.get_object("button_apply").get_child().set_text(_("適用"))

    #
    #  setStringToNNotebook
    def setStringToNotebook(self):
        """ nootbookへの文字をセット"""
        #丸棒page (page0)
        self.builder.get_object("label_maruBar_p0").set_text(_("丸棒"))
        self.builder.get_object("label_calcMethod_p0").set_text(_("ねじり定数Jxの算出方法"))
        self.builder.get_object("label_dia_p0").set_text(_("外径 d"))
        self.builder.get_object("button_calc_p0").get_child().set_text(_("計算"))
        #pipe page (page1)
        self.builder.get_object("label_pipe_p1").set_text(_("パイプ"))
        self.builder.get_object("label_calcMethod_p1").set_text(_("ねじり定数Jxの算出方法"))
        self.builder.get_object("label_dia_p1").set_text(_("外径 d"))
        self.builder.get_object("label_thick_p1").set_text(_("厚さ t"))
        self.builder.get_object("button_calc_p1").get_child().set_text(_("計算"))
        #角棒 page (page2)
        self.builder.get_object("label_square_p2").set_text(_("角棒"))
        self.builder.get_object("label_calcMethod_p2").set_text(_("ねじり定数Jxの算出方法"))
        self.builder.get_object("label_width_p2").set_text(_("幅 B"))
        self.builder.get_object("label_height_p2").set_text(_("高さ h"))
        self.builder.get_object("button_calc_p2").get_child().set_text(_("計算"))
        #中空 page (page3)
        self.builder.get_object("label_squareEmp_p3").set_text(_("中空角棒"))
        self.builder.get_object("label_calcMethod_p3").set_text(_("ねじり定数Jxの算出方法"))
        self.builder.get_object("label_calcConts_p3").set_text(_("薄肉閉断面で算出"))
        self.builder.get_object("label_width_p3").set_text(_("幅 B"))
        self.builder.get_object("label_height_p3").set_text(_("高さ h"))
        self.builder.get_object("label_thick_p3").set_text(_("厚さ t"))
        self.builder.get_object("button_calc_p3").get_child().set_text(_("計算"))
        #H型鋼 page (page4)
        self.builder.get_object("label_Hform_p4").set_text(_("H型鋼"))
        self.builder.get_object("label_calcMethod_p4").set_text(_("ねじり定数Jxの算出方法"))
        self.builder.get_object("label_calcConts_p4").set_text(_("薄肉開断面で算出"))
        self.builder.get_object("label_width_p4").set_text(_("幅 B"))
        self.builder.get_object("label_height_p4").set_text(_("高さ h"))
        self.builder.get_object("label_thick1_p4").set_text(_("板厚 t1"))
        self.builder.get_object("label_thick2_p4").set_text(_("板厚 t2"))
        self.builder.get_object("button_calc_p4").get_child().set_text(_("計算"))
        #I型鋼 page (page5)
        self.builder.get_object("label_Iform_p5").set_text(_("I型"))
        self.builder.get_object("label_calcMethod_p5").set_text(_("ねじり定数Jxの算出方法"))
        self.builder.get_object("label_calcConts_p5").set_text(_("薄肉開断面で算出"))
        self.builder.get_object("label_width_p5").set_text(_("幅 B"))
        self.builder.get_object("label_height_p5").set_text(_("高さ h"))
        self.builder.get_object("label_thick1_p5").set_text(_("板厚 t1"))
        self.builder.get_object("label_thick2_p5").set_text(_("板厚 t2"))
        self.builder.get_object("button_calc_p5").get_child().set_text(_("計算"))
        #L型鋼 page (page6)
        self.builder.get_object("label_Lform_p6").set_text(_("L型"))
        self.builder.get_object("label_calcMethod_p6").set_text(_("ねじり定数Jxの算出方法"))
        self.builder.get_object("label_calcConts_p6").set_text(_("薄肉開断面で算出"))
        self.builder.get_object("label_width_p6").set_text(_("幅 B"))
        self.builder.get_object("label_height_p6").set_text(_("高さ h"))
        self.builder.get_object("label_thick1_p6").set_text(_("板厚 t1"))
        self.builder.get_object("label_thick2_p6").set_text(_("板厚 t2"))
        self.builder.get_object("button_calc_p6").get_child().set_text(_("計算"))
        #C型 page (page7)
        self.builder.get_object("label_Cform_p7").set_text(_("C型"))
        self.builder.get_object("label_calcMethod_p7").set_text(_("ねじり定数Jxの算出方法"))
        self.builder.get_object("label_calcConts_p7").set_text(_("薄肉開断面で算出"))
        self.builder.get_object("label_width_p7").set_text(_("幅 B"))
        self.builder.get_object("label_height_p7").set_text(_("高さ h"))
        self.builder.get_object("label_thick1_p7").set_text(_("板厚 t1"))
        self.builder.get_object("label_thick2_p7").set_text(_("板厚 t2"))
        self.builder.get_object("button_calc_p7").get_child().set_text(_("計算"))
        #U型 page (page8)
        self.builder.get_object("label_Uform_p8").set_text(_("U型"))
        self.builder.get_object("label_calcMethod_p8").set_text(_("ねじり定数Jxの算出方法"))
        self.builder.get_object("label_calcConts_p8").set_text(_("薄肉開断面で算出"))
        self.builder.get_object("label_width_p8").set_text(_("幅 B"))
        self.builder.get_object("label_height_p8").set_text(_("高さ h"))
        self.builder.get_object("label_thick1_p8").set_text(_("板厚 t1"))
        self.builder.get_object("label_thick2_p8").set_text(_("板厚 t2"))
        self.builder.get_object("button_calc_p8").get_child().set_text(_("計算"))

    #
    #  initialize
    #-------------
    def initialize(self):
        """ 初期化"""
        #comboにデータをセット(初期値)
        items = self.comboItems
        self.combo_mat.setItems(items)
        pageNo = 0
        self.combo_mat.selectItem(items[pageNo])
        self.note_mat.set_current_page(pageNo)
        #settingFileからデータを取得してセット
        settingDict = self.getSettingDataFromFile()
        self.setSettingDataToGui(settingDict)

    #
    #  changePage
    #-------------
    def changePage(self):
        item = self.combo_mat.getSelectedItem()
        pageNo = self.comboItems.index(item)
        self.note_mat.set_current_page(pageNo)

    #
    #  calcMaru
    #-----------
    def calcMaru(self):
        """ 丸棒の計算"""
        textDia = self.entryMaru_dia.get_text()
        try:
            dia = float(textDia)
        except:
            title = _("エラー")
            mess = _("丸棒の外径が取得できません。") + ls
            mess += _("外形を入力し直してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        sect = calcSectionCoeff.sectionCircle(dia)
        area = sect.A()
        Izz = sect.Iz()
        Iyy = sect.Iy()
        Jx = sect.Jx()
        self.setCoeffToEntry(area, Iyy, Izz, Jx)
        Zyy = sect.Zy()
        Zzz = sect.Zz()
        self.setCoeffToLabel(Zyy, Zzz)

    #
    #  setCoeffToEntry
    def setCoeffToEntry(self, area, Iyy, Izz, Jx):
        """ 定数をentryにセットする"""
        strArea = "%11.5e" % area
        strIyy = "%11.5e" % Iyy
        strIzz = "%11.5e" % Izz
        strJx = "%11.5e" % Jx
        self.entry_area.set_text(strArea)
        self.entry_Iyy.set_text(strIyy)
        self.entry_Izz.set_text(strIzz)
        self.entry_Jx.set_text(strJx)

    #
    #  setCoeffTolabel
    def setCoeffToLabel(self, Zyy, Zzz):
        """ 断面系数をlabelにセット"""
        strZyy = "%11.5e" % Zyy
        strZzz = "%11.5e" % Zzz
        self.label_Zyy.set_text(strZyy)
        self.label_Zzz.set_text(strZzz)

    #
    #  calcPipe
    #------------
    def calcPipe(self):
        """ パイプの計算"""
        text_d = self.entryPipe_d.get_text()
        text_t = self.entryPipe_t.get_text()
        try:
            d = float(text_d)
            t = float(text_t)
        except:
            title = _("エラー")
            mess = _("パイプの外径 or 厚さが取得できません。") + ls
            mess += _("入力し直してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        sect = calcSectionCoeff.sectionPipe(d, t)
        area = sect.A()
        Izz = sect.Iz()
        Iyy = sect.Iy()
        Jx = sect.Jx()
        self.setCoeffToEntry(area, Iyy, Izz, Jx)
        Zzz = sect.Zz()
        Zyy = sect.Zy()
        self.setCoeffToLabel(Zyy, Zzz)

    #
    #  calcKaku
    #-----------
    def calcKaku(self):
        """ 角棒の計算"""
        text_B = self.entryKaku_B.get_text()
        text_h = self.entryKaku_h.get_text()
        try:
            B = float(text_B)
            h = float(text_h)
        except:
            title = _("エラー")
            mess = _("角棒の定数が取得できません。") + ls
            mess += _("入力し直してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        sect = calcSectionCoeff.sectionRectangle(B, h)
        area = sect.A()
        Izz = sect.Iz()
        Iyy = sect.Iy()
        Jx = sect.Jx()
        self.setCoeffToEntry(area, Iyy, Izz, Jx)
        Zzz = sect.Zz()
        Zyy = sect.Zy()
        self.setCoeffToLabel(Zyy, Zzz)

    #
    #  calcChuKaku
    #--------------
    def calcChuKaku(self):
        """ 中空角棒の計算"""
        text_B = self.entryChuKaku_B.get_text()
        text_h = self.entryChuKaku_h.get_text()
        text_t = self.entryChuKaku_t.get_text()
        try:
            B = float(text_B)
            h = float(text_h)
            t = float(text_t)
        except:
            title = _("エラー")
            mess = _("中空角棒の定数が取得できません。") + ls
            mess += _("入力し直してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        sect = calcSectionCoeff.sectionRectanglePipe(B, h, t)
        area = sect.A()
        Izz = sect.Iz()
        Iyy = sect.Iy()
        Jx = sect.Jx()
        self.setCoeffToEntry(area, Iyy, Izz, Jx)
        Zzz = sect.Zz()
        Zyy = sect.Zy()
        self.setCoeffToLabel(Zyy, Zzz)

    #
    #  calcH
    #--------
    def calcH(self):
        """ H型の計算"""
        text_B = self.entryH_B.get_text()
        text_h = self.entryH_h.get_text()
        text_t1 = self.entryH_t1.get_text()
        text_t2 = self.entryH_t2.get_text()
        try:
            B = float(text_B)
            h = float(text_h)
            t1 = float(text_t1)
            t2 = float(text_t2)
        except:
            title = _("エラー")
            mess = _("H型の定数が取得できません。") + ls
            mess += _("入力し直してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        sect = calcSectionCoeff.sectionH(B, h, t1, t2)
        area = sect.A()
        Izz = sect.Iz()
        Iyy = sect.Iy()
        Jx = sect.Jx()
        self.setCoeffToEntry(area, Iyy, Izz, Jx)
        Zzz = sect.Zz()
        Zyy = sect.Zy()
        self.setCoeffToLabel(Zyy, Zzz)

    #
    #  calcI
    #--------
    def calcI(self):
        """ I型鋼の計算"""
        text_B = self.entryI_B.get_text()
        text_h = self.entryI_h.get_text()
        text_t1 = self.entryI_t1.get_text()
        text_t2 = self.entryI_t2.get_text()
        try:
            B = float(text_B)
            h = float(text_h)
            t1 = float(text_t1)
            t2 = float(text_t2)
        except:
            title = _("エラー")
            mess = _("I型の定数が取得できません。") + ls
            mess += _("入力し直してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        sect = calcSectionCoeff.sectionI(B, h, t1, t2)
        area = sect.A()
        Izz = sect.Iz()
        Iyy = sect.Iy()
        Jx = sect.Jx()
        self.setCoeffToEntry(area, Iyy, Izz, Jx)
        Zzz = sect.Zz()
        Zyy = sect.Zy()
        self.setCoeffToLabel(Zyy, Zzz)

    #
    #  calcL
    #--------
    def calcL(self):
        """ L型鋼の計算"""
        text_B = self.entryL_B.get_text()
        text_h = self.entryL_h.get_text()
        text_t1 = self.entryL_t1.get_text()
        text_t2 = self.entryL_t2.get_text()
        try:
            B = float(text_B)
            h = float(text_h)
            t1 = float(text_t1)
            t2 = float(text_t2)
        except:
            title = _("エラー")
            mess = _("L型の定数が取得できません。") + ls
            mess += _("入力し直してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        sect = calcSectionCoeff.sectionL(B, h, t1, t2)
        area = sect.A()
        Izz = sect.Iz()
        Iyy = sect.Iy()
        Jx = sect.Jx()
        Zzz = sect.Zz()
        Zyy = sect.Zy()
        self.setCoeffToEntry(area, Iyy, Izz, Jx)
        self.setCoeffToLabel(Zyy, Zzz)

    #
    #  calcC
    #--------
    def calcC(self):
        """ C型の計算"""
        text_B = self.entryC_B.get_text()
        text_h = self.entryC_h.get_text()
        text_t1 = self.entryC_t1.get_text()
        text_t2 = self.entryC_t2.get_text()
        try:
            B = float(text_B)
            h = float(text_h)
            t1 = float(text_t1)
            t2 = float(text_t2)
        except:
            title = _("エラー")
            mess = _("C型の定数が取得できません。") + ls
            mess += _("入力し直してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        sect = calcSectionCoeff.sectionC(B, h, t1, t2)
        area = sect.A()
        Izz = sect.Iz()
        Iyy = sect.Iy()
        Jx = sect.Jx()
        Zzz = sect.Zz()
        Zyy = sect.Zy()
        self.setCoeffToEntry(area, Iyy, Izz, Jx)
        self.setCoeffToLabel(Zyy, Zzz)

    #
    #  calcU
    #--------
    def calcU(self):
        """ C型の計算"""
        text_B = self.entryU_B.get_text()
        text_h = self.entryU_h.get_text()
        text_t1 = self.entryU_t1.get_text()
        text_t2 = self.entryU_t2.get_text()
        try:
            B = float(text_B)
            h = float(text_h)
            t1 = float(text_t1)
            t2 = float(text_t2)
        except:
            title = _("エラー")
            mess = _("U型の定数が取得できません。") + ls
            mess += _("入力し直してください。")
            self.errDialog(title, mess, parent=self.dialog)
            return
        sect = calcSectionCoeff.sectionU(B, h, t1, t2)
        area = sect.A()
        Izz = sect.Iz()
        Iyy = sect.Iy()
        Jx = sect.Jx()
        Zzz = sect.Zz()
        Zyy = sect.Zy()
        self.setCoeffToEntry(area, Iyy, Izz, Jx)
        self.setCoeffToLabel(Zyy, Zzz)

    #
    #  getCalcResult
    #----------------
    def getCalcResult(self):
        """ 計算結果を取得して戻す"""
        area = self.entry_area.get_text()
        Iyy = self.entry_Iyy.get_text()
        Izz = self.entry_Izz.get_text()
        Jx = self.entry_Jx.get_text()
        return [area, Iyy, Izz, Jx]
    
    #
    #  getSettingDataFromGui
    #------------------------
    def getSettingDataFromGui(self):
        """ settingDataをGUIから取得する"""
        settingDict = {}
        #設定材料を取得
        matName = self.combo_mat.getSelectedItem()
        idx = self.comboItems.index(matName)
        selMat = self.materials[idx]
        settingDict["material"] = selMat
        #計算結果を取得
        area = self.entry_area.get_text()
        Iyy = self.entry_Iyy.get_text()
        Izz = self.entry_Izz.get_text()
        Jx = self.entry_Jx.get_text()
        Zyy = self.label_Zyy.get_text()
        Zzz = self.label_Zzz.get_text()
        settingDict["area"] = area
        settingDict["Iyy"] = Iyy
        settingDict["Izz"] = Izz
        settingDict["Jx"] = Jx
        settingDict["Zyy"] = Zyy
        settingDict["Zzz"] = Zzz
        #丸棒を取得
        d = self.entryMaru_dia.get_text()
        settingDict["round"] = {}
        settingDict["round"]["d"] = d
        #pipe
        d = self.entryPipe_d.get_text()
        t = self.entryPipe_t.get_text()
        settingDict["pipe"] = {}
        settingDict["pipe"]["d"] = d
        settingDict["pipe"]["t"] = t
        #角棒を取得
        B = self.entryKaku_B.get_text()
        h = self.entryKaku_h.get_text()
        settingDict["square"] = {}
        settingDict["square"]["B"] = B
        settingDict["square"]["h"] = h
        #中空角棒
        B = self.entryChuKaku_B.get_text()
        h = self.entryChuKaku_h.get_text()
        t = self.entryChuKaku_t.get_text()
        settingDict["squarePipe"] = {}
        settingDict["squarePipe"]["B"] = B
        settingDict["squarePipe"]["h"] = h
        settingDict["squarePipe"]["t"] = t
        #H型
        B = self.entryH_B.get_text()
        h = self.entryH_h.get_text()
        t1 = self.entryH_t1.get_text()
        t2 = self.entryH_t2.get_text()
        settingDict["H"] = {}
        settingDict["H"]["B"] = B
        settingDict["H"]["h"] = h
        settingDict["H"]["t1"] = t1
        settingDict["H"]["t2"] = t2
        #I型
        B = self.entryI_B.get_text()
        h = self.entryI_h.get_text()
        t1 = self.entryI_t1.get_text()
        t2 = self.entryI_t2.get_text()
        settingDict["I"] = {}
        settingDict["I"]["B"] = B
        settingDict["I"]["h"] = h
        settingDict["I"]["t1"] = t1
        settingDict["I"]["t2"] = t2
        #L型
        B = self.entryL_B.get_text()
        h = self.entryL_h.get_text()
        t1 = self.entryL_t1.get_text()
        t2 = self.entryL_t2.get_text()
        settingDict["L"] = {}
        settingDict["L"]["B"] = B
        settingDict["L"]["h"] = h
        settingDict["L"]["t1"] = t1
        settingDict["L"]["t2"] = t2
        #C型
        B = self.entryC_B.get_text()
        h = self.entryC_h.get_text()
        t1 = self.entryC_t1.get_text()
        t2 = self.entryC_t2.get_text()
        settingDict["C"] = {}
        settingDict["C"]["B"] = B
        settingDict["C"]["h"] = h
        settingDict["C"]["t1"] = t1
        settingDict["C"]["t2"] = t2
        #U型
        B = self.entryU_B.get_text()
        h = self.entryU_h.get_text()
        t1 = self.entryU_t1.get_text()
        t2 = self.entryU_t2.get_text()
        settingDict["U"] = {}
        settingDict["U"]["B"] = B
        settingDict["U"]["h"] = h
        settingDict["U"]["t1"] = t1
        settingDict["U"]["t2"] = t2
        return settingDict

    #
    #  getSettingDataFromFile
    #-------------------------
    def getSettingDataFromFile(self):
        """ fileからsettingDataを取得する"""
        settingDict = {}
        fileName = self.settingFile
        if os.path.exists(fileName) == False:
            return settingDict

        resKeys = self.resItems
        f = open(self.settingFile, encoding="utf-8"); lines = f.readlines(); f.close()
        lines = pyFistr.deleteComment(lines)
        for line in lines:
            if line[0] != " ":
                words = line.split()
                if len(words) >= 1:
                    key = words[0]
                    if key in resKeys:
                        val = ""
                        if len(words) >= 2:
                            val = words[1]
                        settingDict[key] = val
                    else:
                        settingDict[key] = {}
            else:
                words = line.split()
                if len(words) >= 1:
                    subKey = words[0]
                    val = ""
                    if len(words) >= 2:
                        val = words[1]
                    settingDict[key][subKey] = val
        return settingDict

    #
    #  saveSettingDataToFile
    #-------------------------
    def saveSettingDataToFile(self, settingDict):
        """ settingDataをfileに保存"""
        #file内容を作成
        resLines = []
        lines = []
        lines += ["#\n"]
        lines += ["#  calcSectionCoeffDDialog setting data\n"]
        lines += ["#\n"]
        lines += ["\n"]
        for key in settingDict.keys():
            if type(settingDict[key]) == dict:
                lines += [key + "\n"]
                for subKey in settingDict[key].keys():
                    lines += ["    " + subKey + " " + settingDict[key][subKey] + "\n"]
            elif type(settingDict[key]) == str:
                resLines += [key + " " + settingDict[key] + "\n"]
        lines += resLines
        #書き込み
        f = open(self.settingFile, "w", encoding="utf-8")
        f.writelines(lines)
        f.close()

    #
    #  setSettingDataToGui
    #-----------------------
    def setSettingDataToGui(self, settingDict):
        """ settingDataをGUIにセットする"""
        errFlag = 0
        #comboを設定
        try:
            mat = settingDict["material"]
        except:
            errFlag = 1
            mat = self.materials[0]
            settingDict["material"] = mat
        #comboBox, noteBookを設定
        idx = list(self.materials).index(mat)
        self.combo_mat.selectItem(idx)
        self.note_mat.set_current_page(idx)
        #計算結果を取得
        try:
            area = settingDict["area"]
            Iyy = settingDict["Iyy"]
            Izz = settingDict["Izz"]
            Jx = settingDict["Jx"]
            Zyy = settingDict["Zyy"]
            Zzz = settingDict["Zzz"]
        except:
            errFlag = 1
            area, Iyy, Izz, Jx, Zyy, Zzz = "", "", "", "", "", ""
            settingDict["area"] = ""
            settingDict["Iyy"] = ""
            settingDict["Izz"] = ""
            settingDict["Jx"] = ""
            settingDict["Zyy"] = ""
            settingDict["Zzz"] = ""
        self.entry_area.set_text(area)
        self.entry_Iyy.set_text(Iyy)
        self.entry_Izz.set_text(Izz)
        self.entry_Jx.set_text(Jx)
        self.label_Zyy.set_text(Zyy)
        self.label_Zzz.set_text(Zzz)
        #丸棒を取得
        try:
            d = settingDict["round"]["d"]
        except:
            errFlag = 1
            d = ""
            settingDict["round"] = {"d":""}
        self.entryMaru_dia.set_text(d)
        #pipe
        try:
            d = settingDict["pipe"]["d"]
            t = settingDict["pipe"]["t"]
        except:
            errFlag = 1
            d, t = "", ""
            settingDict["pipe"] = {"d":"", "t":""}
        self.entryPipe_d.set_text(d)
        self.entryPipe_t.set_text(t)
        #角棒を取得
        try:
            B = settingDict["square"]["B"]
            h = settingDict["square"]["h"]
        except:
            errFlag = 1
            B, h = "", ""
            settingDict["square"] = {"B":"", "h":""}
        self.entryKaku_B.set_text(B)
        self.entryKaku_h.set_text(h)
        #中空角棒
        try:
            B = settingDict["squarePipe"]["B"]
            h = settingDict["squarePipe"]["h"]
            t = settingDict["squarePipe"]["t"]
        except:
            errFlag = 1
            B, h, t = "", "", ""
            settingDict["squarePipe"] = {"B":"", "h":"", "t":""}
        self.entryChuKaku_B.set_text(B)
        self.entryChuKaku_h.set_text(h)
        self.entryChuKaku_t.set_text(t)
        #H型
        try:
            B = settingDict["H"]["B"]
            h = settingDict["H"]["h"]
            t1 = settingDict["H"]["t1"]
            t2 = settingDict["H"]["t2"]
        except:
            errFlag = 1
            B, h, t1, t2 = "", "", "", ""
            settingDict["H"] = {"B":"", "h":"", "t1":"", "t2":""}
        self.entryH_B.set_text(B)
        self.entryH_h.set_text(h)
        self.entryH_t1.set_text(t1)
        self.entryH_t2.set_text(t2)
        #I型
        try:
            B = settingDict["I"]["B"]
            h = settingDict["I"]["h"]
            t1 = settingDict["I"]["t1"]
            t2 = settingDict["I"]["t2"]
        except:
            errFlag = 1
            B, h, t1, t2 = "", "", "", ""
            settingDict["I"] = {"B":"", "h":"", "t1":"", "t2":""}
        self.entryI_B.set_text(B)
        self.entryI_h.set_text(h)
        self.entryI_t1.set_text(t1)
        self.entryI_t2.set_text(t2)
        #L型
        try:
            B = settingDict["L"]["B"]
            h = settingDict["L"]["h"]
            t1 = settingDict["L"]["t1"]
            t2 = settingDict["L"]["t2"]
        except:
            errFlag = 1
            B, h, t1, t2 = "", "", "", ""
            settingDict["L"] = {"B":"", "h":"", "t1":"", "t2":""}
        self.entryL_B.set_text(B)
        self.entryL_h.set_text(h)
        self.entryL_t1.set_text(t1)
        self.entryL_t2.set_text(t2)
        #C型
        try:
            B = settingDict["C"]["B"]
            h = settingDict["C"]["h"]
            t1 = settingDict["C"]["t1"]
            t2 = settingDict["C"]["t2"]
        except:
            errFlag = 1
            B, h, t1, t2 = "", "", "", ""
            settingDict["C"] = {"B":"", "h":"", "t1":"", "t2":""}
        self.entryC_B.set_text(B)
        self.entryC_h.set_text(h)
        self.entryC_t1.set_text(t1)
        self.entryC_t2.set_text(t2)
        #U型
        try:
            B = settingDict["U"]["B"]
            h = settingDict["U"]["h"]
            t1 = settingDict["U"]["t1"]
            t2 = settingDict["U"]["t2"]
        except:
            errFlag = 1
            B, h, t1, t2 = "", "", "", ""
            settingDict["U"] = {"B":"", "h":"", "t1":"", "t2":""}
        self.entryU_B.set_text(B)
        self.entryU_h.set_text(h)
        self.entryU_t1.set_text(t1)
        self.entryU_t2.set_text(t2)
        #errorチェック
        if errFlag == 1:
            #辞書を保存する
            self.saveSettingDataToFile(settingDict)
                            
    def errDialog(self, title, message, parent=None):
        dialog = universalDialogs.errDialog(title, message, parent)
        dialog.show()

#-------------------------------------
#  calcSectionCoeffClass class
#-------------------------------------
class calcSectionCoeffClass:
    """ beamの断面の定数を取得するdialog表示、定数取得。
    
    Args:
        funcName (object):  断面の定数取得後の処理関数
        funcArg (str):      処理関数の引数"""
    
    def __init__(self, funcName, funcArgs, parent=None):
        self.funcName = funcName
        self.funcArgs = funcArgs
        self.dialog = dialogApp(parent)
        self.dialog.button_cancel.connect("clicked", self.onCancel)
        self.dialog.button_apply.connect("clicked", self.onApply)
        self.dialog.combo_mat_org.connect("changed", self.onChangeMat)
        self.dialog.button_maru.connect("clicked", self.onMaruCalc)
        self.dialog.button_pipe.connect("clicked", self.onPipeCalc)
        self.dialog.button_kaku.connect("clicked", self.onKakuCalc)
        self.dialog.button_chuKaku.connect("clicked", self.onChuKakuCalc)
        self.dialog.button_H.connect("clicked", self.onHCalc)
        self.dialog.button_I.connect("clicked", self.onICalc)
        self.dialog.button_L.connect("clicked", self.onLCalc)
        self.dialog.button_C.connect("clicked", self.onCCalc)
        self.dialog.button_U.connect("clicked", self.onUCalc)

    def show(self):
        self.dialog.show()

    def close(self):
        self.dialog.close()

    def onCancel(self, event):
        self.close()

    def onApply(self, event):
        self.dialog.stat = self.dialog.getCalcResult()
        settingDict = self.dialog.getSettingDataFromGui()
        self.dialog.saveSettingDataToFile(settingDict)
        self.close()
        stat = self.dialog.stat
        egrp, path, item = self.funcArgs
        self.funcName(stat, egrp, path, item)

    def onChangeMat(self, event):
        self.dialog.changePage()

    def onMaruCalc(self, event):        #丸棒
        self.dialog.calcMaru()
    def onPipeCalc(self, event):        #パイプ
        self.dialog.calcPipe()
    def onKakuCalc(self, event):        #角棒
        self.dialog.calcKaku()
    def onChuKakuCalc(self, event):     #中空角棒
        self.dialog.calcChuKaku()
    def onHCalc(self, event):           #H型鋼
        self.dialog.calcH()
    def onICalc(self, event):           #I型鋼
        self.dialog.calcI()
    def onLCalc(self, event):           #L型鋼
        self.dialog.calcL()
    def onCCalc(self, event):           #C型鋼
        self.dialog.calcC()
    def onUCalc(self, event):           #U型
        self.dialog.calcU()


if __name__ == "__main__":
    import gettext
    gettext.install("app")
    _ = gettext.gettext

    dialog = dialogApp()
    stat = dialog.show()
