#!/usr/bin/python3
#  coding: utf-8
#
#   GtkParts.py
#
#       Gtk+3のGUI部品の設定用関数を準備
#
#   20/01/08    新規作成
#      06/30    path名表記の"/"をos.sepに修正
#      10/09    comboBox classを追加
#      12/16    selectDirs,unselectAllDir:追加
#   21/06/29    collapseChildDir:追加
#      06/30    isExpandedDir:追加
#      07/02    cursorUp, cursorDown:追加
#               getItemsFromTreeData:itemDirがrootの場合を追加
#   22/04/05    comboBoxText.setData:データセット時にentryを
#               クリアする様に修正。
#      07/30    addTreeNodes:バグ修正。
#      10/11    getChildNames:追加
#   23/03/18    treeList.selectItems,selectItemNos:選択cellをscrollする
#               事を追加。
#


import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GdkPixbuf

import os, glob


#--------------------
#  treeWidget class
#--------------------
class treeWidget:

    def __init__(self, treeView):
        self.treeView = treeView
        self.dataDict = {}      #treeの元データ辞書
        self.treeData = []      #treeのnodeデータ(dirのdetaのみ)
        self.itemDict = {}      #nodeデータの辞書
        self.COL_DIR = 0        #dir部（定義のみ。内部で検索する）

    #
    #  defineTreeView
    #-----------------
    def defineTreeView(self, headers, itemDefines, colWidthes=None, multi=True):
        """ treeのcol数を設定し、各colのheaderを設定する。
        
        Args:
            headers (list(str)) :各列のheader名をlist形式で与える。
                                :col数は、header数で設定する。
            itemDefines (list)  :itemの定義(iconは、col数に含まず)
                                :例 [icon, str, icon, str...]
            colWidthes (list)   :colWidthを与える。
                                :例 [20, None,...]
            multi (bool)        :multiSelectionの設定"""
        #treeStoreを定義し、treeViewにセット
        self.treeStore = Gtk.TreeStore(*itemDefines)
        self.treeView.set_model(model=self.treeStore)
        #multiSelectionの設定
        if multi == True:
            selection = self.treeView.get_selection()
            selection.set_mode(Gtk.SelectionMode.MULTIPLE)
        #iconとstrを設定
        cellRenderer = Gtk.CellRendererText()
        iconRenderer = Gtk.CellRendererPixbuf()
        nItem = 0
        nHead = 0
        while nItem < len(itemDefines) and nHead < len(headers):
            tvcol = Gtk.TreeViewColumn(headers[nHead])
            if itemDefines[nItem] != str:
                tvcol.pack_start(iconRenderer, False)
                while itemDefines[nItem] != str:
                    tvcol.add_attribute(iconRenderer, "pixbuf", nItem)
                    nItem += 1
                tvcol.pack_start(cellRenderer, False)
                tvcol.add_attribute(cellRenderer, "text", nItem)
                nItem += 1
            else:
                tvcol.pack_start(cellRenderer, False)
                tvcol.add_attribute(cellRenderer, "text", nItem)
                nItem += 1
            self.treeView.append_column(tvcol)
            nHead += 1
        #fontのweight（BOLD等）を設定
        if nItem < len(itemDefines):
            tvcol = self.treeView.get_column(0)
            tvcol.add_attribute(cellRenderer, "weight", nItem)
            nItem += 1
        #fontColorを設定
        if nItem < len(itemDefines):
            tvcol.add_attribute(cellRenderer, "foreground", nItem)
        #colWidthを設定
        if colWidthes == None:
            return
        for i in range(len(colWidthes)):
            colWidth = colWidthes[i]
            if type(colWidth) == int:
                tvcol = self.treeView.get_column(i)
                tvcol.set_fixed_width(colWidth)
                tvcol.set_resizable(True)
        #itemDefines内のdir名の場所（COL_DIR）を設定
        if itemDefines[self.COL_DIR] != str:
            for i in range(self.COL_DIR, len(itemDefines)):
                if itemDefines[i] == str:
                    self.COL_DIR = i
                    break
        #変数を設定
        self.itemDefines = itemDefines
        self.headers = headers

    #
    #  expandToCurrDir
    #------------------
    def expandToCurrDir(self, currDir):
        expDir = os.path.dirname(currDir)
        path = self.getPathInTreeView(expDir)
        self.treeView.expand_to_path(path)

    #
    #  expandChildDir
    #-----------------
    def expandChildDir(self, parentDir):
        """ treeを開く（collapseの反対）"""
        path = self.getPathInTreeView(parentDir)
        self.treeView.expand_to_path(path)

    #
    #  collapseChildDir
    #-------------------
    def collapseChildDir(self, parentDir):
        """ treeを閉じる（expandの反対）"""
        path = self.getPathInTreeView(parentDir)
        self.treeView.collapse_row(path)

    #
    #  isExpandedDir
    #-----------------
    def isExpandedDir(self, selDir):
        """ 指定のdirが展開されているか確認する"""
        path = self.getPathInTreeView(selDir)
        return self.isExpandedPath(path)

    #
    # isExpandedPath
    #----------------
    def isExpandedPath(self, path):
        return self.treeView.row_expanded(path)

    #
    #  selectDir
    #------------
    def selectDir(self, selDir):
        """ selDirを選択し、scrollして表示する。"""
        treeSelection = self.treeView.get_selection()
        path = self.getPathInTreeView(selDir)
        treeSelection.select_path(path)
        self.treeView.scroll_to_cell(path)

    #
    #  selectDirs
    #-------------
    def selectDirs(self, selDirs):
        """ 複数のdirを選択する。scrollはしない。"""
        self.unselectAllDir()
        treeSelection = self.treeView.get_selection()
        for selDir in selDirs:
            path = self.getPathInTreeView(selDir)
            treeSelection.select_path(path)

    #
    #  unselectAllDir
    #-----------------
    def unselectAllDir(self):
        """ 全項目を非選択にする"""
        treeSelection = self.treeView.get_selection()
        treeSelection.unselect_all()

    #
    #  cursorUp
    #-----------
    def cursorUp(self, path):
        """ pathで指定したcursorを一段上げたpathを返す"""
        selIter = self.treeStore.get_iter(path)
        if selIter == None:
            #存在しないpathは直ぐに戻る
            return path
        #cursorを一段上げる
        if path[-1] == 0:
            parentIter = self.treeStore.iter_parent(selIter)
            if parentIter == None:
                newIter = selIter
            else:
                newIter = parentIter
        else:
            #一段上げる。閉じたpathまで繰り返す
            preIter = self.treeStore.iter_previous(selIter)
            while True:
                prePath = self.treeStore.get_path(preIter)
                if self.isExpandedPath(prePath) == False:
                    #pathが閉じている場合
                    newIter = preIter
                    break
                parentIter = preIter
                nChildren = self.treeStore.iter_n_children(parentIter)
                lastIter = self.treeStore.iter_nth_child(parentIter, nChildren-1)
                preIter = lastIter
        newPath = self.treeStore.get_path(newIter)
        return newPath

    #
    #  cursorDown
    #-------------
    def cursorDown(self, path):
        """ pathで指定したcursorを一段下げたpathを返す"""
        selIter = self.treeStore.get_iter(path)
        if selIter == None:
            return path
        savePath = path
        #cursorを一段下げる
        if self.isExpandedPath(path) == False:
            #指定pathが展開されていない
            while True:
                parentIter = self.treeStore.iter_parent(selIter)
                nChildren = self.treeStore.iter_n_children(parentIter)
                if path[-1] == nChildren - 1:
                    #最後のiterは、親の次のiter
                    nextIter = self.treeStore.iter_next(parentIter)
                    if nextIter != None:
                        newPath = self.treeStore.get_path(nextIter)
                        break
                    path = self.treeStore.get_path(parentIter)
                    if len(path) == 1:
                        #parentIterがrootの場合
                        newPath = savePath
                        break
                    selIter = parentIter
                else:
                    #最後では無い場合は、次のiter
                    nextIter = self.treeStore.iter_next(selIter)
                    newPath = self.treeStore.get_path(nextIter)
                    break
        else:
            #指定pathが展開されている、最初の子に設定
            newIter = self.treeStore.iter_children(selIter)
            newPath = self.treeStore.get_path(newIter)
        return newPath

    #
    #  changeColFromDir
    #-------------------
    def changeColFromDir(self, selDir, col, newColCont):
        """ dirで指定したcol番目の内容をnewColContに変更する。"""
        treeIter = self.getIterInTreeView(selDir)
        self.treeStore.set_value(treeIter, col, newColCont)

    #
    #  getIterInTreeView
    #--------------------
    def getIterInTreeView(self, selDir):
        """ selDirからtreeIterを取得する"""
        path = self.getPathInTreeView(selDir)
        treeIter = self.treeStore.get_iter(path)
        return treeIter

    #
    #  getPathInTreeView
    #--------------------
    def getPathInTreeView(self, selDir):
        """ selDirからtreeのpathを取得する。
        path: 0:1:4:1 """
        rootDir = self.treeData[0]
        rootWords = rootDir.split(os.sep)
        selWords = selDir.split(os.sep)
        #pathを取得
        selConts = selWords[len(rootWords):]
        items = self.treeData[1]
        pathInt = [0]
        for word in selConts:
            for i in range(len(items)):
                if type(items[i]) == str:
                    if word == items[i]:
                        pathInt.append(i)
                        break
                else:
                    if word == items[i][0]:
                        pathInt.append(i)
                        items = items[i][1]
                        break
        path = Gtk.TreePath.new_from_indices(pathInt)
        return path

    #
    #  getSelectedDir
    #---------------
    def getSelectedDir(self):
        """ 選択したdirectoryを返す"""
        treeSelection = self.treeView.get_selection()
        _treestore, pathes = treeSelection.get_selected_rows()
        path = pathes[0]
        selDirList = self.getDirFromTreePath(path)
        selDir = os.sep.join(selDirList)
        return selDir

    #
    #  getSelectedDirs
    #------------------
    def getSelectedDirs(self):
        """ 選択されているdirectoryのlistを返す"""
        treeSelection = self.treeView.get_selection()
        _treestore, pathes = treeSelection.get_selected_rows()
        selDirs = []
        for path in pathes:
            selDirList = self.getDirFromTreePath(path)
            selDir = os.sep.join(selDirList)
            selDirs.append(selDir)
        return selDirs

    #
    #  createTreeData
    #------------------
    def createTreeData(self, itemDict):
        """ itemDict（key=directory）からTreeDataを作成する"""

        def setNameToTreeData(nest, tree, folder):
            """ folderNameをtree状に作成する。"""
            nest += 1
            if nest > 50:
                print("tree nesting is too deep!")
                return
            parentDir = folder[nest]
            n = 0
            while n < len(tree[1]):
                if type(tree[1][n]) == list:
                    if tree[1][n][0] == parentDir:
                        setNameToTreeData(nest, tree[1][n], folder)
                        break
                else:
                    if tree[1][n] == parentDir:
                        tree[1][n] = [parentDir, []]
                        setNameToTreeData(nest, tree[1][n], folder)
                        break
                n += 1
            #親folderの位置ならchildを追加する
            if nest == len(folder) - 1:
                child = folder[-1]
                tree[1].append(child)
            return tree

        folderDirs = list(itemDict.keys())
        folderDirs.sort()
        rootDir = folderDirs[0]
        treeData = [rootDir, []]
        for folder in folderDirs[1:]:
            folderWords = [rootDir] + folder[len(rootDir)+1:].split(os.sep)
            treeData = setNameToTreeData(0, treeData, folderWords)
        return treeData

    #
    #  createTreeNodes
    #------------------
    def createTreeNodes(self):
        """ treeDataからtreeViewのnodeを作成する"""
        self.treeStore.clear()
        treeData = self.treeData
        rootName = treeData[0]
        items = treeData[1]
        itemCont = self.itemDict[rootName]
        rootIter = self.treeStore.append(None, itemCont)
        #rootDir以下を作成
        self.addTreeNodes(self.treeStore, rootIter, items)

    #
    #  addTreeNodes
    #---------------
    def addTreeNodes(self, treeStore, parentIter, items):
        """ treeNodesを追加する"""
        path = treeStore.get_path(parentIter)
        parentDir = self.getDirFromTreePath(path)
        if type(items) != list:
            return
        for item in items:
            if type(item) == str:
                itemDir = os.sep.join(parentDir + [item])
                itemCont = self.itemDict[itemDir]
                newIter = treeStore.append(parentIter, itemCont)
            else:
                itemDir = os.sep.join(parentDir + [item[0]])
                itemCont = self.itemDict[itemDir]
                newIter = treeStore.append(parentIter, itemCont)
                self.addTreeNodes(treeStore, newIter, item[1])

    #
    #  getDirFromTreePath
    #----------------------
    def getDirFromTreePath(self, path):
        """ treePathからdirectoryのlistを取得して返す"""
        words = []
        if len(path) < 20:
            for i in range(len(path)):
                treeiter = self.treeStore.get_iter(path[:i+1])
                value = self.treeStore.get_value(treeiter, self.COL_DIR)
                words.append(value)
        return words

    #
    #  deleteChildrenPath
    #------------------
    def deleteChildrenPath(self, parentPath):
        """ parent以下のchildPathを全て削除する。
        辞書(dataDict, itemDict)も削除する。"""
        #辞書を削除
        #  parentDirを取得
        parentDir = os.sep.join(self.getDirFromTreePath(parentPath))
        #  辞書からparentDir以下を全て削除
        folders = list(self.itemDict.keys())
        for folderDir in folders:
            if folderDir[:len(parentDir)+1] == parentDir + os.sep:
                dummy = self.itemDict.pop(folderDir)
                dummy = self.dataDict.pop(folderDir)
        #treeWidgetからchildを削除
        parentIter = self.getIterInTreeView(parentDir)
        while self.treeStore.iter_has_child(parentIter) == True:
            childIter = self.treeStore.iter_children(parentIter)
            self.treeStore.remove(childIter)

    #
    #  getItemsFromTreeData
    #-----------------------
    def getItemsFromTreeData(self, itemDir):
        """ itemDirをたどって、treeDataのitemsを取得して返す。"""

        def getTreeNodes(items, name):
            """ treeDataのitems内からnameが持っているitemsを取得する"""
            for item in items:
                newItems = ""
                if type(item) == str:
                    newItems = name
                else:
                    if item[0] == name:
                        newItems = item[1]
                        break
            return newItems

        rootDir = self.treeData[0]
        items = self.treeData[1]
        if itemDir == rootDir:
            return items
        words = itemDir[len(rootDir)+1:].split(os.sep)
        for name in words:
            items = getTreeNodes(items, name)
        return items

    #
    #  getChildNames
    #----------------
    def getChildNames(self, parentDir):
        """ 子のnameを取得して返す。
        （parentDir内のfolder名を返す）"""
        items = self.getItemsFromTreeData(parentDir)
        if type(items) == str:
            names = [items]
        else:
            names = []
            for item in items:
                if type(item) == str:
                    names.append(item)
                else:
                    names.append(item[0])
        return names


    #---------- directory取得関連 --------------
    def getDirUnderRootToSelDir(self, rootDir, selDir, maskDirs=[]):
        """ rootDirからselDir系列のdirectoryを取得する"""
        folders = self.getFoldersBetweenDir(rootDir, selDir, maskDirs)
        childFolders = self.getChildFolders(folders, maskDirs)
        folders += childFolders
        return folders

    def getFoldersBetweenDir(self, startDir, endDir, maskDirs=[]):
        """ startDirからendDirまでのfolderを取得する"""
        folders = [startDir]
        folders += self.getFolders(startDir, maskDirs)
        if startDir == endDir:
            return folders
        currDir = ""
        for folder in folders:
            n = len(folder)
            if folder == endDir[:n]:
                currDir = folder
                break
        folders += self.getFolders(currDir, maskDirs)
        names = endDir[len(currDir)+1:].split(os.sep)
        for name in names:
            currDir += os.sep + name
            folders += self.getFolders(currDir, maskDirs)
        return folders 

    def getFolders(self, currDir, maskDirs=[]):
        """ curDirのfolderDirを取得する。
        ただし、maskDirsに含まれるfolderDirは取得しない"""
        if currDir in maskDirs:
            return []
        items = glob.glob(currDir + os.sep + "*")
        folders = list(filter(lambda x: os.path.isdir(x), items))
        return folders

    def getChildFolders(self, folders, maskDirs=[]):
        """ foldersの子foldersを取得する"""
        subFolders = []
        for folder in folders:
            subFolders += self.getFolders(folder, maskDirs)
        return subFolders

#--------------------
#  comboBox class
#--------------------
class comboBox:
    """
    comboBox用の関数。

    Attribute:
        comboBox (Gtk.Object) :Gtk.ComboBoxオブジェクト
    """
    def __init__(self, comboBox):
        self.comboBox = comboBox
        self.items = []

    def setItems(self, items):
        """ 
        comboBoxTextにデータをセットする。
        
        Args:
            items (list(str))   :comboBoxにセットするデータ
        Returns:
            None
        """
        self.setData(items)

    def setData(self, items):
        """ 
        comboBoxTextにデータをセットする。
        
        Args:
            items (list(str))   :comboBoxにセットするデータ
        Returns:
            None
        """
        self.comboBox.clear()
        listModel = Gtk.ListStore(str)
        for name in items:
            listModel.append([name])
        cell = Gtk.CellRendererText()
        self.comboBox.pack_start(cell, True)
        self.comboBox.add_attribute(cell, "text", 0)
        self.comboBox.set_model(listModel)
        self.items = items

    def getSelectedItem(self):
        """
        選択（active）している項目を取得する。

        Args:
            None
        Returns:
            item (str)  :選択している項目
        """
        return self.getSelectedData()

    def getSelectedData(self):
        """
        選択（active）している項目を取得する。

        Args:
            None
        Returns:
            item (str)  :選択している項目
        """
        nIter = self.comboBox.get_active_iter()
        if nIter != None:
            item = self.comboBox.get_model()[nIter][0]
        else:
            item = None
        return item

    def selectItem(self, selItem):
        """ 
        comboBox内のselItem(int or str)をアクティブに設定する。

        Args:
            selItem (int or str)    :アクティブにするNo or 項目名
        Returns:
            None
        """
        if type(selItem) == int:
            self.comboBox.set_active(selItem)
        else:
            listModel = self.comboBox.get_model()
            for i in range(len(listModel)):
                if listModel[i][0] == selItem:
                    self.comboBox.set_active(i)
                    break

    def unselectItem(self):
        """ 非選択の状態に設定する"""
        self.comboBox.set_active_id(None)

    def getAllItems(self):
        listModel = self.comboBox.get_model()
        items = []
        for i in range(len(listModel)):
            item = listModel[i][0]
            items.append(item)
        return items

#---------------------
#  comboBoxText class
#---------------------
class comboBoxText:
    """
    comboBoxText(entry含む)用の関数。

    Attribute:
        comboBox (Gtk.Object) :Gtk.ComboBoxTextオブジェクト
    """
    def __init__(self, comboBox):
        self.comboBox = comboBox

    def setItems(self, items):
        """ 
        comboBoxTextにデータをセットする。
        
        Args:
            items (list(str))   :comboBoxにセットするデータ
        Returns:
            None
        """
        self.setData(items)

    def setData(self, items):
        """ 
        comboBoxTextにデータをセットする。
        
        Args:
            items (list(str))   :comboBoxにセットするデータ
        Returns:
            None
        """
        self.comboBox.clear()
        listModel = Gtk.ListStore(str)
        self.comboBox.get_child().set_text("")  #entryもclearする
        for name in items:
            listModel.append([name])
        cell = Gtk.CellRendererText()
        self.comboBox.pack_start(cell, True)
        self.comboBox.add_attribute(cell, "text", 0)
        self.comboBox.set_model(listModel)

    def getSelectedItem(self):
        """
        選択（active）している項目を取得する。

        Args:
            None
        Returns:
            item (str)  :選択している項目
        """
        return self.getSelectedData()

    def getSelectedData(self):
        """
        選択（active）している項目を取得する。

        Args:
            None
        Returns:
            item (str)  :選択している項目
        """
        entry = self.comboBox.get_child()
        item = entry.get_text()
        return item

    def selectItem(self, selItem):
        """ 
        comboBox内のselItem(int or str)をアクティブに設定する。

        Args:
            selItem (int or str)    :アクティブにするNo or 項目名
        Returns:
            None
        """
        if type(selItem) == int:
            self.comboBox.set_active(selItem)
        else:
            listModel = self.comboBox.get_model()
            for i in range(len(listModel)):
                if listModel[i][0] == selItem:
                    self.comboBox.set_active(i)
                    break


#------------------
#  textView class
#------------------
class textView:
    """
    textViewを扱う場合の関数

    Attribute:
        textView (Gtk.Object)   :Gtk.TextView
    """
    def __init__(self, textView):
        self.textView = textView

    def setText(self, text):
        """ textViewにデータをセットする。
        
        Args:
            text (str) :textViewに表示させるtextデータ
        Returns:
            None
        """
        self.setData(text)

    def setData(self, cont):
        """ textViewにデータをセットする。
        
        Args:
            cont (str) :textViewに表示させるtextデータ
        Returns:
            None
        """
        buffer = self.textView.get_buffer()
        buffer.set_text(cont)

    def getText(self):
        """ textViewからデータを取得する。
        
        Args:
            None
        Returns:
            text (str)  :textView内の全データ
        """
        self.getData()

    def getData(self):
        """ textViewからデータを取得する。
        
        Args:
            None
        Returns:
            text (str)  :textView内の全データ
        """
        buffer = self.textView.get_buffer()
        (start, end) = buffer.get_bounds()
        text = buffer.get_text(start,end,False)
        return text


#-----------------
#  treeList class
#-----------------
class treeList:
    """
    treeViewをtreeListとして扱う時の関数

    Attribute:
        treeList (Gtk.Object)   :Gtk.TreeView
    """
    def __init__(self, treeList):
        self.treeList = treeList

    def create(self, multi=True, header=False, headerCont=""):
        """ treeListの初期化。treeListを作成する。
        
        Args:
            multi (bool)    :MULTIPLE（複数選択）として設定する
            header (bool)   :header(title)行を非表示設定
            headerCont (str):headerの内容
        """
        #multiSelectionの設定
        if multi == True:
            selection = self.treeList.get_selection()
            selection.set_mode(Gtk.SelectionMode.MULTIPLE)
        #titleの設定
        if header == False:
            self.treeList.set_headers_visible(False)
        #treeColumnの設定
        cell = Gtk.CellRendererText()
        col = Gtk.TreeViewColumn(headerCont)
        col.pack_start(cell, True)
        col.add_attribute(cell, "text", 0)
        self.treeList.append_column(col)
        #modelを設定
        listStore = Gtk.ListStore(str)
        self.treeList.set_model(listStore)

    def setItems(self, items):
        """ treeListにデータをセット。
        
        Args:
            items (list(str))   :treeListに表示させるデータ
        Returns:
            None
        """
        self.setData(items)

    def setData(self, items):
        """ treeListにデータをセット。
        
        Args:
            items (list(str))   :treeListに表示させるデータ
        Returns:
            None
        """
        listStore = Gtk.ListStore(str)
        for name in items:
            listStore.append([name])
        self.treeList.set_model(listStore)

    def unselectAllItems(self):
        """ itemの選択を無くす。
        
        Args:
            None"""
        self.treeList.get_selection().unselect_all()

    def selectItems(self, selItems):
        """ selItemsを選択する。
        
        Args:
            selItems (list(str))    :選択表示させるデータ
        Returns:
            None
        """
        selection = self.treeList.get_selection()
        listStore = self.treeList.get_model()
        for i in range(len(listStore)):
            item = listStore[i][0]
            if item in selItems:
                path = Gtk.TreePath(i)
                selection.select_path(path)
                self.treeList.scroll_to_cell(path)
    
    def selectItemNos(self, selNos):
        """ 指定した項目Noを選択する。
        
        Args:
            selNos (list(int))    :選択表示させる項目No
        Returns:
            None
        """
        selection = self.treeList.get_selection()
        for selNo in selNos:
            path = Gtk.TreePath(selNo)
            selection.select_path(path)
            self.treeList.scroll_to_cell(path)

    def getSelectedItems(self):
        """ 選択項目を戻す。
        
        Args:
            None
        Returns:
            selNames (list(str))    :選択している項目
        """
        selection = self.treeList.get_selection()
        (listStore, pathes) = selection.get_selected_rows()
        selNames = []
        for path in pathes:
            iter = listStore.get_iter(path)
            value = listStore.get_value(iter, 0)
            selNames.append(value)
        return selNames
    
    def getAllItems(self):
        """ treeListから全項目を取得する。
        
        Args:
            None
        Returns:
            items (list(str))   :treeList内の全データを取得する
        """
        listStore = self.treeList.get_model()
        items = []
        for i in range(len(listStore)):
            item = listStore[i][0]
            items.append(item)
        return items


#-----------------
#  treeListIcon class
#-----------------
class treeListIcon:
    """
    treeViewをtreeListとして扱う時の関数。
    項目には、アイコンが追加される。
    iconは、pixbufだが、Noneを設定すると、iconが非表示となる。

    Attribute:
        treeList (Gtk.Object)   :Gtk.TreeView
    """
    def __init__(self, treeList):
        self.treeList = treeList

    def create(self, multi=True, header=False, headerCont=""):
        """ treeListの初期化。
        defaultは、複数選択・header非表示。
        
        Args:
            multi (bool)    :MULTIPLE（複数選択）として設定する
            header (bool)   :header(title)行を非表示設定
            headerCont (str):headerの内容
        """
        #multiSelectionの設定
        if multi == True:
            selection = self.treeList.get_selection()
            selection.set_mode(Gtk.SelectionMode.MULTIPLE)
        #titleの設定
        if header == False:
            self.treeList.set_headers_visible(False)
        #modelの設定
        pixbuf = GdkPixbuf.Pixbuf
        listStore = Gtk.ListStore(pixbuf, str)
        self.treeList.set_model(listStore)
        #treeColumnの設定
        icon = Gtk.CellRendererPixbuf()
        cell = Gtk.CellRendererText()
        #cell.set_property("editable", True)
        col = Gtk.TreeViewColumn(headerCont)
        col.pack_start(icon, False)
        col.pack_start(cell, False)
        col.add_attribute(icon, "pixbuf", 0)
        col.add_attribute(cell, "text", 1)
        self.treeList.append_column(col)

    def setItems(self, items):
        """ treeListにデータをセット。
        項目は、[icon, name]のlist形式。
        
        Args:
            items (list(pixbuf, str))   :treeListに表示させるデータ
        """
        self.setData(items)

    def setData(self, items):
        """ treeListにデータをセット。
        項目は、[icon, name]のlist形式。
        
        Args:
            items (list(pixbuf, str))   :treeListに表示させるデータ
        """
        pixbuf = GdkPixbuf.Pixbuf
        listStore = Gtk.ListStore(pixbuf, str)
        for icon, name in items:
            listStore.append([icon, name])
        self.treeList.set_model(listStore)

    def getSelectedItems(self):
        """ 選択項目を戻す。
        
        Returns:
            selNames (list(pixbuf, str))    :選択している項目
        """
        selection = self.treeList.get_selection()
        (listStore, pathes) = selection.get_selected_rows()
        selNames = []
        for path in pathes:
            iter = listStore.get_iter(path)
            icon = listStore.get_value(iter, 0)
            value = listStore.get_value(iter, 1)
            selNames.append([icon, value])
        return selNames

    def unselectAllItems(self):
        """ itemの選択を無くす。
        
        Args:
            None"""
        self.treeList.get_selection().unselect_all()

    def selectItems(self, selItems):
        """ selItemsを追加して選択する。
        
        Args:
            selItems (list(pixbuf, str))    :選択表示させるデータ
        """
        selection = self.treeList.get_selection()
        listStore = self.treeList.get_model()
        for i in range(len(listStore)):
            icon = listStore[i][0]
            name = listStore[i][1]
            if [icon, name] in selItems:
                path = Gtk.TreePath(i)
                selection.select_path(path)
    
    def selectItemNames(self, selNames):
        """ itemを選択する。
        
        Args:
            selNames (list(str))    :選択表示させるitem名"""
        selection = self.treeList.get_selection()
        listStore = self.treeList.get_model()
        for i in range(len(listStore)):
            name = listStore[i][1]
            if name in selNames:
                path = Gtk.TreePath(i)
                selection.select_path(path)

    def getAllItems(self):
        """ treeListから全項目を取得する。
        
        Returns:
            items (list(pixbuf, str))   :treeList内の全データを取得する
        """
        listStore = self.treeList.get_model()
        items = []
        for i in range(len(listStore)):
            icon = listStore[i][0]
            name = listStore[i][1]
            items.append([icon, name])
        return items


#-----------------------
#  treeListCols class
#-----------------------
class treeListCols:
    """
    treeViewをtreeListとして扱う時の関数。
    複数のcolumnを持つtreeList。columnの構成は、modelで設定する。

    Attribute:
        treeList (Gtk.Object)   :Gtk.TreeView
    """

    def __init__(self, treeList):
        self.treeList = treeList

    def create(self, model, multi=True, header=False, headerConts=[], colWidthes=[]):
        """ treeListの初期化。
        defaultは、複数選択・header非表示。
        itemの構成は、modelで定義する。

        Args:
            model (tuple)   :columnの構成を定義
                            :model=(pixbuf, str, pixbuf, str)
            multi (bool)    :MULTIPLE（複数選択）として設定する
            header (bool)   :header(title)行を非表示設定
            headerConts (str):headerの内容（icon含まず）
                            :headerConts=["title", "item"]
            colWidthes (int):列幅 colWidthes=[10, 0]（icon含まず）
        """
        self.model = model
        selection = self.treeList.get_selection()
        #multiSelectionの設定
        if multi == True:
            selection = self.treeList.get_selection()
            selection.set_mode(Gtk.SelectionMode.MULTIPLE)
        #titleの設定
        if header == False:
            self.treeList.set_headers_visible(False)
        #header作成
        pixbuf = GdkPixbuf.Pixbuf
        if len(headerConts) == 0:
            headerConts = ["" for i in range(len(filter(lambda x: x!=pixbuf, model)))]
        #list作成
        icon = Gtk.CellRendererPixbuf()
        cell = Gtk.CellRendererText()
        listStore = Gtk.ListStore(*model)
        self.treeList.set_model(listStore)
        #headerを設定
        i = 0
        for cont in headerConts:
            cell = Gtk.CellRendererText()
            #cell.set_property("editable", True)
            col = Gtk.TreeViewColumn(cont)
            if model[i] == pixbuf:
                col.pack_start(icon, False)
                col.add_attribute(icon, "pixbuf", i)
                i += 1
            col.pack_start(cell, False)
            col.add_attribute(cell, "text", i)
            col.set_resizable(True)
            i += 1
            self.treeList.append_column(col)
        #列幅を設定
        for i in range(len(colWidthes)):
            if i < len(self.model) - 1:
                self.treeList.get_column(i).set_fixed_width(colWidthes[i])

    def setItems(self, items):
        """ treeListにデータをセット。
        項目は、modelで定義した内容を設定する。

        Args:
            items (list(pixbuf, str))   :treeListに表示させるデータ
        """
        listStore = self.treeList.get_model()
        listStore.clear()
        for i in range(len(items)):
            item = items[i]
            listStore.append(item)
        self.treeList.set_model(listStore)
        
    def selectItemNos(self, nums):
        """ 指定されたitemNoを選択する
        
        Args:
            nums (list(int))    :選択するitemNo"""
        selection = self.treeList.get_selection()
        for num in nums:
            path = Gtk.TreePath(num)
            selection.select_path(path)        

    def unselectAllItems(self):
        """ 全itemを非選択する"""
        self.treeList.get_selection().unselect_all()

    def getSelectedItemNos(self):
        """ 選択項目Noを戻す
        
        Returns:
            selNums (list(int))    :選択している項目No"""
        selection = self.treeList.get_selection()
        (_listStore, pathes) = selection.get_selected_rows()
        selNums = []
        for path in pathes:
            selNo = int(Gtk.TreePath.to_string(path))
            selNums.append(selNo)
        return selNums

    def getSelectedItems(self):
        """ 選択項目を戻す。
        
        Returns:
            selItems (list(pixbuf, str)) :選択している項目
        """
        selection = self.treeList.get_selection()
        (listStore, pathes) = selection.get_selected_rows()
        selItems = []
        for path in pathes:
            selNo = int(Gtk.TreePath.to_string(path))
            item = listStore[selNo]
            selItems.append(item)
        return selItems

    def getSelectedItemNames(self):
        """ 選択項目名のみを戻す。（pixbufは戻さない）
        
        Returns:
            selNames (list(str))    :選択している項目
        """
        selection = self.treeList.get_selection()
        (listStore, pathes) = selection.get_selected_rows()
        selNames = []
        for path in pathes:
            selNo = int(Gtk.TreePath.to_string(path))
            item = listStore[selNo]
            name = []
            for val in item:
                if type(val) == str:
                    #selNames.append(val)
                    name.append(val)
            selNames.append(name)
        return selNames

    def getAllItemNames(self):
        """ treeListからstrの全項目を取得する。"""
        listStore = self.treeList.get_model()
        itemNames = []
        for i in range(len(listStore)):
            item = listStore[i]
            name = []
            for val in item:
                if type(val) == str:
                    name.append(val)
            itemNames.append(name)
        return itemNames

        



            


