#!/usr/bin/python3
# coding: utf-8
""" TreeFoam用のモジュール。"""
#
#   pyTreeFoam.py
#
#   22/11/18    pyTreeFoamから抜粋
#   24/07/07    openにencoding="utf-8"を追加
#   25/08/26    getCurrMeshDir:reverse()をsort()に変更。適切なmeshDirが取得できていなかった。
#

import glob
import gzip
import os
import re
import struct
import shutil
import stringOp


#iniCase時に削除するfolder、file名のpattern
delFolderPatternsIniCase = ["processor.*", ".*\\.log.*", "log\\..*",
            "postPro*", "VTK"]
delFilePatternsIniCase = ["log\\..*|.*\\.log", ".*~",
            ".*\\.bak|.*\\.png|.*\\.obj|.*\\.jpg|.*\\.OpenFOAM|.*\\.foam|.*\\.blockMesh",
            "test.*|.*test",
            #"plotWatcher", "README.*",
            "plotWatcher",
            #".*\\.ods|.*\\.odt|.\\*.odp|.*\\.csv",
            ".*\\.ods|.*\\.odt|.\\*.odp",
            "job*"]

def isPatternMatch(patterns, name):
    """ nameがpattern(list)にmatchするかどうか
    
    Args:
        patterns (list(pattern))    :patternのlist
        name (str)                  :対象の文字列
    Returns:
        flag (bool)                 :True(match) or False(no match)"""
    flag = False
    for pattern in patterns:
        if re.match(pattern, name) != None:
            flag = True
            break
    return flag

# ----------- folder取得関連----------------
#
#　getFoldersOnlyCurrDir
#
def getFoldersOnlyCurrDir(currDir, maskDirs=[]):
    """ currDir内のfolderDir（fullPath）を取得する。
    ただし、maskDirsに含まれるDirは取得しない。"""
    if currDir in maskDirs:
        return []
    items = glob.glob(currDir+"/*")
    folders = [f for f in items if os.path.isdir(f)]
    #maskDisに含まれるDirを削除する
    #folders = list(filter(lambda x: not x in maskDirs, folders))
    return folders
    
def getFolders(currDir, maskDirs=[]):
    """ currDir内のfolderDir（fullPath）を取得する。
    ただし、maskDisに含まれるDirは取得しない。"""
    return getFoldersOnlyCurrDir(currDir, maskDirs)

def getFiles(currDir):
    """ currDir内のfile（fullPath）を取得する"""
    items = glob.glob(currDir+"/*")
    files = [f for f in items if os.path.isfile(f)]
    return files

def getFolderNames(currDir):
    """ currDir内のfolder名（baseName）を取得する"""
    folNames = [f for f in os.listdir(currDir) if os.path.isdir(currDir+os.sep+f)]
    return folNames

def getFileNames(currDir):
    """ currDir内のfile名（baseName）を取得する"""
    items = glob.glob(currDir+"/*")
    fileDirs = [f for f in items if os.path.isfile(f)]
    fileNames = list(map(lambda x: x.split(os.sep)[-1], fileDirs))
    return fileNames

def isNeedFolder(path):
    """ pathが必要なfolderかどうかチェックする。
    TreeViewに表示させるfolderかどうかチェック。"""
    name = os.path.basename(path)
    flag = False
    if not (name == "system" or name == "constant"
            or name == "solve.log.analyzed" or name[0] == "."
            or name[:len("processor")] == "processor"):
        try:
            float(name)
        except:
            flag = True
    return flag

def isCaseDir(path):
    """ OpenFOAMのcaseDirかどうかを確認する"""
    if os.path.exists(path + "/constant") == False:
        if os.path.exists(path + os.sep + "system") == False:
            return False
    return True

def getNeedFolders(currDir, maskDirs=[]):
    """ currDir直下の必要なfolderのみ取得する。
    ただし、maskDirsに含まれるdirは取得しない。"""
    folders = getFoldersOnlyCurrDir(currDir, maskDirs)
    folders = list(filter(isNeedFolder, folders))
    return folders

def getFoldersBetweenDir(startDir, endDir, maskDirs=[]):
    """ startDirからendDirまでの必要なfolderを取得する。
    ただし、maskDirsに含まれるdirは取得しない。"""
    folderDir = []
    folders = getNeedFolders(startDir, maskDirs)
    folderDir = folders[:]
    if startDir == endDir:
        return folderDir
    currDir = ""
    for folder in folderDir:
        n = len(folder)
        if folder == endDir[:n]:
            if endDir[:n+1][-1] == os.sep or folder == endDir:
                currDir = folder
                break
    folderDir += getNeedFolders(currDir, maskDirs)
    names = endDir[len(currDir)+1:].split(os.sep)
    for name in names:
        currDir += os.sep + name
        folders = getNeedFolders(currDir, maskDirs)
        folderDir += folders[:]
    folderDir = list(set(folderDir))
    return folderDir

def getChildFolders(folders, maskDirs=[]):
    """ folders(list)のchildFoldersを取得する。
    ただし、maskDirsに含まれるDirは取得しない。"""
    subFolders = []
    for folder in folders:
        subFolders += getNeedFolders(folder, maskDirs)
    return subFolders

def getFolderType(currDir):
    """ 対象のfolderを 'case(Link)', 'salome(Link)', 'folder(Link)'の区分に分ける。
    
    Args:
        None
    Returns:
        (str)   :"case", "caseLink", "salome", "salomeLink, "folder", "folderLink"
    """
    ans = "folder"
    if os.path.exists(currDir+os.sep+"system"+os.sep+"controlDict"):
        ans = "case"
    elif len(glob.glob(currDir+os.sep+"*hdf")) != 0:
        ans = "salome"
    if os.path.islink(currDir):
        ans += "Link"
    return ans

def getHddVals(currDir):
    """ HDDの容量を取得して返す"""
    if os.path.exists(currDir) == False:
        currDir = os.getenv("HOME")
    else:
        if os.path.isfile(currDir) == True:
            currDir = os.path.dirname(currDir)
    disk = shutil.disk_usage(currDir)
    total = disk.total
    free = disk.free
    return (total, free)


#  getNumVersion
#
def getNumVersion(OFversion):
    """ versionを数値(float)に変換する。
    数値変換できない場合は、「-1.0」を返す"""
    numStr = ".".join(OFversion.split(".")[:2])
    if numStr[-1].lower() == "x":
        numStr = numStr[:-1]
    try:
        numVer = float(numStr)
    except:
        numVer = -1.0
    return numVer


#  ----------
#  case class
#  ----------
#
class case:
    """ 解析case内のfileやfolderを取得する。

    Attribute:
        caseDir (str)   :対象の解析caseのdirectory
    """
    def __init__(self, caseDir=os.getcwd()):
        self.caseDir = caseDir

    def getTimeFolders(self, processor="."):
        """ 対象の解析case内のtimeFolderを取得する。
        timeFolderは、数値順にsortした状態で返す。
        
        Args:
            processor (str) :processor名（default="."）
        Returns:
            (list)  :timeFolder名のlistを返す
        """
        wkDir = os.path.abspath(self.caseDir + os.sep + processor)
        folders = getFolderNames(wkDir)
        numFolders = []
        for folder in folders:
            try:
                a = float(folder)
                numFolders.append([a, folder])
            except:
                pass
        numFolders.sort()
        timeFolders = list(map(lambda x: x[1], numFolders))
        return timeFolders

    def getCurrTimeFolder(self):
        """ 解析case内のcurrTimeFolder(計算開始時のtimeFolder)を取得する。
            processor内のtimeFolderは、無視する"""
        timeFolder = ""
        timeFolders = self.getTimeFolders()
        contDict = self.getControlDict()
        if len(timeFolders) != 0 and contDict["startFrom"] != "" and contDict["startTime"]: 
            if contDict["startFrom"] == "firstTime":
                timeFolder = timeFolders[0]
            elif contDict["startFrom"] == "latestTime":
                timeFolder = timeFolders[-1]
            else:
                timeFolder = contDict["startTime"]
                #if len(glob.glob(self.caseDir+os.sep+timeFolder)) == 0:
                if os.path.exists(self.caseDir+os.sep+timeFolder) == False:
                    timeFolder = timeFolders[0]
        return timeFolder

    def getCurrMeshDir(self, maxTime, region, meshName):
        """case内のregionからpolyMesh内のmeshName(boundary等)を探す。
        検索方法は、maxTimeから遡って、検索する。
        region0は、"."で探す。
        戻り値は、caseDirからの相対Dirを返す。
            [例:constant/polyMesh]"""
        savePath = os.getcwd()
        os.chdir(self.caseDir)
        try:
            maxTimeNum = float(maxTime)
        except:
            meshDir = "constant" + os.sep + region + os.sep + "polyMesh"
            return meshDir
        timeFlag = 0
        meshDirs = glob.glob("*/" + region + "/polyMesh/" + meshName)
        if len(meshDirs) > 0:
            ans = []
            for meshDir in meshDirs:
                dirs = meshDir.split(os.sep)
                try:
                    a = float(dirs[0])
                    if a <= maxTimeNum:
                        #ans.append(dirs[:-1])
                        ans.append([a]+dirs[:-1])
                except:
                    pass
            if len(ans) > 0:
                #ans.reverse()
                #meshDir = os.sep.join(ans[0])
                ans.sort()
                meshDir = "/".join(ans[-1][1:])
                timeFlag = 1
        if timeFlag == 0:
            meshDir = "constant" + os.sep + region + os.sep + "polyMesh"
        os.chdir(savePath)
        return meshDir

    def getFieldNames(self, timeFolder, region="."):
        """ 指定したtimeFolder内のfield名を取得する。
        圧縮・非圧縮ファイルを識別してfield名を取得する。"""
        fieldDir = self.caseDir + os.sep + timeFolder + os.sep + region
        names = getFileNames(fieldDir)
        fields = []
        for name in names:
            name = self.getOrgFileName(name)
            if name[-1] != "~" and name[0] != ".":
                #バックアップ、隠しファイルを除く
                fields.append(name)
        #同名を削除
        fields = list(set(fields))
        return fields

    def getOrgFileName(self, fileName):
        """ 非圧縮のfileNameを返す"""
        name = fileName
        if name[-3:] == ".gz":
            name = fileName[:-3]
        return name

    def checkFileName(self, fileName):
        """ 実在するfile名に合わせて、圧縮or非圧縮のfile名に設定する
        非圧縮fileが存在する場合は、非圧縮のfile名を返す。
        両方存在する場合は、非圧縮のfile名を返す。
        両方存在しない場合は、そのままfile名を返す。
        fileNameはfullPathで指定"""
        name = fileName
        if name[-3:] == ".gz":
            name = name[:-3]
        #if len(glob.glob(name)) != 0:
        if os.path.exists(name) == True:
            return name
        name += ".gz"
        #if len(glob.glob(name)) != 0:
        if os.path.exists(name) == True:
            return name
        return fileName

    def clearNullBoundaryAllFields(self, timeFolder, meshDir, region="."):
        """ 空白patchのみのboundaryFieldのクリア
        fieldを取得してクリアする。

        Args:
            timeFolder (str)    :時間folder
            meshDir (str)       :polyMeshの親dir(fullPath)
            region (str)        :region（default:「.」
        Returns:
            None"""
        fields = self.getFieldNames(timeFolder, region)
        if len(fields) == 0:
            return
        #fieldのクリア
        for field in fields:
            print(field + ": ", end="")
            self.clearNullBoundaryFields(timeFolder, meshDir, [field], region)

    def clearNullBoundaryFields(self, timeFolder, meshDir, fields, region="."):
        """ 空白patchのみboundaryFieldをクリアする。
        (各fieldのboundaryFieldのpatch名が同じ場合に、使う。)
        boundaryとboundaryField間の整合性をとる。
        boundary中の空patchは、そのまま残る。
        クリアするfieldを指定してクリアする。
        
        Args:
            timeFolder (str)    :時間フォルダ
            meshDir (str)       :polyMeshの親dir(fullPath)
            fileds (list(str))  :field名のlist
            region (str)        :region（default:「.」
        Returns:
            None"""
        polyMeshDir = meshDir
        #patch名の取得
        patches = self.getPatchNameFromBoundary(polyMeshDir)
        if len(patches) != 0:
            Pname = []
            Ptype = []
            for name in patches:
                Pname.append(name[0])
                Ptype.append(name[1])
            #boundaryFieldの作成
            (dummy, patches, Bcont) = self.getFieldsPatchNames(timeFolder, fields, region)
            #追加patchのチェック
            setFlag = 0
            print(_("以下の追加patchを「zeroGradient, empty, symmetry」で定義します。"))
            for newPatch in Pname:
                same = "no"
                for oldPatch in patches:
                    if newPatch == oldPatch:
                        same = "yes"
                        break
                if same == "no":
                    setFlag = 1
                    print("  " + newPatch)
            if setFlag == 0:
                print(_("  追加patchはありません。"))
            else:
                [dummy, newBcont] = self.setNullBoundaryField(fields, Pname, patches,
                                        Bcont, Ptype)
                error = self.makeNullBoundaryFields(timeFolder, region, fields, Pname, newBcont)
                if error[0] != "OK":
                    print(error[1])

    def setNullBoundaryField(self, fields, newPatches, patches, Bcont, Ptype):
        newBcontents = []
        f=0
        for _field in fields:
            newBcont = []
            p = 0
            for newPatch in newPatches:
                same = 0
                i=0
                for patch in patches:
                    if newPatch == patch:
                        value = self.reformTabContents(Bcont[f][i] + "\n")
                        newBcont.append(value)
                        same = 1
                        break
                    i += 1
                if same == 0:
                    value = self.getBoundaryContFromPatchType(Ptype[p])
                    #valueを整形する。
                    value = self.reformTabContents(value + "\n")  #追加
                    newBcont.append(value)
                p += 1
            newBcontents.append(newBcont)
            f += 1
        return (fields, newBcontents)

    def makeNullBoundaryFields(self, timeFolder, region, fields, patches, Bconts):
        fieldDir = os.sep.join([self.caseDir, timeFolder, region])
        #patchNameを整形（空白追加）
        patchesSp = []
        for patch in patches:
            patchesSp.append("    " + patch)
        #各fieldのboundaryFieldを作成
        i=0
        mess = ""
        while i<len(fields):
            #fieldをオープン
            fileName = fieldDir + os.sep + fields[i]
            contents = foamFile().read(fileName)
            #patchDataから「boundaryField」を作成
            newBoundary = self.makeBoundaryField(patchesSp, Bconts[i])
            #「boundaryField」を置き換える
            newContents = self.replaceBoundaryField(newBoundary, contents)
            if newContents != "":
                foamFile().write(fileName, newContents)
            else:
                print("error:", fields[i], _("が置き換えできませんでした。"))
                mess += fields[i] + ","
            i += 1
        if mess == "":
            error = ["OK", ""]
        else:
            error = ["NG", mess[:-1] + _("が書き込めません。")]
        return error

    #  reformTabContents
    #
    def reformTabContents(self, value):
        """ 行頭に空白、行中にtab追加。tab:8,20で設定"""
        tab1=8; tab2=20
        #各行に分割
        lines = value.split("\n")
        #各行のnul行を削除
        words = []
        for line in lines:
            ll = line.split()
            if len(ll) > 0:
                if ll[0] != "":
                    words.append(line)
        #各行にtab追加
        i = 0
        line = ""
        while i < len(words):
            #行を単語に分割
            wds = words[i].split()
            if len(wds) > 1:
                #2単語以上の場合、行頭に空白(tab1個)追加
                wds[0] = tab1*" " + wds[0]
                if len(wds[0]) < tab2-1:
                    #第1単語がtab2より短い場合、tab2位置まで空白追加
                    wds[0] += (tab2 - len(wds[0]))*" "
                else:
                    #長い場合は、空白1ヶを追加
                    wds[0] += " "
                #1行を作成
                line += wds[0] + " ".join(wds[1:]) + "\n"
            elif len(wds) == 1:
                #行内に1単語のみの場合
                line += tab1*" " + wds[0] + "\n"
            i += 1
        return line

    def getBoundaryContFromPatchType(self, patchCont):
        #patchCont="type empty"の為、empyのみ取得する
        words = patchCont.split()
        if len(words) >= 2:
            pType = words[1]
        else:
            pType = words[0]
        #pType別に内容を取得して返す
        if (pType == "empty;" or pType == "symmetry;" or pType == "symmetryPlane;" or
            pType == "cyclic;" or pType == "cyclicAMI;" or pType == "cyclicACMI;" or
            pType == "cyclicSlip;" or pType == "wedge;"):
            pCont = "type    " + pType
        else:
            pCont = "type    zeroGradient;"
        return pCont

    def makeBoundaryField(self, patches, patchData):
        boundary = ""
        i = 0
        lenPD = len(patchData)
        while i < lenPD:
            #patchDataが空文の場合、作成しない。
            if (patchData[i] != ""):
                patch = patches[i]
                boundary += patch + "\n    {\n"
                boundary += patchData[i]
                boundary += "    }\n"
            i += 1
        return boundary

    def replaceBoundaryField(self, newBoundary, conts):
        """ conts(field内容)から該当するpatch内容に置き換える"""
        #boundaryのポインタを取得
        p = 0
        lineOp = strOp(conts)
        pp = lineOp.getKeywordPointer("boundaryField", p)
        #boundaryの内容を取得
        (boundary, p) = lineOp.getKeywordContents("boundaryField", pp)
        #ポインタをboundaryの頭に戻す
        if boundary != "":
            p = pp
            conts = (conts[:p]
                        + "boundaryField\n{\n"
                        + newBoundary
                        + "}\n\n// ************************************************************************* //")
        else:
            conts = ""
        return conts

    def getPatchNameFromBoundary(self, meshDir):
        """ boundaryからpatch名、patch内容を取得して戻る。
        
        Args:
            meshDir (str)   :polyMeshの親dir、又はboundaryの親dir(fullPath)
        Returns:
            ans (list(list)):[[patchName, type etc, nFaces, startFace], ...]
        """
        if meshDir.split(os.sep)[-1] == "polyMesh":
            fileName = meshDir + os.sep + "boundary"
        else:
            fileName = meshDir + os.sep + "polyMesh" + os.sep + "boundary"
        #if len(glob.glob(fileName)) == 0:
        if os.path.exists(fileName) == False:
            #if len(glob.glob(fileName + ".gz")) == 0:
            if os.path.exists(fileName + ".gz") == False:
                print("error: " + fileName + " " + _("は、polyMeshのDirでは、ありません。"))
                return []
            else:
                #圧縮ファイルを展開する
                cont = foamFile().read(fileName)
                if cont.find("...") < 0:
                    #binaryが存在しないので、そのまま保存
                    f = open(fileName, "w", encoding="utf-8"); f.write(cont); f.close()
                else:
                    #binaryが存在するので、戻る
                    print("error: " + fileName + " " + _("は、polyMeshのDirでは、ありません。"))
                    return []
        ans = []
        #boundar読み込み
        f = open(fileName, encoding="utf-8"); contents = f.read(); f.close()
        contOp = strOp(contents)
        p = contOp.skipFoamFile()
        (bndConts, p) = contOp.getSmallPair(p)
        bndOp = strOp(bndConts)
        ((keyword, bndCont), p) = bndOp.getNextKeywordAndContents(0)
        #patch名毎に処理
        while keyword != "":
            bndContOp = strOp(bndCont)
            #      patchName, type etc., nFaces, strtFace
            name = ["",       "",        "",     ""]
            name[0] = keyword
            #patch内をline毎に処理
            (lineCont, pp, _kind) = bndContOp.get1line(0)
            while lineCont != "":
                words = lineCont.split()
                if len(words) > 1:
                    if words[0] == "type":
                        name[1] += " ".join(words) + "\n"
                    elif words[0] == "nFaces":
                        name[2] = words[1].split(";")[0]
                    elif words[0] == "startFace":
                        name[3] = words[1].split(";")[0]
                    else:
                        line = " ".join(words) + "\n"
                        line = line.replace(" ( ", "(")
                        line = line.replace(" ) ", ")")
                        name[1] += line
                (lineCont, pp, _kind) = bndContOp.get1line(pp)
            ans.append(name)
            ((keyword, bndCont), p) = bndOp.getNextKeywordAndContents(p)
        return ans

    def getFieldsPatchNames(self, timeFolder, fields, region="."):
        """ 指定したfieldのpatch名とboundaryConditionを取得
        
        Args:
            timeFolder (str)    :時間folder
            fields (list(str))  :field名のlist
            region (str)        :region名(default:".")
        Returns:
            fields (list(str))    :field名のlist
            patches (list(str))   :patch名のlist
            patchData (list(str)) :patch内容のlist"""
        fieldsPatch = []
        patchData = []
        fieldDir = os.sep.join([self.caseDir, timeFolder, region])
        for field in fields:
            patchesCont = []
            fileName = fieldDir + os.sep + field
            contents = foamFile().read(fileName)
            contOp = strOp(contents)
            p = 0
            (boundary, p) = contOp.getKeywordContents("boundaryField", p)
            bndOp = strOp(boundary)
            ((patch, cont), p) = bndOp.getNextKeywordAndContents(0)
            while patch != "":
                patchesCont.append([patch, cont])
                ((patch, cont), p) = bndOp.getNextKeywordAndContents(p)
            fieldsPatch.append(patchesCont)
        patches = []
        for patchC in fieldsPatch[0]:
            patches.append(patchC[0])
        for fieldPatch in fieldsPatch:
            col = []
            for patch in patches:
                col.append("")
            for fpatch in fieldPatch:
                flag = 0
                i = 0
                for patch in patches:
                    if fpatch[0] == patch:
                        col[i] = self.compressSPCR(fpatch[1])
                        flag = 1
                        break
                    i += 1
                if flag == 0:
                    patches.append(fpatch[0])
                    col.append(self.compressSPCR(fpatch[1]))
            patchData.append(col)
        return (fields, patches, patchData)

    #
    #  changePatchName
    #
    def changePatchName(self, boundaryDir, fieldDir, fields, oldPatchName, newPatchName):
        """ patch名を変更する。
        
        Args:
            boundaryDir (str)   :boundaryの親dir
            fieldDir (str)      :fieldファイルの親dir
            fields (list(str))  :対象のfield [field1, field2, ...]
            oldPatchName (str)  :修正前のpatchName
            newPatchName (str)  :修正後のpatchName
        Returns:
            error (list(str))   :(<"OK" or "NG">, error内容)
            """
        #文字種をチェック（使える文字かどうか）
        error = self.checkPatchName(newPatchName)
        if error[0] != "OK":
            return error
        #oldPatchNameがboundaryFieldにあるかどうかチェック
        error = self.IsOldPatchNameInBoundary(boundaryDir, fieldDir, fields, oldPatchName)
        if error[0] != "OK":
            return error
        #boundary中のpatchNameを修正
        error = self.changePatchNameInBoundary(boundaryDir, oldPatchName, newPatchName)
        if error[0] != "OK":
            return error
        #boundaryField中のpatchNameを修正
        error = self.changePatchNameInBoundaryField(fieldDir, fields, oldPatchName, newPatchName)
        return error

    def checkPatchName(self, patchName):
        """ patchNameの文字種をチェックする。"""
        error = ("OK", "")
        flag = 0
        nameOp = strOp(patchName)
        for chara in patchName:
            if nameOp.checkChara(chara) == False:
                flag = 1
                break
        if flag == 0:
            if chara == "/" or chara == "\\" or chara == "'" or chara == '"':
                flag = 1
        if flag == 1:
            error = ("NG", _(u"使えない文字「") + chara + _(u"」が使われています"))
        return error

    def IsOldPatchNameInBoundary(self, boundaryDir, fieldDir, fields, oldPatchName):
        """ oldPatchNameがboundaryにあるかどうかチェック"""
        error = ("OK", "")
        #boundaryのチェック
        fileName = boundaryDir + os.sep + "boundary"
        #if len(glob.glob(fileName)) != 0:
        if os.path.exists(fileName) == True:
            f = open(fileName, encoding="utf-8"); cont = f.read(); f.close()
            contOp = strOp(cont)
            #FoamFileをskip
            p = contOp.skipFoamFile()
            #boundary中にoldPatchNameがあるかどうかチェック
            (boundCont, p) = contOp.getSmallPair(p)
            boundContOp = strOp(boundCont)
            (lineCont, p, _kind) = boundContOp.get1line(0)
            (keyword, _pp) = strOp(lineCont).getKeyword(0)
            while lineCont != "" and keyword != oldPatchName:
                (lineCont, p, _kind) = boundContOp.get1line(p)
                (keyword, _pp) = strOp(lineCont).getKeyword(0)
            if keyword != oldPatchName:
                error = ("NG", _(u"boundary中に「") + oldPatchName + _(u"」が存在しません"))
                return error
        else:
            error = ("NG", fileName+_(u"が存在しません。"))
            return error
        #boundaryFieldのチェック
        for field in fields:
            fileName = fieldDir + os.sep + field
            cont = foamFile().read(fileName)
            contOp = strOp(cont)
            #FoamFileをskip
            p = contOp.skipFoamFile()
            #boundaryFieldを取得
            (boundCont, p) = contOp.getKeywordContents("boundaryField", p)
            #boundaryField中にoldPatchNameがあるかどうかチェック
            boundContOp = strOp(boundCont)
            (lineCont, p, _kind) = boundContOp.get1line(0)
            (keyword, _pp) = strOp(lineCont).getKeyword(0)
            while lineCont != "" and keyword != oldPatchName:
                (lineCont, p, _kind) = boundContOp.get1line(p)
                (keyword, _pp) = strOp(lineCont).getKeyword(0)
            if keyword != oldPatchName:
                error = ("NG", _(u"field ") + field + _(u" のboundaryField中に「") + oldPatchName + _(u"」が存在しません"))
                return error
        return error

    def changePatchNameInBoundary(self, boundaryDir, oldPatchName, newPatchName):
        """ boundary中のpatchNameを修正する"""
        error = ("OK", "")
        fileName = boundaryDir + os.sep + "boundary"
        f= open(fileName, encoding="utf-8"); cont = f.read(); f.close()
        contOp = strOp(cont)
        #FoamFileをskip
        p = contOp.skipFoamFile()
        #oldPatchNameを検索
        newCont = cont
        (bndCont, p) = contOp.getSmallPair(p)
        ps = p - len(bndCont) - 1
        bndContOp = strOp(bndCont)
        (lineCont, p, _kind) = bndContOp.get1line(0)
        (keyword, pp) = strOp(lineCont).getKeyword(0)
        while lineCont != "":
            if keyword == oldPatchName:
                newLineCont = newPatchName + lineCont[pp:]
                (newCont, _p, _keyFlag) = contOp.replace1line(ps, lineCont, newLineCont)
                break
            (lineCont, p, _kind) = bndContOp.get1line(p)
            (keyword, pp) = strOp(lineCont).getKeyword(0)
        if lineCont == "":
            error = ("NG", _(u"「") + oldPatchName + _(u"」がboundary中に存在しません"))
            return error
        f = open(fileName, "w", encoding="utf-8"); f.write(newCont); f.close()
        return error

    def changePatchNameInBoundaryField(self, fieldDir, fields, oldPatchName, newPatchName):
        """ 各field内のboundaryFieldのpatchNameを修正する。"""
        error = ("OK", "")
        for field in fields:
            fileName = fieldDir + os.sep + field
            cont = foamFile().read(fileName)
            contOp = strOp(cont)
            #FoamFileをskip
            p = contOp.skipFoamFile()
            #oldPatchNameを検索
            newCont = cont
            (lineCont, p, _kind) = contOp.get1line(p)
            (keyword, pp) = strOp(lineCont).getKeyword(0)
            while lineCont != "" and keyword != "boundaryField":
                (lineCont, p, _kind) = contOp.get1line(p)
                (keyword, pp) = strOp(lineCont).getKeyword(0)
            if keyword == "boundaryField":
                ps = p - len(lineCont) - 1
                (bndCont, p) = strOp(lineCont).getMiddlePair(0)
                bndContOp = strOp(bndCont)
                (lineCont, p, _kind) = bndContOp.get1line(0)
                (keyword, pp) = strOp(lineCont).getKeyword(0)
                while lineCont != "":
                    if keyword == oldPatchName:
                        newLineCont = newPatchName + lineCont[pp:]
                        (newCont, _p, _keyFlag) = contOp.replace1line(ps, lineCont, newLineCont)
                        break
                    (lineCont, p, _kind) = bndContOp.get1line(p)
                    (keyword, pp) = strOp(lineCont).getKeyword(0)
            foamFile().write(fileName, newCont)
        return error

    #
    #  deletePatch
    #
    def deletePatch(self, boundaryDir, fieldDir, fields, delPatch):
        """ boundary, fieldのファイルからpatchを削除する。
        
        Args:
            boundaryDir (str)   :boundaryの親dir
            fieldDir (str)      :fieldの親dir
            fields (list(str))  :fieldのlist
            delPatch (str)      :削除するpatch名
        Returns:
            error (str)     :errorMessage(「""」の場合は、error無し)
        """
        error = self.delPatchBoundary(boundaryDir, delPatch)
        if error == "":
            error = self.delPatchBoundaryFields(fieldDir, fields, delPatch)
        return error

    def delPatchBoundary(self, boundaryDir, delPatch):
        """ bondaryからpatchを削除する"""
        error = ""
        fileName = boundaryDir + os.sep + "boundary"
        #if len(glob.glob(fileName)) == 0:
        if os.path.exists(fileName) == False:
            error = fileName + _(u"が存在しません。")
            return error

        f = open(fileName, encoding="utf-8"); cont = f.read(); f.close()
        contOp = strOp(cont)
        p = contOp.skipFoamFile()
        #patch数を減らす
        (nPatch, p) = contOp.getKeyword(p)
        pst = p - (len(nPatch))             #pointerをkeywordの前に戻す
        newCont = cont[:pst]                #pointerまでを書き込む
        n = int(nPatch) - 1
        newCont += str(n) + "\n("           #patch数を書き込む
        p = pst + len(str(n))               #pointerをpatch数に設定
        #patchを削除
        (bndCont, p) = contOp.getSmallPair(p)
        bndContOp = strOp(bndCont)
        p = 0; flag = 0
        patchName = ("", "")
        while (patchName[0] != delPatch) and (p < len(bndCont)):
            pbak = p
            (patchName, p) = bndContOp.getNextKeywordAndContents(p)
            if patchName[0] == delPatch:
                newCont += bndCont[:pbak] + bndCont[p:]
                flag = 1
        if flag == 1:
            newCont += ")\n\n// ************************************************************************* //"
            f = open(fileName, "w", encoding="utf-8"); f.write(newCont); f.close()
        else:
            error = _(u"「") + delPatch + _(u"」のpatchがboundary中に存在しません")
            print(error)
        return error

    def delPatchBoundaryFields(self, fieldDir, fields, delPatch):
        """ fieldのpatchを削除する"""
        error = ""
        for field in fields:
            fileName = fieldDir + os.sep + field
            cont = foamFile().read(fileName)
            contOp = strOp(cont)
            p = contOp.skipFoamFile()
            p = contOp.getKeywordPointer("boundaryField", p)
            newCont = cont[:p]
            (bndCont, p) = contOp.getKeywordContents("boundaryField", p)
            newCont += "boundaryField\n{"
            bndContOp = strOp(bndCont)
            p = 0; flag = 0
            patchName = ("", "")
            while (patchName[0] != delPatch) and (p < len(bndCont)):
                pbak = p
                (patchName, p) = bndContOp.getNextKeywordAndContents(p)
                if patchName[0] == delPatch:
                    newCont += bndCont[:pbak] + bndCont[p:]
                    flag = 1
            if flag == 1:
                newCont += "}\n\n// ************************************************************************* //"
                foamFile().write(fileName, newCont)
            else:
                error = _(u"「") + delPatch + _(u"」のpatchが「") + field + _(u"」のboundaryField中に存在しません")
                print(error)
                break
        return error

    #
    #  clearInternalFields
    #
    def clearInternalFields(self, fieldDir, fields):
        """ internalFieldをクリアする
        
        Args:
            fieldDir (str)      :対象fieldの親dir(fullPath)
            fields (list(str))  :対象field名
        Returns:
            None"""
        if len(fields) != 0:
            print(_("internal field をクリアします..."))
            for name in fields:
                print("    " + name + "...")
                fileName = fieldDir + os.sep + name
                cont = foamFile().read(fileName)
                lineOp = strOp(cont)
                p = lineOp.skipFoamFile(0)
                n = cont[p:].find("internalField")
                if n >= 0:
                    ps = p + n
                    n = cont[ps:].find(";")
                    if n >= 0:
                        pe = ps + n + 1
                        line = cont[ps:pe]
                        newLine = self.clearInternalField(line)
                        newCont = cont[:ps] + newLine + cont[pe:]
                        foamFile().write(fileName, newCont)
            print(_("internal field をクリアしました。"))

    def clearInternalField(self, line):
        lineOp = strOp(line)
        (dummy, ps) = lineOp.getKeyword(0)
        (valForm, p) = lineOp.getKeyword(ps)
        if valForm == "nonuniform":
            valType = line[p:p+100]
            if valType.find("scalar") >= 0:
                val = "0"
            elif valType.find("vector") >= 0:
                val = "(0 0 0)"
            elif valType.find("symmTensor") >= 0:
                val = "(0 0 0 0 0 0)"
            elif valType.find("tensor") >= 0:
                val = "(0 0 0 0 0 0 0 0 0)"
            else:
                val = "0"
            newLine = line[:ps] + " uniform " + val + ";\n"
            line = newLine
        return line

    def compressSPCR(self, contents):
        """ 余分な空白と改行を削除する。"""
        lines = contents.split("\n")
        ans = []
        for line in lines:
            if line != "":
                words = line.split()
                ans.append(" ".join(words))
        ansR = "\n".join(ans)
        if len(ansR) > 0:
            if ansR[-1] == "\n":
                ansR = "\n".join(ans)[:-1]
        else:
            ansR = "\n"
        return ansR



    def isMultiRegion(self):
        """ 解析caseがmultiRegionかどうか確認する
        
        Aegs:
            None
        Returns:
            ans (bool)              :True(multiRegion) or False(normalCase)
            location (str)          :polyMeshの親folder名(time or constant)
            reginNames (list(str))  :region名のリスト"""
        regions = []
        os.chdir(self.caseDir)
        regMeshDirs = glob.glob("*/*/polyMesh/boundary")
        #timeFolder内のmeshを確認
        currtime = self.getCurrTimeFolder()
        timeMeshes = []
        location = ""
        for meshDir in regMeshDirs:
            time = meshDir.split(os.sep)[0]
            try:
                a = float(time)
                timeMeshes.append([a, meshDir])
            except:
                pass
        #timeFolder内のmeshを確認
        if len(timeMeshes) > 0:
            #locationを取得
            timeMeshes.reverse()
            for timeMesh in timeMeshes:
                time = timeMesh[0]
                if time <= float(currtime):
                    location = timeMesh[1].split(os.sep)[0]
                    numLoc = timeMesh[0]
                    break
            #regionを取得
            for timeMesh in timeMeshes:
                if timeMesh[0] == numLoc:
                    regions.append(timeMesh[1].split(os.sep)[1])
        #constantを確認
        if len(regions) == 0:
            location = "constant"
            for meshDir in regMeshDirs:
                if meshDir.split(os.sep)[0] == location:
                    regions.append(meshDir.split(os.sep)[1])
        #戻り値を設定
        ans = False
        regNames = []
        if len(regions) > 0:
            ans = True
            regNames = regions
            regNames.sort()
            print(_("folder「") + location + _("」内に "), regNames, _("のregionがあります。"))
        return [ans, location, regNames]

    #
    #  getZonesInRegionProperties
    #
    def getZonesInRegionProperties(self, OFversion):
        """ regionPropertiesからfluidRegion, solidRegionを取得する"""
        fluidRegions = []
        solidRegions = []
        fileName = self.caseDir + os.sep + "constant" + os.sep + "regionProperties"
        #if len(glob.glob(fileName)) == 0:
        if os.path.exists(fileName) == False:
            return [fluidRegions, solidRegions]
        f=open(fileName); cont=f.read(); f.close()
        #OFversionチェック
        numVer = getNumVersion(OFversion)
        #if OFversion <= "2.1.x" or OFversion[:3] == "ext":
        if (0.0 < numVer and numVer <= 2.1) or OFversion[:3] == "ext":
            #fluidRegionsを検索
            contOp = strOp(cont)
            ps = contOp.skipFoamFile()
            keyword = " "
            while (keyword != "fluidRegionNames") and (keyword != "") and (p < len(cont)):
                (keyword, p) = contOp.getKeyword(p)
            if keyword == "fluidRegionNames":
                (pairCont, p) = contOp.getSmallPair(p)
                fluidRegions = pairCont.split()
            else:
                fluidRegions = ""
            #solidRegionsを検索
            p = ps
            keyword = " "
            while (keyword != "solidRegionNames") and (keyword != "") and (p<len(cont)):
                (keyword, p) = contOp.getKeyword(p)
            if keyword == "solidRegionNames":
                (pairCont, p) = contOp.getSmallPair(p)
                solidRegions = pairCont.split()
            else:
                solidRegions = ""
        else:
            #ver 2.2.0以降
            contOp = strOp(cont)
            p = contOp.skipFoamFile()
            #regionを取得
            (keyword, p) = contOp.getKeyword(p)
            while (keyword != "" and keyword != "regions"):
                (keyword, p) = contOp.getKeyword(p)
            if keyword == "regions":
                (contents, p) = contOp.getSmallPair(p)
                contentsOp = strOp(contents)
                #fluidを取得
                (keyword, p) = contentsOp.getKeyword(0)
                while (keyword != "" and keyword != "fluid"):
                    (keyword, p) = contentsOp.getKeyword(p)
                if keyword == "fluid":
                    (regionCont, p) = contentsOp.getSmallPair(p)
                    fluidRegions = regionCont.split()
                #solidを取得
                (keyword, p) = contentsOp.getKeyword(0)
                while (keyword != "" and keyword != "solid"):
                    (keyword, p) = contentsOp.getKeyword(p)
                if keyword == "solid":
                    (regionCont, p) = contentsOp.getSmallPair(p)
                    solidRegions = regionCont.split()
        return (fluidRegions, solidRegions)

    #
    #  getSolverBC
    #    solverとBC（ascii/binary, normal/compression）を取得
    def getSolverBC(self):
        """ caseのsolverとBCをcontrolDictから取得"""
        contsDict = self.getControlDict()
        solver = contsDict["application"]
        wFormat = contsDict["writeFormat"]
        if wFormat == "binary":
            BCb = "B"
        else:
            BCb = "a"
        comp = contsDict["writeCompression"]
        if comp == "compressed" or comp == "yes" or comp == "on":
            BCc = "C"
        else:
            BCc = "n"
        BC = BCb + BCc + "P"
        return [solver, BC]

    #
    #  getNumberResult
    #
    """ 計算結果の nR, st, ed を取得"""
    def getNumberResult(self):
        folNames = getFolderNames(self.caseDir)
        #並列数を確認
        nProc = len(list(filter(lambda x: x[:len("processor")] == "processor", folNames)))
        # nProc = 0
        # for folder in folNames:
        #     if folder[:len("processor")] == "processor":
        #         nProc += 1
        #結果folder数を取得
        ans = []
        for folder in folNames:
            try:
                a = float(folder)
                ans.append(a)
            except:
                pass
        nRst = len(ans)
        if nProc > 1:
            #並列処理の計算結果を確認
            ansP = self.getTimeFolders("processor0")
            if len(ansP) > nRst:
                nRst = len(ansP)
                ans = list(map(float, ansP))
        #戻り値を設定
        ansR = ["", "", "", ""]
        if nRst != 0:
            ansR[0] = " " + str(nRst)           # nR
            ansR[1] = " " + str(min(ans))       # st
            if min(ans) != max(ans):
                ansR[2] = " " + str(max(ans))   # ed
            if nProc != 0:
                ansR[3] = str(nProc)            # nProc
        return ansR

    #
    #  getCaseCont
    #
    def getCaseCont(self):
        """ case内の solver, BCPn, nR, st, ed を取得する
        
        Args:
            None
        Returns
            caseCont (list) :[solver, BCPn, nR, st, ed]を返す"""
        [solver, BC] = self.getSolverBC()
        nRst = self.getNumberResult()
        #並列処理?
        if nRst[-1] != "":
            #並列数を追加
            BCPn = BC + nRst[-1]
        else:
            BCPn = BC
        nR=nRst[0]; st=nRst[1]; ed=nRst[2]
        caseCont = [solver, BCPn, nR, st, ed]
        return caseCont

    def getControlDict(self):
        """ controlDict内容の読み込み
        
        Args:
            None
        Returns:
            dict    :辞書形式で戻す
        """
        fileName = os.sep.join([self.caseDir, "system", "controlDict"])
        self.itemsDict = {
            "application": "",
            "startFrom": "",
            "startTime": "",
            "stopAt": "",
            "endTime": "",
            "deltaT": "",
            "writeInterval": "",
            "writeFormat": "",
            "writePrecision": "",
            "writeCompression": "",
            "runTimeModifiable": "",
            "adjustTimeStep": ""
            }
        f = open(fileName, encoding="utf-8"); cont = f.read(); f.close()
        contOp = strOp(cont)
        p = contOp.skipFoamFile()
        line = " "
        while line != "":
            (line, p, kind) = contOp.get1line(p)
            if kind == "line":
                line = line.replace(";", "")
                words = line.split()
                if len(words) >= 2:
                    if words[0] in self.itemsDict.keys():
                        self.itemsDict[words[0]] = words[1]
        return self.itemsDict
    
    def setControlDict(self, setDict):
        """ controlDictの書き込み
        
        Args:
            setDict (dict)  :設定したい項目の辞書
        Returns:
            None
        """
        fileName = os.sep.join([self.caseDir, "system", "controlDict"])
        f = open(fileName, encoding="utf-8"); cont = f.read(); f.close()
        contOp = strOp(cont)
        p = contOp.skipFoamFile()
        lineCont = " "
        while lineCont != "":
            (lineCont, p, kind) = contOp.get1line(p)
            if kind == "line":
                words = lineCont.split()
                if len(words) >= 2:
                    if words[0] in setDict.keys():
                        pp = p - len(lineCont)
                        newWord = setDict[words[0]] + ";"
                        newLineCont = lineCont.replace(words[1], newWord)
                        (cont, p, _keyword) = contOp.replace1lineKeyword(pp, newLineCont)
        fw = open(fileName, "w", encoding="utf-8"); fw.write(cont); fw.close()
        return

    def getCreateMeshData(self):
        """ メッシュ作成用データをhelyxOS_dataファイルから取得する
        
        Args:
            None
        Returns:
            dict    :辞書形式で戻す"""
        dataDict = {
            "stlDir": "",
            "surfaceAngle": "",
            "clearMesh": "",
            "parallel": "",
            "nCpu": "",
            "nSeparate": ""
            }
        helyxDataFile = "helyxOS_data"
        fileNameCase = self.caseDir + os.sep + helyxDataFile
        fileNameTree = os.sep.join([os.getenv("TreeFoamUserPath"), "data", helyxDataFile])
        #if len(glob.glob(fileNameCase)) == 0:
        if os.path.exists(fileNameCase) == False:
            #if len(glob.glob(fileNameTree)) == 0:
            if os.path.exists(fileNameTree) == False:
                #初期値を設定
                cont = "stlDir  ./model\n"
                cont += "surfaceAngle  150\n"
                cont += "clearMesh  yes\n"
                cont += "parallel no\n"
                cont += "nCpu 4\n"
                cont += "nSeparate 2 2 1\n"
                f=open(fileNameTree, "w"); f.write(cont); f.close()
                f=open(fileNameCase, "w"); f.write(cont); f.close()
            else:
                shutil.copy(fileNameTree, self.caseDir)
        f = open(fileNameCase, encoding="utf-8"); lines = f.readlines(); f.close()
        for line in lines:
            words = line.split()
            if len(words) > 1:
                if words[0] in dataDict.keys():
                    dataDict[words[0]] = " ".join(words[1:])
        return dataDict

    def setCreateMeshData(self, dataDict):
        """ メッシュ作成用データをhelyxOS_dataファイルに書き込む
        
        Args:
            dataDict (dict) :辞書形式のデータ
        Returns:
            None"""
        lines = []
        helyxDataFile = "helyxOS_data"
        fileName = self.caseDir + os.sep + helyxDataFile
        #if len(glob.glob(fileName)) == 1:
        if os.path.exists(fileName) == True:
            f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        for key in dataDict.keys():
            if dataDict[key] != "":
                newLine = key + " " + dataDict[key] + "\n"
                flag = 0
                for i in range(len(lines)):
                    words = lines[i].split()
                    if len(words) > 1:
                        if words[0] == key:
                            lines[i] = newLine
                            flag = 1
                            break
                if flag == 0:
                    lines.append(newLine)
        f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

    def getIniCaseFolders(self):
        """ case初期化時に残すfolder名(fullPath)を返す"""
        #削除する不要なfolder名を定義
        delPatterns = delFolderPatternsIniCase
        #folderを取得
        folders = getFolders(self.caseDir)
        #baseNameを抜き出し
        names = list(map(os.path.basename, folders))
        needNames = []      #必要なfolder名
        numFolders = []     #numFolder
        for name in names:
            #削除対象のname？
            if isPatternMatch(delPatterns, name) == False:
                #必要なnameの場合
                try:
                    #numFolderを取得
                    num = float(name)
                    numFolders.append([num, name])
                except:
                    #数値Folder以外の場合
                    needNames.append(name)
        #numFolderが存在するか？
        if len(numFolders) > 0:
            #存在する場合、minのnumFolderを取得
            numFolders.sort()
            numFolder = numFolders[0][1]
            needNames.append(numFolder)
        #取得した必要なfolderをfullPathに変換
        needFolders = list(map(lambda x: self.caseDir+os.sep+x, needNames))
        return needFolders

    def getIniCaseFiles(self):
        """ case初期化時に残すfolder名(fullPath)を返す"""
        files = getFiles(self.caseDir)
        names = list(map(os.path.basename, files))
        #削除対象のpatternを定義
        delPatterns = delFilePatternsIniCase
        needNames = []
        for name in names:
            if isPatternMatch(delPatterns, name) == False:
                needNames.append(name)
        needFiles = list(map(lambda x: self.caseDir+os.sep+x, needNames))
        return needFiles


#  --------------
#  foamFile class
#  --------------
class foamFile:
    """ OpenFOAMのfoamFileの読み込み、書き込み関連。
    読み書きは、圧縮、binaryにも適用。
    
    Attribute:
        nMaxLines (int) :読み込む行数(default:20)"""

    def __init__(self, nMaxLines=20):
        self.nMaxLines = nMaxLines      #読み込む行数

    def read(self, fieldName):
        """ fieldデータの読み込み。戻り値は、ascii(str)データ。
        nMaxLinesの行数まで読み込む
        
        Args:
            fieldName (str) :fullPathで指定
        Returns:
            cont (str)      :読み込み結果"""
        (ascii, data) = self.readSeparate(fieldName)
        cont = self.getFuseDataAtMaxLinesAB(fieldName, ascii, data)
        cont = cont.decode()    #結果をstrに変換
        return cont

    def readSeparate(self, fieldName):
        """ fileをascii部、データ部に分けて読み込む
        
        Args:
            fieldName (str) :fullPathで指定
        Returns:
            ascii (str)     :ascii部の内容
            data (bytes)    :データ部（ascii or binary)"""
        funcDict = {
            #  class                関数名
            "regIOobject"       : self.separateContents,
            "labelList"         : self.separateLabelListContents,
            "faceList"          : self.separateFaceListContents,
            "faceCompactList"   : self.separateFaceListContents,
            "refinementHistory" : self.separateFaceListContents,
            "vectorField"       : self.separateVectorFieldContents,
            "cellSet"           : self.separateLabelListContents,
            "faceSet"           : self.separateLabelListContents,
            "pointSet"          : self.separateLabelListContents,
            "polyBoundaryMesh"  : self.separateContents,
            "volScalarField"    : self.separateContents,
            "volVectorField"    : self.separateContents,
            "volTensorField"    : self.separateContents,
            "volSymmTensorField": self.separateContents,
            "surfaceScalarField": self.separateContents
        }
        cont = self.readFull(fieldName)
        classType = self.getClassInFoamFile(cont)
        classType = classType.decode()
        ascii = ""
        data = []
        if classType in funcDict.keys():
            (ascii, data) = funcDict[classType](cont)
        else:
            if classType[-4:] == "List" or classType[-4:] == "list":
                (ascii, data) = self.separateFaceListContents(cont)
            else:
                (ascii, data) = self.separateContents(cont)
        ascii = ascii.decode()
        return (ascii, data)
        
    #
    #  readFull
    #
    def readFull(self, fileName):
        """ fieldデータを全て読み込む（ascii、binaryの区別なく）
        
        Args:
            fileName (str)  :fullpathで指定
        Returns:
            cont (bytes)    :bytes型で取得する"""
        fileName = case().checkFileName(fileName)
        if fileName[-3:] == ".gz":
            #gzip形式でopen、read
            f = gzip.open(fileName, "rb")
            cont = f.read()
            f.close()
        else:
            #非圧縮でopen、read
            f=open(fileName, "rb")
            cont = f.read()
            f.close()
        return cont

    #
    #  write
    #
    def write(self, fileName, cont):
        """ readメソッドで読み込んだcont(ascii, dataをfuseした結果)を
        fileに書き込む。contをascii, dataに分離し、dataは、fileを再読み込み
        して全内容を再構築し、fileに書き込む。
        
        Args:
            fileName (str)  :fullPathで指定
            cont (str)      :ascii, dataをfuseした内容
        Returns:
            None"""
        allData = self.reconstFuseDataAtMaxLinesAB(fileName, cont)
        #全データの書き込み
        self.writeFull(fileName, allData)
        return

    #
    #  writeFull
    #
    def writeFull(self, fileName, cont):
        """ fieldデータを全て書き込み（ascii、binaryの区別なく）
        
        Args:
            fileName (str)  :fullPathで指定
            cont (bytes)    :書き込む内容
        Returns:
            None"""
        if type(cont) == str:
            cont = cont.encode()
        fileName = case().checkFileName(fileName)
        if fileName[-3:] == ".gz":
            #gzip形式でopen、write
            f = gzip.open(fileName, "wb", encoding="utf-8")
            f.write(cont)
            f.close()
        else:
            #非圧縮でopen、write
            f = open(fileName, "wb", encoding="utf-8")
            f.write(cont)
            f.close()
        return

    #  separateContents
    #
    def separateContents(self, cont):
        """ field内容をasciiとdata部に分ける"""
        if self.checkBinary(cont) == "ascii":
            (ascii, data, error) = self.separateAsciiContents(cont)
            if error == 1:
                cont = self.remakeDataFormat(cont, "ascii", "binary")
                (ascii, data, error) = self.separateBinaryContents(cont)
        else:
            (ascii, data, error) = self.separateBinaryContents(cont)
            if error == 1:
                cont = self.remakeDataFormat(cont, "binary", "ascii")
                (ascii, data, error) = self.separateAsciiContents(cont)
        return (ascii, data)

    # #  separateBoundary
    # #
    # def separateBoundary(self, cont):
    #     """ boundaryの内容をasciiとdata部に分ける。
    #     boundaryは、asciiとして読み込む。"""
    #     if self.checkBinary(cont) != "ascii":
    #         cont = self.remakeDataFormat(cont, "binary", "ascii")
    #     (ascii, data, error) = self.separateAsciiContents(cont)
    #     return (ascii, data)

    #  separateLabelListContents
    #
    def separateLabelListContents(self, cont):
        """ labelList形式の内容を分離する"""
        if self.checkBinary(cont) == "ascii":
            (ascii, data, error) = self.separateAsciiMultiListContents(cont, "label")
            if error == 1:
                cont = self.remakeDataFormat(cont, "ascii", "binary")
                (ascii, data, error) = self.separateBinaryMultiListContents(cont, "label")
        else:
            (ascii, data, error) = self.separateBinaryMultiListContents(cont, "label")
            if error == 1:
                cont = self.remakeDataFormat(cont, "binary", "ascii")
                (ascii, data, error) = self.separateAsciiMultiListContents(cont, "label")
        return (ascii, data)

    #  separateFaceListContents
    #
    def separateFaceListContents(self, cont):
        """ faceList形式の内容を分離する"""
        if self.checkBinary(cont) == "ascii":
            (ascii, data, error) = self.separateAsciiMultiListContents(cont, "faceList")
            if error == 1:
                cont = self.remakeDataFormat(cont, "ascii", "binary")
                (ascii, data, error) = self.separateBinaryMultiListContents(cont, "label")
        else:
            (ascii, data, error) = self.separateBinaryMultiListContents(cont, "label")
            if error == 1:
                cont = self.remakeDataFormat(cont, "binary", "ascii")
                (ascii, data, error) = self.separateAsciiMultiListContents(cont, "faceList")
        return (ascii, data)

    #  separateVectorFieldContents
    #
    def separateVectorFieldContents(self, cont):
        """ vectorField形式の内容を分離する"""
        if self.checkBinary(cont) == "ascii":
            (ascii, data, error) = self.separateAsciiMultiListContents(cont, "vector")
            if error == 1:
                cont = self.remakeDataFormat(cont, "ascii", "binary")
                (ascii, data, error) = self.separateBinaryMultiListContents(cont, "vector")
        else:
            (ascii, data, error) = self.separateBinaryMultiListContents(cont, "vector")
            if error == 1:
                cont = self.remakeDataFormat(cont, "binary", "ascii")
                (ascii, data, error) = self.separateAsciiMultiListContents(cont, "vector")
        return (ascii, data)

    #  separateAsciiContents
    #
    def separateAsciiContents(self, cont):
        """ asciiFile内容をasciiとdata部に分ける"""
        error = 0
        obj = self.getObjectInFoamFile(cont)
        ascii = ""
        data = []
        strOp = stringOp.strings(cont)
        p = strOp.skipFoamFile(0)
        ascii = cont[:p]
        n = 0
        i = 0
        while n >= 0 and error == 0:
            ps = p
            n = cont[p:].find(b" List<")
            if n >= 0:
                p += n
                (dataType, nData) = self.getDataTypeNumData(cont, p)
                #データ数を確認
                if nData == b"0":
                    #データが存在しない場合
                    #  List<scalar> 0の場合（データ数「0」の場合
                    p += 1
                    ascii += cont[ps:p]
                else:
                    #データが存在する場合
                    dataHeader = [dataType, nData]
                    n = cont[p:p+100].find(b"(")
                    if n >= 0:
                        p += n + 1
                        pe = p
                        if int(nData) < 100:
                            #データ数が少ない場合
                            (contData, p) = strOp.getSmallPair(p-1)
                            spId = obj + b"." + str(i).encode()
                            ascii += cont[ps:pe] + (b"\0" + spId + b"\0")
                            data.append([dataHeader, contData])
                            error = self.confirmAscii(contData)
                            p -= 1
                            i += 1
                        else:
                            #データ数が多い場合
                            n = cont[p:].find(b"\n)")
                            if n>=0:
                                spId = obj + b"." + str(i).encode()
                                ascii += cont[ps:pe] + (b"\0" + spId + b"\0")
                                p += n + 1
                                contData = cont[pe:p]
                                data.append([dataHeader, contData])
                                error = self.confirmAscii(contData)
                                i += 1
            #i += 1
        ascii += cont[ps:]
        return (ascii, data, error)

    #  separateBinaryContents
    #
    def separateBinaryContents(self, cont):
        """ binaryFile内容をasciiとdata部に分ける"""

        def findListPosition(cont):
            """ 'List<word>'を除く'List<'を検索する。"""
            i = 0
            while True:
                n = cont.find(b"List<", i)
                if n >= 0:
                    block = cont[n:]
                    ps = block.find(b"<") + 1
                    pe = block.find(b">")
                    dataType = block[ps:pe]
                    if dataType != b"word":
                        break
                    else:
                        i = n + 5
                else:
                    break
            return n

        error = 0
        obj = self.getObjectInFoamFile(cont)
        data = []
        lineOp = stringOp.strings(cont)
        p = lineOp.skipFoamFile(0)
        ascii = cont[:p]
        n = 0
        i = 0
        while n>=0 and error == 0:
            ps = p
            #n = cont[p:].find(b" List<")
            n = findListPosition(cont[p:])
            if n >= 0:
                p += n
                (binaryType, nData) = self.getDataTypeNumData(cont, p)
                #データ数を確認
                if nData == b"0":
                    #データが存在しない場合
                    #  List<scalar> 0の場合（データ数「0」の場合）
                    p += 1
                    ascii += cont[ps:p]
                else:
                    #データが存在する場合
                    dataHeader = [binaryType, nData]
                    n = cont[p:p+100].find(b"(")
                    if n >= 0:
                        p += n + 1
                        pe = p
                        (binary, p) = self.getBinaryData(binaryType, nData, cont, p)
                        if len(binary) != 0:
                            spId = obj + b"." + str(i).encode()
                            ascii += cont[ps:pe] + (b"\0" + spId + b"\0")
                            data.append([dataHeader, binary])
                            #errorチェック
                            if not (cont[p] == b")" or cont[p] == ord(")")):
                                error = 1
            i += 1
        ascii += cont[ps:]
        return (ascii, data, error)

    #  separateAsciiMultiListContents
    #
    def separateAsciiMultiListContents(self, cont, dataType):
        """ 複数のlist形式のascii内容を分離する"""
        error = 0
        obj = self.getObjectInFoamFile(cont)
        data = []
        lineOp = stringOp.strings(cont)
        p = lineOp.skipFoamFile(0)
        ascii = cont[:p]
        n = 0
        i = 0
        while n >= 0 and error == 0:
            ps = p
            n = cont[p:].find(b"(")
            if n >= 0:
                pp = p + n
                [nData, p] = lineOp.getBackKeyword(p+n)
                dataHeader = [dataType, nData]
                if int(nData) < 100:
                    #データ数少ない場合
                    p = pp
                    pe = pp + 1
                    [dataCont, p] = lineOp.getSmallPair(p)
                    ascii += cont[ps:pe] + (b"\0" + obj + b"." + str(i).encode() + b"\0")
                    p -= 1
                    data.append([dataHeader, dataCont])
                    error = self.confirmAscii(dataCont)
                else:
                    #データ数多い場合
                    p = pp
                    pe = pp + 1
                    ascii += cont[ps:pe] + (b"\0" + obj + b"." + str(i).encode() + b"\0")
                    ps = p + 1
                    n = cont[p:].find(b"\n)")
                    p += n + 1
                    dataCont = cont[ps:p]
                    data.append([dataHeader, dataCont])
                    error = self.confirmAscii(dataCont)
            i += 1
        ascii += cont[ps:]
        return (ascii, data, error)

    #  separateBinaryMultiListContents
    #
    def separateBinaryMultiListContents(self, cont, dataType):
        """ 複数のlist形式のbinary内容を分離する"""
        (ascii, data, error) = self.separateBinaryFaceListContents(cont, dataType)
        return (ascii, data, error)

    #  separateBinaryFaceListContents
    #
    def separateBinaryFaceListContents(self, cont, dataType):
        """ faceCompactList形式のbinary内容を分離する"""
        error = 0
        obj = self.getObjectInFoamFile(cont)
        data = []
        lineOp = stringOp.strings(cont)
        p = lineOp.skipFoamFile(0)
        ascii = cont[:p]
        n = 0
        i = 0
        while n >= 0 and error == 0:
            ps = p
            n = cont[p:].find(b"(")
            if n >= 0:
                pss = p + n
                (nData, _p) = lineOp.getBackKeyword(p+n)
                dataHeader = [dataType, nData]
                p = pss + 1
                pe = p
                (binary, p) = self.getBinaryData(dataType, nData, cont, p)
                if len(binary) != 0:
                    spId = obj + b"." + str(i).encode()
                    ascii += cont[ps:pe] + (b"\0" + spId + b"\0")
                    data.append([dataHeader, binary])
                    #errorチェック(bytes型の1文字チェック)
                    if not (cont[p] == b")" or cont[p] == ord(")")):
                        error = 1
                else:
                    ascii += cont[ps:pe]
            i += 1
        ascii += cont[ps:]
        return (ascii, data, error)

    #  reconstFuseDataAtMaxLinesAB
    #
    def reconstFuseDataAtMaxLinesAB(self, fileName, cont):
        """ nMaxLinesまで結合されたfuseDataを再構築して全データを取得"""
        #classType = checkClass(cont)
        classType = self.getClassInFoamFile(cont)
        #fieldタイプのreconst
        if (classType == "regIOobject" or
            classType == "volScalarField"     or classType == "volVectorField" or
            classType == "volSymmTensorField" or classType == "volTensorField" or
            classType == "surfaceScalarField"):
            allCont = self.reconstFuseDataAtMaxLinesField(fileName, cont)
        #listタイプのreconst
        elif (classType == "labelList" or classType == "vectorField" or
            classType == "cellSet"   or classType == "faceSet" or
            classType == "pointSet"  or classType == "faceList"):
            allCont = self.reconstFuseDataAtMaxLinesMultiList(fileName, cont)
        #faceCompactListタイプのreconst
        elif (classType == "faceCompactList" or classType == "refinementHistory"):
            allCont = self.reconstFuseDataAtMaxLinesMultiList(fileName, cont)
        #Listタイプのreconst（複数のListが存在する場合に設定）
        elif (classType[-4:] == "List" or classType[-4:] == "list"):
            allCont = self.reconstFuseDataAtMaxLinesMultiList(fileName, cont)
        #その他のreconst（fieldタイプに設定　List<**> **(** **..); ）
        else:
            allCont = self.reconstFuseDataAtMaxLinesField(fileName, cont)
        return allCont


    #  reconstFuseDataAtMaxLinesMultiList
    #
    def reconstFuseDataAtMaxLinesMultiList(self, fileName, cont):
        """ fuseDataをreconstructして元のデータに戻す
        複数のlist形式に対応"""
        cont = cont.encode()
        #separateデータの読込
        (_ascii, data) = self.readSeparate(fileName)
        #data部までskipしてdataを入れ替え
        obj = self.getObjectInFoamFile(cont)
        lineOp = stringOp.strings(cont)
        p = lineOp.skipFoamFile(0)
        #p = skipFoamFileMem(cont, 0)
        allCont = cont[:p]
        n = 0
        while n >= 0 and p < len(cont):
            #data部までskip
            ps = p
            n = cont[p:].find("\n(")
            if n >= 0:
                p += n + 2
                pks = p
                allCont += cont[ps:p]
                n = cont[pks:].find("\n)")
                p += n + 1
                ple = p
                #keyを取得
                n = cont[pks:p].rfind("...")
                if n >= 0:
                    p = pks + n
                    n = cont[pks:p].rfind("...")
                    p = pks + n
                    key = cont[p:ple].split("...")[1]
                    #recObj = key.split(".")[0]
                    #dataNo = int(key.split(".")[1])
                    recObj = ".".join(key.split(".")[:-1])
                    dataNo = int(key.split(".")[-1])
                    #dataを入れ替え
                    if obj == recObj:
                        #自身のdataに入れ替え
                        allCont += data[dataNo][1]
                        p = ple
                    else:
                        #他のdataに入れ替え
                        newFile = os.sep.join(fileName.split(os.sep)[:-1]) + os.sep + recObj
                        (_newAscii, newData) = self.readSeparate(newFile)
                        allCont += newData[dataNo][1]
                        p = ple
                else:
                    #keyが無い場合は、そのまま取得
                    allCont += cont[pks:ple]
                    p = ple
        allCont += cont[p:]
        return allCont  

    #  reconstFuseDataAtMaxLinesField
    #
    def reconstFuseDataAtMaxLinesField(self, fileName, cont):
        """ fuseDataのcontをreconstructして元のデータに戻す"""

        def skipLineEnd(cont):
            """ 行末「);」までskip"""
            ne = cont.find(b";")
            n = cont[:ne].rfind(b")")
            return n

        #編集内容の文字コードを「bytes」に設定
        if type(cont) == str:
            cont = cont.encode()
        #separateデータを読込
        (_ascii, data) = self.readSeparate(fileName)
        #data部までskipしてdataを入れ替え
        obj = self.getObjectInFoamFile(cont)
        allCont = b""
        n = 0
        p = 0
        while n >= 0 and p < len(cont):
            #data部までskip
            ps = p
            n = cont[p:].find(b" List<")
            if n >= 0:
                p += n
                (_dataType, nData) = self.getDataTypeNumData(cont, p)
                #データ数を確認
                if nData == b"0":
                    #データが存在しない場合
                    #  List<scalar> 0の場合
                    p += 1
                    allCont += cont[ps:p]
                else:
                    #データが存在する場合
                    n = cont[p:].find(b"(")
                    if n >= 0:
                        p += n +1
                        pks = p
                        allCont += cont[ps:p]
                        n = skipLineEnd(cont[pks:])
                        p += n
                        ple = p
                        #keyを取得
                        n = cont[pks:p].rfind(b"...")
                        if n >= 0:
                            p = pks + n
                            n = cont[pks:p].rfind(b"...")
                            p = pks + n
                            key = cont[p:ple].split(b"...")[1]
                            recObj = b".".join(key.split(b".")[:-1])
                            dataNo = int(key.split(b".")[-1])
                            #dataを入れ替え
                            if obj == recObj:
                                #自身のdataに入れ替え
                                allCont += data[dataNo][1]
                                p = ple
                                n = 0
                            else:
                                #他のdataに入れ替え
                                newFile = os.sep.join(fileName.split(os.sep)[:-1]) + os.sep + recObj
                                (_newAscii, newData) = self.readSeparate(newFile)
                                allCont += newData[dataNo][1]
                                p = ple
                                n = 0  
                        else:
                            #keyが無い場合は、そのまま取得
                            allCont += cont[pks:ple]
                            p = ple
                            n = 0
        allCont += cont[p:]
        return allCont

    #  getFuseDataAtMaxLinesAB
    #
    def getFuseDataAtMaxLinesAB(self, fileName, ascii, data):
        """ ascii binary dataをnMaxLinesまで結合して返す"""
        #obj = checkObject(ascii)
        if type(ascii) == str:
            ascii = ascii.encode()
        obj = self.getObjectInFoamFile(ascii)
        cont = b""
        #i = 0
        p = 0
        n = 0
        while n >= 0:
            #separatorまでskip
            ps = p
            n = ascii[p:].find(b"\0")
            if n >= 0:
                #separatorまでのasciiを取得
                p += n
                cont += ascii[ps:p]
                #次のseparatorまでskip
                p += 1
                pks = p
                n = ascii[p:].find(b"\0")
                if n >= 0:
                    #dataObjectを取得
                    p += n
                    key = ascii[pks:p]
                    recObj = b".".join(key.split(b".")[:-1])
                    dataNo = int(key.split(b".")[-1])
                    #dataObjectのdataを取得
                    if obj == recObj:
                        #自身のdataをnMaxLinesまで取得
                        sourceData = data[dataNo]
                    else:
                        #他のdataを取得
                        fileDir = os.sep.join(fileName.split(os.sep)[:-1])
                        (_newAscii, newData) = self.readSeparate(fileDir + os.sep + recObj)
                        sourceData = newData[dataNo]
                    #dataをsourceから取得し、set
                    if self.checkBinary(ascii) == "binary":
                        #binary
                        #smallData = self.getDataAtMaxLinesBinary(data[dataNo])
                        smallData = self.getDataAtMaxLinesBinary(sourceData)
                        cont += smallData + b"..." + key + b"...\n"
                    else:
                        #ascii
                        #smallData = self.getDataAtMaxLinesAscii(data[dataNo])
                        smallData = self.getDataAtMaxLinesAscii(sourceData)
                        if smallData[-3:] == b"...":
                            cont += smallData + key + b"...\n"
                        else:
                            cont += smallData
                    p += 1
        cont += ascii[p:]
        return cont

    #  getDataAtMaxLinesAscii
    #
    def getDataAtMaxLinesAscii(self, data):
        """ ascii data をnMxLinesまで取得"""

        def getCRLine(cont, p):
            """ CRまでの1行を取得"""
            ps = p
            n = cont[ps:].find(b"\n")
            if n >= 0:
                p += n + 1
                line = cont[ps:p]
            else:
                line = cont[ps:]
                p = len(cont)
            return (line, p)

        asciiData = data[1]
        if self.nMaxLines == -1:
            self.nMaxLines = len(asciiData)
        i = 0
        cont = b""
        p = 0
        while i <= self.nMaxLines and p < len(asciiData):
            [line , p] = getCRLine(asciiData, p)
            cont += line
            i += 1
        if i > self.nMaxLines:
            cont += b"..."
        return cont

    #  getDataAtMaxLinesBinary
    #
    def getDataAtMaxLinesBinary(self, data):
        """ binary dataをnMxLinesまで取得"""
        (nNum, nByte, valType) = self.getDataFormat(data[0][0])
        #binaryの読み込みの為、valTypeをbinaryに変換する
        valType = valType.encode()
        if self.nMaxLines == -1:
            convByte = len(data[1])
        else:
            convByte = nNum * nByte * self.nMaxLines

        if len(data[1]) > convByte:
            #nMaxLinesまで変換
            convData = data[1][:convByte]
        else:
            #全データを変換
            convData = data[1]
        n = int(len(convData)/nByte)        #valTypeのデータ数
        numData = struct.unpack(str(n).encode()+valType, convData)
        if nNum == 1:
            stLine = b""; edLine = b""
        else:
            stLine = b"("; edLine = b")"
        cont = b"\n"
        i = 0
        while i<len(numData):
            cont += stLine
            addCont = b""
            for num in numData[i:i+nNum]:
                addCont += str(num).encode() + b" "
            cont += addCont[:-1] + edLine + b"\n"
            i += nNum
        return cont

    #
    #  structBinData
    #
    def structBinData(self, data):
        """ binaryDataから数値を取得"""
        dataType = data[0][0]
        _nData = data[0][1]
        binData = data[1]
        #valTypeを取得
        (_nNum, nByte, valType) = self.getDataFormat(dataType)
        #数値変換する数
        num = int(len(binData) / nByte)
        #数値変換
        numData = struct.unpack(str(num)+valType, binData)
        return numData

    #  remakeDataFormat
    #
    def remakeDataFormat(self, ascii, form, newForm):
        """ dataのformatを書き換え"""
        lineOp = stringOp.strings(ascii)
        p = lineOp.getKeywordPointer(form, 0)
        if type(newForm) == str:
            newForm = newForm.encode()
        if p >= 0:
            newCont = ascii[:p] + newForm + ascii[p+len(form):]
            ascii = newCont
        return ascii

    def getClassInFoamFile(self, cont):
        """ FoamFileからclassを取得する"""
        lineOp = stringOp.strings(cont)
        (foamFile, _p) = lineOp.getKeywordContents("FoamFile", 0)
        #foamFile = foamFile.decode()
        lines = foamFile.split(b";")
        ans = b""
        for line in lines:
            words = line.split()
            if len(words) >= 2:
                if words[0] == b"class":
                    ans = words[1]
                    break
        return ans

    def getObjectInFoamFile(self, cont):
        """ FoamFileからobjectを取得する"""
        strOp = stringOp.strings(cont)
        (foamFile, _p) = strOp.getKeywordContents("FoamFile", 0)
        lines = foamFile.split(b";")
        ans = ""
        for line in lines:
            items = line.split()
            if len(items) >= 2:
                if items[0] == b"object":
                    ans = items[1]
                    break
        return ans

    def getDataTypeNumData(self, cont, p):
        """ dataTypeとnDataを取得"""
        block = cont[p:p+100]
        ps = block.find(b"<") + 1
        pe = block.find(b">")
        dataType = block[ps:pe]
        lineOp = stringOp.strings(block)
        (nData, p) = lineOp.getKeyword(pe+1)
        #[nData, p] = getKeywordMem(block, pe+1)
        return (dataType, nData)

    #  getBinaryData
    #
    def getBinaryData(self, binaryType, nData, cont, p):
        """ typeに応じてbinaryを取得"""
        (nNum, nByte, _valType) = self.getDataFormat(binaryType)
        pe = p + (nNum * int(nData) * nByte)
        dataCont = cont[p:pe]
        return (dataCont, pe)    

    #  getDataFormat
    #
    def getDataFormat(self, dataFormat):
        """ dataのformatを取得
        戻り値：[nNum, nByte, valType]"""
        dataDict = {
        #  type     nNum nByte valtype
        "scalar":     [1,  8,   "d"],
        "vector":     [3,  8,   "d"],
        "symmTensor": [6,  8,   "d"],
        "tensor":     [9,  8,   "d"],
        "label":      [1,  4,   "i"],
        "faceList":   [1,  4,   "i"],
        "bool":       [1,  1,   "b"]
        }
        if type(dataFormat) == bytes:
            dataFormat = dataFormat.decode()
        if dataFormat in dataDict.keys():
            ans = dataDict[dataFormat]
        else:
            ans = [0, 0, b""]
        return ans

    def confirmAscii(self, cont):
        """ dataがasciiかどうかチェック"""
        error = 0
        data = cont[:10]
        for chara in data:
            if chara < ord(chr(0x08)) or chara > ord(chr(0x80)):
                error = 1
                break
        return error

    def checkBinary(self, cont):
        """ データがbinaryかどうかをチェック"""
        lineOp = stringOp.strings(cont)
        (foamFile, _p) = lineOp.getKeywordContents("FoamFile", 0)
        foamFile = foamFile.decode()
        lines = foamFile.split(";")
        ans = "ascii"
        for line in lines:
            items = line.split()
            if len(items) >= 2:
                if items[0] == "format":
                    ans = items[1]
                    break
        return ans


#  -----------
#  strOp class
#  -----------
class strOp:
    """ 文字列操作。文字列は、str型で検索操作する。結果もstr型文字列が戻る。
    対象の文字列が変わる操作（replace、delete関連）は、対象の文字列も変える。

    Attribute:
        line (str)  :対象の文字列"""

    def __init__(self, line):
        if type(line) != str:
            line = line.decode()
        self.line = line

    #
    #  skipFoamFile
    #
    def skipFoamFile(self, p=0):
        """ FoamFileブロックをskipしたpointerを返す。
        FoamFileが存在しない場合は、skipしない。"""
        (_line, pp) = self.getKeywordContents("FoamFile", p)
        #FoamFileの有無チェック
        if pp >= len(self.line):
            #FoamFileが存在しない場合
            pp = p
        return pp

    #
    #  replace1line
    #
    def replace1line(self, ps, oldLine, newLine):
        """ line[ps:]以降を検索し、のoldLineをnewLineに入れ替える。
        置き換え後は、元のself.lineの内容も置き換える。
        戻り値:[newCont, p, keyword]。
        失敗した場合、keywordが「""」になる。
        pointerは、更新される。"""
        #置き換える
        newCont = self.line[:ps] + self.line[ps:].replace(oldLine, newLine, 1)
        #置き換え結果を確認
        p = newCont[ps:].find(newLine)
        if p >= 0:
            #置き換え成功
            p = ps + p + len(newLine)
            keyword = newLine
            self.line = newCont
        else:
            #置き換え失敗
            p = len(newCont)
            keyword = ""
        return (newCont, p, keyword)

    #
    #  replace1lineKeyword
    #
    def replace1lineKeyword(self, ps, newLine):
        """ line[ps:]を検索してnewLineに置き換える。検索は、newLineで使われているkeywordを検索する。
        置き換え後は、元のself.lineも置き換える。
        戻り値:(newCont, p, keyword)。
        失敗した場合、keywordが「""」になる。"""
        (keyword, _pp) = strOp(newLine).getKeyword(0)
        (lineCont, p, _kind) = self.get1line(ps)
        while lineCont != "":
            (tempKey, _pp) = strOp(lineCont).getKeyword(0)
            if tempKey == keyword:
                oldLine = lineCont
                break
            ps = p
            (lineCont, p, _kind) = self.get1line(ps)
        if lineCont == "":
            return ("", -1, "")
        else:
            (newCont, p, keyword) = self.replace1line(ps, oldLine, newLine)
            self.line = newCont
            return (newCont, p, keyword)

    #
    #  get1line
    #
    def get1line(self, p):
        """ 指定したpointerから1行を取得する
        pointerがeofの場合は、lineCont=""が戻る。
        コメント文もそのまま取得する。
        戻り値:(lineCont, p, kind)
        
        kindの種類:
            line        :「...;」
            keyword     :「keyword{}」
            include     :「# include "xxx.H"」
            includeEtc  :「# includeEtc "xxxx.H"」
            other       :「# xxxx yyyy」
            """
        lineCont = ""
        kind = ""
        if p >= len(self.line):
            return (lineCont, p, kind)
        
        chara, p = self.skipNoChara(p)      #pointerを空白以外に設定
        if p >= len(self.line) or chara == "":
            return (lineCont, p, kind)
        
        if chara == ";":
            lineCont = ";"
            p += 1
        elif chara == "#":
            #「#include ...」の取得
            sp = p
            p += 1                          #pointerを「#」の次に進める
            (chara, p) = self.skipNoChara(p)
            (name, p) = self.getKeyword(p)      #「include」取得
            (_keyword, p) = self.getKeyword(p)   #「"xxx.H"」取得
            lineCont = self.line[sp:p-1]
            if name == "include" or name == "includeEtc":
                kind = name
            else:
                kind = "other"
        else:
            sp = p
            (_keyword, p) = self.getKeyword(p)
            (chara, p) = self.skipNoChara(p)
            if chara == "{":
                #「keyword{}」の取得
                p -= 1                      #pointerを戻す
                (_cont, p) = self.getMiddlePair(p)
                lineCont = self.line[sp:p]
                kind = "keyword"
            else:
                #「...;」を取得
                while chara != ";" and chara != "":
                    if chara == "(":
                        p -= 1              #pointerを戻す
                        (dummy, p) = self.getSmallPair(p)
                    elif chara == "{":
                        p -= 1              #pointerを戻す
                        (dummy, p) = self.getMiddlePair(p)
                    (chara, p) = self.get1chara(p)
                if chara != "" and p < len(self.line):
                    if self.line[p] == ";":
                        p += 1
                lineCont = self.line[sp:p]
                kind = "line"
        return (lineCont, p, kind)

    #
    #  getKeywordPointer
    #
    def getKeywordPointer(self, keyword, p):
        """ 指定したkeywordのpointerを取得する。
        （高速検索している為、keywordがコメント中でも、検索するの注意！）
        pointerは、keywordの頭の文字位置。
        合致しない場合は、戻り値「-1」になる。"""
        if p >= 0 and p < len(self.line):
            #高速検索のため、string.find()を使用してポインタを設定
            flag = 0
            while (p >= 0) and (flag == 0):
                n = self.line[p:].find(keyword)
                if n >= 0:
                    #keyword前後の文字をチェック
                    ps = p + n - 1
                    if ps < 0:
                        stChara = self.line[ps+1]
                    else:
                        stChara = self.line[ps]
                    pe = p + n + len(keyword)
                    if pe >= len(self.line):
                        edChara = "\n"
                    else:
                        edChara = self.line[pe]
                    if ((self.checkChara(stChara) == False) and
                        (self.checkChara(edChara) == False)):
                        flag = 1
                        p = pe
                        break
                    else:
                        p = p + n + 1
                else:
                    break
            if flag == 0:
                return -1
            else:
                p = ps + 1
                return p
        return -1

    #
    #  getKeywordContents
    #
    def getKeywordContents(self, keyword, p):
        """ 指定したkeywordの内容を取得する。
        str.find()で検索するため、コメント内の文字列も検索する。
        {}内のデータを取得する。()内は取得しない。"""
        if p >= 0 and p < len(self.line):
            #高速検索のため、str.find()で検索する
            flag = 0
            while p >= 0 and flag == 0:
                n = self.line[p:].find(keyword)
                if n >= 0:
                    #keyword前後の文字をチェック
                    ps = p + n - 1
                    if ps < 0:
                        #stChara = self.line[ps+1]
                        stChara = "\n"
                    else:
                        stChara = self.line[ps]
                    pe = p + n + len(keyword)
                    if pe >= len(self.line):
                        edChara = "\n"
                    else:
                        edChara = self.line[pe]
                    if (self.checkChara(stChara) == False and
                        self.checkChara(edChara) == False):
                        #ループ脱出
                        flag = 1
                        p = pe
                        break
                    else:
                        p = p + n + 1
                else:
                    break
            if flag == 0:
                contents = ""
                p = len(self.line) + 1
            else:
                #内容を取得
                (contents, p) = self.getMiddlePair(p)
        else:
            contents = ""
            p = len(self.line) + 1
        return (contents, p)

    #
    #  getBackKeyword
    #
    def getBackKeyword(self, p):
        """ 遡ってkeywordを取得"""
        #文字をbackする
        chara = self.line[p]
        while self.checkChara(chara) == True:
            p -= 1
            chara = self.line[p]
        #空白をbackSkipする
        while self.checkChara(chara) == False:
            p -= 1
            chara = self.line[p]
        #文字をbackする
        while self.checkChara(chara) == True:
            p -= 1
            chara = self.line[p]
        #keywordを取得
        (keyword, p) = self.getKeyword(p)
        return (keyword, p)

    #
    #  getNextKeywordAndContents
    #
    def getNextKeywordAndContents(self, p):
        """ 次のkeywordと{}の内容を取得する
        戻り値：((keyword, contents), p)"""
        flag = 0
        chara = " "
        while flag == 0 and chara != "":
            (keyword, p) = self.getKeyword(p)
            (chara, p) = self.skipNoChara(p)
            if chara == "{":
                flag = 1
        p -= 1
        (contents, p) = self.getMiddlePair(p)
        return ((keyword, contents), p)

    #
    #  getMiddlePair
    #
    def getMiddlePair(self, p):
        """ {}内の文字列を取得し、その内容とpointerを返す。
        検索はstr.find()で検索する。"""
        pair = ""
        while pair != "{" and p < len(self.line):
            #"{" or "}" を探す
            (pair, p) = self.getNextMiddlePair(p)
        #p += 1
        ap = p
        count  = 1
        while count > 0 and p < len(self.line):
            (pair, p) = self.getNextMiddlePair(p)
            if pair == "{":
                count += 1
            elif pair == "}":
                count -= 1
        contents = self.line[ap:p-1]
        return (contents, p)
    
    #
    #  getNextMiddlePair
    #
    def getNextMiddlePair(self, p):
        """ pointerに近い"{" or "}" 文字を検索し、文字種とpointerを返す。
        検索はstr.finf()で検索する。"""
        lp = self.line[p:].find("{")
        if lp < 0:
            lp = len(self.line)
        rp = self.line[p:].find("}")
        if rp < 0:
            rp = len(self.line)
        if lp < rp:
            ans = "{"
            ap = p + lp + 1
        elif rp < lp:
            ans = "}"
            ap = p + rp + 1
        else:
            ans = ""
            ap = len(self.line)
        return (ans, ap)
    
    #
    #  getSmallPair
    #
    def getSmallPair(self, p):
        """ ()内データを取得し返す。"""
        pair = ""
        while pair != "(" and p < len(self.line):
            #"(" or ")" を探す
            (pair, p) = self.getNextSmallPair(p)
        ap = p
        count  = 1
        while count > 0 and p < len(self.line):
            (pair, p) = self.getNextSmallPair(p)
            if pair == "(":
                count += 1
            elif pair == ")":
                count -= 1
        contents = self.line[ap:p-1]
        return (contents, p)

    #
    #  getNextSmallPair
    #
    def getNextSmallPair(self, p):
        """ pointerに近い"(" or ")" 文字を検索し、文字種とpointerを返す。
        検索はstr.finf()で検索する。"""
        lp = self.line[p:].find("(")
        if lp < 0:
            lp = len(self.line)
        rp = self.line[p:].find(")")
        if rp < 0:
            rp = len(self.line)
        if lp < rp:
            ans = "("
            ap = p + lp + 1
        elif rp < lp:
            ans = ")"
            ap = p + rp + 1
        else:
            ans = ""
            ap = len(self.line)
        return (ans, ap)

    #
    #  getKeyword
    #
    def getKeyword(self, p):
        """ keywordを取得して返す"""
        (chara, p) = self.get1chara(p)
        #文字まで進む
        while (self.checkChara(chara) != True and chara != ""):
            (chara, p) = self.get1chara(p)
        #keyword取得
        keyword = ""
        if chara == '"':
            #""で囲まれる場合
            keyword += chara
            (chara, p) = self.get1chara(p)
            while (chara != '"') and (chara != ""):
                keyword += chara
                (chara, p) = self.get1chara(p)
            #最後の"をkeywordに追加しポインタを1ヶ進める
            keyword += chara
            p += 1
        else:
            #通常の文字列
            while ((self.checkChara(chara) == True) and (chara != "")
                        and p < len(self.line)):
                keyword += chara
                (chara, p) = self.get1chara(p)
            if chara != "":
                p -= 1      #1文字読みすぎているので、ポインタを1ヶ戻す
        return (keyword, p)

    #
    #  get1chara
    #
    def get1chara(self, p):
        """ 1文字を取得"""
        if p >= len(self.line):
            return ("", p)

        (chara, p) = self.get1c(p)
        if chara == "/":
            ps = p
            (nextChara, p) = self.get1c(p)
            if nextChara == "/":
                p = self.skipUntilCR(p)      #行末までスキップ
                (chara, p) = self.get1c(p)
            elif nextChara == "*":
                p= self.skipUntilSS(p)       #"*/"までスキップ
                (chara, p) = self.get1c(p)
            else:
                p = ps                       #pointerを戻す
                #p-=1                            #1文字戻す
        return (chara, p)

    #行末までスキップ
    def skipUntilCR(self, p):
        """ 行末までskipする"""
        if p>=0 and p<len(self.line):
            (chara, p) = self.get1c(p)
            while (chara != "\n") and (chara != ""):
                [chara, p] = self.get1c(p)
            if chara == "\n":
                p-=1                            #1文字戻す
        else:
            p=len(self.line)
        return p

    #*/までスキップ
    def skipUntilSS(self, p):
        """ 「*/」までskipする"""
        if p >= 0 and p < len(self.line)-1:
            charas = self.line[p:p+2]; p+=1
            while (charas != "*/") and (p < len(self.line)-1):
                charas = self.line[p:p+2]; p+=1
            if charas == "*/":
                p+=1
            else:
                p=len(self.line)
        else:
            p=len(self.line)
        return p

    #spスキップ
    def skipSP(self, p):
        """ 空白をskipする"""
        if p >= 0 and p < len(self.line)-1:
            while self.line[p] == " ":
                p += 1
        else:
            p=len(self.line)
        return p

    #1文字取得
    def get1c(self, p):
        """ 1文字取得"""
        #if p >= 0 and p < len(self.line)-1:
        if p >= 0 and p < len(self.line):
            chara = self.line[p]; p += 1
            # if type(chara) == int:
            #     chara = chr(chara).encode()
        else:
            chara = ""; p = len(self.line)
        return (chara, p)

    #
    #  skipNoChara
    #
    def skipNoChara(self, p):
        """ 空白以外の文字までskipする
        ﾎﾟｲﾝﾀは空白以外の文字位置を返す。charaはﾎﾟｲﾝﾀが示す文字を返す"""
        if p >= len(self.line):
            return ("", p)
        (chara, p) = self.get1chara(p)
        while (chara == " " or chara == "\n" or chara == "\t"):
            (chara, p) = self.get1chara(p)
        p -= 1
        return (chara, p)

    #
    #  checkChara
    #
    def checkChara(self, chara):
        """ keywordとして使える文字のチェック"""
        # if type(chara) == int:
        #     chara = chr(chara).encode()
        if chara > " " and chara <= "z":
            if chara == "{" or chara == "}":
                flag = False
            elif chara == "(" or chara == ")":
                flag = False
            elif chara == ";":
                flag = False
            else:
                flag = True
        else:
            flag = False
        return flag

    #
    #  skipIntoPair
    #
    def skipIntoPair(self, p, pair="{"):
        """ 現在のpointerからpair文字（default:"{"）までskipしてpointerを返す """
        if p >= 0 and p < len(self.line)-1:
            #asciiコードで比較
            while (self.line[p] != pair) and (p < len(self.line)-1):
                p += 1
            p += 1
        else:
            p=len(self.line)
        return p

    #
    #  deleteCommentsNullLines
    #
    def deleteCommentsNullLines(self):
        """ コメント、空行の削除。削除後は、元のself.lineも置き換わる"""
        contents = self.deleteComments()
        contents = self.deleteNullLines()
        return contents

    #
    #  deleteComments
    #
    def deleteComments(self):
        """ commentを削除して返す。削除後は、元のself.lineも置き換わる"""
        #/*を削除
        nocomm = ""
        st = 0
        loop = True
        while loop == True:
            # "/" を検索
            pp = self.line[st:].find("/")
            if pp >= 0:
                # "/*" を検索
                if self.line[st:st+2] == "/*":
                    commS = st +  pp
                    nocomm += self.line[st:pp]
                    ped = self.line[commS:].find("*/")
                    if ped >= 0:
                        st = commS + ped + len("*/")
                    else:
                        break
                # "//" を検索
                elif self.line[st:st+2] == "//":
                    commS = st + pp
                    nocomm += self.line[st:commS]
                    ped = self.line[commS:].find("\n")
                    if ped >= 0:
                        st = commS + ped
                    else:
                        break
                else:
                    #pointerを更新
                    nocomm += self.line[st]
                    st += 1
            else:
                nocomm += self.line[st:]
                break
        self.line = nocomm
        return nocomm

    #
    #  deleteNullLines
    #
    def deleteNullLines(self):
        """ 空行を削除する。削除後は、元のself.lineも置き換わる"""
        lines = self.line.split("\n")
        newLines = []
        for line in lines:
            if len(line.split()) != 0:
                newLines.append(line)
        newCont = "\n".join(newLines) + "\n"
        self.line = newCont
        return newCont

    #
    #  compressSPCR
    #
    def compressSPCR(self):
        """ 余分なSPとCRを削除する。削除後は、元のself.lineも置き換える"""
        lines = self.line.split("\n")
        ans = []
        for line in lines:
            if line != "":
                words = line.split()
                ans.append(" ".join(words))
        ansR = "\n".join(ans)
        if len(ansR) > 0:
            if ansR[-1] == "\n":
                ansR = ansR[:-1]
        else:
            ansR = "\n"
        self.line = ansR        
        return ansR
        



if __name__ == "__main__":
    import gettext
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    currDir = '/home/caeuser/CAE/CAE-FOAM/OF-v1906/pitzDaily_binary'
    fileName = currDir + "/100/U"
    cont = foamFile().read(fileName)
    print(cont)
    foamFile().write(fileName, cont)
