#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#   remakeSolidShellBeamModel.py
#
#       solid、shell、beamモデルかどうかをチェックして、この場合は、
#       「!EQUATION」を追加する
#
#   16/11/23    新規作成
#   19/01/06    python3用に書き換え
#      04/20    vsCodeで修正
#   22/03/15    国際化
#      04/12    log表示修正（import logFileCreaterを追加）
#   24/07/07    openにencoding="utf-8"を追加
#


import sys, os

#                [TYPE node数]
solidElements = [["341", 4], ["342", 10],   #四面体1次2次
                ["361", 8], ["362", 20],    #六面体1次2次
                ["351", 6], ["352", 15]]    #五面体1次2次

shellElements = [["731", 3], ["761", 6],    #三角形
                ["741", 4], ["781", 8]]     #四角形

beamElements = [["611", 2], ["641", 4]]     #梁要素


nodeLoc = []        #節点Noをインデックスとして座標を取得
connectShells = []  #shellとの接続部
connectBeams = []   #beamとの接続部
flagS = False       #shell接続でEQUATIONを作成したか
flagB = False       #beam接続でEQUATIONを作成したか

#
#  helpMsg
#
def helpMsg():
    msg = """
<remakeSolidShellBeamModel.py>
shell, beamをsolidに接続する為に、shell, beamのdummy節点（回転角）
とsolid側shellの接続面の変位の関係式を「!EQUATION」で記述し、
mshファイルに追記する。
追記する場所は、!NODE定義の最後（!ELEMENT定義の前）。
この場所に既に!EQUATIONが存在する場合は、これを置き換える。
これにより、solidの回転角と変位がshell, beam側に伝わっていく。
beamのみ、shellのみ、solidのみのmshファイルの場合は、なにもせず戻る。
mshファイル内に入力したconnectFaces、connectLines名が存在しない場合も、
なにもせず戻る。

<使い方>
remakeSolidShellModel.py <mshFile> [<connectFace>] [<connectLine>]

mshFile      :mshファイル
connectFace  :solid側のshellとの接続面の節点グループ名
              省略時は、"toShell"と"shellConnect"が設定される
connectLine  :solid, shell側の接続面の節点グループ名
              省略時は、"toBeam"と"beamConnect"が設定される

例：
remakeSolidShellModel.py FistrModel.msh
remakeSolidShellModel.py FistrModel.msh toShell toBeam
"""
    print (msg)


#
#  crossProduct
#    ベクトル積（外積）
def crossProduct(a, b):
    x = a[1]*b[2] - a[2]*b[1]
    y = a[2]*b[0] - a[0]*b[2]
    z = a[0]*b[1] - a[1]*b[0]
    return [x, y, z]

#
#  dotProduct
#    スカラー積（内積）
def dotProduct(a, b):
    ans = a[0]*b[0] + a[1]*b[1] + a[2]*b[2]
    return ans

#
#  mag
#    vectorの絶対値の取得
def mag(a):
    ans = pow((a[0]*a[0] + a[1]*a[1] + a[2]*a[2]), 0.5)
    return ans

#  getLength
#    2点間の距離を算出
def getLength(a, b):
    vec = getVector(a, b)
    l = mag(vec)
    return l

#  getVector
#    2点間のベクトルを作成
def getVector(a, b):
    x = b[0] - a[0]
    y = b[1] - a[1]
    z = b[2] - a[2]
    return [x,y,z]

#  normalVector
#    単位ベクトルを取得
def normalVector(vec):
    l = mag(vec)
    x = vec[0]/l
    y = vec[1]/l
    z = vec[2]/l
    return [x,y,z]


#  readLines
#    fileの読み込み
def readLines(fileName):
    f = open(fileName, encoding="utf-8")
    lines = f.readlines()
    f.close()
    return lines

#  isSolidElement
#    solid要素?
def isSolidElement(elName):
    ans = False
    for element in solidElements:
        if elName == element[0]:
            ans = True
            break
    return ans

#  isShellElement
#    shell要素?
def isShellElement(elName):
    ans = False
    for element in shellElements:
        if elName == element[0]:
            ans = True
            break
    return ans

#  isBeamElement
#    beam要素?
def isBeamElement(elName):
    ans = False
    for element in beamElements:
        if elName == element[0]:
            ans = True
            break
    return ans

#  isSolidShellModel
#    solidShellかどうかチェック
def isSolidShellModel(lines):
    solid = False
    shell = False
    for line in lines:
        if line[:len("!ELEMENT")] == "!ELEMENT":
            typeName = getValName(line, "TYPE")
            if isSolidElement(typeName) == True:
                solid = True
            elif isShellElement(typeName) == True:
                shell = True
        if solid == True and shell == True:
            break
    if solid == True and shell == True:
        return True
    else:
        return False

#  isSolidBeamModel
#    solidBeamかどうかチェック
def isSolidBeamModel(lines):
    solid = False
    beam = False
    for line in lines:
        if line[:len("!ELEMENT")] == "!ELEMENT":
            typeName = getValName(line, "TYPE")
            if isSolidElement(typeName) == True:
                solid = True
            elif isBeamElement(typeName) == True:
                beam = True
        if solid == True and beam == True:
            break
    if solid == True and beam == True:
        return True
    else:
        return False

#  isInGrpNames
#    nameのheaderがgrpNamesと同じかどうか
def isInGrpNames(name, grpNames):
    ans = False
    for grpName in grpNames:
        if name[:len(grpName)] == grpName:
            ans = True
            break
    return ans

#  isShellConnectNodes
#    shellConnectのnodeGroupがあるか?
def isShellConnectNodes(lines, connectFaces):
    ans = False
    for line in lines:
        if line[:len("!NGROUP")] == "!NGROUP":
            ngrp = getValName(line, "NGRP")
            #if ngrp[:len(connectFace)] == connectFace:
            if isInGrpNames(ngrp, connectFaces) == True:
                ans = True
                break
    return ans

#  isBeamConnectNodes
#    beamConnectのnodeGroupがあるか？
def isBeamConnectNodes(lines, connectLines):
    ans = False
    for line in lines:
        if line[:len("!NGROUP")] == "!NGROUP":
            ngrp = getValName(line, "NGRP")
            #if ngrp[:len(connectLine)] == connectLine:
            if isInGrpNames(ngrp, connectLines) == True:
                ans = True
                break
    return ans

#
#  getValName
#    lineからvalNameを取得
def getValName(line, valName):
    words = line.split(",")
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        if word.find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    newName = ""
    for chara in name:
        if chara > " ":
            newName += chara
    return newName

#
#  deleteSp
#    空白とCRを削除して返す
def deleteSp(line):
    newLine = ""
    i = 0
    for chara in line:
        if chara > " ":
            newLine += chara
        else:
            if i >= 1 and chara == " ":
                if line[i-1] == " " or line[i-1] == ",":
                    pass
                else:
                    newLine += chara
        i += 1 
    return newLine

#
#  getNodeLocation
#    nodeの座標を取得
def getNodeLocation(lines):
    #nodeの行を取得
    nodeLines = []
    flag = 0
    for line in lines:
        if line[0] == "!":
            if line[:len("!NODE")] == "!NODE":
                flag = 1
            else:
                flag = 0
        else:
            if flag == 1:
                words = deleteSp(line).split(",")
                nodeNo = int(words[0])
                x = float(words[1])
                y = float(words[2])
                z = float(words[3])
                nodeLines.append([nodeNo,x,y,z])
    nodeLines.sort()
    #maxNodeNo = nodeLines[-1][0]
    nodeLoc = []
    for i in range(len(nodeLines)+1):
        nodeLoc.append([])
    for node in nodeLines:
        nodeNo = node[0]
        nodeLoc[nodeNo] = node[1:]
    return nodeLoc            

#
#  getShellConnectNodes
#    shellConnect面のnodeを取得
def getShellConnectNodes(lines, grpNames):
    nodeLines = []
    flag = 0
    for line in lines:
        if line[0] == "!":
            if line[:len("!NGROUP")] == "!NGROUP":
                name = getValName(line, "NGRP")
                #if name[:len(grpName)] == grpName:
                if isInGrpNames(name, grpNames) == True:
                    flag = 1
                else:
                    flag = 0
            else:
                flag = 0
        else:
            if flag == 1:
                words = deleteSp(line).split(",")
                for word in words:
                    try:
                        a = int(word)
                        nodeLines += [a]
                    except:
                        pass
    return nodeLines

#  delSameNodes
#    同じnodeNoを削除
def delSameNodes(nodes):
    nodes.sort()
    newNodes = [nodes[0]]
    for node in nodes:
        if newNodes[-1] != node:
            newNodes.append(node)
    return newNodes

#
#  getShellNodesOnConnect
#    shell接続面上のshellの節点を取得
def getShellNodesOnConnect(lines, shellConnect):
    #shell行を取得
    shellLines = []
    flag = 0
    for line in lines:
        if line[0] == "!":
            if line[:len("!ELEMENT")] == "!ELEMENT":
                typeName = getValName(line, "TYPE")
                if isShellElement(typeName) == True:
                    flag = 1
            else:
                flag = 0
        else:
            if flag == 1:
                shellLines.append(line)
    #数字に変換
    shellEl = []
    for line in shellLines:
        words = deleteSp(line).split(",")
        data = []
        for word in words:
            try:
                a = int(word)
                data += [a]
            except:
                pass
        shellEl.append(data)
    #connect面のnodeを取得
    #  conNodes：[[shellNode, dummyNode], ..]
    conNodes = []; shells = []
    for shell in shellEl:
        flag = 0
        for i in range(len(shell[1:])):
            shellNode = shell[i+1]
            for cnNode in shellConnect:
                if shellNode == cnNode:
                    p = int(len(shell[1:])/2) + i + 1
                    conNodes.append([shellNode, shell[p]])
                    flag = 1
                    break
        if flag == 1:
            shells.append(shell)
    conNodes = delSameNodes(conNodes)
    return [conNodes, shells]

#
#  getBeamNodesOnConnect
#    beam接続面上のbeamの節点を取得
def getBeamNodesOnConnect(lines, beamConnect):
    #beam行を取得
    beamLines = []
    flag = 0
    for line in lines:
        if line[0] == "!":
            if line[:len("!ELEMENT")] == "!ELEMENT":
                typeName = getValName(line, "TYPE")
                if isBeamElement(typeName) == True:
                    flag = 1
            else:
                flag = 0
        else:
            if flag == 1:
                beamLines.append(line)
    #数字に自変換
    beamEl = []
    for line in beamLines:
        words = deleteSp(line).split(",")
        data = []
        for word in words:
            try:
                a = int(word)
                data += [a]
            except:
                pass
        beamEl.append(data)
    #connect面のnodeを取得
    #  conNodes: [[beamNode, dummyNode], ...]
    conNodes = []; beams = []
    for beam in beamEl:
        flag = 0
        for i in range(len(beam[1:])):
            beamNode = beam[i+1]
            for cnNode in beamConnect:
                if beamNode == cnNode:
                    p = int(len(beam[1:])/2) + i + 1
                    conNodes.append([beamNode, beam[p]])
                    flag = 1
                    break
        if flag == 1:
            beams.append(beam)
    conNodes = delSameNodes(conNodes)
    return [conNodes, beams]

#
# separateConnectFaceNodes
#    shellの接続面の節点を裏表に分ける
def separateConnectFaceNodes(faceNodes, nodes, shells):

    def isInShellNode(node, nVecs):
        ans = False
        for nVec in nVecs:
            for No in nVec[1]:
                if node == No:
                    ans = True
                    break
        return ans

    # def deleteSameNode(nodes):
    #     nodes.sort()
    #     newNodes = [nodes[0]]
    #     for node in nodes:
    #         if newNodes[-1][0] != node[0]:
    #             newNodes.append(node)
    #     return newNodes

    upperNodes = []
    lowerNodes = []
    nVecs = []
    #shellの単位ベクトルを作成
    for i in range(len(shells)):
        shell = shells[i]
        loc1 = nodeLoc[shell[1]]
        loc2 = nodeLoc[shell[2]]
        vec1 = getVector(loc1, loc2)
        loc1 = nodeLoc[shell[2]]
        loc2 = nodeLoc[shell[3]]
        vec2 = getVector(loc1, loc2)
        vec = crossProduct(vec1, vec2)
        nVec = normalVector(vec)
        sameNodes = []
        nNodes = int((len(shell) - 1) / 2)
        for n in range(nNodes):
            shellNode = shell[n+1]
            for faceNode in faceNodes:
                if shellNode == faceNode:
                    sameNodes.append(shellNode)
                    break
        if len(sameNodes) == 2:
            nVecs.append([nVec, sameNodes])
    #nodeを表裏に分ける
    #up = []; low = []
    for faceNode in faceNodes:
        if isInShellNode(faceNode, nVecs) == False:
            #最短のnodeNoを調べる
            nearNodes = []
            for nVec in nVecs:
                pnt1 = nVec[1][0]
                pnt2 = nVec[1][1]
                l1 = getLength(nodeLoc[pnt1], nodeLoc[faceNode])
                l2 = getLength(nodeLoc[pnt2], nodeLoc[faceNode])
                nearNodes.append([l1, pnt1, nVec])
                nearNodes.append([l2, pnt2, nVec])
            nearNodes.sort()
            nearNode = nearNodes[0][1]
            #裏と表を調べる
            vec = getVector(nodeLoc[nearNode], nodeLoc[faceNode])
            nVec = nearNodes[0][2]
            dotPro = dotProduct(vec, nVec[0])
            if dotPro >= 0:
                #表側
                upperNodes.append([faceNode, nVec])
            else:
                #裏側
                lowerNodes.append([faceNode, nVec])
    return [upperNodes, lowerNodes]

#
#  getUnsetDummyNodes
#    セットされていないdummy節点を取得する
def getUnsetDummyNodes(connectNodes, setDummyNodes):
    unsetNodes = []
    for [node, dummy] in connectNodes:
        flag = 0
        for setDummyNode in setDummyNodes:
            if dummy == setDummyNode:
                flag = 1
                break
        if flag == 0:
            unsetNodes.append([node, dummy])
    return unsetNodes

#
#  createEquationNodesLines
#    表側の節点の!EQUATIONを作成
def createEquationNodesLines(nodes, connectNodes):

    #  createLine
    #    追加する行を作成
    def createLine(direct, faceNode, shellNode, rDirect1, dummyNode1, r1,
                   rDirect2, dummyNode2, r2):
        lineList = [str(faceNode), direct, "1", str(shellNode), direct, "-1", 
                    str(dummyNode1), rDirect1, str(-r1),
                    str(dummyNode2), rDirect2, str(-r2)]
        line = ", ".join(lineList) + "\n"
        return line

    #メイン
    newLinesx = []
    newLinesy = []
    newLinesz = []
    #各方向の単位ベクトル
    ox = [1.0, 0.0, 0.0]
    oy = [0.0, 1.0, 0.0]
    oz = [0.0, 0.0, 1.0]
    setDummyNodes = []
    #faceNode毎にチェック
    for [node, [nFaceVec, pnts]] in nodes:
        #node    :faceの節点
        #nFaceVec:shellの向き（表裏）
        #pnts    :shellと共有するedge（2ヶの節点）
        #shellのedgeの単位ベクトル取得

        #最短のshellNode取得
        l1 = getLength(nodeLoc[node], nodeLoc[pnts[0]])
        l2 = getLength(nodeLoc[node], nodeLoc[pnts[1]])
        if l1 <= l2:
            shellNode = pnts[0]
        else:
            shellNode = pnts[1]
        #shellEdgeの単位ベクトル取得
        vec = getVector(nodeLoc[pnts[0]], nodeLoc[pnts[1]])
        nVec = normalVector(vec)
        #dummy節点を取得
        for cNode in connectNodes:
            if cNode[0] == shellNode:
                dummyNode = cNode[1]    #dummy節点
                setDummyNodes.append(dummyNode)
                break
        #接続点からのベクトル
        vec = getVector(nodeLoc[shellNode], nodeLoc[node])
        #接続ラインからの距離（ベクトル）
        r = dotProduct(vec, nFaceVec)
        rVec = [nFaceVec[0]*r, nFaceVec[1]*r, nFaceVec[2]*r]

        #x方向変位
        ry = dotProduct(rVec, oz)
        rz = -dotProduct(rVec, oy)
        newLinesx += ["4, 0.0\n"]
        newLinesx += [createLine("1", node, shellNode, "2",
                        dummyNode, ry, "3", dummyNode, rz)]
        #y方向変位
        rz = dotProduct(rVec, ox)
        rx = -dotProduct(rVec, oz)
        newLinesy += ["4, 0.0\n"]
        newLinesy += [createLine("2", node, shellNode, "3",
                        dummyNode, rz, "1", dummyNode, rx)]
        #z方向変位
        rx = dotProduct(rVec, oy)
        ry = -dotProduct(rVec, ox)
        newLinesz += ["4, 0.0\n"]
        newLinesz += [createLine("3", node, shellNode, "1",
                        dummyNode, rx, "2", dummyNode, ry)]
    #セットされていないdummy節点を取得
    unsetDummyNodes = getUnsetDummyNodes(connectNodes, setDummyNodes)
    if len(unsetDummyNodes) > 0:
        print (_(u"未セットのdummy節点「") + str(len(unsetDummyNodes)) + _(u"」ヶにEQUATIONをセットします。"))
        #未セットのdummy節点にEQUATIONをセット
        addLinesxyz = createUnsetDummyNode(nodes, connectNodes, unsetDummyNodes)
        newLinesx += addLinesxyz[0]
        newLinesy += addLinesxyz[1]
        newLinesz += addLinesxyz[2]
    newLines = newLinesx + newLinesy + newLinesz
    return newLines

#
#  createBeamEquationNodesLines
#    beamの!EQUATIONを作成
#       nodes:接続面のnodeNo
#       connectNodes:beamの接続点（[[nodeNo, dummyNodeNo], ...])
def createBeamEquationNodesLines(nodes, connectNodes):

    #  createLine
    #    追加する行を作成
    def createLine(direct, faceNode, shellNode, rDirect1, dummyNode1, r1,
                   rDirect2, dummyNode2, r2):
        lineList = [str(faceNode), direct, "1", str(shellNode), direct, "-1", 
                    str(dummyNode1), rDirect1, str(-r1),
                    str(dummyNode2), rDirect2, str(-r2)]
        line = ", ".join(lineList) + "\n"
        return line

    #メイン
    newLinesx = []
    newLinesy = []
    newLinesz = []
    #各方向の単位ベクトル
    ox = [1.0, 0.0, 0.0]
    oy = [0.0, 1.0, 0.0]
    oz = [0.0, 0.0, 1.0]
    for node in nodes:
        #最短のbeamNodeNoを取得
        lengthes = []
        for connectNode in connectNodes:
            beamNodeNo = connectNode[0]
            length = getLength(nodeLoc[node], nodeLoc[beamNodeNo])
            lengthes.append([length] + connectNode)
        lengthes.sort()
        beamNode = lengthes[0][1]
        beamDummyNode = lengthes[0][2]
        #beamNode以外の場合、EQUATIONを作成
        if node != beamNode:
            #接続点からのベクトル
            vec = getVector(nodeLoc[beamNode], nodeLoc[node])
            
            #x方向
            ry = dotProduct(vec, oz)
            rz = -dotProduct(vec, oy)
            newLinesx += ["4, 0.0\n"]
            newLinesx += [createLine("1", node, beamNode, "2",
                            beamDummyNode, ry, "3", beamDummyNode, rz)]
            #y方向
            rz = dotProduct(vec, ox)
            rx = -dotProduct(vec, oz)
            newLinesy += ["4, 0.0\n"]
            newLinesy += [createLine("2", node, beamNode, "3",
                            beamDummyNode, rz, "1", beamDummyNode, rx)]
            #z方向
            rx = dotProduct(vec, oy)
            ry = -dotProduct(vec, ox)
            newLinesz += ["4, 0.0\n"]
            newLinesz += [createLine("3", node, beamNode, "1",
                            beamDummyNode, rx, "2", beamDummyNode, ry)]
    newLines = newLinesx + newLinesy + newLinesz
    return newLines

#
#  createUnsetDummyNode
#    dummy接点を基準にEQUATIONを作成
def createUnsetDummyNode(nodes, connectNodes, unsetDummyNodes):

    #  createLine
    #    追加する行を作成
    def createLine(direct, faceNode, shellNode, rDirect1, dummyNode1, r1,
                   rDirect2, dummyNode2, r2):
        lineList = [str(faceNode), direct, "1", str(shellNode), direct, "-1", 
                    str(dummyNode1), rDirect1, str(-r1),
                    str(dummyNode2), rDirect2, str(-r2)]
        line = ", ".join(lineList) + "\n"
        return line

    #メイン
    newLinesx = []
    newLinesy = []
    newLinesz = []
    #各方向の単位ベクトル
    ox = [1.0, 0.0, 0.0]
    oy = [0.0, 1.0, 0.0]
    oz = [0.0, 0.0, 1.0]
    #dummyNode毎にチェック
    for [shellNode, dummyNode] in unsetDummyNodes:
        #最短のnodeを取得
        lengthes = []
        for [node, [nFaceVec, pnts]] in nodes:
            l = getLength(nodeLoc[shellNode], nodeLoc[node])
            lengthes.append([l, [node, [nFaceVec, pnts]]])
        lengthes.sort()
        [l, [node, [nFaceVec, pnts]]] = lengthes[0]
        #shellのedgeの単位ベクトル取得
        vec = getVector(nodeLoc[pnts[0]], nodeLoc[pnts[1]])
        nVec = normalVector(vec)
        #接続点からのベクトル
        vec = getVector(nodeLoc[shellNode], nodeLoc[node])
        #接続ラインからの距離（ベクトル）
        r = dotProduct(vec, nFaceVec)
        rVec = [nFaceVec[0]*r, nFaceVec[1]*r, nFaceVec[2]*r]

        #x方向変位
        ry = dotProduct(rVec, oz)
        rz = -dotProduct(rVec, oy)
        newLinesx += ["4, 0.0\n"]
        newLinesx += [createLine("1", node, shellNode, "2",
                        dummyNode, ry, "3", dummyNode, rz)]
        #y方向変位
        rz = dotProduct(rVec, ox)
        rx = -dotProduct(rVec, oz)
        newLinesy += ["4, 0.0\n"]
        newLinesy += [createLine("2", node, shellNode, "3",
                        dummyNode, rz, "1", dummyNode, rx)]
        #z方向変位
        rx = dotProduct(rVec, oy)
        ry = -dotProduct(rVec, ox)
        newLinesz += ["4, 0.0\n"]
        newLinesz += [createLine("3", node, shellNode, "1",
                        dummyNode, rx, "2", dummyNode, ry)]
    newLines = [newLinesx, newLinesy, newLinesz]
    return newLines

#
#  createEquationLines
#    「!EQUATION」を作成
def createEquationLines(lines, faceNames):
    global nodeLoc
    #節点座標の読み込み
    nodeLoc = getNodeLocation(lines)
    #shellとの接続面の節点を取得
    connectFaceNodes = getShellConnectNodes(lines, faceNames)
    print (_(u"接続面の節点数: ") + str(len(connectFaceNodes)) + _(u"  （共有節点含む）"))
    #接続面に接続されるshellと節点を取得
    [connectNodes, connectShells] = getShellNodesOnConnect(lines, connectFaceNodes)
    print (_(u"shellの共有節点数: ") + str(len(connectNodes)))
    #接続面の節点をshellの表裏側に分け、その節点に近いshellとshell節点を取得
    [upperNodes, lowerNodes] = separateConnectFaceNodes(connectFaceNodes,
                                    connectNodes, connectShells)
    #追加するEQUATIONの行を作成
    upperAddLines = createEquationNodesLines(upperNodes, connectNodes)
    lowerAddLines = createEquationNodesLines(lowerNodes, connectNodes)
    addLines = ["!EQUATION\n"] + upperAddLines + lowerAddLines
    return addLines

#
#  checkConnectPairCorrectDummyNode
#    beamと接する相手側の要素がshellの場合、dummy節点をshellに合わせる
def checkConnectPairCorrectDummyNode(lines, connectNodes):
    #shell行を取得
    shellLines = []
    getFlag = 0
    for line in lines:
        if line[0] == "!" and line[:2] != "!!" and line[0] != "#":
            if line[:len("!ELEMENT")] == "!ELEMENT":
                elType = getValName(line, "TYPE")
                flag = 0
                for shellElement in shellElements:
                    if shellElement[0] == elType:
                        flag = 1
                        break
                if flag == 1:
                    getFlag = 1
                else:
                    getFlag = 0
            else:
                getFlag = 0
        else:
            if getFlag == 1:
                shellLines.append(line)
    #shellが存在しない場合は、そのまま戻る
    if len(shellLines) == 0:
        return [lines, connectNodes] 
    #shellのnodeNoを取得
    shellNodes = []
    for line in shellLines:
        words = deleteSp(line).split(",")
        data = []
        for word in words[1:]:
            data.append(int(word))
        shellNodes.append(data)
    #connectNodes内にshellnodeが存在するか確認
    for i in range(len(connectNodes)):
        cNode = connectNodes[i]
        flag = 0
        for shellEl in shellNodes:
            n = int(len(shellEl)/2)
            for j in range(len(shellEl[:n])):
                shellNode = shellEl[:n][j]
                if shellNode == cNode[0]:
                    #beamDummy節点をshellDummyに設定
                    shellDummy = shellEl[n+j]
                    connectNodes[i][1] = shellDummy
                    flag = 1
                    break
            if flag == 1:
                break
    #beam要素のdummy節点Noを修正（lines内全てを修正)
    setFlag = 0
    for i in range(len(lines)):
        line = lines[i]
        if line[0] == "!" and line[:2] != "!!" and line[0] != "#":
            if line[:len("!ELEMENT")] == "!ELEMENT":
                elType = getValName(line, "TYPE")
                flag = 0
                for beamElement in beamElements:
                    if beamElement[0] == elType:
                        flag = 1
                        break
                if flag == 1:                
                    setFlag = 1
                else:
                    setFlag = 0
            else:
                setFlag = 0
        else:
            if setFlag == 1:
                #lineを修正
                words = deleteSp(line).split(",")
                data = []
                for word in words[1:]:
                    data.append(int(word))
                n = int(len(data)/2)
                rFlag = 0
                for j in range(n):
                    nodeNo = data[j]
                    for connectNode in connectNodes:
                        if nodeNo == connectNode[0]:
                            #dummyを修正
                            words[1+n+j] = str(connectNode[1])
                            lines[i] = ", ".join(words) + "\n"
                            rFlag = 1
                            break
                    if rFlag == 1:
                        break
    return [lines, connectNodes]

#
#  createBeamEquations
#    beam用の「!EQUATION」を作成
def createBeamEquations(lines, lineNames):
    global nodeLoc
    #節点座標の読み込み
    if len(nodeLoc) == 0:
        nodeLoc = getNodeLocation(lines)
    #beamとの接続面の節点を取得
    connectLineNodes = getShellConnectNodes(lines, lineNames)
    print (_(u"接続面の節点数: ") + str(len(connectLineNodes)) + _(u"  （共有節点含む）"))
    #接続面に接続されるbeamと節点を取得
    [connectNodes, connectBeams] = getBeamNodesOnConnect(lines, connectLineNodes)
    #beamと接続する相手側の要素がshellの場合、dummy節点をshellのdummy節点に設定する
    [lines, connectNodes] = checkConnectPairCorrectDummyNode(lines, connectNodes)
    print (_(u"beamの共有節点数: ") + str(len(connectNodes)))
    #追加するEQUATIONの行を作成
    addLines = createBeamEquationNodesLines(connectLineNodes, connectNodes)
    addLines = ["!EQUATION\n"] + addLines
    return addLines

#  skipNextHeader
#    次のheaderまでskipする
def skipNextHeader(lines, i):
    ii = len(lines)
    while i < len(lines):
        if lines[i][0] == "!":
            ii = i
            break
        i += 1
    return ii

#
#  replaceEquationLines
#  --------------------
#    !EQUATIONのblockを置き換える
#    場所は、!NODEと!ELEMENTの間（!ELEMENTの前）
def replaceEquationLines(lines, addLines):
    i = 0
    while i < len(lines):
        i = skipNextHeader(lines, i)
        if lines[i][:len("!EQUATION")] == "!EQUATION":
            ps = i
            while i < len(lines):
                i = skipNextHeader(lines, i)
                if lines[i][:len("!ELEMENT")] == "!ELEMENT":
                    pe = i
                    newLines = lines[:ps] + addLines + lines[pe:]
                    break
                i += 1
            break
        elif lines[i][:len("!ELEMENT")] == "!ELEMENT":
            ps = i
            newLines = lines[:ps] + addLines + lines[ps:]
            break
        i += 1
    return newLines

#
#  addEquationLines
#  ----------------
#    !EQUATIONのblockを追加する
#    場所は、!NODEと!ELEMENTの間（!ELEMENTの前）
def addEquationLines(lines, addLines):
    i = 0
    while i < len(lines):
        i = skipNextHeader(lines, i)
        if lines[i][:len("!ELEMENT")] == "!ELEMENT":
            ps = i
            newLines = lines[:ps] + addLines + lines[ps:]
            break
        i += 1
    return newLines

#
#  checkAndSetEquationShell
#    solidShell混在モデルかどうかをチェックし、混在モデルの場合はEQUATIONをセット
def checkAndSetEquationShell(lines, fileName, connectFaces):
    if isSolidShellModel(lines) == False:
        return [False, lines]
    if isShellConnectNodes(lines, connectFaces) == False:
        return [False, lines]

    print (_(u"\nshellのdummy節点にEQUATIONを設定します。"))
    addLines = createEquationLines(lines, connectFaces)
    newLines = replaceEquationLines(lines, addLines)
    return [True, newLines]

#
#  checkAndSetEquationBeam
#    solidBeam混在モデルかどうかをチェックし、混在モデルの場合はEQUATIONをセット
def checkAndSetEquationBeam(lines, fileName, connectLines):
    if isSolidBeamModel(lines) == False:
        return [False, lines]
    if isBeamConnectNodes(lines, connectLines) == False:
        return [False, lines]
    print (_(u"\nbeamのdummy節点にEQUATIONを設定します。"))
    addLines = createBeamEquations(lines, connectLines)
    if flagS == True:
        newLines = addEquationLines(lines, addLines)
    else:
        newLines = replaceEquationLines(lines, addLines)
    return [True, newLines]

#
#  remakeSolidShellBeamModel
#  ---------------------
#    solidShellモデルの場合、EQUATIONを追加する。
def remakeSolidShellBeamModel(fileName, connectFaces, connectLines):
    global flagS, flagB
    lines = readLines(fileName)
    #EQUATIONを追加
    [flagS, newLines] = checkAndSetEquationShell(lines, fileName, connectFaces)
    [flagB, newLines] = checkAndSetEquationBeam(newLines, fileName, connectLines)
    if flagS == True or flagB == True:
        #書き込み
        f=open(fileName, "w")
        for line in newLines:
            f.write(line)
        f.close()
        print (_(u"solidとshell, beamの接続面の「!EQUATION」を設定しました。"))


if __name__=='__main__':
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    #log表示
    import logFileCreater
    logFileCreater.log()

    if len(sys.argv) == 1:
        helpMsg()
        exit()
    elif len(sys.argv) == 2:
        if sys.argv[1] == "-h" or sys.argv[1] == "-help" or sys.argv[1] == "--help":
            helpMsg()
            exit()
        mshFile = sys.argv[1]
        connectFaces = ["shellConnect", "toShell"]
        connectLines = ["beamConnect", "toBeam"]
    elif len(sys.argv) == 3:
        mshFile = sys.argv[1]
        connectFaces = [sys.argv[2]]
        connectLines = ["beamConnect", "toBeam"]
    else:
        mshFile = sys.argv[1]
        connectFaces = [sys.argv[2]]
        connectLines = [sys.argv[3]]
    remakeSolidShellBeamModel(mshFile, connectFaces, connectLines)

