#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
#   neutral2fistr.py
#
#   16/03/16    新規作成
#      04/25    getMeshParts：修正
#               elementの取得方法を修正
#      08/04    elGrpの名称を***Fistrに変更
#      11/28    引数scaleを追加
#      12/08	getMeshParts:文字列に小数点を追加
#               FrontISTR側が小数点が無い文字列をfloatと認識しない為
#               座標値の読み込みで「1e-5」, 「5」を読み込めない。
#   17/01/05    getMeshParts:バグ修正。
#      01/23	nodeGroupの取得を追加
#      02/15	elementGroupの取得を追加
#      02/16	createSurfaceGroupLines:追加（surfaceGroupの取得）
#      02/22    separateElementGroups,separateSurfaceGroups:grpsが設定されていない場合
#               直ぐに戻る。
#               getMeshParts:elementの取得方法を修正
#      03/25    createNodeGroupLines:バグ修正（returnの位置を修正）
#      03/30	createNodeGroupLines:バグ修正（最終行が取得できない時あり）
#      04/07	getGroups:「elemnt:」を「solid:」に修正
#      12/14    getShellElements:シェルモデルの要素をgrp別に取得を追加
#               createShellElementLines:fistr用に変換を追加。
#   18/01/08    getShellElements:バグ修正（elNoが「0」から始まっていた）
#      01/18    createSuGrpsShellElementLines:shell要素が扱える様に追加
#   20/09/22    python3用に書き換え
#

import sys
import os
import glob
from operator import itemgetter 

import convertMesh as cm

def printMsg():
    cont = "\n"
    cont +=  "neutral2fistr.py\n"
    cont += "Mesh of neutral format is converted to mesh of FrontISTR format.\n"
    cont += "\n"
    cont += "Usage\n"
    cont += "neutral2fistr.py <neutralFile>\n"
    cont += "\n"
    cont += "example:\n"
    cont += "neutral2fistr.py name.neutral\n"
    cont += "\n"
    print (cont)

def readFileLines(fileName):
    f=open(fileName); lines=f.readlines(); f.close()
    return lines
    
def writeFileLines(fileName, lines):
    f = open(fileName, "w")
    for line in lines:
        f.write(line)
    f.close()

#
#  getMeshParts
#----------------    
def getMeshParts(lines):
    """ meshPartsを取得して返す"""
    nodes = []
    elements = []
    elmGroups = []
    surGroups = []
    #ENDOFSECTION行を取得
    i = 0
    while i < len(lines)-1:
        #if lines[i].find("ENDOFSECTION") >= 0:
        #    i += 1
        if lines[i].find("NODAL") >= 0:
            i, nodes = getNodesData(i, lines)
        elif lines[i].find("ELEMENTS/CELLS") >= 0:
            i, elements = getElementsData(i, lines)
        elif lines[i].find("ELEMENT GROUP") >= 0:
            i, group = getElmGroupsData(i, lines)
            elmGroups.append(group)
        elif lines[i].find("BOUNDARY CONDITIONS") >= 0:
            i, group = getSurGroupsData(i, lines)
            surGroups.append(group)
        else:
            i += 1
    return [nodes, elements, elmGroups, surGroups]

#
#  getNodesData
def getNodesData(si, lines):
    """ nodesを取得"""
    nodes = []
    #nodesの取得
    n = si + 1
    for i in range(n, len(lines), 1):
        line = lines[i]
        if line.find("ENDOFSECTION") >= 0:
            si = i
            break
        words = line.split()
        #数値のcheck
        for i in range(1, len(words), 1):
            if words[i].find(".") < 0:
                m = words[i].lower().find("e")
                if m >= 0:
                    words[i] = words[i][:m] + ".0" + words[i][m:]
                else:
                    words[i] += ".0"
        nodes.append(words)
    return si, nodes
    
#
#  getElementsData
def getElementsData(si, lines):
    """ elementsを取得"""
    elements = []
    n = si + 1
    for i in range(n, len(lines), 1):
        line = lines[i]
        words = line.split()
        if line[:14] != " " * 14:
            words = line.split()
            if words[0] == "ENDOFSECTION":
                si = i
                break
            elements.append(words)
        else:
            elements[-1] += words
    return si, elements 

#
#  getElmGroupsData
def getElmGroupsData(si, lines):
    """ elmGroupを取得"""
    group = []
    n = si + 4
    for i in range(n, len(lines), 1):
        line = lines[i]
        words = line.split()
        if words[0] == "ENDOFSECTION":
            si = i
            break
        group += words
    return si, group

#
#  getSurGroupsData
def getSurGroupsData(si, lines):
    """ surGroupを取得"""
    surfaces = []
    n = si + 2
    for i in range(n, len(lines), 1):
        line = lines[i]
        words = line.split()
        if words[0] == "ENDOFSECTION":
            si = i
            break
        surfaces.append(words)
    return si, surfaces



    # #elementsの取得
    # n = n + nNodes
    # nElements = int(lines[n])
    # n += 1
    # for i in range(nElements):
    #     line = lines[n + i]
    #     words = line.split()
    #     elements.append(words)
    # #surfacesの取得
    # n = n + nElements
    # nSurfaces = int(lines[n])
    # n += 1
    # for i in range(nSurfaces):
    #     line = lines[n+i]
    #     words = line.split()
    #     surfaces.append(words[1:])
    # #取得データを整形
    # #  node
    # for i in range(len(nodes)):
    #     nodes[i] = [str(i+1)] + nodes[i]
    # #  element
    # elements.sort()
    # newEl = []
    # elData = []
    # if len(elements) > 0:
    #     elNo = elements[0][0]
    # for element in elements:
    #     if element[0] == elNo:
    #         elData.append(element[1:])
    #     else:
    #         newEl.append(["", elData])
    #         elNo = element[0]
    #         elData = []
    #         elData.append(element[1:])
    # newEl.append(["", elData])
    # elNo = 1
    # for n in range(len(newEl)):
    #     elData = newEl[n][1]
    #     for i in range(len(elData)):
    #         newEl[n][1][i] = [str(elNo)] + newEl[n][1][i]
    #         elNo += 1
    # #  surface
    # for i in range(len(surfaces)):
    #     surfaces[i] = [str(i+1)] + surfaces[i]
    # return [nodes, newEl, surfaces]

# #  getGroups
# #    groupの内容を取得
# def getGroups(grpFileName):

#     #  getGrps
#     #    keywordのgroupを取得
#     def getGrps(gLines, keyword):
#         si = -1
#         i = 0
#         for line in gLines:
#             if line[:len(keyword)] == keyword:
#                 si = i
#                 break
#             i += 1
#         if si == -1:
#             return []
#         aGrps = []
#         i = si + 1
#         while i < len(gLines):
#             line = gLines[i]
#             words = line.split()
#             if len(words) == 0:
#                 break
#             name = words[0]
#             grps = []
#             num = int(words[1])
#             i += 1
#             while num > 0:
#                 line = gLines[i]
#                 words = line.split()
#                 grps += words
#                 num -= len(words)
#                 i += 1
#             aGrps.append([name, grps])
#         return aGrps

#     if len(glob.glob(grpFileName)) == 0:
#         return [[], [], []]

#     gLines = readFileLines(grpFileName)
#     ndGrps = getGrps(gLines, "node:")
#     suGrps = getGrps(gLines, "surface:")
#     #elGrps = getGrps(gLines, "element:")
#     elGrps = getGrps(gLines, "solid:")
#     return [ndGrps, elGrps, suGrps]

# #
# #  separateElementGroups
# #    elementを設定されているgroupに分割する
# def separateElementGroups(elements, elGrps):
#     #groupが定義されていない場合は、直ぐに戻る
#     if len(elGrps) == 0:
#         return elements

#     elNos = []
#     for elementList in elements:
#         for element in elementList[1]:
#             elNo = int(element[0])
#             elNos.append(elNo)
#     maxElNo = max(elNos)
#     elGrpNames = []
#     for i in range(maxElNo+1):
#         elGrpNames.append("")
#     i = 0
#     for elGrp in elGrps:
#         name = elGrp[0]
#         for elNo in elGrp[1]:
#             elGrpNames[int(elNo)] = [name, i]
#         i += 1
#     #elementをgroup分け
#     #  未定義のelementを取得
#     newElements = []
#     grpElements = []
#     for elementList in elements:
#         newElementList = []
#         for element in elementList[1]:
#             elNo = int(element[0])
#             namei = elGrpNames[elNo]
#             if namei == "":
#                 newElementList.append(element)
#             else:
#                 grpElements.append([namei[1]] + element)
#         if len(newElementList) > 0:
#             newElements.append(["", newElementList])
#     #  groupName定義済みのelementを取得
#     grpNameElms = []
#     for elGrp in elGrps:
#         name = elGrp[0]
#         grpNameElms.append([name, []])
#     for grpElement in grpElements:
#         i = grpElement[0]
#         element = grpElement[1:]
#         grpNameElms[i][1].append(element)
#     newElements = grpNameElms + newElements
#     return newElements


def createTitle():
    lines =  ["!HEADER\n"]
    lines += ["  generate by neutral2fistr.py\n"]
    return lines

def createNodeLines(nodes):
    lines = ["!NODE\n"]
    for node in nodes:
        line = ", ".join(node) + "\n"
        lines.append(line)
    return lines
    
def createElementLines(elements, name):
    lines = []
    #elementsをelmType,nNodesでsortする
    elms = sorted(elements, key=itemgetter(1, 2))
    neuType = elms[0][1]
    n = int(elms[0][2])
    element = elms[0][-n:]
    elmType, newElm = cm.neu2fistr_el(neuType, element)
    nNodes = elms[0][2]
    line = "!ELEMENT, TYPE=" + elmType + ", EGRP=" + name + "_" + elmType + "\n"
    lines += [line]
    for elm in elms:
        if elm[1] == neuType or elm[2] == nNodes:
            #element行を作成
            elmNo = elm[0]
            n = int(nNodes)
            element = elm[-n:]
            elmType, newElm = cm.neu2fistr_el(neuType, element)
            line = ", ".join([elmNo] + newElm) + "\n"
            lines += [line]
        else:
            #sectionを作成
            elmConts = cm.elmContsDict[elmType]
            secType = elmConts[0].upper()
            line = "!SECTION, TYPE=" + secType + ", EGRP=" + name + "_" + elmType + "\n"
            lines += [line]
            #新しいheaderを作成
            neuType = elm[1]
            nNodes = elm[2]
            elmNo = elm[0]
            n = int(nNodes)
            element = elm[-n:]
            elmType, newElm = cm.neu2fistr_el(neuType, element)
            line = "!ELEMENT, TYPE=" + elmType + ", EGRP=" + name + "\n"
            lines += [line]
            #element行を作成
            line = ", ".join([elmNo] + newElm) + "\n"
            lines += [line]
    #sectionを作成
    elmConts = cm.elmContsDict[elmType]
    secType = elmConts[0].upper()
    line = "!SECTION, TYPE=" + secType + ", EGRP=" + name + "_" + elmType + "\n"
    lines += [line]
    return lines

#
#  createElmGroupslines
def createElmGroupslines(elmGroups):
    """ 要素Groupのlinesを作成"""
    lines = []
    for i in range(len(elmGroups)):
        name = "elmGroup_" + str(i)
        group = elmGroups[i]
        line = "!EGROUP, EGRP=" + name + "\n"
        lines += [line]
        for j in range(0, len(group), 10):
            line = ", ".join(group[j:j+10]) + "\n"
            lines += [line]
    return lines

#
#  createSurGroupsLines
def createSurGroupsLines(surGroups):
    """ surGroupのlinesを作成"""
    neuTypeFaceDict = {
        #  neu      fistr
        #type face  face
        ("6", "1"): "1",        #
        ("6", "2"): "2",
        ("6", "3"): "3",
        ("6", "4"): "4",
        ("4", "1"): "3",
        ("4", "2"): "4",
        ("4", "3"): "5",
        ("4", "4"): "6",
        ("4", "5"): "1",
        ("4", "6"): "2"
    }
    lines = []
    for i in range(len(surGroups)):
        name = "surGroup_" + str(i)
        group = surGroups[i]
        line = "!SGROUP, SGRP=" + name + "\n"
        lines += [line]
        for j in range(0, len(group), 5):
            surs = group[j:j+5]
            words = []
            for elmNo, neuType, faceNo in surs:
                key = (neuType, faceNo)
                newFaceNo = neuTypeFaceDict[key]
                words += [elmNo, newFaceNo]
            line = ", ".join(words) + "\n"
            lines += [line]
    return lines



    # lines = []
    # for n in range(len(elements)):
    #     grpName = elements[n][0]
    #     if grpName == "":
    #         egrp = "EGRP=" + name + "_" + str(n+1)
    #     else:
    #         egrp = "EGRP=" + grpName
    #     elData = elements[n][1]
    #     if len(elData[0][1:]) == 4:
    #         elType = "TYPE=341"
    #         lines += ["!ELEMENT, " + elType + ", " + egrp + "\n"]
    #         for el in elData:
    #             newEl = [el[0], el[1], el[2], el[4], el[3]]
    #             line = ", ".join(newEl) + "\n"
    #             lines.append(line)
    #     else:
    #         elType = "TYPE=342"
    #         lines += ["!ELEMENT, " + elType + ", " + egrp + "\n"]
    #         for el in elData:
    #             newEl = [el[0], el[1], el[2], el[4], el[3],
    #                      el[9], el[7], el[5], el[6], el[8], el[10]]
    #             line = ", ".join(newEl) + "\n"
    #             lines.append(line)
    #     lines.append("!SECTION, TYPE=SOLID, " + egrp + "\n")
    # return lines

#  getShellElements(lines)
#    shell用のpartsをneutralメッシュから取得
def getShellElements(lines):
    #nodeをskip
    nNode = int(lines[0])
    n = nNode + 1
    #elementをskip
    nElms = int(lines[n])
    n += nElms + 1
    #surfaceのみを取得
    nSur = int(lines[n])
    n += 1
    surLines = []
    for _i in range(nSur):
        line = lines[n]
        words = line.split()
        words[0] = int(words[0])
        surLines.append(words)
        n += 1
    #grp分の配列を準備
    surLines.sort()
    maxGrpNo = surLines[-1][0]
    surfaces = []
    for _i in range(maxGrpNo):
        surfaces.append([])
    #surfaceの要素を取得
    n = 1
    for words in surLines:
        grpId = words[0] - 1
        surElm = [str(n)] + words[1:]
        surfaces[grpId].append(surElm)
        n += 1
    return surfaces

#  createShellElementLines
#    shellの要素を取得
def createShellElementLines(surfaces, name):
    lines = []
    for n in range(len(surfaces)):
        egrp = "EGRP=" + name + "_" + str(n+1)
        elData = surfaces[n]
        if len(elData[0][1:]) == 3:
            elType = "TYPE=731"
            lines += ["!ELEMENT, " + elType + ", " + egrp + "\n"]
            for el in elData:
                newEl = [el[0], el[1], el[3], el[2]]
                line = ", ".join(newEl) + "\n"
                lines.append(line)
        else:
            elType = "TYPE=732"
            lines += ["!ELEMENT, " + elType + ", " + egrp + "\n"]
            for el in elData:
                newEl = [el[0], el[1], el[3], el[2], el[5], el[4], el[6]]
                line = ", ".join(newEl) + "\n"
                lines.append(line)
        lines.append("!SECTION, TYPE=SHELL, " + egrp + "\n")
        lines.append("0.001, 5\n")
    return lines

#
#  createSuGrpsShellElementLines
#    shell要素を取得
def createSuGrpsShellElementLines(surfaces, nodes, suGrps, name):
    #elementListを作成
    maxElNo = int(surfaces[-1][0])
    elementList = [[]] * (maxElNo+1)
    #shell要素名を取得
    shellNames = []
    for suGrp in suGrps:
        shellNames.append(suGrp[0])
    shellNames.append(name+"_1")
    #elementListにnodeNoを設定
    for i in range(len(surfaces)):
        suNo = int(surfaces[i][0])
        nodeNos = surfaces[i][1:]
        elementList[suNo] = [nodeNos]
    #shell要素名をelementListに追加
    for i in range(len(suGrps)):
        suData = suGrps[i][1]
        for suNo in suData:
            n = int(suNo)
            elementList[n] = elementList[n] + [i]
    #未定義の要素を取得
    noNameElms = []
    for i in range(len(elementList)):
        if len(elementList[i]) == 1:
            noNameElms.append(i)
    if len(noNameElms) != 0:
        suGrps = suGrps + [name+"_1", noNameElms]
    #shell要素行を作成
    lines = []
    elNo = 1
    for suGrp in suGrps:
        egrp = "EGRP=" + suGrp[0]
        elData = suGrp[1]
        n = int(elData[0])
        if len(elementList[n][0]) == 3:
            elType = "TYPE=731"
        else:
            elType = "TYPE=732"
        lines += ["!ELEMENT, " + elType + ", " + egrp + "\n"]
        for el in elData:
            n = int(el)
            nodeNos = elementList[n][0]
            lines += [str(elNo) + ", " + ", ".join(nodeNos) + "\n"]
            elNo += 1
        lines += ["!SECTION, TYPE=SHELL, " + egrp + "\n"]
        lines += ["0.001, 5\n"]
    return lines

#
#  getFaceNoOfElement
#    要素のfaceNoを取得する
def getFaceNoOfElement(element, surface):
    #node順をfistr用に入れ替え
    elm = [element[0], element[1], element[3], element[2]]
    surface1 = [elm[0], elm[1], elm[2]]
    surface2 = [elm[0], elm[1], elm[3]]
    surface3 = [elm[1], elm[2], elm[3]]
    surface4 = [elm[2], elm[0], elm[3]]
    surface.sort()
    surface1.sort()
    surface2.sort()
    surface3.sort()
    surface4.sort()
    if surface1 == surface:
        face = 1
    elif surface2 == surface:
        face = 2
    elif surface3 == surface:
        face = 3
    elif surface4 == surface:
        face = 4
    return face

#
#  createSurfaceGroupLines
#    surfaceGroupを取得
def createSurfaceGroupLines(elements, surfaces, nodes, suGrps):
    if len(suGrps) == 0:
        return []

    #nodeの配列の準備
    nodeNos = []
    for node in nodes:
        nodeNo = int(node[0])
        nodeNos.append(nodeNo)
    maxNodeNo = max(nodeNos)
    nodeList = []
    for i in range(maxNodeNo+1):
        nodeList.append([])
    #surfaceの配列を準備
    suNos = []
    for surface in surfaces:
        suNo = int(surface[0])
        suNos.append(suNo)
    maxSuNo = max(suNos)
    surfaceList = []
    for i in range(maxSuNo+1):
        surfaceList.append([])
    #elementの配列を準備
    elNos = []
    for elementGrp in elements:
        for element in elementGrp[1]:
            elNo = int(element[0])
            elNos.append(elNo)
    maxElNo = max(elNos)
    elementList = []
    for i in range(maxElNo+1):
        elementList.append([])

    #surfaceNoをindexとしてnodeNoを保存
    for surface in surfaces:
        suNo = int(surface[0])
        #1次要素のみ取得
        for nodeNo in surface[1:4]:
            surfaceList[suNo] += [int(nodeNo)]
    #nodeNoをindexとして要素Noを保存
    #elementNoをindexとしてelementのnodeNoを保存
    for elementGrp in elements:
        elementData = elementGrp[1]
        for element in elementData:
            elNo = int(element[0])
            #1次要素のnodeNoのみ取得
            data = []
            for node in element[1:5]:
                nodeNo = int(node)
                data += [nodeNo]
                nodeList[nodeNo] += [elNo]
            elementList[elNo] = data

    #surfaceの面Noを取得
    surfaceGrps = []
    for suGrp in suGrps:
        suName = suGrp[0]
        suData = suGrp[1]
        surfaceGrp = [suName, []]
        for suNoStr in suData:
            #面要素の節点を含む全要素を取得
            suNo = int(suNoStr)
            surface = surfaceList[suNo]
            suElms = []
            for suNode in surface:
                suElms += nodeList[suNode]
            #各nodeに共通する要素を取得
            suElms.sort()
            elNo = suElms[0]
            i = 1
            n = 1
            while i < len(suElms):
                if suElms[i] == elNo:
                    n += 1
                else:
                    elNo = suElms[i]
                    n = 1
                if n == 3:
                    break
                i += 1
            #要素の面Noを取得
            element = elementList[elNo]
            surface = surfaceList[suNo]
            faceNo = getFaceNoOfElement(element, surface)
            surfaceGrp[1].append([elNo, faceNo])
        surfaceGrps.append(surfaceGrp)

    #surfaceGroupのlineを取得
    lines = []
    for surfaceGrp in surfaceGrps:
        suName = surfaceGrp[0]
        lines += ["!SGROUP, SGRP=" + suName + "\n"]
        suData = surfaceGrp[1]
        for [el, face] in suData:
            lines += [str(el) + ", " + str(face) + "\n"]
    return lines

#
#  createNodeGroupLines
#    nodeGroupを取得
def createNodeGroupLines(ndGrps):
    if len(ndGrps) == 0:
        return []

    lines = []
    for ndGrp in ndGrps:
        name = ndGrp[0]
        data = ndGrp[1]
        line = "!NGROUP, NGRP=" + name + "\n"
        lines += [line]
        num = 10
        line = ""
        lData = []
        i = 0
        for ndNo in data:
            if num > 0:
                lData.append(ndNo)
            else:
                line = ", ".join(lData) + ",\n"
                lines += [line]
                num = 10
                line = ""
                lData = [ndNo]
            num -= 1
            i += 1
        #if num > 0:
        if len(lData) > 0:
            line = ", ".join(lData) + "\n"
            lines += [line]
    return lines

#
#  changeNodesScale
#    nodesLocのscaleを変換
def changeNodesScale(nodes, scale):
    newNodes = []
    for node in nodes:
        nodeNo = node[0]
        x = str(float(node[1]) * scale)
        y = str(float(node[2]) * scale)
        z = str(float(node[3]) * scale)
        newNodes.append([nodeNo, x, y, z])
    return newNodes

#
#  convNeutralToFistr
#---------------------
def convNeutralToFistr(fileName):
    lines = readFileLines(fileName)
    name = fileName.split(os.sep)[-1].split(".")[0] + "Fistr"
    #meshPartsを取得する
    [nodes, elements, elmGroups, surGroups] = getMeshParts(lines)
    #[ndGrps, elGrps, suGrps] = getGroups(fileName+".group")
    #elements = separateElementGroups(elements, elGrps)
    fLines = createTitle()
    fLines += createNodeLines(nodes)
    fLines += createElementLines(elements, name)

    # if len(elements) == 1 and len(elements[0][1]) == 0:
    #     #shellのみのモデル
    #     #  shellの要素groupを作成
    #     if len(suGrps) == 0:
    #         surfaces = getShellElements(lines)
    #         fLines += createShellElementLines(surfaces, name)
    #     else:
    #         fLines += createSuGrpsShellElementLines(surfaces, nodes, suGrps, name)
    # else:
    #     #solidのみのモデル
    #     fLines += createElementLines(elements, name)
    #     fLines += createSurfaceGroupLines(elements, surfaces, nodes, suGrps)
    # fLines += createNodeGroupLines(ndGrps)
    fLines += createElmGroupslines(elmGroups)
    fLines += createSurGroupsLines(surGroups)
    fLines += ["!END\n"]
    fileName = ".".join(fileName.split(".")[:-1]) + ".msh"
    writeFileLines(fileName, fLines)


if __name__=='__main__':
    if len(sys.argv) == 1:
        printMsg()
        exit()
    else:
        fileName = sys.argv[1]
    if (fileName == "-h" or fileName == "-help"
        or fileName == "--help"):
        printMsg()
        exit()
    #if len(glob.glob(fileName)) == 0:
    if os.path.exists(fileName) == False:
        print ("'" + fileName + "' can not open.")
        exit()
    print
    print ("converting " + fileName + " to msh file of FrontISTR...", end="")
    convNeutralToFistr(fileName)
    print ("done.")
