#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#   getReactionForce.py
#
#       nodeGroupの反力合計を算出する
#
#   18/08/27    新規作成
#      09/02    結果fileをsortする様に修正
#   19/01/10    python3用に書き換え
#   22/02/16    getResNodeValues:fistr2vtu.py側の修正に伴い、修正。
#               修正忘れ。
#   22/03/15    国際化
#   24/07/07    openにencoding="utf-8"を追加
#

import sys, os
import glob
import convertMesh as cm
import fistr2vtk
import fistr2vtu

#
#  helpMsg
#
def helpMsg():
    cont = """
-------------- getReactionForce.py ----------------------------------
FrontISTRの結果ファイルから、指定したnodeGroupに発生する反力を計算
する。
<nodeGroup>を「ALL」とした場合は、全nodeが対象で反力を計算する。
-header option を省略した場合は、headerNameとして
「FistrModel.res.merge」「FistrModel.res.0」を順に検索し、先に検出で
きたheaderNameを使う。両方とも検出できなかった場合は、エラー停止。

＜使い方＞
getReactionForce.py [option] <nodeGroup>

    nodeGroup:  反力を計算するnodeGroup名
                「ALL」を入力した場合は、全節点が対象
option
    -header <headerName>    fileのheaderName（省略可）
    -h                      ヘルプ
    --help                  ヘルプ

＜使用例＞
getReactionForce.py fix
getReactionForce.py ALL
getReactionForce.py fix -header FistrModel.res.0
getReactionForce.py fix -header FistrModel.res.merge
"""
    print (cont)

def deleteSp(line):
    line = line.replace(" ", "").replace("\n", "")
    return line

def readFileLines(fileName):
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    return lines

def getMeshConts(meshFile):
    lines = readFileLines(meshFile)
    meshConts = cm.getFistrHeaderNumData(lines)
    return meshConts

def getReactionNodeGrp(meshHeaderNumConts, nodeGrpName):
    nodeGrp = []
    for meshCont in meshHeaderNumConts:
        header = meshCont[0]
        words = deleteSp(header).split(",")
        if words[0].upper() == "!NGROUP":
            grpName = cm.getValName(header, "NGRP")
            if grpName == nodeGrpName:
                nodeGrp = [grpName, meshCont[1]]
                break
    return nodeGrp

def addNewNodeNo(nodeList):
    newNo = 0
    for i in range(len(nodeList)):
        if len(nodeList[i]) != 0:
            nodeList[i].append(newNo)
            newNo += 1
    return nodeList

def getNewNodeNoOfNodeGrp(nodeGrp, nodeList):
    newGrp = []
    for node in nodeGrp[1]:
        newNo = nodeList[node][-1]
        newGrp.append(newNo)
    nodeGrp[1] = newGrp
    return nodeGrp

def getNumberOfNodes(meshHeaderConts):
    nNode = 0
    for meshCont in meshHeaderConts:
        header = meshCont[0]
        words = deleteSp(header).split(",")
        if words[0].upper() == "!NODE":
            nNode += len(meshCont[1])
    return nNode

def getResNodeValues(resFile, nNode, elementList):
    lines = readFileLines(resFile)
    [nodalFields, elmFields, elmToOrderDict] = fistr2vtu.getNodalResults(lines, elementList)
    return [nodalFields, elmFields]

#
#  getAllNodes
#    全nodeNoを取得
def getAllNodes(nodeList):
    nodeNos = []
    for node in nodeList:
        if len(node) > 0:
            if len(node) == 3:
                newNo = node[-1]
                nodeNos.append(newNo)
    return ["ALL", nodeNos]

#
#  getReacForce
#    nodeGroup内の反力を計算
def getReacForce(reacForceVals, nodeGrp):
    reacForce = [0, 0, 0]
    for node in nodeGrp[1]:
        #reac = reacForceVals[1][node].split()
        reac = reacForceVals[1][node]
        reacForce[0] += float(reac[0])
        reacForce[1] += float(reac[1])
        reacForce[2] += float(reac[2])
    return reacForce

#
#  getReactonForce
#    反力を算出する
def getReactionForce(nodeGrpName, fileHeader):
    if fileHeader == "":
        meshFile = "FistrModel.msh"
    else:
        meshFile = ".".join(fileHeader.split(".")[:-2]) + ".msh"
    #if len(glob.glob(meshFile)) == 0:
    if os.path.exists(meshFile) == False:
        print (_("メッシュファイルが存在しません"))
        return
    if fileHeader == "":
        resFiles = glob.glob("FistrModel.res.merge.*")
        if len(resFiles) == 0:
            resFiles = glob.glob("FistrModel.res.0.*")
            if len(resFiles) == 0:
                print (_("結果ファイルが存在しません"))
                return
    else:
        resFiles = glob.glob(fileHeader + ".*")
        if len(resFiles) == 0:
            print (_("結果ファイルが存在しません"))
            return
    numResFiles = []
    flag = 0
    for resFile in resFiles:
        try:
            num = float(resFile.split(".")[-1])
            numResFiles.append([num, resFile])
        except:
            print (_("結果ファイル名が間違っています。"))
            flag = 1
    if flag == 1:
        return

    print()
    print ("calculating reaction force of '" + nodeGrpName + "'.")
    numResFiles.sort()
    meshHeaderNumConts = getMeshConts(meshFile)
    [nodeList, elementList] = cm.getFistrMeshConts(meshHeaderNumConts)
    nodeList = addNewNodeNo(nodeList)
    if nodeGrpName == "ALL":
        nodeGrp = getAllNodes(nodeList)
    else:
        nodeGrp = getReactionNodeGrp(meshHeaderNumConts, nodeGrpName)
        if len(nodeGrp) == 0:
            print ("nodeGroup '" + nodeGrpName + _("' は存在しません"))
            return
        nodeGrp = getNewNodeNoOfNodeGrp(nodeGrp, nodeList)
    nNode = getNumberOfNodes(meshHeaderNumConts)
    for numResFile in numResFiles:
        resFile = numResFile[1]
        #nodeVals = getResNodeValues(resFile, nNode)
        [nodeVals, elmVals] = getResNodeValues(resFile, nNode, elementList)
        reacForceVals = []
        for vals in nodeVals:
            header = vals[0]
            words = header.split()
            if words[0] == "REACTION_FORCE":
                reacForceVals = vals
                break
        if len(reacForceVals) == 0:
            print (resFile + _(" 内に「REACTION_FORCE」のデータがありません"))
        else:
            reacForce = getReacForce(reacForceVals, nodeGrp)
            print (resFile + ":", reacForce)

#
#  getArgument
#    argumentを取得
def getArgument():
    nodeGrpName = ""
    fileHeader = ""
    flagHeader = 0
    for i in range(1, len(sys.argv)):
        if sys.argv[i][0] == "-":
            if sys.argv[i] == "-h" or sys.argv[i] == "--help":
                nodeGrpName = ""
                break
            elif sys.argv[i] == "-header":
                flagHeader = 1
            else:
                print (_(u"「") + sys.argv[i] + _(u"」は、不明なoptionです。"))
                nodeGrpName = ""
                break
        else:
            if flagHeader == 1:
                fileHeader = sys.argv[i]
            else:
                nodeGrpName = sys.argv[i]
    return [nodeGrpName, fileHeader]


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    if len(sys.argv) == 1:
        helpMsg()
        exit()
    [nodeGrpName, fileHeader] = getArgument()
    if nodeGrpName == "":
        helpMsg()
        exit()
    getReactionForce(nodeGrpName, fileHeader)

