#!/usr/bin/python3
#  coding: utf-8
#
#   getBeamRefDir.py
#
#       2本beam要素から、平面を作り出して、その平面の垂直方向のベクトル
#       （beamの参照Z軸方向）を出力する。
#
#   21/01/25    新規作成
#      02/20    getBeamRefDirWithP:egrpとngrpでも取得できるように変更。
#   22/03/15    国際化
#

import os, sys
import geometryFunc as geo
import pyFistr


beamGrpNames = []
meshHeaderData = []
nodeDict = {}
elementDict = {}


#
#  createNodeElementDict
def createNodeElementDict():
    """ nodeDict, elementDictを作成する"""
    nodeDict = {}
    elementDict = {}
    for headerData in meshHeaderData:
        header = headerData[0]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NODE":
            data = headerData[1]
            for node in data:
                nodeNo = node[0]
                loc = node[1:]
                nodeDict[nodeNo] = [loc]
        elif words[0] == "!ELEMENT":
            data = headerData[1]
            elmType = pyFistr.getValName(header, "TYPE")
            for elm in data:
                elmNo = elm[0]
                nodes = elm[1:]
                elementDict[elmNo] = [nodes, elmType]
    return nodeDict, elementDict

#
#  getBeamGroups
def getBeamGroups():
    """ nameFiltersをpassした要素groupの全要素Noを取得する"""
    beamsData = [[], []]
    for headerData in meshHeaderData:
        header = headerData[0]
        data = headerData[1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            egrp = pyFistr.getValName(header, "EGRP")
            if egrp == beamGrpNames[0]:
                elmNos = []
                for elm in data:
                    elmNo = elm[0]
                    elmNos.append(elmNo)
                beamsData[0] = elmNos
            elif egrp == beamGrpNames[1]:
                elmNos = []
                for elm in data:
                    elmNo = elm[0]
                    elmNos.append(elmNo)
                beamsData[1] = elmNos
        elif words[0] == "!EGROUP":
            egrp = pyFistr.getValName(header, "EGRP")
            if egrp == beamGrpNames[0]:
                beamsData[0] = data
            elif egrp == beamGrpNames[1]:
                beamsData[1] = data
        if len(beamsData[0]) > 0 and len(beamsData[1]) > 0:
            break
    return beamsData

#
#  get1BeamGroup
def get1BeamGroup(elmGrp):
    """ elmGrp内の全要素Noを取得する"""
    beamsData = []
    for headerData in meshHeaderData:
        header = headerData[0]
        data = headerData[1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            egrp = pyFistr.getValName(header, "EGRP")
            if egrp == elmGrp:
                elmNos = []
                for elm in data:
                    elmNo = elm[0]
                    elmNos.append(elmNo)
                beamsData += elmNos
        elif words[0] == "!EGROUP":
            egrp = pyFistr.getValName(header, "EGRP")
            if egrp == elmGrp:
                beamsData += data
    return beamsData

#
#  getBeamElmNoInGrp
def getBeamElmNoInGrp(beamGrp):
    elmNos = []
    for headerData in meshHeaderData:
        header = headerData[0]
        data = headerData[1]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            egrp = pyFistr.getValName(header, "EGRP")
            if egrp == beamGrp:
                eNos = list(map(lambda x: x[0], data))
                elmNos += eNos
        elif words[0] == "!EGROUP":
            egrp = pyFistr.getValName(header, "EGRP")
            if egrp == beamGrp:
                elmNos += data
    elmNos = list(set(elmNos))
    return elmNos

#
#  getBeamVector
def getBeamVector(elmNos):
    """ beamの単位ベクトルを取得"""
    locs = []
    for elmNo in elmNos:
        nodes = elementDict[elmNo][0]
        nodesLoc = list(map(lambda x: nodeDict[x][0], nodes))
        locs += nodesLoc
    locs.sort(key=lambda x: (x[0], x[1]))
    loc1 = locs[0]
    loc2 = locs[-1]
    vec = geo.vector(loc1, loc2)
    nVec = geo.normal(vec)
    return nVec

#
#  getElmVector
def getElmVector(elmNo):
    """ 要素Noからベクトルを取得する"""
    nodes = elementDict[elmNo][0]
    loc1 = nodeDict[nodes[0]][0]
    loc2 = nodeDict[nodes[1]][0]
    vec = geo.vector(loc1, loc2)
    vec = geo.normal(vec)
    return vec

#
#  getVectorsInGrp
def getVectorsInGrp(beamData):
    """ grp内から2ヶのベクトルを抽出する"""
    #vec1を取得
    elmNo = beamData[0]
    vec1 = getElmVector(elmNo)
    vec2 = vec1[:]
    #vec2を取得
    for elmNo in beamData:
        vec = getElmVector(elmNo)
        l = geo.length(vec1, vec)
        if l > 0.01 and l < 1.99:
            vec2 = vec
            break
    if vec1 == vec2:
        return [vec1]
    else:
        return [vec1, vec2]

#
#  getBeamVectorFromPoints
#--------------------------
def getBeamVectorFromPoints(vec1, elmNos, ngrp):
    """ beamとngrpで単位ベクトルを取得"""
    nodeNos = []
    for header, data in meshHeaderData:
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            name = pyFistr.getValName(header, "NGRP")
            if name == ngrp:
                nodeNos += data
    vec2 = []
    for nodeNo in nodeNos:
        loc1 = nodeDict[nodeNo][0]
        flag = 0
        vec2 = []
        for elmNo in elmNos:
            elmNode = elementDict[elmNo][0][0]
            loc2 = nodeDict[elmNode][0]
            if loc1 != loc2:
                vec = geo.vector(loc1, loc2)
                vec = geo.normal(vec)
                l = geo.length(vec1, vec)
                if 0.01 < l and l < 1.99:
                    vec2 = vec
                    flag = 1
                    break
        if flag == 1:
            break
    return vec2

#
#  getBeamRefDirWithVec
#-----------------------
def getBeamRefDirWithVec(egrp, vec2):
    """ egrpとvec2から参照方向を取得する"""
    global meshHeaderData, nodeDict, elementDict
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    nodeDict, elementDict = createNodeElementDict()
    #beamGrp = beamGrpNames[0]
    beamGrp = egrp
    #group内の要素Noを取得する
    beamData = getBeamElmNoInGrp(beamGrp)
    #vec1を取得
    elmNo = beamData[0]
    vec1 = getElmVector(elmNo)
    zVec = geo.crossProduct(vec1, vec2)
    zVec = geo.normal(zVec)
    vec = list(map(lambda x: "%10.8f" % x, zVec))
    print(_("参照軸方向(beamのz軸): ") + " ".join(beamGrpNames))
    print("  ", vec)
    return vec

#
#  getBeamRefDir
#-----------------
def getBeamRefDir():
    """ 2ヶのgrp名からbeamの参照方向を取得する。"""
    global meshHeaderData, nodeDict, elementDict
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    nodeDict, elementDict = createNodeElementDict()
    #beamGrps内の要素Noを取得
    beamsData = getBeamGroups()
    if len(beamsData[0]) == 0 or len(beamsData[1]) == 0:
        print(_("指定したbeam要素Grpは、存在しません。"))
        return []
    #beamの単位ベクトルを取得
    beamVec1 = getBeamVector(beamsData[0])
    beamVec2 = getBeamVector(beamsData[1])
    zVec = geo.crossProduct(beamVec1, beamVec2)
    zVec = geo.normal(zVec)
    vec = list(map(lambda x: "%10.8f" % x, zVec))
    print(_("参照軸方向(beamのz軸): ") + beamGrpNames[0] + "," + beamGrpNames[1])
    print("  ", vec)
    return vec

#
#  getBeamRefDirInGrp
#---------------------
def getBeamRefDirInGrp():
    """ 1ヶのgrpのデータのみでbeamの参照方向を取得する。"""
    global meshHeaderData, nodeDict, elementDict
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    nodeDict, elementDict = createNodeElementDict()
    beamGrp = beamGrpNames[0]
    #group内の要素Noを取得する
    beamData = getBeamElmNoInGrp(beamGrp)
    #単位ベクトルを取得
    vectors = getVectorsInGrp(beamData)
    if len(vectors) == 1:
        print(_("2種類のベクトルが取得できません。"))
        return []
    zVec = geo.crossProduct(vectors[0], vectors[1])
    zVec = geo.normal(zVec)
    vec = list(map(lambda x: "%10.8f" % x, zVec))
    print(_("参照軸方向(beamのz軸): ") + beamGrp)
    print("  ", vec)
    return vec

#
#  getBeamRefDirXYZ
#-------------------
def getBeamRefDirXYZ():
    """ 1ヶのgrpと指定したXYZ軸を使って、参照方向を取得する。"""
    global meshHeaderData, nodeDict, elementDict
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    nodeDict, elementDict = createNodeElementDict()
    beamGrp = beamGrpNames[0]
    #group内の要素Noを取得する
    beamData = getBeamElmNoInGrp(beamGrp)
    #vec1を取得
    elmNo = beamData[0]
    vec1 = getElmVector(elmNo)
    #vec2を取得
    if beamGrpNames[1] == "-x":
        vec2 = [1.0, 0.0, 0.0]
    elif beamGrpNames[1] == "-y":
        vec2 = [0.0, 1.0, 0.0]
    elif beamGrpNames[1] == "-z":
        vec2 = [0.0, 0.0, 1.0]
    zVec = geo.crossProduct(vec1, vec2)
    zVec = geo.normal(zVec)
    print(_("参照軸方向(beamのz軸): ") + " ".join(beamGrpNames))
    print("  ", zVec)

#
#  getBeamRefDirWithP
#---------------------
def getBeamRefDirWithP(elmGrp, ngrp):
    """ beam要素GrpとNGRPで参照方向を取得する"""
    global meshHeaderData, nodeDict, elementDict
    #lines = pyFistr.readFistrModelMsh()
    #meshHeaderData = pyFistr.getHeaderNumData(lines)
    meshHeaderData = pyFistr.getHeaderNumDataMsh(os.getcwd())
    nodeDict, elementDict = createNodeElementDict()
    elmNos = get1BeamGroup(elmGrp)
    if len(elmNos) == 0:
        print(_("指定したbeam要素Grpは、存在しません。"))
        return []
    #beamの単位ベクトルを取得
    beamVec1 = getBeamVector(elmNos)
    beamVec2 = getBeamVectorFromPoints(beamVec1, elmNos, ngrp)
    if len(beamVec2) == 0:
        print(_("参照軸方向が取得できません。"))
        return []
    zVec = geo.crossProduct(beamVec1, beamVec2)
    zVec = geo.normal(zVec)
    vec = list(map(lambda x: "%10.8f" % x, zVec))
    print(_("参照軸方向(beamのz軸): ") + " ".join(beamGrpNames) + " -p " + ngrp)
    print("  ", vec)
    return vec

#
#  printHelp
#------------
def printHelp():
    """ helpを出力する"""
    cont  = "\n"
    cont += "-------- getBeamRefDir.py ---------------------------------------------------\n"
    cont += "beam要素GrpからbeamのZ軸方向の単位ベクトルを出力する。\n"
    cont += "その作成方法は、以下のいずれかの方法で平面を作り出す。\n"
    cont += "  ・2本のbeam要素Grpから平面を作り出す。\n"
    cont += "  ・1本のbeam要素GrpとXYZ軸から平面を作り出す\n"
    cont += "  ・1ヶの要素Grpから2方向のベクトルを検索し、そのベクトルから平面を作り出す。\n"
    cont += "  ・1本のbeam要素Grpと離れた点で平面を作り出す。\n"
    cont += "作り出した平面に垂直な単位ベクトルをbeamのZ軸方向として出力する。\n"
    cont += "<使い方>\n"
    cont += "  getBeamRefDir.py <beam要素Grp1> <beam要素Grp2>\n"
    cont += "       beam要素Grp1 と beam要素Grp2 は、各々直線のbeam要素Grp\n"
    cont += "  getBeamRefDir.py <beam要素Grp> <-x or -y or -z>\n"
    cont += "       直線のbeam要素Grp、各軸方向を指定\n"
    cont += "  getBeamRefDir.py <beam要素Grp>\n"
    cont += "       複数の直線を含むbeam要素Grp\n"
    cont += "  getBeamRefDir.py <beam要素Grp> -p <節点Grp>\n"
    cont += "<使用例>\n"
    cont += "  getBeamRefDir.py beam1 beam2\n"
    cont += "  getBeamRefDir.py beam1 -x\n"
    cont += "  getBeamRefDir.py beamGrp\n"
    cont += "  getBeamRefDir.py beam1 -p point\n"
    print(cont)


#
#  getArguments
#------------------
def getArguments():
    """ 引数の内容を取得する"""
    beamGrps = []
    ngrp = ""
    if len(sys.argv) < 2:
        printHelp()
        return beamGrps, ngrp
    elif len(sys.argv) == 2:
        beamGrps = [sys.argv[1]]
    elif len(sys.argv) == 3:
        beamGrps = [sys.argv[1], sys.argv[2]]
    elif len(sys.argv) >= 4:
        beamGrps = [sys.argv[1]]
        ngrp = sys.argv[3]
    if beamGrps[0] == "-h" or beamGrps[0] == "--help":
        printHelp()
        return [], ngrp
    return beamGrps, ngrp

if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    #_ = gettext.gettext

    beamGrpNames, ngrp = getArguments()
    currDir = os.getcwd()
    if len(beamGrpNames) == 2:
        if beamGrpNames[1] == "-x" or beamGrpNames[1] == "-y" or beamGrpNames[1] == "-z":
            getBeamRefDirXYZ()
        else:
            vec = getBeamRefDir()
    elif len(beamGrpNames) == 1 and ngrp == "":
        vec = getBeamRefDirInGrp()
    elif len(beamGrpNames) == 1 and ngrp != "":
        vec = getBeamRefDirWithP(beamGrpNames[0], ngrp)
        