#!/usr/bin/env python3
#  coding; utf-8
#
#   fistrRes2vtu.py
#
#       fistrが出力したvtuファイルにelementGroupを追加する。
#       下表の要素を変換する。
#
#       fistr   node    vtk     note
#       -----------------------------------------
#       341     4       10      四面体1次
#       342     10      24      四面体2次
#       361     8       12      六面体1次
#       362     20      25      六面体2次
#       351     6       13      五面体1次
#       352     15      26      五面体2次
#       611     2       3       beam要素
#       641     4       3       beam要素（3自由度）
#       731     3       5       シェル三角形1次
#       732     6       22      シェル三角形2次
#       741     4       9       シェル四角形1次
#       761     6       5       シェル三角形1次（3自由度）
#       781     8       9       シェル四角形1次（3自由度）
#
#
#   20/04/06    新規作成
#   22/03/15    国際化
#   24/07/07    openにencoding="utf-8"を追加
#

import os
import sys
import glob
import multiprocessing
import convertMesh as cm

ls = "\n"

#以下は、bufferサイズを「0」にする設定
#  subprocess等で起動した時、print内容をbufferに溜め込んでrealTimeに出力されない
#  ので、以下の設定でbufferサイズを「0」にする。
#  又は、起動時に「python3 -u」オプション付きで起動する。
#sys.stdout = os.fdopen(sys.stdout.fileno(), 'w', buffering=1)
#sys.stderr = os.fdopen(sys.stderr.fileno(), 'w', buffering=1)
#sys.stdin = os.fdopen(sys.stdin.fileno(), 'r', buffering=1)

#
#  readFileLines
def readFileLines(fileName):
    """ fileをlines(list)形式で読込返す。"""
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    return lines

#
#  writeFileLines
def writeFileLines(fileName, lines):
    """ linesを保存する"""
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  deleteSp
def deleteSp(line):
    """ 空白とCRを削除して返す"""
    line = line.replace(" ", "")
    line = line.replace(ls, "")
    return line

#
#  getValName
def getValName(line, valName, sep=","):
    """ valName(type等）を取得して返す"""
    if sep == " ":
        words = line.split()
    else:
        line = deleteSp(line)
        words = line.split(sep)
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        if word.find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    return name

#
#  getNumberOfSurfaces
def getNumberOfSurfaces(vsuGrps):
    """ surfaceGroupからfaceの数を取得して返す"""
    nFaces = 0
    for grp in vsuGrps:
        nFaces += len(grp[1])
    return nFaces

#
#  getMeshFileName
def getMeshFileName(headerName):
    """ headerNameからmshファイルを取得して返す。
    ファイルが存在しない場合は、「""」を返す"""
    if headerName.split(".")[-1] == "msh":
        fileName = headerName
    else:
        fileName = headerName + ".msh"
    if os.path.exists(fileName) == False:
        print("error: " + _("「") + headerName + _("」が存在しません。"))
        fileName = ""
    return fileName

#
#  getSingleResultFileNames
def getSingleResultFileNames():
    """ singleCoreの場合、resFileから結果fileNameを取得して返す"""
    names = glob.glob(resFile + ".res.0.*")
    fileNames = []
    for name in names:
        if name[-1] != "~" and name.split(".")[-1] != "inp":
            fileNames.append(name)
    return fileNames

#
#  getSinglePvtuFileNames
def getSinglePvtuFileNames():
    """ singleCoreの場合、pvtuファイルを取得して返す。"""
    names = glob.glob(resFile + ".vis_psf.*.pvtu")
    return names    

#
#  sortStepFileName
def sortStepFileName(names):
    """ fileNameをstepでsortする。"""
    steps = []
    for name in names:
        step = int(name.split(".")[-1])
        steps.append(step)
    steps.sort()
    nameHeader = ".".join(names[0].split(".")[:-1])
    names = []
    for step in steps:
        fileName = nameHeader + "." + str(step)
        names.append(fileName)
    return names

#
#  sortStepPvtuFileName
def sortStepPvtuFileName(names):
    """ pvtuファイルをstepでsortする。"""
    steps = []
    for name in names:
        step = int(name.split(".")[-2])
        steps.append(step)
    steps.sort()
    nameHeader = ".".join(names[0].split(".")[:-2])
    names = []
    for step in steps:
        stepName = "0"*(4-len(str(step))) + str(step)
        fileName = nameHeader + "." + stepName + ".pvtu"
        names.append(fileName)
    return names

#
#  getNewNodeNo
#----------------
def getNewNodeNo(nodeList, meshHeaderNumConts):
    """ 未使用のnodeNoを調べ、新しいnodeNoを設定（「0」始まり）。
    nodeNoは、出現順に設定する。（結果dataが出現順の為）
    
    Args:
        nodeList (list)             :旧nodeList
        meshHeaderNumConts (list)   :header毎のnumData
    Returns:
        nodeList (list) :使用有無、新nodeNo追加
        vnList (list)   :vtk用のnodeList"""
    vnList = []
    newNo = 0
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!NODE":
            nodeData = headerCont[1]
            for nodeCont in nodeData:
                nodeNo = nodeCont[0]
                if len(nodeList[nodeNo][1]) > 0:    #使用有無チェック
                    loc = nodeCont[1:]
                    nodeList[nodeNo].append(newNo)
                    vnList.append([loc])
                    newNo += 1
    return [nodeList, vnList]

#
#  renumberNodeElement
#----------------------
def renumberNodeElement(nodeList, elementList, meshHeaderNumConts):
    """ nodeNoとelNoを書き換え。
    elNoは、出現順に設定する。（結果dateが出現順の為）
    
    Args:
        nodeList (list)             :旧nodeList
        elementList (list)          :旧elementList
        meshHeaderNumConts (list)   :header毎のnumData
    Returns:
        elementList (list)  :新elNoを追加
        veList (list)       :vtk用のelNo
                            :[[<grpNo>, nodeNos], <eltype>],..."""
    #grpNameのgrpNoを設定
    grpNos = {}             #grpNameとgrpNoの辞書
    grpNameNo = 1
    for headerCont in meshHeaderNumConts:
        words = deleteSp(headerCont[0]).split(",")
        if words[0] == "!ELEMENT":
            grpName = getValName(headerCont[0], "EGRP")
            if not grpName in grpNos.keys():
                grpNos[grpName] = grpNameNo
                grpNameNo += 1
    #elNoとgrpNoを設定
    veList = []
    newNo = 0
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            grpName = getValName(header, "EGRP")
            elData = headerCont[1]
            for elCont in elData:
                elNo = elCont[0]
                elNodes = elCont[1:]
                newNodes = []
                for elNode in elNodes:
                    newNodeNo = nodeList[elNode][-1]
                    newNodes.append(newNodeNo)
                elType = elementList[elNo][1]
                grpNo = grpNos[grpName]             #辞書からgrpNoを取得
                [vtkName, newNodes] = cm.fistr2vtk_el(elType, newNodes)
                newElCont = [[grpNo] + newNodes] + [vtkName]
                veList.append(newElCont)
                elementList[elNo] += [newNo]
                newNo += 1
    return [elementList, veList]

#
#  renumberVeList
#-----------------
def renumberVeList(veList):
    """ veListをsolid、シェル、ビームの順に並べ替えて返す"""
    elDict = {
        "3": 2,     #"beam"
        "4": 2,     #"beam"
        "5": 1,     #"shell"
        "7": 1,     #"shell"
        "9": 1,     #"shell"
        "10": 0,    #"solid"
        "12": 0,    #"solid"
        "13": 0,    #"solid"
        "14": 0,    #"solid"
        "21": 2,    #"beam"
        "22": 1,    #"shell"
        "23": 1,    #"shell"
        "24": 0,    #"solid"
        "25": 0,    #"solid"
        "26": 0     #"solid"
        }
    newList = [[], [], []]
    for i in range(len(veList)):
        elm = veList[i]
        newList[elDict[elm[-1]]].append(elm)
    ve = newList[0] + newList[1] + newList[2]
    return ve

#
#  getElmGrpField
#-----------------
def getElmGrpField(veList):
    """ elementGroupNoを取得する。"""
    grpData = list(map(lambda x: [str(x[0][0])], veList))
    elmGrpField = ["elementGroup", grpData]
    return elmGrpField

#
#  addFieldsToVtkFile
#----------------------
def addFieldsToVtkFile(fileName, elmGrpField):
    """ vtuファイルにelmGrpNoを追加する。"""
    lines = readFileLines(fileName)
 


#
#  getNodalResults
def getNodalResults(lines, elementList):
    """ nodeのdataを取得する"""

    def getGrpNoFromElmNo(elmNo):
        """ elmNoから要素grpNoを取得する"""
        if elmNo >= len(elementList):
            grpNo = "0"
        elif len(elementList[elmNo]) == 0:
            grpNo = "0"
        else:
            grpNo = str(elementList[elmNo][0][0])
        return grpNo

    nodalFields = []
    elmFields = []
    #*dataを検索
    flag = 0
    for i in range(len(lines)):
        if lines[i][:len("*data")] == "*data":
            flag = 1
            break
    if flag == 0:
        print(_("error: nodalResult が読み取れません"))
        return [nodalFields, elmFields]

    #node数、要素数を取得
    i += 1
    words = lines[i].split()
    (nNodes, nElms) = (int(words[0]), int(words[1]))
    #変数の数を取得
    i += 1
    words = lines[i].split()
    (nNodalVals, nElmVals) = (int(words[0]), int(words[1]))

    #nodal側を取得-------------------
    #nodal変数の桁数を取得
    i += 1
    words = lines[i].split()
    comps = list(map(lambda x: int(x), words))
    nFloat = sum(comps)
    #nodal変数名を取得
    i += 1
    valNames = []
    for _ii in range(nNodalVals):
        valName = lines[i].split()[0]
        valNames.append(valName)
        i += 1
    #nodal変数の配列を準備
    nodalVals = [ [] for ii in range(nNodalVals)]
    #各nodal変数の値を取得
    for _ii in range(nNodes):
        _nodeNo = lines[i]
        i += 1
        vals = []
        loop = True
        while loop:
            # NANをチェックし、NANがあれば、0.0に置き換える
            line = lines[i].replace(" NAN ", " 0.0 ")
            words = line.split()
            vals += words
            i += 1
            if len(vals) >= nFloat:
                break
        #値を取得
        st = 0
        for jj in range(len(comps)):
            ed = st + comps[jj]
            val = vals[st:ed]
            nodalVals[jj].append(val)
            st += comps[jj]
    #field名をセット
    for ii in range(nNodalVals):
        valName = valNames[ii]
        vals = nodalVals[ii]
        nodalFields.append([valName, vals])

    #要素側を取得----------------------
    if i >= len(lines):
        return [nodalFields, elmFields]
    grpVals = []
    #桁数を取得
    words = lines[i].split()
    comps = list(map(lambda x: int(x), words))
    nFloat = sum(comps)
    #変数名を取得
    i += 1
    valNames = []
    for _ii in range(nElmVals):
        valName = lines[i].split()[0]
        valNames.append(valName)
        i += 1
    #変数の配列を準備
    elmVals = [ [] for ii in range(nElmVals)]
    #各変数の値を取得
    for _ii in range(nElms):
        #grpNoを取得
        elmNo = lines[i]
        grpNo = getGrpNoFromElmNo(int(elmNo))
        grpVals.append([grpNo])
        #要素データを取得
        i += 1
        vals = []
        loop = True
        while loop:
            # NANをチェックし、NANがあれば、0.0に置き換える
            line = lines[i].replace(" NAN ", " 0.0 ")
            words = line.split()
            vals += words
            i += 1
            if len(vals) >= nFloat:
                break
        #値を取得
        st = 0
        for jj in range(len(comps)):
            ed = st + comps[jj]
            val = vals[st:ed]
            elmVals[jj].append(val)
            st += comps[jj]
    #field名をセット
    for ii in range(nElmVals):
        valName = valNames[ii]
        vals = elmVals[ii]
        elmFields.append([valName, vals])
    elmFields.append(["elementGroup", grpVals])
    return [nodalFields, elmFields]

#
#  getResultFields
#------------------
def getResultFields(fileName, elementList):
    """ 結果fieldをnodeとcellに分けて取得する"""
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    #nodeFieldを作成
    (nodeFields, elmFields) = getNodalResults(lines, elementList)
    return (nodeFields, elmFields)



#
#  addElmGrpToVtu
#------------------
def addElmGrpToVtu(meshFile):
    """ fistrが出力したvtuファイルにelementGroupを追加する。"""
    meshFileName = getMeshFileName(meshFile)
    if meshFile == "":
        return
    lines = readFileLines(meshFileName)
    #mesh内容を読込
    meshHeaderNumConts = cm.getFistrHeaderNumData(lines)
    #メッシュ内容を取得
    [nodeList, elementList] = cm.getFistrMeshConts(meshHeaderNumConts)
    print (_("meshファイルを変換中..."))
    #node番号の振り直し（未使用nodeを確認）vtk用のvnList, veListを取得
    [nodeList, vnList] = getNewNodeNo(nodeList, meshHeaderNumConts)
    [elementList, veList] = renumberNodeElement(nodeList, elementList, meshHeaderNumConts)
    #veListをsolid、シェル、ビーム要素順に並べ替える
    veList = renumberVeList(veList)
    if parallel == False:
        #singleCoreの場合
        fileNames = getSingleResultFileNames()
        (nodeFields, cellFields) = getResultFields(fileNames[0], elementList)


        fileNames = getSinglePvtuFileNames()
        if len(fileNames) == 0:
            print(_("error: 結果fileが存在しません"))
            return
        fileNames = sortStepPvtuFileName(fileNames)
        names = list(map(lambda x: os.path.basename(x), fileNames))
        for i in range(len(fileNames)):
            fileName = fileNames[i]
            print("  converting " + names[i] + "...")
            #計算結果のfieldを取得
            elmGrpField = getElmGrpField(veList)
            addFieldsToVtkFile(fileName, elmGrpField)



if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))
    if len(sys.argv) < 2:
        print(_("error: headerが入力されていません。"))
        exit()
    resFile = sys.argv[1]
    parallel = False
    addElmGrpToVtu(resFile)

