#!/usr/bin/env python3
#  coding; utf-8
#
#   fistrMesh2vtu.py
#
#       fistr形式のmeshをvtu形式（vtk）に変換する。
#       下表の要素を変換する。
#
#       fistr   node    vtk     note
#       -----------------------------------------
#       341     4       10      四面体1次
#       342     10      24      四面体2次
#       361     8       12      六面体1次
#       362     20      25      六面体2次
#       351     6       13      五面体1次
#       352     15      26      五面体2次
#       611     2       3       beam要素
#       641     4       3       beam要素（3自由度）
#       731     3       5       シェル三角形1次
#       732     6       22      シェル三角形2次
#       741     4       9       シェル四角形1次
#       761     6       5       シェル三角形1次（3自由度）
#       781     8       9       シェル四角形1次（3自由度）
#
#
#   20/04/06    新規作成
#   22/03/15    国際化
#      04/12    log表示修正（import logFileCreaterを追加）
#      04/26    log表示削除
#   24/07/07    openにencoding="utf-8"を追加
#

import os
import sys
import glob
import multiprocessing
import convertMesh as cm

ls = "\n"

#以下は、bufferサイズを「0」にする設定
#  subprocess等で起動した時、print内容をbufferに溜め込んでrealTimeに出力されない
#  ので、以下の設定でbufferサイズを「0」にする。
#  又は、起動時に「python3 -u」オプション付きで起動する。
#sys.stdout = os.fdopen(sys.stdout.fileno(), 'w', buffering=1)
#sys.stderr = os.fdopen(sys.stderr.fileno(), 'w', buffering=1)
#sys.stdin = os.fdopen(sys.stdin.fileno(), 'r', buffering=1)

#
#  readFileLines
def readFileLines(fileName):
    """ fileをlines(list)形式で読込返す。"""
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    return lines

#
#  writeFileLines
def writeFileLines(fileName, lines):
    """ linesを保存する"""
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  deleteSp
def deleteSp(line):
    """ 空白とCRを削除して返す"""
    line = line.replace(" ", "")
    line = line.replace(ls, "")
    return line

#
#  getValName
def getValName(line, valName, sep=","):
    """ valName(type等）を取得して返す"""
    if sep == " ":
        words = line.split()
    else:
        line = deleteSp(line)
        words = line.split(sep)
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        if word.find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    return name

#
#  getNumberOfSurfaces
def getNumberOfSurfaces(vsuGrps):
    """ surfaceGroupからfaceの数を取得して返す"""
    nFaces = 0
    for grp in vsuGrps:
        nFaces += len(grp[1])
    return nFaces

#----------- vtu作成 ------------------
#
#  getLinesWordsN
def getLinesWordsN(words, n):
    """ n words毎に改行して返す"""
    lines = []
    for i in range(0, len(words), n):
        line = " ".join(words[i:i + n]) + ls
        lines.append(line)
    return lines

#
#  createVtkHeader
def createVtkHeader():
    """ vtkファイルのheader部を作成して返す"""
    lines  = ['<?xml version="1.0"?>' + ls]
    lines += ['<VTKFile type="UnstructuredGrid" version="1.0">' + ls]
    lines += ['<UnstructuredGrid>' + ls]
    return lines

#
#  createVtkFooter
def createVtkFooter():
    """ vtkファイルのfooterを作成して返す"""
    lines  = ['</UnstructuredGrid>' + ls]
    lines += ['</VTKFile>' + ls]
    return lines

#
#  createVtkPieceHeader
def createVtkPieceHeader(vnList, veList, vsuGrps):
    """ pieceSectionのheaderを設定"""
    nNodes = len(vnList)
    nCells = len(veList)
    nFaces = getNumberOfSurfaces(vsuGrps)
    lines = ['<Piece NumberOfPoints="' + str(nNodes) + '" NumberOfCells="' + str(nCells + nFaces) + '">' + ls]
    return lines

#
#  createVtkPieceFooter
def createVtkPieceFooter():
    """ pieceSectionのfooterを設定"""
    lines = ['</Piece>' + ls]
    return lines

#
#  createVtkNodes
def createVtkNodes(vnList):
    """ node部を作成"""
    lines = ['<Points>' + ls]
    lines += ['<DataArray type="Float32" NumberOfComponents="3" format="ascii">' + ls]
    for nData in vnList:
        loc = nData[0]
        loc = list(map(lambda x: str(x), loc))
        line = " ".join(loc) + ls
        lines.append(line)
    lines += ['</DataArray>' + ls]
    lines += ['</Points>' + ls] 
    return lines

#
#  createVtkCells
def createVtkCells(veList, vsuGrps):
    """ cell部を作成する"""
    #node順入れ替え用の辞書
    changeOrderDict = {
     #node数 新order  入れ替え関数名
        3:[[0,1,2],   cm.order_3z],
        4:[[0,1,2,3], cm.order_4z],
        6:[[0,2,4,1,3,5], cm.order_6z],
        8:[[0,2,4,6,1,3,5,7], cm.order_8z]
        }
    #node数からface要素名を取得する辞書
    #  面要素のnode数とelType名
    faceNumToFistrVtkTypeDict = {
        #node数: fistr名、vtk名
            3:   ["231",  "5"],
            4:   ["241",  "9"],
            6:   ["232",  "22"],
            8:   ["242",  "23"]
        }

    lines = ['<Cells>' + ls]
    #elementを作成
    lines += ['<DataArray type="Int32" Name="connectivity" format="ascii">' + ls]
    for eData in veList:
        eNodes = eData[0][1:]
        eNodes = list(map(lambda x: str(x), eNodes))
        line = " ".join(eNodes) + ls
        lines.append(line)
    #surface要素を作成
    cells = []
    for grpData in vsuGrps:
        suNodes = grpData[1]
        for nodes in suNodes:
            #2次要素の場合、node順を入れ替え
            #  左回り順　→　2次要素順（頂点→中間点）
            [newOrder, func] = changeOrderDict[len(nodes)]
            cell = func(nodes, newOrder)
            cells.append(cell)
            words = list(map(str, cell))    #cellをstrのlistに変換
            line = " ".join(words) + ls
            lines.append(line)
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="Int32" Name="offsets" format="ascii">' + ls]
    #offsetを作成
    offset = 0
    data = []
    #  solid要素
    for eData in veList:
        eNodes = eData[0][1:]
        offset += len(eNodes)
        data.append(str(offset))
    #  surfaceを作成
    for cell in cells:
        offset += len(cell)
        data.append(str(offset))
    #lines += [" ".join(data) + ls]
    lines += getLinesWordsN(data, 10)      #10 words毎に改行する
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="UInt8" Name="types" format="ascii">' + ls]
    #typesを作成
    data = []
    #  solid要素
    for eData in veList:
        elType = eData[1]
        data.append(elType)
    #  surface要素
    for cell in cells:
        elType = faceNumToFistrVtkTypeDict[len(cell)][1]
        data.append(elType)
    #lines += [" ".join(data) + ls]
    lines += getLinesWordsN(data, 20)
    lines += ['</DataArray>' + ls]
    lines += ['</Cells>' + ls]
    return lines

#
#  createVtkNodesData
def createVtkNodesData(vnList, vndGrps):
    """ nodesData(nodeGroupNo)を作成"""
    nNodes = len(vnList)
    ndNos = [ 0 for i in range(nNodes) ]
    #ndGrpNoを設定
    for i in range(len(vndGrps)):
        ndGrpNo = i + 1
        nodes = vndGrps[i][1]
        for ndNo in nodes:
            ndNos[ndNo] = ndGrpNo
    #linesを作成
    lines = ['<PointData>' + ls]
    lines += ['<DataArray type="Int16" Name="nodeGroup" NumberOfComponents="1" format="ascii">' + ls]
    ndGrpNos = list(map(lambda x:str(x), ndNos))
    #line = " ".join(ndGrpNos) + ls
    #lines += [line]
    lines += getLinesWordsN(ndGrpNos, 20)
    lines += ['</DataArray>' + ls]
    lines += ['</PointData>' + ls]
    return lines

#
#  createVtkCellsData
def createVtkCellsData(veList, vsuGrps):
    """ cellData(elGroupNo, suGroupNo)を作成"""
    #faceを含めた全cell数を取得
    nFaces = 0
    for grpData in vsuGrps:
        nFaces += len(grpData[1])
    nCells = len(veList) + nFaces
    #solid要素の要素grpNoをセット
    elGrpNos = [ 0 for i in range(nCells)]
    for elNo in range(len(veList)):
        elGrpNo = veList[elNo][0][0]
        elGrpNos[elNo] = elGrpNo
    #surfaceのgrpNoをセット
    suGrpNos = [ 0 for i in range(nCells)]
    faceNo = len(veList) - 1
    for i in range(len(vsuGrps)):
        grpData = vsuGrps[i]
        faces = grpData[1]
        suGrpNo = i + 1
        for _ii in range(len(faces)):
            faceNo += 1
            suGrpNos[faceNo] = suGrpNo
    #linesを作成
    lines = ['<CellData>' + ls]
    lines += ['<DataArray type="Int16" Name="elementGroup" NumberOfComponents="1" format="ascii">' + ls]
    elGrpNos = list(map(lambda x: str(x), elGrpNos))
    lines += getLinesWordsN(elGrpNos, 20)
    lines += ['</DataArray>' + ls]
    lines += ['<DataArray type="Int16" Name="surfaceGroup" NumberOfComponents="1" format="ascii">' + ls]
    suGrpNos = list(map(lambda x: str(x), suGrpNos))
    lines += getLinesWordsN(suGrpNos, 20)
    lines += ['</DataArray>' + ls]
    lines += ['</CellData>' + ls]
    return lines

def getDtypeFromName(name):
    """ Nameからdtypeを返す。"""
    dtype = "Float64"
    if name == "elementGroup":
        dtype = "Int32"
    return dtype

#
#  createVtkAddValue
def createVtkAddValue(fields):
    """ 追加するvtkDataArrayを作成する"""
    lines = []
    for field in fields:
        name = field[0]
        data = field[1]
        dtype = getDtypeFromName(name)
        nComp = len(data[0])
        lines += ['<DataArray type="' + dtype + '" Name="' + name + '" NumberOfComponents="' + str(nComp) + '" format="ascii">' + ls]
        for val in data:
            lines += [" ".join(val) + ls]
        lines += ['</DataArray>' + ls]
    return lines

#
#  insertVtkPointData
def insertVtkPointData(lines, addLines):
    """ pointDataにaddLinesを追加する"""
    for i in range(len(lines)):
        line = lines[i]
        if line.find("</PointData>") >= 0:
            lines = lines[:i] + addLines + lines[i:]
            break
    return lines

#
#  insertVtkCellData
def insertVtkCellData(lines, addLines):
    """ cellDataにaddLinesを追加する"""
    for i in range(len(lines)):
        line = lines[i]
        if line.find("</CellData>") >= 0:
            lines = lines[:i] + addLines + lines[i:]
            break
    return lines
#--------------------------------------

#
#  getMeshFileName
def getMeshFileName(headerName):
    """ headerNameからmshファイルを取得して返す。
    ファイルが存在しない場合は、「""」を返す"""
    if headerName.split(".")[-1] == "msh":
        fileName = headerName
    else:
        fileName = headerName + ".msh"
    if os.path.exists(fileName) == False:
        print("error: " + _("「") + headerName + _("」が存在しません。"))
        fileName = ""
    return fileName

#
#  getNewNodeNo
#----------------
def getNewNodeNo(nodeList, meshHeaderNumConts):
    """ 未使用のnodeNoを調べ、新しいnodeNoを設定（「0」始まり）。
    nodeNoは、出現順に設定する。（結果dataが出現順の為）
    
    Args:
        nodeList (list)             :旧nodeList
        meshHeaderNumConts (list)   :header毎のnumData
    Returns:
        nodeList (list) :使用有無、新nodeNo追加
        vnList (list)   :vtk用のnodeList"""
    vnList = []
    newNo = 0
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!NODE":
            nodeData = headerCont[1]
            for nodeCont in nodeData:
                nodeNo = nodeCont[0]
                if len(nodeList[nodeNo][1]) > 0:    #使用有無チェック
                    loc = nodeCont[1:]
                    nodeList[nodeNo].append(newNo)
                    vnList.append([loc])
                    newNo += 1
    return [nodeList, vnList]

#
#  renumberNodeElement
#----------------------
def renumberNodeElement(nodeList, elementList, meshHeaderNumConts):
    """ nodeNoとelNoを書き換え。
    elNoは、出現順に設定する。（結果dateが出現順の為）
    
    Args:
        nodeList (list)             :旧nodeList
        elementList (list)          :旧elementList
        meshHeaderNumConts (list)   :header毎のnumData
    Returns:
        elementList (list)  :新elNoを追加
        veList (list)       :vtk用のelNo
                            :[[<grpNo>, nodeNos], <eltype>],..."""
    #grpNameのgrpNoを設定
    grpNos = {}             #grpNameとgrpNoの辞書
    grpNameNo = 1
    for headerCont in meshHeaderNumConts:
        words = deleteSp(headerCont[0]).split(",")
        if words[0] == "!ELEMENT":
            grpName = getValName(headerCont[0], "EGRP")
            if not grpName in grpNos.keys():
                grpNos[grpName] = grpNameNo
                grpNameNo += 1
    #elNoとgrpNoを設定
    veList = []
    newNo = 0
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            grpName = getValName(header, "EGRP")
            elData = headerCont[1]
            for elCont in elData:
                elNo = elCont[0]
                elNodes = elCont[1:]
                newNodes = []
                for elNode in elNodes:
                    newNodeNo = nodeList[elNode][-1]
                    newNodes.append(newNodeNo)
                elType = elementList[elNo][1]
                #3自由度のbeam、シェル（三角形、四角形）の場合
                #  dummyNodeを削除して変換
                if elType == "641":
                    newNodes = newNodes[:len(newNodes)//2]
                    elType = "611"
                elif elType == "761":
                    newNodes = newNodes[:len(newNodes)//2]
                    elType = "731"
                elif elType == "781":
                    newNodes = newNodes[:len(newNodes)//2]
                    elType = "741"
                grpNo = grpNos[grpName]             #辞書からgrpNoを取得
                [vtkName, newNodes] = cm.fistr2vtk_el(elType, newNodes)
                newElCont = [[grpNo] + newNodes] + [vtkName]
                veList.append(newElCont)
                elementList[elNo] += [newNo]
                newNo += 1
    return [elementList, veList]

#
#  renumberVeList
#-----------------
def renumberVeList(veList):
    """ veListをsolid、シェル、ビームの順に並べ替えて返す"""
    elDict = {
        "3": 2,     #"beam"
        "4": 2,     #"beam"
        "5": 1,     #"shell"
        "7": 1,     #"shell"
        "9": 1,     #"shell"
        "10": 0,    #"solid"
        "12": 0,    #"solid"
        "13": 0,    #"solid"
        "14": 0,    #"solid"
        "21": 2,    #"beam"
        "22": 1,    #"shell"
        "23": 1,    #"shell"
        "24": 0,    #"solid"
        "25": 0,    #"solid"
        "26": 0     #"solid"
        }
    newList = [[], [], []]
    for i in range(len(veList)):
        elm = veList[i]
        newList[elDict[elm[-1]]].append(elm)
    ve = newList[0] + newList[1] + newList[2]
    return ve

#
#  getDummyNodes
#----------------
def getDummyNodes(elementList, nodeList, meshHeaderNumConts):
    """ dummy節点とその相手nodeNoを取得"""
    dummyNodes = []
    for meshCont in meshHeaderNumConts:
        header = meshCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            elType = getValName(header, "TYPE")
            #dummyNodeを取得
            if elType == "761":
                #三角形
                elmData = meshCont[1]
                for elm in elmData:
                    nodes = elm[1:]
                    dummyNodes.append([nodeList[nodes[3]][2], nodeList[nodes[0]][2]])
                    dummyNodes.append([nodeList[nodes[4]][2], nodeList[nodes[1]][2]])
                    dummyNodes.append([nodeList[nodes[5]][2], nodeList[nodes[2]][2]])
            elif elType == "781":
                #四角形
                elmData = meshCont[1]
                for elm in elmData:
                    nodes = elm[1:]
                    dummyNodes.append([nodeList[nodes[4]][2], nodeList[nodes[0]][2]])
                    dummyNodes.append([nodeList[nodes[5]][2], nodeList[nodes[1]][2]])
                    dummyNodes.append([nodeList[nodes[6]][2], nodeList[nodes[2]][2]])
                    dummyNodes.append([nodeList[nodes[7]][2], nodeList[nodes[3]][2]])
            elif elType == "641":
                #beam
                elmData = meshCont[1]
                for elm in elmData:
                    nodes = elm[1:]
                    dummyNodes.append([nodeList[nodes[2]][2], nodeList[nodes[0]][2]])
                    dummyNodes.append([nodeList[nodes[3]][2], nodeList[nodes[1]][2]])
    return dummyNodes

#
#  renumberGrps
#--------------
def renumberGrps(meshHeaderNumConts, nodeList, elementList):
    """ node, surfaceのgroupを取得する"""
    vndGrps = []
    vsuGrps = []
    for headerData in meshHeaderNumConts:
        header = headerData[0]
        words = deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            #nodeGrpを取得
            grpName = getValName(header, "NGRP")
            nodes = headerData[1]
            newNodes = []
            for node in nodes:
                newNo = nodeList[node][-1]
                newNodes.append(newNo)
            grp = [grpName, newNodes]
            vndGrps.append(grp)
        elif words[0] == "!SGROUP":
            #surfaceGrpを取得
            grpName = getValName(header, "SGRP")
            suData = headerData[1]
            grpData = []
            for i in range(0, len(suData), 2):
                elNo = suData[i]
                faceNo = suData[i+1]
                newNo = elementList[elNo][-1]
                elType = elementList[elNo][1]
                elNodes = elementList[elNo][0][1:]
                #faceNodeを左回りで連続して取得
                nodes = cm.fistr_elFaceNode(elType, elNodes, faceNo)
                #nodeNoをvtkのnodeNoに置き換え
                newNodes = []
                for node in nodes:
                    newNo = nodeList[node][-1]
                    newNodes.append(newNo)
                #vtkNodeNoを保存
                grpData.append(newNodes)
            #grpNameと共にnodeNoを保存
            grp = [grpName, grpData]
            vsuGrps.append(grp)
    return [vndGrps, vsuGrps]

#
#  createVtkFileMesh
#---------------------
def createVtkFileMesh(vnList, veList, vndGrps, vsuGrps):
    """ vtkファイルのmeshデータを作成"""
    #vtk行を作成
    newLines = createVtkHeader()
    newLines += createVtkPieceHeader(vnList, veList, vsuGrps)
    newLines += createVtkNodes(vnList)
    newLines += createVtkCells(veList, vsuGrps)
    newLines += createVtkNodesData(vnList, vndGrps)
    newLines += createVtkCellsData(veList, vsuGrps)
    newLines += createVtkPieceFooter()
    newLines += createVtkFooter()
    return newLines

#
#  deleteDummyNodes
#----------------------
def deleteDummyNodes(vnList, veList, vndGrps, dummyNodes):
    """ dummyNodeGroupを削除する。"""
    #dummyNodesを削除する
    for i in range(len(dummyNodes)):
        (dummy, _node) = dummyNodes[i]
        vnList[dummy] = []
    #newNoを設定
    newNodeList = []
    n = 0
    for i in range(len(vnList)):
        if len(vnList[i]) > 0:
            newNodeList.append(vnList[i])
            vnList[i].append(n)
            n += 1
    #veListのnodeNoを更新
    for i in range(len(veList)):
        for ii in range(len(veList[i][0][1:])):
            nodeNo = veList[i][0][ii+1]
            newNo = vnList[nodeNo][1]
            veList[i][0][ii+1] = newNo
    #vndGrpsのnodeNoを更新、dummyのGroupを削除
    newGrps = []
    for i in range(len(vndGrps)):
        grpName = vndGrps[i][0]
        if grpName[:len("dummy")] != "dummy":
            for ii in range(len(vndGrps[i][1])):
                nodeNo = vndGrps[i][1][ii]
                newNo = vnList[nodeNo][1]
                vndGrps[i][1][ii] = newNo
            newGrps.append(vndGrps[i])
    return (newNodeList, veList, newGrps)

#
#  meshConvertToVtu
#-------------------
def meshConvertToVtu(meshFile):
    """ fistr形式のmshファイルををvtu形式に変換する。"""
    meshFileName = getMeshFileName(meshFile)
    if meshFile == "":
        return
    lines = readFileLines(meshFileName)
    #mesh内容を読込
    meshHeaderNumConts = cm.getFistrHeaderNumData(lines)
    #メッシュ内容を取得
    [nodeList, elementList] = cm.getFistrMeshConts(meshHeaderNumConts)
    print (_("meshファイルを変換中..."))
    #node番号の振り直し（未使用nodeを確認）vtk用のvnList, veListを取得
    [nodeList, vnList] = getNewNodeNo(nodeList, meshHeaderNumConts)
    [elementList, veList] = renumberNodeElement(nodeList, elementList, meshHeaderNumConts)
    #veListをsolid、シェル、ビーム要素順に並べ替える
    veList = renumberVeList(veList)
    #dummyNodeを取得(dummyと相手nodeNoを取得）
    dummyNodes = getDummyNodes(elementList, nodeList, meshHeaderNumConts)
    #groupを取得
    [vndGrps, vsuGrps] = renumberGrps(meshHeaderNumConts, nodeList, elementList)
    #dummyのnodeを削除
    (vnList, veList, vndGrps) = deleteDummyNodes(vnList, veList, vndGrps, dummyNodes)
    #point, cell定義行を作成
    defineLines = createVtkFileMesh(vnList, veList, vndGrps, vsuGrps)
    #fileName = "conv" + meshFileName + ".vtu"
    fileName = meshFileName + ".vtu"
    writeFileLines(fileName, defineLines)
    print(_("  変換しました"))
    return

#
#  printMsg
#-----------
def printMsg():
    """ helpを出力する。"""
    msg = """

<fistrMesh2vtk.py>---------------------------------------------------------------
FrontISTRフォーマットのmeshファイルをvtkフォーマットに変換する。

<使い方>
fistr2vtk.py <meshFileのheader>

例:
fistr2vtk.py FistrModel
"""
    print (msg)


if __name__ == "__main__":
    import gettext
    gettext.install("easyistr", os.getenv("LOCALE_DIR"))

    #log出力の設定
    #import logFileCreater
    #logFileCreater.log()

    if len(sys.argv) > 2:
        print(_("error: meshファイルが入力されていません。"))
        printMsg()
        exit()
    meshFile = sys.argv[1]
    if meshFile == "-h" or meshFile == "--help":
        printMsg()
        exit()
    meshConvertToVtu(meshFile)
