#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#   fistr2vtk.py
#
#       fistrからvtkにファイル変換する
#
#       fistr   node    vtk
#       -----------------------------------------
#       341     4       10      四面体1次
#       342     10      24      四面体2次
#       361     8       12      六面体1次
#       362     20      25      六面体2次
#       351     6       13      五面体1次
#       352     15      26      五面体2次
#       611     2       3       beam要素
#       641     4       3       beam要素（3自由度）
#       731     3       5       シェル三角形1次
#       732     6       22      シェル三角形2次
#       741     4       9       シェル四角形1次
#       761     6       5       シェル三角形1次（3自由度）
#       781     8       9       シェル四角形1次（3自由度）
#
#
#   15/05/19    新規作成
#      05/21    faceGroupの値を「1」→「0」からに変更
#               surfaceVtkフォルダ内に表面要素のvtkを作成（表面よvolume要素は別ファイル）
#      05/23    表面要素とvolume要素を１ヶのfileに設定
#               elementGroup 0     :表面要素
#                            1以上 :volume要素のelementGroupNo
#               faceGroup    0     :volume要素
#                            1以上 :表面要素のfaceGroupNo
#      06/08    結果fileにpointデータのみの場合があるので、これに対応
#      09/02    "\n"→os.linesepに修正
#      10/18    print u""形式に全て修正。
#               ファイル読み込み時、"\n"行は、読み飛ばす
#      12/25    行末コードlsを修正
#   16/03/27    シェル要素（731、741）追加
#               createVtkFileResult:DISPLACEMENT内に回転が含まれている場合は、削除
#      04/08    シェル要素（761、781）追加
#      06/11    梁要素（641）追加
#      06/27    checkNanData:nodeDataが「NAN」の場合、「0.0」に修正する事を追加
#      11/08    getNodesAllArrayData:!NODE内のコメント行をskipするように修正
#   17/09/27    高速化の為、ファイルの書き出し方法を修正
#      10/08    getNodeNo:バグ修正（未使用のnodeNoも取得していた。）
#   18/03/28    高速化の為、メッシュデータの取得方法を修正
#      04/01    createVtkNodeNoFields:未使用nodeの削除追加
#      04/05    createVtkCellGrpFields:バグ修正。（要素Grpが正しく表示されない）
#      04/06    getDummyNodes:追加。（dummyNodeを相手nodeの値に置き換え、dummyを隠す。）
#      11/03    getElementalResults:resファイル内の要素Noの並びが、mshファイルの要素並び
#               と異なっているので、要素Noを基準に読み取る様に修正。
#               （要素タイプが異なる要素で構成した場合、並びが異なってくる）
#      12/02    getElementalResults:dummyの要素が有る場合、エラー発生するので修正。
#      12/14    renumberNodeElement:grpName毎にgrpNoを設定
#               同一要素grp名内に異なる要素typeが存在する場合に対応
#   19/01/06    python3用に書き換え
#      03/04    renumberNodeElement:要素GrpNoの設定方法修正。（出現順）
#      03/07    main:旧の結果ファイルが残っていた場合、変換エラーが発生するので、
#               エラー処理を追加
#      05/03    vsCodeによる修正
#   22/03/15    国際化
#      04/12    log表示修正（import logFileCreaterを追加）
#      04/26    log表示を削除
#      10/26    getResulsDataForVersion:追加
#               resFileのversionに応じて読み込みを修正（fistr5以降に対応）
#   23/01/10    並列処理を追加。
#   24/06/05    createVtkFileResult:ver5用に修正
#      07/07    openにencoding="utf-8"を追加
#      10/18    convertFile:print内容を修正。複数のprocsが同時にprintする
#               ので、\nの出力が遅れる事がある為、print内容を\nを含める。
#

import os
import sys
import glob
import multiprocessing
import convertMesh as cm

import gettext
localeDir = os.getenv("LOCALE_DIR")
gettext.install("easyistr", localeDir)

ls = "\n"


#以下は、bufferサイズを「0」にする設定
#  subprocess等で起動した時、print内容をbufferに溜め込んでrealTimeに出力されない
#  ので、以下の設定でbufferサイズを「0」にする。
#  又は、起動時に「python3 -u」オプション付きで起動する。
#sys.stdout = os.fdopen(sys.stdout.fileno(), 'w', buffering=1)
#sys.stderr = os.fdopen(sys.stderr.fileno(), 'w', buffering=1)
#sys.stdin = os.fdopen(sys.stdin.fileno(), 'r', buffering=1)

#
#  createVtkHeader
#
def createVtkHeader():
    lines  = ["# vtk DataFile Version 2.0" + ls]
    lines += ["Create by fistr2vtk.py" + ls]
    lines += ["ASCII" + ls]
    lines += ["DATASET UNSTRUCTURED_GRID" + ls]
    return lines

#
#  createVtkNodesData
#    vtkのnodeデータを作成
def createVtkNodesData(vnList):
    newLines = ["POINTS " + str(len(vnList)) + " float" + ls]
    for node in vnList:
        loc = node[0]
        line = " ".join([str(loc[0]), str(loc[1]), str(loc[2])]) + ls
        newLines += [line]        
    return newLines

#
#  createVtkCellsData
#    vtkファイルにcellデータを追加
def createVtkCellsData(veList, vsuGrps):
    #node順入れ替え用の辞書
    changeOrderDict = {
     #node数 新order  入れ替え関数名
        3:[[0,1,2],   cm.order_3z],
        4:[[0,1,2,3], cm.order_4z],
        6:[[0,2,4,1,3,5], cm.order_6z],
        8:[[0,2,4,6,1,3,5,7], cm.order_8z]
        }
    cellsNum = 0
    allNum = 0
    newLines = []
    #要素データを作成
    for cont in veList:
        cell = cont[0][1:]
        words = list(map(str, cell))    #cellをstrのlistに変換
        # words = []
        # for nodeNo in cell:
        #     words.append(str(nodeNo))
        line = str(len(words)) + " " + " ".join(words) + ls
        newLines += [line]
        cellsNum += 1
        allNum += len(cell) + 1
    #表面要素を作成
    for grpData in vsuGrps:
        suNodes = grpData[1]
        for nodes in suNodes:
            #2次要素の場合、node順を入れ替え
            #  左回り順　→　2次要素順（頂点→中間点）
            [newOrder, func] = changeOrderDict[len(nodes)]
            cell = func(nodes, newOrder)
            #行を作成
            words = list(map(str, cell))    #cellをstrのlistに変換
            # words = []
            # for nodeNo in cell:
            #     words.append(str(nodeNo))
            line = str(len(words)) + " " + " ".join(words) + ls
            newLines += [line]
            cellsNum += 1
            allNum += len(cell) + 1
    line = "CELLS " + str(cellsNum) + " " + str(allNum) + ls
    newLines = [line] + newLines
    return newLines

#
#  createVtkCellType
#    vtkファイルにcellTypeを追加
def createVtkCellType(veList, vsuGrps):
    #node数からface要素名を取得する辞書
    #  面要素のnode数とelType名
    faceNumToFistrVtkTypeDict = {
        #node数: fistr名、vtk名
        3:["231", "5"],
        4:["241", "9"],
        6:["232", "22"],
        8:["242", "23"]
        }
    #header行を作成
    allNum = len(veList)
    for grp in vsuGrps:
        allNum += len(grp[1])
    lines = ["CELL_TYPES " + str(allNum) + ls]
    #要素行を作成
    for elCont in veList:
        cellName = elCont[-1]
        lines += [cellName + ls]
    #表面要素行を作成
    for grpData in vsuGrps:
        for grp in grpData[1]:
            #要素数に応じてcellNameを取得
            cellName = faceNumToFistrVtkTypeDict[len(grp)][1]
            #行を作成
            lines += [cellName + ls]
    return lines

#  checkNanData
#    NANを0.0に修正する
def checkNanData(vals):
    newVal = []
    for val in vals:
        if val.find("NAN") >= 0:
            newVal.append("0.0")
        else:
            newVal.append(val)
    return newVal

#
#  getNodalResults
#    nodeのdataを取得
def getNodalResults(lines, i, nNode):
    #各変数のcomponent数を取得
    comps = lines[i].split()
    nVal = len(comps)
    nFloat = 0
    for comp in comps:
        nFloat += int(comp)
    #各変数名を取得
    i += 1
    ip = i
    valNames = []
    while i < ip+nVal:
        line = lines[i]
        valNames.append(line.split()[0])
        i += 1
    ip = i
    #各変数分の配列を準備
    valLines = []
    j = 0
    for valName in valNames:
        line = [valName + " " + comps[j] + " " + str(nNode) + " float" + ls]
        valLines.append(line)
        j += 1
    #各変数の値を取得
    m = 0
    while i < len(lines) and m < nNode:
        [nums, i] = get1data(lines, i, nFloat)
        j = 0
        n = 0
        for comp in comps:
            vals = nums[n:n+int(comp)]
            #--------NANをチェック-------
            vals = checkNanData(vals)
            #---------------------------
            line = [" ".join(vals) + ls]
            valLines[j] += line
            j += 1
            n += int(comp)
        m += 1
    return [valLines, i]

#
#  getElementalResults
#    elementのdataを取得
def getElementalResults(lines, i, nCell, nElms, elementList):
    #各変数のcomponent数を取得
    comps = lines[i].split()
    nVal = len(comps)
    nFloat = 0
    for comp in comps:
        nFloat += int(comp)
    #各変数名を取得
    i += 1
    ip = i
    valNames = []
    while i < ip+nVal:
        line = lines[i]
        valNames.append(lines[i].split()[0])
        i += 1
    #各変数分の配列を準備
    ip = i
    valLines = []
    j = 0
    for valName in valNames:
        line = [valName + " " + comps[j] + " " + str(nCell) + " float" + ls]
        valLines.append(line)
        j += 1
    vtkElmsVal = [ [] for ii in range(nElms) ]
    #各変数の値を取得（resファイルから要素Noを読み取り、vtk要素Noに変換する）
    while i < len(lines):
        elmNo = int(lines[i].split()[0])
        if elmNo < len(elementList):
            #実要素の場合値を取得
            vtkElmNo = elementList[elmNo][-1]
            [nums, i] = get1data(lines, i, nFloat)
            vtkElmsVal[vtkElmNo] = nums
        else:
            #skipする
            [nums, i] = get1data(lines, i, nFloat)
    #各変数の値をvtk要素Noに保存
    m = 0
    for nums in vtkElmsVal:
        j = 0
        n = 0
        for comp in comps:
            vals = nums[n:n+int(comp)]
            line = [" ".join(vals) + ls]
            valLines[j] += line
            j += 1
            n += int(comp)
        m += 1
    return [valLines, i]

#  get1data
#    1行分のデータを取得
def get1data(lines, i, nFloat):
    nums = []
    while len(nums) < nFloat+1 and i < len(lines):
        nums += lines[i].split()
        i += 1
    return [nums[1:], i]

#  getValName
#    valName(TYPEなど）を取得
def getValName(line, valName):
    words = line.split(",")
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        word = deleteSp(word)
        if word.find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    newName = ""
    for chara in name:
        if chara > " ":
            newName += chara
    return newName

#  deleteSp
#    空白とCRを削除して返す
def deleteSp(line):
    newLine = ""
    for chara in line:
        if chara > " ":
            newLine += chara
    return newLine

#
#  remakeLines
#    761, 781要素のdummy節点削除
def remakeLines(lines):
    for i in range(len(lines)):
        line = lines[i]
        if line[0] == "!":
            if line[:len("!ELEMENT")] == "!ELEMENT":
                elType = getValName(line, "TYPE")
                if elType == "761":
                    flag = 1
                elif elType == "781":
                    flag = 2
            else:
                flag = 0
        else:
            if flag == 1:
                words = deleteSp(line).split(",")
                newLine = ", ".join(words[:4]) + "\n"
                lines[i] = newLine
            elif flag == 2:
                words = deleteSp(line).split(",")
                newLine = ", ".join(words[:5]) + "\n"
                lines[i] = newLine
    return lines

#
#  getNewNodeNo
#    未使用のnodeNoを調べ、新しいnodeNoを設定（「0」始まり）
#    nodeNoは、出現順に設定する（結果dataが出現順の為）
def getNewNodeNo(nodeList, meshHeaderNumConts):
    vnList = []
    newNo = 0
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!NODE":
            nodeData = headerCont[1]
            for nodeCont in nodeData:
                nodeNo = nodeCont[0]
                if len(nodeList[nodeNo][1]) > 0:    #使用有無チェック
                    loc = nodeCont[1:]
                    nodeList[nodeNo].append(newNo)
                    vnList.append([loc])
                    newNo += 1
    return [nodeList, vnList]

#
#  renumberNodeElement
#    nodeNoとelNoを書き換え
#    elNoは、出現順に設定する。（結果dataが出現順のため）
def renumberNodeElement(nodeList, elementList, meshHeaderNumConts):
    #grpNameのgrpNoを設定
    grpNos = {}             #grpNameとgrpNoの辞書
    grpNameNo = 1
    for headerCont in meshHeaderNumConts:
        words = deleteSp(headerCont[0]).split(",")
        if words[0] == "!ELEMENT":
            grpName = getValName(headerCont[0], "EGRP")
            if not grpName in grpNos.keys():
                grpNos[grpName] = grpNameNo
                grpNameNo += 1
    """
    grpNames = list(grpNos.keys())
    grpNames.sort()
    grpNameNo = 1
    for grpName in grpNames:
        grpNos[grpName] = grpNameNo
        grpNameNo += 1
    """

    #elNoとgrpNoを設定
    veList = []
    newNo = 0
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            grpName = getValName(header, "EGRP")
            elData = headerCont[1]
            for elCont in elData:
                elNo = elCont[0]
                elNodes = elCont[1:]
                newNodes = []
                for elNode in elNodes:
                    newNodeNo = nodeList[elNode][-1]
                    newNodes.append(newNodeNo)
                elType = elementList[elNo][1]
                #grpNo = elementList[elNo][0][0]
                grpNo = grpNos[grpName]             #辞書からgrpNoを取得
                [vtkName, newNodes] = cm.fistr2vtk_el(elType, newNodes)
                newElCont = [[grpNo] + newNodes] + [vtkName]
                veList.append(newElCont)
                elementList[elNo] += [newNo]
                newNo += 1
    return [elementList, veList]

#
#  renumberGrps
#    node, surfaceのgroupを取得
def renumberGrps(meshHeaderNumConts, nodeList, elementList):
    vndGrps = []
    vsuGrps = []
    for headerData in meshHeaderNumConts:
        header = headerData[0]
        words = deleteSp(header).split(",")
        if words[0] == "!NGROUP":
            #nodeGrpを取得
            grpName = getValName(header, "NGRP")
            nodes = headerData[1]
            newNodes = []
            for node in nodes:
                newNo = nodeList[node][-1]
                newNodes.append(newNo)
            grp = [grpName, newNodes]
            vndGrps.append(grp)
        elif words[0] == "!SGROUP":
            #surfaceGrpを取得
            grpName = getValName(header, "SGRP")
            suData = headerData[1]
            grpData = []
            for i in range(0, len(suData), 2):
                elNo = suData[i]
                faceNo = suData[i+1]
                newNo = elementList[elNo][-1]
                elType = elementList[elNo][1]
                elNodes = elementList[elNo][0][1:]
                #faceNodeを左回りで連続して取得
                nodes = cm.fistr_elFaceNode(elType, elNodes, faceNo)
                #nodeNoをvtkのnodeNoに置き換え
                newNodes = []
                for node in nodes:
                    newNo = nodeList[node][-1]
                    newNodes.append(newNo)
                #vtkNodeNoを保存
                grpData.append(newNodes)
            #grpNameと共にnodeNoを保存
            grp = [grpName, grpData]
            vsuGrps.append(grp)
    return [vndGrps, vsuGrps]

#
#  createVtkFileMesh
#    vtkファイルのmeshデータを作成
def createVtkFileMesh(vnList, veList, vsuGrps):
    #vtk行を作成
    newLines = createVtkHeader()
    newLines += createVtkNodesData(vnList)
    newLines += createVtkCellsData(veList, vsuGrps)
    newLines += createVtkCellType(veList, vsuGrps)
    return newLines

#
#  checkDisplacement
#    変位の他に回転が含まれている場合、回転を削除する
def checkDisplacement(lines):
    flag = 0
    for i in range(len(lines)):
        line = lines[i]
        if line[:len("DISPLACEMENT")] == "DISPLACEMENT":
            words = line.split()
            if words[1] == "3":
                flag = 1
                break
            else:
                flag = 2
                n = int(words[1])
                #nData = int(words[2])
                si = i
                break
    if flag != 2:
        return lines

    header = " ".join([words[0], "3", words[2], words[3]]) + "\n"
    i += 1
    while i < len(lines):
        line = lines[i]
        if ("0" <= line[0] and line[0] <= "9") or line[0] == "." or line[0] == "-" or line[0] == "+":
            i += 1
        else:
            break
    cont = "".join(lines[si+1:i])
    nums = cont.split()
    frontLines = lines[:si]
    afterLines = lines[i:]
    addLines = [header]
    i = 0
    while i < len(nums):
        line = " ".join(nums[i:i+3]) + "\n"
        addLines.append(line)
        i += n
    newLines = frontLines + addLines + afterLines
    return newLines

#
#  getResulsDataForVersion
def getResulsDataForVersion(lines):
    """ FrintISTRのversionに応じてdataLineを返す"""
    words = lines[0].split()
    if len(words) > 1:
        if words[0] == "*fstrresult" and words[1] == "2.0":
            #*data行を探す
            n = 0
            for i in range(len(lines)):
                line = lines[i]
                if line[:len("*data")] == "*data":
                    n = i
                    break
            lines = lines[n:]
    return lines

#
#  createVtkFileResult
#    vtkファイルの結果データを作成
def createVtkFileResult(fileName, nNodes, nCells, nElms, dummyNodes, elementList):
    nodeFields = []
    cellFields = []
    f=open(fileName); lines=f.readlines(); f.close()
    lines = getResulsDataForVersion(lines)
    #lines内の*data行を検索
    si = 3
    for i in range(20):
        if lines[i][:len("*data")] == "*data":
            si = i + 3
    #nodeFieldを作成（si行以降から）
    [nodeFields, i] = getNodalResults(lines, si, nNodes)
    if i < len(lines):
        #cellFieldを作成
        [cellFields, i] = getElementalResults(lines, i, nCells, nElms, elementList)
    #DISPLACEMENTをチェック修正
    for i in range(len(nodeFields)):
        line = nodeFields[i][0]
        if line.find("DISPLACEMENT") >= 0:
            disLines = nodeFields[i]
            newDisLines = checkDisplacement(disLines)
            nodeFields[i] = newDisLines
            break
    #dummyNodesの処理
    if len(dummyNodes) > 0:
        for i in range(len(nodeFields)):
            for [dummy, node] in dummyNodes:
                nodeFields[i][dummy+1] = nodeFields[i][node+1]
    return [nodeFields, cellFields]

#
#  addFaceValueToCellFields
#    cellFieldにfaceの値を追加する
def addFaceValueToCellFields(elementList, meshHeaderNumConts, cellFields):
    #faceが属する全てのelmNoを取得
    faceElmNos = []
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!SGROUP":
            faceData = headerCont[1]
            for i in range(0, len(faceData), 2):
                elmNo = faceData[i]
                newElmNo = elementList[elmNo][-1]
                faceElmNos.append(newElmNo)
    #cellFieldに値を追加（最初の2ヶは、elGrp,faceGrpの為、値を追加しない）
    for i in range(2, len(cellFields)):
        addLines = []
        for elmNo in faceElmNos:
            faceValueLine = cellFields[i][elmNo+1]
            addLines.append(faceValueLine)
        cellFields[i] += addLines
    return cellFields

#
#  createVtkNodeNoFields
#    nodeNoのfieldを作成
def createVtkNodeNoFields(nodeList, dummyNodes):
    nodeNos = []
    for i in range(len(nodeList)):
        if len(nodeList[i]) > 0:            #未定義？
            if len(nodeList[i][1]) > 0:     #未使用？
                nodeNos.append(i)
    nodeLines = ["nodeNo 1 " + str(len(nodeNos)) + " int" + ls]
    for node in nodeNos:
        nodeLines += [str(node) + ls]
    #dummyNodeの処理
    if len(dummyNodes) > 0:
        for [dummy, node] in dummyNodes:
            nodeLines[dummy+1] = nodeLines[node+1]
    return [nodeLines]    

#
#  createVtkNodeGrpFields
#    nodeGrpのfieldを作成
def createVtkNodeGrpFields(vnList, vndGrps, meshHeaderNumConts, dummyNodes):
    nNodes = len(vnList)
    #nodeGrpを取得
    nGrps = [ 0 for i1 in range(len(vnList)) ]
    grpNo = 1
    for grpData in vndGrps:
        nodes = grpData[1]
        for node in nodes:
            #要素に属さないnodeを削除しているので、判定している。
            #存在しないnodeNoの場合、[]のlistが入っている。
            if type(node) == int:
                nGrps[node] = grpNo
        grpNo += 1
    nodeLines = ["nodeGroup 1 " + str(nNodes) + " int" + ls]
    for grp in nGrps:
        nodeLines += [str(grp) + ls]
    #dummyNodeの処理
    if len(dummyNodes) > 0:
        for [dummy, node] in dummyNodes:
            nodeLines[dummy+1] = nodeLines[node+1]
    return [nodeLines]    

#
#  createVtkCellNoFields
#    cellNoのfieldを作成
def createVtkCellNoFields(elementList, meshHeaderNumConts):
    #element側
    elNos = []
    for i in range(len(elementList)):
        if len(elementList[i]) > 0:
            elNos.append(i)
            #grpNo = elementList[i][0][0]
    #表面要素を追加
    #grpNo = 0
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!SGROUP":
            suData = headerCont[1]
            for i in range(0, len(suData), 2):
                elNo = suData[i]
                elNos.append(elNo)
    #cellField行を作成
    cellLines = ["elementNo 1 " + str(len(elNos)) + " int" + ls]
    for el in elNos:
        cellLines += [str(el) + ls]
    return [cellLines]

#
#  createVtkCellGrpFields
#    cellGroupのfieldを作成
def createVtkCellGrpFields(elementList, meshHeaderNumConts, veList):
    #要素Grpを取得
    elGrps = []
    for i in range(len(veList)):
        grpNo = veList[i][0][0]
        elGrps.append(grpNo)
    #表面要素を追加
    grpNo = 0
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!SGROUP":
            suData = headerCont[1]
            for i in range(0, len(suData), 2):
                elNo = suData[i]
                grpNo = elementList[elNo][0][0]
                elGrps.append(grpNo)
    cellLines = ["elementGroup 1 " + str(len(elGrps)) + " int" + ls]
    for grp in elGrps:
        cellLines += [str(grp) + ls]
    return [cellLines]    

#
#  sortStepFileName
#
def sortStepFileName(names):
    steps = []
    for name in names:
        step = int(name.split(".")[-1])
        steps.append(step)
    steps.sort()
    nameHeader = ".".join(names[0].split(".")[:-1])
    names = []
    for step in steps:
        fileName = nameHeader + "." + str(step)
        names.append(fileName)
    return names

#
#  createVtkFaceFields
#    face行を作成
def createVtkFaceFields(elementList, meshHeaderNumConts):
    cellLines = []
    faceLines = []
    faceGrpNo = 1
    for headerCont in meshHeaderNumConts:
        header = headerCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            #要素Dataを作成（全て「0」を設定）
            elData = headerCont[1]
            line = "0" + ls
            cellLines += [ line for i in range(len(elData)) ]
            #cellLines += [line] * len(elData)
            #for elm in elData:
            #    cellLines += ["0" + ls]
        elif words[0] == "!SGROUP":
            #faceGroupNoを設定
            faceData = headerCont[1]
            line = str(faceGrpNo) + ls
            faceLines += [ line for i in range(len(faceData)//2) ]
            #faceLines += [line] * (len(faceData)//2)
            #for i in range(0, len(faceData), 2):
            #    faceLines += [str(faceGrpNo) + ls]
            faceGrpNo += 1
    lines = cellLines + faceLines
    header = ["faceGroup 1 " + str(len(lines)) + " int" + ls]
    lines = header + lines
    return lines

#
#  createNodeLines
#    nodeField行を作成
def createNodeLines(nodeFields):
    nNodes = len(nodeFields[0]) - 1
    newLines  = ["POINT_DATA " + str(nNodes) + ls]
    newLines += ["FIELD attributes " + str(len(nodeFields)) + ls]
    for lines in nodeFields:
        newLines += lines
    return newLines

#
#  createCellLines
#    cellField行を作成
def createCellLines(cellFields):
    nCell = len(cellFields[0]) - 1
    newLines =  ["CELL_DATA " + str(nCell) + ls]
    newLines += ["FIELD attribites " + str(len(cellFields)) + ls]
    for lines in cellFields:
        newLines += lines
    return newLines

#
#  getDummyNodes(elementList)
#    dummy節点とその相手nodeNoを取得
def getDummyNodes(elementList, nodeList, meshHeaderNumConts):
    dummyNodes = []
    for meshCont in meshHeaderNumConts:
        header = meshCont[0]
        words = deleteSp(header).split(",")
        if words[0] == "!ELEMENT":
            elType = getValName(header, "TYPE")
            #dummyNodeを取得
            if elType == "761":
                #三角形
                elmData = meshCont[1]
                for elm in elmData:
                    nodes = elm[1:]
                    dummyNodes.append([nodeList[nodes[3]][2], nodeList[nodes[0]][2]])
                    dummyNodes.append([nodeList[nodes[4]][2], nodeList[nodes[1]][2]])
                    dummyNodes.append([nodeList[nodes[5]][2], nodeList[nodes[2]][2]])
            elif elType == "781":
                #四角形
                elmData = meshCont[1]
                for elm in elmData:
                    nodes = elm[1:]
                    dummyNodes.append([nodeList[nodes[4]][2], nodeList[nodes[0]][2]])
                    dummyNodes.append([nodeList[nodes[5]][2], nodeList[nodes[1]][2]])
                    dummyNodes.append([nodeList[nodes[6]][2], nodeList[nodes[2]][2]])
                    dummyNodes.append([nodeList[nodes[7]][2], nodeList[nodes[3]][2]])
            elif elType == "641":
                #beam
                elmData = meshCont[1]
                for elm in elmData:
                    nodes = elm[1:]
                    dummyNodes.append([nodeList[nodes[2]][2], nodeList[nodes[0]][2]])
                    dummyNodes.append([nodeList[nodes[3]][2], nodeList[nodes[1]][2]])
    return dummyNodes

#--------------------
#  multiFilesConvert
#--------------------
def multiFilesConvert():
    """ 複数のfileを並列変換する"""
    meshFileName = meshFile + ".msh"
    print (_(u"meshデータを取得中..."))
    f=open(meshFileName)
    lines = []
    for line in f.readlines():
        if line != "\n":
            lines.append(line)
    f.close()
    #mesh内容を読み込み
    meshHeaderNumConts = cm.getFistrHeaderNumData(lines)
    #メッシュ内容を取得
    [nodeList, elementList] = cm.getFistrMeshConts(meshHeaderNumConts)
    print (_(u"meshファイルを変換中..."))
    #node番号の振り直し（未使用nodeを確認）vtk用のvnList, veListを取得
    [nodeList, vnList] = getNewNodeNo(nodeList, meshHeaderNumConts)
    [elementList, veList] = renumberNodeElement(nodeList, elementList, meshHeaderNumConts)
    #dummyNodeを取得(dummyと相手nodeNoを取得）
    dummyNodes = getDummyNodes(elementList, nodeList, meshHeaderNumConts)
    #groupを取得
    [vndGrps, vsuGrps] = renumberGrps(meshHeaderNumConts, nodeList, elementList)
    #point, cell定義行を作成
    defineLines = createVtkFileMesh(vnList, veList, vsuGrps)
    if resFile == "":
        #nodeNo,nodeGroupのfield行を作成
        nodeFields = createVtkNodeNoFields(nodeList, dummyNodes)
        nodeFields += createVtkNodeGrpFields(vnList, vndGrps, meshHeaderNumConts, dummyNodes)
        #要素No、要素groupのfield行を作成
        cellFields = createVtkCellNoFields(elementList, meshHeaderNumConts)
        cellFields += createVtkCellGrpFields(elementList, meshHeaderNumConts, veList)
        #faceGroup行を作成
        faceLines = createVtkFaceFields(elementList, meshHeaderNumConts)
        cellFields.append(faceLines)
        #各nodeFieldから全体のnode行を作成
        nodeLines = createNodeLines(nodeFields)
        #各cellFieldから全体のcell行を作成
        cellLines = createCellLines(cellFields)
        #全体のvtkを作成
        newLines = defineLines + nodeLines + cellLines
        #ファイル作成
        writeName = "conv" + meshFileName + ".vtk"
        f = open(writeName, "w", encoding="utf-8")
        for line in newLines:
            f.write(line)
        f.close()
        print (_(u"変換しました"))
        return

    fileName = resFile
    if paraRes == True:
        #並列処理の場合
        names = fileName + ".res.merge.*"
    else:
        #シングルコアの場合
        names = fileName + ".res.0.*"
    resFileNames = glob.glob(names)
    names = []
    for name in resFileNames:
        if name[-1] != "~" and name.split(".")[-1] != "inp":
            names.append(name)
    if len(names) > 0:
        names = sortStepFileName(names)
        #メッシュのfieldを作成
        nodeFields = createVtkNodeGrpFields(vnList, vndGrps, meshHeaderNumConts, dummyNodes)
        cellFields = createVtkCellGrpFields(elementList, meshHeaderNumConts, veList)
        #faceGroupNo行をcellFielに追
        faceLines = createVtkFaceFields(elementList, meshHeaderNumConts)
        cellFields.append(faceLines)
        nNodes = len(nodeFields[0][1:])     #node数
        nCells = len(cellFields[0][1:])     #cell数（face含む）
        nElms = len(veList)                 #cell数（要素のみ）
        nValues = [nNodes, nCells, nElms]
        sgrpHeaderNumConts = getSgrpNumConts(meshHeaderNumConts)
        #並列処理で変換
        np = int(nCpu)
        if np > len(names):
            np = len(names)
        print(_("全file数:") + str(len(names)) + "  " + _("並列数:") + str(np) + " --> " + _("vtk変換します。"))
        value = [nValues, nodeFields, cellFields, dummyNodes,
                 elementList, sgrpHeaderNumConts, defineLines]
        valueList = list(map(lambda x: [x] + value, names))
        p = multiprocessing.Pool(np)
        res = p.map(convertFile, valueList)
        print (_(u"変換しました"))
    else:
        print (_(u"結果fileがありません"))

#
#  getSgrpNumConts
#------------------
def getSgrpNumConts(meshHeaderNumConts):
    """ meshHeaderNumContsから!SGRPを取得して返す"""
    headerConts = []
    for headerData in meshHeaderNumConts:
        header = headerData[0]
        words = deleteSp(header).split(",")
        if words[0] == "!SGROUP":
            headerConts.append(headerData)
    return headerConts

#
#  convertFile
#--------------
def convertFile(valueList):
    """ 1個のfileを変換する。並列計算用で作成"""
    (fileName, nValues, nodeFields, cellFields, dummyNodes,
     elementList, meshHeaderNumConts, defineLines) = valueList
    print (fileName + _(u" を変換中...") + "\n", end="", flush=True)
    [nNodes, nCells, nElms] = nValues
    #fieldの初期化（必要部分のみ取得）
    #nodeFields = nodeFields[:1]
    #cellFields = cellFields[:2]
    try:
        #計算結果のfieldを取得
        [nodeField, cellField] = createVtkFileResult(fileName, nNodes, nCells, nElms, dummyNodes, elementList)
    except:
        print ("  --> error: " + fileName + _(" 変換中にエラーが発生しました。"))
        return fileName
    nodeFields += nodeField
    cellFields += cellField
    #cellFieldにfaceValueを追加
    cellFields = addFaceValueToCellFields(elementList, meshHeaderNumConts, cellFields)
    #fieldから行を完成させる
    nodeLines = createNodeLines(nodeFields)
    cellLines = createCellLines(cellFields)
    writeLines = defineLines + nodeLines + cellLines
    #書き込み
    writeName = "conv" + fileName + ".vtk"
    f=open(writeName, "w")
    cont = "".join(writeLines)
    f.write(cont)
    f.close()
    return fileName


#---------------------
#  singleFileConvert
#---------------------
def singleFileConvert():
    """ resFileをsingleで変換する。"""
    #処理開始
    meshFileName = meshFile + ".msh"
    print (_(u"meshデータを取得中..."))
    f=open(meshFileName)
    lines = []
    for line in f.readlines():
        if line != "\n":
            lines.append(line)
    f.close()
    #mesh内容を読み込み
    meshHeaderNumConts = cm.getFistrHeaderNumData(lines)
    #メッシュ内容を取得
    [nodeList, elementList] = cm.getFistrMeshConts(meshHeaderNumConts)
    print (_(u"meshファイルを変換中..."))
    #node番号の振り直し（未使用nodeを確認）vtk用のvnList, veListを取得
    [nodeList, vnList] = getNewNodeNo(nodeList, meshHeaderNumConts)
    [elementList, veList] = renumberNodeElement(nodeList, elementList, meshHeaderNumConts)
    #dummyNodeを取得(dummyと相手nodeNoを取得）
    dummyNodes = getDummyNodes(elementList, nodeList, meshHeaderNumConts)
    #groupを取得
    [vndGrps, vsuGrps] = renumberGrps(meshHeaderNumConts, nodeList, elementList)
    #point, cell定義行を作成
    defineLines = createVtkFileMesh(vnList, veList, vsuGrps)
    if resFile == "":
        #nodeNo,nodeGroupのfield行を作成
        nodeFields = createVtkNodeNoFields(nodeList, dummyNodes)
        nodeFields += createVtkNodeGrpFields(vnList, vndGrps, meshHeaderNumConts, dummyNodes)
        #要素No、要素groupのfield行を作成
        cellFields = createVtkCellNoFields(elementList, meshHeaderNumConts)
        cellFields += createVtkCellGrpFields(elementList, meshHeaderNumConts, veList)
        #faceGroup行を作成
        faceLines = createVtkFaceFields(elementList, meshHeaderNumConts)
        cellFields.append(faceLines)
        #各nodeFieldから全体のnode行を作成
        nodeLines = createNodeLines(nodeFields)
        #各cellFieldから全体のcell行を作成
        cellLines = createCellLines(cellFields)
        #全体のvtkを作成
        newLines = defineLines + nodeLines + cellLines
        #ファイル作成
        writeName = "conv" + meshFileName + ".vtk"
        f = open(writeName, "w", encoding="utf-8")
        for line in newLines:
            f.write(line)
        f.close()
        print (_(u"変換しました"))
    else:
        fileName = resFile
        #if len(glob.glob(fileName + ".res.1.*")):
        if paraRes == True:
            #並列処理の場合
            names = fileName + ".res.merge.*"
        else:
            #シングルコアの場合
            names = fileName + ".res.0.*"
        resFileNames = glob.glob(names)
        names = []
        for name in resFileNames:
            if name[-1] != "~" and name.split(".")[-1] != "inp":
                names.append(name)
        if len(names) > 0:
            names = sortStepFileName(names)
            #メッシュのfieldを作成
            nodeFields = createVtkNodeGrpFields(vnList, vndGrps, meshHeaderNumConts, dummyNodes)
            cellFields = createVtkCellGrpFields(elementList, meshHeaderNumConts, veList)
            #faceGroupNo行をcellFielに追
            faceLines = createVtkFaceFields(elementList, meshHeaderNumConts)
            cellFields.append(faceLines)
            nNodes = len(nodeFields[0][1:])     #node数
            nCells = len(cellFields[0][1:])     #cell数（face含む）
            nElms = len(veList)                 #cell数（要素のみ）
            for fileName in names:
                print (fileName + _(u" を変換中..."))
                #fieldの初期化（必要部分のみ取得）
                nodeFields = nodeFields[:1]
                cellFields = cellFields[:2]
                try:
                    #計算結果のfieldを取得
                    [nodeField, cellField] = createVtkFileResult(fileName, nNodes, nCells, nElms, dummyNodes, elementList)
                except:
                    print ("  --> error: " + fileName + _(" 変換中にエラーが発生しました。"))
                    break
                nodeFields += nodeField
                cellFields += cellField
                #cellFieldにfaceValueを追加
                cellFields = addFaceValueToCellFields(elementList, meshHeaderNumConts, cellFields)
                #fieldから行を完成させる
                nodeLines = createNodeLines(nodeFields)
                cellLines = createCellLines(cellFields)
                writeLines = defineLines + nodeLines + cellLines
                #書き込み
                writeName = "conv" + fileName + ".vtk"
                f=open(writeName, "w")
                cont = "".join(writeLines)
                f.write(cont)
                f.close()
            print (_(u"変換しました"))
        else:
            print (_(u"結果fileがありません"))


#
#  printMsg
#
def printMsg():
    msg = """

<fistr2vtk.py>---------------------------------------------------------------
FrontISTRフォーマットをvtkフォーマットに変換する。
FrontISTR側のmshファイルとresファイルを読み込みファイル変換する。
「-res」を省略した場合は、meshファイルのみvtk変換する。

<使い方>
fistr2vtk.py -mesh <meshFileのheader> [-res <resFileのheader>] [-para yes/no]

<option>
 -mesh fileHeader   meshファイル名を設定
 -res fileHeader    結果ファイル名を設定
 -para yeas/no      並列計算の場合
 -np nProcs         並列数
 -h                 ヘルプ
 -help              ヘルプ

例:
fistr2vtk.py -mesh FistrModel -res FistrModel
fistr2vtk.py -mesh FistrModel -res FistrModel_eigen -para yes -np 4
"""
    print (msg)

#
#  getOption
#    optionを取得
def getOption():
    #comm = ""
    meshFile = ""
    resFile = ""
    para = ""
    nCpu = "1"
    i = 0
    while i < len(sys.argv):
        if sys.argv[i] == "-mesh":
            i += 1
            meshFile = sys.argv[i]
        elif sys.argv[i] == "-res":
            i += 1
            resFile = sys.argv[i]
        elif sys.argv[i] == "-para":
            i += 1
            para = sys.argv[i]
        elif sys.argv[i] == "-np":
            i += 1
            nCpu = sys.argv[i]
        i += 1
    return [meshFile, resFile, para, nCpu]


if __name__=='__main__':
    #log表示
    #import logFileCreater
    #logFileCreater.log()

    [meshFile, resFile, para, nCpu] = getOption()
    if meshFile == "":
        print (_(u"error:meshファイルが入力されていません。"))
        printMsg()
        exit()

    #parallel結果かどうかを判断
    if len(glob.glob(resFile + ".res.1.*")):
        paraRes = True
    else:
        paraRes = False
    #並列変換のチェック
    if para == "yes":
        #複数fileを並列変換
        multiFilesConvert()
    else:
        #fileを1個づつ変換
        singleFileConvert()
