#!/usr/bin/python3
# -*- coding: utf-8 -*-    
#
#   fineSeparateMesh.py
#
#       中間節点を追加し、要素を分割して、メッシュを細分化する。
#       fistr用のメッシュについて細分化する
#
#   18/09/07    新規作成
#      09/20    fistr形式メッシュのみに対応させる。
#               四面体1次要素に対応させる
#      09/27    六面体1次要素に対応させる。
#      09/30    四面体、六面体、五面体の1次2次要素に対応させる。
#   19/01/10    python3用に書き換え
#   24/07/07    openにencoding="utf-8"を追加
#


import convertMesh as cm
import geometryFunc as geo
import sys

ls = "\n"

#
#  helpMsg
#    helpを出力
def helpMsg():
    cont = """
----- fineSeparateMesh.py -------------------------------------------
各要素に中間節点を追加して、その節点を使って要素を増やす。
四面体（tetra）、六面体（hexa）、五面体（prism）の1次2次に対応。

<使い方>
fineSeparateMesh.py <file> [<numlevel>]
file:       fistr形式のメッシュファイル
numlevel:   中間節点を追加する回数。1回で要素が8倍に増える。
            調略した場合は、numlevel=1として処理。

<使用例>
fineSeparateMesh.py FistrModel.msh
fineSeparateMesh.py FistrModel.msh 2
"""
    print (cont)

#
#  readFileLines
#    fileの読み込み
def readFileLines(fileName):
    f = open(fileName, encoding="utf-8")
    lines = f.readlines()
    f.close()
    return lines

#
#  writeFileLines
#    fileの書き込み
def writeFileLines(fileName, lines):
    f = open(fileName, "w", encoding="utf-8")
    for line in lines:
        f.write(line)
    f.close()

#
#  getEachGroups
#    各group別にdataを取得
def getEachGroups(headerNumData):
    nodeConts = []
    elementConts = []
    sgrpConts = []
    ngrpConts = []
    otherConts = []
    for headerCont in headerNumData:
        header = headerCont[0]
        words = cm.deleteSp(header).split(",")
        if words[0].upper() == "!NODE":
            nodeConts.append(headerCont)
        elif words[0].upper() == "!ELEMENT":
            elementConts.append(headerCont)
        elif words[0].upper() == "!SGROUP":
            sgrpConts.append(headerCont)
        elif words[0].upper() == "!NGROUP":
            ngrpConts.append(headerCont)
        else:
            otherConts.append(headerCont)
    return [nodeConts, elementConts, sgrpConts, ngrpConts, otherConts]

#
#  getMiddlePoint
#    中間点の座標を取得
def getMiddlePoint(edge, nodeList):
    xyz1 = nodeList[edge[0]][0]
    xyz2 = nodeList[edge[1]][0]
    x = (xyz1[0] + xyz2[0]) / 2.0
    y = (xyz1[1] + xyz2[1]) / 2.0
    z = (xyz1[2] + xyz2[2]) / 2.0
    return [x,y,z]

#
#  getNodeGrpNoOfMiddPoint
#    中間節点のnodeGrpNoを取得
def getNodeGrpNoOfMiddPoint(key, nodeGrpNos):
    grpNo1 = nodeGrpNos[key[0]]
    grpNo2 = nodeGrpNos[key[1]]
    grpNos = []
    if len(grpNo1) == 0 or len(grpNo2) == 0:
        return grpNos
    for grp in grpNo1:
        if grp in grpNo2:
            grpNos.append(grp)
    return grpNos

#
#  getNodeGrpNoOfFgPoint(key, nodeGrpNos)
#    face中心のnodeGrpNoを取得
def getNodeGrpNoOfFgPoint(key, nodeGrpNos):
    grpNo1 = nodeGrpNos[key[0]]
    grpNo2 = nodeGrpNos[key[1]]
    grpNo3 = nodeGrpNos[key[2]]
    grpNo4 = nodeGrpNos[key[3]]
    grpNos = []
    if len(grpNo1) == 0 or len(grpNo2) == 0 or len(grpNo3) == 0 or len(grpNo4) == 0:
        return grpNos
    for grp in grpNo1:
        if (grp in grpNo2) and (grp in grpNo3) and (grp in grpNo4):
            grpNos.append(grp)
    return grpNos

#
#  getFaceNo
#    要素内から該当する面を取得する
def getFaceNo(elm, tris):
    faceNo = 0
    for triNodes in tris:
        flag = 0
        for triNode in triNodes:
            if not (triNode in elm[1:]):
                flag = 1
                break
        if flag == 0:
            faceNo = cm.fistr_elFace(triNodes, elm[1:])
            break
    return faceNo


#
#  createNodesElements341
#  ----------------------
#    四面体1次：node, elementを追加
def createNodesElements341(meshes, nodeNo, elmNo):
    [nodeList, elms, ngrpConts, nodeGrpNos, surGrpNos, edgeDict, faceDict] = meshes
    #surGrpNoの最大値を取得
    maxNo = -1
    for surGrpNo in surGrpNos:
        if len(surGrpNo) > 0:
            for [grpNo, faceNo] in surGrpNo:
                if grpNo > maxNo:
                    maxNo = grpNo
    #配列を準備
    newSurConts = [ [] for i in range(maxNo+1) ]
    newElms = []
    #要素毎に処理
    for i in range(len(elms)):
        elm = elms[i]
        #edgeを取得
        edges = []
        edge1 = [elm[1], elm[2]]; edge1.sort(); edges.append(edge1)
        edge2 = [elm[2], elm[3]]; edge2.sort(); edges.append(edge2)
        edge3 = [elm[3], elm[1]]; edge3.sort(); edges.append(edge3)
        edge4 = [elm[1], elm[4]]; edge4.sort(); edges.append(edge4)
        edge5 = [elm[2], elm[4]]; edge5.sort(); edges.append(edge5)
        edge6 = [elm[3], elm[4]]; edge6.sort(); edges.append(edge6)
        #中間節点を取得
        midNodes = []
        for edge in edges:
            key = (edge[0], edge[1])
            if not (key in edgeDict):
                #node追加
                newLoc = getMiddlePoint(key, nodeList)
                edgeDict[key] = nodeNo
                midNodes.append(nodeNo)
                nodeList.append([newLoc])                   #nodeListに保存
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfMiddPoint(key, nodeGrpNos)
                for grpNo in newNgrpNos:
                    ngrpConts[grpNo][1].append(nodeNo)      #nodeGrpに保存
                nodeNo += 1
            else:
                midNodes.append(edgeDict[key])
        #新elementを作成
        #       要素No       1               2           3           4
        elm1 = [elmNo] +   [midNodes[3], midNodes[4], midNodes[5], elm[4]]
        elm2 = [elmNo+1] + [midNodes[4], midNodes[0], midNodes[3], midNodes[2]]
        elm3 = [elmNo+2] + [midNodes[1], midNodes[4], midNodes[5], midNodes[2]]
        elm4 = [elmNo+3] + [midNodes[4], midNodes[3], midNodes[5], midNodes[2]]
        elm5 = [elmNo+4] + [midNodes[0], midNodes[4], midNodes[1], midNodes[2]]
        elm6 = [elmNo+5] + [midNodes[0], elm[1],      midNodes[3], midNodes[2]]
        elm7 = [elmNo+6] + [elm[2],      midNodes[4], midNodes[1], midNodes[0]]
        elm8 = [elmNo+7] + [elm[3],      midNodes[1], midNodes[5], midNodes[2]]
        elmNo += 8
        addElms = [elm1] + [elm2] + [elm3] + [elm4] + [elm5] + [elm6] + [elm7] + [elm8]
        newElms += addElms
        #新sgrpを作成
        eNo = elm[0]
        if len(surGrpNos[eNo]) != 0:
            #sgrpが設定されている場合
            for j in range(len(surGrpNos[eNo])):
                [grpNo, faceNo] = surGrpNos[eNo][j]
                surNodes = cm.fistr_elFaceNode("341", elm[1:], faceNo)
                edge1 = [surNodes[0], surNodes[1]]; edge1.sort(); key=(edge1[0], edge1[1])
                node1 = edgeDict[key]
                edge2 = [surNodes[1], surNodes[2]]; edge2.sort(); key=(edge2[0], edge2[1])
                node2 = edgeDict[key]
                edge3 = [surNodes[2], surNodes[0]]; edge3.sort(); key=(edge3[0], edge3[1])
                node3 = edgeDict[key]
                #新しい面
                tri1 = [surNodes[0], node1, node3]
                tri2 = [node1, node2, node3]
                tri3 = [node1, surNodes[1], node2]
                tri4 = [node2, surNodes[2], node3]
                tris = [tri1, tri2, tri3, tri4]
                #追加した要素毎に面Noを取得
                for addElm in addElms:
                    newFaceNo = getFaceNo(addElm, tris)
                    if newFaceNo != 0:
                        #該当するfaceが存在する場合
                        newElmNo = addElm[0]
                        newSurConts[grpNo].append([newElmNo, newFaceNo])
    meshes = [nodeList, newElms, ngrpConts, nodeGrpNos, newSurConts, edgeDict, faceDict]
    return [meshes, nodeNo, elmNo]

#
#  createNodesElement361
#  ---------------------
#    六面体1次：node、element作成
def createNodesElement361(meshes, nodeNo, elmNo):
    [nodeList, elms, ngrpConts, nodeGrpNos, surGrpNos, edgeDict, faceDict] = meshes
    #surGrpNoの最大値を取得
    maxNo = -1
    for surGrpNo in surGrpNos:
        if len(surGrpNo) > 0:
            for [grpNo, faceNo] in surGrpNo:
                if grpNo > maxNo:
                    maxNo = grpNo
    #配列を準備
    newSurConts = [ [] for i in range(maxNo+1) ]
    newElms = []
    #要素毎に処理
    for i in range(len(elms)):
        elm = elms[i]
        #edgeを取得
        edges = []
        edge1 = [elm[1], elm[2]]; edge1.sort(); edges.append(edge1)
        edge2 = [elm[2], elm[3]]; edge2.sort(); edges.append(edge2)
        edge3 = [elm[3], elm[4]]; edge3.sort(); edges.append(edge3)
        edge4 = [elm[4], elm[1]]; edge4.sort(); edges.append(edge4)
        edge5 = [elm[5], elm[6]]; edge5.sort(); edges.append(edge5)
        edge6 = [elm[6], elm[7]]; edge6.sort(); edges.append(edge6)
        edge7 = [elm[7], elm[8]]; edge7.sort(); edges.append(edge7)
        edge8 = [elm[8], elm[5]]; edge8.sort(); edges.append(edge8)
        edge9 = [elm[1], elm[5]]; edge9.sort(); edges.append(edge9)
        edge10 = [elm[2], elm[6]]; edge10.sort(); edges.append(edge10)
        edge11 = [elm[3], elm[7]]; edge11.sort(); edges.append(edge11)
        edge12 = [elm[4], elm[8]]; edge12.sort(); edges.append(edge12)
        #中間節点を取得
        midNodes = []
        #  edgeの中間
        for edge in edges:
            key = (edge[0], edge[1])
            if not (key in edgeDict):
                #node追加
                newLoc = getMiddlePoint(key, nodeList)
                edgeDict[key] = nodeNo
                midNodes.append(nodeNo)
                nodeList.append([newLoc])                   #nodeListに保存
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfMiddPoint(key, nodeGrpNos)
                for grpNo in newNgrpNos:
                    ngrpConts[grpNo][1].append(nodeNo)      #nodeGrpに保存
                nodeNo += 1
            else:
                midNodes.append(edgeDict[key])
        #  面の中心
        faces = []
        quad = [elm[1], elm[2], elm[3], elm[4]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[5], elm[6], elm[7], elm[8]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[1], elm[2], elm[6], elm[5]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[2], elm[3], elm[7], elm[6]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[3], elm[4], elm[8], elm[7]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[4], elm[1], elm[5], elm[8]]; k=sorted(quad); faces.append([k,quad])
        elmNgrpNos = []
        fgNodes = []
        for k, q in faces:
            key = (k[0], k[1], k[2], k[3])
            if not (key in faceDict):
                #node追加
                qLocs = [ nodeList[q[0]][0], nodeList[q[1]][0],
                          nodeList[q[2]][0], nodeList[q[3]][0] ]
                newLoc = geo.quadG(qLocs)
                faceDict[key] = nodeNo
                fgNodes.append(nodeNo)
                nodeList.append([newLoc])
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfFgPoint(key, nodeGrpNos)
                elmNgrpNos.append(newNgrpNos)
                for grpNo in newNgrpNos:
                    ngrpConts[grpNo][1].append(nodeNo)
                nodeNo += 1
            else:
                fgNodes.append(faceDict[key])
        #  6面体の中心
        node1 = edgeDict[(edges[8][0], edges[8][1])]        #edge1-5
        node2 = edgeDict[(edges[9][0], edges[9][1])]        #edge2-6
        node3 = edgeDict[(edges[10][0], edges[10][1])]      #edge3-7
        node4 = edgeDict[(edges[11][0], edges[11][1])]      #edge4-8
        quad = [ nodeList[node1][0], nodeList[node2][0],
                 nodeList[node3][0], nodeList[node4][0] ]
        newLoc = geo.quadG(quad)
        nodeList.append([newLoc])
        #    節点Grpを設定
        sameGrpNos = []
        for grpNo in elmNgrpNos[0]:
            flag = 0
            for grpNos in elmNgrpNos[1:]:
                if not (grpNo in grpNos):
                    flag = 1
                    break
            if flag == 0:
                sameGrpNos.append(grpNo)
        for grpNo in sameGrpNos:
            ngrpConts[grpNo][1].append(nodeNo)
        volNode = nodeNo
        nodeNo += 1
        #新elementを作成
        #       要素No       1               2           3           4
        elm1 = [elmNo]   + [elm[1],      midNodes[0], fgNodes[0],   midNodes[3],
                            midNodes[8], fgNodes[2],  volNode,      fgNodes[5]]
        elm2 = [elmNo+1] + [midNodes[0], elm[2],      midNodes[1],  fgNodes[0],
                            fgNodes[2],  midNodes[9], fgNodes[3],   volNode]
        elm3 = [elmNo+2] + [midNodes[3], fgNodes[0],  midNodes[2],  elm[4],
                            fgNodes[5],  volNode,     fgNodes[4],   midNodes[11]]
        elm4 = [elmNo+3] + [fgNodes[0],  midNodes[1], elm[3],       midNodes[2],
                            volNode,     fgNodes[3],  midNodes[10], fgNodes[4]]
        elm5 = [elmNo+4] + [midNodes[8], fgNodes[2],  volNode,      fgNodes[5],
                            elm[5],      midNodes[4], fgNodes[1],   midNodes[7]]
        elm6 = [elmNo+5] + [fgNodes[2],  midNodes[9], fgNodes[3],   volNode,
                            midNodes[4], elm[6],      midNodes[5],  fgNodes[1]]
        elm7 = [elmNo+6] + [fgNodes[5],  volNode,     fgNodes[4],   midNodes[11],
                            midNodes[7], fgNodes[1],  midNodes[6],  elm[8]]
        elm8 = [elmNo+7] + [volNode,     fgNodes[3],  midNodes[10], fgNodes[4],
                            fgNodes[1],  midNodes[5], elm[7],       midNodes[6]]
        elmNo += 8
        addElms = [elm1] + [elm2] + [elm3] + [elm4] + [elm5] + [elm6] + [elm7] + [elm8]
        newElms += addElms
        #新sgrpを作成
        eNo = elm[0]
        if len(surGrpNos[eNo]) != 0:
            #sgrpが設定されている場合
            for j in range(len(surGrpNos[eNo])):
                [grpNo, faceNo] = surGrpNos[eNo][j]
                surNodes = cm.fistr_elFaceNode("361", elm[1:], faceNo)
                edge1 = [surNodes[0], surNodes[1]]; edge1.sort(); key1=(edge1[0], edge1[1])
                edge2 = [surNodes[1], surNodes[2]]; edge2.sort(); key2=(edge2[0], edge2[1])
                edge3 = [surNodes[2], surNodes[3]]; edge3.sort(); key3=(edge3[0], edge3[1])
                edge4 = [surNodes[3], surNodes[0]]; edge4.sort(); key4=(edge4[0], edge4[1])
                face = sorted(surNodes); key5 = (face[0], face[1], face[2], face[3])
                node1 = edgeDict[key1]
                node2 = edgeDict[key2]
                node3 = edgeDict[key3]
                node4 = edgeDict[key4]
                nodeG = faceDict[key5]
                #新しい面
                quad1 = [surNodes[0], node1, nodeG, node4]
                quad2 = [node1, surNodes[1], node2, nodeG]
                quad3 = [node4, nodeG, node3, surNodes[3]]
                quad4 = [nodeG, node2, surNodes[2], node3]
                quads = [quad1, quad2, quad3, quad4]
                #追加した要素毎に面Noを取得
                for addElm in addElms:
                    newFaceNo = getFaceNo(addElm, quads)
                    if newFaceNo != 0:
                        #該当するfaceが存在する場合
                        newElmNo = addElm[0]
                        newSurConts[grpNo].append([newElmNo, newFaceNo])
    meshes = [nodeList, newElms, ngrpConts, nodeGrpNos, newSurConts, edgeDict, faceDict]
    return [meshes, nodeNo, elmNo]

#
#  createNodesElement351
#  ---------------------
#    五面体1次の細分化（node、element作成）
def createNodesElement351(meshes, nodeNo, elmNo):
    [nodeList, elms, ngrpConts, nodeGrpNos, surGrpNos, edgeDict, faceDict] = meshes
    #surGrpNoの最大値を取得
    maxNo = -1
    for surGrpNo in surGrpNos:
        if len(surGrpNo) > 0:
            for [grpNo, faceNo] in surGrpNo:
                if grpNo > maxNo:
                    maxNo = grpNo
    #配列を準備
    newSurConts = [ [] for i in range(maxNo+1) ]
    newElms = []
    #要素毎に処理
    for i in range(len(elms)):
        elm = elms[i]
        #edgeを取得
        edges = []
        edge1 = [elm[1], elm[2]]; edge1.sort(); edges.append(edge1)
        edge2 = [elm[2], elm[3]]; edge2.sort(); edges.append(edge2)
        edge3 = [elm[3], elm[1]]; edge3.sort(); edges.append(edge3)
        edge4 = [elm[4], elm[5]]; edge4.sort(); edges.append(edge4)
        edge5 = [elm[5], elm[6]]; edge5.sort(); edges.append(edge5)
        edge6 = [elm[6], elm[4]]; edge6.sort(); edges.append(edge6)
        edge7 = [elm[1], elm[4]]; edge7.sort(); edges.append(edge7)
        edge8 = [elm[2], elm[5]]; edge8.sort(); edges.append(edge8)
        edge9 = [elm[3], elm[6]]; edge9.sort(); edges.append(edge9)
        #中間節点を取得
        midNodes = []
        #  edgeの中間
        for edge in edges:
            key = (edge[0], edge[1])
            if not (key in edgeDict):
                #nodeを追加
                newLoc = getMiddlePoint(key, nodeList)
                edgeDict[key] = nodeNo
                midNodes.append(nodeNo)
                nodeList.append([newLoc])
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfMiddPoint(key, nodeGrpNos)
                for grpNo in newNgrpNos:
                    ngrpConts[grpNo][1].append(nodeNo)
                nodeNo += 1
            else:
                midNodes.append(edgeDict[key])
        #  面の中心
        faces = []
        quad = [elm[1], elm[2], elm[5], elm[4]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[2], elm[3], elm[6], elm[5]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[3], elm[1], elm[4], elm[6]]; k=sorted(quad); faces.append([k,quad])
        elmNgrpNos = []
        fgNodes = []
        for k, q in faces:
            key = (k[0], k[1], k[2], k[3])
            if not (key in faceDict):
                #node追加
                qLocs = [ nodeList[q[0]][0], nodeList[q[1]][0],
                          nodeList[q[2]][0], nodeList[q[3]][0] ]
                newLoc = geo.quadG(qLocs)
                faceDict[key] = nodeNo
                fgNodes.append(nodeNo)
                nodeList.append([newLoc])
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfFgPoint(key, nodeGrpNos)
                elmNgrpNos.append(newNgrpNos)
                for grpNo in newNgrpNos:
                    ngrpConts[grpNo][1].append(nodeNo)
                nodeNo += 1
            else:
                fgNodes.append(faceDict[key])
        #新elementを作成
        elm1 = [elmNo] +   [elm[1],      midNodes[0], midNodes[2], midNodes[6],
                            fgNodes[0],  fgNodes[2]]
        elm2 = [elmNo+1] + [midNodes[0], elm[2],      midNodes[1], fgNodes[0],
                            midNodes[7], fgNodes[1]]
        elm3 = [elmNo+2] + [midNodes[2], midNodes[0], midNodes[1], fgNodes[2],
                            fgNodes[0],  fgNodes[1]]
        elm4 = [elmNo+3] + [midNodes[2], midNodes[1], elm[3],      fgNodes[2],
                            fgNodes[1],  midNodes[8]]
        elm5 = [elmNo+4] + [midNodes[6], fgNodes[0],  fgNodes[2],  elm[4],
                            midNodes[3], midNodes[5]]
        elm6 = [elmNo+5] + [fgNodes[0],  midNodes[7], fgNodes[1],  midNodes[3],
                            elm[5],      midNodes[4]]
        elm7 = [elmNo+6] + [fgNodes[0],  fgNodes[1],  fgNodes[2],  midNodes[3],
                            midNodes[4], midNodes[5]]
        elm8 = [elmNo+7] + [fgNodes[2],  fgNodes[1],  midNodes[8], midNodes[5],
                            midNodes[4], elm[6]]
        elmNo += 8
        addElms = [elm1] + [elm2] + [elm3] + [elm4] + [elm5] + [elm6] + [elm7] + [elm8]
        newElms += addElms
        #新sgrpを作成
        eNo = elm[0]
        if len(surGrpNos[eNo]) != 0:
            #sgrpが設定されている場合
            for j in range(len(surGrpNos[eNo])):
                [grpNo, faceNo] = surGrpNos[eNo][j]
                surNodes = cm.fistr_elFaceNode("351", elm[1:], faceNo)
                if len(surNodes) == 3:
                    #三角形の場合
                    edge1 = [surNodes[0], surNodes[1]]; edge1.sort(); key1=(edge1[0],edge1[1])
                    edge2 = [surNodes[1], surNodes[2]]; edge2.sort(); key2=(edge2[0],edge2[1])
                    edge3 = [surNodes[2], surNodes[0]]; edge3.sort(); key3=(edge3[0],edge3[1])
                    node1 = edgeDict[key1]
                    node2 = edgeDict[key2]
                    node3 = edgeDict[key3]
                    #新しい面
                    tri1 = [surNodes[0], node1, node3]
                    tri2 = [node1, node2, node3]
                    tri3 = [node1, surNodes[1], node2]
                    tri4 = [node2, surNodes[2], node3]
                    tris = [tri1, tri2, tri3, tri4]
                    #追加した要素毎に面Noを取得
                    for addElm in addElms:
                        newFaceNo = getFaceNo(addElm, tris)
                        if newFaceNo != 0:
                            #該当するfaceが存在する場合
                            newElmNo = addElm[0]
                            newSurConts[grpNo].append([newElmNo, newFaceNo])
                else:
                    #四角形の場合
                    edge1 = [surNodes[0], surNodes[1]]; edge1.sort(); key1=(edge1[0],edge1[1])
                    edge2 = [surNodes[1], surNodes[2]]; edge2.sort(); key2=(edge2[0],edge2[1])
                    edge3 = [surNodes[2], surNodes[3]]; edge3.sort(); key3=(edge3[0],edge3[1])
                    edge4 = [surNodes[3], surNodes[0]]; edge4.sort(); key4=(edge4[0],edge4[1])
                    face = sorted(surNodes); key5 = (face[0], face[1], face[2], face[3])
                    node1 = edgeDict[key1]
                    node2 = edgeDict[key2]
                    node3 = edgeDict[key3]
                    node4 = edgeDict[key4]
                    nodeG = faceDict[key5]
                    #新しい面
                    quad1 = [surNodes[0], node1, nodeG, node4]
                    quad2 = [node1, surNodes[1], node2, nodeG]
                    quad3 = [node4, nodeG, node3, surNodes[3]]
                    quad4 = [nodeG, node2, surNodes[2], node3]
                    quads = [quad1, quad2, quad3, quad4]
                    #追加した要素毎に面Noを取得
                    for addElm in addElms:
                        newFaceNo = getFaceNo(addElm, quads)
                        if newFaceNo != 0:
                            #該当するfaceが存在する場合
                            newElmNo = addElm[0]
                            newSurConts[grpNo].append([newElmNo, newFaceNo])
    meshes = [nodeList, newElms, ngrpConts, nodeGrpNos, newSurConts, edgeDict, faceDict]
    return [meshes, nodeNo, elmNo]

#
#  createNodesElement342
#  ---------------------
#    四面体2次の細分化
def createNodesElement342(meshes, nodeNo, elmNo):
    [nodeList, elms, ngrpConts, nodeGrpNos, surGrpNos, edgeDict, faceDict] = meshes
    #surGrpNoの最大値を取得
    maxNo = -1
    for surGrpNo in surGrpNos:
        if len(surGrpNo) > 0:
            for [grpNo, faceNo] in surGrpNo:
                if grpNo > maxNo:
                    maxNo = grpNo
    #配列を準備
    newSurConts = [ [] for i in range(maxNo+1) ]
    newElms = []
    #要素毎に処理（中間節点を使ってメッシュを細分化）
    for i in range(len(elms)):
        elm = elms[i]
        #中間節点を取得
        midNodes = [elm[7], elm[5], elm[6], elm[8], elm[9], elm[10]]
        #新elementを作成
        #       要素No       1               2           3           4
        elm1 = [elmNo] +   [midNodes[3], midNodes[4], midNodes[5], elm[4]]
        elm2 = [elmNo+1] + [midNodes[4], midNodes[0], midNodes[3], midNodes[2]]
        elm3 = [elmNo+2] + [midNodes[1], midNodes[4], midNodes[5], midNodes[2]]
        elm4 = [elmNo+3] + [midNodes[4], midNodes[3], midNodes[5], midNodes[2]]
        elm5 = [elmNo+4] + [midNodes[0], midNodes[4], midNodes[1], midNodes[2]]
        elm6 = [elmNo+5] + [midNodes[0], elm[1],      midNodes[3], midNodes[2]]
        elm7 = [elmNo+6] + [elm[2],      midNodes[4], midNodes[1], midNodes[0]]
        elm8 = [elmNo+7] + [elm[3],      midNodes[1], midNodes[5], midNodes[2]]
        elmNo += 8
        addElms = [elm1] + [elm2] + [elm3] + [elm4] + [elm5] + [elm6] + [elm7] + [elm8]
        newElms += addElms
        #新sgrpを作成
        eNo = elm[0]
        if len(surGrpNos[eNo]) != 0:
            #sgrpが設定されている場合
            for j in range(len(surGrpNos[eNo])):
                [grpNo, faceNo] = surGrpNos[eNo][j]
                surNodes = cm.fistr_elFaceNode("342", elm[1:], faceNo)
                #新しい面
                tri1 = [surNodes[0], surNodes[1], surNodes[5]]
                tri2 = [surNodes[1], surNodes[3], surNodes[5]]
                tri3 = [surNodes[1], surNodes[2], surNodes[3]]
                tri4 = [surNodes[3], surNodes[4], surNodes[5]]
                tris = [tri1, tri2, tri3, tri4]
                #追加した要素毎に面Noを取得
                for addElm in addElms:
                    newFaceNo = getFaceNo(addElm, tris)
                    if newFaceNo != 0:
                        #該当するfaceが存在する場合
                        newElmNo = addElm[0]
                        newSurConts[grpNo].append([newElmNo, newFaceNo])
    #中間節点を追加し2次要素に変換
    for i in range(len(newElms)):
        elm = newElms[i]
        #edgeを取得
        edges = []
        edge1 = [elm[1], elm[2]]; edge1.sort(); edges.append(edge1)
        edge2 = [elm[2], elm[3]]; edge2.sort(); edges.append(edge2)
        edge3 = [elm[3], elm[1]]; edge3.sort(); edges.append(edge3)
        edge4 = [elm[1], elm[4]]; edge4.sort(); edges.append(edge4)
        edge5 = [elm[2], elm[4]]; edge5.sort(); edges.append(edge5)
        edge6 = [elm[3], elm[4]]; edge6.sort(); edges.append(edge6)
        #中間節点を取得
        midNodes = []
        for edge in edges:
            key = (edge[0], edge[1])
            if not (key in edgeDict):
                #node追加
                newLoc = getMiddlePoint(key, nodeList)
                edgeDict[key] = nodeNo
                midNodes.append(nodeNo)
                nodeList.append([newLoc])                   #nodeListに保存
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfMiddPoint(key, nodeGrpNos)
                for grpNo in newNgrpNos:
                    ngrpConts[grpNo][1].append(nodeNo)      #nodeGrpに保存
                nodeNo += 1
            else:
                midNodes.append(edgeDict[key])
        #2次要素に変換
        addNodes = [midNodes[1], midNodes[2], midNodes[0], midNodes[3],
                    midNodes[4], midNodes[5]]
        newElms[i] += addNodes
    meshes = [nodeList, newElms, ngrpConts, nodeGrpNos, newSurConts, edgeDict, faceDict]
    return [meshes, nodeNo, elmNo]

#
#  createNodesElement362
#  ---------------------
#    六面体2次の細分化
def createNodesElement362(meshes, nodeNo, elmNo):
    [nodeList, elms, ngrpConts, nodeGrpNos, surGrpNos, edgeDict, faceDict] = meshes
    #surGrpNoの最大値を取得
    maxNo = -1
    for surGrpNo in surGrpNos:
        if len(surGrpNo) > 0:
            for [grpNo, faceNo] in surGrpNo:
                if grpNo > maxNo:
                    maxNo = grpNo
    #配列を準備
    newSurConts = [ [] for i in range(maxNo+1) ]
    newElms = []
    #要素毎に処理（中間節点を使ってメッシュを細分化）
    for i in range(len(elms)):
        elm = elms[i]
        #中間節点を取得
        midNodes = [elm[9], elm[10], elm[11], elm[12], elm[13], elm[14],
                    elm[15], elm[16], elm[17], elm[18], elm[19], elm[20]]
        #面中心点を取得
        faces = []
        quad = [elm[1], elm[2], elm[3], elm[4]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[5], elm[6], elm[7], elm[8]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[1], elm[2], elm[6], elm[5]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[2], elm[3], elm[7], elm[6]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[3], elm[4], elm[8], elm[7]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[4], elm[1], elm[5], elm[8]]; k=sorted(quad); faces.append([k,quad])
        elmNgrpNos = []
        fgNodes = []
        for k, q in faces:
            key = (k[0], k[1], k[2], k[3])
            if not (key in faceDict):
                #node追加
                qLocs = [ nodeList[q[0]][0], nodeList[q[1]][0],
                          nodeList[q[2]][0], nodeList[q[3]][0] ]
                newLoc = geo.quadG(qLocs)
                faceDict[key] = nodeNo
                fgNodes.append(nodeNo)
                nodeList.append([newLoc])
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfFgPoint(key, nodeGrpNos)
                elmNgrpNos.append(newNgrpNos)       #六面体中心点追加時に使用
                #  nodeGrpNosへ追加
                if len(nodeGrpNos) > nodeNo:
                    nodeGrpNos[nodeNo] += newNgrpNos
                else:
                    nodeGrpNos.append(newNgrpNos)
                #  ngrpContsへ追加
                for grpNo in newNgrpNos:
                    ngrpConts[grpNo][1].append(nodeNo)
                nodeNo += 1
            else:
                fgNodes.append(faceDict[key])
        #六面体中心点を取得
        #  6面体の中心
        node1 = elm[17]         #edge1-5
        node2 = elm[18]         #edge2-6
        node3 = elm[19]         #edge3-7
        node4 = elm[20]         #edge4-8
        quad = [ nodeList[node1][0], nodeList[node2][0],
                 nodeList[node3][0], nodeList[node4][0] ]
        newLoc = geo.quadG(quad)
        nodeList.append([newLoc])
        #    節点Grpを設定
        sameGrpNos = []
        for grpNo in elmNgrpNos[0]:
            flag = 0
            for grpNos in elmNgrpNos[1:]:
                if not (grpNo in grpNos):
                    flag = 1
                    break
            if flag == 0:
                sameGrpNos.append(grpNo)
        nodeGrpNos.append(sameGrpNos)
        for grpNo in sameGrpNos:
            ngrpConts[grpNo][1].append(nodeNo)
        volNode = nodeNo
        nodeNo += 1
        #新elementを作成
        #       要素No       1               2           3           4
        elm1 = [elmNo]   + [elm[1],      midNodes[0], fgNodes[0],   midNodes[3],
                            midNodes[8], fgNodes[2],  volNode,      fgNodes[5]]
        elm2 = [elmNo+1] + [midNodes[0], elm[2],      midNodes[1],  fgNodes[0],
                            fgNodes[2],  midNodes[9], fgNodes[3],   volNode]
        elm3 = [elmNo+2] + [midNodes[3], fgNodes[0],  midNodes[2],  elm[4],
                            fgNodes[5],  volNode,     fgNodes[4],   midNodes[11]]
        elm4 = [elmNo+3] + [fgNodes[0],  midNodes[1], elm[3],       midNodes[2],
                            volNode,     fgNodes[3],  midNodes[10], fgNodes[4]]
        elm5 = [elmNo+4] + [midNodes[8], fgNodes[2],  volNode,      fgNodes[5],
                            elm[5],      midNodes[4], fgNodes[1],   midNodes[7]]
        elm6 = [elmNo+5] + [fgNodes[2],  midNodes[9], fgNodes[3],   volNode,
                            midNodes[4], elm[6],      midNodes[5],  fgNodes[1]]
        elm7 = [elmNo+6] + [fgNodes[5],  volNode,     fgNodes[4],   midNodes[11],
                            midNodes[7], fgNodes[1],  midNodes[6],  elm[8]]
        elm8 = [elmNo+7] + [volNode,     fgNodes[3],  midNodes[10], fgNodes[4],
                            fgNodes[1],  midNodes[5], elm[7],       midNodes[6]]
        elmNo += 8
        addElms = [elm1] + [elm2] + [elm3] + [elm4] + [elm5] + [elm6] + [elm7] + [elm8]
        newElms += addElms
        #新sgrpを作成
        eNo = elm[0]
        if len(surGrpNos[eNo]) != 0:
            #sgrpが設定されている場合
            for j in range(len(surGrpNos[eNo])):
                [grpNo, faceNo] = surGrpNos[eNo][j]
                surNodes = cm.fistr_elFaceNode("362", elm[1:], faceNo)
                #新しい面
                keyG = [surNodes[0], surNodes[2], surNodes[4], surNodes[6]]
                keyG.sort(); key = (keyG[0], keyG[1], keyG[2], keyG[3])
                nodeG = faceDict[key]
                quad1 = [surNodes[0], surNodes[1], nodeG, surNodes[7]]
                quad2 = [surNodes[1], surNodes[2], surNodes[3], nodeG]
                quad3 = [surNodes[7], nodeG, surNodes[5], surNodes[6]]
                quad4 = [nodeG, surNodes[3], surNodes[4], surNodes[5]]
                quads = [quad1, quad2, quad3, quad4]
                #追加した要素毎に面Noを取得
                for addElm in addElms:
                    newFaceNo = getFaceNo(addElm, quads)
                    if newFaceNo != 0:
                        #該当するfaceが存在する場合
                        newElmNo = addElm[0]
                        newSurConts[grpNo].append([newElmNo, newFaceNo])
    #中間節点を追加し、2次要素に変換
    for i in range(len(newElms)):
        elm = newElms[i]
        #edgeを取得
        edges = []
        edge1 = [elm[1], elm[2]]; edge1.sort(); edges.append(edge1)
        edge2 = [elm[2], elm[3]]; edge2.sort(); edges.append(edge2)
        edge3 = [elm[3], elm[4]]; edge3.sort(); edges.append(edge3)
        edge4 = [elm[4], elm[1]]; edge4.sort(); edges.append(edge4)
        edge5 = [elm[5], elm[6]]; edge5.sort(); edges.append(edge5)
        edge6 = [elm[6], elm[7]]; edge6.sort(); edges.append(edge6)
        edge7 = [elm[7], elm[8]]; edge7.sort(); edges.append(edge7)
        edge8 = [elm[8], elm[5]]; edge8.sort(); edges.append(edge8)
        edge9 = [elm[1], elm[5]]; edge9.sort(); edges.append(edge9)
        edge10 = [elm[2], elm[6]]; edge10.sort(); edges.append(edge10)
        edge11 = [elm[3], elm[7]]; edge11.sort(); edges.append(edge11)
        edge12 = [elm[4], elm[8]]; edge12.sort(); edges.append(edge12)
        #中間節点を取得
        midNodes = []
        #  edgeの中間
        for edge in edges:
            key = (edge[0], edge[1])
            if not (key in edgeDict):
                #node追加
                newLoc = getMiddlePoint(key, nodeList)
                edgeDict[key] = nodeNo
                midNodes.append(nodeNo)
                nodeList.append([newLoc])                   #nodeListに保存
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfMiddPoint(key, nodeGrpNos)
                for grpNo in newNgrpNos:
                    ngrpConts[grpNo][1].append(nodeNo)      #nodeGrpに保存
                nodeNo += 1
            else:
                midNodes.append(edgeDict[key])
        #2次要素に変換
        addNodes = midNodes
        newElms[i] += addNodes
    meshes = [nodeList, newElms, ngrpConts, nodeGrpNos, newSurConts, edgeDict, faceDict]
    return [meshes, nodeNo, elmNo]

#
#  createNodesElement352
#  ---------------------
#    五面体2次の細分化
def createNodesElement352(meshes, nodeNo, elmNo):
    [nodeList, elms, ngrpConts, nodeGrpNos, surGrpNos, edgeDict, faceDict] = meshes
    #surGrpNoの最大値を取得
    maxNo = -1
    for surGrpNo in surGrpNos:
        if len(surGrpNo) > 0:
            for [grpNo, faceNo] in surGrpNo:
                if grpNo > maxNo:
                    maxNo = grpNo
    #配列を準備
    newSurConts = [ [] for i in range(maxNo+1) ]
    newElms = []
    #要素毎に処理（中間節点を使ってメッシュを細分化）
    for i in range(len(elms)):
        elm = elms[i]
        #中間節点を取得
        midNodes = [elm[9], elm[7], elm[8], elm[12], elm[10],
                    elm[11], elm[13], elm[14], elm[15]]
        #面の中心点を取得
        faces = []
        quad = [elm[1], elm[2], elm[5], elm[4]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[2], elm[3], elm[6], elm[5]]; k=sorted(quad); faces.append([k,quad])
        quad = [elm[3], elm[1], elm[4], elm[6]]; k=sorted(quad); faces.append([k,quad])
        elmNgrpNos = []
        fgNodes = []
        for k, q in faces:
            key = (k[0], k[1], k[2], k[3])
            if not (key in faceDict):
                #node追加
                qLocs = [ nodeList[q[0]][0], nodeList[q[1]][0],
                          nodeList[q[2]][0], nodeList[q[3]][0] ]
                newLoc = geo.quadG(qLocs)
                faceDict[key] = nodeNo
                fgNodes.append(nodeNo)
                nodeList.append([newLoc])
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfFgPoint(key, nodeGrpNos)
                elmNgrpNos.append(newNgrpNos)       #六面体中心点追加時に使用
                #  nodeGrpNosへ追加
                if len(nodeGrpNos) > nodeNo:
                    nodeGrpNos[nodeNo] += newNgrpNos
                else:
                    nodeGrpNos.append(newNgrpNos)
                #  ngrpContsへ追加
                for grpNo in newNgrpNos:
                    ngrpConts[grpNo][1].append(nodeNo)
                nodeNo += 1
            else:
                fgNodes.append(faceDict[key])
        #新elementを作成
        elm1 = [elmNo] +   [elm[1],      midNodes[0], midNodes[2], midNodes[6],
                            fgNodes[0],  fgNodes[2]]
        elm2 = [elmNo+1] + [midNodes[0], elm[2],      midNodes[1], fgNodes[0],
                            midNodes[7], fgNodes[1]]
        elm3 = [elmNo+2] + [midNodes[2], midNodes[0], midNodes[1], fgNodes[2],
                            fgNodes[0],  fgNodes[1]]
        elm4 = [elmNo+3] + [midNodes[2], midNodes[1], elm[3],      fgNodes[2],
                            fgNodes[1],  midNodes[8]]
        elm5 = [elmNo+4] + [midNodes[6], fgNodes[0],  fgNodes[2],  elm[4],
                            midNodes[3], midNodes[5]]
        elm6 = [elmNo+5] + [fgNodes[0],  midNodes[7], fgNodes[1],  midNodes[3],
                            elm[5],      midNodes[4]]
        elm7 = [elmNo+6] + [fgNodes[0],  fgNodes[1],  fgNodes[2],  midNodes[3],
                            midNodes[4], midNodes[5]]
        elm8 = [elmNo+7] + [fgNodes[2],  fgNodes[1],  midNodes[8], midNodes[5],
                            midNodes[4], elm[6]]
        elmNo += 8
        addElms = [elm1] + [elm2] + [elm3] + [elm4] + [elm5] + [elm6] + [elm7] + [elm8]
        newElms += addElms
        #新sgrpを作成
        eNo = elm[0]
        if len(surGrpNos[eNo]) != 0:
            #sgrpが設定されている場合
            for j in range(len(surGrpNos[eNo])):
                [grpNo, faceNo] = surGrpNos[eNo][j]
                surNodes = cm.fistr_elFaceNode("352", elm[1:], faceNo)
                if len(surNodes) == 6:
                    #三角形の場合
                    #新しい面
                    tri1 = [surNodes[0], surNodes[1], surNodes[5]]
                    tri2 = [surNodes[1], surNodes[3], surNodes[5]]
                    tri3 = [surNodes[1], surNodes[2], surNodes[3]]
                    tri4 = [surNodes[3], surNodes[4], surNodes[5]]
                    tris = [tri1, tri2, tri3, tri4]
                    #追加した要素毎に面Noを取得
                    for addElm in addElms:
                        newFaceNo = getFaceNo(addElm, tris)
                        if newFaceNo != 0:
                            #該当するfaceが存在する場合
                            newElmNo = addElm[0]
                            newSurConts[grpNo].append([newElmNo, newFaceNo])
                else:
                    #四角形の場合
                    #新しい面
                    keyG = [surNodes[0], surNodes[2], surNodes[4], surNodes[6]]
                    keyG.sort(); key = (keyG[0], keyG[1], keyG[2], keyG[3])
                    nodeG = faceDict[key]
                    quad1 = [surNodes[0], surNodes[1], nodeG, surNodes[7]]
                    quad2 = [surNodes[1], surNodes[2], surNodes[3], nodeG]
                    quad3 = [surNodes[7], nodeG, surNodes[5], surNodes[6]]
                    quad4 = [nodeG, surNodes[3], surNodes[4], surNodes[5]]
                    quads = [quad1, quad2, quad3, quad4]
                    #追加した要素毎に面Noを取得
                    for addElm in addElms:
                        newFaceNo = getFaceNo(addElm, quads)
                        if newFaceNo != 0:
                            #該当するfaceが存在する場合
                            newElmNo = addElm[0]
                            newSurConts[grpNo].append([newElmNo, newFaceNo])
    #肘関節点を追加し、2次要素に変換
    for i in range(len(newElms)):
        elm = newElms[i]
        #edgeを取得
        edges = []
        edge1 = [elm[1], elm[2]]; edge1.sort(); edges.append(edge1)
        edge2 = [elm[2], elm[3]]; edge2.sort(); edges.append(edge2)
        edge3 = [elm[3], elm[1]]; edge3.sort(); edges.append(edge3)
        edge4 = [elm[4], elm[5]]; edge4.sort(); edges.append(edge4)
        edge5 = [elm[5], elm[6]]; edge5.sort(); edges.append(edge5)
        edge6 = [elm[6], elm[4]]; edge6.sort(); edges.append(edge6)
        edge7 = [elm[1], elm[4]]; edge7.sort(); edges.append(edge7)
        edge8 = [elm[2], elm[5]]; edge8.sort(); edges.append(edge8)
        edge9 = [elm[3], elm[6]]; edge9.sort(); edges.append(edge9)
        #中間節点を取得
        midNodes = []
        #  edgeの中間を取得
        for edge in edges:
            key = (edge[0], edge[1])
            if not (key in edgeDict):
                #nodeを追加
                newLoc = getMiddlePoint(key, nodeList)
                edgeDict[key] = nodeNo
                midNodes.append(nodeNo)
                nodeList.append([newLoc])               #nodeListに保存
                #nodeのgrpNoを取得
                newNgrpNos = getNodeGrpNoOfMiddPoint(key, nodeGrpNos)
                for grpNo in newNgrpNos:        
                    ngrpConts[grpNo][1].append(nodeNo)  #nodeGrpに保存
                nodeNo += 1
            else:
                midNodes.append(edgeDict[key])
        #2次要素に変換
        addNodes = [midNodes[1], midNodes[2], midNodes[0], midNodes[4],
                    midNodes[5], midNodes[3], midNodes[6], midNodes[7], midNodes[8]]
        newElms[i] += addNodes
    meshes = [nodeList, newElms, ngrpConts, nodeGrpNos, newSurConts, edgeDict, faceDict]
    return [meshes, nodeNo, elmNo]

#
#  createNodesElementsAllElms
#    中間節点を追加してelementを作り出す
def createNodesElementsAllElms(meshGrps, meshList):

    #  createNodeConts
    #    nodeListからnodeContsを作り出す
    def createNodeConts(nodeList, nodeConts):
        nConts = []
        for i in range(len(nodeList)):
            if len(nodeList[i]) > 0:
                cont = [i] + nodeList[i][0]
                nConts.append(cont)
        nodeConts[0][1] = nConts
        return nodeConts

    #  createSgrpConts
    #    surContsからsgrpContsを作成する
    def createSgrpConts(surConts, sgrpConts):
        for i in range(len(surConts)):
            setData = surConts[i]
            data = []
            for [elmNo, faceNo] in setData:
                data += [elmNo, faceNo]
            sgrpConts[i][1] = data
        return sgrpConts

    #メイン
    print ("    getting node and surface group data...")
    [nodeConts, elementConts, sgrpConts, ngrpConts, otherConts] = meshGrps
    [nodeList, elementList] = meshList
    #辞書定義
    edgeDict = {}
    #nodeNoとnodeGrpNoを取得
    nodeGrpNos = [ [] for i in range(len(nodeList)+1) ]
    for i in range(len(ngrpConts)):
        nodeNos = ngrpConts[i][1]
        for nodeNo in nodeNos:
            nodeGrpNos[nodeNo].append(i)
    #elmNoとsgrpNo,faceNoを取得
    #  surGrpNos[elmNo] = [[sgrpNo, faceNo], ...]
    surGrpNos = [ [] for i in range(len(elementList)+1) ]
    for i in range(len(sgrpConts)):
        elmNos = sgrpConts[i][1]
        for j in range(0, len(elmNos), 2):
            elmNo = elmNos[j]
            faceNo = elmNos[j+1]
            surGrpNos[elmNo].append([i, faceNo])
    #変数を定義
    nodeNo = len(nodeList)      #追加するnodeNo
    elmNo = 1                   #要素のelementNo
    #配列と辞書を定義
    newSurConts = [ [] for i in range(len(sgrpConts)) ]
    edgeDict = {}       #edge辞書を定義
    faceDict = {}       #face辞書を定義

    #要素毎に中間節点、新elementを取得
    print ("    creating new nodes and elements...")
    for i in range(len(elementConts)):
        header = elementConts[i][0]
        elms = elementConts[i][1]
        elType = cm.getValName(header, "TYPE")
        print ("    " + elType + " elements are getting fine...")
        meshes = [nodeList, elms, ngrpConts, nodeGrpNos, surGrpNos, edgeDict, faceDict]
        if elType == "341":
            #四面体1次
            retMesh = createNodesElements341(meshes, nodeNo, elmNo)
        elif elType == "361":
            #六面体1次
            retMesh = createNodesElement361(meshes, nodeNo, elmNo)
        elif elType == "351":
            #五面体1次
            retMesh = createNodesElement351(meshes, nodeNo, elmNo)
        elif elType == "342":
            #四面体2次
            retMesh = createNodesElement342(meshes, nodeNo, elmNo)
        elif elType == "362":
            #六面体2次
            retMesh = createNodesElement362(meshes, nodeNo, elmNo)
        elif elType == "352":
            #五面体2次
            retMesh = createNodesElement352(meshes, nodeNo, elmNo)
        #  配列を展開
        [meshes, nodeNo, elmNo] = retMesh
        [nodeList, elms, ngrpConts, nodeGrpNos, surConts, edgeDict, faceDict] = meshes
        #  element, surfaceGroupのデータを保存
        elementConts[i][1] = elms
        for i in range(len(surConts)):
            newSurConts[i] += surConts[i]
            
    #戻り値を作成
    print ("    binding new nodes and elements...")
    nodeConts = createNodeConts(nodeList, nodeConts)
    sgrpConts = createSgrpConts(newSurConts, sgrpConts)
    return [nodeConts, elementConts, sgrpConts, ngrpConts, otherConts]

#
#  getNewHeaderNumData
#    headerNumDataを再作成する
def getNewHeaderNumData(meshGrps, headerNumData):
    [nodeConts, elementConts, sgrpConts, ngrpConts, otherConts] = meshGrps
    #nodeを書き換え
    for i in range(len(headerNumData)):
        header = headerNumData[i][0]
        for j in range(len(nodeConts)):
            nodeHeader = nodeConts[j][0]
            if header == nodeHeader:
                headerNumData[i][1] = nodeConts[j][1]
    #elementの書き換え
    for i in range(len(headerNumData)):
        header = headerNumData[i][0]
        for j in range(len(elementConts)):
            elmHeader = elementConts[j][0]
            if header == elmHeader:
                headerNumData[i][1] = elementConts[j][1]
    #sgrpの書き換え
    for i in range(len(headerNumData)):
        header = headerNumData[i][0]
        for j in range(len(sgrpConts)):
            sgrpHeader = sgrpConts[j][0]
            if header == sgrpHeader:
                headerNumData[i][1] = sgrpConts[j][1]
    #ngrpの書き換え
    for i in range(len(headerNumData)):
        header = headerNumData[i][0]
        for j in range(len(ngrpConts)):
            ngrpHeader = ngrpConts[j][0]
            if header == ngrpHeader:
                headerNumData[i][1] = ngrpConts[j][1]
    return headerNumData

#
#  createLinesFromConts(meshGrps, headerNumData)
#    meshファイルの行を作り出す
def createLinesFromConts(meshGrps, headerNumData):
    #[nodeConts, elementConts, sgrpConts, ngrpConts, otherConts] = meshGrps
    #linesを作成
    lines = []
    for headerCont in headerNumData:
        header = headerCont[0]
        lines.append(header)
        if len(headerCont[1]) > 0:
            words = cm.deleteSp(header).split(",")
            if words[0].upper() == "!NODE" or words[0].upper() == "!ELEMENT":
                for vals in headerCont[1]:
                    strVals = []
                    for val in vals:
                        strVals.append(str(val))
                    line = ", ".join(strVals) + ls
                    lines.append(line)
            elif words[0].upper() == "!SGROUP":
                for i in range(0, len(headerCont[1]), 2):
                    line = str(headerCont[1][i]) + ", " + str(headerCont[1][i+1]) + ls
                    lines.append(line)
            elif words[0].upper() == "!NGROUP":
                for i in range(0, len(headerCont[1]), 8):
                    vals = headerCont[1][i:i+8]
                    strVals = []
                    for val in vals:
                        strVals.append(str(val))
                    line = ", ".join(strVals) + ls
                    lines.append(line)
            else:
                for line in headerCont[1]:
                    lines.append(line)
    return lines

#
#  printElementNumber
#    要素数を出力する
def printElementNumber(headerNumData):
    num = 0
    for headerCont in headerNumData:
        words = cm.deleteSp(headerCont[0]).split(",")
        if words[0] =="!ELEMENT":
             num += len(headerCont[1])
    print ("  -- nElements: " + str(num))

#
#  addNodesSepElms
#    中間節点を追加し要素を細分化
def addNodesSepElms(meshFile, numlevel):
    #meshファイルの読み込み
    print ("\nFistr format mesh will be created fine.")
    print ("  reading '" + meshFile + "' mesh file...")
    lines = readFileLines(meshFile)
    headerNumData = cm.getFistrHeaderNumData(lines)
    printElementNumber(headerNumData)

    for i in range(numlevel):
        print ("  *** numlevel " + str(i+1) + " ***")
        #各groupを取得
        meshGrps = getEachGroups(headerNumData)
        #[nodeConts, elementConts, sgrpConts, ngrpConts, otherConts] = meshGrps
        meshList = cm.getFistrMeshConts(headerNumData)
        #edge辞書作成、node追加、新element作成
        meshGrps = createNodesElementsAllElms(meshGrps, meshList)
        #headerNumDataを再作成
        headerNumData = getNewHeaderNumData(meshGrps, headerNumData)
        print ("  ", end=" ")
        printElementNumber(headerNumData)

    #meshファイル書き込み
    print ("  writing mesh file...")
    lines = createLinesFromConts(meshGrps, headerNumData)
    writeFileLines(meshFile, lines)
    print ("... done")


if __name__ == "__main__":
    if len(sys.argv) >= 3:
        fileName = sys.argv[1]
        numlevel = int(sys.argv[2])
    elif len(sys.argv) == 2:
        fileName = sys.argv[1]
        numlevel = 1
    else:
        helpMsg()
        exit()
    if fileName == "-h" or fileName == "--help":
        helpMsg()
        exit()
    
    addNodesSepElms(fileName, numlevel)

