#!/usr/bin/python3
#  coding: utf-8
#
#           exodus2csv.py
#
#       exodus形式のdataをvtkExodusIIReaderで読み込み、csv形式で出力する
#       elementGroup毎にcellData、pointDataに分けて取得する。
#         cellData: Id,cellTypeとarrayData
#         pointData:Id,座標値とarrayData
#       file名
#           cells0.x.csv    points0.x.csv   x:timeStepのNo
#           cells1.x.csv    points1.x.csv   0,1:elementGroupのNo
#                :                :
#           <cellData>      <pointData>                
#
#   24/08/03    新規作成
#



import vtk

import os, sys
import glob

#初期値を設定
saveDir = os.getcwd() + os.sep + "convData"
exoFile = "Peridigm_Model.e"

#
#  blockNameOfMetaData
#----------------------
def blockNameOfMetaData(metaData):
    """ metaDataのblock名を取得して返す"""
    lines = str(metaData).split("\n")
    name = None
    for line in lines:
        words = line.split()
        if words[0] == "NAME:":
            name = " ".join(words[1:])
            break
    return name

#
#  getOrgCellData
#-----------------
def getOrgCellData(blockData):
    """ cellDataの基本data（ids、cellType）を取得して返す"""
    allData = []
    ids = ["ids"]
    cellTypes = ["cell_type"]
    nCells = blockData.GetNumberOfCells()
    for nn in range(nCells):
        cellType = blockData.GetCellType(nn)
        ids.append([nn])
        cellTypes.append([cellType])
    allData.append(ids)
    allData.append(cellTypes)
    return allData
    
#
#  getOrgPointData
#------------------
def getOrgPointData(blockData):
    """ pointDataの基本data（ids、座標）を取得して返す"""
    allData = []
    ids = ["ids"]
    locs = ["points"]
    nPoints = blockData.GetNumberOfPoints()
    for nn in range(nPoints):
        loc = blockData.GetPoints().GetPoint(nn)
        ids.append([nn])
        locs.append(loc)
    allData.append(ids)
    allData.append(locs)
    return allData
    
#
#  getArrayDataFromVtk
#-----------------------
def getArrayDataFromVtk(dataArray, nVals, nCompo):
    """ block内の全arrayデータを取得して返す"""
    data = []
    for m in range(0, nVals, nCompo):
        item = []
        for i in range(nCompo):
            value = dataArray.GetValue(m)
            item.append(value)
        data.append(item)
    return data

#
#  createCsvLines
#-----------------
def createCsvLines(allData):
    """ 取得した全dataをcsv形式に変換したlinesを返す"""
    csvData = []
    nArray = len(allData)
    #title行作成
    row = []
    for i in range(nArray):
        title = allData[i][0] 
        item = allData[i][1]
        nCompo = len(item)
        if nCompo > 1:
            for ii in range(nCompo):
                row.append(title + ":" + str(ii))
        else:
            row.append(title)
    csvData.append(row)
    #data行作成
    nData = len(allData[0])
    for m in range(1, nData):
        row = []
        for i in range(nArray):
            item = allData[i][m]
            nCompo = len(item)
            for ii in range(nCompo):
                row.append(item[ii])
        csvData.append(row)
    #csv行を作成（dataを文字列に変換）
    lines = []
    row = csvData[0]
    words = list(map(lambda x: '"' + x + '"', row))
    line = ",".join(words) + "\n"
    lines.append(line)
    for i in range(1, nData):
        row = csvData[i]
        words = list(map(str, row))
        line = ",".join(words) + "\n"
        lines.append(line)
    return lines

#-----------------------
#  createCsvAtTimeStep
#-----------------------
def createCsvAtTimeStep(reader, timeStep):
    """ 指定したtimeStep時のcell、pointのデータを変換する。"""
    #timeStepを設定
    reader.SetTimeStep(timeStep)
    reader.Update()

    #exodusDataの全resultDataの読み込みを設定
    nNames = reader.GetNumberOfPointResultArrays()
    for n in range(nNames):
        arrayName = reader.GetPointResultArrayName(n)
        reader.SetPointResultArrayStatus(arrayName, 1)
    nNames = reader.GetNumberOfElementResultArrays()
    for n in range(nNames):
        arrayName = reader.GetElementResultArrayName(n)
        reader.SetElementResultArrayStatus(arrayName, 1)    
    #全data読み込み開始
    reader.Update()

    #Exodus形式を通常のmultiblockに変換
    output = reader.GetOutput()

    #計算結果があるblock「Element Block」を取得
    nBlocks = output.GetNumberOfBlocks()
    for n in range(nBlocks):
        metaData = output.GetMetaData(n)
        blockName = blockNameOfMetaData(metaData)
        if blockName == "Element Blocks":
            elementBlock = output.GetBlock(n)
            break

    #ElementGroup毎に結果を取得
    nBlocks = elementBlock.GetNumberOfBlocks()
    for n in range(nBlocks):
        #ElementGroup毎
        blockData = elementBlock.GetBlock(n)
        #cellのids, cellTypeを取得    
        allData = getOrgCellData(blockData)
        #cellDataを取得
        nArray = blockData.GetCellData().GetNumberOfArrays()
        for m in range(nArray):
            #変数名毎にdataを取得
            cellData = blockData.GetCellData()
            dataName = cellData.GetArrayName(m)
            dataArray = cellData.GetArray(m)
            nVals = dataArray.GetNumberOfValues()
            nCompo = dataArray.GetNumberOfComponents()
            data = getArrayDataFromVtk(dataArray, nVals, nCompo)
            data = [dataName] + data
            allData.append(data)
        #csvファイルを作成
        lines = createCsvLines(allData)
        csvName = "cells" + str(n) + "." + str(timeStep) + ".csv"
        fileName = saveDir + os.sep + csvName
        f = open(fileName, "w")
        f.writelines(lines)
        f.close()

    #pointDataを取得
    for n in range(nBlocks):
        #ElementGroup毎
        blockData = elementBlock.GetBlock(n)
        #pointのidsと座標値を取得
        allData = getOrgPointData(blockData)
        #pointDataを取得
        nArray = blockData.GetPointData().GetNumberOfArrays()
        for m in range(nArray):
            #変数名毎にdataを取得
            pointData = blockData.GetPointData()
            dataName = pointData.GetArrayName(m)
            dataArray = pointData.GetArray(m)
            nVals = dataArray.GetNumberOfValues()
            nCompo = dataArray.GetNumberOfComponents()
            data = getArrayDataFromVtk(dataArray, nVals, nCompo)
            data = [dataName] + data
            allData.append(data)
        #csvファイルを作成
        lines = createCsvLines(allData)
        csvName = "points" + str(n) + "." + str(timeStep) + ".csv"
        fileName = saveDir + os.sep + csvName
        f = open(fileName, "w")
        f.writelines(lines)
        f.close()

#----------------
#  convertToCsv
#----------------
def convertToCsv():
    """ 全TimeStepにおいてcsvファイルを作成する"""
    global exoFile
    if os.path.exists(exoFile) == False:
        print("ERROR: exoFile「" + exoFile + "」が存在しません。")
        return
    reader = vtk.vtkExodusIIReader()
    reader.SetFileName(exoFile)
    reader.UpdateInformation()
    (stepMin, stepMax) = reader.GetTimeStepRange()
    #全stepの変換開始
    print("exodusFileを読み込み、csvファイルを作成中...")
    for timeStep in range(stepMin, stepMax+1):
        print("creating csv file at timeStep " + str(timeStep))
        createCsvAtTimeStep(reader, timeStep)

#
#  getOption
#------------
def getOption():
    """ optionを取得する"""
    global exoFile, saveDir
    i = 1
    while i < len(sys.argv):
        if sys.argv[i] == "-i" or sys.argv[i] == "--input":
            #exodus形式の読み込みfile
            i += 1
            exoFile = sys.argv[i]
        elif sys.argv[i] == "-o" or sys.argv[i] == "--outDir":
            #出力するdirを設定
            i += 1
            saveDir = sys.argv[i]
        i += 1

#
#  checkSaveDir
#--------------
def checkSaveDir():
    """ csvを保存するsaveDirの有無を確認し、無ければ作成、
    有ればfolder内のfileを削除する"""
    #saveDirのチェック
    if os.path.exists(saveDir) == False:
        #folder作成
        os.mkdir(saveDir)
    else:
        #folder内をクリア
        files  = glob.glob(saveDir + os.sep + "cells*.csv")
        files += glob.glob(saveDir + os.sep + "points*.csv")
        files += glob.glob(saveDir + os.sep + "*.vtk")
        for file in files:
            os.remove(file)


if __name__ == "__main__":
    getOption()
    checkSaveDir()
    convertToCsv()


                    
