#!/usr/bin/python3
#  coding: utf-8
#
#           createMesh_netgen.py
#
#           netgenによるメッシュ作成
#
#   25/02/22    新規作成
#

import os, sys
import netgen.occ as occ
import netgen.meshing as ngmesh
import pyngcore as ngcore
import time
import pyFistr
import convertMesh as cm
import geometryFunc as geom

fineDict = {
    "very_coarse": occ.meshsize.very_coarse,
    "coarse":      occ.meshsize.coarse,
    "moderate":    occ.meshsize.moderate,
    "fine":        occ.meshsize.fine,
    "very_fine":   occ.meshsize.very_fine
    }

#
#  getNodesElementsDict
#-----------------------
def getNodesElementsDict(mesh):
    """ vol形式をfistr形式に変換して返す"""
    #nodesを取得
    points = mesh.Points()
    nodeDict = {}
    nodeNo = 1
    for point in points:
        nodeDict[nodeNo] = point
        nodeNo += 1
    #elements取得
    elms3D = mesh.Elements3D()
    elms2D = mesh.Elements2D()
    elms1D = mesh.Elements1D()
    elementDict = {}
    elmNo = 1
    nD = ""         #3D or 2D or 1D
    if len(elms3D) > 0:
        for elm in elms3D:
            nodes = list(elm.points)
            nD = "3D"
            elmType, newNodes = cm.vol2fistr_el(nD, nodes)
            elementDict[elmNo] = [elmType, newNodes]
            elmNo += 1
    elif len(elms2D) > 0:
        for elm in elms2D:
            nodes = list(elm.points)
            nD = "2D"
            elmType, newNodes = cm.vol2fistr_el(nD, nodes)
            elementDict[elmNo] = [elmType, newNodes]
            elmNo += 1
    elif len(elms1D) > 0:
        for elm in elms1D:
            nodes = list(elm.points)
            nodes = [nodes[0], nodes[1]]    #1次要素に固定
            nD = "1D"
            elmType, newNodes = cm.vol2fistr_el(nD, nodes)
            elementDict[elmNo] = [elmType, newNodes]
            elmNo += 1
    return nodeDict, elementDict, nD

#
#  runNetgen
#-------------
def runNetgen():
    """ netgen実行"""
    #thread並列で実行
    nCores = 1
    if thread == "yes":
        nCores = int(nThread)
    ngcore.SetNumThreads(nCores)        #nThreadを設定
    with ngcore.TaskManager(): 
        #thread並列で実行
        meshFineness = fineDict[fineness]
        geo = occ.OCCGeometry(cadFilePath)
        mesh = geo.GenerateMesh(meshFineness, maxh=maxh, minh=minh)
        if mesh2nd == "yes":
            mesh.SecondOrder()
    return mesh
    # meshFineness = fineDict[fineness]
    # geo = occ.OCCGeometry(cadFilePath)
    # mesh = geo.GenerateMesh(meshFineness, maxh=maxh, minh=minh, try_hexes=True)
    # if mesh2nd == "yes":
    #     mesh.SecondOrder()
    # return mesh

#
#  readVolMesh
#--------------
def readVolMesh(fileName):
    """ volファイルをopen"""
    #fileが出来上がるまで待つ
    count = 20
    while os.path.exists(fileName) == False:
        time.sleep(0.1)
        count -= 1
        if count < 0:
            break
    #file読み込み
    lines = []
    f = open(fileName)
    while True:
        line = f.readline()
        if line[:len("endmesh")] == "endmesh":
            break
        lines.append(line)
    return lines

#
#  getElementSurfaceLines
#------------------
def getElementSurfaceLines(lines):
    """ elm, surfaceに関するlineを取得して返す"""
    keywords = ["volumeelements", "surfaceelementsuv"]
    elmLines, surLines = get2BlockLines(lines, keywords)
    return elmLines, surLines

#
#  getSurfaceEdgeLines
#----------------------
def getSurfaceEdgeLines(lines):
    """ surface, edgeに関するlineを取得して返す"""
    keywords = ["surfaceelementsuv", "edgesegmentsgi2"]
    surLines, edgeLines = get2BlockLines(lines, keywords)
    return surLines, edgeLines

#
#  getEdgePointLines
#--------------------
def getEdgePointLines(lines):
    """ edge, pointElementに関するlineを取得"""
    keywords = ["edgesegmentsgi2", "pointelements"]
    edgeLines, pointLines = get2BlockLines(lines, keywords)
    return edgeLines, pointLines

#  get2BlockLines
def get2BlockLines(lines, keywords):
    """ keywordのblockを取得する"""
    flag = 0
    i = 0
    while i < len(lines):
        line = lines[i]
        if line[:len(keywords[0])] == keywords[0]:
            elmi = i + 2
            flag += 1
            elmLines, i = getBlockLines(lines, elmi)
        elif line[:len(keywords[1])] == keywords[1]:
            suri = i + 2
            flag += 1
            surLines, i = getBlockLines(lines, suri)
        if flag == 2:
            break
        i += 1
    return elmLines, surLines

#  getBlockLines
def getBlockLines(lines, si):
    """ block行を取得して返す"""
    i = si
    blockLineWords = []
    while i < len(lines):
        line = lines[i]
        words = line.split()
        if len(words) == 0:
            break
        blockLineWords.append(words)
        i += 1
    return blockLineWords, i

#
#  getEgrpDict
#--------------
def getEgrpDict(elmLines):
    """ EGRPを取得して返す"""
    egrpDict = {}
    elmNo = 1
    for words in elmLines:
        egrpNo = words[0]
        if egrpNo in egrpDict.keys():
            egrpDict[egrpNo] += [elmNo]
        else:
            egrpDict[egrpNo] = [elmNo]
        elmNo += 1
    return egrpDict

#
#  getSgrpNgrpDict
#------------------
def getSgrpNgrpDict(surLines, elementDict, mesh):
    """ SGRP, NGRPを取得して返す"""
    #SGRP、NGRPを取得    
    sgrpDict = {}
    ngrpDict = {}
    i = 0
    for words in surLines:
        surfaceNo = words[0]               #SGRPのNo
        nNodes = int(words[4])
        aFace = words[5:5+nNodes]
        aFace = list(map(int, aFace))
        rFace = aFace[:3]
        rFace.sort()
        face = tuple(rFace)                #key
        #netgenのsurfaceNoからelmNoを取得                    #elmNo
        elmNo, dummy = mesh.GetVolumeNeighboursOfSurfaceElement(i)
        if elmNo > 0:
            #elmNoからnode列を取得
            elmType, elNodes = elementDict[elmNo]
            #faceNoを取得
            faceNo = cm.fistr_elFace(face, elNodes)     #faceNo
            #結果を取得
            if surfaceNo in sgrpDict.keys():
                sgrpDict[surfaceNo] += [[elmNo, faceNo]]
                ngrpDict[surfaceNo] += aFace            
            else:
                sgrpDict[surfaceNo] = [[elmNo, faceNo]]
                ngrpDict[surfaceNo] = aFace
        i += 1
    #NGRPを修正（ダブリを無くす）
    for key in ngrpDict.keys():
        nodes = ngrpDict[key]
        newNodes = list(set(nodes))
        newNodes.sort()
        ngrpDict[key] = newNodes
    return sgrpDict, ngrpDict

#
#  getNgrpFromEdge
#-------------------
def getNgrpFromEdge(edgeLines, elementDict):
    """ sheetのedgeからngrpDictを作成"""
    
    def getEdgeKey(edge):
        """ edgeをkey(int)に変更する"""
        edge = list(map(str, edge))
        edge = list(map(int, edge))
        edge.sort()
        key = tuple(edge)
        return key, edge
    
    #edgeをkeyとしてnodesを取得する辞書を作成
    edgeDict = {}
    for elmNo in elementDict.keys():
    #for elmNo in shapeDict["2D"]:
        elmType, nodes = elementDict[elmNo]
        edge0 = [nodes[0], nodes[1]]
        key0, edge = getEdgeKey(edge0)
        edgeDict[key0] = edge
        edge1 = [nodes[1], nodes[2]]
        key1, edge = getEdgeKey(edge1)
        edgeDict[key1] = edge
        edge2 = [nodes[2], nodes[0]]
        key2, edge = getEdgeKey(edge2)
        edgeDict[key2] = edge
        #2次要素？
        if len(nodes) == 6:
            #中間節点を追加
            edgeDict[key0] += [nodes[5]]
            edgeDict[key1] += [nodes[3]]
            edgeDict[key2] += [nodes[4]]
    #ngrpを取得
    ngrpDict = {}
    for words in edgeLines:
        #edgeNoを取得
        edgeNo = words[0]
        #edgeのkeyを取得
        nodes = words[2], words[3]
        nodes = list(map(int, nodes))
        nodes.sort()
        key = tuple(nodes)
        #edgeのnodesを取得
        edgeNodes = edgeDict[key]
        #ngrpDictに保存
        if edgeNo in ngrpDict.keys():
            ngrpDict[edgeNo] += edgeNodes
        else:
            ngrpDict[edgeNo] = edgeNodes
    #ngrpを修正（ダブリを削除）
    for key in ngrpDict.keys():
        nodes = ngrpDict[key]
        newNodes = list(set(nodes))
        ngrpDict[key] = newNodes
    return ngrpDict

#
#  getNgrpFromPointElm
#----------------------
def getNgrpFromPointElm(pntLines):
    """ beamのPointElementsからngrpDictを作成"""
    ngrpDict = {}
    for words in pntLines:
        pntNo = words[0]
        nodeNo = int(words[1])
        if pntNo in ngrpDict.keys():
            ngrpDict[pntNo] += [nodeNo]
        else:
            ngrpDict[pntNo] = [nodeNo]
    return ngrpDict

#
#  saveFisrMesh
#---------------
def saveFistrMesh(nodeDict, elementDict, egrpDict, sgrpDict, ngrpDict, saveFilePath):
    """ ファイルを作成し、保存する"""
    lines = ["!HEADER\n"]
    lines += ["  create netgen, convert to FrontISTR mesh.\n"]
    #NODE行を作成
    lines += ["!NODE\n"]
    nodeNos = list(nodeDict.keys())
    nodeNos.sort()
    for nodeNo in nodeNos:
        loc = nodeDict[nodeNo]
        x = loc[0]
        y = loc[1]
        z = loc[2]
        sx = pyFistr.float2strAuto(x)
        sy = pyFistr.float2strAuto(y)
        sz = pyFistr.float2strAuto(z)
        sNo = str(nodeNo)
        line = ", ".join([sNo, sx, sy, sz]) + "\n"
        lines += [line]
    #ELEMENT行を作成
    for egrp in egrpDict.keys():
        elmNo = egrpDict[egrp][0]
        elmType, nodes = elementDict[elmNo]
        line = "!ELEMENT, TYPE=" + elmType + ", EGRP=EGRP" + egrp + "\n"
        lines += [line]
        elmNos = egrpDict[egrp]
        elmNos.sort()
        for elmNo in elmNos:
            elmType, nodes = elementDict[elmNo]
            sNodes = list(map(str, nodes))
            sNo = str(elmNo)
            line = ", ".join([sNo]+sNodes) + "\n"
            lines += [line]
    #SGRP行を作成
    for sgrp in sgrpDict.keys():
        line = "!SGROUP, SGRP=SGRP" + sgrp + "\n"
        lines += [line]
        for i in range(0, len(sgrpDict[sgrp]), 8):
            sets = sgrpDict[sgrp][i:i+8]
            nums = []
            for elm, face in sets:
                nums += [elm, face]
            words = list(map(str, nums))
            line = ", ".join(words) + "\n"
            lines += [line]
    #NGRP行を作成
    for ngrp in ngrpDict.keys():
        line = "!NGROUP, NGRP=NGRP" + ngrp + "\n"
        lines += [line]
        for i in range(0, len(ngrpDict[ngrp]), 10):
            nums = ngrpDict[ngrp][i:i+10]
            words = list(map(str, nums))
            line = ", ".join(words) + "\n"
            lines += [line]
    lines += "!END\n"
    #ファイル置き換え
    fileName = saveFilePath
    f = open(fileName, "w"); f.writelines(lines); f.close()

#----------------------------
#  createHexMeshFromTetra2nd
#----------------------------
def createHexMeshFromTetra2nd(nodeDict, elementDict, fistrFilePath):
    """ tetra2次メッシュをhexa1次に変換して保存する"""
    #六面体に変換
    #faceの辞書作成
    faceNodeDict, nodeDict = getFaceCenterLoc(nodeDict, elementDict)
    #六面体作成
    newElmDict = {}
    nodeNos = nodeDict.values()
    maxNodeNo = len(nodeNos)
    nodeNo = maxNodeNo + 1
    hexNo = 1
    for elmNo in elementDict.keys():
        elmType, nodes = elementDict[elmNo]
        cNodes = faceNodeDict[elmNo]
        #六面体要素4個を作成
        #面1-7-2-5, tetG-面2-9-面3
        hexNodes = [cNodes[0], nodes[6], nodes[1], nodes[4],       #面1-7-2-5
                    cNodes[4], cNodes[1], nodes[8], cNodes[2]]   #tetG, 面2, 9, 面3
        newElmDict[hexNo] = ["361", hexNodes]
        hexNo += 1
        #面1-5-3-6 ,tetG-面3-10-面4
        hexNodes = [cNodes[0], nodes[4], nodes[2], nodes[5],
                    cNodes[4], cNodes[2], nodes[9], cNodes[3]]
        newElmDict[hexNo] = ["361", hexNodes]
        hexNo += 1
        #面1-6-1-7, tetG-面4-8-面2
        hexNodes = [cNodes[0], nodes[5], nodes[0], nodes[6],
                    cNodes[4], cNodes[3], nodes[7], cNodes[1]]
        newElmDict[hexNo] = ["361", hexNodes]
        hexNo += 1
        #tetG-面4-8-面2, 面3-10-4-9
        hexNodes = [cNodes[4], cNodes[3], nodes[7], cNodes[1],
                    cNodes[2], nodes[9], nodes[3], nodes[8]]
        newElmDict[hexNo] = ["361", hexNodes]
        hexNo += 1
    egrpDict = {}; sgrpDict = {}; ngrpDict = {}
    #currDir = os.path.dirname(fistrFilePath)
    #saveFilePath = currDir + os.sep + "FistrModel_Hex.msh"
    saveFilePath = fistrFilePath
    saveFistrMesh(nodeDict, newElmDict, egrpDict, sgrpDict, ngrpDict, saveFilePath)

#  getFaceCenterLoc
def getFaceCenterLoc(nodeDict, elementDict):
    """ faceの中心座標を辞書形式で取得する"""
    
    def getTriCelterLoc(nodes, nodeDict):
        loc0 = nodeDict[nodes[0]]
        loc1 = nodeDict[nodes[1]]
        loc2 = nodeDict[nodes[2]]
        locG = geom.triG([loc0, loc1, loc2])
        return locG
    
    def getTetraCenterLoc(nodes, nodeDict):
        loc0 = nodeDict[nodes[0]]
        loc1 = nodeDict[nodes[1]]
        loc2 = nodeDict[nodes[2]]
        loc3 = nodeDict[nodes[3]]
        locVG = geom.tetG([loc0, loc1, loc2, loc3])
        return locVG       

    maxNodeNo = len(nodeDict.keys())
    nodeNo = maxNodeNo + 1
    faceLocDict = {}
    faceNodeDict = {} 
    for elmNo in elementDict.keys():
        elmType, aNodes = elementDict[elmNo]
        #face中心座標
        faceLocs = []
        for faceNo in range(1, 5, 1):
            faceNodes = cm.fistr_elFaceNode(elmType, aNodes, faceNo)
            nodes = (faceNodes[0], faceNodes[2], faceNodes[4])
            words = list(map(str, nodes))
            nodes = list(map(int, words))
            nodes.sort()
            faceKey = tuple(nodes)
            if not faceKey in faceLocDict.keys():
                nodes = (faceNodes[1], faceNodes[3], faceNodes[5])
                loc = getTriCelterLoc(nodes, nodeDict)
                nodeDict[nodeNo] = loc
                faceLocDict[faceKey] = [nodeNo, loc]
                faceLocs.append(nodeNo)
                nodeNo += 1
            else:
                nNo, loc = faceLocDict[faceKey]
                faceLocs.append(nNo)
        #elm中心座標
        loc = getTetraCenterLoc(aNodes[:4], nodeDict)
        nodeDict[nodeNo] = loc
        faceLocs.append(nodeNo)
        nodeNo += 1
        faceNodeDict[elmNo] = faceLocs
    return faceNodeDict, nodeDict

#--------------------
#  convertVolToFistr
#--------------------
def convertVolToFistr(mesh, newMeshPath, fistrFilePath):
    """ vol形式meshをfistr形式に変換する"""
    nodeDict, elementDict, nD = getNodesElementsDict(mesh)
    #volメッシュを読み込み
    lines = readVolMesh(newMeshPath)
    egrpDict = {}; sgrpDict = {}; ngrpDict = {}
    if nD == "3D":
        #必要な行のみ取得
        elmLines, surLines = getElementSurfaceLines(lines)
        #EGRPを取得
        egrpDict = getEgrpDict(elmLines)    
        #SGRP,NGRPを取得
        #sgrpDict, ngrpDict = getSgrpNgrpDict(surLines, elementDict, mesh)
    elif nD == "2D":
        #必要な行を取得
        surLines, edgeLines = getSurfaceEdgeLines(lines)
        #EGRPを取得
        egrpDict = getEgrpDict(surLines)
        sgrpDict = {}
        #NGRPを取得
        #ngrpDict = getNgrpFromEdge(edgeLines, elementDict)
    elif nD == "1D":
        edgeLines, pntLines = getEdgePointLines(lines)
        #EGRPを取得
        egrpDict = getEgrpDict(edgeLines)
        sgrpDict = {}
        #ngrpDict = getNgrpFromPointElm(pntLines)
    #mshファイルを作成
    saveFilePath = fistrFilePath
    saveFistrMesh(nodeDict, elementDict, egrpDict, sgrpDict, ngrpDict, saveFilePath)

#--------------
#  createMesh
#==============
def createMesh():
    """ netgenによるメッシュ作成"""
    currDir = os.path.dirname(cadFilePath)
    #メッシュ作成
    startTime = time.time()
    print("creating mesh at Netgen...")
    mesh = runNetgen()
    #volメッシュを保存
    newMeshPath = currDir + os.sep + "newMesh.vol"
    mesh.Save(newMeshPath)
    convTime = time.time()
    print("  finished. " + ("%.2f" % (convTime-startTime)) + " s")
    #node, elementを取得
    print("converting Neggen to FrontISTR format...")
    fistrFilePath = currDir + os.sep + "FistrModel_new.msh"
    #mesh変換する
    mesh = ngmesh.Mesh()
    mesh.Load(newMeshPath)
    convertVolToFistr(mesh, newMeshPath, fistrFilePath)
    endTime = time.time()
    print("  finished. " + ("%.2f" % (endTime-convTime)) + " s")

if __name__ == "__main__":
    fineness = sys.argv[1]
    maxh = float(sys.argv[2])
    minh = float(sys.argv[3])
    mesh2nd = sys.argv[4]
    cadFilePath = sys.argv[5]
    thread = sys.argv[6]
    nThread = sys.argv[7]
    createMesh()


