#!/usr/bin/python3
#  coding: utf-8
#
#           createMesh_gmsh.py
#
#       gmshによるメッシュ作成
#
#   25/03/06    新規作成
#      04/17    getYesNoValues:不要なaddMeshを削除。
#               作成したmeshの保存file名変更
#      06/03    mesh作成条件のdataを引数からfileで受け取る様に修正。
#               部分的なmeshSize変更。NGRP、SGRPの設定を追加。
#      06/04    getSgrpDict:円錐の頂点付近の要素の場合、faceのnode取得漏れ
#               が発生するので、修正。
#               checkNodeElementDictOfHexa:sgrpを修正の対象に追加。
#      06/10    points.vtuファイルからpointMeshSizeの設定を読み込み
#               pointのmeshSizeを設定するように修正。
#      06/27    saveFistrMesh:「!SECTION」行を追加。
#      07/01    setDefaultPointMeshSize:defaultのmeshSizeがmaxSizeよりも大きい
#               は、maxSizeに修正する様に追加。
#      08/20    getSgrpDict:SGRP取得方法変更。nodeElmDictを使って取得。
#               高速化の為。
#      09/25    選択部品の非表示設定、EGRPの設定を追加する大幅修正。
#

import vtk

import os, sys
import gmsh
import meshio
import time
import pickle
import pyFistr
import convertMesh as cm
import geometryFunc as geom
import readCadFileCreateVtkFiles_gmsh as readCad

import logFileCreater
logFileCreater.log()

elmContDict = {
    #type dim  fistr
    1:  ["1D", "611"],      #beam1次
    2:  ["2D", "731"],      #三角形1次tri
    3:  ["2D", "741"],      #四角形1次quar
    4:  ["3D", "341"],      #四面体1次tet
    5:  ["3D", "361"],      #六面体1次hex
    6:  ["3D", "351"],      #五面体1次
    7:  ["3D", ""],         #pyramid
    8:  ["1D", "612"],      #beam2次
    9:  ["2D", "732"],      #三角形2次tri
    10: ["2D", "743"],      #四角形3次
    11: ["3D", "342"],      #四面体2次tet
    12: ["3D", ""],         #27nodesHex
    13: ["3D", ""],         #18nodesPrism
    14: ["3D", ""],         #14nodePyramid
    15: ["0D", ""],         #点point
    16: ["2D", "742"],      #四角形2次
    17: ["3D", "362"],      #六面体2次hex
    18: ["3D", "352"]       #五面体2次
    }

#
#  getBigestDimType
def getBiggestDimType():
    """ メッシュ内の最大の次元のmeshTypeを取得"""
    elmTypes = gmsh.model.mesh.getElementTypes()
    elms = []
    for elmType in elmTypes:
        if elmType in elmContDict.keys():
            dim = elmContDict[elmType][0]
            elms.append([dim, elmType])
    elms.sort()
    biggestDim = elms[-1][1]
    return biggestDim

#
#  getLocalMeshSizeOfPoints
def getLocalMeshSizeOfPoints(localSizes, pointIdDict):
    """ localMeshの設定内容からmeshSizeとpointNoを取得し返す"""
    meshSize, selType, gmshNos = localSizes
    #readerを取得
    surReader = getSurfaceReader()
    edgeReader = getEdgeReader()
    pntReader = getPointReader()
    #pointsのlocを取得
    pntLocs = getPointLocsFromVtuFile(pntReader)
    #selType毎に処理
    if selType == "volume":
        volNos = list(map(lambda x: x-1, gmshNos))
        locs = getVolumeLocsFromVtuFile(volNos, surReader)
        pointNos = getIncludePointsInSurface(pntLocs, locs)
        pointNos = remakePointNosAtDefMeshSize(pointNos, pointIdDict, meshSize)
        meshSize = float(meshSize)
    elif selType == "surface":
        surNos = list(map(lambda x: x-1, gmshNos))
        locs = getSurfaceLocsFromVtuFile(surNos, surReader)
        pointNos = getIncludePointsInSurface(pntLocs, locs)
        meshSize = float(meshSize)
    elif selType == "edge":
        curNos = list(map(lambda x: x-1, gmshNos))
        locs = getEdgeLocsFromVtuFile(curNos, edgeReader)
        pointNos = getIncludePointsInSurface(pntLocs, locs)
        meshSize = float(meshSize)
    else:
        meshSize = float(meshSize)
        pointNos = list(map(int, gmshNos))
    return meshSize, pointNos

#  getSurfaceReader
def getSurfaceReader():
    """ surfaces.vtuを読み込むreaderを定義"""
    fileName = CurrDir + os.sep + "vtkCadData" + os.sep + "surfaces.vtu"
    reader = vtk.vtkXMLUnstructuredGridReader()
    reader.SetFileName(fileName)
    reader.Update()
    return reader

#  getEdgeReader
def getEdgeReader():
    """ edges.vtuを読み込むreaderを定義"""
    global CurrDir
    fileName = CurrDir + os.sep + "vtkCadData" + os.sep + "edges.vtu"
    reader = vtk.vtkXMLUnstructuredGridReader()
    reader.SetFileName(fileName)
    reader.Update()
    return reader

#  getPointReader
def getPointReader():
    """ points.vtuを読み込むreaderを定義"""
    global CurrDir
    fileName = CurrDir + os.sep + "vtkCadData" + os.sep + "points.vtu"
    reader = vtk.vtkXMLUnstructuredGridReader()
    reader.SetFileName(fileName)
    reader.Update()
    return reader

#  getVolumeLocsFromVtuFile
def getVolumeLocsFromVtuFile(volNos, reader):
    """ volNoに相当する要素の主節点座標を取得する"""
    global currDir
    volNosSet = set(volNos)
    cellData = reader.GetOutput().GetCellData().GetArray("volumeNo")
    nData = cellData.GetSize()
    elmNos = []
    for i in range(nData):
        volNo = cellData.GetValue(i)
        if volNo in volNosSet:
            elmNos.append(i)
    #主nodeNoを取得
    cellNodeData = getVtkCellConts(reader)
    nodeNos = []
    for elmNo in elmNos:
        cellConts = cellNodeData[elmNo]
        nNodes = cellConts[0]
        nodes = cellConts[1:]
        if len(nodes) > 3:
            nodeNos += [nodes[0], nodes[2], nodes[4]]
        else:
            nodeNos += [nodes[0], nodes[1], nodes[2]]
    nodeNos = list(set(nodeNos))
    #座標を取得
    allPoints = reader.GetOutput().GetPoints().GetData()
    locs = []
    for nodeNo in nodeNos:
        i = nodeNo * 3
        x = allPoints.GetValue(i)
        y = allPoints.GetValue(i+1)
        z = allPoints.GetValue(i+2)
        loc = [x,y,z]
        locs.append(loc)
    return locs

#  remakePointNosAtDefMeshSize
def remakePointNosAtDefMeshSize(pointNos, pointIdToSizeDict, meshSize):
    """ default値がmeshSizeより小さい場合、pointNosから除く"""
    newPointNos = []
    meshSize = float(meshSize)
    for pointNo in pointNos:
        defSize = pointIdToSizeDict[pointNo]
        if defSize > meshSize:
            newPointNos.append(pointNo)
    return newPointNos

#  getSurfaceLocsFromVtuFile
def getSurfaceLocsFromVtuFile(surNos, reader):
    """ surNosに相当する要素の主節点座標を取得する"""
    #該当する要素Noを取得
    global CurrDir
    surNosSet = set(surNos)
    cellData = reader.GetOutput().GetCellData().GetArray("surNo")
    nData = cellData.GetSize()
    elmNos = []
    for i in range(nData):
        surNo = cellData.GetValue(i)
        if surNo in surNosSet:
            elmNos.append(i)
    #主nodeNoを取得
    cellNodeData = getVtkCellConts(reader)
    nodeNos = []
    for elmNo in elmNos:
        cellConts = cellNodeData[elmNo]
        nNodes = cellConts[0]
        nodes = cellConts[1:]
        if len(nodes) > 3:
            nodeNos += [nodes[0], nodes[2], nodes[4]]
        else:
            nodeNos += [nodes[0], nodes[1], nodes[2]]
    nodeNos = list(set(nodeNos))
    #座標を取得
    allPoints = reader.GetOutput().GetPoints().GetData()
    locs = []
    for nodeNo in nodeNos:
        i = nodeNo * 3
        x = allPoints.GetValue(i)
        y = allPoints.GetValue(i+1)
        z = allPoints.GetValue(i+2)
        loc = [x,y,z]
        locs.append(loc)
    return locs

#  getEdgeLocsFromVtuFile
def getEdgeLocsFromVtuFile(curNos, reader):
    """ curNosに相当する要素の主節点座標を取得する"""
    #該当する要素Noを取得
    curNosSet = set(curNos)
    cellData = reader.GetOutput().GetCellData().GetArray("curveNo")
    nData = cellData.GetSize()
    elmNos = []
    for i in range(nData):
        curNo = cellData.GetValue(i)
        if curNo in curNosSet:
            elmNos.append(i)
    #主nodeNoを取得
    cellNodeData = getVtkCellConts(reader)
    nodeNos = []
    for elmNo in elmNos:
        cellConts = cellNodeData[elmNo]
        nNodes = cellConts[0]
        nodes = cellConts[1:]
        nodeNos += [nodes[0], nodes[1]]
    nodeNos = list(set(nodeNos))
    #座標を取得
    allPoints = reader.GetOutput().GetPoints().GetData()
    locs = []
    for nodeNo in nodeNos:
        i = nodeNo * 3
        x = allPoints.GetValue(i)
        y = allPoints.GetValue(i+1)
        z = allPoints.GetValue(i+2)
        loc = [x,y,z]
        locs.append(loc)
    return locs

#  getPointLocsFromVtuFile
def getPointLocsFromVtuFile(reader):
    """ point.vtuファイルの全座標を取得する"""
    allPoints = reader.GetOutput().GetPoints().GetData()
    nPoints = allPoints.GetSize()
    locs = []
    for i in range(0, nPoints, 3):
        x = allPoints.GetValue(i)
        y = allPoints.GetValue(i+1)
        z = allPoints.GetValue(i+2)
        loc = [x,y,z]
        locs.append(loc)
    return locs

#  getIncludePointsInSurface
def getIncludePointsInSurface(pntLocs, locs):
    """ locs内に含まれるpntLocsを取得"""
    #toleranceを取得
    xx = list(map(lambda x: x[0], locs))
    yy = list(map(lambda x: x[1], locs))
    zz = list(map(lambda x: x[2], locs))
    lx = max(xx) - min(xx)
    ly = max(yy) - min(yy)
    lz = max(zz) - min(zz)
    tol = max([lx, ly, lz]) / 1.0e4
    #locs内に含まれるpntlocsを取得
    i = 1
    pointNos = []
    for pntLoc in pntLocs:
        for loc in locs:
            l = geom.length(pntLoc, loc)
            if l < tol:
                pointNos.append(i)
                break
        i += 1
    return pointNos

#
#  getVtkCellConts
def getVtkCellConts(reader):
    """ readerからcell内容のcellCont(nNodesとnodes)のlistを取得"""
    cells = reader.GetOutput().GetCells().GetData()
    nData = cells.GetSize()
    cellNodeData = []
    nNodes = 0
    data = []
    for i in range(nData):
        if nNodes == 0:
            if len(data) > 0:
                cellNodeData.append(data)
            #cellのnode数を保存
            data = []
            #vtk-8.2対応
            try:
                nNodes = cells.GetValue(i)
                data.append(nNodes)
            except:
                break
        else:
            #nodeNoを保存
            nodeNo = cells.GetValue(i)
            data.append(nodeNo)
            nNodes -= 1
    if len(data) > 0:
        cellNodeData.append(data)
    return cellNodeData


#
#  runGmsh
#----------
def runGmsh():
    """ gmshを実行してメッシュ作成する"""
    global elmShape, CurrDir
    #ファイル読み込み
    dimTags = gmsh.model.occ.importShapes(cadFilePath)
    gmsh.model.occ.synchronize()
    #modelの次元を確認
    dim = dimTags[0][0]
    if dim == 2:
        #2Dの場合
        if elmShape == "tetra":
            elmShape = "tri"
        elif elmShape == "hexa":
            elmShape = "quad"
    elif dim == 1:
        #1Dの場合
        elmShape = "beam"

    #計算開始
    if elmShape == "tetra":
        #四面体
        runGmsh_tetra()
    elif elmShape == "hexa":
        #六面体
        runGmsh_hexa()
    elif elmShape == "tri":
        #三角形
        runGmsh_tri()
    elif elmShape == "quad":
        #四角形
        runGmsh_quad()
    elif elmShape == "beam":
        #beam
        runGmsh_beam()
    
    #meshを保存
    currDir = CurrDir
    name = os.path.basename(cadFilePath)
    newName = ".".join(name.split(".")[:-1]) + "_gmsh.msh"
    gmsh.write(currDir + os.sep + newName)


#  setDefaultPointMeshSize
def setDefaultPointMeshSize(maxSize):
    """ defaultのpointMeshSizeを設定する"""
    pointIdDict = {}
    pntReader = getPointReader()
    cellData = pntReader.GetOutput().GetCellData().GetArray("meshSize")
    nData = cellData.GetSize()
    for i in range(nData):
        meshSize = cellData.GetValue(i)
        pointId = i + 1
        if meshSize > 0:
            if meshSize < maxSize:
                pointIdDict[pointId] = meshSize
            else:
                pointIdDict[pointId] = maxSize
        else:
            pointIdDict[pointId] = maxSize
    return pointIdDict

#
#  runGmsh_beam
def runGmsh_beam():
    """ beamのメッシュを作成"""
    runGmsh_tri()

#
#  runGmsh_tri
def runGmsh_tri():
    """ tri（三角形）でshellのメッシュを作成"""
    #要素サイズを設定
    maxSize = float(meshSize)
    #minSize = float(minh)
    #minSize = 0.0
    #nCore = int(nThread)

    #全pointsにmaxzSizeを設定
    dimTags = gmsh.model.occ.getEntities(0)
    gmsh.model.occ.mesh.setSize(dimTags, maxSize)
    
    pointIdDict = setDefaultPointMeshSize(maxSize)
    #localMeshSizeを設定
    #  設定値をdictで取得
    pointDict = {}
    for localSizes in localMeshSizes:
        mSize, points = getLocalMeshSizeOfPoints(localSizes, pointIdDict)
        for point in points:
            if point in pointDict.keys():
                pointDict[point] += [mSize]
            else:
                pointDict[point] = [mSize]
    #  同pointに複数設定されている場合は、平均値を設定
    for point in pointDict.keys():
        mSize = sum(pointDict[point]) / len(pointDict[point])
        dimTags = [(0, point)]
        gmsh.model.occ.mesh.setSize(dimTags, mSize)
    gmsh.model.occ.synchronize()

    # # max size, 20 (Mesh.MeshSizeMax, 20)
    # gmsh.option.setNumber("Mesh.MeshSizeMax", maxSize) 
    # # min size, 0.0 (Mesh.MeshSizeMin, 0.0)
    # gmsh.option.setNumber("Mesh.MeshSizeMin", minSize)
    # # finess(elementSizeFactor)
    # gmsh.option.setNumber("Mesh.MeshSizeFactor", 1)

    # #thread並列
    # gmsh.option.setNumber("General.NumThreads", nCore)    
    # if thread == "yes":
    #     #threadの設定
    #     gmsh.option.setNumber("Mesh.Algorithm3D", 10)

    # # 2次要素
    # if mesh2nd == "yes":
    #     gmsh.option.setNumber("Mesh.ElementOrder", 2)

    #メッシュ作成
    gmsh.model.mesh.generate(dim=3)
    #gmsh.model.mesh.optimize()

#
#  runGmsh_quad
def runGmsh_quad():
    """ quad（四角形）でshellのメッシュを作成"""
    #要素サイズを設定
    maxSize = float(meshSize)
    #minSize = float(minh)
    #minSize = 0.0
    #nCore = int(nThread)

    #全pointsにmaxzSizeを設定
    dimTags = gmsh.model.occ.getEntities(0)
    gmsh.model.occ.mesh.setSize(dimTags, maxSize)
    
    pointIdDict = setDefaultPointMeshSize(maxSize)
    #localMeshSizeを設定
    #  設定値をdictで取得
    pointDict = {}
    for localSizes in localMeshSizes:
        mSize, points = getLocalMeshSizeOfPoints(localSizes, pointIdDict)
        for point in points:
            if point in pointDict.keys():
                pointDict[point] += [mSize]
            else:
                pointDict[point] = [mSize]
    #  同pointに複数設定されている場合は、平均値を設定
    for point in pointDict.keys():
        mSize = sum(pointDict[point]) / len(pointDict[point])
        dimTags = [(0, point)]
        gmsh.model.occ.mesh.setSize(dimTags, mSize)
    gmsh.model.occ.synchronize()

    # # max size, 20 (Mesh.MeshSizeMax, 20)
    # gmsh.option.setNumber("Mesh.MeshSizeMax", maxSize) 
    # # min size, 0.0 (Mesh.MeshSizeMin, 0.0)
    # gmsh.option.setNumber("Mesh.MeshSizeMin", minSize)
    # # finess(elementSizeFactor)
    # gmsh.option.setNumber("Mesh.MeshSizeFactor", 1)

    if reformHex == "no":
        gmsh.option.setNumber("Mesh.SubdivisionAlgorithm", 1)   #subdivisionAlgorithm
    else:
        gmsh.option.setNumber('Mesh.RecombineAll', 1)           #Reconbine all trianguler meshes
        gmsh.option.setNumber("Mesh.SubdivisionAlgorithm", 1)   #subdivisionAlgorithm

    # #thread並列
    # gmsh.option.setNumber("General.NumThreads", nCore)    
    # if thread == "yes":
    #     #threadの設定
    #     gmsh.option.setNumber("Mesh.Algorithm3D", 10)
    # # 2次要素
    # if mesh2nd == "yes":
    #     gmsh.option.setNumber("Mesh.ElementOrder", 2)

    #メッシュ作成
    gmsh.model.mesh.generate(dim=3)

#
#  runGmsh_tetra
def runGmsh_tetra():
    """tetraでメッシュ作成"""
    #要素サイズを設定
    maxSize = float(meshSize)
    #minSize = float(minh)
    minSize = 0.0
    nCore = int(nThread)

    #全pointのmeshSizeを設定
    pointIdDict = setDefaultPointMeshSize(maxSize)
    #localMeshSizeを設定
    #  設定値をdictで取得
    pointDict = {}
    for localSizes in localMeshSizes:
        mSize, points = getLocalMeshSizeOfPoints(localSizes, pointIdDict)
        for point in points:
            if point in pointDict.keys():
                pointDict[point] += [mSize]
            else:
                pointDict[point] = [mSize]
    #  同pointに複数設定されている場合は、平均値を設定
    for pointId in pointDict.keys():
        mSize = sum(pointDict[pointId]) / len(pointDict[pointId])
        pointIdDict[pointId] = mSize
    #各pointにmeshSizeを設定
    for pointId in pointIdDict.keys():
        dimTag = [(0, pointId)]
        mSize = pointIdDict[pointId]
        gmsh.model.occ.mesh.setSize(dimTag, mSize)        
    gmsh.model.occ.synchronize()

    # # max size, 20 (Mesh.MeshSizeMax, 20)
    # gmsh.option.setNumber("Mesh.MeshSizeMax", maxSize) 
    # # min size, 0.0 (Mesh.MeshSizeMin, 0.0)
    # gmsh.option.setNumber("Mesh.MeshSizeMin", minSize)
    # finess(elementSizeFactor)
    # gmsh.option.setNumber("Mesh.MeshSizeFactor", 1.0)

    #thread並列
    if thread == "yes":
        #threadの設定
        gmsh.option.setNumber("General.NumThreads", nCore)    
        gmsh.option.setNumber("Mesh.Algorithm3D", 10)

    # 2次要素
    if mesh2nd == "yes":
        gmsh.option.setNumber("Mesh.ElementOrder", 2)

    #メッシュ作成
    gmsh.model.mesh.generate(dim=1)
    gmsh.model.mesh.generate(dim=2)
    gmsh.model.mesh.generate(dim=3)
    #  optimizeを追加するとmultiThreadsでエラー発生する。
    #gmsh.model.mesh.optimize()

#
#  runGmsh_hexa
def runGmsh_hexa():
    """ hexaでメッシュ作成"""
    #cadFile読み込み
    gmsh.model.occ.importShapes(cadFilePath)
    gmsh.model.occ.synchronize()
    #要素サイズを取得
    maxSize = float(meshSize)
    #minSize = float(minh)
    minSize = 0.0

    gmsh.option.setNumber('Mesh.Algorithm', 2)              #2D algorithm
    gmsh.option.setNumber("Mesh.Algorithm3D", 1)            #3D algorithm
    gmsh.option.setNumber("Mesh.RecombinationAlgorithm", 0) #2D reconbination algorithm

    #meshサイズ設定
    #全pointsにmaxzSizeを設定
    dimTags = gmsh.model.occ.getEntities(0)
    gmsh.model.occ.mesh.setSize(dimTags, maxSize)

    pointIdDict = setDefaultPointMeshSize(maxSize)
    #localMeshSizeを設定
    #  設定値をdictで取得
    pointDict = {}
    for localSizes in localMeshSizes:
        mSize, points = getLocalMeshSizeOfPoints(localSizes, pointIdDict)
        for point in points:
            if point in pointDict.keys():
                pointDict[point] += [mSize]
            else:
                pointDict[point] = [mSize]
    #  同pointに複数設定されている場合は、平均値を設定
    for point in pointDict.keys():
        mSize = sum(pointDict[point]) / len(pointDict[point])
        dimTags = [(0, point)]
        gmsh.model.occ.mesh.setSize(dimTags, mSize)
    gmsh.model.occ.synchronize()

    # a = gmsh.model.occ.getEntities(0)
    # gmsh.model.occ.mesh.setSize(a, maxSize)
    # gmsh.model.occ.mesh.setSize([(0,5),(0,6),(0,7),(0,8)], 2.0)
    # gmsh.model.occ.synchronize()

    # gmsh.option.setNumber("Mesh.MeshSizeMax", maxSize)           #max size
    # gmsh.option.setNumber("Mesh.MeshSizeMin", minSize)

    if reformHex == "no":
        #分割のみ
        #allHex
        gmsh.option.setNumber("Mesh.SubdivisionAlgorithm", 2)   #subdivisionAlgorithm
        #gmsh.option.setNumber('Mesh.RecombineAll', 1)           #Reconbine all trianguler meshes

    else:
        #成形
        #allHex
        gmsh.option.setNumber("Mesh.SubdivisionAlgorithm", 2)   #subdivisionAlgorithm
        gmsh.option.setNumber('Mesh.RecombineAll', 1)       #Reconbine all trianguler meshes
        #均等にedge,surface,volumeを分割
        gmsh.model.mesh.setTransfiniteAutomatic()

    # 2次要素
    if mesh2nd == "yes":
        gmsh.option.setNumber("Mesh.ElementOrder", 2)
        #hexaの2次の場合、以下の指定が必要
        gmsh.option.setNumber("Mesh.SecondOrderIncomplete", 1)

    # Generate mesh
    gmsh.model.mesh.generate(dim=3)

#
#  getSurfaceToNodesElmsDict
def getSurfaceToNodesElmsDict():
    """ 座標からsurfaceNoを取得する辞書を作成"""
    #全solid内のfaceTagを取得
    maxTag = gmsh.model.occ.getMaxTag(dim=3)
    surfaceTags = []
    surNos = []
    for i in range(maxTag):
        tag = i + 1
        loopTags, surTags = gmsh.model.occ.getSurfaceLoops(tag)
        surfaceTags += surTags
    #faceTagからphysicalGroupを作成
    i = 1
    for surTag in surfaceTags:
        tags = list(surTag)
        for tag in tags:
            surNos.append(i)
            gmsh.model.addPhysicalGroup(2, [tag], name=str(i))
            i += 1
    #face上のnodeを取得し、Dictを作成
    gmTypes = gmsh.model.mesh.getElementTypes()
    gmType = gmTypes[2]
    elmTags, elmNodeTags = gmsh.model.mesh.getElementsByType(gmType)
    surfaceToElmsDict = {}
    surfaceToNodesDict = {}
    for tag in surNos:
        #surface上のnodeを取得
        nodeTags, coord = gmsh.model.mesh.getNodesForPhysicalGroup(2, tag)
        surfaceToNodesDict[tag] = nodeTags
        #surFace上のnodeに属する要素を取得
        for i in range(len(nodeTags)):
            #nodeTag = nodeTags[i]
            j = i * 3
            loc = coord[j:j+3]
            #座標locのnodeを持っている要素(elmTags)を取得
            elmTags = gmsh.model.mesh.getElementsByCoordinates(loc[0], loc[1], loc[2], dim=3)
            #全elmTags要素のtag(要素の区分:egrp)を取得
            elms = []
            for elmTag in elmTags:
                elmType, nodeTags, dim, eTag = gmsh.model.mesh.getElement(elmTag)
                elms.append([eTag, elmTag])
            #若いeTagを取得
            elms.sort()
            minTag = elms[0][0]
            #複数のelmがあれば、若いTga側のelmを取得
            tagElms = []
            for eTag, elmTag in elms:
                if eTag == minTag:
                    tagElms.append(elmTag)
            if tag in surfaceToElmsDict.keys():
                surfaceToElmsDict[tag] += tagElms
            else:
                surfaceToElmsDict[tag]  = tagElms
    return surfaceToNodesDict, surfaceToElmsDict

#  getEgrpDict
def getEgrpDict():
    """ egrpDictを取得する"""
    gmType = getBiggestDimType()
    #全要素を取得
    elmTags, nodeTags = gmsh.model.mesh.getElementsByType(gmType, tag=-1)
    nAllElms = len(elmTags.tolist())
    tag = 1
    egrpDict = {}
    while True:
        #要素tag毎に要素を取得
        elmTags, nodeTags = gmsh.model.mesh.getElementsByType(gmType, tag=tag)
        elms = elmTags.tolist()
        nElms = len(elms)
        egrpDict[str(tag)] = elms
        nAllElms -= nElms
        #全て取得？
        if nAllElms <= 0:
            break
        tag += 1
    if gmType == 5 or gmType == 17:
        #六面体1次2次の場合、egrpがダブっているため、ここで修正
        nTags = len(egrpDict.keys()) // 2
        newDict = {}
        for i in range(nTags):
            tag = str(i + 1)
            newDict[tag] = egrpDict[tag]
        return newDict
    return egrpDict

#
#  createEgrpNamesFromOption
def createEgrpNamesFromOption(egrpDict):
    """ mesh作成用のoptionFileで定義されているegrp名を取得して、egrpDictを再作成する"""
    global egrpNames
    #egrpNoからegrp名を取得する辞書作成
    num2nameDict = {}
    #   egrp名 対象
    for name,  item, egrpNos in egrpNames:
        for egrpNo in egrpNos:
            strNo = str(egrpNo)
            num2nameDict[strNo] = name
    #egrpDictを再作成
    newDict = {}
    for egrpNo in egrpDict.keys():
        #egrpNameを作成
        if egrpNo in num2nameDict.keys():
            egrpName = num2nameDict[egrpNo]
        else:
            egrpName = "EGRP" + egrpNo
        #egrpNameで辞書再作成
        if egrpName in newDict.keys():
            newDict[egrpName] += egrpDict[egrpNo]
        else:
            newDict[egrpName] = egrpDict[egrpNo]
    return newDict

#
#  checkHeaderData
def checkHeaderData(meshHeaderData):
    """ 数値のtypeを修正する"""
    for i in range(len(meshHeaderData)):
        header, data = meshHeaderData[i]
        words = pyFistr.deleteSp(header).split(",")
        if words[0] == "!SGROUP":
            data = list(map(int, data))
            meshHeaderData[i][1] = data
        elif words[0] == "!NGROUP":
            data = list(map(int, data))
            meshHeaderData[i][1] = data
    return meshHeaderData

#
#  saveFistrMesh
def saveFistrMesh(nodeDict, elementDict, egrpDict, sgrpDict, ngrpDict, saveFilePath):
    """ ファイルを作成し、保存する"""
    lines = ["!HEADER\n"]
    lines += ["  created by Gmeh, converted to FrontISTR format.\n"]
    #NODE行を作成
    lines += ["!NODE\n"]
    nodeNos = list(nodeDict.keys())
    nodeNos.sort()
    for nodeNo in nodeNos:
        loc = nodeDict[nodeNo]
        x = loc[0]
        y = loc[1]
        z = loc[2]
        sx = str(x); sy = str(y); sz = str(z)
        # sx = pyFistr.float2strAuto(x, prec=12)
        # sy = pyFistr.float2strAuto(y, prec=12)
        # sz = pyFistr.float2strAuto(z, prec=12)
        sNo = str(nodeNo)
        line = ", ".join([sNo, sx, sy, sz]) + "\n"
        lines += [line]
    #ELEMENT行を作成
    sectLines = []
    for egrp in egrpDict.keys():
        elmNo = egrpDict[egrp][0]
        elmType, nodes = elementDict[elmNo]
        line = "!ELEMENT, TYPE=" + elmType + ", EGRP=" + egrp + "\n"
        lines += [line]
        sectType = cm.elmContsDict[elmType][0].upper()
        sectLines += ["!SECTION, TYPE=" + sectType + ", EGRP=" + egrp + "\n"] 
        elmNos = egrpDict[egrp]
        elmNos.sort()
        for elmNo in elmNos:
            elmType, nodes = elementDict[elmNo]
            sNodes = list(map(str, nodes))
            sNo = str(elmNo)
            line = ", ".join([sNo]+sNodes) + "\n"
            lines += [line]
    #SECTION行を作成
    lines += sectLines
    #SGRP行を作成
    for sgrp in sgrpDict.keys():
        line = "!SGROUP, SGRP=" + sgrp + "\n"
        lines += [line]
        for i in range(0, len(sgrpDict[sgrp]), 8):
            sets = sgrpDict[sgrp][i:i+8]
            nums = []
            for elm, face in sets:
                nums += [elm, face]
            words = list(map(str, nums))
            line = ", ".join(words) + "\n"
            lines += [line]
    #NGRP行を作成
    for ngrp in ngrpDict.keys():
        line = "!NGROUP, NGRP=" + ngrp + "\n"
        lines += [line]
        for i in range(0, len(ngrpDict[ngrp]), 10):
            nums = ngrpDict[ngrp][i:i+10]
            words = list(map(str, nums))
            line = ", ".join(words) + "\n"
            lines += [line]
    #END行を作成
    lines += ["!END\n"]
    #ファイル置き換え
    fileName = saveFilePath
    f = open(fileName, "w"); f.writelines(lines); f.close()
    return

#  getEgrpSgrpNgrpDict
def getEgrpSgrpNgrpDict(nodeDict, elementDict, nodeElmDict):
    """ hexaのdictを取得して返す"""
    egrpDict = getEgrpDict()
    egrpDict = createEgrpNamesFromOption(egrpDict)
    sgrpDict = getSgrpDict(elementDict, nodeElmDict)
    ngrpDict = getNgrpDict()
    return egrpDict, sgrpDict, ngrpDict

#  getNgrpDict
def getNgrpDict():
    """ ngrpDictを作成する"""
    ngrpDict = {}
    if len(ngrpNames) == 0:
        return ngrpDict
    #作成するngrpの内容を取得
    ngrpSurfaces = []; ngrpEdges = []; ngrpPoints = []
    for ngrp in ngrpNames:
        nameType = ngrp[1]
        if nameType == "surface":
            ngrpSurfaces.append(ngrp)
        elif nameType == "edge":
            ngrpEdges.append(ngrp)
        else:
            ngrpPoints.append(ngrp)
    #ngrpSurfacesを取得
    surNos, curNos = getSurfaceCurveTags()
    #surface上のnodesを取得
    for i in range(len(ngrpSurfaces)):
        name = ngrpSurfaces[i][0]
        ngrp = ngrpSurfaces[i][2]
        for tag in surNos:
            if tag in ngrp:
                nodeTags, coord = gmsh.model.mesh.getNodesForPhysicalGroup(2, tag)
                if name in ngrpDict.keys():
                    ngrpDict[name] += list(nodeTags)
                else:
                    ngrpDict[name] = list(nodeTags)
    #edge上のnodeを取得
    for i in range(len(ngrpEdges)):
        name = ngrpEdges[i][0]
        ngrp = ngrpEdges[i][2]
        for tag in curNos:
            if tag in ngrp:
                nodeTags, coord = gmsh.model.mesh.getNodesForPhysicalGroup(1, tag)
                if name in ngrpDict.keys():
                    ngrpDict[name] += list(nodeTags)
                else:
                    ngrpDict[name] = list(nodeTags)
    #point上のnodeを取得
    allPoints = getAllPointsFromPointsVtuFile()
    for i in range(len(ngrpPoints)):
        name = ngrpPoints[i][0]
        ngrp = ngrpPoints[i][2]
        locs = getPointLocationsFromPointsVtuFile(ngrp, allPoints)
        for ii in range(len(locs)):
            loc = locs[ii]
            x, y, z = loc
            ans = gmsh.model.mesh.getElementByCoordinates(x, y, z, dim=0)
            elmTag, elmType, nodeTags, u, v, w = ans
            nodeNo = nodeTags[0]
            if name in ngrpDict.keys():
                ngrpDict[name] += [nodeNo]
            else:
                ngrpDict[name] = [nodeNo]
    return ngrpDict

#
#  getSgrpDict
def getSgrpDict(elementDict, nodeElmDict):
    """ sgrpDictを作成する"""
    sgrpDict = {}
    #sgrpの内容を取得する
    sgrpSurfaces = []; sgrpEdges = []; sgrpPoints = []
    for sgrp in sgrpNames:
        nameType = sgrp[1]
        if nameType == "surface":
            sgrpSurfaces.append(sgrp)
        elif nameType == "edge":
            sgrpEdges.append(sgrp)
        else:
            sgrpPoints.append(sgrp)
    #sgrpSurfacesを取得
    surNos, curNos = getSurfaceCurveTags()
    #surface上のnodesを取得
    surfaceToNodeSetsDict = {}
    for i in range(len(sgrpSurfaces)):
        name = sgrpSurfaces[i][0]
        sgrp = sgrpSurfaces[i][2]
        for tag in surNos:
            if tag in sgrp:
                nodeTags, coord = gmsh.model.mesh.getNodesForPhysicalGroup(2, tag)
                surfaceToNodeSetsDict[tag] = [set(nodeTags), name]
    #その他surfaceを取得
    for surNo in surNos:
        if not (surNo in surfaceToNodeSetsDict.keys()):
            tag = surNo
            nodeTags, coord = gmsh.model.mesh.getNodesForPhysicalGroup(2, tag)
            surfaceToNodeSetsDict[tag] = [set(nodeTags), "otherS"]

    #surface上のnodeを含むelement,faceNoを取得
    sgrpDict = getSgrpDictFromSurfaceNodeSets(surfaceToNodeSetsDict, elementDict, nodeElmDict)
    return sgrpDict

#  getSgrpDictFromSurfaceNodeSets
def getSgrpDictFromSurfaceNodeSets(surfaceToNodeSetsDict, elementDict, nodeElmDict):
    """ surface上のnodeを含むelement,faceNoを取得
    nodeElmDictを使って取得するように修正。"""
    sgrpDict = {}
    for surNo in surfaceToNodeSetsDict.keys():
        surNodesSet, name = surfaceToNodeSetsDict[surNo]
        #surfaceNodeを含むelementを取得
        elements = []
        for nodeNo in list(surNodesSet):
            elms = nodeElmDict[nodeNo]
            elements += elms
        elements = list(set(elements))
        #関係するelementについて確認する
        for elmNo in elements:
            elmType = elementDict[elmNo][0]
            nodes = elementDict[elmNo][1]
            nodesSet = set(nodes)
            nFaceNodes = len(cm.faceNodeOfElementDict[elmType][1])
            #      surNoの全node  elmNoの全node　
            comms = surNodesSet & nodesSet  #共通のnodeを取得
            #共通node数は、faceNode数を超える場合あり。（円錐の頂点部分は、超える）
            if len(comms) >= nFaceNodes:
                nFaces = len(cm.faceNodeOfElementDict[elmType])
                commsSet = set(comms)
                for faceNo in range(1, nFaces, 1):
                    faceNodes = cm.fistr_elFaceNode(elmType, nodes, faceNo)
                    if len(set(faceNodes) & commsSet) == nFaceNodes:
                        if name in sgrpDict.keys():
                            sgrpDict[name].append([elmNo, faceNo])
                        else:
                            sgrpDict[name] = [[elmNo, faceNo]]
                        if len(comms) == nFaceNodes:
                            break
    return sgrpDict

#  getSurfaceCurveTags
def getSurfaceCurveTags():
    """ cad file内のsurfaceとcurveのtagを取得する。"""
    surNos, curveNos = readCad.getSurfaceCurveTags()
    return surNos, curveNos

#  getAllPointsFromPointsVtuFile
def getAllPointsFromPointsVtuFile():
    """ points.vtuファイルからallPointsを取得して返す"""
    pointsFile = CurrDir + os.sep + "vtkCadData" + os.sep + "points.vtu"
    reader = vtk.vtkXMLUnstructuredGridReader()
    reader.SetFileName(pointsFile)
    reader.Update()
    allPoints = reader.GetOutput().GetPoints().GetData()
    return allPoints

#  getAllPointsFromEdgesVtuFile
def getAllPointsFromEdgesVtuFile():
    """ edges.vtuファイルからallPointsを取得して返す"""
    pointsFile = CurrDir + os.sep + "vtkCadData" + os.sep + "edges.vtu"
    reader = vtk.vtkXMLUnstructuredGridReader()
    reader.SetFileName(pointsFile)
    reader.Update()
    allPoints = reader.GetOutput().GetPoints().GetData()
    return allPoints

#  getPointLocationsFromPointsVtuFile
def getPointLocationsFromPointsVtuFile(nodeNos, allPoints):
    """ points.vtuからgmshTagsに相当するlocを取得して返す。"""
    #nodeNosをvtkのNoに変換
    pointNos = map(lambda x: x-1, nodeNos)
    locs = []
    for pointNo in pointNos:
        i = pointNo * 3
        x = allPoints.GetValue(i)
        y = allPoints.GetValue(i+1)
        z = allPoints.GetValue(i+2)
        loc = [x,y,z]
        locs.append(loc)
    return locs


#  getSgrpNgrpDict
def getSgrpNgrpDict(elementDict):
    """ hexaのsgrpDictを取得
    faceNodesの取得に時間がかかるため、ここでは、使っていない。"""
    #surfaceTag毎のnodes、elmsを取得する辞書を取得
    surfaceToNodesDict, surfaceToElmsDict = getSurfaceToNodesElmsDict()
    #sgrpDictを取得
    sgrpDict = {}
    for surTag in surfaceToElmsDict.keys():
        #surfaceTag上のnodesを取得
        nodes = surfaceToNodesDict[surTag]
        nodesSet = set(nodes)
        #surfaceTag上のnodesが属するelmsを取得
        elms = surfaceToElmsDict[surTag]
        #elmsのfaceNoを取得
        for elm in elms:
            [elmType, elNodes] = elementDict[elm]
            nFaces = len(cm.faceNodeOfElementDict[elmType]) - 1
            for i in range(nFaces):
                faceNo = i + 1
                faceNodes = cm.fistr_elFaceNode(elmType, elNodes, faceNo)
                if elmType == "342":
                    faceNodes = [faceNodes[0], faceNodes[2], faceNodes[4]]
                elif elmType == "362":
                    faceNodes = [faceNodes[0], faceNodes[2], faceNodes[4], faceNodes[6]]
                flag = 0
                for nodeTag in faceNodes:
                    if not nodeTag in nodesSet:
                        flag = 1 
                        break
                if flag == 0:
                    surStr = str(surTag)
                    if surStr in sgrpDict.keys():
                        sgrpDict[surStr] += [[elm, faceNo]]
                    else:
                        sgrpDict[surStr] = [[elm, faceNo]]
    #ngrpDict-----
    ngrpDict = {}
    for surTag in surfaceToNodesDict.keys():
        nodes = surfaceToNodesDict[surTag]
        ngrpDict[str(surTag)] = nodes
    return sgrpDict, ngrpDict

#
#  checkNodeElementDictOfHexa
def checkNodeElementDictOfHexa(nodeDict, elementDict, egrpDict, sgrpDict):
    """ node数、element数をegrpDictに合わせるchechする。
    nodeをelement内で使用しているnodeのみ抽出して返す。"""
    ndDict = {}
    elmDict = {}
    for egrp in egrpDict.keys():
        elms = egrpDict[egrp]
        for elmNo in elms:
            elmType, elNodes = elementDict[elmNo]
            elmDict[elmNo] = [elmType, elNodes]
            for nodeNo in elNodes:
                if nodeNo in ndDict.keys():
                    pass
                else:
                    loc = nodeDict[nodeNo]
                    ndDict[nodeNo] = loc
    #sgrpを修正
    maxElmNo = max(list(elmDict.keys()))
    sgDict = {}
    for sgrp in sgrpDict.keys():
        sgrpVals = sgrpDict[sgrp]
        sgDict[sgrp] = []
        for elmNo, faceNo in sgrpVals:
            if elmNo <= maxElmNo:
                sgDict[sgrp].append([elmNo, faceNo])
    return ndDict, elmDict, sgDict

#
#  showMeshQuality
def showMeshQuality():
    """ meshQualityを取得する"""
    gmType = getBiggestDimType()
    elmTags, nodeTags = gmsh.model.mesh.getElementsByType(gmType, tag=-1)
    elmQ = gmsh.model.mesh.getElementQualities(elmTags)
    print("Element Quality --")
    maxQ = pyFistr.float2strAuto(max(elmQ), prec=4)
    minQ = pyFistr.float2strAuto(min(elmQ), prec=4)
    print("  quality max:" + maxQ + "  min:" + minQ)
    distQ = [0 for i in range(10)]
    for q in elmQ:
        i = int(q*10)
        if 0 < i and i < 10:
            distQ[i] += 1
        elif i <= 0:
            distQ[0] += 1
        elif i >= 10:
            distQ[9] += 1
    for i in range(len(distQ)):
        a = (i + 1)/10
        a = pyFistr.float2strAuto(a, prec=1 )
        print("    < " + a + ":", distQ[i])
    print("  ------------------------------")
    print("    total ", len(elmQ), "elements")


#
#  convertGmshToFistr
#--------------------
def convertGmshToFistr(saveFilePath):
    """ FrontISTR用に変換する"""

    def getNodesElements(elmType, nNodes):
        """ elmTypeとnode数を指定してnodeDict, elementDictを取得して返す
        六面体1次2次要素の場合は、node、elementともダブリが発生しているので注意
        要素のダブリは、egrpDictで調整する。"""
        #nodeを取得
        ndDict = {}
        ndElmDict = {}
        nodeTags, coord, parametricCoord = gmsh.model.mesh.getNodesByElementType(elmType)
        for i in range(len(nodeTags)):
            nodeNo = nodeTags[i]
            j = i * 3
            loc = coord[j:j+3]
            ndDict[nodeNo] = loc
            ndElmDict[nodeNo] = []
        #elemを取得
        elmDict = {}
        elementTags, nodeTags = gmsh.model.mesh.getElementsByType(elmType)
        for i in range(len(elementTags)):
            elmNo = elementTags[i]
            j = i * nNodes
            nodes = nodeTags[j:j+nNodes]
            if elmType == 12:
                #elmType=12の場合27nodesあるので、elmType=17に変更する。
                nodes = nodes[:20]
                elmType = 17        #node数20の要素
            name, newNodes = cm.gmsh2fistr_el(elmType, nodes)
            elmDict[elmNo] = [name, newNodes]
            #nodeElmDictを作成
            for nodeNo in newNodes:
                ndElmDict[nodeNo].append(elmNo)
        return ndDict, elmDict, ndElmDict

    global elmShape
    print("creating FrontISTR format mesh...", flush=True)
    print("  getting nodes and elements dictionary...", flush=True)
    #tetra
    if elmShape == "tetra":
        #tetra分を取得
        elmType = 4; nNodes = 4
        if mesh2nd == "yes":
            elmType = 11; nNodes = 10
        nodeDict, elementDict, nodeElmDict = getNodesElements(elmType, nNodes)
    #hexa
    elif elmShape == "hexa":
        #hexa分を取得
        elmType = 5; nNodes = 8
        if mesh2nd == "yes":
            elmType = 17; nNodes = 20
        nodeDict, elementDict, nodeElmDict = getNodesElements(elmType, nNodes)
    elif elmShape == "tri":
        #tri分を取得
        elmType = 2; nNodes = 3
        nodeDict, elementDict, nodeElmDict = getNodesElements(elmType, nNodes)
    elif elmShape == "quad":
        #quad分を取得
        elmType = 3; nNodes = 4
        nodeDict, elementDict, nodeElmDict = getNodesElements(elmType, nNodes)
    elif elmShape == "beam":
        #beam分を取得
        elmType = 1; nNodes = 2
        nodeDict, elementDict, nodeElmDict = getNodesElements(elmType, nNodes)
    print("  getting nodes, surfaces and elements groups...", flush=True)
    egrpDict, sgrpDict, ngrpDict = getEgrpSgrpNgrpDict(nodeDict, elementDict, nodeElmDict)
    print("  saving mesh for FrontISTR...", flush=True)
    if elmShape == "hexa":
        nodeDict, elementDict, sgrpDict = checkNodeElementDictOfHexa(nodeDict, elementDict, egrpDict, sgrpDict)
    saveFistrMesh(nodeDict, elementDict, egrpDict, sgrpDict, ngrpDict, saveFilePath)

#--------------
#  createMesh
#==============
def createMesh():
    """ gmshによるメッシュ作成"""
    global CurrDir
    currDir = CurrDir
    #メッシュ作成
    startTime = time.time()
    print("creating mesh at Gmsh...")
    #メッシュ作成
    gmsh.initialize()
    runGmsh()
    convTime = time.time()
    print("Gmsh finished. " + ("%.2f" % (convTime-startTime)) + " s")
    #メッシュ品質確認
    showMeshQuality()
    #node, elementを取得
    cadName = os.path.basename(cadFilePath)
    meshName = ".".join(cadName.split(".")[:-1]) + "_fistr.msh"
    fistrFilePath = currDir + os.sep + meshName
    convertGmshToFistr(fistrFilePath)
    gmsh.finalize()

#
#  getParameter
def getParameter():
    localMeshSizes = []; ngrpNames = []; sgrpNames = []
    egrpNames = []
    fileName = CurrDir + os.sep + "createMeshData.gmsh"
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    for line in lines:
        words = line.split()
        if words[0] == "cadFile":
            cadFilePath = words[1]
        elif words[0] == "meshSize":
            meshSize = words[1]
        elif words[0] == "localMeshSize":
            ws = words[1].split(":")
            localMeshSize = ws[0]
            selType = ws[1]
            gmshNos = list(map(int, ws[2].split(",")))
            localMeshSizes.append([localMeshSize, selType, gmshNos])
        elif words[0] == "2ndOrder":
            mesh2nd = words[1]
        elif words[0] == "allHexa":
            allHexa = words[1]
        elif words[0] == "thread":
            thread = words[1]
        elif words[0] == "nThread":
            nThread = words[1]
        elif words[0] == "reformHex":
            reformHex = words[1]
        elif words[0] == "ngrp":
            ws = words[1].split(":")
            name = ws[0]
            nameType = ws[1]
            gmshNos = list(map(int, ws[2].split(",")))
            ngrpNames.append([name, nameType, gmshNos])
        elif words[0] == "sgrp":
            ws = words[1].split(":")
            name = ws[0]
            nameType = ws[1]
            gmshNos = list(map(int, ws[2].split(",")))
            sgrpNames.append([name, nameType, gmshNos])
        elif words[0] == "egrp":
            ws = words[1].split(":")
            name = ws[0]
            nameType = ws[1]
            gmshNos = list(map(int, ws[2].split(",")))
            egrpNames.append([name, nameType, gmshNos])
    return (cadFilePath, meshSize, localMeshSizes, mesh2nd,
            allHexa, thread, nThread, reformHex, ngrpNames,
            sgrpNames, egrpNames)

if __name__ == "__main__":
    CurrDir = sys.argv[1]
    (cadFilePath, meshSize, localMeshSizes, mesh2nd, 
     allHexa, thread, nThread, reformHex, ngrpNames,
     sgrpNames, egrpNames) = getParameter()
    elmShape = "tetra"
    if allHexa == "yes":
        elmShape = "hexa"
    createMesh()
